/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.exception;

import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.type.ASN1Type;

/**
 * Detailed message of exceptions which related to this library.
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
class DetailedMessage {

	private static final String NEW_LINE = System.getProperty("line.separator");

	private String description;

	private Throwable cause;

	private Class<? extends ASN1Type> type;

	private String element;

	private ASN1Type data;

	/**
	 * Instantiate a {@code DetailedMessage}.
	 * 
	 * @param description
	 * @param cause
	 * @param type
	 * @param element
	 * @param data
	 * @param value
	 */
	DetailedMessage(String description, Throwable cause,
			Class<? extends ASN1Type> type, String element, ASN1Type data) {
		this.description = description;
		this.cause = cause;
		this.type = type;
		this.element = element;
		this.data = data;
	}

	/**
	 * Returns detailed message of this instance.
	 * 
	 * @return The detailed message.
	 */
	String getMessage() {
		StringBuilder builder = new StringBuilder();
		if (description != null) {
			builder.append(NEW_LINE).append(" *** ").append(description);
		}
		if (type != null) {
			builder.append(NEW_LINE).append("\tWhile processing ASN.1 type '");
			ASN1Identifier id = type.getAnnotation(ASN1Identifier.class);
			if (id != null) {
				builder.append(id.value());
			} else {
				builder.append(type.getSimpleName());
			}
			builder.append('\'');
			if (element != null) {
				builder.append(", element '").append(element).append('\'');
			}
		}
		if (data != null) {
			builder.append(NEW_LINE).append("\tData description :")
					.append(NEW_LINE);
			try {
				String[] strings = data.toString().split(NEW_LINE);
				for (int i = 0;;) {
					builder.append("\t\t").append(strings[i]);
					if (++i == strings.length) {
						break;
					}
					builder.append(NEW_LINE);
				}
			} catch (Exception ex) {
				builder.append("\t\t")
						.append("(Failed to convert the ASN.1 data to text.)")
						.append(NEW_LINE);
			}
		}
		return builder.toString();
	}

	/**
	 * Returns the {@code Throwable} object that caused an exception.
	 * 
	 * @return The throwable object.
	 */
	Throwable getCause() {
		return cause;
	}

}
