/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.processor;

import jp.bitmeister.asn1.type.ASN1Type;
import jp.bitmeister.asn1.type.StringType;
import jp.bitmeister.asn1.type.TimeType;
import jp.bitmeister.asn1.type.UnknownType;
import jp.bitmeister.asn1.type.builtin.ANY;
import jp.bitmeister.asn1.type.builtin.BIT_STRING;
import jp.bitmeister.asn1.type.builtin.BOOLEAN;
import jp.bitmeister.asn1.type.builtin.CHOICE;
import jp.bitmeister.asn1.type.builtin.ENUMERATED;
import jp.bitmeister.asn1.type.builtin.INTEGER;
import jp.bitmeister.asn1.type.builtin.NULL;
import jp.bitmeister.asn1.type.builtin.OBJECT_IDENTIFIER;
import jp.bitmeister.asn1.type.builtin.OCTET_STRING;
import jp.bitmeister.asn1.type.builtin.REAL;
import jp.bitmeister.asn1.type.builtin.SEQUENCE;
import jp.bitmeister.asn1.type.builtin.SEQUENCE_OF;
import jp.bitmeister.asn1.type.builtin.SET;
import jp.bitmeister.asn1.type.builtin.SET_OF;

/**
 * Interface for classes which behave as visitor of ASN.1 types.
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
public interface ASN1Visitor<R, E extends Throwable> {

	/**
	 * Visits the {@code BOOLEAN} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(BOOLEAN data) throws E;

	/**
	 * Visits the {@code INTEGER} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(INTEGER data) throws E;

	/**
	 * Visits the {@code BIT_STRING} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(BIT_STRING data) throws E;

	/**
	 * Visits the {@code OCTET_STRING} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(OCTET_STRING data) throws E;

	/**
	 * Visits the {@code NULL} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(NULL data) throws E;

	/**
	 * Visits the {@code OBJECT_IDENTIFIER} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(OBJECT_IDENTIFIER data) throws E;

	/**
	 * Visits the {@code REAL} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(REAL data) throws E;

	/**
	 * Visits the {@code ENUMERATED} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(ENUMERATED data) throws E;

	/**
	 * Visits the {@code ANY} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(ANY data) throws E;

	/**
	 * Visits the {@code CHOICE} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(CHOICE data) throws E;

	/**
	 * Visits the {@code SEQUENCE_OF} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(SEQUENCE_OF<? extends ASN1Type> data) throws E;

	/**
	 * Visits the {@code SEQUENCE} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(SEQUENCE data) throws E;

	/**
	 * Visits the {@code SET_OF} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(SET_OF<? extends ASN1Type> data) throws E;

	/**
	 * Visits the {@code SET} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(SET data) throws E;

	/**
	 * Visits the {@code StringType} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(StringType data) throws E;

	/**
	 * Visits the {@code TimeType} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(TimeType data) throws E;

	/**
	 * Visits the {@code UnknownType} data.
	 * 
	 * @param data
	 *            The data to be visited.
	 * @return Result.
	 * @throws E
	 *             When an error occured in the {@code accept} method.
	 */
	public R visit(UnknownType data) throws E;

}
