/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import static java.lang.reflect.Modifier.ABSTRACT;
import static java.lang.reflect.Modifier.INTERFACE;
import static java.lang.reflect.Modifier.PUBLIC;
import static java.lang.reflect.Modifier.STATIC;

import java.util.HashMap;
import java.util.Map;

import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.annotation.ASN1ModuleRef;
import jp.bitmeister.asn1.annotation.ASN1ModuleTags;
import jp.bitmeister.asn1.exception.ASN1IllegalDefinition;

/**
 * The base class for classes which represents an ASN.1 module.
 * 
 * <p>
 * An instance of sub-class of {@code ASN1Module} represents an ASN.1 module and
 * provides {@code instantiate} methods that instantiates an ASN.1 data from an
 * ASN.1 tag or a type identifier. ASN.1 types can indicate a module that they
 * are included in by {@code ASN1ModuleRef} annotation. In addition, ASN.1 types
 * declared as {@code public} {@code static} member classes of an sub-class of
 * {@code ASN1Module} class are automatically included in the module.
 * </p>
 * <p>
 * Default tagging mode for this module may be indicated by an
 * {@code @ASN1ModuleTags} annotation or explicit tagging is used as default.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1Modules
 * @see ASN1ModuleTags
 * @see ASN1ModuleRef
 */
public abstract class ASN1Module {

	/**
	 * Returns the default tagging mode of the ASN.1 module.
	 * 
	 * @param moduleref
	 *            The ASN.1 module.
	 * @return The default tagging mode of the module.
	 */
	static ASN1TagDefault tagDefault(Class<? extends ASN1Module> moduleref) {
		if (moduleref.isAnnotationPresent(ASN1ModuleTags.class)) {
			return moduleref.getAnnotation(ASN1ModuleTags.class).value();
		}
		return ASN1TagDefault.EXPLICIT_TAGS;
	}

	/**
	 * Returns the identifier of the ASN.1 module.
	 * 
	 * @param moduleref
	 *            The ASN.1 module.
	 * @return The identifier of the module.
	 */
	static String identifier(Class<? extends ASN1Module> moduleref) {
		if (moduleref.isAnnotationPresent(ASN1Identifier.class)) {
			return moduleref.getAnnotation(ASN1Identifier.class).value();
		}
		return moduleref.getSimpleName();
	}

	private String identifier;

	private Map<String, Class<? extends ASN1Type>> identifierMap = new HashMap<String, Class<? extends ASN1Type>>();

	private Map<ASN1TagClass, Map<Integer, Class<? extends ASN1Type>>> tagMap = new HashMap<ASN1TagClass, Map<Integer, Class<? extends ASN1Type>>>();

	/**
	 * Instantiates an {@code ASN1Module}. The parameter indicates that this
	 * module is universal or not.
	 * 
	 * @param isUniversal
	 *            {@code true} if this module is universal.
	 */
	ASN1Module(boolean isUniversal) {
		identifier = identifier(getClass());
		if (isUniversal) {
			tagMap.put(ASN1TagClass.UNIVERSAL,
					new HashMap<Integer, Class<? extends ASN1Type>>());
		} else {
			tagMap.put(ASN1TagClass.APPLICATION,
					new HashMap<Integer, Class<? extends ASN1Type>>());
			tagMap.put(ASN1TagClass.PRIVATE,
					new HashMap<Integer, Class<? extends ASN1Type>>());
		}
		for (Class<?> e : getClass().getDeclaredClasses()) {
			if (ASN1Type.class.isAssignableFrom(e)) {
				@SuppressWarnings("unchecked")
				Class<? extends ASN1Type> type = (Class<? extends ASN1Type>) e;
				register(type);
			}
		}
	}

	/**
	 * Instantiates an {@code ASN1Module}.
	 */
	protected ASN1Module() {
		this(false);
	}

	/**
	 * Returns the identifier of this module.
	 * 
	 * @return The identifier of this module.
	 */
	public String identifier() {
		return identifier;
	}

	/**
	 * Registers the ASN.1 type to this module.
	 * 
	 * @param type
	 *            The type to be registered.
	 */
	protected void register(Class<? extends ASN1Type> type) {
		if ((type.getModifiers() & (ABSTRACT | INTERFACE)) != 0) {
			ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
			ex.setMessage("An ASN.1 type class must be instantiatable.", null,
					type, null, null);
			throw ex;
		}
		if ((type.getModifiers() & PUBLIC) == 0) {
			ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
			ex.setMessage("An ASN.1 type class must be a public class", null,
					type, null, null);
			throw ex;
		}
		if (type.isMemberClass() && (type.getModifiers() & STATIC) == 0) {
			ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
			ex.setMessage(
					"If an ASN.1 type class is a member class, it must be static.",
					null, type, null, null);
			throw ex;
		}
		TypeSpecification typeSpec = TypeSpecification.getSpecification(type);
		if (typeSpec.tag() != null) {
			Map<Integer, Class<? extends ASN1Type>> map = tagMap.get(typeSpec
					.tag().tagClass());
			if (map == null) {
				ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
				ex.setMessage("Tag class '" + typeSpec.tag().tagClass()
						+ "'is not allowed to this module.", null, type, null,
						null);
				throw ex;
			}
			if (map.containsKey(typeSpec.tag().tagNumber())) {
				ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
				ex.setMessage("Tag number '" + typeSpec.tag().tagNumber()
						+ "' is dupulicated.", null, type, null, null);
				throw ex;
			}
			map.put(typeSpec.tag().tagNumber(), type);
		}
		identifierMap.put(typeSpec.identifier(), type);
	}

	/**
	 * Instantiates an ASN.1 data specified by the tag.
	 * 
	 * @param tagClass
	 *            The tag class.
	 * @param tagNumber
	 *            The tag number.
	 * @return An instance of ASN.1 data.
	 */
	public ASN1Type instantiate(ASN1TagClass tagClass, int tagNumber) {
		Class<? extends ASN1Type> type = tagMap.get(tagClass).get(tagNumber);
		if (type != null) {
			return ASN1Type.instantiate(type);
		}
		return null;
	}

	/**
	 * Instantiates an ASN.1 data specified by the type identifier.
	 * 
	 * @param typeIdentifier
	 *            The identifier of the type.
	 * @return An instance of ASN.1 data.
	 */
	public ASN1Type instantiate(String typeIdentifier) {
		Class<? extends ASN1Type> type = identifierMap.get(typeIdentifier);
		if (type != null) {
			return ASN1Type.instantiate(type);
		}
		return null;
	}

}
