/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.bitmeister.asn1.annotation.ASN1Element;
import jp.bitmeister.asn1.exception.ASN1IllegalArgument;
import jp.bitmeister.asn1.exception.ASN1IllegalDefinition;
import jp.bitmeister.asn1.exception.ASN1InvalidDataValue;
import jp.bitmeister.asn1.type.builtin.SEQUENCE;
import jp.bitmeister.asn1.type.builtin.SET;

/**
 * The base class for structured types defined by referencing a list of ASN.1
 * types.
 * 
 * <p>
 * This class provides generic interfaces and common methods for classes that
 * represents structured types which defined by referencing a list of ASN.1
 * types. This class is the parent class of {@link SEQUENCE} and {@link SET}.
 * </p>
 * 
 * @see SEQUENCE
 * @see SET
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
public abstract class ConstructiveType extends StructuredType {

	private static Map<Class<? extends ConstructiveType>, ElementSpecification[]> ELEMENTS_MAP = new HashMap<Class<? extends ConstructiveType>, ElementSpecification[]>();

	/**
	 * Returns an array of {@code ElementSpecification} that represents elements
	 * of the type.
	 * 
	 * @param type
	 *            The type.
	 * @return An array of {@code ElementSpecification}.
	 */
	private static ElementSpecification[] getElementTypeList(
			Class<? extends ConstructiveType> type) {
		if (ELEMENTS_MAP.containsKey(type)) {
			return ELEMENTS_MAP.get(type);
		}
		List<ElementSpecification> elements = new ArrayList<ElementSpecification>();
		for (Field f : type.getDeclaredFields()) {
			if (f.isAnnotationPresent(ASN1Element.class)) {
				elements.add(new ElementSpecification(f
						.getAnnotation(ASN1Element.class), f));
			}
		}
		@SuppressWarnings("unchecked")
		Class<? extends ConstructiveType> parent = (Class<? extends ConstructiveType>) type
				.getSuperclass();
		ElementSpecification[] array;
		if (parent == SET.class || parent == SEQUENCE.class) {
			if (elements.isEmpty()) {
				ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
				ex.setMessage(
						"SET and SEQUENCE type shall have at least one element.",
						null, type, null, null);
				throw ex;
			}
			Collections.sort(elements);
			array = elements.toArray(new ElementSpecification[0]);
			if (TypeSpecification.getSpecification(type).tagDefault() == ASN1TagDefault.AUTOMATIC_TAGS) {
				generateAutomaticTags(array);
			}
			if (parent == SET.class) {
				new UnorderedElementsChecker(type).check(array);
			} else {
				new OrderedElementsChecker(type).check(array);
			}
		} else {
			if (!elements.isEmpty()) {
				ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
				ex.setMessage(
						"If a class does not extend SET or SEQUENCE directly, it can not define own elements.",
						null, type, null, null);
				throw ex;
			}
			array = getElementTypeList(parent);
		}
		ELEMENTS_MAP.put(type, array);
		return array;
	}

	/**
	 * Returns the array of {@code ElementSpecification} that associated to the
	 * elements contained in the type.
	 * 
	 * @return The array of {@code ElementSpecification}.
	 */
	public ElementSpecification[] getElementTypeList() {
		return getElementTypeList(getClass());
	}

	/**
	 * Returns the ASN.1 data that assigned to the element of this instance.
	 * 
	 * @param element
	 *            The {@code ElementSpecification} associated to the element.
	 * @return The ASN.1 data.
	 */
	public ASN1Type getComponent(ElementSpecification element) {
		return element.retrieve(this);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.StructuredType#set(jp.bitmeister.asn1.type.
	 * NamedTypeSpecification, jp.bitmeister.asn1.type.ASN1Type)
	 */
	@Override
	public void set(NamedTypeSpecification component, ASN1Type data) {
		component.assign(this, data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.StructuredType#set(java.lang.String,
	 * jp.bitmeister.asn1.type.ASN1Type)
	 */
	@Override
	public void set(String elementName, ASN1Type component) {
		for (ElementSpecification e : getElementTypeList()) {
			if (e.identifier().equals(elementName)) {
				e.assign(this, component);
				return;
			}
		}
		ASN1IllegalArgument ex = new ASN1IllegalArgument();
		ex.setMessage("No such element '" + elementName + "' in this type.",
				null, getClass(), null, null);
		throw ex;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.StructuredType#get(java.lang.String)
	 */
	@Override
	public ASN1Type get(String elementName) {
		for (ElementSpecification e : getElementTypeList()) {
			if (e.identifier().equals(elementName)) {
				return e.retrieve(this);
			}
		}
		ASN1IllegalArgument ex = new ASN1IllegalArgument();
		ex.setMessage("No such element '" + elementName + "' in this type.",
				null, getClass(), null, null);
		throw ex;
	}
	
	/* 
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#validate()
	 */
	@Override
	public void validate() {
		for (ElementSpecification e: getElementTypeList()) {
			ASN1Type component = e.retrieve(this);
			if (component != null) {
				component.validate();
			}
			else if (!e.optional() && !e.hasDefault()) {
				ASN1InvalidDataValue ex = new ASN1InvalidDataValue();
				ex.setMessage("Mandatory data value is not present.",
						null, getClass(),
						e.identifier(), this);
				throw ex;
			}
		}
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#clear()
	 */
	@Override
	public void clear() {
		for (ElementSpecification e : getElementTypeList()) {
			e.assign(this, null);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#hasValue()
	 */
	@Override
	public boolean hasValue() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#valueEquals(java.lang.Object)
	 */
	@Override
	public boolean valueEquals(Object other) {
		if (other instanceof ConstructiveType) {
			ConstructiveType comparison = (ConstructiveType) other;
			if (getElementTypeList() == comparison.getElementTypeList()) {
				for (ElementSpecification e : getElementTypeList()) {
					ASN1Type thisComponent = e.retrieve(this);
					ASN1Type compComponent = e.retrieve(comparison);
					if (thisComponent == compComponent) {
						continue;
					}
					if (thisComponent == null || compComponent == null
							|| !thisComponent.equals(compComponent)) {
						return false;
					}
				}
				return true;
			}
		}
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#hashCode()
	 */
	@Override
	public int hashCode() {
		int hash = 0;
		for (ElementSpecification e : getElementTypeList()) {
			ASN1Type element = e.retrieve(this);
			if (element != null) {
				hash += element.hashCode();
			}
		}
		return hash;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#clone()
	 */
	@Override
	public Object clone() {
		ConstructiveType clone = ASN1Type.instantiate(getClass());
		for (ElementSpecification e : getElementTypeList()) {
			ASN1Type element = e.retrieve(this);
			if (element != null) {
				e.assign(clone, (ASN1Type) element.clone());
			}
		}
		return clone;
	}

}
