package bodybuilder.util.jdom;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.jdom.Element;

import bodybuilder.exception.BodyBuilderException;
import bodybuilder.util.Config;

/**
 * JDOMユーティリティ
 */
public class JDOMUtils {

    /**
     * タブ
     */
    private static final String TAB = "  ";

    /////////////////////////////////////////////////////////////////
    // child utility

    /**
     * 子要素を取得する。
     * 
     * @param element XML要素
     * @return 子要素
     */
    public static Element getChild(Element element) {
        List children = element.getChildren();

        // 未定義の場合はnull。
        if (children == null || children.size() < 0) {
            return null;
        }

        // 子要素が多すぎる場合はエラー。
        if (children.size() > 1) {
            throw new BodyBuilderException("too many children as '"
                    + element.getName() + "'.");
        }

        return (Element) children.get(0);
    }

    /**
     * 子要素を取得する。未定義の場合はエラー。
     * 
     * @param element XML要素
     * @return 子要素
     */
    public static Element getRequiredChild(Element element) {
        List children = element.getChildren();

        // 未定義の場合はエラー。
        if (children == null || children.size() < 0) {
            throw new BodyBuilderException("undefined child as '"
                    + element.getName() + "'.");
        }

        return (Element) children.get(0);
    }

    /**
     * 子要素を取得する。未定義の場合はエラー。
     * 
     * @param element XML要素
     * @param name 子要素名
     * @return 子要素
     */
    public static Element getRequiredChild(Element element, String name) {
        Element child = element.getChild(name);

        // 未定義の場合はエラー。
        if (child == null) {
            throw new BodyBuilderException("undefined element '" + name
                    + "' as '" + element.getName() + "'.");
        }

        return child;
    }

    /**
     * 子要素リストを取得する。未定義の場合はエラー。
     * 
     * @param element XML要素
     * @param name 子要素名
     * @return 子要素リスト
     */
    public static List getRequiredChildren(Element element, String name) {
        List children = element.getChildren(name);

        // 未定義の場合はエラー。
        if (children == null || children.size() < 0) {
            throw new BodyBuilderException("undefined elements '" + name
                    + "' as '" + element.getName() + "'.");
        }

        return children;
    }

    /**
     * 子要素があるかどうかを返。
     * 
     * @param element XML要素
     * @param name 子要素名
     * @return 子要素がある場合はtrue
     */
    public static boolean hasChild(Element element, String name) {
        Element child = element.getChild(name);

        return (child != null);
    }

    /////////////////////////////////////////////////////////////////
    // attribute utility

    /**
     * 属性値を取得する。
     * 
     * @param element XML要素
     * @param name 属性名
     * @param defaultValue デフォルト値
     * @return 属性値
     */
    public static String getAttrValue(Element element, String name,
            String defaultValue) {
        String value = element.getAttributeValue(name);

        // 未定義の場合はデフォルト値を返す。
        if (value == null) {
            value = defaultValue;
        }

        return value;
    }

    /**
     * 属性値を取得する。未定義の場合はエラー。
     * 
     * @param element XML要素
     * @param name 属性名
     * @return 属性値
     */
    public static String getRequiredAttrValue(Element element, String name) {
        String value = element.getAttributeValue(name);

        // 未定義の場合はエラー。
        if (value == null) {
            throw new BodyBuilderException("undefined attribute '" + name
                    + "' as '" + element.getName() + "'.");
        }

        return value;
    }

    /**
     * 属性値(ブール値)を取得する。未定義の場合はtrue。
     * 
     * @param element XML要素
     * @param name 属性名
     * @return 属性値
     */
    public static boolean getAttrBoolValue(Element element, String name) {
        return getAttrBoolValue(element, name, true);
    }

    /**
     * 属性値(ブール値)を取得する。
     * 
     * @param element XML要素
     * @param name 属性名
     * @param defaultValue デフォルト値
     * @return 属性値
     */
    public static boolean getAttrBoolValue(Element element, String name,
            boolean defaultValue) {
        String value = element.getAttributeValue(name);

        if ("true".equalsIgnoreCase(value)) {
            return true;
        } else if ("false".equalsIgnoreCase(value)) {
            return false;
        } else {
            return defaultValue;
        }
    }

    /**
     * 要素を文字列に変換する。
     * 
     * @param element 要素
     * @return 文字列
     */
    public static String toString(Element element) {
        List elems = new ArrayList();
        elems.add("<" + element.getName() + ">");

        while (!element.isRootElement()) {
            element = element.getParentElement();
            elems.add("<" + element.getName() + ">");
        }

        Collections.reverse(elems);
        StringBuffer buf = new StringBuffer();
        int size = elems.size();

        for (int i = 0; i < size; i++) {
            for (int j = 0; j < i; j++) {
                buf.append(TAB);
            }

            buf.append(elems.get(i));

            if ((i + 1) >= size) {
                buf.append(" <-");
            }

            buf.append(Config.NL);
        }

        return buf.toString();
    }

}