/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_coverage.c - coverage information display program                     */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "bfd_if.h"
#include "bt.h"
#include <libgen.h>

#define BT_COVERAGE_VER	"0.0.1"
#define	COPYRIGHT	"Copyright (c) Hitachi, Ltd. 2005-2006"

#define	chk_next(argc, i)	if ((argc) <= (i)+1) { usage(); err_exit(); }

struct range_to_name {
	unsigned long	begin;
	unsigned long	end;
	unsigned long	offset;
	char		name[MAX_NAME_LEN];
	char		*dirname;
	char		*basename;

	struct bfd_if	bi;
	node		*pt;
};

/*-----------------------------------------------------------------------------
 *  build range to name structure
 *-----------------------------------------------------------------------------
 */
struct range_to_name **all_r2n;
int r2n_max = 128;

void dump_r2n(void)
{
	int i;
	struct range_to_name *r2n;

	if (!all_r2n)
		return;
	for (i = 0; i < r2n_max; i++) {
		r2n = all_r2n[i];
		if (r2n) {
			printf("0x%08lx:0x%08lx\t%s\n",
			       r2n->begin, r2n->end, r2n->name);
		}
	}
}

void free_r2n(void)
{
	int i;
	struct range_to_name *r2n;

	if (!all_r2n)
		return;
	for (i = 0; i < r2n_max; i++) {
		r2n = all_r2n[i];
		if (r2n) {
			//close_mmapfile(r2n->fd, r2n->p_map, r2n->mapped_size);
			//free(r2n->fnames);
			free(r2n);
		}
	}
	free(all_r2n);
}

struct range_to_name* addr_to_r2n(unsigned long addr)
{
	int i;
	struct range_to_name *r2n;

	for (i = 0; i < r2n_max; i++) {
		r2n = all_r2n[i];
		if (r2n && addr >= r2n->begin && addr <= r2n->end)
			return r2n;
	}
	return NULL;
}

int prepare_obj_file(struct range_to_name *r2n, char *kallsyms)
{
	char path[MAX_LINE_LEN];

	snprintf(path, MAX_LINE_LEN, "%s/%s", r2n->dirname, r2n->basename);
	printf("checking %s...\n", path);
	if (init_bfd_if(&r2n->bi, strdup(path), kallsyms) < 0)
		return -1;
	if (!r2n->begin && !r2n->end) {
		struct addr_range *range = get_pid_addr_range(ALL_PID);
		bfd_vma min, max;
		struct bt_record rec;

		get_min_max_addr(&r2n->bi, &min, &max);
		rec.from = min;
		rec.to = max;
		if (is_addr_match(&rec, range)) {
			r2n->begin = min;
			r2n->end = max;
		} else
			return 1;
	} else {
		r2n->offset = get_offset_addr(&r2n->bi, r2n->begin);
		/* for DEBUG*/
		printf("OFFSET: ");
		printf_bfd_vma(r2n->offset);
		printf("\n");
		/**/
	}
	return chk_path_tree(&r2n->bi, &r2n->pt, r2n->offset);
}

int add_range_to_name(unsigned long begin, unsigned long end, char *name,
		      char *kallsyms, int is_module)
{
	int rc, i, step = 128;
	struct range_to_name *r2n;

	r2n = malloc(sizeof(struct range_to_name));
	if (!r2n) {
		fprintf(stderr, "malloc failed.(%s)\n", strerror(errno));
		return -1;
	}
	memset(r2n, 0, sizeof(struct range_to_name));
	r2n->begin = begin;
	r2n->end = end;
	snprintf(r2n->name, MAX_NAME_LEN, "%s", name);
	r2n->basename = basename(r2n->name);
	r2n->dirname = dirname(r2n->name);

	for (i = 0; i < r2n_max && all_r2n[i]; i++);
	if (i >= r2n_max) {
		all_r2n = realloc(all_r2n, (r2n_max + step) * sizeof(r2n));
		if (!all_r2n) {
			free(r2n);
			fprintf(stderr, "realloc failed.(%s)\n",
				strerror(errno));
			return -1;
		}
		memset(&all_r2n[i], 0, step * sizeof(r2n));
		i = r2n_max;
		r2n_max += step;
	}
	all_r2n[i] = r2n;
	rc = prepare_obj_file(r2n, kallsyms);
	if (rc < 0)
		return -1;
	else if (rc > 0) {
		free(r2n);
		all_r2n[i] = NULL;
	}
	return 0;
}

#define INC_CNT(p_cnt)						\
	do {							\
		if (*(p_cnt) != 0xffffffff) *(p_cnt) += 1;	\
	} while(0)

#define ADD_CNT(p_cnt, cnt)					\
	do {							\
		*(p_cnt) += (cnt);				\
		if (*(p_cnt) < (cnt))				\
			*(p_cnt) = 0xffffffff;			\
	} while(0)

int inc_tracking_cnt_without_addr(struct branch *b, int is_branch)
{
	if (is_branch)
		INC_CNT(&b->cnt);
	else
		INC_CNT(&b->ft_cnt);
	return 0;
}

int inc_tracking_cnt_with_addr(struct branch *b, bfd_vma addr)
{
	struct unknown *b_uk;
	unsigned long i, step = 8;
	int size = sizeof(struct unknown);

	for (i = 0; i < b->cnt; i++) {
		b_uk = &b->unknowns[i];
		if (b_uk->to == addr) {
			INC_CNT(&b_uk->cnt);
			return 0;
		}
	}
	if (!(b->cnt % step)) {
		if (b->cnt == 0) {
			b->unknowns = calloc(step, size);
			if (!b->unknowns) {
				fprintf(stderr, "calloc failed.(%s)\n",
					strerror(errno));
				return -1;
			}
		} else {
			b->unknowns = realloc(b->unknowns,
					      (b->cnt + step) * size);
			if (!b->unknowns) {
				fprintf(stderr, "realloc failed.(%s)\n",
					strerror(errno));
				return -1;
			}
			memset(&b->unknowns[b->cnt], 0, step * size);
		}
	}
	b_uk = &b->unknowns[b->cnt];
	b_uk->to = addr;
	INC_CNT(&b_uk->cnt);
	b->cnt++;
	return 0;
}

int check_match_branch(struct range_to_name *r2n, struct branch *b,
		       struct bt_record *rec, int *found)
{
	int rc = 0;

	if (b->from == rec->from) {
		if (b->to == UNKNOWN_BADDR) {
			rc = inc_tracking_cnt_with_addr(b, rec->to);
			goto FOUND;
		} else if (b->to == rec->to) {
			inc_tracking_cnt_without_addr(b, 1);
			goto FOUND;
		}
	} else if (b->from > rec->from)
		goto FOUND;
	if (found)
		*found = 0;
	return rc;
FOUND:
	if (found)
		*found = 1;
	return rc;
}

int proc_each_record(struct bt_record *rec, struct addr_range *range,
		     unsigned long *last)
{
	int rc = 0, found;
	struct range_to_name *r2n;
	struct path *p_last, *p_from;
	struct branch *b;

	if (is_pid_record(rec) || !is_addr_match(rec, range))
		goto EXIT;
	r2n = addr_to_r2n(rec->from);
	if (!r2n)
		goto EXIT;
	p_from = find_path_from_addr(r2n->pt, rec->from);
	if (!p_from)
		goto EXIT;
	p_last = find_path_from_addr(r2n->pt, *last);

	/*
	printf("DBG:");
	printf_bfd_vma(*last);
	printf("->");
	printf_bfd_vma(rec->from);
	printf("->");
	printf_bfd_vma(rec->to);
	printf(" ");
	printf_path(p_from);
	*/
	for (b = p_from->branches; b; b = b->next) {
		if (b->from >= rec->from)
			break;
		/* check fallthrough */
		if (b->from >= *last && p_last == p_from) {
			inc_tracking_cnt_without_addr(b, 0);
			/*
			printf("BR: ");
			if (b) {
				printf_bfd_vma(b->from);
				printf("->");
				printf_bfd_vma(b->to);
			}
			printf("\n");
			*/
		}
	}
	/* check branch */
	/*
	printf("BR:%p ", p_from->branches);
	if (b) {
		printf_bfd_vma(b->from);
		printf("->");
		printf_bfd_vma(b->to);
	}
	printf("\n");
	*/
	for (; b; b = b->next) {
		rc = check_match_branch(r2n, b, rec, &found);
		if (rc < 0 || found)
			goto EXIT;
	}
	if (p_from->call)
		rc = check_match_branch(r2n, p_from->call, rec, NULL);
EXIT:
	*last = rec->to;
	return rc;
}

void printf_funcname(struct bfd_if *bi, bfd_vma addr)
{
	int rc;
	size_t offset;
	const char *func_name;

	if (addr == UNKNOWN_BADDR) {
		printf("----------");
		return;
	}
	rc = addr_to_func_name_and_offset(bi, addr, &func_name, &offset);
	if (rc >= 0) {
		printf_func_name(func_name, offset);
	} else
		printf_bfd_vma(addr);
}

void printf_srcname_and_lno(struct bfd_if *bi, bfd_vma addr)
{
	int rc, lno;
	const char *src_name, *func_name;

	if (addr == UNKNOWN_BADDR) {
		printf("----------");
		return;
	}
	rc = get_source_info(bi, addr, &src_name, &func_name, &lno);
	if (rc >= 0 && src_name)
		printf("%s,%d", src_name, lno);
	else
		printf_bfd_vma(addr);
		//printf_funcname(bi, addr);
}

/*-----------------------------------------------------------------------------
 *  display function coverage
 *-----------------------------------------------------------------------------
 */
struct func_chk {
	bfd_vma		addr;
	unsigned long	cnt;
};

struct fc_r2n_pack {
	node			*fc_list;
	struct range_to_name	*r2n;
};

static int f_cmp_fc(void *__target, void *__dt)
{
	bfd_vma addr = *((bfd_vma*)__target);
	struct func_chk *fc = (struct func_chk*)__dt;

	if (addr < fc->addr)
		return -1;
	if (addr > fc->addr)
		return 1;
	return 0;
}

int __check_func(node **fc_list, bfd_vma addr, unsigned long cnt)
{
	struct func_chk *fc;

	fc = search_tree(&addr, *fc_list, f_cmp_fc);
	if (fc) {
		ADD_CNT(&fc->cnt, cnt);
		return 0;
	}
	fc = malloc(sizeof(struct func_chk));
	if (!fc) {
		fprintf(stderr, "malloc failed.(%s)\n", strerror(errno));
		return -1;
	}
	fc->addr = addr;
	fc->cnt = cnt;

	*fc_list = insert_tree(fc, *fc_list, f_cmp_fc);
	if (!(*fc_list))
		return -1;
	return 0;
}

int check_func(struct range_to_name *r2n, node **fc_list, struct branch *b)
{
	unsigned long i;
	struct unknown *b_uk;

	if (b->to == UNKNOWN_BADDR) {
		for (i = 0; i < b->cnt; i++) {
			b_uk = &b->unknowns[i];
			if (__check_func(fc_list, b_uk->to, b_uk->cnt) < 0)
				return -1;
		}
	} else if (__check_func(fc_list, b->to, b->cnt) < 0)
			return -1;
	return 0;
}

int f_check_func(void *__dt, void *user_data)
{
	struct path *p = (struct path*)__dt;
	struct fc_r2n_pack *pack = (struct fc_r2n_pack*)user_data;

	if (p->call) {
		if (check_func(pack->r2n, &pack->fc_list, p->call) < 0)
			return -1;
	}
	return 0;
}

int chk_all_func_syms(node **fc_list, struct range_to_name *r2n)
{
	unsigned long i;
	struct bfd_if *bi = &r2n->bi;
	asymbol *sym;

	for (i = 0; i < bi->n_fsyms; i++) {
		sym = bi->p_fsyms[i];
		if (sym->flags & BSF_FUNCTION) {
			if (__check_func(fc_list,
					 bfd_asymbol_value(sym) + r2n->offset,
					 0) < 0)
				return -1;
		}
	}
	return 0;
}

static void f_fc_free(void *__dt)
{
	struct func_chk *fc = (struct func_chk*)__dt;

	free(fc);
}

int chk_func_coverage(node **fc_list)
{
	unsigned long i;
	struct range_to_name *r2n;
	struct fc_r2n_pack pack;

	pack.fc_list = NULL;
	for (i = 0; i < r2n_max; i++) {
		r2n = all_r2n[i];
		if (r2n) {
			pack.r2n = r2n;
			if (for_each_node(r2n->pt, f_check_func, &pack) < 0)
				goto ERR_EXIT;
			if (chk_all_func_syms(&pack.fc_list, r2n) < 0)
				goto ERR_EXIT;
		}
	}
	*fc_list = pack.fc_list;
	return 0;

ERR_EXIT:
	free_tree(pack.fc_list, f_fc_free);
	return -1;
}

struct call_cnt_pack {
	unsigned long		called;
	unsigned long		all;
	struct range_to_name	*r2n;
};

static int f_cnt_each_fc(void *__dt, void *user_data)
{
	struct call_cnt_pack *pack = (struct call_cnt_pack*)user_data;
	struct range_to_name *r2n = pack->r2n;
	struct func_chk *cc = (struct func_chk*)__dt;

	if (cc->addr >= r2n->begin && cc->addr <= r2n->end) {
		pack->all++;
		if (cc->cnt)
			pack->called++;
	} else if (cc->addr > r2n->end)
		return 1;	/* break */
	return 0;
}

static int f_dump_each_fc(void *__dt, void *user_data)
{
	struct range_to_name *r2n = (struct range_to_name*)user_data;
	struct func_chk *cc = (struct func_chk*)__dt;

	if (cc->addr >= r2n->begin && cc->addr <= r2n->end) {
		printf(cc->cnt ? "(OK) " : "(NT) ");
		printf_funcname(&r2n->bi, cc->addr - r2n->offset);
		if (cc->cnt)
			printf("\t(%ld)\n", cc->cnt);
		else
			printf("\n");
	} else if (cc->addr > r2n->end)
		return 1;	/* break */
	return 0;
}

void dump_func_coverage(struct range_to_name *r2n, node *cc_list)
{
	struct call_cnt_pack pack = { 0, 0, r2n };

	for_each_node(cc_list, f_cnt_each_fc, &pack);
	printf("------ function coverage (%ld/%ld) ------\n",
	       pack.called, pack.all);
	for_each_node(cc_list, f_dump_each_fc, r2n);
}

/*-----------------------------------------------------------------------------
 *  display branch coverage
 *-----------------------------------------------------------------------------
 */
void dump_one_branch_coverage(struct range_to_name *r2n, struct branch *b)
{
	int i;
	struct unknown *b_uk;
	struct bfd_if *bi = &r2n->bi;

	if (b->to == UNKNOWN_BADDR) {
		for (i = 0; i < b->cnt; i++) {
			b_uk = &b->unknowns[i];
			printf("(UK) ");
			printf_srcname_and_lno(bi, b->from - r2n->offset);
			printf(" [%ld/-] ", b_uk->cnt);
			printf_srcname_and_lno(bi, b_uk->to - r2n->offset);
			printf(":----------");
			printf("\n");
		}
		printf("(UK) ");
		printf_srcname_and_lno(bi, b->from - r2n->offset);
		printf(" [-/%ld] ", b->ft_cnt);
		printf("----------:");
		printf_srcname_and_lno(bi, b->end - r2n->offset);
		printf("\n");
	} else {
		if (b->cnt != 0 && b->ft_cnt != 0)
			printf("(OK) ");
		else if (b->cnt == 0 && b->ft_cnt == 0)
			printf("(NT) ");
		else
			printf("(HT) ");
		printf_srcname_and_lno(bi, b->from - r2n->offset);
		printf(" [%ld/%ld] ", b->cnt, b->ft_cnt);
		printf_srcname_and_lno(bi, b->to - r2n->offset);
		printf(":");
		printf_srcname_and_lno(bi, b->end - r2n->offset);
		printf("\n");
	}
}

struct branch_cnts {
	unsigned long ok;
	unsigned long uk;
	unsigned long nt;
	unsigned long ht;
};

int f_chk_branch_coverage(void *__dt, void *user_data)
{
	struct path *p = (struct path*)__dt;
	struct branch_cnts *bcs = (struct branch_cnts*)user_data;
	struct branch *b;

	for (b = p->branches; b; b = b->next) {
		if (b->to == UNKNOWN_BADDR)
			bcs->uk += b->cnt;
		else if (b->cnt != 0 && b->ft_cnt != 0)
			bcs->ok++;
		else if (b->cnt == 0 && b->ft_cnt == 0)
			bcs->nt++;
		else
			bcs->ht++;
	}
	return 0;
}

int f_dump_branch_coverage(void *__dt, void *user_data)
{
	struct path *p = (struct path*)__dt;
	struct range_to_name *r2n = (struct range_to_name*)user_data;
	struct branch *b;

	for (b = p->branches; b; b = b->next)
		dump_one_branch_coverage(r2n, b);
	return 0;
}

void dump_branch_coverage(struct range_to_name *r2n)
{
	struct branch_cnts bcs = {0,0,0,0};

	for_each_node(r2n->pt, f_chk_branch_coverage, &bcs);
	printf("------ branch coverage (OK:%ld,UK:%ld,HT:%ld,NT:%ld / %ld)\n",
	       bcs.ok, bcs.uk, bcs.ht, bcs.nt,
	       bcs.ok + bcs.uk + bcs.ht + bcs.nt);
	for_each_node(r2n->pt, f_dump_branch_coverage, r2n);
}

/*-----------------------------------------------------------------------------
 *  display coverage
 *-----------------------------------------------------------------------------
 */
void dump_coverage(void)
{
	int i;
	struct range_to_name *r2n;
	node *cc_list;

	if (!all_r2n)
		return;

	if (chk_func_coverage(&cc_list) < 0)
		return;

	for (i = 0; i < r2n_max; i++) {
		r2n = all_r2n[i];
		if (r2n) {
			printf("====== %s coverage ======\n", r2n->basename);
			dump_func_coverage(r2n, cc_list);
			dump_branch_coverage(r2n);
		}
	}
}

int proc_logfile(char *logfile)
{
	int fd, rc = -1;
	//struct stat st;
	size_t size;
	struct bt_record *rec, *p_max;
	char *dir, *fname;
	struct addr_range *range;
	unsigned long last_addr;

	if (open_mmapfile(logfile, &fd, (void**)&rec, &size) < 0)
		goto EXIT;

	fname = basename(logfile);
	dir = dirname(logfile);
	if (is_pid_record(rec)) {
		if (parse_maps(add_range_to_name, dir, fname) < 0)
			goto EXIT;
	} else {
		if (parse_modules(add_range_to_name, dir) < 0)
			goto EXIT;
	}
	printf("start\n");
	range = get_pid_addr_range(ALL_PID);
	last_addr = UNKNOWN_BADDR;
	for (p_max = (struct bt_record*)((char*)rec + size);
	     rec < p_max; rec++) {
		/*
		if (!((p_max - rec) % 1000))
			printf("%d records left...\n", p_max - rec);
			*/
		if (proc_each_record(rec, range, &last_addr) < 0) {
			/* for DEBUG */
			printf("MUMU: %08lx->%08lx\n", rec->from, rec->to);
			goto EXIT;
		}
	}
	rc = 0;
	dump_coverage();
EXIT:
	close_mmapfile(fd, rec, size);
	return rc;
}

void err_exit(void)
{
	exit(1);
}

void usage(void)
{
	fprintf(stderr, "bt_coverage %s\n", BT_COVERAGE_VER);
	fprintf(stderr, "    %s\n\n", COPYRIGHT);
	fprintf(stderr, "bt_coverage [-a top:end [...]] [-d top:end [...]] -f logfile\n");
	fprintf(stderr, "  -a: add address range\n");
	fprintf(stderr, "  -d: delete address range\n");
	fprintf(stderr, "  -f: logfile\n");
}

int main(int argc, char *argv[])
{
	int i;
	char *logfile = NULL;
	unsigned long begin, end;

	if (alloc_pid_range(ALL_PID) < 0)
		err_exit();
	for (i = 1; i < argc;) {
		if (strcmp(argv[i], "-a") == 0) {
			chk_next(argc, i);
			i++;
			while (i < argc && argv[i][0] != '-') {
				if (!range2ulongs(argv[i], &begin, &end))
					err_exit();
				add_range(ALL_PID, begin, end);
				i++;
			}
		} else if (strcmp(argv[i], "-d") == 0) {
			chk_next(argc, i);
			i++;
			while (i < argc && argv[i][0] != '-') {
				if (!range2ulongs(argv[i], &begin, &end))
					err_exit();
				printf("del_range\n");
				del_range(ALL_PID, begin, end);
				i++;
			}
		} else if (strcmp(argv[i], "-f") == 0) {
			chk_next(argc, i);
			i++;
			logfile = argv[i];
			i++;
		} else {
			usage();
			err_exit();
		}
	}
	if (!logfile) {
		usage();
		err_exit();
	}
	all_r2n = calloc(r2n_max, sizeof(struct range_to_name*));
	if (!all_r2n) {
		fprintf(stderr, "calloc failed.(%s)\n", strerror(errno));
		err_exit();
	}
	if (proc_logfile(logfile) < 0)
		err_exit();
	exit(0);
}
