/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_relfs.c - branch trace module (relayfs interface)                     */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <linux/version.h>
#include <linux/file.h>
#include <linux/proc_fs.h>
#include "linux/relayfs_fs.h"
#include "bt.h"

static char *relfs_dirname = "bts";
static char *relfs_basename = "cpu";
static struct dentry *dir = NULL;
static struct rchan *channel = NULL;
static size_t dropped;

static DECLARE_WAIT_QUEUE_HEAD(relfs_write_wait);

/* produced/consumed control files */
extern struct proc_dir_entry *proc_bts;
static struct proc_dir_entry *proc_subbuf_sz;
static struct proc_dir_entry *proc_n_subbufs;
static struct proc_dir_entry *proc_dropped;
static struct proc_dir_entry *produced_ctrl[NR_CPUS];
static struct proc_dir_entry *consumed_ctrl[NR_CPUS];

extern size_t subbuf_size;
extern size_t subbuf_num;
extern size_t subbuf_sleep_threshold;
extern int do_not_overwrite;

#ifdef DEBUG
unsigned long wcnt[NR_CPUS];
static void relfs_relay_write(struct rchan *channel, void *p, size_t size)
{
	int cpu = smp_processor_id();
	struct bt_record rec = {0,0,BT_FLAG_DEBUG};
	size_t i;

	for (i = 0; i < size / sizeof(rec); i++) {
		rec.from = wcnt[cpu]++;
		relay_write(channel, &rec, sizeof(rec));
	}
}
#else
#  define relfs_relay_write	relay_write
#endif

static int in_flush;
void relfs_flush(void)
{
	/* Do flush twice.
	 * 1st: Flush left trace data.
	 * 2st: Set size zero trace data. Thus, read process can detect there's
	 *      no trace data.
	 */
	in_flush = 1;
	relay_flush(channel);
	relay_flush(channel);
	in_flush = 0;
}

void write_pid_record(pid_t pid)
{
	struct pid_record rec;
	struct timeval tv;

	do_gettimeofday(&tv);
	rec.pid = pid;
	rec.tv_sec = tv.tv_sec;
	rec.tv_usec = tv.tv_usec | BT_FLAG_PID;

	relfs_relay_write(channel, &rec, sizeof(rec));
}

void write_bt_records(void *p, size_t size)
{
	relfs_relay_write(channel, p, size);
}

static int is_relfs_writeable(void)
{
	unsigned long flags;
	struct rchan_buf *buf;
	size_t ready;
	int rc, cpu;

	local_irq_save(flags);
	cpu = smp_processor_id();
	buf = channel->buf[cpu];
	ready = buf->subbufs_produced - buf->subbufs_consumed;
	rc = (ready < subbuf_sleep_threshold);
	/* for DEBUG
	serial_prints("(%d)is writable: %d(p:%d,c:%d)\n",
		      cpu, rc, buf->subbufs_produced, buf->subbufs_consumed);
		      */
	local_irq_restore(flags);
	return rc;
}

void check_and_wait_relfs_write(void)
{
	wait_event_interruptible(relfs_write_wait, is_relfs_writeable());
}

static int subbuf_start_handler(struct rchan_buf *buf, void *subbuf,
				void *prev_subbuf,
				unsigned int prev_padding)
{
	/* for DEBUG
	if (!do_not_overwrite)
		serial_prints("subbuf_start(cpu:%d, full:%d, p:%d)\n",
			      smp_processor_id(), relay_buf_full(buf),
			      prev_padding);
			      */
	if (do_not_overwrite && !in_flush) {
		size_t ready = buf->subbufs_produced - buf->subbufs_consumed;
		if (ready >= buf->chan->n_subbufs - 1)	/* buffer nearly full */
			return 0;
	}
	/* When do-not-overwrite mode, log data is never dropped. */
	if (!do_not_overwrite && relay_buf_full(buf))
		dropped++;
	if (prev_subbuf)
		*(unsigned*)prev_subbuf = prev_padding;
	subbuf_start_reserve(buf, sizeof(unsigned int));
	return 1;
}

static struct rchan_callbacks relayfs_callbacks =
	{ .subbuf_start = subbuf_start_handler };

static int subbuf_sz_read(char *buffer, char **start, off_t off,
			  int count, int *eof, void *data)
{
	return sprintf(buffer, "%d\n", subbuf_size);
}

static int n_subbufs_read(char *buffer, char **start, off_t off,
			  int count, int *eof, void *data)
{
	return sprintf(buffer, "%d\n", subbuf_num);
}

static int dropped_read(char *buffer, char **start, off_t off,
			int count, int *eof, void *data)
{
	return sprintf(buffer, "%d\n", dropped);
}

static int produced_read(char *buffer, char **start, off_t off,
			 int count, int *eof, void *data)
{
	struct rchan_buf *rbuf = (struct rchan_buf*)data;
	int len = sizeof(rbuf->subbufs_produced);

	memcpy(buffer, &rbuf->subbufs_produced, len);
	return len;
}

static int consumed_read(char *buffer, char **start, off_t off,
			 int count, int *eof, void *data)
{
	struct rchan_buf *rbuf = (struct rchan_buf*)data;
	int len = sizeof(rbuf->subbufs_consumed);

	memcpy(buffer, &rbuf->subbufs_consumed, len);
	return len;
}

static int consumed_write(struct file *file, const char *user_buf,
			  unsigned long count, void *data)
{
	struct rchan_buf *rbuf = (struct rchan_buf*)data;
	size_t consumed;

	if (copy_from_user(&consumed, user_buf, sizeof(consumed))) {
		printk("%s: copy_from_user failed.\n", MOD_NAME);
		return -EFAULT;
	}
		
	relay_subbufs_consumed(rbuf->chan, rbuf->cpu, consumed);

	if (is_relfs_writeable())
		wake_up_interruptible(&relfs_write_wait);

	return count;
}

int relfs_init(void)
{
	int i, max=16;
	char buf[max];

	dir = relayfs_create_dir(relfs_dirname, NULL);
	if (dir == NULL) {
		printk("%s: cannot create relayfs directory\n", MOD_NAME);
		return -ENOMEM;
	}
	channel = relay_open(relfs_basename, dir, subbuf_size, subbuf_num,
			     &relayfs_callbacks);
	if (channel == NULL) {
		printk("%s: cannot open relayfs channel\n", MOD_NAME);
		return -ENOMEM;
	}
	proc_subbuf_sz = create_proc_entry("subbuf_size", 0400, proc_bts);
	proc_subbuf_sz->read_proc = subbuf_sz_read;
	proc_n_subbufs = create_proc_entry("n_subbufs", 0400, proc_bts);
	proc_n_subbufs->read_proc = n_subbufs_read;
	proc_dropped = create_proc_entry("dropped", 0400, proc_bts);
	proc_dropped->read_proc = dropped_read;

	for_each_online_cpu(i) {
		snprintf(buf, max, "%s%d.produced", relfs_basename, i);
		produced_ctrl[i] = create_proc_entry(buf, 0600, proc_bts);
		if (!produced_ctrl[i]) {
			printk("%s: create_proc_entry failed.\n", MOD_NAME);
			return -ENOMEM;
		}
		produced_ctrl[i]->data = channel->buf[i];
		produced_ctrl[i]->read_proc = produced_read;

		snprintf(buf, max, "%s%d.consumed", relfs_basename, i);
		consumed_ctrl[i] = create_proc_entry(buf, 0600, proc_bts);
		if (!consumed_ctrl[i]) {
			printk("%s: create_proc_entry failed.\n", MOD_NAME);
			return -ENOMEM;
		}
		consumed_ctrl[i]->data = channel->buf[i];
		consumed_ctrl[i]->read_proc = consumed_read;
		consumed_ctrl[i]->write_proc = consumed_write;
	}
	return 0;
}

static void chk_procs_using_relfs(struct dentry *d,
				  pid_t **pp_pid, pid_t *p_max)
{
	struct task_struct *p;
	struct files_struct *files;
	struct file *file = NULL;
	unsigned int i;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,14)
	struct fdtable *fdt;
#endif

	write_lock_irq(&tasklist_lock);
	p = &init_task;
	do {
		p = prev_task(p);
		files = p->files;
		if (!files)
			continue;
		spin_lock(&files->file_lock);
		//serial_prints("chk-pid(%d)\n", p->pid);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,14)
		fdt = files_fdtable(files);
		for (i = 0; i < fdt->max_fds; i++) {
			file = fdt->fd[i];
#else
		for (i = 0; i < files->max_fds; i++) {
			file = files->fd[i];
#endif
			if (file && file->f_dentry == d) {
				*(*pp_pid)++ = p->pid;
				if (*pp_pid >= p_max) {
					spin_unlock(&files->file_lock);
					goto EXIT;
				}
				break;
			}
		}
		spin_unlock(&files->file_lock);
	} while (p != &init_task);
EXIT:
	write_unlock_irq(&tasklist_lock);
}

static void kill_procs_using_relfs(void)
{
	struct dentry *child;
	int max = 16;
	pid_t pid[max], *p, *p_max;

	if (!dir)
		return;
	do {
		p = pid;
		p_max = p + max;
		spin_lock(&dcache_lock);
		list_for_each_entry(child, &dir->d_subdirs, d_child) {
			chk_procs_using_relfs(child, &p, p_max);
			if (p >= p_max)
				break;
		}
		spin_unlock(&dcache_lock);
		p_max = p;
		for (p = pid; p < p_max; p++) {
			//serial_prints("  kill(%d)\n", *p);
			kill_proc(*p, SIGKILL, 1);
		}
		cpu_relax();
	} while (p != pid);
}

void relfs_cleanup(void)
{
	int i;

	if (proc_subbuf_sz)
		remove_proc_entry(proc_subbuf_sz->name, proc_bts);
	if (proc_n_subbufs)
		remove_proc_entry(proc_n_subbufs->name, proc_bts);
	if (proc_dropped)
		remove_proc_entry(proc_dropped->name, proc_bts);

	kill_procs_using_relfs();

	for (i = 0; i < NR_CPUS; i++) {
		if (produced_ctrl[i])
			remove_proc_entry(produced_ctrl[i]->name, proc_bts);
		if (consumed_ctrl[i])
			remove_proc_entry(consumed_ctrl[i]->name, proc_bts);
	}
	if (channel)
		relay_close(channel);
	if (dir)
		relayfs_remove_dir(dir);
}
