/*
 * XMLFileWriterImpl.java
 * Copyright (C) 2008 Cyber Beans Corporation. All rights reserved.
 */
package jp.co.cybec.cb3.accessor.fileaccess.xml;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Stack;

import jp.co.cybec.cb3.accessor.AccessorsLifeCycle;
import jp.co.cybec.cb3.exception.AccessorAlreadyOpenedException;
import jp.co.cybec.cb3.exception.AccessorNotOpenException;
import jp.co.cybec.cb3.exception.AccessorRuntimeException;
import jp.co.cybec.cb3.exception.AccessorUnknownException;
import jp.co.cybec.cb3.exception.BadXmlGrammerException;
import jp.co.cybec.cb3.exception.ConfigFileDefinitionException;
import jp.co.cybec.cb3.exception.ConfigFileIOException;
import jp.co.cybec.cb3.exception.ConfigFileNotFoundException;
import jp.co.cybec.cb3.exception.ConfigFileParseException;
import jp.co.cybec.cb3.exception.DataFileIOException;
import jp.co.cybec.cb3.exception.DataFileNotFoundException;
import jp.co.cybec.cb3.util.ConfigurationLoader;
import jp.co.cybec.cb3.util.logging.ErrorCodeConstant;
import jp.co.cybec.cb3.util.logging.LogMessageConstant;
import jp.co.cybec.cb3.util.logging.LogMessageFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.xml.sax.Attributes;
import org.xml.sax.helpers.AttributesImpl;
import org.xml.sax.helpers.NamespaceSupport;

/**
 * XMLt@C̏݃NXB
 * @author Daisuke Suga
 */
public class XMLFileWriterImpl implements XMLFileWriter, AccessorsLifeCycle {

	/** O */
	private static final Log LOG = LogFactory.getLog(XMLFileWriterImpl.class);

	/** bZ[W */
	private static final LogMessageFactory MESSAGE = new LogMessageFactory(
			LogMessageConstant.CB3_MESSAGE_FILENAME);

	/**
	 * XML`t@Cdigester[̃t@CpXB
	 */
	private static final String XML_FILE_WRITER_RULE_FILE_NAME = "XMLFileWriterRule.xml";

	/**
	 * ftHg̉sR[hB
	 */
	private static final String DEFAULT_LINE_SEPARATOR = System.getProperty("line.separator");

	/**
	 * ftHg̃GR[hB
	 */
	private static final String DEFAULT_ENCODE = "UTF-8";

	/**
	 * ݂̏ԁ|ȂB
	 */
	private static final Object SEEN_NOTHING = new Object();

	/**
	 * ݂̏ԁ|vfB
	 */
	private static final Object SEEN_ELEMENT = new Object();

	/**
	 * ݂̏ԁ|f[^B
	 */
	private static final Object SEEN_DATA = new Object();

	/**
	 * ANZbT̃L[B
	 */
	private String key = null;

	/**
	 * XML`t@CB
	 */
	private XMLFileWriterConfig config = null;

	/**
	 * t@C̃I[vԂێtOB
	 */
	private boolean isOpen = false;

	/**
	 * XMLt@COutputStreamWriterB
	 */
	private OutputStreamWriter writer = null;

	/**
	 * XMLt@CBufferedWriterB
	 */
	private Object state = SEEN_NOTHING;

	/**
	 * ݂̏ԂێX^bNB
	 */
	private Stack<Object> stateStack = new Stack<Object>();

	/**
	 * Cfg̋󔒕B
	 */
	private int indentSpace = 2;

	/**
	 * vf̐[B
	 */
	private int depth = 0;

	/**
	 * OԂ̐ړێMapB
	 */
	private Map<String, String> prefixMap = null;

	/**
	 * OԂێMapB
	 */
	private Map<String, Boolean> namespaceMap = null;

	/**
	 * ݒ肳ꂽOԂƐړێMapB
	 */
	private Map<String, String> declaredNamespaceMap = null;

	/**
	 * NamespaceSupportB
	 */
	private NamespaceSupport nsSupport = null;

	/**
	 * {@inheritDoc}
	 */
	public void startElement(XMLElement element) throws AccessorNotOpenException,
			DataFileIOException {

		// t@CI[vĂȂꍇAOX[܂B
		if (!isOpen) {
			AccessorNotOpenException ex = new AccessorNotOpenException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_019,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_001, key));
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}

		// XML̗vf̊Jn^Oo͂܂B
		try {
			startElement(element.getURI(), element.getName(), element.getQName(),
					getAttributes(element));
		} catch (AccessorRuntimeException e) {
			throw e;
		} catch (Exception e) {
			AccessorUnknownException ex = new AccessorUnknownException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_020,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_006, key), e);
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void endElement(XMLElement element) throws AccessorNotOpenException,
			BadXmlGrammerException, DataFileIOException {

		// t@CI[vĂȂꍇAOX[܂B
		if (!isOpen) {
			AccessorNotOpenException ex = new AccessorNotOpenException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_021,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_001, key));
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}

		// XML̗vf̏I^Oo͂܂B
		try {
			endElement(element.getURI(), element.getName(), element.getQName());
		} catch (AccessorRuntimeException e) {
			throw e;
		} catch (Exception e) {
			AccessorUnknownException ex = new AccessorUnknownException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_022,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_006, key), e);
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void dataElement(XMLElement element) throws AccessorNotOpenException,
			DataFileIOException {

		// t@CI[vĂȂꍇAOX[܂B
		if (!isOpen) {
			AccessorNotOpenException ex = new AccessorNotOpenException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_023,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_001, key));
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}

		// XML̗vf݂܂B
		try {
			dataElement(element.getURI(), element.getName(), element.getQName(),
					getAttributes(element), element.getString());
		} catch (AccessorRuntimeException e) {
			throw e;
		} catch (Exception e) {
			AccessorUnknownException ex = new AccessorUnknownException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_024,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_006, key), e);
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void characters(XMLElement element) throws AccessorNotOpenException, DataFileIOException {

		// t@CI[vĂȂꍇAOX[܂B
		if (!isOpen) {
			AccessorNotOpenException ex = new AccessorNotOpenException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_025,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_001, key));
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}

		// XML̗vf̒l݂܂B
		try {
			characters(element.getString());
		} catch (AccessorRuntimeException e) {
			throw e;
		} catch (Exception e) {
			AccessorUnknownException ex = new AccessorUnknownException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_026,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_006, key), e);
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void open() throws AccessorAlreadyOpenedException, DataFileNotFoundException,
			DataFileIOException {

		// łopenĂꍇAOX[܂B
		if (isOpen) {
			AccessorAlreadyOpenedException ex = new AccessorAlreadyOpenedException(MESSAGE
					.getErrorMessage(ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_027,
							LogMessageConstant.CB3_ACCESSOR_FILEACCESS_002, key));
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}

		// t@CpXݒ肳ĂȂꍇ͗OX[܂B
		if (config.getFilepath() == null || config.getFilepath().length() == 0) {
			DataFileNotFoundException ex = new DataFileNotFoundException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_042,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_018));
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}

		try {

			// XMLt@CI[v܂B
			File file = new File(config.getFilepath());
			writer = new OutputStreamWriter(new FileOutputStream(file), config.getEncode());

			// ϐ̏
			init();

			// XMLhLgJn܂B
			startDocument();

			// t@CI[vԂɂ܂B
			isOpen = true;

		} catch (FileNotFoundException e) {
			DataFileNotFoundException ex = new DataFileNotFoundException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_028,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_003, config.getFilepath()), e);
			LOG.info(ex.getMessage(), ex);
			throw ex;
		} catch (IOException e) {
			DataFileIOException ex = new DataFileIOException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_029,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_004, config.getFilepath()), e);
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}

	}

	/**
	 * {@inheritDoc}
	 */
	public void open(String filepath) throws AccessorAlreadyOpenedException, DataFileNotFoundException,
			DataFileIOException {
		config.setFilepath(filepath);
		open();
	}

	/**
	 * {@inheritDoc}
	 */
	public void close() throws DataFileIOException {

		if (isOpen) {
			try {

				// XMLhLgI܂B
				endDocument();

				// XMLt@CN[Y܂B
				writer.flush();
				writer.close();
				writer = null;
				isOpen = false;

			} catch (IOException e) {
				DataFileIOException ex = new DataFileIOException(MESSAGE.getErrorMessage(
						ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_030,
						LogMessageConstant.CB3_ACCESSOR_FILEACCESS_004, config.getFilepath()), e);
				LOG.info(ex.getMessage(), ex);
				throw ex;
			}
		}

	}

	/**
	 * {@inheritDoc}
	 *
	 * @throws ConfigFileNotFoundException ݒt@C݂Ȃꍇɔ܂B
	 * @throws ConfigFileDefinitionException ݒt@C̊ԈႢɂ蔭܂B
	 * @throws ConfigFileIOException ݒt@CǂݍݎIOG[ɂ蔭܂B
	 * @throws ConfigFileParseException ݒt@CXMLt@C͂sɔ܂B
	 */
	public void initialize(String key) throws ConfigFileNotFoundException,
			ConfigFileDefinitionException, ConfigFileIOException, ConfigFileParseException {

		// ANZbT̃L[ێ܂B
		this.key = key;

		// keyiݒt@Cjݒt@Cǂݍ
		config = (XMLFileWriterConfig) ConfigurationLoader
				.load(XML_FILE_WRITER_RULE_FILE_NAME, key);

		// ݒt@CɉsR[hݒ肳ĂȂꍇ̓ftHg̒lݒ肵܂B
		if (config.getLineSeparator() == null || config.getLineSeparator().length() == 0) {
			config.setLineSeparator(DEFAULT_LINE_SEPARATOR);
		} else {
			config.setLineSeparator(config.getLineSeparator().replaceAll("\\\\r", "\r"));
			config.setLineSeparator(config.getLineSeparator().replaceAll("\\\\n", "\n"));
		}

		// ݒt@CɃGR[h񂪐ݒ肳ĂȂꍇ̓ftHg̒lݒ肵܂B
		if (config.getEncode() == null || config.getEncode().length() == 0) {
			config.setEncode(DEFAULT_ENCODE);
		}

		// Cfg̋󔒕ݒ肵܂B
		if (config.getIndentSpace() != null && config.getIndentSpace().length() > 0) {
			indentSpace = Integer.parseInt(config.getIndentSpace());
		}

	}

	/**
	 * {@inheritDoc}
	 *
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	public void release() throws DataFileIOException {
		close();
	}

	/**
	 * OԂ̒`ǉ܂B
	 *
	 * @param uri OԂURI
	 * @param prefix ړ
	 * @throws AccessorNotOpenException {@link XMLFileWriterImpl#open()}Ă΂ĂȂꍇɔ܂B
	 */
	public void addNamespace(String uri, String prefix) throws AccessorNotOpenException {

		// t@CI[vĂȂꍇAOX[܂B
		if (!isOpen) {
			AccessorNotOpenException ex = new AccessorNotOpenException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_031,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_001, key));
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}

		prefixMap.put(uri, prefix);
		addNamespace(uri);
	}

	/**
	 * OԂ̒`ǉ܂B
	 *
	 * @param uri OԂURI
	 * @throws AccessorNotOpenException {@link XMLFileWriterImpl#open()}Ă΂ĂȂꍇɔ܂B
	 */
	public void addNamespace(String uri) throws AccessorNotOpenException {

		// t@CI[vĂȂꍇAOX[܂B
		if (!isOpen) {
			AccessorNotOpenException ex = new AccessorNotOpenException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_032,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_001, key));
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}

		namespaceMap.put(uri, Boolean.TRUE);
	}

	/**
	 * XMLhLgJn܂B
	 *
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void startDocument() throws DataFileIOException {
		write("<?xml version=\"1.0\" encoding=\"");
		write(config.getEncode());
		write("\"?>");
		write(config.getLineSeparator());
		write(config.getLineSeparator());
	}

	/**
	 * XMLhLgI܂B
	 *
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void endDocument() throws DataFileIOException {
		write(config.getLineSeparator());
	}

	/**
	 * XML̗vf̊Jn^Oo͂܂B
	 *
	 * @param uri OԂURI
	 * @param localName vf̖O
	 * @param qName vfQName
	 * @param atts vf̑
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void startElement(String uri, String localName, String qName, Attributes atts)
			throws DataFileIOException {
		stateStack.push(SEEN_ELEMENT);
		state = SEEN_NOTHING;
		if (depth > 0) {
			characters(config.getLineSeparator());
		}
		doIndent();
		nsSupport.pushContext();
		write('<');
		writeName(uri, localName, qName, true);
		writeAttributes(atts);
		if (depth == 0) {
			declareNamespace();
		}
		writeNSDecls();
		write('>');
		depth++;
	}

	/**
	 * XML̗vf̏I^Oo͂܂B
	 *
	 * @param uri OԂURI
	 * @param localName vf̖O
	 * @param qName vfQName
	 * @throws BadXmlGrammerException XML̕@ԈĂƂɔ܂B
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void endElement(String uri, String localName, String qName)
			throws BadXmlGrammerException, DataFileIOException {

		if (depth == 0) {
			BadXmlGrammerException ex = new BadXmlGrammerException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_033,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_XML_006, key, localName));
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}
		depth--;
		if (state == SEEN_ELEMENT) {
			characters(config.getLineSeparator());
			doIndent();
		}
		write("</");
		writeName(uri, localName, qName, true);
		write('>');
		nsSupport.popContext();
		state = stateStack.pop();

	}

	/**
	 * XML̗vf݂܂B<br>
	 * startElementcharactersAendElements܂B
	 *
	 * @param uri OԂURI
	 * @param localName vf̖O
	 * @param qName vfQName
	 * @param atts vf̑
	 * @param content vf̒l
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void dataElement(String uri, String localName, String qName, Attributes atts,
			String content) throws DataFileIOException {
		startElement(uri, localName, qName, atts);
		characters(content);
		endElement(uri, localName, qName);
	}

	/**
	 * XML̗vf̒l݂܂B
	 *
	 * @param data vf̒l
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void characters(String data) throws DataFileIOException {
		String writeData = data;
		if (writeData == null) {
			writeData = "";
		}
		char ch[] = writeData.toCharArray();
		characters(ch, 0, ch.length);
	}

	/**
	 * XML̗vf̒l݂܂B
	 *
	 * @param ch vf̒l
	 * @param start vf̒l̊Jnʒu
	 * @param len vf̒l̒
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void characters(char[] ch, int start, int len) throws DataFileIOException {
		state = SEEN_DATA;
		writeEsc(ch, start, len, false);
	}

	/**
	 * ϐ܂B
	 */
	private void init() {
		depth = 0;
		state = SEEN_NOTHING;
		stateStack = new Stack<Object>();
		prefixMap = new HashMap<String, String>();
		namespaceMap = new LinkedHashMap<String, Boolean>();
		declaredNamespaceMap = new HashMap<String, String>();
		nsSupport = new NamespaceSupport();
	}

	/**
	 * ׂĂ̖OԂݒ肵܂B
	 */
	private void declareNamespace() {

		for (String uri : namespaceMap.keySet()) {
			doPrefix(uri, null, true);
		}

	}

	/**
	 * OԂɐړݒ肵܂B
	 *
	 * @param uri OԂURI
	 * @param qName vfQName
	 * @param isElement vf̐ݒꍇtrueA̐ݒꍇfalse
	 *
	 * @return ړ
	 */
	private String doPrefix(String uri, String qName, boolean isElement) {

		if (uri == null || uri.length() == 0) {
			return null;
		}

		String prefix = nsSupport.getPrefix(uri);
		if (prefix != null) {
			return prefix;
		}

		prefix = (String) declaredNamespaceMap.get(uri);
		if (prefix != null) {
			return prefix;
		}

		prefix = (String) prefixMap.get(uri);
		if (prefix == null && qName != null && !"".equals(qName)) {
			int i = qName.indexOf(':');
			if (i == -1) {
				return null;
			} else {
				prefix = qName.substring(0, i);
			}
		}
		if (prefix == null) {
			prefix = "";
		}
		nsSupport.declarePrefix(prefix, uri);
		declaredNamespaceMap.put(uri, prefix);
		return prefix;
	}

	/**
	 * PoCg݂܂B
	 *
	 * @param c ޕ
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void write(char c) throws DataFileIOException {
		try {
			writer.write(c);
		} catch (IOException e) {
			DataFileIOException ex = new DataFileIOException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_034,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_004, config.getFilepath()), e);
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}
	}

	/**
	 * ݂܂B
	 *
	 * @param s ޕ
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void write(String s) throws DataFileIOException {
		try {
			writer.write(s);
		} catch (IOException e) {
			DataFileIOException ex = new DataFileIOException(MESSAGE.getErrorMessage(
					ErrorCodeConstant.ACCESSOR_FILEACCESS_XML_035,
					LogMessageConstant.CB3_ACCESSOR_FILEACCESS_004, config.getFilepath()), e);
			LOG.info(ex.getMessage(), ex);
			throw ex;
		}
	}

	/**
	 * ׂĂ݂̑܂B
	 *
	 * @param atts 
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void writeAttributes(Attributes atts) throws DataFileIOException {
		int len = atts.getLength();
		for (int i = 0; i < len; i++) {
			char ch[] = atts.getValue(i).toCharArray();
			write(' ');
			writeName(atts.getURI(i), atts.getLocalName(i), atts.getQName(i), false);
			write("=\"");
			writeEsc(ch, 0, ch.length, true);
			write('"');
		}
	}

	/**
	 * GXP[vď݂܂B
	 *
	 * @param ch ޕ
	 * @param start ޕ̊Jnʒu
	 * @param length ޕ̒
	 * @param isAttVal ޕ񂪑̏ꍇtrueAȊOfalse
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void writeEsc(char ch[], int start, int length, boolean isAttVal)
			throws DataFileIOException {

		for (int i = start; i < start + length; i++) {
			switch (ch[i]) {
			case '&':
				write("&amp;");
				break;
			case '<':
				write("&lt;");
				break;
			case '>':
				write("&gt;");
				break;
			case '\"':
				if (isAttVal) {
					write("&quot;");
				} else {
					write('\"');
				}
				break;
			default:
				write(ch[i]);
			}
		}

	}

	/**
	 * OԂ݂܂B
	 *
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void writeNSDecls() throws DataFileIOException {

		Enumeration<String> prefixes = nsSupport.getDeclaredPrefixes();
		while (prefixes.hasMoreElements()) {
			String prefix = prefixes.nextElement();
			String uri = nsSupport.getURI(prefix);
			char ch[] = uri.toCharArray();
			write(' ');
			if ("".equals(prefix)) {
				write("xmlns=\"");
			} else {
				write("xmlns:");
				write(prefix);
				write("=\"");
			}
			writeEsc(ch, 0, ch.length, true);
			write('\"');
		}

	}

	/**
	 * vf܂̖͑O݂܂B
	 *
	 * @param uri OԂURI
	 * @param localName vf܂̖͑O
	 * @param qName vf܂͑QName
	 * @param isElement vf̐ݒꍇtrueA̐ݒꍇfalse
	 *
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void writeName(String uri, String localName, String qName, boolean isElement)
			throws DataFileIOException {

		String prefix = doPrefix(uri, qName, isElement);
		if (prefix != null && !"".equals(prefix)) {
			write(prefix);
			write(':');
		}
		write(localName);

	}

	/**
	 * Cfg݂܂B
	 *
	 * @throws DataFileIOException f[^t@CǂݏIOG[ɂ蔭܂B
	 */
	private void doIndent() throws DataFileIOException {

		if (indentSpace > 0 && depth > 0) {
			int n = indentSpace * depth;
			char ch[] = new char[n];
			for (int i = 0; i < n; i++) {
				ch[i] = ' ';
			}
			characters(ch, 0, n);
		}

	}

	/**
	 * w肳ꂽXMLElementAttributesɕϊĎ擾܂B
	 *
	 * @param element XMLElement
	 * @return XMLElement̑SĂ̑
	 */
	private Attributes getAttributes(XMLElement element) {

		AttributesImpl attributes = new AttributesImpl();
		for (XMLAttribute attribute : element.getAttributes().values()) {
			attributes.addAttribute(attribute.getURI(), attribute.getName(), attribute.getQName(),
					"CDATA", attribute.getString());
		}
		return attributes;

	}

}
