/*
 * XMLFileWriterImplTest.java
 * Copyright (C) 2008 Cyber Beans Corporation. All rights reserved.
 */
package jp.co.cybec.cb3.accessor.fileaccess.xml;

import static org.junit.Assert.fail;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;

import jp.co.cybec.cb3.accessor.fileaccess.xml.XMLFileWriterImpl;
import jp.co.cybec.cb3.exception.AccessorAlreadyOpenedException;
import jp.co.cybec.cb3.exception.AccessorNotOpenException;
import jp.co.cybec.cb3.exception.AccessorUnknownException;
import jp.co.cybec.cb3.exception.BadXmlGrammerException;
import jp.co.cybec.cb3.exception.ConfigFileNotFoundException;
import jp.co.cybec.cb3.exception.DataFileIOException;
import jp.co.cybec.cb3.exception.DataFileNotFoundException;
import junit.framework.Assert;

import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

/**
 * {@link XMLFileWriterImpl} ̃eXgP[XB
 * @author Daisuke Suga
 */
public class XMLFileWriterImplTest {

	XMLFileWriterImpl target = null;

	/** ݒt@C  */
	private static final String CONFIG_WRITER1 = "jp/co/cybec/cb3/accessor/fileaccess/xml/writer1Config";

	/** ݒt@C 2 */
	private static final String CONFIG_WRITER2 = "jp/co/cybec/cb3/accessor/fileaccess/xml/writer2Config";

	/** ݒt@C f[^t@C݂Ȃ */
	private static final String CONFIG_WRITER_FILE_NOT_FOUND = "jp/co/cybec/cb3/accessor/fileaccess/xml/writerFileNotFoundConfig";

	/** f[^t@C  */
	private static final String DATA_WRITER1 = "./src/test/testdata/xml/writer1.xml";

	/** f[^t@C 2 */
	private static final String DATA_WRITER2 = "./src/test/testdata/xml/writer2.xml";

	/** f[^t@C */
	private static final String FILE_NORMAL= "src/test/testdata/xml/writer1.xml";

	@Before
	public void setUp() {
		target = new XMLFileWriterImpl();
	}

	@After
	public void tearDown() {
		target.release();
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfo͂邱ƂmFB
	 */
	@Test
	public void testStartElement() {
		XMLElement element = new XMLElement("test1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.startElement(element);
		target.release();
		Assert.assertEquals("<test1>", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfɖOURIo͂邱ƂmFB
	 */
	@Test
	public void testStartElementNamespace() {
		XMLElement element = new XMLElement("test1");
		element.setURI("uri1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.addNamespace("uri1", "p1");
		target.startElement(element);
		target.release();
		Assert.assertEquals("<p1:test1 xmlns:p1=\"uri1\">", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfɖOURIo͂邱ƂmFB
	 */
	@Test
	public void testStartElementNamespace2() {
		XMLElement element = new XMLElement("test1");
		element.setURI("uri\"1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.addNamespace("uri\"1", "p1");
		target.startElement(element);
		target.release();
		Assert.assertEquals("<p1:test1 xmlns:p1=\"uri&quot;1\">", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfɖOURIo͂邱ƂmFB
	 */
	@Test
	public void testStartElementDefaultNamespace() {
		XMLElement element = new XMLElement("test1");
		element.setURI("uri1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.addNamespace("uri1");
		target.startElement(element);
		target.release();
		Assert.assertEquals("<test1 xmlns=\"uri1\">", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfɕ̖OURIo͂邱ƂmFB
	 */
	@Test
	public void testStartElementNamespaces() {
		XMLElement element = new XMLElement("test1");
		element.setURI("uri1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.addNamespace("uri1", "p1");
		target.addNamespace("uri2", "p2");
		target.addNamespace("uri3", "");
		target.startElement(element);
		target.release();
		Assert.assertEquals("<p1:test1 xmlns:p1=\"uri1\" xmlns:p2=\"uri2\" xmlns=\"uri3\">",
				getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfQNameo͂邱ƂmFB
	 */
	@Test
	public void testStartElementQName() {
		XMLElement element = new XMLElement("test1");
		element.setURI("uri1");
		element.setQName("p1:test1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.startElement(element);
		target.release();
		Assert.assertEquals("<p1:test1 xmlns:p1=\"uri1\">", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfQNameɒ`ς݂̖OURIݒ肳ĂȂꍇɐo͂邱ƂmFB
	 */
	@Test
	public void testStartElementQNameNoDeclaredNamespace() {
		XMLElement element = new XMLElement("test1");
		element.setQName("p1:test1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.startElement(element);
		target.release();
		Assert.assertEquals("<test1>", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfQNameɖOԐړݒ肳ĂȂꍇɐo͂邱ƂmFB
	 */
	@Test
	public void testStartElementQNameNoPrefix() {
		XMLElement element = new XMLElement("test1");
		element.setURI("uri1");
		element.setQName("test1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.startElement(element);
		target.release();
		Assert.assertEquals("<test1>", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfɑo͂邱ƂmFB
	 */
	@Test
	public void testStartElementAttribute() {
		XMLElement element = new XMLElement("test1");
		element.putAttribute(new XMLAttribute("aaa1", "value1"));
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.startElement(element);
		target.release();
		Assert.assertEquals("<test1 aaa1=\"value1\">", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfɕ̑o͂邱ƂmFB
	 */
	@Test
	public void testStartElementAttributes() {
		XMLElement element = new XMLElement("test1");
		element.putAttribute(new XMLAttribute("aaa1", "value1"));
		element.putAttribute(new XMLAttribute("aaa2", "value2"));
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.startElement(element);
		target.release();
		Assert.assertEquals("<test1 aaa1=\"value1\" aaa2=\"value2\">", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfɖOURIƑo͂邱ƂmFB
	 */
	@Test
	public void testStartElementNamespacesAttributes() {
		XMLElement element = new XMLElement("test1");
		element.setURI("uri1");
		element.putAttribute(new XMLAttribute("aaa1", "value1"));
		element.putAttribute(new XMLAttribute("aaa2", "value2"));
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.addNamespace("uri1", "p1");
		target.addNamespace("uri2", "p2");
		target.startElement(element);
		target.release();
		Assert.assertEquals(
				"<p1:test1 aaa1=\"value1\" aaa2=\"value2\" xmlns:p1=\"uri1\" xmlns:p2=\"uri2\">",
				getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfCfgďo͂邱ƂmFB
	 */
	@Test
	public void testStartElementIndent() {
		XMLElement element1 = new XMLElement("test1");
		XMLElement element2 = new XMLElement("test2");
		XMLElement element3 = new XMLElement("test3");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.startElement(element1);
		target.startElement(element2);
		target.startElement(element3);
		target.release();
		Assert.assertEquals("    <test3>", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfݒt@CŎw肵Cfgŏo͂邱ƂmFB
	 */
	@Test
	public void testStartElementIndent2() {
		XMLElement element1 = new XMLElement("test1");
		XMLElement element2 = new XMLElement("test2");
		XMLElement element3 = new XMLElement("test3");
		target.initialize(CONFIG_WRITER2);
		target.open();
		target.startElement(element1);
		target.startElement(element2);
		target.startElement(element3);
		target.release();
		Assert.assertEquals("        <test3>", getWriteText(DATA_WRITER2));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfɑo͂邱ƂmFB
	 */
	@Test
	public void testStartElementAttributeNamespaces() {
		XMLElement element = new XMLElement("test1");
		XMLAttribute attribute = new XMLAttribute("aaa1", "value1");
		attribute.setURI("uri1");
		element.putAttribute(attribute);
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.addNamespace("uri1", "p1");
		target.startElement(element);
		target.release();
		Assert.assertEquals("<test1 p1:aaa1=\"value1\" xmlns:p1=\"uri1\">",
				getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * f[^t@CI[vĂȂƂɗO邱ƂmFB
	 */
	@Test(expected = AccessorNotOpenException.class)
	public void testStartElementAccessorNotOpenException() {
		target.initialize(CONFIG_WRITER1);
		target.startElement(new XMLElement("test1"));
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * f[^t@C݂̏ɎsƂɗO邱ƂmFB
	 */
	@Ignore("DataFileIOException𔭐Ȃ")
	@Test(expected = DataFileIOException.class)
	public void testStartElementDataFileIOException() {
	}

	/**
	 * {@link XMLFileWriterImpl#startElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * f[^t@C݂̏ɎsƂɗO邱ƂmFB
	 */
	@Ignore("AccessorUnknownException𔭐Ȃ")
	@Test(expected = AccessorUnknownException.class)
	public void testStartElementAccessorUnknownException() {
	}

	/**
	 * {@link XMLFileWriterImpl#endElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfo͂邱ƂmFB
	 */
	@Test
	public void testEndElement() {
		XMLElement element = new XMLElement("test1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.startElement(element);
		target.endElement(element);
		target.release();
		Assert.assertEquals("<test1></test1>", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#endElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * ̗vfo͂邱ƂmFB
	 */
	@Test
	public void testEndElements() {
		XMLElement element1 = new XMLElement("test1");
		XMLElement element2 = new XMLElement("test2");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.startElement(element1);
		target.startElement(element2);
		target.endElement(element2);
		target.endElement(element1);
		target.release();
		Assert.assertEquals("</test1>", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#endElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfCfgďo͂邱ƂmFB
	 */
	@Test
	public void testEndElementIndent() {
		XMLElement element1 = new XMLElement("test1");
		XMLElement element2 = new XMLElement("test2");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.startElement(element1);
		target.startElement(element2);
		target.endElement(element2);
		target.release();
		Assert.assertEquals("  <test2></test2>", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#endElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfɐړo͂邱ƂmFB
	 */
	@Test
	public void testEndElementNamespace() {
		XMLElement element = new XMLElement("test1");
		element.setURI("uri1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.addNamespace("uri1", "p1");
		target.startElement(element);
		target.startElement(element);
		target.endElement(element);
		target.endElement(element);
		target.release();
		Assert.assertEquals("</p1:test1>", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#endElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfɐړo͂邱ƂmFB
	 */
	@Test
	public void testEndElementQName() {
		XMLElement element = new XMLElement("test1");
		element.setURI("uri1");
		element.setQName("p1:test1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.startElement(element);
		target.startElement(element);
		target.endElement(element);
		target.endElement(element);
		target.release();
		Assert.assertEquals("</p1:test1>", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#endElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * f[^t@CI[vĂȂƂɗO邱ƂmFB
	 */
	@Test(expected = AccessorNotOpenException.class)
	public void testEndElementAccessorNotOpenException() {
		target.initialize(CONFIG_WRITER1);
		target.endElement(new XMLElement("test1"));
	}

	/**
	 * {@link XMLFileWriterImpl#endElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * {@link XMLFileWriterImpl#startElement(XMLElement)}ɎsȂƂɗO邱ƂmFB
	 */
	@Test(expected = BadXmlGrammerException.class)
	public void testEndElementBadXmlGrammerException() {
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.endElement(new XMLElement("test1"));
	}

	/**
	 * {@link XMLFileWriterImpl#endElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * f[^t@C݂̏ɎsƂɗO邱ƂmFB
	 */
	@Ignore("DataFileIOException𔭐Ȃ")
	@Test(expected = DataFileIOException.class)
	public void testEndElementDataFileIOException() {
	}

	/**
	 * {@link XMLFileWriterImpl#endElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * f[^t@C݂̏ɎsƂɗO邱ƂmFB
	 */
	@Ignore("AccessorUnknownException𔭐Ȃ")
	@Test(expected = AccessorUnknownException.class)
	public void testEndElementAccessorUnknownException() {
	}

	/**
	 * {@link XMLFileWriterImpl#dataElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vfo͂邱ƂmFB
	 */
	@Test
	public void testDataElement() {
		XMLElement element = new XMLElement("test1");
		element.setString("aaa1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.dataElement(element);
		target.release();
		Assert.assertEquals("<test1>aaa1</test1>", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#dataElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vf̒lȂꍇɏo͂邱ƂmFB
	 */
	@Test
	public void testDataElementNoData() {
		XMLElement element = new XMLElement("test1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.dataElement(element);
		target.release();
		Assert.assertEquals("<test1></test1>", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#dataElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * f[^t@CI[vĂȂƂɗO邱ƂmFB
	 */
	@Test(expected = AccessorNotOpenException.class)
	public void testDataElementAccessorNotOpenException() {
		target.initialize(CONFIG_WRITER1);
		target.dataElement(new XMLElement("test1"));
	}

	/**
	 * {@link XMLFileWriterImpl#dataElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * f[^t@C݂̏ɎsƂɗO邱ƂmFB
	 */
	@Ignore("DataFileIOException𔭐Ȃ")
	@Test(expected = DataFileIOException.class)
	public void testDataElementDataFileIOException() {
	}

	/**
	 * {@link XMLFileWriterImpl#dataElement(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * f[^t@C݂̏ɎsƂɗO邱ƂmFB
	 */
	@Ignore("AccessorUnknownException𔭐Ȃ")
	@Test(expected = AccessorUnknownException.class)
	public void testDataElementAccessorUnknownException() {
	}

	/**
	 * {@link XMLFileWriterImpl#characters(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vf̒lo͂邱ƂmFB
	 */
	@Test
	public void testCharacters() {
		XMLElement element = new XMLElement("test1");
		element.setString("aaa1");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.characters(element);
		target.release();
		Assert.assertEquals("aaa1", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#characters(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * vf̒lGXP[vďo͂邱ƂmFB
	 */
	@Test
	public void testCharactersEscape() {
		XMLElement element = new XMLElement("test1");
		element.setString("aaa<>&\"");
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.characters(element);
		target.release();
		Assert.assertEquals("aaa&lt;&gt;&amp;\"", getWriteText(DATA_WRITER1));
	}

	/**
	 * {@link XMLFileWriterImpl#characters(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * f[^t@CI[vĂȂƂɗO邱ƂmFB
	 */
	@Test(expected = AccessorNotOpenException.class)
	public void testCharactersAccessorNotOpenException() {
		target.initialize(CONFIG_WRITER1);
		target.characters(new XMLElement("test1"));
	}

	/**
	 * {@link XMLFileWriterImpl#characters(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * f[^t@C݂̏ɎsƂɗO邱ƂmFB
	 */
	@Ignore("DataFileIOException𔭐Ȃ")
	@Test(expected = DataFileIOException.class)
	public void testCharactersDataFileIOException() {
	}

	/**
	 * {@link XMLFileWriterImpl#characters(XMLElement)} ̂߂̃eXgE\bhB<br>
	 * f[^t@C݂̏ɎsƂɗO邱ƂmFB
	 */
	@Ignore("AccessorUnknownException𔭐Ȃ")
	@Test(expected = AccessorUnknownException.class)
	public void testCharactersAccessorUnknownException() {
	}

	/**
	 * {@link XMLFileWriterImpl#open()} ̂߂̃eXgE\bhB<br>
	 * OȂƂmFB
	 */
	@Test
	public void testOpen() {
		target.initialize(CONFIG_WRITER1);
		target.open();
	}

	/**
	 * {@link XMLFileWriterImpl#open()} ̂߂̃eXgE\bhB<br>
	 * QI[vƂɗO邱ƂmFB
	 */
	@Test(expected = AccessorAlreadyOpenedException.class)
	public void testOpenAccessorAlreadyOpenedException() {
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.open();
	}

	/**
	 * {@link XMLFileWriterImpl#open()} ̂߂̃eXgE\bhB<br>
	 * f[^t@C݂ȂƂɗO邱ƂmFB
	 */
	@Test(expected = DataFileNotFoundException.class)
	public void testOpenDataFileNotFoundException() {
		target.initialize(CONFIG_WRITER_FILE_NOT_FOUND);
		target.open();
	}

	/**
	 * {@link XMLFileWriterImpl#open()} ̂߂̃eXgE\bhB<br>
	 * f[^t@C݂̏ɎsƂɗO邱ƂmFB
	 */
	@Ignore("DataFileIOException𔭐Ȃ")
	@Test(expected = DataFileIOException.class)
	public void testOpenDataFileIOException() {
	}

	/**
	 * {@link XMLFileWriterImpl#open(String)} ̂߂̃eXgE\bhB<br>
	 * OȂƂmFB
	 */
	@Test
	public void testOpenString() {
		target.initialize(CONFIG_WRITER1);
		target.open(FILE_NORMAL);
	}

	/**
	 * {@link XMLFileWriterImpl#open(String)} ̂߂̃eXgE\bhB
	 * f[^t@Cnull̎ɗO邱ƂmFB
	 */
	@Test(expected = DataFileNotFoundException.class)
	public void testOpenStringNullFilepath() {
		target.initialize(CONFIG_WRITER1);
		target.open(null);
	}

	/**
	 * {@link XMLFileWriterImpl#close()} ̂߂̃eXgE\bhB<br>
	 * OȂƂmFB
	 */
	@Test
	public void testClose() {
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.close();
		target.open();
	}

	/**
	 * {@link XMLFileWriterImpl#close()} ̂߂̃eXgE\bhB<br>
	 * f[^t@CI[vĂȂĂOȂƂmFB
	 */
	@Test
	public void testCloseNotOpen() {
		target.initialize(CONFIG_WRITER1);
		target.close();
	}

	/**
	 * {@link XMLFileWriterImpl#close()} ̂߂̃eXgE\bhB<br>
	 * f[^t@C̃N[YɎsƂɗO邱ƂmFB
	 */
	@Ignore("DataFileIOException𔭐Ȃ")
	@Test(expected = DataFileIOException.class)
	public void testCloseDataFileIOException() {
	}

	/**
	 * {@link XMLFileWriterImpl#initialize(String)} ̂߂̃eXgE\bhB<br>
	 * OȂƂmFB
	 */
	@Test
	public void testInitialize() {
		target.initialize(CONFIG_WRITER1);
	}

	/**
	 * {@link XMLFileWriterImpl#open()} ̂߂̃eXgE\bhB<br>
	 * ݒt@C̕R[hAsR[hŏo͂邱ƂmFB
	 */
	@Test
	public void testInitializeEncodeLineSeparator() {
		target.initialize(CONFIG_WRITER2);
		target.open();
		target.release();
		Assert.assertEquals("<?xml version=\"1.0\" encoding=\"Shift_JIS\"?>\r\r\r",
				getWriteHeader(DATA_WRITER2));
	}

	/**
	 * {@link XMLFileWriterImpl#initialize(String)} ̂߂̃eXgE\bhB<br>
	 * ɑ݂ȂL[iݒt@Cjw肵ƂɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileNotFoundException.class)
	public void testInitializeConfigFileNotFoundException() {
		target.initialize("aaa");
	}

	/**
	 * {@link XMLFileWriterImpl#close()} ̂߂̃eXgE\bhB<br>
	 * OȂƂmFB
	 */
	@Test
	public void testRelease() {
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.release();
		target.open();
	}

	/**
	 * {@link XMLFileWriterImpl#close()} ̂߂̃eXgE\bhB<br>
	 * f[^t@CI[vĂȂĂOȂƂmFB
	 */
	@Test
	public void testReleaseNotOpen() {
		target.initialize(CONFIG_WRITER1);
		target.release();
	}

	/**
	 * {@link XMLFileWriterImpl#addNamespace(String, String)} ̂߂̃eXgE\bhB<br>
	 * OȂƂmFB
	 */
	@Test
	public void testAddNamespaceStringString() {
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.addNamespace("test1", "aaa1");
		target.addNamespace("test2", "aaa2");
		target.addNamespace("test3", "aaa3");
	}

	/**
	 * {@link XMLFileWriterImpl#addNamespace(String, String)} ̂߂̃eXgE\bhB<br>
	 * f[^t@CI[vĂȂƂɗO邱ƂmFB
	 */
	@Test(expected = AccessorNotOpenException.class)
	public void testAddNamespaceStringStringAccessorNotOpenException() {
		target.initialize(CONFIG_WRITER1);
		target.addNamespace("test1", "aaa1");
	}

	/**
	 * {@link XMLFileWriterImpl#addNamespace(String)} ̂߂̃eXgE\bhB<br>
	 * OȂƂmFB
	 */
	@Test
	public void testAddNamespaceString() {
		target.initialize(CONFIG_WRITER1);
		target.open();
		target.addNamespace("test1");
	}

	/**
	 * {@link XMLFileWriterImpl#addNamespace(String)} ̂߂̃eXgE\bhB<br>
	 * f[^t@CI[vĂȂƂɗO邱ƂmFB
	 */
	@Test(expected = AccessorNotOpenException.class)
	public void testAddNamespaceStringAccessorNotOpenException() {
		target.initialize(CONFIG_WRITER1);
		target.addNamespace("test1");
	}

	/**
	 * f[^t@CŌɏo͂s擾܂B
	 *
	 * @return o͂
	 */
	private String getWriteText(String fileName) {
		try {
			File file = new File(fileName);
			BufferedReader reader = new BufferedReader(new FileReader(file));
			String lastLine = null;
			String text = null;
			while ((text = reader.readLine()) != null) {
				lastLine = text;
			}
			return lastLine;
		} catch (Exception e) {
			fail();
			return null;
		}
	}

	/**
	 * f[^t@Co͂wb_̕擾܂B
	 *
	 * @return o͂
	 */
	private String getWriteHeader(String fileName) {
		try {
			File file = new File(fileName);
			BufferedReader reader = new BufferedReader(new FileReader(file));
			char[] cbuf = new char[(int) file.length()];
			reader.read(cbuf, 0, (int) file.length());
			return new String(cbuf);
		} catch (Exception e) {
			fail();
			return null;
		}
	}
}
