/*
 * ConfigurationLoaderTest.java
 * Copyright (C) 2008 Cyber Beans Corporation. All rights reserved.
 */
package jp.co.cybec.cb3.util;

import java.util.Iterator;

import jp.co.cybec.cb3.accessor.dbaccess.DBAccessConfig;
import jp.co.cybec.cb3.accessor.dbaccess.DeleteSql;
import jp.co.cybec.cb3.accessor.dbaccess.InsertSql;
import jp.co.cybec.cb3.accessor.dbaccess.SelectSql;
import jp.co.cybec.cb3.accessor.dbaccess.UpdateSql;
import jp.co.cybec.cb3.accessor.fileaccess.flat.delimited.DelimitedLineRecordReaderConfig;
import jp.co.cybec.cb3.accessor.fileaccess.flat.delimited.DelimitedLineRecordWriterConfig;
import jp.co.cybec.cb3.accessor.fileaccess.flat.fixed.FixedLengthRecordReaderConfig;
import jp.co.cybec.cb3.accessor.fileaccess.flat.fixed.FixedLengthRecordWriterConfig;
import jp.co.cybec.cb3.accessor.fileaccess.xml.XMLFileReaderConfig;
import jp.co.cybec.cb3.accessor.fileaccess.xml.XMLFileWriterConfig;
import jp.co.cybec.cb3.exception.ConfigFileDefinitionException;
import jp.co.cybec.cb3.exception.ConfigFileNotFoundException;
import jp.co.cybec.cb3.exception.ConfigFileParseException;
import jp.co.cybec.cb3.util.ConfigurationLoader;
import junit.framework.Assert;

import org.junit.Test;

/**
 * {@link ConfigurationLoader} ̃eXgP[XB
 * @author Daisuke Suga
 */
public class ConfigurationLoaderTest {

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * l擾ł邱ƂmFB
	 */
	@Test
	public void testLoad() {

		XMLFileReaderConfig info = (XMLFileReaderConfig) ConfigurationLoader.load(
				"XMLFileReaderRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/normalConfig.xml");

		Assert.assertEquals("./src/test/testdata/xml/test.xml", info.getFilepath());

	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Ɋgqw肵Ȃꍇɐl擾ł邱ƂmFB
	 */
	@Test
	public void testLoadAddExtension() {

		XMLFileReaderConfig info = (XMLFileReaderConfig) ConfigurationLoader.load(
				"XMLFileReaderRule", "jp/co/cybec/cb3/util/ConfigurationLoader/normalConfig");

		Assert.assertEquals("./src/test/testdata/xml/test.xml", info.getFilepath());

	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * digester̃[t@CȂꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileNotFoundException.class)
	public void testNoRuleFile() {
		ConfigurationLoader.load("aaa", "jp/co/cybec/cb3/util/ConfigurationLoader/normalConfig");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ݒt@CȂꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileNotFoundException.class)
	public void testNoConfigFile() {
		ConfigurationLoader.load("XMLFileReaderRule", "aaa");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * digester̃[t@C̉͂ɎsꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileParseException.class)
	public void testRuleFileParseError() {
		ConfigurationLoader.load("jp/co/cybec/cb3/util/ConfigurationLoader/parseErrorConfig",
				"jp/co/cybec/cb3/util/ConfigurationLoader/normalConfig");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ݒt@C̉͂ɎsꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileParseException.class)
	public void testConfigFileParseError() {
		ConfigurationLoader.load("XMLFileReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/parseErrorConfig");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ݒt@C̊ԈႢɂǂݍ݂ɎsꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testConfigFileDefinitionException() {
		ConfigurationLoader.load("XMLFileReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/badDifinition");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷[_[:l擾ł邱ƂmFB
	 */
	@Test
	public void testFixedReaderConfig() {

		FixedLengthRecordReaderConfig info = (FixedLengthRecordReaderConfig) ConfigurationLoader.load(
				"FixedLengthRecordReaderRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedRecordReaderConfig.xml");

		Assert.assertEquals("./src/test/testdata/fixed/test1.xml", info.getFilepath());

	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷[_[:ݒt@C̉͂ɎsꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedReaderConfigFileParseError() {
		ConfigurationLoader.load("FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedReaderConfigParseError");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷[_[:vffield̏o񐔁F0̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedReaderNoField() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedReaderConfigNoField");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷[_[:vffieldname lengthF0̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedReaderFieldNoNameAndLength() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedReaderConfigFieldNoNameAndLength");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷[_[:vffieldnameFnull̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileParseException.class)
	public void testFixedReaderFieldNameNull() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedReaderConfigFieldNameNull");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷[_[:vffieldnameF("")̏ꍇɗOȂƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedReaderFieldNameEmptyStr() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedReaderConfigFieldNameEmptyStr");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷[_[:vffieldlengthF̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedReaderFieldLengthNoInteger() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedReaderConfigFieldLengthNoInteger");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷[_[:vffield̑:
	 * type̒lunumberAdatevȊȌꍇAO邱ƂmFB
	 * align̒luleftArightvȊȌꍇAO邱ƂmFB
	 * length̒lȊȌꍇO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedReaderConfigErrorvalue() {
		ConfigurationLoader.load("FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedReaderErrorConfig");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷[_[:vffield()nameFꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedReaderConfigMultipleSameKey() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedReaderConfigMultipleSameKey");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷C^[:l擾ł邱ƂmFB
	 */
	@Test
	public void testFixedWriterConfig() {

		FixedLengthRecordWriterConfig info = (FixedLengthRecordWriterConfig) ConfigurationLoader.load(
				"FixedLengthRecordWriterRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedRecordWriterConfig.xml");
		Assert.assertEquals("D:\\test_appendwrite1.txt", info.getFilepath());

	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷C^[:vfAppend̃ftHgl擾ł邱ƂmFB
	 */
	@Test
	public void testFixedWriterConfigDefault() {

		FixedLengthRecordWriterConfig info = (FixedLengthRecordWriterConfig) ConfigurationLoader.load(
				"FixedLengthRecordWriterRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedRecordWriterConfig.xml");
		Assert.assertEquals("false", info.getAppend());

	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷C^[:vfAppend̃ftHgl擾łȂƂmFB
	 */
	@Test
	public void testFixedWriterConfigNoDefault() {

		FixedLengthRecordWriterConfig info = (FixedLengthRecordWriterConfig) ConfigurationLoader.load(
				"FixedLengthRecordWriterRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedWriterConfigNoDefault.xml");
		Assert.assertEquals(null, info.getAppend());

	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷C^[:ݒt@C̉͂ɎsꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedWriterConfigFileParseError() {
		ConfigurationLoader.load("FixedLengthRecordWriterRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedWriterConfigParseError");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷C^[:vffield̏o񐔁F0̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedWriterNoField() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedWriterConfigNoField");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷C^[:vffieldname lengthF0̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedWriterFieldNoNameAndLength() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedWriterConfigFieldNoNameAndLength");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷C^[:vffieldnameFnull̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileParseException.class)
	public void testFixedWriterFieldNameNull() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedWriterConfigFieldNameNull");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷C^[:vffieldnameF("")̏ꍇɗOȂƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedWriterFieldNameEmptyStr() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedWriterConfigFieldNameEmptyStr");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷C^[:vffieldlengthF̏ꍇ̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedWriterFieldLengthNoInteger() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedWriterConfigFieldLengthNoInteger");
	}


	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷C^[:vffield̑:
	 * type̒lunumberAdatevȊȌꍇAO邱ƂmFB
	 * align̒luleftArightvȊȌꍇAO邱ƂmFB
	 * length̒lȊȌꍇO邱ƂmFB
	 * lineSeparator̒lu\n / \r\n / \rvȊȌꍇAO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedWriterConfigErrorvalue() {
		ConfigurationLoader.load("FixedLengthRecordWriterRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedWriterErrorConfig");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * Œ蒷C^[:vffield()nameFꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testFixedWriterConfigMultipleSameKey() {
		ConfigurationLoader.load(
				"FixedLengthRecordWriterRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/fixedWriterConfigMultipleSameKey");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒ[_[:l擾ł邱ƂmFB
	 */
	@Test
	public void testDelimitedReaderConfig() {
		DelimitedLineRecordReaderConfig info = (DelimitedLineRecordReaderConfig) ConfigurationLoader.load(
				"DelimitedLineRecordReaderRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedReaderConfig.xml");
		Assert.assertEquals("D:\\test.csv", info.getFilepath());
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒ[_[:~j}Zbg̏ꍇɐl擾ł邱ƂmFB
	 */
	@Test
	public void testDelimitedReaderMiniConfig() {
		DelimitedLineRecordReaderConfig info = (DelimitedLineRecordReaderConfig) ConfigurationLoader.load(
				"DelimitedLineRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedReaderMiniConfig");
		Assert.assertEquals("D:\\test.csv", info.getFilepath());
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒ[_[:vffilepath̏o񐔁F2̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDelimitedReaderRepeatFilepath() {
		ConfigurationLoader.load(
				"DelimitedLineRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedReaderConfigRepeatFilepath");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒ[_[:vffield̑namȅo񐔁F0̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDelimitedReaderAttributeNoName() {
		ConfigurationLoader.load(
				"DelimitedLineRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedReaderConfigAttributeNoName");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒ[_[:vffield̑namȅo񐔁F2̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileParseException.class)
	public void testDelimitedReaderAttributeRepeatName() {
		ConfigurationLoader.load(
				"DelimitedLineRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedReaderConfigAttributeRepeatName");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒ[_[:vffield̑name̒l("")̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDelimitedReaderAttributeNameIsNull() {
		ConfigurationLoader.load(
				"DelimitedLineRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedReaderConfigAttributeNameIsNull");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒ[_[:vffield̑:
	 * type̒lunumberAdatevȊȌꍇAO邱ƂmFB
	 * align̒luleftArightvȊȌꍇAO邱ƂmFB
	 * length̒lnonNegativeIntegerȊȌꍇO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDelimitedReaderConfigErrorValue01() {
		ConfigurationLoader.load("DelimitedLineRecordReaderRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedReaderConfigErrorValue.xml");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒ[_[:vffield:length̒l̏ꍇO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDelimitedReaderConfigErrorValue02() {
		ConfigurationLoader.load("DelimitedLineRecordReaderRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedReaderConfigErrorValue2.xml");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒ[_[:vffield()nameFꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDelimitedReaderConfigMultipleSameKey() {
		ConfigurationLoader.load(
				"DelimitedLineRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedReaderConfigMultipleSameKey");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒC^[:l擾ł邱ƂmFB
	 */
	@Test
	public void testDelimitedWriterConfig() {
		DelimitedLineRecordWriterConfig info = (DelimitedLineRecordWriterConfig) ConfigurationLoader.load(
				"DelimitedLineRecordWriterRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedWriterConfig.xml");
		Assert.assertEquals("D:\\test_appendwrite.csv", info.getFilepath());

	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒC^[:~j}Zbg̏ꍇɐl擾ł邱ƂmFB
	 */
	@Test
	public void testDelimitedWriterMiniConfig() {
		DelimitedLineRecordWriterConfig info = (DelimitedLineRecordWriterConfig) ConfigurationLoader.load(
				"DelimitedLineRecordWriterRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedWriterMiniConfig");
		Assert.assertEquals("D:\\test_write.csv", info.getFilepath());

	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒC^[:vffilepath̏o񐔁F2̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDelimitedWriterRepeatFilepath() {
		ConfigurationLoader.load(
				"DelimitedLineRecordWriterRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedWriterConfigRepeatFilepath");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒC^[:ϒC^[:vffield̑namȅo񐔁F0̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDelimitedWriterAttributeNoName() {
		ConfigurationLoader.load(
				"DelimitedLineRecordWriterRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedWriterConfigAttributeNoName");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒC^[:vffield̑namȅo񐔁F2̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileParseException.class)
	public void testDelimitedWriterAttributeRepeatName() {
		ConfigurationLoader.load(
				"DelimitedLineRecordWriterRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedWriterConfigAttributeRepeatName");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒC^[:vffield̑name̒l("")̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDelimitedWriterAttributeNameIsNull() {
		ConfigurationLoader.load(
				"DelimitedLineRecordWriterRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedWriterConfigAttributeNameIsNull");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒC^[:vffield̑:
	 * type̒lunumberAdatevȊȌꍇAO邱ƂmFB
	 * align̒luleftArightvȊȌꍇAO邱ƂmFB
	 * length̒lnonNegativeIntegerȊȌꍇO邱ƂmFB
	 * lineSeparator̒lu\nA\r\nA\rvȊȌꍇO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDelimitedWriterConfigErrorValue01() {
		ConfigurationLoader.load("DelimitedLineRecordWriterRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedWriterConfigErrorValue.xml");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒC^[:length̒l̏ꍇO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDelimitedWriterConfigErrorValue02() {
		ConfigurationLoader.load("DelimitedLineRecordWriterRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedWriterConfigErrorValue2.xml");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * ϒC^[:vffield()nameFꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDelimitedWriterConfigMultipleSameKey() {
		ConfigurationLoader.load(
				"DelimitedLineRecordWriterRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/delimitedWriterConfigMultipleSameKey");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * XML[_[:l擾ł邱ƂmFB
	 */
	@Test
	public void testXMLFileReaderConfig() {
		XMLFileReaderConfig info = (XMLFileReaderConfig) ConfigurationLoader.load(
				"XMLFileReaderRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/xmlFileReaderConfig.xml");
		Assert.assertEquals("./src/test/testdata/xml/reader1.xml", info.getFilepath());
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * XML[_[:vffilepath̏o񐔁F2̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testXMLFileReaderRepeatFilepath() {
		ConfigurationLoader.load("XMLFileReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/xmlFileReaderConfigRepeatFilepath");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * XMLC^[:l擾ł邱ƂmFB
	 */
	@Test
	public void testXMLFileWriterConfig() {
		XMLFileWriterConfig info = (XMLFileWriterConfig) ConfigurationLoader.load(
				"XMLFileWriterRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/xmlFileWriterConfig.xml");
		Assert.assertEquals("./src/test/testdata/xml/writer.xml", info.getFilepath());
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * XMLC^[:~j}Zbg̏ꍇɐl擾ł邱ƂmFB
	 */
	@Test
	public void testXMLFileWriterMiniConfig() {

		XMLFileWriterConfig info = (XMLFileWriterConfig) ConfigurationLoader.load(
				"XMLFileWriterRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/xmlFileWriterMiniConfig");
		Assert.assertEquals("./src/test/testdata/xml/writer.xml", info.getFilepath());
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * XMLC^[:vffilepath̏o񐔁F2̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testXMLFileWriterRepeatFilepath() {
		ConfigurationLoader.load("XMLFileWriterRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/xmlFileWriterConfigRepeatFilepath");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * XMLC^[:vfindentSpace̒lnonNegativeIntegerȊȌꍇO邱ƂmFB
	 * XMLC^[:vflineSeparator̒lu\nA\r\nA\rvȊȌꍇAO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testXMLFileWriterConfigFileErrorValue() {
		ConfigurationLoader.load("XMLFileReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/xmlFileWriterConfigErrorValue");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * XMLC^[:vfindentSpace̒lchaȑꍇO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testXMLFileWriterConfigFileErrorValue02() {
		ConfigurationLoader.load("XMLFileReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/xmlFileWriterConfigErrorValue2");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * DBANZX:l擾ł邱ƂmFB
	 */
	@Test
	public void testDBAccessorConfig() {
		int i=1;

		DBAccessConfig info = (DBAccessConfig) ConfigurationLoader.load(
				"dbaccessConfigRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/dbaccessConfig.xml");
		for(Iterator<SelectSql> it = info.getSelectSql().iterator(); it.hasNext() ; ){
			SelectSql sql = (SelectSql)it.next();
			Assert.assertEquals("selRecordTest", sql.getId());
			Assert.assertEquals("select * from RECORDTEST", sql.getSelect());
		}
		for(Iterator<InsertSql> it = info.getInsertSql().iterator(); it.hasNext() ; ){
			InsertSql sql = (InsertSql)it.next();
			if(i==1){
			   	Assert.assertEquals("insert1", sql.getId());
				Assert.assertEquals("INSERT INTO EMPLOYEE VALUES('tst00','','08088888888')", sql.getInsert());

			}else{
				Assert.assertEquals("insert2", sql.getId());
				Assert.assertEquals("INSERT INTO EMPLOYEE VALUES('tst00','','08088888888')", sql.getInsert());
			}
			i++;
		}
		for(Iterator<DeleteSql> it = info.getDeleteSql().iterator(); it.hasNext() ; ){
			DeleteSql sql = (DeleteSql)it.next();
			Assert.assertEquals("delete1", sql.getId());
			Assert.assertEquals("DELETE FROM  EMPLOYEE", sql.getDelete());
		}
		for(Iterator<UpdateSql> it = info.getUpdateSql().iterator(); it.hasNext() ; ){
			UpdateSql sql = (UpdateSql)it.next();
			Assert.assertEquals("update1", sql.getId());
			Assert.assertEquals("UPDATE EMPLOYEE SET EMP_NAME = :EMP_NAME WHERE emp_code = :EMP_CODE", sql.getUpdate());
		}
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * DBANZX:vf(select/update/delete/insert)Ȃ̏ꍇO邱ƂmF
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDBAccessorConfigNoElement() {

		ConfigurationLoader.load(
				"dbaccessConfigRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/dbaccessConfigNoElement.xml");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * vfselect̑id̏o񐔁F0̏ꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDBAccessorConfigAttributeNoId() {

		ConfigurationLoader.load(
				"dbaccessConfigRule.xml",
				"jp/co/cybec/cb3/util/ConfigurationLoader/dbaccessConfigAttributeNoId.xml");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * DBANZX:vfselect̑id̒lNULL̏ꍇɗO邱ƂmF
	 */
	@Test(expected = ConfigFileParseException.class)
	public void testDBAccessorConfigSelectIdNull() {

		ConfigurationLoader.load(
				"DelimitedLineRecordWriterRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/DBAccessorConfigSelectIdNull");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * DBANZX:vfselect̑id̒l("")̏ꍇɗOȂƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDBAccessorConfigSelectIdEmptyStr() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/DBAccessorConfigSelectIdEmptyStr");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * DBANZX:vfupdatȇcommitLimitCount̒l̏ꍇɗOȂƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDBAccessorConfigcommitLimitCountNoNumber() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/DBAccessorConfigcommitLimitCountNoNumber");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * DBANZX:vfupdatȇcommitLimitCount̒l'-10'̏ꍇɗOȂƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDBAccessorConfigcommitLimitCountNoInteger() {
		ConfigurationLoader.load(
				"FixedLengthRecordReaderRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/DBAccessorConfigcommitLimitCountNoInteger");
	}

	/**
	 * {@link ConfigurationLoader#load(String, String)} ̂߂̃eXgE\bhB<br>
	 * DBANZX:vf(select/update/delete/insert)()idFꍇɗO邱ƂmFB
	 */
	@Test(expected = ConfigFileDefinitionException.class)
	public void testDBAccessorConfigMultipleSameKey() {
		ConfigurationLoader.load(
				"DelimitedLineRecordWriterRule",
				"jp/co/cybec/cb3/util/ConfigurationLoader/dbAccessorConfigMultipleSameId");
	}

}
