/*
 * NumberUtilTest.java
 * Copyright (C) 2008 Cyber Beans Corporation. All rights reserved.
 */
package jp.co.cybec.cb3.util;

import java.math.BigDecimal;
import java.text.ParseException;

import jp.co.cybec.cb3.util.NumberUtil;
import junit.framework.Assert;

import org.junit.Test;

/**
 * NumberUtilTest̃eXgP[XB
 * @author Jianming Guo
 */
public class NumberUtilTest {

	/**
	 * {@link NumberUtil#getFractionDigits(String)} ̂߂̃eXgE\bhB
	 * ̏_̌𐳂擾邱ƂmFB
	 */
	@Test
	public void testGetFractionDigits() {
		// Fnull
		int result = NumberUtil.getFractionDigits(null);
		Assert.assertEquals(0, result);

		// F
		result = NumberUtil.getFractionDigits("");
		Assert.assertEquals(0, result);

		// F12345.230
		result = NumberUtil.getFractionDigits("12345.230");
		Assert.assertEquals(3, result);

		// F12345.0
		result = NumberUtil.getFractionDigits("12345.0");
		Assert.assertEquals(1, result);

		// F12345.
		result = NumberUtil.getFractionDigits("12345.");
		Assert.assertEquals(0, result);

		// F12345
		result = NumberUtil.getFractionDigits("12345");
		Assert.assertEquals(0, result);

		// F0.123
		result = NumberUtil.getFractionDigits("0.123");
		Assert.assertEquals(3, result);

		// F.123
		result = NumberUtil.getFractionDigits(".123");
		Assert.assertEquals(3, result);
		// F0.0
		result = NumberUtil.getFractionDigits("0.0");
		Assert.assertEquals(1, result);
	}

	/**
	 * {@link NumberUtil#parseBigDecimal(String, String)} ̂߂̃eXgE\bhB
	 * BigDecimal^ɐp[X邱ƂmFB
	 */
	@Test
	public void testParseBigDecimal() throws Exception{
		BigDecimal expected;
		BigDecimal result;

		expected = new BigDecimal("1234567.89");

		result = NumberUtil.parseBigDecimal("####.#", "1234567.89");
		Assert.assertEquals(expected, result);

		result = NumberUtil.parseBigDecimal("#,###.#", "1,234,567.89");
		Assert.assertEquals(expected, result);

		result = NumberUtil.parseBigDecimal("#,##0.#", "1,234,567.89");
		Assert.assertEquals(expected, result);

		result = NumberUtil.parseBigDecimal("#,##0.000", "1,234,567.89");
		Assert.assertEquals(expected, result);

		result = NumberUtil.parseBigDecimal("###0.#E0", "12345.6789E2");
		Assert.assertEquals(expected, result);

		result = NumberUtil.parseBigDecimal("#,##0.00%", "1,234,56789%");
		Assert.assertEquals(expected, result);

		result = NumberUtil.parseBigDecimal("\u00A4#,##0.000", "1,234,567.89");
		Assert.assertEquals(expected, result);

		expected = new BigDecimal("-1234567.89");

		result = NumberUtil.parseBigDecimal("####.#", "-1234567.89");
		Assert.assertEquals(expected, result);

		result = NumberUtil.parseBigDecimal("#,###.#", "-1,234,567.89");
		Assert.assertEquals(expected, result);

		result = NumberUtil.parseBigDecimal("#,##0.#", "-1,234,567.89");
		Assert.assertEquals(expected, result);

		result = NumberUtil.parseBigDecimal("#,##0.000", "-1,234,567.89");
		Assert.assertEquals(expected, result);

		result = NumberUtil.parseBigDecimal("###0.#E0", "-12345.6789E2");
		Assert.assertEquals(expected, result);

		result = NumberUtil.parseBigDecimal("#,##0.00%", "-123,456,789%");
		Assert.assertEquals(expected, result);

		result = NumberUtil.parseBigDecimal("\u00A4#,##0.000", "-1,234,567.89");
		Assert.assertEquals(expected, result);

	}

	/**
	 * {@link NumberUtil#parseBigDecimal(String, String)} ̂߂̃eXgE\bhB
	 * 𐔎ȂꍇAO邱ƂmFB
	 */
	@Test(expected = ParseException.class)
	public void testParseBigDecimalParseException() throws Exception{
		NumberUtil.parseBigDecimal("####.#", "a12aaa.89");
	}

	/**
	 * {@link NumberUtil#parseBigDecimal(String, String)} ̂߂̃eXgE\bhB
	 * ȃtH[}bg̏ꍇAO邱ƂmFB
	 */
	@Test(expected = IllegalArgumentException.class)
	public void testParseBigDecimalIllegalArgumentException() throws Exception{
		NumberUtil.parseBigDecimal("####.#0000", "12356.89");

	}

	/**
	 * {@link NumberUtil#formatNumber(String, String)} ̂߂̃eXgE\bhB
	 * 𐳂tH[}bg邱ƂmFB
	 */
	@Test
	public void testFormatNumber() throws Exception{
		String expected;
		String result;

		expected = "1234567.83";
		result = NumberUtil.formatNumber("####.0#", "1234567.83");
		Assert.assertEquals(expected, result);

		expected = "1,234,567.8";
		result = NumberUtil.formatNumber("#,###.#", "1234567.82");
		Assert.assertEquals(expected, result);

		expected = "1,234,567";
		result = NumberUtil.formatNumber("#,##0.#", "1234567.00");
		Assert.assertEquals(expected, result);

		expected = "1,234,567.890";
		result = NumberUtil.formatNumber("#,##0.000", "1234567.89");
		Assert.assertEquals(expected, result);

		expected = "123.46E4";
		result = NumberUtil.formatNumber("###0.#E0", "1234567.82");
		Assert.assertEquals(expected, result);

		expected = "123,456,789.00%";
		result = NumberUtil.formatNumber("#,##0.00%", "1234567.89");
		Assert.assertEquals(expected, result);

		expected = "1,234,567.890";
		result = NumberUtil.formatNumber("\u00A4#,##0.000", "1234567.89");
		Assert.assertEquals(expected, result);


		expected = "-1234567.83";
		result = NumberUtil.formatNumber("####.0#", "-1234567.83");
		Assert.assertEquals(expected, result);

		expected = "-1,234,567.8";
		result = NumberUtil.formatNumber("#,###.#", "-1234567.82");
		Assert.assertEquals(expected, result);

		expected = "-1,234,567";
		result = NumberUtil.formatNumber("#,##0.#", "-1234567.00");
		Assert.assertEquals(expected, result);

		expected = "-1,234,567.890";
		result = NumberUtil.formatNumber("#,##0.000", "-1234567.89");
		Assert.assertEquals(expected, result);

		expected = "-123.46E4";
		result = NumberUtil.formatNumber("###0.#E0", "-1234567.82");
		Assert.assertEquals(expected, result);

		expected = "-123,456,789.00%";
		result = NumberUtil.formatNumber("#,##0.00%", "-1234567.89");
		Assert.assertEquals(expected, result);

		expected = "-1,234,567.890";
		result = NumberUtil.formatNumber("\u00A4#,##0.000", "-1234567.89");
		Assert.assertEquals(expected, result);
	}

	/**
	 * {@link NumberUtil#formatNumber(String, String)} ̂߂̃eXgE\bhB
	 * 𐔎ȂꍇAO邱ƂmFB
	 */
	@Test(expected = NumberFormatException.class)
	public void testFormatNumberNumberFormatException() throws Exception{
		NumberUtil.formatNumber("####.#", "a12aaa.89");
	}

	/**
	 * {@link NumberUtil#formatNumber(String, String)} ̂߂̃eXgE\bhB
	 * ȃtH[}bg̏ꍇAO邱ƂmFB
	 */
	@Test(expected = IllegalArgumentException.class)
	public void testFormatNumberIllegalArgumentException() throws Exception {
		NumberUtil.formatNumber("####.#0000", "12356.89");
	}

	/**
	 * {@link NumberUtil#numberToString(BigDecimal, int)} ̂߂̃eXgE\bhB
	 * BigDecimal^w̏̕ɐϊ邱ƂmFB
	 */
	@Test
	public void testNumberToString() {
		String expected;
		String result;

		BigDecimal bigDecimal= new BigDecimal("123.456780");

		result = NumberUtil.numberToString(null, 2);
		Assert.assertNull(result);

		expected = "123";
		result = NumberUtil.numberToString(bigDecimal, 0);
		Assert.assertEquals(expected, result);

		expected = "123.5";
		result = NumberUtil.numberToString(bigDecimal, 1);
		Assert.assertEquals(expected, result);

		expected = "123.46";
		result = NumberUtil.numberToString(bigDecimal, 2);
		Assert.assertEquals(expected, result);

		expected = "123.45678";
		result = NumberUtil.numberToString(bigDecimal, 5);
		Assert.assertEquals(expected, result);

		expected = "123.45678";
		result = NumberUtil.numberToString(bigDecimal, 8);
		Assert.assertEquals(expected, result);

		bigDecimal= new BigDecimal("12345");

		expected = "12345";
		result = NumberUtil.numberToString(bigDecimal, 0);
		Assert.assertEquals(expected, result);

		expected = "12345";
		result = NumberUtil.numberToString(bigDecimal, 1);
		Assert.assertEquals(expected, result);

		expected = "12345";
		result = NumberUtil.numberToString(bigDecimal, 2);
		Assert.assertEquals(expected, result);
	}

}
