/* Copyright (C) 2003 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id: CCUnitList.c,v 1.6 2003/10/05 11:29:59 tsutsumi Exp $
 */

/**
 * @file
 * Linked list module implementation.
 */
#include <ccunit/CCUnitList.h>

/** @addtogroup CCUnitList
 * @{
 */

/** @defgroup CCUnitListCell ListCell
 * Linked list cell.
 * @{
 */

/**
 * List Cell class.
 */
typedef struct CCUnitListCell
{
  struct CCUnitListCell* next;			/**< next cell pointer  */
  void* contents;				/**< list content object  */
} CCUnitListCell;

/**
 * create new list cell object.
 * @return new list cell object.
 */
static inline CCUnitListCell* ccunit_newListCell ()
{
  return calloc (1, sizeof (CCUnitListCell));
}

/**
 * delete list cell object.
 * @param cell deleting cell.
 */
static inline void ccunit_deleteListCell (CCUnitListCell* cell)
{
  safe_free (cell);
}

/** @} */

CCUnitList* ccunit_initList (CCUnitList* list)
{
  list->length = 0;
  list->head = NULL;
  list->tailp = &list->head;
  list->isAllocated = false;
  return list;
}

inline CCUnitList* ccunit_newList ()
{
  CCUnitList* newList = calloc (1, sizeof (*newList));
  if (!newList)
    return NULL;
  ccunit_initList (newList);
  newList->isAllocated = true;
  return newList;
}

void ccunit_addList (CCUnitList* list, void* contents)
{
  CCUnitListCell* cell;
  if (!list)
    return;
  cell = ccunit_newListCell ();
  if (!cell)
    return;
  cell->contents = contents;
  cell->next = NULL;
  *list->tailp = cell;
  list->tailp = &cell->next;
  list->length ++;
}

void ccunit_deleteList (CCUnitList* list, void (*deleteContents)(void*))
{
  CCUnitListCell* cell;
  while (list->head)
    {
      cell = list->head;
      if (deleteContents && cell->contents)
	deleteContents (cell->contents);
      list->head = cell->next;
      ccunit_deleteListCell (cell);
    }
  if (list->isAllocated)
    safe_free (list);
}

inline CCUnitListIterator* ccunit_initListIterator (const struct CCUnitList* list,
						    struct CCUnitListIterator* it)
{
  it->current = list->head;
  it->isAllocated = false;
  return it;
}

CCUnitListIterator* ccunit_newListIterator (const struct CCUnitList* list)
{
  CCUnitListIterator* it;
  if (!list)
    return NULL;
  it = calloc (1, sizeof (*it));
  if (!it)
    return NULL;
  ccunit_initListIterator (list, it);
  it->isAllocated = true;
  return it;
}

bool ccunit_hasNextListIterator (struct CCUnitListIterator* it)
{
  if (!it)
    return false;
  return it->current != NULL;
}

void* ccunit_nextListIterator (struct CCUnitListIterator* it)
{
  void* contents;
  if (!ccunit_hasNextListIterator (it))
    return NULL;
  contents = it->current->contents;
  it->current = it->current->next;
  return contents;
}

inline void ccunit_deleteListIterator (struct CCUnitListIterator* it)
{
  if (it && it->isAllocated)
    safe_free (it);
}

/** @} */
