/* Copyright (C) 2003, 2010 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/
/*
 * $Id$
 */
/**@file
 * WriteSuite module implementation.
 */

#include <ccunit/CCUnitMakeSuite.h>
#include <ccunit/CCUnitLogMessage.h>
#include <stdio.h>

/**
 * @ingroup CCUnitMakeSuite
 * @defgroup CCUnitPrintSuite _PrintSuite
 * 
 * @{
 */

/**
 * CCUnitTestType_t name string.
 */
static const char* ccunitTypeNames[] = {
  "ccunitTypeTest",				/**< Test class */
  "ccunitTypeSuite",				/**< TestSuite class */
  "ccunitTypeTestCase",				/**< TestCase class */
};

/**
 * print function prototype.
 *
 * @param ofp output stream.
 * @param type function type; ctor, dtor, setUp, tearDown, testCase.
 * @param fdef funcdef object to print.
 */
static void printPrototype (FILE* ofp, const char* type, _CCUnitFuncDef* fdef)
{
  fprintf (ofp,
	   "/* %s */\n"
	   "%s %s %s ();\n",
	   fdef->desc,
	   !fdef->scope ? "extern" : fdef->scope, fdef->type, fdef->name);
  ccunit_log ("%s: %s %s", type, fdef->type, fdef->name);  
}

/**
 * print test suite/test case prototypes.
 *
 * @param ofp output stream.
 * @param suitedef test suite object to print.
 */
static void printPrototypes (FILE* ofp, _CCUnitTestSuiteDef* suitedef)
{
  CCUnitListIterator itor;
  _CCUnitTestDef* testdef;
  ccunit_initListIterator (&suitedef->testdefs, &itor);
  while ((testdef = ccunit_nextListIterator (&itor)) != NULL)
    {
      if (testdef->type == ccunitTypeSuite)
	printPrototypes (ofp, (_CCUnitTestSuiteDef*)testdef);
      else if (testdef->type == ccunitTypeTestCase)
	{
	  _CCUnitTestCaseDef* tcdef = (_CCUnitTestCaseDef*)testdef;
	  ccunit_log ("tcdef: %s", tcdef->testdef.name);
	  if (tcdef->testdef.name)
	    fprintf (ofp, "/* test case: %s */\n", tcdef->testdef.name);
	  /* print testFunc prototypes */
	  {
	    CCUnitListIterator fitor;
	    _CCUnitFuncDef* funcdef;
	    ccunit_initListIterator (&tcdef->testFuncs, &fitor);
	    while ((funcdef = ccunit_nextListIterator (&fitor)) != NULL)
	      printPrototype (ofp, "testFunc", funcdef);
	  }
	}
    }
  fputc ('\n', ofp);
}

static inline void printTestFunc (FILE* ofp,
				  _CCUnitFuncDef* fndef)
{
  fprintf (ofp, "  {\n");
  if (!fndef)
    fprintf (ofp, "    NULL, NULL, NULL\n");
  else
    fprintf (ofp,
	     "    \"%s\",\n"
	     "    \"%s\",\n"
	     "    %s\n",
	     fndef->name,
	     fndef->desc,
	     fndef->name);
  fprintf (ofp, "  },\n");
}

/**
 * print test case adding function.
 *
 * @param ofp output stream.
 * @param tcdef test case to print.
 */
static void printTestCase  (FILE* ofp, _CCUnitTestCaseDef* tcdef)
{
  static int tcid = 0;
  CCUnitListIterator fnitor;
  _CCUnitFuncDef* fndef;
  if (!tcdef->testdef.idname)
    {
      tcdef->testdef.idname = calloc (1, 8);
      tcid ++;
      sprintf (tcdef->testdef.idname, "tc_%03d", tcid);
    }
  fprintf (ofp, "static CCUnitTestFunc %s_funcs[] = {\n", tcdef->testdef.idname);
  ccunit_initListIterator (&tcdef->testFuncs, &fnitor);
  while ((fndef = ccunit_nextListIterator (&fnitor)) != NULL)
    {
      printTestFunc (ofp, fndef);
    }
  printTestFunc (ofp, NULL);
  fputs ("};\n\n", ofp);
  fprintf (ofp,
	   "static CCUnitTestCaseDfn %s = {\n"
	   "  { %s },\n"
	   "  \"%s\",\n",
	   tcdef->testdef.idname,
	   ccunitTypeNames[tcdef->testdef.type],
	   !tcdef->testdef.name ? "NULL" : tcdef->testdef.name);
  fprintf (ofp, "  %s_funcs,\n", tcdef->testdef.idname);
  fputs ("};\n\n", ofp);
}

/**
 * print test suite adding function.
 *
 * @param ofp output stream.
 * @param name function name.
 * @param suite test suite to print.
 */
static void printSuite (FILE* ofp, const char* name, _CCUnitTestSuiteDef* suite)
{
  CCUnitListIterator itor;
  _CCUnitTestDef* testdef;
  static int suiteid = 0;
  if (!suite->testdef.idname)
    {
      suiteid ++;
      suite->testdef.idname = calloc (1, 12);
      sprintf (suite->testdef.idname, "suite_%03d", suiteid);
    }
  ccunit_initListIterator (&suite->testdefs, &itor);
  while ((testdef = ccunit_nextListIterator (&itor)) != NULL)
    {
      if (testdef->type == ccunitTypeSuite)
	printSuite (ofp, testdef->name, (_CCUnitTestSuiteDef*)testdef);
      else if (testdef->type == ccunitTypeTestCase)
	printTestCase (ofp, (_CCUnitTestCaseDef*)testdef);
      else
	;
    }
  fprintf (ofp, "static CCUnitTestDfn* %s_test[] = {\n", suite->testdef.idname);
  ccunit_initListIterator (&suite->testdefs, &itor);
  while ((testdef = ccunit_nextListIterator (&itor)) != NULL)
    {
      fprintf (ofp, "    &%s.test,\n", testdef->idname);
    }
  fputs ("    NULL,\n};\n\n", ofp);
  fprintf (ofp,
	   "static CCUnitTestSuiteDfn %s = {\n"
	   "  { %s },\n"
	   "  \"%s\",\n"
	   "  %s_test\n",
	   suite->testdef.idname,
	   ccunitTypeNames[suite->testdef.type],
	   !suite->testdef.name ? "" : suite->testdef.name,
	   suite->testdef.idname);
  fputs ("};\n\n", ofp);
}

/*
 * print test suite.
 */
void ccunit_printSuite (FILE* ofp, const char* name, _CCUnitTestSuiteDef* suite)
{
  fprintf (ofp,
	   "#include <ccunit/CCUnitTestSuite.h>\n"
	   "#include <ccunit/CCUnitTestCase.h>\n"
	   "#include <ccunit/CCUnitTestFunc.h>\n"
	   "\n");
  printPrototypes (ofp, suite);
  printSuite (ofp, NULL, suite);
  if (!name)
    name = "ccunit_suite";
  fprintf (ofp,
	   "\n"
	   "CCUnitTestSuite* %s (const char* name)\n"
	   "{\n"
	   "  if (!%s.name[0])\n"
	   "    %s.name = name;\n"
	   "  return ccunit_newTestSuiteFromDfn (&%s);\n"
	   "}\n",
	   name,
	   suite->testdef.idname, suite->testdef.idname, suite->testdef.idname);
}

/** @} */
