/*
 * client.c - ChaSen client
 *
 * Copyright (C) 1996-1997 Nara Institute of Science and Technology
 *
 * Author: A.Kitauchi <akira-k@is.aist-nara.ac.jp>, Apr. 1996
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <sys/wait.h>

#include "chalib.h"

/* max of the command line length */
#define CHA_COMM_LINE_MAX 1024

/*
 * check_status
 */
static void check_status(ifp, mes)
    FILE *ifp;
    char *mes;
{
    char line[CHA_INPUT_SIZE];

    fgets(line, sizeof(line), ifp);
    if (strncmp(line, "200 ", 4)) {
	if (mes == NULL)
	  fputs(line + 4, stderr);
	else
	  cha_exit(1, mes);
	exit(1);
    }
}

/*
 * do_chasen_client
 */
static void fp_copy(ofp, ifp)
    FILE *ofp, *ifp;
{
    char line[CHA_INPUT_SIZE];
    /* if output is stdout or not */
    int istty = isatty(fileno(stdout));

    while (cha_fgets(line, sizeof(line), ifp) != NULL) {
	if (line[0] == '.')
	  fputc('.', ofp);
	fputs(line, ofp);
	if (istty)
	  fflush(ofp);
    }

    if (ifp != stdin)
      fclose(ifp);
}

/*
 * send_chasenrc
 */
static void send_chasenrc(ifp, ofp)
    FILE *ifp, *ofp;
{
    /* If chasenrc file is "*", don't read it */
    if (!strcmp(get_chasenrc_path(), "*"))
      return;

    fputs("RC\n", ofp);
    fp_copy(ofp, fopen_chasenrc());
    fputs(".\n", ofp);
    fflush(ofp);

    check_status(ifp, NULL);
}

/*
 * escape_string
 */
static char *escape_string(dst_str, src_str)
    char *dst_str, *src_str;
{
    char *src, *dst;

    dst = dst_str;
    for (src = src_str; *src; src++) {
	if (*src == ' ' || *src == '"' || *src == '\'' || *src == '\\')
	  *dst++ = '\\';
	*dst++ = *src;
    }
    *dst = '\0';

    return dst_str;
}

/*
 * getopt_client
 */
static char *getopt_client(argv)
    char **argv;
{
    static char option[CHA_COMM_LINE_MAX];
    static char arg[CHA_COMM_LINE_MAX];
    char *op;
    int c;

    op = option;

    Cha_optind = 0;
    while ((c = cha_getopt_chasen(argv, stderr)) != EOF) {
	switch (c) {
	  case 'a':
	  case 's':
	  case 'D':
	  case 'P': break;
	  default:
	    if (Cha_optarg != NULL)
	      sprintf(op, "-%c %s ", c, escape_string(arg, Cha_optarg));
	    else
	      sprintf(op, "-%c ", c);
	    op += strlen(op);
	}
    }

    return option;
}

static int fork_and_gets(ifp)
    FILE *ifp;
{
    int pid;
    int istty, i;
    char line[CHA_INPUT_SIZE];

    if ((pid = fork()) < 0) {
	cha_perror("fork");
	return -1;
    }

    if (pid)
      return pid;

    /* child process */
    /* if output is stdout or not */
    istty = isatty(fileno(stdout));

    check_status(ifp, "error");

    while (fgets(line, sizeof(line), ifp) != NULL) {
	if (line[0] == '.' && 
	    (line[1] == '\n' || line[1] == '\r' && line[2] == '\n'))
	  break;
	fputs(line[0] == '.' ? line + 1 : line, stdout);
	if (istty)
	  fflush(stdout);
    }
    fclose(ifp);
    exit(0);
}

/*
 * close_connection
 */
static void close_connection(pid, ofp)
    int pid;
    FILE *ofp;
{
    int status;

    fputs(".\nQUIT\n", ofp);
    fclose(ofp);
    while (wait(&status) != pid)
      ;
}

/*
 * connect_server
 */
static int open_connection(server, port)
    char *server;
    unsigned short port;
{
    int sfd;
    struct sockaddr_in sin;
    struct hostent *host;

    if((sfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	cha_perror("socket");
	return -1;
    }

    memset(&sin, 0, sizeof(sin));
    sin.sin_port = htons(port);

    if ((host = gethostbyname(server)) != NULL) {
	memcpy(&sin.sin_addr.s_addr, host->h_addr, host->h_length);
	sin.sin_family = host->h_addrtype;
    } else if ((sin.sin_addr.s_addr = inet_addr(server)) != (unsigned long)-1) {
	sin.sin_family = AF_INET;
    } else {
#if 0
	cha_perror("inet_addr");
#endif
	cha_exit(-1, "Can't get address: %s\n", server);
	return -1;
    }

    if (connect(sfd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
	cha_perror("connect");
	close(sfd);
	return -1;
    }

    return sfd;
}

/*
 * chasen_client
 *
 * return code: exit code
 */
int chasen_client(argv, server, port)
    char **argv;
    char *server;
    int port;
{
    int pid, sfd;
    char *option;
    FILE *ifp, *ofp;

    /* open connection to server */
    if ((sfd = open_connection(server, port)) < 0)
      return 1;

    ifp = fdopen(sfd, "r");
    ofp = fdopen(sfd, "w");
    check_status(ifp, "connection error");

    send_chasenrc(ifp, ofp);

    /* send RUN command with option */
    option = getopt_client(argv);
    argv += Cha_optind;
    fprintf(ofp, "RUN %s\n", option);

    if ((pid = fork_and_gets(ifp)) < 0)
      return 1;

    if (*argv == NULL)
      fp_copy(ofp, stdin);
    else
      for (; *argv; argv++)
	fp_copy(ofp, cha_fopen(*argv, "r", 1));

    close_connection(pid, ofp);
    close(sfd);

    return 0;
}

