/*
 * client.c - ChaSen client
 *
 * Copyright (C) 1996-1997 Nara Institute of Science and Technology
 *
 * Author: A.Kitauchi <akira-k@is.aist-nara.ac.jp>, Apr. 1996
 *
 */

#ifndef NO_SERVER

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>

#if defined _WIN32 && ! defined __CYGWIN__
#include <winsock.h>
#include <windows.h>
#else
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/wait.h>
#endif

#include <string.h>
#include <signal.h>
#include <errno.h>

#include "chalib.h"

/* max of the command line length */
#define CHA_COMM_LINE_MAX 1024

#if defined _WIN32 && ! defined __CYGWIN__
#define fgets	fgets_sock

DWORD WINAPI fork_and_gets(LPVOID, FILE *);

/*
 * fgets_sock:  get strings not from file stream but from socket
 */
char *fgets_sock(char *buf, int size, int s)
{
    int i, ret;

    i = 0;
    for (i = 0; i < size - 1; i++) {
	ret = recv(s, buf + i, 1, 0);
	if (ret == 0) {
	    *(buf + i) = 0;
	    return (char *)NULL;
	}
	    
	if (*(buf + i) == '\n') {
	    i++;
	    break;
	}
    }
    *(buf + i) = 0;

    return buf;
}
#endif /* _WIN32 */


/*
 * check_status
 */
static void check_status(ifp, mes)
#if defined _WIN32 && ! defined __CYGWIN__
    SOCKET ifp;
#else
    FILE *ifp;
#endif /* _WIN32 */
    char *mes;
{
    char line[CHA_INPUT_SIZE];

    fgets(line, sizeof(line), ifp);
    if (strncmp(line, "200 ", 4)) {
	if (mes == NULL)
	  fputs(line + 4, stderr);
	else
	  cha_exit(1, mes);
	exit(1);
    }
}

/*
 * do_chasen_client
 */
static void fp_copy(ofp, ifp)
#if defined _WIN32 && ! defined __CYGWIN__
    SOCKET ofp;
    FILE *ifp;
#else
    FILE *ofp, *ifp;
#endif
{
    char line[CHA_INPUT_SIZE];
    /* whether output is stdout or not */
    int istty = isatty(fileno(stdout));

    /* sizeof(line)-1: space for "\n" at the end of line */
    while (cha_fgets(line, sizeof(line)-1, ifp) != NULL) {
	int len = strlen(line);
	if (line[len-1] != '\n') {
	    line[len] = '\n';
	    line[len+1] = '\0';
	}
#if defined _WIN32 && ! defined __CYGWIN__
	if (line[0] == '.')
	  send(ofp, ".", 1, 0);
	send(ofp, line, strlen(line), 0);
#else
	if (line[0] == '.')
	  fputc('.', ofp);
	fputs(line, ofp);
	if (istty)
	  fflush(ofp);
#endif /* _WIN32 */
    }

    if (ifp != stdin)
      fclose(ifp);
}

/*
 * send_chasenrc
 */
static void send_chasenrc(ifp, ofp)
#if defined _WIN32 && ! defined __CYGWIN__
    SOCKET ifp, ofp;
#else
    FILE *ifp, *ofp;
#endif /* _WIN32 */
{
    /* If chasenrc file is "*", don't read it */
    if (!strcmp(cha_get_rcpath(), "*"))
      return;

#if defined _WIN32 && ! defined __CYGWIN__
    send(ofp, "RC\n", 3, 0);
    fp_copy(ofp, cha_fopen_rcfile());
    send(ofp, ".\n", 2, 0);
#else
    fputs("RC\n", ofp);
    fp_copy(ofp, cha_fopen_rcfile());
    fputs(".\n", ofp);
    fflush(ofp);
#endif /* _WIN32 */

    check_status(ifp, NULL);
}

/*
 * escape_string
 */
static char *escape_string(dst_str, src_str)
    char *dst_str, *src_str;
{
    char *src, *dst;

    dst = dst_str;
    for (src = src_str; *src; src++) {
	if (*src == ' ' || *src == '"' || *src == '\'' || *src == '\\')
	  *dst++ = '\\';
	*dst++ = *src;
    }
    *dst = '\0';

    return dst_str;
}

/*
 * getopt_client
 */
static char *getopt_client(argv)
    char **argv;
{
    static char option[CHA_COMM_LINE_MAX];
    static char arg[CHA_COMM_LINE_MAX];
    char *op;
    int c;

    op = option;

    Cha_optind = 0;
    while ((c = cha_getopt_chasen(argv, stderr)) != EOF) {
	switch (c) {
	  case 'a':
	  case 's':
	  case 'D':
	  case 'P': break;
	  default:
	    if (Cha_optarg != NULL)
	      sprintf(op, "-%c %s ", c, escape_string(arg, Cha_optarg));
	    else
	      sprintf(op, "-%c ", c);
	    op += strlen(op);
	}
    }

    return option;
}

#if defined _WIN32 && ! defined __CYGWIN__
DWORD WINAPI fork_and_gets(LPVOID s, FILE *output)
{
    SOCKET ifp;
#else
static int fork_and_gets(ifp, output)
    FILE *ifp, *output;
{
#endif /* _WIN32 */

    int pid;
    int istty;
    char line[CHA_INPUT_SIZE];

#if defined _WIN32 && ! defined __CYGWIN__
    ifp = (SOCKET)s;
#else
    if ((pid = fork()) < 0) {
	cha_perror("fork");
	return -1;
    }

    if (pid)
      return pid;
#endif /* _WIN32 */

    /* child process */
    /* if output is stdout or not */
    istty = output == stdout && isatty(fileno(stdout));

    check_status(ifp, "error");

    while (fgets(line, sizeof(line), ifp) != NULL) {
	if (line[0] == '.' && 
	    (line[1] == '\n' || (line[1] == '\r' && line[2] == '\n')))
	  break;
#if defined _WIN32 && ! defined __CYGWIN__
	euc2sjis(line);
#endif /* _WIN32 */
	fputs(line[0] == '.' ? line + 1 : line, output);
	if (istty)
	  fflush(output);
    }
#if ! (defined _WIN32 && ! defined __CYGWIN__)
    fclose(ifp);
    exit(0);
#endif /* NOT _WIN32 */
}

/*
 * close_connection
 */
static void close_connection(pid, ofp)
#if defined _WIN32 && ! defined __CYGWIN__
    int pid;
    SOCKET ofp;
#else
    int pid;
    FILE *ofp;
#endif /* _WIN32 */
{
    int status;

#if defined _WIN32 && ! defined __CYGWIN__
    send(ofp, ".\nQUIT\n", 7, 0);
#else
    fputs(".\nQUIT\n", ofp);
    fclose(ofp);
    while (wait(&status) != pid)
      ;
#endif	/* _WIN32 */
}

/*
 * connect_server
 */
#if defined _WIN32 && ! defined __CYGWIN__
static SOCKET open_connection(server, port)
#else
static int open_connection(server, port)
#endif /* _WIN32 */
    char *server;
    unsigned short port;
{
#if defined _WIN32 && ! defined __CYGWIN__
    SOCKET sfd;
#else
    int sfd;
#endif /* _WIN32 */
    struct sockaddr_in sin;
    struct hostent *host;
#if defined _WIN32 && ! defined __CYGWIN__    
    WORD wVersionRequested;
    WSADATA wsaData;
    int err;

    wVersionRequested = MAKEWORD(1, 1);
    err = WSAStartup(wVersionRequested, &wsaData);
#endif

#if defined _WIN32 && ! defined __CYGWIN__
    if((sfd = socket(AF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET) {
#else
    if((sfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
#endif /* _WIN32 */
	cha_perror("socket");
	return -1;
    }

    memset(&sin, 0, sizeof(sin));
    sin.sin_port = htons(port);

    if ((host = gethostbyname(server)) != NULL) {
	memcpy(&sin.sin_addr.s_addr, host->h_addr, host->h_length);
	sin.sin_family = host->h_addrtype;
    } else if ((sin.sin_addr.s_addr = inet_addr(server)) != (unsigned long)-1) {
	sin.sin_family = AF_INET;
    } else {
#if 0
	cha_perror("inet_addr");
#endif
	cha_exit(-1, "Can't get address: %s\n", server);
	return -1;
    }

    if (connect(sfd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
	cha_perror("connect");
#if defined _WIN32 && ! defined __CYGWIN__
	closesocket(sfd);
#else
	close(sfd);
#endif /* _WIN32 */
	return -1;
    }

    return sfd;
}

/*
 * chasen_client
 *
 * return code: exit code
 */
int chasen_client(argv, output, server, port)
    char **argv;
    FILE *output;
    char *server;
    int port;
{
    char *option;
#if defined _WIN32 && ! defined __CYGWIN__
    SOCKET pid, sfd;
    SOCKET ifp, ofp;
    char tmp[512];

    HANDLE hThrd;
    DWORD threadId;
#else
    int pid, sfd;
    FILE *ifp, *ofp;
#endif

    /* open connection to server */
    if ((sfd = open_connection(server, port)) < 0)
      return 1;

#if defined _WIN32 && ! defined __CYGWIN__
    ifp = sfd;
    ofp = sfd;
#else
    ifp = fdopen(sfd, "r");
    ofp = fdopen(sfd, "w");
#endif
    check_status(ifp, "connection error");

    send_chasenrc(ifp, ofp);

    /* send RUN command with option */
    option = getopt_client(argv);
    argv += Cha_optind;
#if defined _WIN32 && ! defined __CYGWIN__
    sprintf(tmp, "RUN %s\n", option);
    send(ofp, tmp, strlen(tmp), 0);

    hThrd = CreateThread(NULL, 0, fork_and_gets, (LPVOID)ifp, 0, &threadId);

#else
    fprintf(ofp, "RUN %s\n", option);

    if ((pid = fork_and_gets(ifp, output)) < 0)
      return 1;

#endif /* _WIN32 */

    if (*argv == NULL)
      fp_copy(ofp, stdin);
    else
      for (; *argv; argv++)
	fp_copy(ofp, cha_fopen(*argv, "r", 1));

    close_connection(pid, ofp);
#if defined _WIN32 && ! defined __CYGWIN__
    WaitForSingleObject(hThrd, INFINITE);

    closesocket(ofp);
    closesocket(sfd);
#else
    close(sfd);
#endif

    return 0;
}

#endif /* !NO_SERVER */
