/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.authn.filter.saml2;

import java.util.Date;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.log.CitkLogger;
import jp.co.ogis_ri.citk.common.util.DataUtil;

import org.apache.commons.httpclient.Cookie;
import org.apache.commons.lang.StringUtils;
import org.mule.api.MuleMessage;
import org.mule.api.routing.filter.Filter;
import org.mule.api.transport.PropertyScope;
import org.mule.transport.http.HttpConnector;

/**
 * Mule ESBの認証Cookie確認用カスタムフィルタークラス。
 * 
 * @author toyota
 * 
 */
public class AccessFilter implements Filter {

    /**
     * ログメッセージ出力用オブジェクト。
     */
    private static final CitkLogger logger =
            CitkLogger.getLog(AccessFilter.class);

    /**
     * 操作ログメッセージ出力用オブジェクト。
     */
    private static final CitkLogger opLogger = CitkLogger.getOperationLog();

    /**
     * アイドルタイムアウト判定用スパン(msec)
     * 
     */
    private long idleTime = 0;

    /**
     * @return idleTime
     */
    public long getIdleTime() {
        return idleTime;
    }

    /**
     * @param idleTime セットする idleTime
     */
    public void setIdleTime(long idleTime) {
        this.idleTime = idleTime;
    }

    @Override
    /**
     * 認証済みかどうか判定するMuleのフィルタメソッド。
     *
     * @param message 判定対象となるMuleMessage
     * @return boolean true:認証済み、false:未認証
     */
    public boolean accept(MuleMessage message) {

        String messageID = message.getMessageRootId();

        Cookie[] cookies =
                (Cookie[]) message.getInboundProperty(HttpConnector.HTTP_COOKIES_PROPERTY);

        String nameId =
                DataUtil.getCookieValue(cookies, CitkConstants.NAMEID_KEY);

        // nameIdのCookieの有無
        if (nameId == null || nameId.isEmpty()) {
            opLogger.info("I-0102", messageID, CitkConstants.NAMEID_KEY);
            return false;
        }
        logger.debug("nameId : " + nameId);

        // IdPEntityIDのCookieの有無
        String cookie_idpEntityId =
                DataUtil.getCookieValue(cookies, CitkConstants.IDP_ENTITYID_KEY);

        if (cookie_idpEntityId == null || cookie_idpEntityId.isEmpty()) {
            opLogger.info("I-0102", messageID, CitkConstants.IDP_ENTITYID_KEY);
            return false;
        }

        String param_idpEntityId =
                message.getInboundProperty(CitkConstants.IDP_ENTITYID_KEY);
        boolean SetIdpEntityID = true;
        if (param_idpEntityId == null || param_idpEntityId.isEmpty()) {
            SetIdpEntityID = false;
        }

        // expireのCookieの検証
        String expireString =
                DataUtil.getCookieValue(cookies,
                        CitkConstants.CITK_EXPIRE_COOKIE_NAME);
        if (expireString == null || expireString.isEmpty()) {
            opLogger.info("I-0102", messageID,
                    CitkConstants.CITK_EXPIRE_COOKIE_NAME);
            return false;
        } else {
            Long l = Long.valueOf(expireString);
            if (l.longValue() != 0) {
                Date expire = new Date(l);
                Date nowDate = new Date();
                if (nowDate.after(expire)) {
                    // タイムアウト
                    message.setProperty(CitkConstants.CITK_ISTIMEOUT_KEY,
                            "true", PropertyScope.SESSION);
                    if (!SetIdpEntityID) {
                        // パラメータでIdP指定がなければ元の認証先への再認証をさせる
                        message.setProperty(CitkConstants.IDP_ENTITYID_KEY,
                                cookie_idpEntityId, PropertyScope.INBOUND);
                    }
                    return false;
                }
            }
        }

        // AccessDateのCookieの検証
        String accessDateString =
                DataUtil.getCookieValue(cookies,
                        CitkConstants.CITK_ACCESS_DATE_COOKIE_NAME);
        if (accessDateString == null || accessDateString.isEmpty()) {
            opLogger.info("I-0102", messageID,
                    CitkConstants.CITK_ACCESS_DATE_COOKIE_NAME);
            return false;
        } else if (idleTime != 0) { // idleTimeが0のときは判定無し
            Long l = Long.valueOf(accessDateString);
            Date expire = new Date(l + idleTime);
            Date nowDate = new Date();
            if (nowDate.after(expire)) {
                // タイムアウト
                message.setProperty(CitkConstants.CITK_ISTIMEOUT_KEY,
                        Boolean.TRUE, PropertyScope.SESSION);
                if (!SetIdpEntityID) {
                    // パラメータでIdP指定がなければ元の認証先への再認証をさせる
                    message.setProperty(CitkConstants.IDP_ENTITYID_KEY,
                            cookie_idpEntityId, PropertyScope.INBOUND);
                }
                return false;
            }
        }

        // 認証Cookieが揃っていても、パラメータが指すIdPがCookieのIdPと異なる場合には、パラメータ指定のIdPで再認証させる
        if (SetIdpEntityID) {
            if (!cookie_idpEntityId.equals(param_idpEntityId)) {
                return false;
            }
        }

        opLogger.info("I-0101", messageID);
        for (int i = 0; i < cookies.length; i++) {
            opLogger.info("[" + messageID + "]    " + cookies[i].toString());
        }

        String authzRequestParam =
                message.getInboundProperty(CitkConstants.AUTHZ_REQUEST_PARAM);
        // 認可の有無
        if (StringUtils.equalsIgnoreCase(authzRequestParam, "false")) {
            message.setProperty(CitkConstants.AUTHZ_REQUEST_PARAM, "false",
                    PropertyScope.OUTBOUND);
            opLogger.info("I-0103", messageID);
        } else {
            message.setSessionProperty(CitkConstants.NAMEID_KEY,
                    nameId.replace("/", "%2F"));
        }

        return true;
    }

}
