/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.authn.transformer.http;

import java.util.Date;
import java.util.TimeZone;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.log.CitkLogger;

import org.apache.commons.httpclient.Cookie;
import org.mule.api.MuleMessage;
import org.mule.api.transformer.TransformerException;
import org.mule.api.transport.PropertyScope;
import org.mule.transformer.AbstractMessageTransformer;
import org.mule.transport.http.HttpConstants;

/**
 * Mule ESBの認証Cookie発行用トランスフォーマクラス。
 * 
 * @author ISP Shiraishi
 * 
 */
public class AuthnSetCookiesTransformer extends AbstractMessageTransformer {

    /**
     * ログメッセージ出力用ロガート。
     */
    private static final CitkLogger logger =
            CitkLogger.getLog(AuthnSetCookiesTransformer.class);

    /**
     * 発行cookieのタイムゾーン。
     */
    private String timezone = null;

    /**
     * 発行cookieの、発行時刻からexpireまでの間隔（ミリ秒）。
     */
    private long expire = 0;

    /**
     * 発行cookieのドメイン。
     */
    private String domain = null;

    /**
     * 発行cookieのタイムゾーンを取得定する。
     * 
     * @return timezone　 発行cookieのタイムゾーン
     */
    public String getTimezone() {
        return timezone;
    }

    /**
     * 発行cookieのタイムゾーンを設定する。
     * 
     * @param timezone  発行cookieのタイムゾーン。
     */
    public void setTimezone(String timezone) {
        this.timezone = timezone;
    }

    /**
     * 有効期限を取得する。
     * 
     * @return 有効期限。
     */
    public long getExpire() {
        return expire;
    }

    /**
     * 有効期限を設定する。
     * 
     * @param expire 有効期限。
     */
    public void setExpire(long expire) {
        this.expire = expire;
    }

    /**
     * 発行cookieのドメインを取得する。
     * 
     * @return 発行cookieのドメイン。
     */
    public String getDomain() {
        return domain;
    }

    /**
     * 発行cookieのドメインを設定する。
     * 
     * @param domain 発行cookieのドメイン。
     */
    public void setDomain(String domain) {
        this.domain = domain;
    }

    /**
     * コンストラクタ。
     */
    public AuthnSetCookiesTransformer() {
        super();
    }

    /**
     * 認証判定用のCookieをセットするMuleのトランスﾌｫｰﾏ。
     * 
     * @param message 変換対象のMuleメッセージ
     * @param outputEncoding エンコード。
     * @return 変換後のMuleメッセージ。
     * @throws TransformerException
     */
    @Override
    public MuleMessage transformMessage(MuleMessage message,
            String outputEncoding) throws TransformerException {

        TimeZone.setDefault(TimeZone.getTimeZone(timezone));
        Date expireDate = getExpireDate();
        String nameId = message.getSessionProperty(CitkConstants.NAMEID_KEY);
        String idpEntityId =
                message.getSessionProperty(CitkConstants.IDP_ENTITYID_KEY);
        String expireString = "" + expireDate.getTime();
        String accessDate = "" + getNowTime().getTime();

        Cookie[] cookies = new Cookie[4];
        cookies[0] =
                createCookie(domain, CitkConstants.NAMEID_KEY, nameId, null);
        cookies[1] =
                createCookie(domain, CitkConstants.IDP_ENTITYID_KEY,
                        idpEntityId, null);
        cookies[2] =
                createCookie(domain, CitkConstants.CITK_EXPIRE_COOKIE_NAME,
                        expireString, null);
        cookies[3] =
                createCookie(domain,
                        CitkConstants.CITK_ACCESS_DATE_COOKIE_NAME, accessDate,
                        null);
        message.setProperty(HttpConstants.HEADER_COOKIE_SET, cookies,
                PropertyScope.INBOUND);

        if (logger.isDebugEnabled()) {
            for (Cookie cookie : cookies) {
                logger.debug("COOKIE: " + cookie.toString());
            }
        }
        return message;
    }

    /**
     * 有効期限時刻を取得する。
     * 
     * @return 有効期限時刻のDate。
     */
    private Date getExpireDate() {
        Date date = getNowTime();
        Date expireDate;
        if (expire == 0) {
            expireDate = new Date(expire);
        } else {
            expireDate = new Date(date.getTime() + expire);
        }
        return expireDate;
    }

    /**
     * 現在時刻を取得する。
     * 
     * @return 現在時刻のDate。
     */
    private Date getNowTime() {
        return new Date();
    }

    /**
     * Cookieを生成する。
     * 
     * @param domain ドメイン。
     * @param cookieName Cookie名称。
     * @param value Cookie値。
     * @param expires 有効期限。
     * @return 生成したcCokie。
     */
    private Cookie createCookie(String domain, String cookieName, String value,
            Date expires) {
        return new Cookie(domain, cookieName, value, "/", expires, false);
    }
}
