/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.authn.transformer.http;

import java.util.Date;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.exception.CitkSystemException;
import jp.co.ogis_ri.citk.common.log.CitkLogger;
import jp.co.ogis_ri.citk.common.util.RequestParameterCache;

import org.apache.commons.httpclient.Cookie;
import org.mule.api.MuleMessage;
import org.mule.api.transformer.TransformerException;
import org.mule.api.transport.PropertyScope;
import org.mule.transformer.AbstractMessageTransformer;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;

/**
 * Mule ESBのリダイレクト処理トランスフォーマークラス。
 * 
 * @author ISP Shiraishi
 * 
 */
public class RedirectionTransformer extends AbstractMessageTransformer {

    /**
     * ログメッセージ出力用ロガー。
     */
    private static final CitkLogger logger =
            CitkLogger.getLog(RedirectionTransformer.class);

    /**
     * リクエストプロトコル。
     */
    private String requestProtocol = null;

    /**
     * リクエストプロトコルを取得する。
     * 
     * @return requestProtocol リクエストプロトコル。
     */
    public String getRequestProtocol() {
        return requestProtocol;
    }

    /**
     * リクエストプロトコルを取得する。
     * 
     * @param requestProtocol リクエストプロトコル。
     */
    public void setRequestProtocol(String requestProtocol) {
        this.requestProtocol = requestProtocol;
    }

    /**
     * リクエストMuleMessgaeを収めたキャッシュ。
     * 
     * @see RequestParameterCache
     */
    private RequestParameterCache cache = null;

    /**
     * 発行Cookieのドメイン。
     */
    private String domain = null;

    /**
     * IdPの認証Cookie名称。
     */
    private String idpAuthnCookieName = null;

    /**
     * リクエストMuleMessgaeを収めたキャッシュを取得する。
     *  
     * @return リクエストMuleMessgaeを収めたキャッシュ。
     */
    public RequestParameterCache getCache() {
        return cache;
    }

    /**
     * リクエストMuleMessgaeを収めたキャッシュ設定する。
     * 
     * @param cache リクエストMuleMessgaeを収めたキャッシュ。
     */
    public void setCache(RequestParameterCache cache) {
        this.cache = cache;
    }

    /**
     * 発行Cookieのドメインを取得する。
     * 
     * @return 発行Cookieのドメイン。
     */
    public String getDomain() {
        return domain;
    }

    /**
     * 発行Cookieのドメインを設定する。
     * 
     * @param domain 発行Cookieのドメイン。
     */
    public void setDomain(String domain) {
        this.domain = domain;
    }

    /**
     * IdPの認証Cookie名称を取得する。
     * 
     * @return IdPの認証Cookie名称。
     */
    public String getIdpAuthnCookieName() {
        return idpAuthnCookieName;
    }

    /**
     * IdPの認証Cookie名称を設定する。
     * 
     * @param idpAuthnCookieName IdPの認証Cookie名称。
     */
    public void setIdpAuthnCookieName(String idpAuthnCookieName) {
        this.idpAuthnCookieName = idpAuthnCookieName;
    }

    /**
     * コンストラクタ。
     */
    public RedirectionTransformer() {
        super();
    }

    /**
     * リダイレクト処理を行う。
     * 
     * @param message 変換対象のMuleMessage。
     * @param outputEncoding エンコード。
     * @return 変換後のMuleメッセージ。
     * @throws TransformerException。
     */
    @Override
    public Object transformMessage(MuleMessage message, String outputEncoding)
            throws TransformerException {
        String acsStatus =
                message.getOutboundProperty(CitkConstants.CITK_ACS_STATUS_KEY);
        String location = null;
        if (acsStatus == null) {
            location =
                    message.getInboundProperty(HttpConstants.HEADER_LOCATION);
        } else if (acsStatus.equals(CitkConstants.CITK_ACS_STATUS_OK)) {
            // RelayStateをリダイレクトのGetパラメータとして付与
            String relayState =
                    message.getSessionProperty(CitkConstants.CITK_RELAYSTATE_KEY);
            MuleMessage originalMessage = cache.get(relayState);
            location = buildLocation(originalMessage, relayState);
        } else if (acsStatus.equals(CitkConstants.CITK_ACS_STATUS_NG)) {
            // 認証Cookieのクリア
            clearAuthnCookies(message);
            location =
                    message.getInboundProperty(HttpConstants.HEADER_LOCATION);
            String relayState =
                    message.getSessionProperty(CitkConstants.CITK_RELAYSTATE_KEY);
            if (location.contains("?")) {
                location =
                        location + "&" + CitkConstants.RELAYSTATE_PARAM + "="
                                + relayState;
            } else {
                location =
                        location + "?" + CitkConstants.RELAYSTATE_PARAM + "="
                                + relayState;
            }
        } else {
            throw new CitkSystemException(
                    CitkConstants.ERROR_MSG_INVALID_ACSSTATUS, message);
        }

        if (location != null) {
            message.setOutboundProperty(HttpConnector.HTTP_STATUS_PROPERTY,
                    HttpConstants.SC_MOVED_TEMPORARILY);
            message.setOutboundProperty(HttpConstants.HEADER_LOCATION, location);
        } else {
            throw new CitkSystemException(
                    CitkConstants.ERROR_MSG_LOCATION_IS_NULL, message);
        }
        logger.debug("acs status: " + acsStatus);
        logger.debug("location: " + location);
        return message;
    }

    /**
     * リダイレクト先(Location)を生成する。
     * 
     * @param originalMessage 変換対象のMuleメッセージ。
     * @param relayState Locationのパラメータに含めるRelayState値。
     * @return Location文字列。
     */
    private String buildLocation(MuleMessage originalMessage, String relayState) {
        String location;

        String host =
                originalMessage.getInboundProperty(HttpConstants.HEADER_HOST);
        String path =
                originalMessage.getInboundProperty(HttpConnector.HTTP_REQUEST_PATH_PROPERTY);
        location =
                this.requestProtocol + host + path + "?"
                        + CitkConstants.RELAYSTATE_PARAM + "=" + relayState;
        return location;
    }

    /**
     * MuleMessageへ認証Cookieを除去するパラメータをセットする。
     * 
     * @param message Muleメッセージ。
     * 
     */
    private void clearAuthnCookies(MuleMessage message) {
        String dummyValue = "";
        Date deleteExpire = new Date(0);
        int cookiesNum = 4;

        // タイムアウトの場合、IdpのSSO認証Cookieも削除
        Boolean isTimeOut = null;
        isTimeOut =
                message.getSessionProperty(CitkConstants.CITK_ISTIMEOUT_KEY);
        if (isTimeOut != null && isTimeOut.booleanValue()) {
            cookiesNum = 5;
        }

        Cookie[] cookies = new Cookie[cookiesNum];

        for (int i = 0; i < cookiesNum; i++) {

            switch (i) {
            case 0:
                cookies[i] =
                        new Cookie(domain, CitkConstants.NAMEID_KEY,
                                dummyValue, "/", deleteExpire, false);
                break;
            case 1:
                cookies[i] =
                        new Cookie(domain, CitkConstants.IDP_ENTITYID_KEY,
                                dummyValue, "/", deleteExpire, false);
                break;
            case 2:
                cookies[i] =
                        new Cookie(domain,
                                CitkConstants.CITK_EXPIRE_COOKIE_NAME,
                                dummyValue, "/", deleteExpire, false);
                break;
            case 3:
                cookies[i] =
                        new Cookie(domain,
                                CitkConstants.CITK_ACCESS_DATE_COOKIE_NAME,
                                dummyValue, "/", deleteExpire, false);
                break;
            case 4:
                cookies[i] =
                        new Cookie(domain, idpAuthnCookieName, dummyValue, "/",
                                deleteExpire, false);
                break;
            default:
                break;
            }
        }

        message.setProperty(HttpConstants.HEADER_COOKIE_SET, cookies,
                PropertyScope.INBOUND);
    }
}
