/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.authn.transformer.http;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.exception.IllegalRequestException;
import jp.co.ogis_ri.citk.common.log.CitkLogger;
import jp.co.ogis_ri.citk.common.util.RequestParameterCache;

import org.mule.api.MuleMessage;
import org.mule.api.transformer.TransformerException;
import org.mule.transformer.AbstractMessageTransformer;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;
import org.mule.util.StringUtils;

/**
 * Mule ESBのカスタムトランスフォーマークラス。
 * 
 * @author ISP Shiraishi
 * 
 */
public class RequestParameterTransformer extends AbstractMessageTransformer {

    /**
     * ログメッセージ出力用ロガー。
     */
    private static final CitkLogger logger =
            CitkLogger.getLog(RequestParameterTransformer.class);

    /**
     * リクエストMuleMessgaeを収めたキャッシュ。
     * 
     * @see SubjectCache
     */
    private RequestParameterCache cache = null;

    /**
     * リクエストMuleMessgaeを収めたキャッシュを取得する。
     * 
     * @return RequestParameterCache　リクエストMuleMessgaeを収めたキャッシュ。
     */
    public RequestParameterCache getCache() {
        return cache;
    }

    /**
     * リクエストMuleMessgaeを収めたキャッシュを設定する。
     * 
     * @param cache リクエストMuleMessgaeを収めたキャッシュ。
     */
    public void setCache(RequestParameterCache cache) {
        this.cache = cache;
    }

    /**
     * コンストラクタ。
     */
    public RequestParameterTransformer() {
        super();
    }

    /**
     * リクエストパラメータの変換を行う。 Getパラメータに"RelayState"が含まれていた場合、認証後の再呼び出しと判定し
     * キャッシュに保持された元のリクエストメソッド・パラメータを付与する。
     * 
     * @param message 変換対象のMuleメッセージ。
     * @param outputEncoding エンコード。
     * @return 変換後のMuleメッセージ。
     * @throws TransformerException。
     */
    @Override
    public Object transformMessage(MuleMessage message, String outputEncoding)
            throws TransformerException {

        MuleMessage originalMessage = null;
        String relayState =
                message.getInboundProperty(CitkConstants.RELAYSTATE_PARAM);
        // RelayStateパラメータが含まれていた場合、キャッシュから元のメッセージを引き出す
        if (relayState != null) {
            originalMessage = cache.get(relayState);
            if (originalMessage == null) {
                throw new IllegalRequestException("E-0002", message);
            }
            cache.remove(relayState);
        } else {
            originalMessage = message;
        }

        // 必要な値をOutboundパラメータにセット
        for (String name : originalMessage.getInboundPropertyNames()) {
            Object inValue = originalMessage.getInboundProperty(name);
            if (inValue != null) {
                if (!name.equals(HttpConstants.HEADER_ACCEPT_ENCODING)) {
                    // CITK用パラメータの除去
                    if (name.equals(HttpConnector.HTTP_REQUEST_PROPERTY)) {
                        String path =
                                deleteUrlParam((String) inValue,
                                        CitkConstants.IDP_ENTITYID_KEY);
                        path =
                                deleteUrlParam(path,
                                        CitkConstants.RELAYSTATE_PARAM);
                        path =
                                deleteUrlParam(path,
                                        CitkConstants.AUTHZ_REQUEST_PARAM);
                        message.setOutboundProperty(name, path);
                    } else {
                        message.setOutboundProperty(name, inValue);
                    }
                }
            }
        }
        message.setOutboundProperty(HttpConnector.HTTP_COOKIES_PROPERTY,
                message.getInboundProperty(HttpConnector.HTTP_COOKIES_PROPERTY));
        message.setPayload(originalMessage.getPayload());
        return message;
    }

    /**
     * URLから特定のパラメータを除去する。
     * 
     * @param url 対象URL文字列。
     * @param key 削除対象のキー文字列。
     * @return 削除後URL。
     */
    protected String deleteUrlParam(String url, String key) {

        logger.debug("url : {0}, query key : {1} ", url, key);

        if (StringUtils.isEmpty(url) || StringUtils.isEmpty(key)) {
            return url;
        }

        StringBuilder returnUrl = new StringBuilder(url);
        // パラメータ部分のインデックスを取得
        int st = url.indexOf("?");
        if (st >= 0) {
            String queryString = url.substring(st + 1);
            returnUrl = new StringBuilder(url.substring(0, st));
            String[] params = queryString.split("&");
            // パラメータを判定
            for (int i = 0; i < params.length; i++) {
                String param = params[i];
                // keyとの判定
                if (!param.startsWith(key + "=")) {

                    if (returnUrl.indexOf("?") < 0
                            && !returnUrl.toString().endsWith("?")) {
                        returnUrl.append("?");
                    } else if (!returnUrl.toString().endsWith("&")) {
                        returnUrl.append("&");
                    }
                    returnUrl.append(param);
                }
            }
        }

        return returnUrl.toString();
    }
}
