/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.authn.transformer.http;

import java.util.Date;
import java.util.TimeZone;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.log.CitkLogger;

import org.apache.commons.httpclient.Cookie;
import org.mule.api.MuleMessage;
import org.mule.api.transformer.TransformerException;
import org.mule.api.transport.PropertyScope;
import org.mule.transformer.AbstractMessageTransformer;
import org.mule.transport.http.HttpConstants;

/**
 * Mule ESBの認証Cookie発行用トランスフォーマ。
 * 
 * @author ISP Shiraishi
 * 
 */
public class UpdateAccessCookieTransformer extends AbstractMessageTransformer {

    /**
     * ログメッセージ出力用ロガー。
     */
    private static final CitkLogger logger =
            CitkLogger.getLog(UpdateAccessCookieTransformer.class);

    /**
     * 発行Cookieのタイムゾーン。
     */
    private String timezone = null;

    /**
     * 発行Cookieのドメイン。
     */
    private String domain = null;

    /**
     * 発行Cookieのタイムゾーンを取得する。
     * 
     * @return 発行Cookieのタイムゾーン。 
     */
    public String getTimezone() {
        return timezone;
    }

    /**
     * 発行Cookieのタイムゾーンを設定する。
     * 
     * @param timezone 発行Cookieのタイムゾーン。
     */
    public void setTimezone(String timezone) {
        this.timezone = timezone;
    }

    /**
     * 発行Cookieのタイムゾーンを取得する。
     * 
     * @return 発行Cookieのタイムゾーン。
     */
    public String getDomain() {
        return domain;
    }

    /**
     * 発行Cookieのタイムゾーンを設定する。
     * 
     * @param domain 発行Cookieのタイムゾーン。
     */
    public void setDomain(String domain) {
        this.domain = domain;
    }

    /**
     * コンストラクタ.
     */
    public UpdateAccessCookieTransformer() {
        super();
    }

    /**
     * 認証判定用の最新アクセス時刻CookieをセットするMuleのトランスフォーマメソッド。 認可結果がPermitの場合のみ更新。
     * 
     * @param message 変換対象のMuleメッセージ。
     * @param outputEncoding エンコード。
     * @return 変換後のMuleメッセージ。
     * @throws TransformerException。
     */
    @Override
    public MuleMessage transformMessage(MuleMessage message,
            String outputEncoding) throws TransformerException {

        String authzResult =
                message.getSessionProperty(CitkConstants.AUTHZ_RESULT_KEY);

        if (CitkConstants.AUTHZ_PERMIT.equals(authzResult)) {
            TimeZone.setDefault(TimeZone.getTimeZone(timezone));

            String accessDate = "" + getNowTime().getTime();

            Cookie[] cookies = new Cookie[1];
            cookies[0] =
                    createCookie(domain,
                            CitkConstants.CITK_ACCESS_DATE_COOKIE_NAME,
                            accessDate, null);
            message.setProperty(HttpConstants.HEADER_COOKIE_SET, cookies,
                    PropertyScope.INBOUND);

            if (logger.isDebugEnabled()) {
                for (Cookie cookie : cookies) {
                    logger.debug("COOKIE: " + cookie.toString());
                }
            }
        }

        return message;
    }

    /**
     * Cookieを生成する。
     * 
     * @param domain  発行Cookieドメイン。
     * @param cookieName  発行Cookie名称。
     * @param value  発行Cookie値。
     * @param expires 有効期限。
     * @return 生成したCookie。
     */
    private Cookie createCookie(String domain, String cookieName, String value,
            Date expires) {
        return new Cookie(domain, cookieName, value, "/", expires, false);
    }

    /**
     * 現在時刻を取得する。
     * 
     * @return 現在時刻のDate。
     */
    private Date getNowTime() {
        return new Date();
    }
}
