/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.authn.transformer.saml2;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.exception.CitkSystemException;
import jp.co.ogis_ri.citk.common.exception.FedletException;
import jp.co.ogis_ri.citk.common.log.CitkLogger;
import jp.co.ogis_ri.citk.common.util.ExceptionUtil;

import org.mule.api.MuleMessage;
import org.mule.api.transformer.TransformerException;
import org.mule.transformer.AbstractMessageTransformer;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transport.NullPayload;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;
import org.mule.util.StringUtils;

/**
 * Mule ESBのSAMLアサーションパース用トランスフォーマークラス。
 * 
 * @author toyota
 * 
 */
public class SAML2InfoIntoMuleMessage extends AbstractMessageTransformer {

    /**
     * ログメッセージ出力用ロガー。
     */
    private static final CitkLogger logger =
            CitkLogger.getLog(SAML2InfoIntoMuleMessage.class);

    /**
     * 操作ログメッセージ出力用ロガー。
     */
    private static final CitkLogger opLogger = CitkLogger.getOperationLog();

    /**
     * SAMLアサーションのNameIDタグ開始文字列。
     */
    private static final String NAMEID_START_TAG = "<saml:NameID";

    /**
     * SAMLアサーションのNameIDタグ終了文字列。
     */
    private static final String NAMEID_END_TAG = "</saml:NameID>";

    /**
     * SAMLアサーションのNameID内NameQualifierパラメータの開始文字列。
     */
    private static final String NAMEID_NAMEQUALIFIER_TAG = "NameQualifier=\"";

    /**
     * fedletへのリクエストURL（FedletException用）
     * 
     */
    private String fedletURL = null;

    /**
     * FedletURLを取得する。
     * 
     * @return fedletURL。
     */
    public String getFedletURL() {
        return fedletURL;
    }

    /**
     * FedletURLを設定する。
     * 
     * @param fedletURL FedletURL。
     */
    public void setFedletURL(String fedletURL) {
        this.fedletURL = fedletURL;
    }

    /**
     * コンストラクタ。
     */
    public SAML2InfoIntoMuleMessage() {
        super();
        registerSourceType(DataTypeFactory.INPUT_STREAM);
        registerSourceType(DataTypeFactory.MULE_MESSAGE);
        setReturnDataType(DataTypeFactory.MULE_MESSAGE);
    }

    /**
     * OpenAMのfedletのAssersion Consumer Service であるfedletapplication
     * からのレスポンス(SAML2 ResponseメッセージのXMLを含むHTMLドキュメント)から
     * ResponseメッセージのXML部分のみを切り出し、さらにNameID、IdPEntityIDを切り出す。
     * MuleMessageのセッションプロパティにそれらをセットし、NameIDをキャッシュする。
     * 
     * @param message 変換対象のMuleメッセージ。
     * @param outputEncoding エンコード。
     * @return 変換後のMuleメッセージ。
     * @throws TransformerException。
     */
    @Override
    public Object transformMessage(MuleMessage message, String outputEncoding)
            throws TransformerException {
        String messageID = message.getMessageRootId();

        String encode = message.getEncoding();
        String body = null;

        Object httpStatus =
                message.getInboundProperty(HttpConnector.HTTP_STATUS_PROPERTY);
        String contentType =
                message.getInboundProperty(HttpConstants.HEADER_CONTENT_TYPE);

        if (!httpStatus.equals("" + HttpConstants.SC_OK)
                || !contentType.contains("text/html")
                || message.getPayload() instanceof NullPayload) {
            throw new CitkSystemException(
                    CitkConstants.ERROR_MSG_NOT_CONTAIN_PAYLOARD, message);
        }

        try {
            body = message.getPayloadAsString(encode);
        } catch (Exception e) {
            logger.debug(e);
            throw ExceptionUtil.convertRuntimeException(e);
        }

        logger.debug("Fedletレスポンス：{0}", body);

        String nameIdElement = getNameIdElement(body);

        if (nameIdElement == null) {
            throw new FedletException("E-0102", fedletURL, body);
        }

        message.setOutboundProperty(HttpConnector.HTTP_STATUS_PROPERTY,
                httpStatus);
        String nameID = getNameId(nameIdElement);

        if (StringUtils.isEmpty(nameID)) {
            throw new FedletException("E-0103", fedletURL, body);
        }

        String idpEntityId = getIdpEntityId(nameIdElement);

        if (StringUtils.isEmpty(idpEntityId)) {
            throw new FedletException("E-0104", fedletURL, body);
        }
        String relayState =
                message.getSessionProperty(CitkConstants.CITK_RELAYSTATE_KEY);

        message.setSessionProperty(CitkConstants.NAMEID_KEY, nameID);
        message.setSessionProperty(CitkConstants.IDP_ENTITYID_KEY, idpEntityId);

        opLogger.info("I-0105", messageID, relayState, nameID, idpEntityId);

        return message;
    }

    /**
     * SAMLアサーションからNameIDの切り出し。
     * 
     * @param data SAMLアサーション。
     * @return NameID。
     */
    private String getNameId(String data) {
        String nameId = data.replaceAll(NAMEID_END_TAG, "");
        return nameId.replaceAll(NAMEID_START_TAG + ".*\\>", "");
    }

    /**
     * SAMLアサーションからIdpEntityIdの切り出し。
     * 
     * @param data SAMLアサーション。
     * @return IdpEntityId。
     */
    private String getIdpEntityId(String data) {
        int sp =
                data.indexOf(NAMEID_NAMEQUALIFIER_TAG)
                        + NAMEID_NAMEQUALIFIER_TAG.length();
        String tmp = data.substring(sp);
        String[] temps = tmp.split("\"");

        return temps[0];
    }

    /**
     * SAMLアサーションからNameIDタグの切り出し。
     * 
     * @param body SAMLアサーションを含むhtml。
     * @return NameIDタグ文字列。
     */
    private String getNameIdElement(String body) {

        String value =
                StringUtils.substringBetween(body, NAMEID_START_TAG,
                        NAMEID_END_TAG);
        if (value == null) {
            return null;
        }

        StringBuilder sb = new StringBuilder();
        sb.append(NAMEID_START_TAG).append(value).append(NAMEID_END_TAG);
        return sb.toString();

    }
}
