/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.authn.transformer.http;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.exception.IllegalRequestException;
import jp.co.ogis_ri.citk.common.util.RequestParameterCache;
import mockit.Deencapsulation;

import org.apache.commons.httpclient.Cookie;
import org.junit.Test;
import org.mule.DefaultMuleMessage;
import org.mule.RequestContext;
import org.mule.api.MuleMessage;
import org.mule.api.endpoint.EndpointBuilder;
import org.mule.api.endpoint.ImmutableEndpoint;
import org.mule.api.transformer.Transformer;
import org.mule.api.transport.PropertyScope;
import org.mule.endpoint.EndpointURIEndpointBuilder;
import org.mule.tck.junit4.AbstractMuleContextTestCase;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;

public class RequestParameterTransformerTestCase extends
        AbstractMuleContextTestCase {
    private static String encoding = "UTF-8";
    private static String requestPath = "/sp1/app/test/test.html";
    private static String cachePath = "/sp1/test/test.html";
    private static final String PAYLOAD = "http body!";
    private static Map<String, Object> requestMessageProperties;
    private static Map<String, Object> cacheMessageProperties;

    private static RequestParameterCache cache;
    private static MuleMessage cacheMessage;
    private static String relayState = "abcd1234";

    private static final String BASE_URL =
            "http://localhost:8080/sp1/test/test.html";

    static {
        Map<String, Object> map = new HashMap<String, Object>();
        map.put(HttpConnector.HTTP_METHOD_PROPERTY, "GET");
        map.put(HttpConstants.HEADER_ACCEPT,
                "text/html, application/xhtml+xml, */*");
        map.put(HttpConstants.HEADER_ACCEPT_LANGUAGE, "ja,en-US;q=0.8,en;q=0.6");
        map.put(HttpConstants.HEADER_CONNECTION, "Keep-Alive");
        map.put(HttpConstants.HEADER_HOST, "www.acme.com");
        map.put(HttpConnector.HTTP_CONTEXT_PATH_PROPERTY, "/");
        map.put(HttpConnector.HTTP_REQUEST_PROPERTY, requestPath
                + "?RelayState=" + relayState);
        map.put(CitkConstants.RELAYSTATE_PARAM, relayState);
        map.put(HttpConnector.HTTP_REQUEST_PATH_PROPERTY, requestPath);
        map.put(HttpConnector.HTTP_VERSION_PROPERTY, HttpConstants.HTTP11);

        requestMessageProperties = Collections.unmodifiableMap(map);

        Map<String, Object> map2 = new HashMap<String, Object>();
        map2.put(HttpConnector.HTTP_METHOD_PROPERTY, "POST");
        map2.put(HttpConstants.HEADER_ACCEPT,
                "text/html, application/xhtml+xml, */*");
        map2.put(HttpConstants.HEADER_ACCEPT_LANGUAGE,
                "ja,en-US;q=0.8,en;q=0.6");
        map2.put(HttpConstants.HEADER_CONNECTION, "Keep-Alive");
        map2.put(HttpConstants.HEADER_HOST, "www.abcd.com");
        map2.put(HttpConnector.HTTP_CONTEXT_PATH_PROPERTY, "/");
        map2.put(HttpConnector.HTTP_REQUEST_PROPERTY, cachePath + "?"
                + CitkConstants.IDP_ENTITYID_KEY + "=idp202");
        map2.put(HttpConnector.HTTP_REQUEST_PATH_PROPERTY, cachePath);
        map2.put(HttpConnector.HTTP_VERSION_PROPERTY, HttpConstants.HTTP11);

        cacheMessageProperties = Collections.unmodifiableMap(map2);

    }

    // setup 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    public void doSetUp() throws Exception {
        super.doSetUp();
        // トランスフォーマのテストをする際は、必ずRequestContextのイベントが設定されてなければならない。
        RequestContext.setEvent(getTestEvent("test"));

        cacheMessage =
                new DefaultMuleMessage(PAYLOAD, cacheMessageProperties, null,
                        null, muleContext);
        cache = Deencapsulation.newInstance(RequestParameterCache.class);

        Map<String, MuleMessage> cacheMap = new HashMap<String, MuleMessage>();
        cacheMap.put(relayState, cacheMessage);

        cache.setCache(cacheMap);
    }

    // tear down 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    protected void doTearDown() throws Exception {
        super.doTearDown();
        // RequestContext の値をnullに戻し、次のテストメソッドに備える。
        RequestContext.setEvent(null);
    }

    public RequestParameterTransformer getTransformer() throws Exception {
        RequestParameterTransformer trans =
                createObject(RequestParameterTransformer.class);
        trans.setReturnDataType(DataTypeFactory.create(MuleMessage.class));

        EndpointBuilder builder =
                new EndpointURIEndpointBuilder("test://test", muleContext);
        builder.setEncoding(encoding);
        ImmutableEndpoint endpoint =
                muleContext.getEndpointFactory().getInboundEndpoint(builder);
        trans.setEndpoint(endpoint);

        trans.setCache(cache);

        return trans;
    }

    public Object getTestData() {
        MuleMessage mes =
                new DefaultMuleMessage(PAYLOAD, requestMessageProperties, null,
                        null, muleContext);

        Cookie[] cookies = new Cookie[2];
        Date date = new Date();
        cookies[0] =
                new Cookie(".citk202.com", CitkConstants.NAMEID_KEY,
                        "naneIDValue", "/", date, false);
        cookies[1] =
                new Cookie(".citk202.com", CitkConstants.IDP_ENTITYID_KEY,
                        "idp202", "/", date, false);

        mes.setProperty(HttpConnector.HTTP_COOKIES_PROPERTY, cookies,
                PropertyScope.INBOUND);

        return mes;
    }

    public Object getTestData_InvalidRelayState_Exception() {
        MuleMessage mes =
                new DefaultMuleMessage(PAYLOAD, requestMessageProperties, null,
                        null, muleContext);
        mes.setProperty(CitkConstants.RELAYSTATE_PARAM, "InvalidRelayState",
                PropertyScope.INBOUND);
        mes.setProperty(HttpConnector.HTTP_REQUEST_PROPERTY, requestPath
                + "?RelayState=InvalidRelayState", PropertyScope.INBOUND);

        Cookie[] cookies = new Cookie[2];
        Date date = new Date();
        cookies[0] =
                new Cookie(".citk202.com", CitkConstants.NAMEID_KEY,
                        "naneIDValue", "/", date, false);
        cookies[1] =
                new Cookie(".citk202.com", CitkConstants.IDP_ENTITYID_KEY,
                        "idp202", "/", date, false);

        mes.setProperty(HttpConnector.HTTP_COOKIES_PROPERTY, cookies,
                PropertyScope.INBOUND);
        return mes;
    }

    public Object getResultData() {
        MuleMessage mes =
                new DefaultMuleMessage(PAYLOAD, requestMessageProperties,
                        cacheMessageProperties, null, muleContext);
        mes.setProperty(HttpConnector.HTTP_REQUEST_PROPERTY, cachePath,
                PropertyScope.OUTBOUND);

        Cookie[] cookies = new Cookie[2];
        Date date = new Date();
        cookies[0] =
                new Cookie(".citk202.com", CitkConstants.NAMEID_KEY,
                        "naneIDValue", "/", date, false);
        cookies[1] =
                new Cookie(".citk202.com", CitkConstants.IDP_ENTITYID_KEY,
                        "idp202", "/", date, false);

        mes.setProperty(HttpConnector.HTTP_COOKIES_PROPERTY, cookies,
                PropertyScope.OUTBOUND);
        return mes;
    }

    @Test
    public void testTransform() throws Exception {
        Transformer trans = this.getTransformer();
        Object result = trans.transform(getTestData());
        Object expectedResult = this.getResultData();

        MuleMessage expectedMessage = (MuleMessage) expectedResult;
        MuleMessage resultMessage = (MuleMessage) result;

        Set<String> resultProperties = resultMessage.getOutboundPropertyNames();
        Set<String> expectedProperties =
                expectedMessage.getOutboundPropertyNames();

        assertEquals(expectedProperties.size(), resultProperties.size());

        Object resultValue = null;
        Object expectedValue = null;
        for (String name : resultProperties) {
            resultValue = resultMessage.getOutboundProperty(name);
            expectedValue = expectedMessage.getOutboundProperty(name);

            if (!name.equals(HttpConnector.HTTP_COOKIES_PROPERTY)) {
                assertEquals(expectedValue, resultValue);
            }
        }
    }

    @Test(expected = IllegalRequestException.class)
    public void testTransform_InvalidRelayState_Exception() throws Exception {
        Transformer trans = this.getTransformer();

        // Exception発生
        trans.transform(getTestData_InvalidRelayState_Exception());

        fail("期待するExceptionが発生しませんでした。");
    }

    @Test
    public void testDeleteUrlParam() throws Exception {
        RequestParameterTransformer trans = this.getTransformer();
        String url =
                BASE_URL + "?key=value&" + CitkConstants.IDP_ENTITYID_KEY
                        + "=idp201";
        String result =
                trans.deleteUrlParam(url, CitkConstants.IDP_ENTITYID_KEY);

        assertEquals(BASE_URL + "?key=value", result);

    }

    @Test
    public void testDeleteUrlParam_RelayState_Param() throws Exception {
        RequestParameterTransformer trans = this.getTransformer();
        String url2 =
                BASE_URL + "?" + CitkConstants.RELAYSTATE_PARAM + "="
                        + relayState;
        String result2 =
                trans.deleteUrlParam(url2, CitkConstants.RELAYSTATE_PARAM);
        assertEquals(BASE_URL, result2);
    }
}
