/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.authn.transformer.http;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;

import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import jp.co.ogis_ri.citk.common.CitkConstants;
import mockit.Expectations;

import org.apache.commons.httpclient.Cookie;
import org.junit.Test;
import org.mule.DefaultMuleMessage;
import org.mule.RequestContext;
import org.mule.api.MuleMessage;
import org.mule.api.endpoint.EndpointBuilder;
import org.mule.api.endpoint.ImmutableEndpoint;
import org.mule.api.transformer.Transformer;
import org.mule.api.transport.PropertyScope;
import org.mule.endpoint.EndpointURIEndpointBuilder;
import org.mule.tck.junit4.AbstractMuleContextTestCase;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;

public class UpdateAccessCookieTransformerTestCase extends
        AbstractMuleContextTestCase {

    private static String encoding = "UTF-8";
    private static Map<String, Object> messageProperties;
    private static String path = "/acs";
    private static String payload = "testPayloard";

    private static String nameID = "aaabbb1234";
    private static String timeZone = "GMT";
    private static String domain = ".citk202.com";
    private static final Date nowDate = new Date();

    static {
        Map<String, Object> map = new HashMap<String, Object>();
        map.put(HttpConnector.HTTP_METHOD_PROPERTY, "POST");
        map.put(HttpConstants.HEADER_ACCEPT,
                "text/html, application/xhtml+xml, */*");
        map.put(HttpConstants.HEADER_ACCEPT_LANGUAGE, "ja,en-US;q=0.8,en;q=0.6");
        map.put(HttpConstants.HEADER_CONNECTION, "Keep-Alive");
        map.put(HttpConstants.HEADER_HOST, "www.acme.com");
        map.put(HttpConstants.HEADER_COOKIE, "testCookie=value");
        map.put(HttpConnector.HTTP_CONTEXT_PATH_PROPERTY, "/");
        map.put(HttpConnector.HTTP_REQUEST_PROPERTY, path);
        map.put(HttpConnector.HTTP_REQUEST_PATH_PROPERTY, path);
        map.put(HttpConnector.HTTP_VERSION_PROPERTY, HttpConstants.HTTP11);

        messageProperties = Collections.unmodifiableMap(map);
    }

    // setup 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    public void doSetUp() throws Exception {
        super.doSetUp();
        // トランスフォーマのテストをする際は、必ずRequestContextのイベントが設定されてなければならない。
        RequestContext.setEvent(getTestEvent("test"));
    }

    // tear down 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    protected void doTearDown() throws Exception {
        super.doTearDown();
        // RequestContext の値をnullに戻し、次のテストメソッドに備える。
        RequestContext.setEvent(null);
    }

    public Transformer getTransformer() throws Exception {
        final UpdateAccessCookieTransformer trans =
                createObject(UpdateAccessCookieTransformer.class);
        new Expectations(trans) {
            {
                invoke(trans, "getNowTime");
                returns(nowDate);
            }
        };

        trans.setReturnDataType(DataTypeFactory.create(MuleMessage.class));

        EndpointBuilder builder =
                new EndpointURIEndpointBuilder("test://test", muleContext);
        builder.setEncoding(encoding);
        ImmutableEndpoint endpoint =
                muleContext.getEndpointFactory().getInboundEndpoint(builder);
        trans.setEndpoint(endpoint);
        trans.setDomain(domain);
        trans.setTimezone(timeZone);
        return trans;
    }

    public Object getTestData() {
        MuleMessage mes =
                new DefaultMuleMessage(payload, messageProperties, null, null,
                        muleContext);
        mes.setSessionProperty(CitkConstants.AUTHZ_RESULT_KEY,
                CitkConstants.AUTHZ_PERMIT);
        return mes;
    }

    public Object getResultData() {
        MuleMessage mes =
                new DefaultMuleMessage(payload, messageProperties, null, null,
                        muleContext);
        Cookie[] cookies = new Cookie[1];

        cookies[0] =
                new Cookie(domain, CitkConstants.CITK_ACCESS_DATE_COOKIE_NAME,
                        "" + nowDate.getTime(), "/", null, false);

        mes.setProperty(HttpConstants.HEADER_COOKIE_SET, cookies,
                PropertyScope.INBOUND);
        return mes;
    }

    @Test
    public void testTransform() throws Exception {
        Transformer trans = this.getTransformer();
        Object result = trans.transform(getTestData());
        Object expectedResult = this.getResultData();

        MuleMessage resultMes = (MuleMessage) result;
        MuleMessage expectedMes = (MuleMessage) expectedResult;

        Cookie[] resultCookies =
                resultMes.getInboundProperty(HttpConstants.HEADER_COOKIE_SET);
        Cookie[] expectedCookies =
                expectedMes.getInboundProperty(HttpConstants.HEADER_COOKIE_SET);

        // NameID
        assertEquals(expectedCookies[0].getName(), resultCookies[0].getName());
        assertEquals(expectedCookies[0].getDomain(),
                resultCookies[0].getDomain());
        assertEquals(expectedCookies[0].getValue(), resultCookies[0].getValue());
        assertNull(resultCookies[0].getExpiryDate());

    }
}
