/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.authz;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.exception.CitkSystemException;
import jp.co.ogis_ri.citk.common.exception.FedletException;
import jp.co.ogis_ri.citk.common.log.CitkLogger;
import jp.co.ogis_ri.citk.common.util.DataUtil;
import jp.co.ogis_ri.citk.common.util.ExceptionUtil;

import org.mule.api.MuleMessage;
import org.mule.api.transformer.TransformerException;
import org.mule.api.transport.PropertyScope;
import org.mule.transformer.AbstractMessageTransformer;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;
import org.mule.util.StringUtils;

/**
 * Mule ESBの認可処理用トランスフォーマークラス。
 * 
 * @author ISP Shiraishi
 * 
 */
public class AuthzTransformer extends AbstractMessageTransformer {

    /**
     * コンストラクタ。
     */
    public AuthzTransformer() {
        super();
    }

    /**
     * ログメッセージ出力用ロガー。
     */
    private static final CitkLogger opLogger = CitkLogger.getOperationLog();

    /**
     * Fedletのレスポンスhtml内の認可結果開始タグ文字列。
     */
    private static final String POLICY_TAG = "<th>Policy Decision</th>";

    /**
     * Fedletへのリクエストパラメータ文字列（idpEntityID）。
     */
    private static final String FEDLET_IDPENTITYID_PARAM = "&idpEntityID=";

    /**
     * Fedletへのリクエストパラメータ文字列（resource）。
     */
    private static final String FEDLET_RESOURCE_PARAM = "&resource=";

    /**
     * Fedletへのリクエストパラメータ文字列（action）。
     */
    private static final String FEDLET_ACTION_PARAM = "&action=";

    /**
     * Fedletへのリクエストパラメータ文字列（nameIDValue）。
     */
    private static final String FEDLET_NAMEID_PARAM = "&nameIDValue=";

    /**
     * FedletへのリクエストURL。
     */
    private String requestUrl = null;

    /**
     * Fedletへのリクエストアドレス。
     */
    private String address = null;

    /**
     * FedletへのリクエストURLを取得する。
     * 
     * @return requestUrl FedletへのリクエストURL。
     */
    public String getRequestUrl() {
        return requestUrl;
    }

    /**
     * FedletへのリクエストURLを設定する。
     * 
     * @param requestUrl FedletへのリクエストURL。
     */
    public void setRequestUrl(String requestUrl) {
        this.requestUrl = requestUrl;
    }

    /**
     * Fedletへのリクエストアドレスを取得する。
     * 
     * @return address　 Fedletへのリクエストアドレス。
     */
    public String getAddress() {
        return address;
    }

    /**
     * Fedletへのリクエストアドレスを設定する。
     * 
     * @param address Fedletへのリクエストアドレス。
     */
    public void setAddress(String address) {
        this.address = address;
    }

    /**
     * 認可処理を行う。(Fedletを利用)
     * 
     * @param message 変換対象のMuleメッセージ。
     * @param outputEncoding エンコード。
     * @return 変換後のMuleMessageメッセージ。
     * @throws TransformerException。
     */
    @Override
    public Object transformMessage(MuleMessage message, String outputEncoding)
            throws TransformerException {
        String messageID = message.getUniqueId();

        String requestPath =
                message.getInboundProperty(HttpConnector.HTTP_REQUEST_PATH_PROPERTY);
        String resource = address + requestPath;
        String nameId = message.getSessionProperty(CitkConstants.NAMEID_KEY);
        String encodedNameId = null;

        try {
            encodedNameId = URLEncoder.encode(nameId, "UTF-8");
        } catch (UnsupportedEncodingException e1) {
            logger.debug(e1);
            ExceptionUtil.convertRuntimeException(e1);
        }

        String method =
                message.getInboundProperty(HttpConnector.HTTP_METHOD_PROPERTY);
        String idpEntityId =
                DataUtil.getCookieValue(
                        message.getInboundProperty(HttpConnector.HTTP_COOKIES_PROPERTY),
                        CitkConstants.IDP_ENTITYID_KEY);
        String url = buildUrl(resource, encodedNameId, method, idpEntityId);

        opLogger.info("I-0201", messageID, idpEntityId, method, nameId,
                resource);

        String response = null;

        response = fedletAuthrization(url);

        if (response == null) {
            // fedletAuthrization 内で、うまくストリームが読めなかった場合。
            throw new CitkSystemException(
                    CitkConstants.ERROR_MSG_AUTHORIZE_FAILUER, message);
        }

        if (response.contains(POLICY_TAG) == false) {
            // XACMLレスポンス不正（PolicyDecisionタグなし）
            throw new FedletException("E-0105", url, response); 
        }

        String authzResult = StringUtils.substringAfter(response, POLICY_TAG);

        if (authzResult == null || authzResult.isEmpty()) {
            throw new FedletException("E-0105", url, response); // XACMLレスポンス不正（PolicyDecisionタグなし）
        }

        String decision = getPolicyDecision(authzResult);

        if (decision == null) {
            throw new FedletException("E-0106", url, response); // XACMLレスポンス不正（PolicyDecisionなし）
        }

        opLogger.info("I-0202", messageID, decision);

        message.setProperty(CitkConstants.AUTHZ_RESULT_KEY, decision,
                PropertyScope.SESSION);

        return message;
    }

    /**
     * Fedletに対する認可リクエストURLを生成する。
     * 
     * @param resource 認可対象リソース。
     * @param encodedNameID URLエンコード済みNameID。
     * @param method 認可対象メソッド。
     * @param idpEntityId IdpEntityID。
     * @return Fedletの認可リクエストURL文字列。
     */
    private String buildUrl(String resource, String encodedNameId,
            String method, String idpEntityId) {
        return requestUrl + FEDLET_IDPENTITYID_PARAM + idpEntityId
                + FEDLET_RESOURCE_PARAM + resource + FEDLET_ACTION_PARAM
                + method + FEDLET_NAMEID_PARAM + encodedNameId;
    }

    /**
     * Fedletに対し認可リクエストを行う。
     * 
     * @param urlString Fedletの認可リクエストURL文字列。
     * @return 認可結果。
     * @throws IOException。
     */
    private String fedletAuthrization(String urlString) {
        String body = null;
        URL url = null;
        try {
            url = new URL(urlString);
        } catch (MalformedURLException e) {
            logger.debug(e);
            ExceptionUtil.convertRuntimeException(e);
        }

        HttpURLConnection urlconn = null;
        BufferedReader reader = null;
        try {
            urlconn = (HttpURLConnection) url.openConnection();
            urlconn.setRequestMethod(HttpConstants.METHOD_GET);
            urlconn.setInstanceFollowRedirects(false);

            urlconn.connect();

            int responseCode = urlconn.getResponseCode();

            if (responseCode != HttpConstants.SC_OK) {
                throw new FedletException(
                        CitkConstants.ERROR_MSG_AUTHORIZE_FAILUER, urlString,
                        "HTTP Status code : " + responseCode);
            }

            reader =
                    new BufferedReader(new InputStreamReader(
                            urlconn.getInputStream()));

            String line;
            StringBuilder sb = new StringBuilder();
            while ((line = reader.readLine()) != null) {
                sb.append(line);
            }
            body = sb.toString();

        } catch (IOException e) {
            logger.debug(e);
            ExceptionUtil.convertRuntimeException(e);
        } finally {
            try {
                if (reader != null) {
                    reader.close();
                }
            } catch (IOException e) {
                // no op
            }

            try {
                if (urlconn != null) {
                    urlconn.disconnect();
                }
            } catch (Exception e) {
                // no op
            }
        }

        return body;
    }

    /**
     * Fedletの認可レスポンスから認可結果を抽出。
     * 
     * @param data Fedletの認可レスポンス。
     * @return 認可結果。
     */
    private String getPolicyDecision(String data) {

        String decision = null;

        if (data.indexOf(CitkConstants.AUTHZ_PERMIT) != -1) {
            decision = CitkConstants.AUTHZ_PERMIT;

        } else if (data.indexOf(CitkConstants.AUTHZ_DENY) != -1) {
            decision = CitkConstants.AUTHZ_DENY;
        } else if (data.indexOf(CitkConstants.AUTHZ_INDETERMINATE) != -1) {
            decision = CitkConstants.AUTHZ_INDETERMINATE;
        }
        return decision;
    }
}
