/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.authz.common;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;
import java.net.URLEncoder;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.exception.FedletException;
import jp.co.ogis_ri.citk.common.log.CitkLogger;
import jp.co.ogis_ri.citk.common.util.ExceptionUtil;

import org.apache.commons.io.IOUtils;
import org.mule.transport.http.HttpConstants;

/**
 * 認可処理用ユーティリティ
 * 
 */
public class AuthzUtil {

    /**
     * ログメッセージ出力用ロガー。
     */
    private static final CitkLogger opLogger = CitkLogger.getOperationLog();

    /**
     * コンストラクタ.
     */
    private AuthzUtil() {
    }
    
    /**
     * Fedlet認可用URLを作成する。
     * 
     * @param authzUrl 認可用URL。
     * @param spEntityId SPエンティティID。
     * @param action アクション(HTTPメソッド)。
     * @param idpEntityId IDPエンティティID。
     * @param nameId NameID。
     * @param resource 認可対象のリソースURL。
     * @return Fedlet認可用URL。
     * @throws UnsupportedEncodingException サポートされていないコードでエンコードした場合。
     * 
     */
    public static String createFedletUrl(String authzUrl, String spEntityId,
            String action, String idpEntityId, String nameId, String resource)
            throws UnsupportedEncodingException {

        StringBuilder sb = new StringBuilder();
        sb.append(authzUrl + "?");
        sb.append("spEntityID=" + spEntityId + "&");
        sb.append("action=" + action + "&");
        sb.append("idpEntityID=" + idpEntityId + "&");
        sb.append("nameIDValue=" + URLEncoder.encode(nameId, "UTF-8") + "&");
        sb.append("resource=" + URLEncoder.encode(resource, "UTF-8"));

        return sb.toString();
    }

    /**
     * 認可結果を取得する(Fedlet用)。
     * 
     * @param urlString Fedletの認可リクエストURL文字列。
     * @return　認可結果（Permit,Deny,Indeterminate）。
     * 
     */
    public static String getDecisionForFedlet(String urlString) {

        String response = AuthorizeForFedlet(urlString);

        String decision = null;

        if (response.indexOf(CitkConstants.AUTHZ_PERMIT) != -1) {
            decision = CitkConstants.AUTHZ_PERMIT;

        } else if (response.indexOf(CitkConstants.AUTHZ_DENY) != -1) {
            decision = CitkConstants.AUTHZ_DENY;

        } else if (response.indexOf(CitkConstants.AUTHZ_INDETERMINATE) != -1) {
            decision = CitkConstants.AUTHZ_INDETERMINATE;

        }
        return decision;
    }

    /**
     * Fedletに対し認可リクエストを行う。
     * 
     * @param urlString Fedletの認可リクエストURL文字列。
     * @return 認可結果。
     */
    private static String AuthorizeForFedlet(String urlString) {

        String ret = null;
        HttpURLConnection conn = null;
        try {
            conn = createConnection(createURL(urlString));

            checkResponseCode(urlString, conn);

            ret = readResponse(conn.getInputStream());

        } catch (IOException e) {
            opLogger.debug(e);
            ExceptionUtil.convertRuntimeException(e);

        } finally {
            disconnect(conn);
        }
        return ret;
    }

    private static void checkResponseCode(String urlString,
            HttpURLConnection conn) throws IOException {

        int responseCode = conn.getResponseCode();
        if (responseCode != HttpConstants.SC_OK) {
            throw new FedletException(
                    CitkConstants.ERROR_MSG_AUTHORIZE_FAILUER, urlString,
                    "HTTP Status code : " + responseCode);
        }
    }

    private static String readResponse(InputStream inputStream)
            throws IOException {

        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new InputStreamReader(inputStream));

            String line;
            StringBuilder sb = new StringBuilder();
            while ((line = reader.readLine()) != null) {
                sb.append(line);
            }
            return sb.toString();

        } finally {
            IOUtils.closeQuietly(reader);
        }
    }

    private static HttpURLConnection createConnection(URL url)
            throws IOException, ProtocolException {
        HttpURLConnection conn;
        conn = (HttpURLConnection) url.openConnection();
        conn.setRequestMethod(HttpConstants.METHOD_GET);
        conn.setInstanceFollowRedirects(false);

        conn.connect();
        return conn;
    }

    private static URL createURL(String urlString) {
        URL url = null;
        try {
            url = new URL(urlString);
        } catch (MalformedURLException e) {
            opLogger.debug(e);
            ExceptionUtil.convertRuntimeException(e);
        }
        return url;
    }

    private static void disconnect(HttpURLConnection conn) {
        if (conn != null) {
            try {
                conn.disconnect();
            } catch (Exception e) {
                // no operation
            }
        }
    }
}
