/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.authz;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.exception.CitkSystemException;
import jp.co.ogis_ri.citk.common.exception.FedletException;
import mockit.Expectations;

import org.apache.commons.httpclient.Cookie;
import org.junit.Test;
import org.mule.DefaultMuleMessage;
import org.mule.RequestContext;
import org.mule.api.MuleMessage;
import org.mule.api.endpoint.EndpointBuilder;
import org.mule.api.endpoint.ImmutableEndpoint;
import org.mule.api.transformer.Transformer;
import org.mule.api.transport.PropertyScope;
import org.mule.endpoint.EndpointURIEndpointBuilder;
import org.mule.tck.junit4.AbstractMuleContextTestCase;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;

public class AuthzTransformerTestCase extends AbstractMuleContextTestCase {
    private static String encoding = "UTF-8";
    private static String path = "/sp1/test/test.html";
    private static final String PAYLOAD = "http body!";
    private static Map<String, Object> messageProperties;
    private static String testNameID = "aaa";
    private static String address = "http://sp.citk202.com:8008";
    private static String requestUrl = "http://dummy.citk202.com:9080/fedlet/fedletXACMLResp.jsp?spEntityID=sp202";
    private static String expectedURL = "http://dummy.citk202.com:9080/fedlet/fedletXACMLResp.jsp?spEntityID=sp202&idpEntityID=idp202&resource=http://sp.citk202.com:8008/sp1/test/test.html&action=GET&nameIDValue=aaa";
    private static String dummyResponse = "";

    static {
        Map<String, Object> map = new HashMap<String, Object>();
        map.put(HttpConnector.HTTP_METHOD_PROPERTY, "GET");
        map.put(HttpConstants.HEADER_ACCEPT,
                "text/html, application/xhtml+xml, */*");
        map.put(HttpConstants.HEADER_ACCEPT_LANGUAGE, "ja,en-US;q=0.8,en;q=0.6");
        map.put(HttpConstants.HEADER_CONNECTION, "Keep-Alive");
        map.put(HttpConstants.HEADER_HOST, "www.acme.com");
        map.put(HttpConnector.HTTP_CONTEXT_PATH_PROPERTY, "/");
        map.put(HttpConnector.HTTP_REQUEST_PROPERTY, path
                + "?testKey=testValue");
        map.put(HttpConnector.HTTP_REQUEST_PATH_PROPERTY, path);
        map.put(HttpConnector.HTTP_VERSION_PROPERTY, HttpConstants.HTTP11);

        messageProperties = Collections.unmodifiableMap(map);
    }

    static {
        String uri = "./src/test/resources/dummyResponse.html";

        try {
            FileReader f = new FileReader(uri);
            BufferedReader b = new BufferedReader(f);
            String s = null;
            while ((s = b.readLine()) != null) {
                dummyResponse = dummyResponse.concat(s);
            }
        } catch (Exception e) {
            fail("ファイル読み込み失敗");
        }

    }

    // setup 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    public void doSetUp() throws Exception {
        super.doSetUp();
        // トランスフォーマのテストをする際は、必ずRequestContextのイベントが設定されてなければならない。
        RequestContext.setEvent(getTestEvent("test"));
    }

    // tear down 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    protected void doTearDown() throws Exception {
        super.doTearDown();
        // RequestContext の値をnullに戻し、次のテストメソッドに備える。
        RequestContext.setEvent(null);
    }

    public Transformer getTransformer() throws Exception {
        final Transformer trans = createObject(AuthzTransformer.class);
        new Expectations(trans) {
            {
                invoke(trans, "fedletAuthrization", expectedURL);
                returns(dummyResponse);
            }
        };

        trans.setReturnDataType(DataTypeFactory.create(MuleMessage.class));

        EndpointBuilder builder = new EndpointURIEndpointBuilder("test://test",
                muleContext);
        builder.setEncoding(encoding);
        ImmutableEndpoint endpoint = muleContext.getEndpointFactory()
                .getInboundEndpoint(builder);
        trans.setEndpoint(endpoint);

        ((AuthzTransformer) trans).setAddress(address);
        ((AuthzTransformer) trans).setRequestUrl(requestUrl);
        return trans;
    }

    public Transformer getTransformer_AuthorizationFail_Exception()
            throws Exception {
        final Transformer trans = createObject(AuthzTransformer.class);
        new Expectations(trans) {
            {
                invoke(trans, "fedletAuthrization", expectedURL);
                result = new IOException();
            }
        };

        trans.setReturnDataType(DataTypeFactory.create(MuleMessage.class));

        EndpointBuilder builder = new EndpointURIEndpointBuilder("test://test",
                muleContext);
        builder.setEncoding(encoding);
        ImmutableEndpoint endpoint = muleContext.getEndpointFactory()
                .getInboundEndpoint(builder);
        trans.setEndpoint(endpoint);

        ((AuthzTransformer) trans).setAddress(address);
        ((AuthzTransformer) trans).setRequestUrl(requestUrl);
        return trans;
    }

    public Transformer getTransformer_NullAuthorization_Exception()
            throws Exception {
        final Transformer trans = createObject(AuthzTransformer.class);
        new Expectations(trans) {
            {
                invoke(trans, "fedletAuthrization", expectedURL);
                returns(null);
            }
        };

        trans.setReturnDataType(DataTypeFactory.create(MuleMessage.class));

        EndpointBuilder builder = new EndpointURIEndpointBuilder("test://test",
                muleContext);
        builder.setEncoding(encoding);
        ImmutableEndpoint endpoint = muleContext.getEndpointFactory()
                .getInboundEndpoint(builder);
        trans.setEndpoint(endpoint);

        ((AuthzTransformer) trans).setAddress(address);
        ((AuthzTransformer) trans).setRequestUrl(requestUrl);
        return trans;
    }

    public Transformer getTransformer_NotContainFedletResponse_Exception()
            throws Exception {
        final Transformer trans = createObject(AuthzTransformer.class);
        new Expectations(trans) {
            {
                invoke(trans, "fedletAuthrization", expectedURL);
                returns("<th>Policy Decision</th>");
            }
        };

        trans.setReturnDataType(DataTypeFactory.create(MuleMessage.class));

        EndpointBuilder builder = new EndpointURIEndpointBuilder("test://test",
                muleContext);
        builder.setEncoding(encoding);
        ImmutableEndpoint endpoint = muleContext.getEndpointFactory()
                .getInboundEndpoint(builder);
        trans.setEndpoint(endpoint);

        ((AuthzTransformer) trans).setAddress(address);
        ((AuthzTransformer) trans).setRequestUrl(requestUrl);
        return trans;
    }

    public Transformer getTransformer_InvalidFedletResponse_Exception()
            throws Exception {
        final Transformer trans = createObject(AuthzTransformer.class);
        new Expectations(trans) {
            {
                invoke(trans, "fedletAuthrization", expectedURL);
                returns("Invalidresponse");
            }
        };

        trans.setReturnDataType(DataTypeFactory.create(MuleMessage.class));

        EndpointBuilder builder = new EndpointURIEndpointBuilder("test://test",
                muleContext);
        builder.setEncoding(encoding);
        ImmutableEndpoint endpoint = muleContext.getEndpointFactory()
                .getInboundEndpoint(builder);
        trans.setEndpoint(endpoint);

        ((AuthzTransformer) trans).setAddress(address);
        ((AuthzTransformer) trans).setRequestUrl(requestUrl);
        return trans;
    }

    public Transformer getTransformer_InvalidAuthorization_Exception()
            throws Exception {
        final Transformer trans = createObject(AuthzTransformer.class);

        final String invalidDummyResponse = dummyResponse.replaceAll(
                CitkConstants.AUTHZ_PERMIT, "");
        new Expectations(trans) {
            {
                invoke(trans, "fedletAuthrization", expectedURL);
                returns(invalidDummyResponse);
            }
        };

        trans.setReturnDataType(DataTypeFactory.create(MuleMessage.class));

        EndpointBuilder builder = new EndpointURIEndpointBuilder("test://test",
                muleContext);
        builder.setEncoding(encoding);
        ImmutableEndpoint endpoint = muleContext.getEndpointFactory()
                .getInboundEndpoint(builder);
        trans.setEndpoint(endpoint);

        ((AuthzTransformer) trans).setAddress(address);
        ((AuthzTransformer) trans).setRequestUrl(requestUrl);
        return trans;
    }

    public Object getTestData() {
        MuleMessage mes = new DefaultMuleMessage(PAYLOAD, messageProperties,
                null, null, muleContext);
        mes.setSessionProperty(CitkConstants.NAMEID_KEY, testNameID);

        Cookie[] cookies = new Cookie[1];
        Date date = new Date();
        cookies[0] = new Cookie(".citk202.com", CitkConstants.IDP_ENTITYID_KEY,
                "idp202", "/", date, false);

        mes.setProperty(HttpConnector.HTTP_COOKIES_PROPERTY, cookies,
                PropertyScope.INBOUND);
        return mes;
    }

    public Object getTestData_AuthorizationFail_Exception() {
        MuleMessage mes = new DefaultMuleMessage(PAYLOAD, messageProperties,
                null, null, muleContext);

        mes.setProperty(HttpConnector.HTTP_REQUEST_PATH_PROPERTY, "aaa",
                PropertyScope.INBOUND);

        mes.setSessionProperty(CitkConstants.NAMEID_KEY, testNameID);

        Cookie[] cookies = new Cookie[1];
        Date date = new Date();
        cookies[0] = new Cookie(".citk202.com", CitkConstants.IDP_ENTITYID_KEY,
                "idp202", "/", date, false);

        mes.setProperty(HttpConnector.HTTP_COOKIES_PROPERTY, cookies,
                PropertyScope.INBOUND);
        return mes;
    }

    public Object getResultData() {
        MuleMessage mes = new DefaultMuleMessage(PAYLOAD, messageProperties,
                null, null, muleContext);
        mes.setSessionProperty(CitkConstants.AUTHZ_RESULT_KEY,
                CitkConstants.AUTHZ_PERMIT);
        return mes;
    }

    @Test
    public void testTransform() throws Exception {
        Transformer trans = this.getTransformer();
        Object result = trans.transform(getTestData());
        Object expectedResult = this.getResultData();

        MuleMessage expectedMessage = (MuleMessage) expectedResult;
        MuleMessage resultMessage = (MuleMessage) result;

        String expectedDecision = expectedMessage
                .getSessionProperty(CitkConstants.AUTHZ_RESULT_KEY);
        String resultDecision = resultMessage
                .getSessionProperty(CitkConstants.AUTHZ_RESULT_KEY);

        assertEquals(resultDecision, expectedDecision);
    }

    @Test(expected = CitkSystemException.class)
    public void testTransform_AuthorizationFail_Exception() throws Exception {
        Transformer trans = this.getTransformer_AuthorizationFail_Exception();

        // Exception発生
        trans.transform(getTestData_AuthorizationFail_Exception());

        fail("期待するExceptionが発生しませんでした。");
    }

    @Test(expected = CitkSystemException.class)
    public void testTransform_NullAuthorization_Exception() throws Exception {
        Transformer trans = this.getTransformer_NullAuthorization_Exception();

        // Exception発生
        trans.transform(getTestData());

        fail("期待するExceptionが発生しませんでした。");
    }

    @Test(expected = FedletException.class)
    public void testTransform_InvalidFedletResponse_Exception()
            throws Exception {
        Transformer trans = this
                .getTransformer_InvalidFedletResponse_Exception();

        // Exception発生
        trans.transform(getTestData());

        fail("期待するExceptionが発生しませんでした。");
    }

    @Test(expected = FedletException.class)
    public void testTransform_NotContainFedletResponse_Exception()
            throws Exception {
        Transformer trans = this
                .getTransformer_NotContainFedletResponse_Exception();

        // Exception発生
        trans.transform(getTestData());

        fail("期待するExceptionが発生しませんでした。");
    }

    @Test(expected = FedletException.class)
    public void testTransform_InvalidAuthorization_Exception() throws Exception {
        Transformer trans = this
                .getTransformer_InvalidAuthorization_Exception();

        // Exception発生
        trans.transform(getTestData());

        fail("期待するExceptionが発生しませんでした。");
    }

}
