/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.common.exception;

import java.nio.charset.Charset;

import jp.co.ogis_ri.citk.common.log.CitkLogger;

import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.exception.RollbackSourceCallback;
import org.mule.api.transport.PropertyScope;
import org.mule.config.ExceptionHelper;
import org.mule.exception.AbstractMessagingExceptionStrategy;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;

/**
 * Mule ESBのカスタムExceptionStrategyクラス実装。{@link CitkSystemException} および、
 * {@link CitkApplicationException} のサブクラスに関して、ハンドリングを行う。
 * これらの例外をハンドルする際は、ユーザエージェントに対して通知するエラーメッセージも生成する。
 * 
 * @author ISP Shiraishi
 * 
 */
public class CitkExceptionStrategy extends AbstractMessagingExceptionStrategy {

    /**
     * Content-Type。
     */
    private static final String contentType = "text/html";

    /**
     * エンコード。
     */
    private String encoding = "charset=Shift_JIS";

    /**
     * ログメッセージ出力用ロガー。
     */
    private static final CitkLogger logger = CitkLogger
            .getLog(CitkExceptionStrategy.class);

    /**
     * コンストラクタ。
     */
    public CitkExceptionStrategy(MuleContext muleContext) {
        super(muleContext);
    }

    /**
     * エンコードを取得する。
     * 
     * @return　エンコード。
     * 
     */
    public String getEncoding() {
        return encoding;
    }

    /**
     * エンコードを設定する。
     * 
     * @param encoding エンコード。
     * 
     */
    public void setEncoding(String encoding) {
        Charset.forName(encoding); // check encode setting

        this.encoding = encoding;
    }

    /**
     * Exceptionハンドラ。
     * 
     * @param e 対象のException。
     * @param event Muleイベント。
     * @return Muleイベント。
     */
    @Override
    public MuleEvent handleException(Exception e, MuleEvent event) {

        MuleEvent resultEvent = event;

        Throwable throwable = e.getCause();

        if (throwable instanceof jp.co.ogis_ri.citk.common.exception.CitkSystemException) {
            exceptionLogging(throwable);
            return setResult(resultEvent, throwable);
        } else if (throwable instanceof jp.co.ogis_ri.citk.common.exception.CitkApplicationException) {
            exceptionLogging(throwable);
            return setResult(resultEvent, throwable);
        } else {
            return super.handleException(e, event);
        }
    }

    /**
     * 例外の型から、ユーザエージェントに返すHTTPステータスコードを判定する。
     * 
     * @param throwable　例外。
     * @return 不正アクセス時に起こる例外ならば、404。さもなければ500。
     */
    private int getStatusCode(Throwable throwable) {
        if (throwable instanceof IllegalRequestException) {
            return 404;
        }
        return 500;
    }

    /**
     * Exceptionハンドラ。
     * @param　ex 対象のException。
     * @param event Muleイベント
     * @param rollbackMethod。
     * @return Muleイベント。
     */
    @Override
    public MuleEvent handleException(Exception ex, MuleEvent event,
            RollbackSourceCallback rollbackMethod) {
        return super.handleException(ex, event, rollbackMethod);
    }

    /**
     * Exceptionハンドリング結果のセット。
     * 
     * @param event Muleイベント。
     * @param message　レスポンスメッセージ。
     * @param status httpステータスコード。
     * @return Muleイベント。
     */
    private MuleEvent setResult(MuleEvent event, Throwable throwable) {
        // Citkの例外が発生した場合、ユーザエージェントに対して、このメソッド内 で生成するエラーメッセージを返す必要がある。
        // そのため、この位置でMuleMessageの exceptionPayload の値にnullを設定している。
        // さもなければ、Mule 内部で例外が起こったと判断され、Mule
        // のAbstractExceptionStrategy#routeExceptionにハンドルされてしまう。
        // そうなると、ユーザエージェントにレスポンスが返されなくなってしまう。
        event.getMessage().setExceptionPayload(null);

        String message = null;
        if (throwable instanceof jp.co.ogis_ri.citk.common.exception.IllegalRequestException) {
            message = ((IllegalRequestException) throwable)
                    .getMessageToPayloard();
            event.getMessage().setPayload(message);
        } else {
            message = throwable.getMessage();
            event.getMessage().setPayload(message);
        }

        event.getMessage().setOutboundProperty(
                HttpConstants.HEADER_CONTENT_LENGTH, message.length());
        event.getMessage().setOutboundProperty(
                HttpConnector.HTTP_STATUS_PROPERTY, getStatusCode(throwable));

        String contentTypeParam = contentType + "; " + getEncoding();
        event.getMessage().setProperty(HttpConstants.HEADER_CONTENT_TYPE,
                contentTypeParam, PropertyScope.OUTBOUND);
        return event;
    }

    /**
     * Exceptionログ出力。
     * 
     * @param t　ログ対象throwable。
     */
    private void exceptionLogging(Throwable t) {
        logger.error(t);

        Throwable lootThrowable = ExceptionHelper.getRootException(t);
        logger.error(lootThrowable, "Caught exception in Exception Strategy: ");
    }

}
