/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.common.log;

import java.text.MessageFormat;
import java.util.ResourceBundle;

import jp.co.ogis_ri.citk.common.CitkConstants;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * <p>
 * CITKLoggerクラス実装。 Mule ESBで使われている {@link org.apache.commons.logging.Log}
 * オブジェクトのラッパーとして実装している。<br />
 * また、このLoggerでは、{@link java.text.MessageFormat#format(String, Object...)}
 * で利用できる、フォーマットのパターンが用いられている。
 * </p>
 * <p>
 * ログレベルについては、以下のとおり。
 * <ol>
 * <li>trace (the least serious)</li>
 * <li>debug</li>
 * <li>info</li>
 * <li>warn</li>
 * <li>error</li>
 * <li>fatal (the most serious)</li>
 * </ol>
 * </p>
 * 
 * <b>CitkLogger のメソッドの仕様</b><br/>
 * <dl>
 * <dt><code>keyOrMessage</code></dt>
 * <dd>ログメッセージパターンの文字列または、リソースバンドルのメッセージID。<br/>
 * ログメッセージパターンの文字列を直接指定する場合、 {@link java.text.MessageFormat}で扱えるパターンの形式が使える。</dd>
 * <dt><code>param</code></dt>
 * <dd>メッセージパターンのパラメータ。{@link java.text.MessageFormat#format(String, Object...)}
 * の可変長引数として使われる。</dd>
 * <dt><code>Throwable t</code></dt>
 * <dd>Logger で記録する例外。</dd>
 * </dl>
 * 
 * @author ISP Shiraishi
 * 
 */
public class CitkLogger {

    private static final String OPERATION = "operation";

    private static final ResourceBundle message_rb = ResourceBundle
            .getBundle(CitkConstants.LOG_MESSAGE_PROPERTIES);

    /**
     * ラップ対象のLogger。
     */
    private Log log = null;

    /**
     * 開発者用のログを出力するLoggerを返す。
     * 
     * @param clazz 対象クラス。
     * @return ロガーインスタンス。
     */
    public static CitkLogger getLog(Class<?> clazz) {
        return new CitkLogger(clazz);
    }

    /**
     * 開発者用のログを出力するLoggerを返す。
     * 
     * @param clazz  対象クラス。
     * @return　ロガーインスタンス。
     */
    public static CitkLogger getLog(String clazz) {
        return new CitkLogger(clazz);
    }

    /**
     * 運用管理者用の操作ログを出力するLoggerを返す。 前提として、 log4j.properties などに、
     * {@link CitkLogger#OPERATION} という定数で取得できる Logger を用意されていること。
     * 
     * @return　ロガーインスタンス。
     */
    public static CitkLogger getOperationLog() {
        return new CitkLogger();
    }

    /**
     * コンストラクタ。
     * 
     * @param clazz 対象クラス。
     */
    private CitkLogger(Class<?> clazz) {
        log = LogFactory.getLog(clazz);
    }

    /**
     * コンストラクタ。
     * 
     * @param clazz キー。
     */
    private CitkLogger(String clazz) {
        log = LogFactory.getLog(clazz);
    }

    /**
     * コンストラクタ。
     */
    private CitkLogger() {
        log = LogFactory.getLog(OPERATION);
    }

    /**
     * ログメッセージのパターンを取得し、
     * {@link java.text.MessageFormat#format(String, Object...)}
     * を用いて、argsからログメッセージを作成する。 <br />
     * CitkLogMapping に該当のメッセージコードがない場合、 keyOrMessage のテンプレートの文字列として使う。
     * 
     * @param keyOrMessage ログメッセージパターンの文字列または、リソースバンドルのメッセージID。
     * @param args メッセージパラメータ。
     * @return ログメッセージ。
     * @throws IllegalArgumentException see {@link
     *             java.text.MessageFormat.format(String, Object...)}
     */
    private String getMessage(String keyOrMessage, Object... args) {
        String message = null;
        if (message_rb.containsKey(keyOrMessage)) {
            message = message_rb.getString(keyOrMessage);
        } else {
            message = keyOrMessage;
        }

        if (args.length == 0) {
            return message;
        }
        return MessageFormat.format(message, args);
    }

    /**
     * fatalレベルログ。
     * 
     * @param keyOrMessage　リソースキーまたはメッセージ。
     * @param param 引数。
     */
    public void fatal(String keyOrMessage, Object... param) {
        if (log.isFatalEnabled()) {
            log.fatal(getMessage(keyOrMessage, param));
        }
    }

    /**
     * fatalレベルログ。
     * 
     * @param t 例外。
     * @param keyOrMessage　リソースキーまたはメッセージ。
     * @param param 引数。
     */
    public void fatal(Throwable t, String keyOrMessage, Object... args) {
        if (log.isFatalEnabled()) {
            log.fatal(getMessage(keyOrMessage, args), t);
        }
    }

    /**
     * fatalレベルログ。例外をラップして投げる前などに、例外の発生地点を記録する目的で使う。
     * 
     * @param t 例外。
     */
    public void fatal(Throwable t) {
        if (log.isFatalEnabled()) {
            log.fatal(t);
        }
    }

    /**
     * errorレベルログ。
     * 
     * @param keyOrMessage　リソースキーまたはメッセージ。
     * @param param 引数。
     */
    public void error(String keyOrMessage, Object... param) {
        if (log.isErrorEnabled()) {
            log.error(getMessage(keyOrMessage, param));
        }
    }

    /**
     * errorレベルログ。 例外をラップして投げる前などに、例外の発生地点を記録する目的で使う。
     * 
     * @param t 例外。
     */
    public void error(Throwable t) {
        if (log.isErrorEnabled()) {
            log.error(t);
        }
    }

    /**
     * errorレベルログ
     * 
     * @param t 例外。
     * @param keyOrMessage　リソースキーまたはメッセージ。
     * @param param 引数。
     */
    public void error(Throwable t, String keyOrMessage, Object... param) {
        if (log.isErrorEnabled()) {
            log.error(getMessage(keyOrMessage, param), t);
        }
    }

    /**
     * warnレベルログ。
     * 
     * @param keyOrMessage　リソースキーまたはメッセージ。
     * @param param 引数。
     */
    public void warn(String keyOrMessage, Object... param) {
        if (log.isWarnEnabled()) {
            log.warn(getMessage(keyOrMessage, param));
        }
    }

    /**
     * warnレベルログ。
     * 
     * @param t 例外。
     * @param keyOrMessage　リソースキーまたはメッセージ。
     * @param param 引数。
     */
    public void warn(Throwable t, String keyOrMessage, Object... param) {
        if (log.isWarnEnabled()) {
            log.warn(getMessage(keyOrMessage, param), t);
        }
    }

    /**
     * warnレベルログ。例外をラップして投げる前などに、例外の発生地点を記録する目的で使う。
     * 
     * @param t 例外。
     */
    public void warn(Throwable t) {
        if (log.isWarnEnabled()) {
            log.warn(t);
        }
    }

    /**
     * infoレベルログ。
     * 
     * @param keyOrMessage　リソースキーまたはメッセージ。
     * @param param 引数。
     */
    public void info(String keyOrMessage, Object... param) {
        if (log.isInfoEnabled()) {
            log.info(getMessage(keyOrMessage, param));
        }
    }

    /**
     * infoレベルログ。
     * 
     * @param t 例外。
     * @param keyOrMessage　リソースキーまたはメッセージ。
     * @param param 引数。
     */
    public void info(Throwable t, String keyOrMessage, Object... param) {
        if (log.isInfoEnabled()) {
            log.info(getMessage(keyOrMessage, param), t);
        }
    }

    /**
     * infoレベルログ. 例外をラップして投げる前などに、例外の発生地点を記録する目的で使う。
     * 
     * @param t 例外。
     */
    public void info(Throwable t) {
        if (log.isInfoEnabled()) {
            log.info(t);
        }
    }

    /**
     * debugレベルログ。
     * 
     * @param keyOrMessage　リソースキーまたはメッセージ。
     * @param param 引数。
     */
    public void debug(String keyOrMessage, Object... param) {
        if (log.isDebugEnabled()) {
            log.debug(getMessage(keyOrMessage, param));
        }
    }

    /**
     * debugレベルログ。
     * 
     * @param t 例外。
     * @param keyOrMessage　リソースキーまたはメッセージ。
     * @param param 引数。
     */
    public void debug(Throwable t, String keyOrMessage, Object... param) {
        if (log.isDebugEnabled()) {
            log.debug(getMessage(keyOrMessage, param), t);
        }
    }

    /**
     * debugレベルログ. 例外をラップして投げる前などに、例外の発生地点を記録する目的で使う。
     * 
     * @param t 例外。
     */
    public void debug(Throwable t) {
        if (log.isDebugEnabled()) {
            log.debug(t);
        }
    }

    /**
     * traceレベルログ。
     * 
     * @param keyOrMessage　リソースキーまたはメッセージ。
     * @param param 引数。
     */
    public void trace(String keyOrMessage, Object... param) {
        if (log.isTraceEnabled()) {
            log.trace(getMessage(keyOrMessage, param));
        }
    }

    /**
     * traceレベルログ。
     * 
     * @param t 例外。
     * @param keyOrMessage　リソースキーまたはメッセージ。
     * @param param 引数。
     */
    public void trace(Throwable t, String keyOrMessage, Object... param) {
        if (log.isTraceEnabled()) {
            log.trace(getMessage(keyOrMessage, param), t);
        }
    }

    /**
     * traceレベルログ. 例外をラップして投げる前などに、例外の発生地点を記録する目的で使う。
     * 
     * @param t 例外。
     */
    public void trace(Throwable t) {
        if (log.isTraceEnabled()) {
            log.trace(t);
        }
    }

    /**
     * Fatalレベル設定状態確認。
     * 
     * @return 有効(true)/無効(false)。
     */
    public boolean isFatalEnabled() {
        return log.isFatalEnabled();
    }

    /**
     * Errorレベル設定状態確認。
     * 
     * @return 有効(true)/無効(false)。
     */
    public boolean isErrorEnabled() {
        return log.isErrorEnabled();
    }

    /**
     * Warnレベル設定状態確認。
     * 
     * @return 有効(true)/無効(false)。
     */
    public boolean isWarnEnabled() {
        return log.isWarnEnabled();
    }

    /**
     * Infoレベル設定状態確認。
     * 
     * @return 有効(true)/無効(false)。
     */
    public boolean isInfoEnabled() {
        return log.isInfoEnabled();
    }

    /**
     * Debugレベル設定状態確認。
     * 
     * @return 有効(true)/無効(false)。
     */
    public boolean isDebugEnabled() {
        return log.isDebugEnabled();
    }

    /**
     * Traceレベル設定状態確認。
     * 
     * @return 有効(true)/無効(false)。
     */
    public boolean isTraceEnabled() {
        return log.isTraceEnabled();
    }

}
