/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.common.transformer.saml2;

import java.net.URI;
import java.net.URISyntaxException;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.exception.IllegalRequestException;
import jp.co.ogis_ri.citk.common.util.ExceptionUtil;

import org.mule.api.MuleMessage;
import org.mule.api.transformer.TransformerException;
import org.mule.transformer.AbstractMessageTransformer;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;

/**
 * Mule ESBのACS処理トランスフォーマークラス。
 * 
 * @author ISP Shiraishi
 * 
 */
public class ReceiveRelayStateTransformer extends AbstractMessageTransformer {

    /**
     * コンストラクタ。
     */
    public ReceiveRelayStateTransformer() {
        super();
    }

    /**
     * RelayStateパラメータキー。
     */
    public static final String KEY = CitkConstants.RELAYSTATE_PARAM + "=";

    /**
     * AssertionConsumerServiceとして設けたMuleのエンドポイントに対し
     * 認証結果として送られてきたPOSTリクエストから
     * RelayState値を取り出しMuleMessageのセッションプロパティにセットする。
     * 
     * @param message 変換対象のMuleメッセージ。
     * @param outputEncoding エンコード。
     * @return 変換後のMuleメッセージ。
     * @throws TransformerException。
     */
    @Override
    public Object transformMessage(MuleMessage message, String outputEncoding)
            throws TransformerException {

        // HTTP Inbound Endpoint Transformer
        String method = message
                .getInboundProperty(HttpConnector.HTTP_METHOD_PROPERTY);
        String queryString = null;
        String relayState = null;

        queryString = getQueryString(message, method);

        if (queryString == null) {
            throw new IllegalRequestException(
                    "E-0003", message);
        }

        if (queryString.contains(KEY)) {
            int startPosition = queryString.indexOf("&" + KEY) + KEY.length()
                    + 1;
            int endPosition = queryString.indexOf("&", startPosition);

            if (endPosition < 0) {
                // if RelayState is the last parameter.
                endPosition = queryString.length();
            }

            if (startPosition < endPosition) {
                relayState = queryString.substring(startPosition, endPosition);
            }
        }

        if (relayState == null) {
            throw new IllegalRequestException(
                    "E-0003", message);
        }

        message.setSessionProperty(CitkConstants.CITK_RELAYSTATE_KEY,
                relayState);

        return message;
    }

    /**
     * MuleMessageからリクエストクエリを文字列として抽出する。
     * 
     * @param message 対象のMuleメッセージ。
     * @param method リクエストメソッド。
     * @return リクエストクエリ文字列。
     * @throws TransformerException。
     * @throws URISyntaxException。
     */
    private String getQueryString(MuleMessage message, String method) {
        String query = null;
        if (HttpConstants.METHOD_GET.equals(method)) {
            URI uri = null;
            try {
                uri = new URI(message.getInboundProperty(
                        HttpConnector.HTTP_REQUEST_PROPERTY, ""));
            } catch (URISyntaxException e) {
                logger.debug(e);
                throw ExceptionUtil.convertRuntimeException(e);
            }

            query = uri.getRawQuery();

        } else if (HttpConstants.METHOD_POST.equals(method)) {
            try {
                query = message.getPayload(DataTypeFactory.STRING);
            } catch (TransformerException e) {
                logger.debug(e);
                throw ExceptionUtil.convertRuntimeException(e);
            }
        }
        return query;
    }
}
