/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.common.util;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.mule.api.MuleMessage;

/**
 * リクエストパラメータキャッシュ用クラス。 認証リクエストユーザのRelayStateの値をキーとし、
 * リソースに対するリクエストパラメータの値を含むMuleMessageオブジェクトを
 * 値として持つMapオブジェクトを「キャッシュ」とし、その保持と管理を行うクラス。
 * 
 * @author ISP Shiraishi
 * 
 */
public class RequestParameterCache {

    /**
     * コンストラクタ。クラスの外部からは利用しない。
     */
    protected RequestParameterCache() {
        cache = new ConcurrentHashMap<String, MuleMessage>();
        payloardCache = new ConcurrentHashMap<String, Object>();
    }

    /**
     * このクラスのシングルトンオブジェクト。
     */
    private static RequestParameterCache me = new RequestParameterCache();

    /**
     * cacheオブジェクト。
     */
    private Map<String, MuleMessage> cache;

    /**
     * 　ペイロードのcacheオブジェクト。
     */
    private Map<String, Object> payloardCache;

    /**
     * cacheオブジェクトを設定する。
     * 
     * @param cache　cacheオブジェクト。
     */
    public void setCache(Map<String, MuleMessage> cache) {
        this.cache = null;
        this.cache = cache;
        this.payloardCache = null;
        this.payloardCache = new ConcurrentHashMap<String, Object>();
        Set<String> cacheSet = cache.keySet();
        for (String name : cacheSet) {
            this.payloardCache.put(name, cache.get(name).getPayload());
        }
    }

    /**
     * cacheオブジェクトを取得する。
     * 
     * @return cacheオブジェクト。
     */
    public Map<String, MuleMessage> getCache() {
        return cache;
    }

    /**
     * 認証リクエストユーザのMuleメッセージをキャッシュに追加する。
     * 
     * @param key 追加するユーザのRelayStateの値。
     * @param value リクエストパラメータを含むMuleメッセージ。
     */
    public void add(String key, Object value) {
        if (key == null || value == null) {
            return;
        }
        if (key.isEmpty()) {
            return;
        }
        MuleMessage message = null;
        if (MuleMessage.class.isInstance(value)) {
            message = (MuleMessage) value;
        } else {
            return;
        }

        if (!cache.containsKey(key)) {
            cache.put(key, message);
            payloardCache.put(key, message.getPayload());
        }
        return;
    }

    /**
     * キャッシュ内のMuleMessageを削除する。
     * 
     * @param key 削除対象のRelayStateの値。
     */
    public void remove(String key) {
        if (key == null) {
            return;
        }
        if (cache.containsKey(key)) {
            cache.remove(key);
            payloardCache.remove(key);
        }
    }

    /**
     * キャッシュ内のMuleMessageを更新する。
     * 
     * @param key 更新対象のRelayStateの値。
     * @param value 更新する値。
     */
    public synchronized void update(String key, Object value) {
        remove(key);
        add(key, value);
    }

    /**
     * キャッシュ内のMuleMessageを取得する。
     * 
     * @param key RelayStateの値。
     * @return キャッシュ内に存在する、keyに該当するMuleメッセージ。 存在しない場合はnullを返す。
     */
    public MuleMessage get(String key) {
        if (key == null) {
            return null;
        }
        MuleMessage val = null;
        if (cache.containsKey(key)) {
            val = cache.get(key);
            val.setPayload(payloardCache.get(key));
        }
        return val;
    }

    /**
     * このオブジェクトのシングルトンインスタンスを取得する。
     * 
     * @return RequestParameterCacheオブジェクト。
     */
    public static RequestParameterCache getInstance() {
        return me;
    }
}
