/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.common.exception;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;

import org.junit.Test;
import org.mule.DefaultMuleEvent;
import org.mule.DefaultMuleMessage;
import org.mule.MessageExchangePattern;
import org.mule.api.MuleEvent;
import org.mule.api.MuleMessage;
import org.mule.api.MuleSession;
import org.mule.api.transformer.TransformerMessagingException;
import org.mule.tck.MuleTestUtils;
import org.mule.tck.junit4.AbstractMuleContextTestCase;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;

public class CitkExceptionStrategyTestCase extends AbstractMuleContextTestCase {

    private static String systemExceptionTestMessage = "認可処理に失敗しました";

    private static String illegalRequestTestMessage =
            "[E-0002]RelayStateが不正です。リクエストメッセージキャッシュが取得できません.";

    private static final String fedleTestMessage =
            "[E-0002]RelayStateが不正です。リクエストメッセージキャッシュが取得できません."
                    + " Fedlet-URL : http://dummyurl.com"
                    + " , Fedlet-Response : testPayloard";

    private static final String endPointMappingTestMessage =
            "[E-0002]RelayStateが不正です。リクエストメッセージキャッシュが取得できません."
                    + " Key : idp101/app1" + ", Decision : Permit"
                    + ", NameID : testNameID";

    @Test
    public void testHandleException_CitkSystemException() {
        CitkExceptionStrategy target = new CitkExceptionStrategy(muleContext);

        MuleMessage okMessage = new DefaultMuleMessage("OK", muleContext);
        systemExceptionTestMessage =
                systemExceptionTestMessage + " Mule Message: "
                        + okMessage.toString();

        MuleSession session = MuleTestUtils.getTestSession(muleContext);

        MuleEvent event =
                new DefaultMuleEvent(okMessage,
                        MessageExchangePattern.REQUEST_RESPONSE, session);

        CitkSystemException citkSystemException =
                new CitkSystemException("認可処理に失敗しました", okMessage);
        TransformerMessagingException transformerMessagingException =
                new TransformerMessagingException(null, event, null,
                        citkSystemException);

        MuleEvent result = null;

        result = target.handleException(transformerMessagingException, event);

        MuleMessage resultMessage = result.getMessage();

        assertNull(resultMessage.getExceptionPayload());
        assertEquals(resultMessage.getPayload(), systemExceptionTestMessage);
        assertEquals(
                resultMessage.getOutboundProperty(HttpConstants.HEADER_CONTENT_LENGTH),
                systemExceptionTestMessage.length());
        assertEquals(
                resultMessage.getOutboundProperty(HttpConnector.HTTP_STATUS_PROPERTY),
                500);
        assertEquals(
                resultMessage.getOutboundProperty(HttpConstants.HEADER_CONTENT_TYPE),
                "text/html; charset=Shift_JIS");
    }

    @Test
    public void testHandleException_IllegalRequestException() {
        CitkExceptionStrategy target = new CitkExceptionStrategy(muleContext);

        MuleMessage okMessage = new DefaultMuleMessage("OK", muleContext);

        MuleSession session = MuleTestUtils.getTestSession(muleContext);

        MuleEvent event =
                new DefaultMuleEvent(okMessage,
                        MessageExchangePattern.REQUEST_RESPONSE, session);
        IllegalRequestException illegalRequestException =
                new IllegalRequestException(new NullPointerException(),
                        "E-0002", okMessage, "nameID=aaabbbccc");
        TransformerMessagingException transformerMessagingException =
                new TransformerMessagingException(null, event, null,
                        illegalRequestException);

        MuleEvent result = null;

        result = target.handleException(transformerMessagingException, event);

        MuleMessage resultMessage = result.getMessage();

        assertNull(resultMessage.getExceptionPayload());

        assertEquals(resultMessage.getPayload(), illegalRequestTestMessage);
        assertEquals(
                resultMessage.getOutboundProperty(HttpConstants.HEADER_CONTENT_LENGTH),
                illegalRequestTestMessage.length());
        assertEquals(
                resultMessage.getOutboundProperty(HttpConnector.HTTP_STATUS_PROPERTY),
                404);
        assertEquals(
                resultMessage.getOutboundProperty(HttpConstants.HEADER_CONTENT_TYPE),
                "text/html; charset=Shift_JIS");
    }

    @Test
    public void testHandleException_fedletException() {
        CitkExceptionStrategy target = new CitkExceptionStrategy(muleContext);

        MuleMessage okMessage = new DefaultMuleMessage("OK", muleContext);
        MuleSession session = MuleTestUtils.getTestSession(muleContext);

        MuleEvent event =
                new DefaultMuleEvent(okMessage,
                        MessageExchangePattern.REQUEST_RESPONSE, session);
        FedletException fedletException =
                new FedletException(new NullPointerException(), "E-0002",
                        "http://dummyurl.com", "testPayloard",
                        "nameID=aaabbbccc");
        TransformerMessagingException transformerMessagingException =
                new TransformerMessagingException(null, event, null,
                        fedletException);

        MuleEvent result = null;

        result = target.handleException(transformerMessagingException, event);

        MuleMessage resultMessage = result.getMessage();

        assertNull(resultMessage.getExceptionPayload());
        assertEquals(resultMessage.getPayload(), fedleTestMessage);
        assertEquals(
                resultMessage.getOutboundProperty(HttpConstants.HEADER_CONTENT_LENGTH),
                fedleTestMessage.length());
        assertEquals(
                resultMessage.getOutboundProperty(HttpConnector.HTTP_STATUS_PROPERTY),
                500);
        assertEquals(
                resultMessage.getOutboundProperty(HttpConstants.HEADER_CONTENT_TYPE),
                "text/html; charset=Shift_JIS");
    }

    @Test
    public void testHandleException_endPointMappingException() {
        CitkExceptionStrategy target = new CitkExceptionStrategy(muleContext);

        MuleMessage okMessage = new DefaultMuleMessage("OK", muleContext);
        MuleSession session = MuleTestUtils.getTestSession(muleContext);

        MuleEvent event =
                new DefaultMuleEvent(okMessage,
                        MessageExchangePattern.REQUEST_RESPONSE, session);
        EndPointMappingException endPointException =
                new EndPointMappingException(new NullPointerException(),
                        "E-0002", "idp101/app1", "Permit", "testNameID",
                        "nameID=aaabbbccc");
        TransformerMessagingException transformerMessagingException =
                new TransformerMessagingException(null, event, null,
                        endPointException);

        MuleEvent result = null;

        result = target.handleException(transformerMessagingException, event);

        MuleMessage resultMessage = result.getMessage();

        assertNull(resultMessage.getExceptionPayload());
        assertEquals(resultMessage.getPayload(), endPointMappingTestMessage);
        assertEquals(
                resultMessage.getOutboundProperty(HttpConstants.HEADER_CONTENT_LENGTH),
                endPointMappingTestMessage.length());
        assertEquals(
                resultMessage.getOutboundProperty(HttpConnector.HTTP_STATUS_PROPERTY),
                500);
        assertEquals(
                resultMessage.getOutboundProperty(HttpConstants.HEADER_CONTENT_TYPE),
                "text/html; charset=Shift_JIS");
    }

}
