/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.common.transformer.http;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.fail;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.commons.httpclient.Cookie;
import org.apache.commons.httpclient.cookie.MalformedCookieException;
import org.junit.Test;
import org.mule.DefaultMuleMessage;
import org.mule.RequestContext;
import org.mule.api.MuleMessage;
import org.mule.api.endpoint.EndpointBuilder;
import org.mule.api.endpoint.ImmutableEndpoint;
import org.mule.api.transformer.Transformer;
import org.mule.api.transport.PropertyScope;
import org.mule.endpoint.EndpointURIEndpointBuilder;
import org.mule.tck.junit4.AbstractMuleContextTestCase;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transport.http.CookieHelper;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;

public class PostPropertiesTransformerTestCase extends
        AbstractMuleContextTestCase {

    private static String encoding = "UTF-8";
    private static String cookieName = "testCookie";
    private static final String PAYLOAD =
            "IDToken0=&IDToken1=amadmin&IDToken2=testadmin&IDButton=Submit"
                    + "&goto=L1NTT1JlZGlyZWN0L21ldGFBbGlhcy9jaXRraWRwP1JlcUlEPXMyYjljZjBiMmE3MjkzMWFhYzU2MDNiMzFiN"
                    + "GQxZTQ5NTJiZWU0OTcyZQ%3D%3D&SunQueryParamsString=cmVhbG09LyZmb3J3YXJkPXRydWUmU0FNTFJlcXVlc3"
                    + "Q9blZOTGo5b3dFTDd2cjRoOGh6eFlDbGdFaVlLcUltMjdLVWw3NktWeW5FblhhbUtubnNrdSsrOXJCNHFvdXVLQVpDb"
                    + "lNlUHpOOTVnc1ViUk54OWM5UGVrOS9PNEJLUWdPYmFPUkR6Y3A2NjNtUnFCQ3JrVUx5RW55ZlAzcGdTZmppSGZXa0pH"
                    + "bVlYZTdiY293S1JleWpzcEV6SkxGSkJaQ1R0OUZrM0lTbC9kVkRQZUxhVktDKzh3U1lNRTNzS2lNVHBtRFljRU9zWWV"
                    + "kUmhLYVhDbUs0NUU3MGJ5STVqeWF1Zk9kQlZ0SFRXbEJ3NnNub282SElibmFXQ3I2TlphbTVmTm9Ib1dtQThmengySW"
                    + "FoM24rdUlkS1daQVV0a0JpM1NpQm9XOVhWY2VDRDhaS0dJU25yQllOZ3VlUkNVVDFET2RLZGxMNFh1bEs2Wi9YN1NpU"
                    + "FRjZy9Ga1UyeWg3emdnVnJSTENlOU1abzdGdXdPZGhuSmVIci91RXNBN3RMRWRFOEZCTFo2bTdwRStDRE4vWWlrK3Nj"
                    + "eE45NWJPVXhzVnVHRnpBbjBJNS9kZzkzMjh3MFNyN2VFcmgzcnhWMHZkdFhWRFdxaDFaT1ZtaFVvSWtGZWVibmYrbEZ"
                    + "vMm9GTm1WSHFzNnRwakV2R3d1Q1hBUmtlMkNySS85L0daOWxuRllXcWlGSFp6SEJnVzZSc3pGdEo2eEN2MXh3RUpMYz"
                    + "RNRzNTK0JONDl6ZFEzMUxHRmZiSkpjZTJwWDkvcjBZVy9tOWMzc0xWZUZkNjR5bFU0NXY4VGw3OUtZZHpxencvNzk4O"
                    + "VFjPSZSZWxheVN0YXRlPS8mc3BFbnRpdHlJRD1jaXRrc3A%3D"
                    + "&encoded=true&gx_charset=UTF-8";

    private static Map<String, Object> messageProperties;

    static {
        Map<String, Object> map = new HashMap<String, Object>();
        map.put(HttpConnector.HTTP_METHOD_PROPERTY, HttpConstants.METHOD_POST);
        map.put(HttpConstants.HEADER_ACCEPT,
                " text/html, application/xhtml+xml, */*");
        map.put(HttpConstants.HEADER_ACCEPT_LANGUAGE, " ja-JP");
        map.put(HttpConstants.HEADER_CONNECTION, " Keep-Alive");
        map.put(HttpConstants.HEADER_HOST, " www.acme.com");
        try {
            map.put(HttpConnector.HTTP_COOKIES_PROPERTY,
                    CookieHelper.parseCookiesAsAClient("testCookie=value", null));
        } catch (MalformedCookieException e) {
            fail(e.getMessage());
        }
        map.put(HttpConstants.HEADER_CONTENT_TYPE,
                PostPropertiesTransformer.CONTENT_TYPE_X_WWW_FORM_URL_ENCODED);
        map.put(HttpConstants.HEADER_CONTENT_LENGTH, PAYLOAD.length());

        messageProperties = Collections.unmodifiableMap(map);
    }

    // setup 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    public void doSetUp() throws Exception {
        super.doSetUp();
        // トランスフォーマのテストをする際は、必ずRequestContextのイベントが設定されてなければならない。
        RequestContext.setEvent(getTestEvent("test"));
    }

    // tear down 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    protected void doTearDown() throws Exception {
        super.doTearDown();
        // RequestContext の値をnullに戻し、次のテストメソッドに備える。
        RequestContext.setEvent(null);
    }

    public Transformer getTransformer() throws Exception {
        Transformer trans = createObject(PostPropertiesTransformer.class);
        trans.setReturnDataType(DataTypeFactory.create(MuleMessage.class));

        EndpointBuilder builder =
                new EndpointURIEndpointBuilder("test://test", muleContext);
        builder.setEncoding(encoding);
        ImmutableEndpoint endpoint =
                muleContext.getEndpointFactory().getOutboundEndpoint(builder);
        trans.setEndpoint(endpoint);
        return trans;
    }

    public Object getTestData() {
        MuleMessage mes =
                new DefaultMuleMessage(PAYLOAD, messageProperties, null, null,
                        muleContext);
        return mes;
    }

    public Object getResultData() {
        Map<String, Object> map = new HashMap<String, Object>();
        String[] names =
                new String[] {HttpConstants.HEADER_CONTENT_TYPE,
                        HttpConnector.HTTP_METHOD_PROPERTY,
                        HttpConstants.HEADER_CONTENT_LENGTH,
                        HttpConstants.HEADER_CONTENT_LENGTH};

        for (String name : names) {
            copyProperty(map, name);
        }

        MuleMessage mes =
                new DefaultMuleMessage(PAYLOAD, messageProperties, map, null,
                        muleContext);

        mes.setSessionProperty(HttpConnector.HTTP_COOKIES_PROPERTY,
                messageProperties.get(HttpConnector.HTTP_COOKIES_PROPERTY));

        return mes;
    }

    @Test
    public void testTransform() throws Exception {
        Transformer trans = this.getTransformer();
        Object result = trans.transform(getTestData());

        MuleMessage resultMes = (MuleMessage) result;
        Map<String, Object> props = new HashMap<String, Object>();
        Set<String> names = resultMes.getPropertyNames(PropertyScope.OUTBOUND);

        assertFalse(names.contains(HttpConstants.HEADER_ACCEPT_ENCODING));

        for (String name : names) {
            props.put(name, resultMes.getOutboundProperty(name));
        }

        Map<String, Object> expectedProps = new HashMap<String, Object>();
        expectedProps.putAll(messageProperties);
        expectedProps.remove(HttpConstants.HEADER_ACCEPT_ENCODING);

        assertFalse(expectedProps.equals(props));

        Cookie[] cookies =
                resultMes.getSessionProperty(HttpConnector.HTTP_COOKIES_PROPERTY);

        assertNotNull(cookies);
        assertFalse(cookies.length == 0);
        assertEquals(cookies[0].getName(), cookieName);
    }

    private void copyProperty(Map<String, Object> map, String name) {
        map.put(name, messageProperties.get(name));
    }

}
