/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.common.transformer.saml2;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import jp.co.ogis_ri.citk.common.exception.CitkSystemException;
import jp.co.ogis_ri.citk.common.exception.IllegalRequestException;

import org.junit.Test;
import org.mule.DefaultMuleMessage;
import org.mule.RequestContext;
import org.mule.api.MuleMessage;
import org.mule.api.endpoint.EndpointBuilder;
import org.mule.api.endpoint.ImmutableEndpoint;
import org.mule.api.transformer.Transformer;
import org.mule.api.transport.PropertyScope;
import org.mule.endpoint.EndpointURIEndpointBuilder;
import org.mule.tck.junit4.AbstractMuleContextTestCase;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;

public class ReceiveRelayStateTransformerTestCase extends
        AbstractMuleContextTestCase {
    private static String encoding = "UTF-8";
    private static Map<String, Object> messageProperties;
    private static String relayState = "388e2436-2182-11e1-b4fa-af7760430a0a";
    private static String path = "/acs";
    private static String invalidPath = "/%";

    private static String payload =
            "SAMLResponse=DUMMY%3CPHNhbWxwOlJlc3BvbnNlIHhtbG5zOnNhbWxwPSJ1cm46b2FzaXM6bm"
                    + "&" + ReceiveRelayStateTransformer.KEY + relayState;

    static {
        Map<String, Object> map = new HashMap<String, Object>();
        map.put(HttpConnector.HTTP_METHOD_PROPERTY, "POST");
        map.put(HttpConstants.HEADER_ACCEPT,
                "text/html, application/xhtml+xml, */*");
        map.put(HttpConstants.HEADER_ACCEPT_LANGUAGE, "ja,en-US;q=0.8,en;q=0.6");
        map.put(HttpConstants.HEADER_CONNECTION, "Keep-Alive");
        map.put(HttpConstants.HEADER_HOST, "www.acme.com");
        map.put(HttpConstants.HEADER_COOKIE, "testCookie=value");
        map.put(HttpConnector.HTTP_CONTEXT_PATH_PROPERTY, "/");
        map.put(HttpConnector.HTTP_REQUEST_PROPERTY, path);
        map.put(HttpConnector.HTTP_REQUEST_PATH_PROPERTY, path);
        map.put(HttpConnector.HTTP_VERSION_PROPERTY, HttpConstants.HTTP11);

        messageProperties = Collections.unmodifiableMap(map);
    }

    // setup 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    public void doSetUp() throws Exception {
        super.doSetUp();
        // トランスフォーマのテストをする際は、必ずRequestContextのイベントが設定されてなければならない。
        RequestContext.setEvent(getTestEvent("test"));
    }

    // tear down 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    protected void doTearDown() throws Exception {
        super.doTearDown();
        // RequestContext の値をnullに戻し、次のテストメソッドに備える。
        RequestContext.setEvent(null);
    }

    public Transformer getTransformer() throws Exception {
        Transformer trans = createObject(ReceiveRelayStateTransformer.class);
        trans.setReturnDataType(DataTypeFactory.create(MuleMessage.class));

        EndpointBuilder builder =
                new EndpointURIEndpointBuilder("test://test", muleContext);
        builder.setEncoding(encoding);
        ImmutableEndpoint endpoint =
                muleContext.getEndpointFactory().getInboundEndpoint(builder);
        trans.setEndpoint(endpoint);
        return trans;
    }

    public Object getTestData() {
        MuleMessage mes =
                new DefaultMuleMessage(payload, messageProperties, null, null,
                        muleContext);
        return mes;
    }

    public Object getTestData_InvalidPath_Exception() {
        MuleMessage mes =
                new DefaultMuleMessage(payload, messageProperties, null, null,
                        muleContext);
        mes.setProperty(HttpConnector.HTTP_METHOD_PROPERTY, "GET",
                PropertyScope.INBOUND);
        mes.setProperty(HttpConnector.HTTP_REQUEST_PROPERTY, invalidPath,
                PropertyScope.INBOUND);
        mes.setProperty(HttpConnector.HTTP_REQUEST_PATH_PROPERTY, invalidPath,
                PropertyScope.INBOUND);

        return mes;
    }

    public Object getTestData_NullQuery_Exception() {
        MuleMessage mes =
                new DefaultMuleMessage(payload, messageProperties, null, null,
                        muleContext);
        mes.setProperty(HttpConnector.HTTP_METHOD_PROPERTY, "GET",
                PropertyScope.INBOUND);
        mes.setProperty(HttpConnector.HTTP_REQUEST_PROPERTY, "",
                PropertyScope.INBOUND);
        mes.setProperty(HttpConnector.HTTP_REQUEST_PATH_PROPERTY, "",
                PropertyScope.INBOUND);

        return mes;
    }

    public Object getTestData_NoRelayState_Exception() {
        MuleMessage mes =
                new DefaultMuleMessage(payload, messageProperties, null, null,
                        muleContext);
        mes.setProperty(HttpConnector.HTTP_METHOD_PROPERTY, "GET",
                PropertyScope.INBOUND);
        mes.setProperty(HttpConnector.HTTP_REQUEST_PROPERTY, path
                + "?testkey=testValue", PropertyScope.INBOUND);
        mes.setProperty(HttpConnector.HTTP_REQUEST_PATH_PROPERTY, path
                + "?testkey=testValue", PropertyScope.INBOUND);

        return mes;
    }

    public Object getResultData() {
        MuleMessage mes =
                new DefaultMuleMessage(payload, messageProperties,
                        messageProperties, null, muleContext);
        return mes;
    }

    @Test
    public void testTransform() throws Exception {
        Transformer trans = this.getTransformer();
        Object result = trans.transform(getTestData());

        MuleMessage resultMes = (MuleMessage) result;

        assertEquals(relayState,
                (String) resultMes.getSessionProperty("citk.relaystate"));

    }

    @Test(expected = CitkSystemException.class)
    public void testTransform_InvalidPath_Exception() throws Exception {
        Transformer trans = this.getTransformer();
        // Exception発生
        trans.transform(getTestData_InvalidPath_Exception());

        fail("期待するExceptionが発生しませんでした。");

    }

    @Test(expected = IllegalRequestException.class)
    public void testTransform_NoRelayState_Exception() throws Exception {
        Transformer trans = this.getTransformer();
        // Exception発生
        trans.transform(getTestData_NoRelayState_Exception());

        fail("期待するExceptionが発生しませんでした。");

    }

}
