/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.common.util;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.fail;

import java.io.IOException;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.httpclient.Cookie;
import org.junit.Test;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

public class DataUtilTestCase {
    public static String xmlheader =
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
    public static String xmlbody =
            "<samlp:Response xmlns:samlp=\"urn:oasis:names:tc:SAML:2.0:protocol\" ID=\"s20872a3bfbbe9329b48b061b8db2fbd1bf51cc76b\" InResponseTo=\"s25dd2e066a32af777bb073f90a3dbec9c35e71320\" Version=\"2.0\" IssueInstant=\"2011-02-24T07:31:42Z\" Destination=\"http://my.citk.org:8888/mule/fedletapplication\"><saml:Issuer xmlns:saml=\"urn:oasis:names:tc:SAML:2.0:assertion\">idp2</saml:Issuer><samlp:Status xmlns:samlp=\"urn:oasis:names:tc:SAML:2.0:protocol\">"
                    + "<samlp:StatusCode  xmlns:samlp=\"urn:oasis:names:tc:SAML:2.0:protocol\" "
                    + "Value=\"urn:oasis:names:tc:SAML:2.0:status:Success\">"
                    + "</samlp:StatusCode>"
                    + "</samlp:Status><saml:Assertion xmlns:saml=\"urn:oasis:names:tc:SAML:2.0:assertion\" xmlns:samlp=\"urn:oasis:names:tc:SAML:2.0:protocol\" ID=\"s2b558bfa6b18769837e177072262a059af36d12b6\" IssueInstant=\"2011-02-24T07:31:42Z\" Version=\"2.0\">"
                    + "<saml:Issuer>idp2</saml:Issuer>"
                    + "<saml:Subject>"
                    + "<saml:NameID Format=\"urn:oasis:names:tc:SAML:2.0:nameid-format:transient\" NameQualifier=\"idp2\" SPNameQualifier=\"mule\">"
                    + "9dc5tjemuEQzz83oAxWcqSIdUNRA </saml:NameID><saml:SubjectConfirmation Method=\"urn:oasis:names:tc:SAML:2.0:cm:bearer\">"
                    + "<saml:SubjectConfirmationData InResponseTo=\"s25dd2e066a32af777bb073f90a3dbec9c35e71320\" NotOnOrAfter=\"2011-01-31T07:41:42Z\" Recipient=\"http://my.citk.org:8888/mule/fedletapplication\"/></saml:SubjectConfirmation>"
                    + "</saml:Subject><saml:Conditions NotBefore=\"2011-03-31T07:21:42Z\" NotOnOrAfter=\"2011-02-24T07:41:42Z\">"
                    + "<saml:AudienceRestriction>"
                    + "<saml:Audience>mule</saml:Audience>"
                    + "</saml:AudienceRestriction>"
                    + "</saml:Conditions>"
                    + "<saml:AuthnStatement AuthnInstant=\"2011-02-24T07:31:42Z\" SessionIndex=\"s2b25fd021dd54b42edb8d8024d2355dfb6864ce01\"><saml:AuthnContext><saml:AuthnContextClassRef>urn:oasis:names:tc:SAML:2.0:ac:classes:PasswordProtectedTransport</saml:AuthnContextClassRef></saml:AuthnContext></saml:AuthnStatement></saml:Assertion></samlp:Response>";

    @Test
    public void testGetCookieValue() {
        Cookie[] cookies =
                new Cookie[] {new Cookie(".citk.org", "cookie1", "value1"),
                        new Cookie(".citk.org", "cookie2", "value2"),
                        new Cookie(".citk.org", "cookie3", "value3"),};
        String name = null;
        String value = null;
        name = "cookie1";
        value = DataUtil.getCookieValue(cookies, name);
        assertNotNull(value);
        assertEquals("value1", value);
        name = "cookie2";
        value = DataUtil.getCookieValue(cookies, name);
        assertNotNull(value);
        assertEquals("value2", value);
        name = "cookie3";
        value = DataUtil.getCookieValue(cookies, name);
        assertNotNull(value);
        assertEquals("value3", value);
        name = "nosuchcookie";
        value = DataUtil.getCookieValue(cookies, name);
        assertNull(value);

    }

    @Test
    public void testParseDocument() {
        Object obj = null;
        Document doc = null;
        String src = xmlheader + xmlbody;

        String value = null;
        Date date = null;
        try {
            doc = DataUtil.parseDocument(null);
            assertNull(doc);
            doc = DataUtil.parseDocument("");
            assertNull(doc);
            try {
                doc = DataUtil.parseDocument(xmlheader);
                fail("期待するExceptionが発生しませんでした。");
            } catch (Exception e) {
            }
            try {
                doc = DataUtil.parseDocument(xmlbody);
                assertNotNull(doc);
                value = DataUtil.getNameId(doc);
                assertEquals("9dc5tjemuEQzz83oAxWcqSIdUNRA", value);
            } catch (Exception e) {
                fail(e.getMessage());
            }

            doc = DataUtil.parseDocument(src);
            assertNotNull(doc);

            value = DataUtil.getNameId(doc);
            assertEquals("9dc5tjemuEQzz83oAxWcqSIdUNRA", value);
            value = DataUtil.getNameId(src);
            assertEquals("9dc5tjemuEQzz83oAxWcqSIdUNRA", value);
            obj = doc;
            value = DataUtil.getNameId(obj);
            assertEquals("9dc5tjemuEQzz83oAxWcqSIdUNRA", value);
            obj = src;
            value = DataUtil.getNameId(obj);
            assertEquals("9dc5tjemuEQzz83oAxWcqSIdUNRA", value);

            value = DataUtil.getExpierDateTime(doc);
            assertEquals("2011-01-31T07:41:42Z", value);
            value = DataUtil.getExpierDateTime(src);
            assertEquals("2011-01-31T07:41:42Z", value);
            obj = doc;
            value = DataUtil.getExpierDateTime(obj);
            assertEquals("2011-01-31T07:41:42Z", value);
            obj = src;
            value = DataUtil.getExpierDateTime(obj);
            assertEquals("2011-01-31T07:41:42Z", value);

            date = DataUtil.parseStringToDate(value);
            assertNotNull(date);

            Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
            cal.clear(); // it is needed for clear below of the second
            cal.set(2011, 0, 31, 7, 41, 42);
            assertEquals(cal.getTime().getTime(), date.getTime());

        } catch (ParserConfigurationException e) {
            fail(e.getMessage());
        } catch (IOException e) {
            fail(e.getMessage());
        } catch (SAXException e) {
            fail(e.getMessage());
        } catch (Exception e) {
            fail(e.getMessage());
        }
    }

    @Test
    public void testStringToDate() {
        Calendar cal = Calendar.getInstance();
        String value = null;
        Date date = null;

        try {
            value = "2011-01-31T07:41:42Z";
            date = DataUtil.parseStringToDate(value);
            assertNotNull(date);
            cal.setTimeZone(TimeZone.getTimeZone("UTC"));
            cal.clear();
            cal.set(2011, 0, 31, 7, 41, 42);
            assertEquals(cal.getTime().getTime(), date.getTime());
        } catch (ParseException e) {
            fail(e.getMessage());
        }

        try {
            value = "2011-01-31T07:41:42+0000";
            date = DataUtil.parseStringToDate(value);
            assertNotNull(date);
            cal.setTimeZone(TimeZone.getTimeZone("UTC"));
            cal.clear();
            cal.set(2011, 0, 31, 7, 41, 42);
            assertEquals(cal.getTime().getTime(), date.getTime());
        } catch (ParseException e) {
            fail(e.getMessage());
        }

        try {
            value = "2011-01-31T07:41:42+0900";
            date = DataUtil.parseStringToDate(value);
            assertNotNull(date);
            cal.setTimeZone(TimeZone.getTimeZone("JST"));
            cal.clear();
            cal.set(2011, 0, 31, 7, 41, 42);
            assertEquals(cal.getTime().getTime(), date.getTime());
        } catch (ParseException e) {
            fail(e.getMessage());
        }

        try {
            value = "2011-01-31T07:41:42+09:00";
            date = DataUtil.parseStringToDate(value);
            assertNotNull(date);
            cal.setTimeZone(TimeZone.getTimeZone("JST"));
            cal.clear();
            cal.set(2011, 0, 31, 7, 41, 42);
            assertEquals(cal.getTime().getTime(), date.getTime());
        } catch (ParseException e) {
            fail(e.getMessage());
        }

        try {
            value = "2011-01-31T07:41:42";
            date = DataUtil.parseStringToDate(value);
            assertNull(date);
        } catch (ParseException e) {
            fail(e.getMessage());
        }

        try {
            value = "2011/01/31 07:41:42";
            date = DataUtil.parseStringToDate(value);
            assertNull(date);
        } catch (ParseException e) {
            fail(e.getMessage());
        }

        try {
            value = "2011-01-31";
            date = DataUtil.parseStringToDate(value);
            assertNull(date);
        } catch (ParseException e) {
            fail(e.getMessage());
        }

        try {
            value = "03:02:11";
            date = DataUtil.parseStringToDate(value);
            assertNull(date);
        } catch (ParseException e) {
            fail(e.getMessage());
        }

        try {
            value = "2011-01-32T00:00:00Z";
            date = DataUtil.parseStringToDate(value);
            cal.setTimeZone(TimeZone.getTimeZone("UTC"));
            cal.clear();
            cal.set(2011, 1, 1, 0, 0, 0);
            assertEquals(cal.getTime().getTime(), date.getTime());
        } catch (ParseException e) {
            fail(e.getMessage());
        }

        try {
            value = "2011/01/31 07:41:42+0000";
            date = DataUtil.parseStringToDate(value);
            fail("期待するExceptionが発生しませんでした。");
        } catch (ParseException e) {
        }

    }

}
