/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.forwarding.transformer;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.exception.EndPointMappingException;
import jp.co.ogis_ri.citk.common.log.CitkLogger;

import org.apache.commons.lang.StringUtils;
import org.mule.api.MuleMessage;
import org.mule.api.transformer.TransformerException;
import org.mule.transformer.AbstractMessageTransformer;
import org.mule.transport.http.HttpConnector;

/**
 * Mule ESBのフォワーディング処理トランスフォーマークラス。
 * 
 * @author ISP Shiraishi
 * 
 */
public class ForwardingOutboundTransformer extends AbstractMessageTransformer {
    /**
     * ログメッセージ出力用ロガー。
     */
    private static final CitkLogger logger =
            CitkLogger.getLog(ForwardingOutboundTransformer.class);

    /**
     * ログメッセージ出力用ロガー。
     */
    private static final CitkLogger opLogger = CitkLogger.getOperationLog();

    /**
     * 　アウトバウンド解決クラス。
     */
    OutboundEndpointResolver outboundEndpointResolver;

    /**
     * コンストラクタ。
     */
    public ForwardingOutboundTransformer() {
        super();
    }

    /**
     * アウトバウンド解決クラスを取得する。
     * 
     * @return アウトバウンド解決クラス
     */
    public OutboundEndpointResolver getOutboundEndpointResolver() {
        return outboundEndpointResolver;
    }

    /**
     * アウトバウンド解決クラスを設定する。
     * 
     * @param outboundEndpointResolver　アウトバウンド解決クラス
     */
    public void setOutboundEndpointResolver(
            OutboundEndpointResolver outboundEndpointResolver) {
        this.outboundEndpointResolver = outboundEndpointResolver;
    }

    /**
     * フォワーディング処理を行う。
     * 
     * @param message 変換対象のMuleメッセージ。
     * @param outputEncoding エンコード
     * @return 変換後のMuleメッセージ。
     * @throws TransformerException。
     */
    @Override
    public Object transformMessage(MuleMessage message, String outputEncoding)
            throws TransformerException {
        String messageID = message.getMessageRootId();
        String targetKey = null;
        String authzResult = null;
        String path = null;

        // Resource URI を取得する。（ホスト名は含まない。パスとクエリが含まれる。）
        path = message.getInboundProperty(HttpConnector.HTTP_REQUEST_PROPERTY);

        // Authzコンポーネントから結果を受け取る
        authzResult =
                message.getSessionProperty(CitkConstants.AUTHZ_RESULT_KEY);

        // 先頭の'/' を削除する。 Mule が 余分な'/' をつけてしまうらしい。
        path = path.substring(1);

        targetKey = getTargetKey(message, path);

        logger.debug(
                "call resolve method [ muleMessageId : {0}, nameId : {1}, targetKey : {2}, authzResult : {3}]",
                message.getUniqueId(),
                message.getInboundProperty(CitkConstants.NAMEID_KEY),
                targetKey, authzResult);
        OutboundEndpoint outboundEndpoint =
                outboundEndpointResolver.resolve(targetKey, authzResult);

        if (outboundEndpoint == null) { // no hit targetKey
            throw new EndPointMappingException(
                    "E-0202",
                    targetKey,
                    authzResult,
                    (String) message.getInboundProperty(CitkConstants.NAMEID_KEY));
        }

        String address = null;

        String protocol = outboundEndpoint.getProtocol();
        String host = outboundEndpoint.getHost();
        String port = Integer.toString(outboundEndpoint.getPort());
        String appContext = outboundEndpoint.getAppcontext();

        address =
                buildRequestURL(path, targetKey, host, port, appContext,
                        authzResult);

        opLogger.info("I-0301", messageID, protocol + "://" + address);

        message.setOutboundProperty(CitkConstants.OUTBOUND_ADDRESS, address);
        message.setOutboundProperty(CitkConstants.OUTBOUND_PROTOCOL, protocol);

        return message;
    }

    /**
     * 認可結果に応じたForwarding先URLを生成する。
     * 
     * @param path　パス。
     * @param targetKey　ターゲットキー。
     * @param host　ホスト。
     * @param port　ポート。
     * @param appContext　アプリケーションコンテキスト。
     * @param authzResult　認可結果。
     * @return　Forwarding先URL文字列。
     */
    private String buildRequestURL(String path, String targetKey, String host,
            String port, String appContext, String authzResult) {
        String address;

        if (CitkConstants.AUTHZ_PERMIT.equals(authzResult)) {
            String request = StringUtils.remove(path, targetKey + "/");
            address = host + ":" + port + appContext + request;
        } else {
            address = host + ":" + port + appContext; // error page
        }
        logger.debug("address : " + address);

        return address;
    }

    /**
     * パスからターゲットキー（サービス事業者ID/appID）を切り出す。
     * 
     * @param message　MuleMessage。
     * @param path　対象のパス。
     * @return　ターゲットキー。
     */
    private String getTargetKey(MuleMessage message, String path) {
        String targetKey;
        // サービス事業者IDとアプリケーションIDを切り離す
        String[] id = path.split("/");
        if (id.length < 2) {
            logger.debug("Invalid path : " + path);
            throw new EndPointMappingException("E-0004", null, null, null); // 不正なターゲットキー（フォーマット不正。）

        }
        targetKey = id[0] + "/" + id[1];
        return targetKey;
    }
}
