/*
 * Copyright (C) 2011-2012 OGIS-RI Co.,Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.wsp.cxf.jaxws.handler;

import static jp.co.ogis_ri.citk.common.CitkConstants.CITK_PREFIX;
import static jp.co.ogis_ri.citk.common.CitkConstants.IDP_ENTITYID_KEY;
import static jp.co.ogis_ri.citk.common.CitkConstants.NAMEID_KEY;

import java.util.Iterator;
import java.util.Map;

import javax.security.auth.Subject;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.ws.ProtocolException;
import javax.xml.ws.handler.MessageContext;
import javax.xml.ws.handler.soap.SOAPMessageContext;

import jp.co.ogis_ri.citk.common.log.CitkLogger;

import org.apache.cxf.binding.soap.SoapMessage;
import org.mule.api.MuleEvent;
import org.mule.api.MuleMessage;

/**
 * JAX-WS Handler(SOAPHandler) inbound component.
 * 
 */
public class WsHandler extends AbstractWsHandler {

    private static CitkLogger logger = CitkLogger.getLog(WsHandler.class);

    @Override
    protected void handleSecureMessage(SoapMessage message,
            SOAPMessageContext context, ThreadLocal<Subject> cred,
            Subject subject, Map<Object, Object> sharedMap) {

        // outboundMsg : [true: Outgoing RESPONSE, false :Incoming REQUEST]
        Boolean outboundMsg =
                (Boolean) context.get(MessageContext.MESSAGE_OUTBOUND_PROPERTY);

        if (!(outboundMsg.booleanValue())) {
            // incoming Request process
            setCredentialsToMuleMessage(context);
            handleIncomingRequest(message, context, cred, subject, sharedMap);

        } else {
            // outgoing Response process
            handleOutgoingResponse(message, context, cred, subject, sharedMap);
        }
    }

    private void setCredentialsToMuleMessage(SOAPMessageContext context) {

        Iterator<?> elements;
        SOAPElement citkElem = null;
        try {
            elements =
                    context.getMessage()
                            .getSOAPPart()
                            .getEnvelope()
                            .getHeader()
                            .getChildElements();
            while (elements.hasNext()) {
                SOAPElement tmp = (SOAPElement) elements.next();
                if (CITK_PREFIX.equals(tmp.getPrefix())) {
                    citkElem = tmp;
                    break;
                }
            }

        } catch (SOAPException e) {
            throw new ProtocolException(e);
        }

        if (citkElem == null) {
            throw new ProtocolException("CITK-credentials are not existed.");
        }

        MuleMessage message =
                ((MuleEvent) context.get("mule.event")).getMessage();

        elements = citkElem.getChildElements();
        while (elements.hasNext()) {
            SOAPElement tmp = (SOAPElement) elements.next();
            if (NAMEID_KEY.equals(tmp.getLocalName())) {
                message.setSessionProperty(NAMEID_KEY,
                        tmp.getAttribute("value"));
            } else if (IDP_ENTITYID_KEY.equals(tmp.getLocalName())) {
                message.setSessionProperty(IDP_ENTITYID_KEY,
                        tmp.getAttribute("value"));
            }
        }
    }

    private void handleIncomingRequest(SoapMessage message,
            SOAPMessageContext context, ThreadLocal<Subject> cred,
            Subject subject, Map<Object, Object> sharedMap) {

        logger.debug("Incomming Request : "
                + requestHandler.print(context.getMessage().getSOAPPart()));

        try {
            // validate REQUEST from Client
            requestHandler.validateRequest(context.getMessage(), subject,
                    sharedMap, null, null);
            synchronized (this) {
                cred.set(subject);
            }

            // set XMLReader for SOAP
            message.setContent(XMLStreamReader.class, getSOAPReader(context));

        } catch (Exception ex) {
            logger.error(ex, "validateRequest failed.");
            throw new ProtocolException(ex);
        }
    }

    private void handleOutgoingResponse(SoapMessage message,
            SOAPMessageContext context, ThreadLocal<Subject> cred,
            Subject subject, Map<Object, Object> sharedMap) {

        logger.debug("Outgoing Response : "
                + requestHandler.print(context.getMessage().getSOAPPart()));

        try {
            // secure RESPONSE to Client
            requestHandler.secureResponse(context.getMessage(), sharedMap);

        } catch (Exception ex) {
            logger.error(ex, "secureResponse failed.");
            throw new ProtocolException(ex);
        }
    }
}
