/* MushSignal.cc */
/* Created by Enomoto Sanshiro on 4 May 1998. */
/* Last updated by Enomoto Sanshiro on 2 Nobember 2000. */


#include <unistd.h>
#include <signal.h>
#include "MushSignal.hh"

using namespace std;


TMushSignalClient *TMushSignalHandler::_ClientList[NSIG];


TMushSignalHandler::TMushSignalHandler(void) 
{
    sigemptyset(&_SignalSet);
    sigemptyset(&_OldSignalSet);    
    _IsBlocking = false;
}

TMushSignalHandler::~TMushSignalHandler() 
{
    Reset();
}

void TMushSignalHandler::RegisterClient(int SignalId, TMushSignalClient *SignalClient) 
{
    if ((SignalId > 0) && (SignalId < NSIG)) {
	_ClientList[SignalId] = SignalClient;
	sigaddset(&_SignalSet, SignalId);
    }
    else if (SignalId == 0) {
	for (int i = 1; i < NSIG; i++) {
	    RegisterClient(i, SignalClient);
	}
    }
    else {
	// errno = EINVAL;
	// throw TSystemCallException("TMushSignalHandler::RegisterClient()");
    }
}

void TMushSignalHandler::StartHandling(int SignalId, THandlingOptionFlags OptionFlags) 
{
    _SignalAction.sa_handler = Handler;
    _SignalAction.sa_mask = _SignalSet;
    _SignalAction.sa_flags = 0;
    
    if (OptionFlags & hoRestartSystemCall) {
	_SignalAction.sa_flags |= SA_RESTART;
    }
#if 0
    if (OptionFlags & hoAddInformation) {
	_SignalAction.sa_flags |= SA_SIGINFO;
    }
    if (OptionFlags & hoNoChildWait) {
	_SignalAction.sa_flags |= SA_NOCLDWAIT;
    }
#endif
    if (OptionFlags & hoNoChildStop) {
	_SignalAction.sa_flags |= SA_NOCLDSTOP;
    }

    DoAction(SignalId, _SignalAction);
}

void TMushSignalHandler::StartIgnoring(int SignalId) 
{
    _SignalAction.sa_handler = SIG_IGN;
    _SignalAction.sa_mask = _SignalSet;
    _SignalAction.sa_flags = 0;

    DoAction(SignalId, _SignalAction);
}

void TMushSignalHandler::StartDefaultDisposition(int SignalId) 
{
    _SignalAction.sa_handler = SIG_DFL;
    _SignalAction.sa_mask = _SignalSet;
    _SignalAction.sa_flags = 0;

    DoAction(SignalId, _SignalAction);
}

void TMushSignalHandler::StartBlocking(void) 
{
    if (! _IsBlocking) {
	sigprocmask(SIG_BLOCK, &_SignalSet, &_OldSignalSet);
	_IsBlocking = true;
    }
}

void TMushSignalHandler::StopBlocking(void) 
{
    if (_IsBlocking) {
	sigprocmask(SIG_SETMASK, &_OldSignalSet, 0);
	_IsBlocking = false;
    }
}

int TMushSignalHandler::HasPending(int SignalId)
{
    sigset_t PendingSet;
    sigpending(&PendingSet);
    
    return sigismember(&PendingSet, SignalId);
}

void TMushSignalHandler::Wait(void) 
{
    sigsuspend(&_OldSignalSet);
}

void TMushSignalHandler::Reset(void)
{
    StartDefaultDisposition();

    sigemptyset(&_SignalSet);
}

void TMushSignalHandler::DoAction(int SignalId, struct sigaction &SignalAction)
{
    if (SignalId != 0) {
	sigaction(SignalId, &SignalAction, 0);
    }
    else {
	for (int i = 1; i < NSIG; i++) {
	    if (sigismember(&_SignalSet, i)) {
		DoAction(i, SignalAction);
	    }
	}
    }
}

void TMushSignalHandler::Handler(int SignalId) 
{
    if (_ClientList[SignalId] != 0) {
	_ClientList[SignalId]->OnCatchSignal(SignalId);
    }
}

int TMushSignalSender::Raise(int SignalId)
{
    return raise(SignalId);
}

int TMushSignalSender::SendSignal(long TargetProcessId, int SignalId)
{
    return kill(TargetProcessId, SignalId);
}
