// ============================================================================
//  $Id$
//  $Name$
// ============================================================================
#include "TSoftwareAdcModule.hh"
#include "TDataSegment.hh"
#include "TDataElement.hh"

Tlong TSoftwareAdcModule::theSeed = (Tlong)time( 0 );
TRandomEngine TSoftwareAdcModule::theRandomEngine = TRandomEngine( theSeed );

TSoftwareAdcModule::TSoftwareAdcModule( Tint nchannel, Tint scale, Tint id )
  : TSoftwareModule( nchannel, tStatusSuccess, id ), theScale( scale ),
    theRandomFlat( TSoftwareAdcModule::theRandomEngine ), 
    theRandomGaussian( TSoftwareAdcModule::theRandomEngine )
{
  if ( scale < 0 )
    theScale = tDefaultScale;
  theChannel = new TChannel( theNumberOfChannels );
  for ( Tint i = 0; i < theNumberOfChannels; i ++ )
    theChannel -> Add( new Tint( 0 ) );

  setParameters();
}

TSoftwareAdcModule::~TSoftwareAdcModule()
{
  delete [] theMean;
  delete [] theSigma;
  delete theChannel;
}

Tint TSoftwareAdcModule::Clear()
{
  theChannel -> Clear();
  return( theStatus = tStatusSuccess );
}

Tint TSoftwareAdcModule::Update()
{
  theStatus = tStatusSuccess;
  Clear();
  for ( Tint i = 0; i < theNumberOfChannels; i ++ ) {
    Tdouble ddata = theRandomGaussian.fire( (Tdouble)theMean[ i ], (Tdouble)theSigma[ i ] );
    Tint idata = (Tint)ddata;
    if ( idata > theScale ) {
      idata = tDataOverFlow;
      theStatus = -EFAULT;
    }
    theChannel -> Add( new Tint( idata ) );
  }
  return( theStatus );
}

Tint TSoftwareAdcModule::Initialize()
{
  Clear();
  delete [] theMean;
  delete [] theSigma;
  for ( Tint i = 0; i < theNumberOfChannels; i ++ )
    theChannel -> Add( new Tint( 0 ) );
  setParameters();
  return( theStatus = tStatusSuccess );
}

Tvoid TSoftwareAdcModule::FillData( TDataSegment* segment )
{
  for ( Tint i = 0; i < theNumberOfChannels; i ++ )
    segment -> Add( new TDataElement( theChannel -> Get( i ), tTypeInt, i ) );
  return;
}

Tvoid TSoftwareAdcModule::FillData( TDataElement* element )
{
  Tint ch = element -> GetID();
  if ( ch < 0 || ( ch >= theChannel -> GetNumberOfEntries() ) ) {
    Tint tmp = -EFAULT;
    element -> FillData( &tmp, tTypeInt );
  } else {
    element -> FillData( theChannel -> Get( ch ), tTypeInt );
  }
  return;
}

Tvoid TSoftwareAdcModule::setParameters()
{
  theMean = new Tint[ theNumberOfChannels ];
  theSigma = new Tint[ theNumberOfChannels ];

  for ( Tint i = 0; i < theNumberOfChannels; i ++ ) {
    theMean[ i ] = theRandomFlat.fireInt( theScale );

    Tdouble distance = (Tdouble)theMean[ i ];
    if ( (Tdouble)theScale / 2.0 < distance )
      distance = theScale - distance;
    // 3sigma = 0.997
    theSigma[ i ] = theRandomFlat.fireInt( (Tint)( distance / 3.0 ) );
  }
  return;
}

Tvoid TSoftwareAdcModule::Print( Tostream& tos ) const
{
  Tstring head = Twspace + Twspace + Twspace + "* Software ADC, ";
  tos << head << "Status: " << theStatus << Twspace;
  tos << "Scale: " << theScale << Twspace;
  tos << "ID: " << theID << Tendl;

  for ( Tint i = 0; i < theNumberOfChannels; i ++ ) {
    tos << Twspace << head << "Channel: " << i;
    tos << Twspace << "Mean: " << theMean[ i ];
    tos << Twspace << "Sigma: " << theSigma[ i ];
    tos << Twspace << "Data: " << *( theChannel -> Get( i ) ) << Tendl;
  }

  return;
}
