// ============================================================================
//  $Id$
//  $Name$
// ============================================================================
#ifndef __TVECTOR_HH
#define __TVECTOR_HH

#include "TTypes.hh"

template <class T> class Tvector
{

  protected:
    enum { tDefaultCapacity = 1, tDefaultReallocationParameter = 2 };

  protected:
    T** theObjects;

  protected:
    Tsize_t theCapacity;

    Tint theNumberOfEntries;
    Tint thePosition;
    Tint theBeginOfPosition;
    Tint theEndOfPosition;
    Tint theReallocationParameter;

  public:
    Tvector( Tsize_t capacity = tDefaultCapacity, 
		    Tint reallocation = tDefaultReallocationParameter );
    virtual ~Tvector();

  public:
    Tint Add( T* object );
    Tint Remove( Tint position );
    Tvoid Swap();

  public:
    Tvoid Clear();
    Tbool Resize( Tsize_t capacity );

  public:
    T* Next();
    T* Begin() const;
    T* End() const;
    T* Get( Tint position );
    T* Get();
    T* At( Tint position );

  public:
    Tsize_t GetCapacity() const;
    Tint GetNumberOfEntries() const;
    Tint GetPosition() const;
    Tint GetBeginOfPosition() const;
    Tint GetEndOfPosition() const;
    Tint GetReallocationParameter() const;

  public:
    Tvoid SetPosition( Tint position );
    Tvoid SetReallocationParameter( Tint reallocation );

  public:
    T* operator []( Tint position );

  private:
    Tvoid free();
    Tvoid allocate( Tsize_t capacity );

};

template <class T> Tvector<T>::Tvector( Tsize_t capacity, Tint reallocation )
{
  if ( capacity < 1 )
    theCapacity = 1;
  else
    theCapacity = capacity;
  theReallocationParameter = reallocation;
  theNumberOfEntries = 0;
  theObjects = 0;
  thePosition = 0;
  theBeginOfPosition = 0;
  theEndOfPosition = 0;

  if ( theCapacity > 0 )
    allocate( theCapacity );
}

template <class T> Tvector<T>::~Tvector()
{
  free();
}

template <class T> Tint Tvector<T>::Add( T* object )
{
  if ( theNumberOfEntries >= (Tint)theCapacity )
    Resize( theCapacity *= theReallocationParameter );

  theObjects[ theNumberOfEntries ] = object;

  theNumberOfEntries ++;
  thePosition = theNumberOfEntries - 1;
  theEndOfPosition = theNumberOfEntries - 1;

  return( theNumberOfEntries );
}

template <class T> Tint Tvector<T>::Remove( Tint position )
{
  if ( position < 0 || position >= theNumberOfEntries )
    return( 0 );
  thePosition = position;
  delete theObjects[ thePosition ];
  for ( Tint i = thePosition; i < theNumberOfEntries; i ++ )
    theObjects[ i ] = theObjects[ i + 1 ];
  theNumberOfEntries --;
  theEndOfPosition = theNumberOfEntries - 1;

  return( theNumberOfEntries );
}

template <class T> Tvoid Tvector<T>::Swap()
{
  if ( theNumberOfEntries < 2 )
    return;

  Tint times = theNumberOfEntries / 2;
  for ( Tint i = 0; i < times; i ++ ) {
    T* tmp = theObjects[ i ];
    theObjects[ i ] = theObjects[ theEndOfPosition - i ];
    theObjects[ theEndOfPosition - i ] = tmp;
  }

  return;
}

template <class T> Tvoid Tvector<T>::Clear()
{
  for ( Tint i = 0; i < theNumberOfEntries; i ++ )
    if ( theObjects[ i ] )
      delete theObjects[ i ];
  theNumberOfEntries = 0;
  thePosition = 0;
  theEndOfPosition = 0;
  return;
}

template <class T> Tbool Tvector<T>::Resize( Tsize_t capacity )
{
  if ( theNumberOfEntries >= (Tint)capacity )
    return( Tfalse );

  theCapacity = capacity;
  T** newobjects = new T* [ theCapacity ];
  for ( Tint i = 0; i < theNumberOfEntries; i ++ )
    newobjects[ i ] = theObjects[ i ];
  delete [] theObjects;
  theObjects = newobjects;

  thePosition = theNumberOfEntries - 1;
  
  return( Ttrue );
}

template <class T> T* Tvector<T>::Next()
{
  if ( thePosition < 0 || thePosition >= theNumberOfEntries )
    return( 0 );
  return( theObjects[ thePosition ++ ] );
}

template <class T> T* Tvector<T>::Begin() const
{
  return( theObjects[ theBeginOfPosition ] );
}

template <class T> T* Tvector<T>::End() const
{
  return( theObjects[ theEndOfPosition ] );
}

template <class T> T* Tvector<T>::Get( Tint position )
{
  if ( position < 0 || position >= theNumberOfEntries )
    return( 0 );
  thePosition = position;
  return( theObjects[ thePosition ] );
}

template <class T> T* Tvector<T>::Get()
{
  return( Get( thePosition ) );
}

template <class T> T* Tvector<T>::At( Tint position )
{
  return( Get( position ) );
}

template <class T> Tsize_t Tvector<T>::GetCapacity() const
{
  return( theCapacity );
}

template <class T> Tint Tvector<T>::GetNumberOfEntries() const
{
  return( theNumberOfEntries );
}

template <class T> Tint Tvector<T>::GetPosition() const
{
  return( thePosition );
}

template <class T> Tint Tvector<T>::GetBeginOfPosition() const
{
  return( theBeginOfPosition );
}

template <class T> Tint Tvector<T>::GetEndOfPosition() const
{
  return( theEndOfPosition );
}

template <class T> Tint Tvector<T>::GetReallocationParameter() const
{
  return( theReallocationParameter );
}

template <class T> Tvoid Tvector<T>::SetPosition( Tint position )
{
  thePosition = position;
  return;
}

template <class T> Tvoid Tvector<T>::SetReallocationParameter( Tint reallocation )
{
  theReallocationParameter = reallocation;
  return;
}

template <class T> T* Tvector<T>::operator []( Tint position )
{
  return( Get( position ) );
}

template <class T> Tvoid Tvector<T>::free()
{
  for ( Tint i = 0; i < theNumberOfEntries; i ++ )
    if ( theObjects[ i ] )
      delete theObjects[ i ];
  delete [] theObjects;

  theCapacity = 0;
  theNumberOfEntries = 0;
  thePosition = 0;
  theBeginOfPosition = 0;
  theEndOfPosition = 0;
}

template <class T> Tvoid Tvector<T>::allocate( Tsize_t capacity )
{
  theCapacity = capacity;
  theObjects = new T* [ theCapacity ];
}

#endif
