// ============================================================================
//  $Id$
//  $Name$
// ============================================================================
#include "TDataElement.hh"
#include "TOutputObjectStream.hh"
#include "TOutputObjectFile.hh"
#include "TOutputObjectSocket.hh"
//#include "TOutputObjectSharedMemory.hh"

TDataElement::TDataElement( Telement_t type, Tint id )
  : TStreamableObject( tDataElement, id ),
    theData( 0 ), theElementType( type ), theNumberOfPrimitives( 0 )
{;}

TDataElement::TDataElement( Tvoid* data, Telement_t type, Tint id, Tint ndata )
  : TStreamableObject( tDataElement, id ), 
    theData( 0 ), theElementType( type ), theNumberOfPrimitives( ndata )
{
  allocateDataSpace( data );
}

TDataElement::TDataElement( const TDataElement& right )
  : TStreamableObject( right ),
    theData( 0 ), theElementType( right.theElementType ),
    theNumberOfPrimitives( right.theNumberOfPrimitives )
{
  allocateDataSpace( right.theData );
}

TDataElement::~TDataElement()
{
  freeDataSpace();
}

Tint TDataElement::WriteData( TOutputObjectStream* output )
{
  Tint datasize = 0;
  Tstream_t streamtype = output -> GetStreamType();

  switch ( streamtype ) {
    case tFileStream:
      datasize = writeToFile( (TOutputObjectFile*)output );
      break;
    case tSocketStream:
      datasize = writeToSocket( (TOutputObjectSocket*)output );
      break;
    case tSharedMemoryStream:
      datasize = writeToSharedMemory( (TOutputObjectSharedMemory*)output );
      break;
    case tUnknownStream:
    default:
      return( 0 );
      break;
  }


  return( datasize );
}

Tint TDataElement::GetDataSize()
{
  Tsize_t total = Tsizeof( theID ) + Tsizeof( theElementType ) + Tsizeof( theNumberOfPrimitives );

  switch ( theElementType ) {

    case tTypeWord:
    case tTypeUnsignedShort:
    case tTypeShort:
      total += ( Tsizeof( TUshort ) ) * theNumberOfPrimitives;
      break;

    case tTypeString:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	total += ( Tsizeof( Tint ) );
	Tsize_t ss = ( ( (Tstring*)theData )[ i ] ).size() + 1;
	total += ( Tsizeof( Tchar ) ) * ss;
      }
      break;

    case tTypeDouble:
      total += ( Tsizeof( Tdouble ) ) * theNumberOfPrimitives;
      break;

    case tTypeInt:
      total += ( Tsizeof( Tint ) ) * theNumberOfPrimitives;
      break;

    case tTypeUnknown:
    case tTypeObject:
    case tTypeFloat:
    default:
      break;
  }

  return( (Tint)total );
}

const TDataElement& TDataElement::operator=( const TDataElement& right )
{
  *( (TStreamableObject*)this ) = *( (TStreamableObject*)(&right) );

  freeDataSpace();

  theElementType = right.theElementType;
  theNumberOfPrimitives = right.theNumberOfPrimitives;

  allocateDataSpace( right.theData );

  return( *this );
}

Tbool TDataElement::operator==( const TDataElement& right ) const
{
  Tbool ret = Ttrue;
  ret &= ( *((TStreamableObject*)this) == *( (TStreamableObject*)(&right) ) );
  ret &= ( theElementType == right.theElementType );
  ret &= ( theNumberOfPrimitives == right.theNumberOfPrimitives );

  if ( ret != Tfalse ) {
    switch ( theElementType ) {
      case tTypeString:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tstring*)theData)[ i ] == ((Tstring*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeDouble:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tdouble*)theData)[ i ] == ((Tdouble*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeInt:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tint*)theData)[ i ] == ((Tint*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeUnsignedShort:
      case tTypeShort:
      case tTypeWord:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((TUshort*)theData)[ i ] == ((TUshort*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeUnknown:
      case tTypeObject:
      case tTypeFloat:
      default:
	ret &= Tfalse;
	break;
    }
  }
  return( ret );
}

inline Tbool TDataElement::operator!=( const TDataElement& right ) const
{
  Tbool ret = Ttrue;
  ret &= ( *((TStreamableObject*)this) != *((TStreamableObject*)(&right)) );
  ret &= ( theElementType != right.theElementType );
  ret &= ( theNumberOfPrimitives != right.theNumberOfPrimitives );


  if ( ret != Tfalse ) {
    switch ( theElementType ) {
      case tTypeString:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tstring*)theData)[ i ] != ((Tstring*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeDouble:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tdouble*)theData)[ i ] != ((Tdouble*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeInt:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tint*)theData)[ i ] != ((Tint*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeUnsignedShort:
      case tTypeShort:
      case tTypeWord:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((TUshort*)theData)[ i ] != ((TUshort*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeUnknown:
      case tTypeObject:
      case tTypeFloat:
      default:
	ret &= Tfalse;
	break;
    }
  }
  return( ret );
}

Tint TDataElement::writeToFile( TOutputObjectFile* ofile )
{
  Tsize_t datasize = 0;
  static const Tsize_t nmemb = 1;
  Tsize_t ss = 0;
  Tstring strbuf;


  // write ID
  ss = fwrite( &theID, Tsizeof( theID ), nmemb, ofile -> GetFileStream() );
  if ( ss != nmemb )
    perror( "TDataElement::writeToFile" );
  else
    datasize += Tsizeof( theID );


  // write Element Type
  ss = fwrite( &theElementType, Tsizeof( theElementType ), nmemb, ofile -> GetFileStream() );
  if ( ss != nmemb )
    perror( "TDataElement::writeToFile" );
  else
    datasize += Tsizeof( theID );


  // write Number of Primitives
  ss = fwrite( &theNumberOfPrimitives, Tsizeof( theNumberOfPrimitives ), nmemb, ofile -> GetFileStream() );
  if ( ss != nmemb )
    perror( "TDataElement::writeToFile" );
  else
    datasize += Tsizeof( theNumberOfPrimitives );


  switch ( theElementType ) {

    case tTypeString:
      Tchar cc;
      Tint nchar;

      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	strbuf = ( (Tstring*)theData )[ i ];

	nchar = (Tint)( strbuf.size() + 1 );
	ss = fwrite( &nchar, Tsizeof( Tint ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( "TDataElement::writeToFile" );
	else
	  datasize += ss * Tsizeof( Tint );


	for ( Tint j = 0; j < nchar; j ++ ) {
	  if ( j == ( nchar - 1 ) )
	    cc = '\0';
	  else
	    cc = strbuf[ j ];
	  ss = fwrite( &cc, Tsizeof( Tchar ), nmemb, ofile -> GetFileStream() );
	  if ( ss != nmemb )
	    perror( "TDataElement::writeToFile" );
	  else
	    datasize += ss * Tsizeof( Tchar );
	}
      }
      break;

    case tTypeDouble:
      Tdouble dd;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	dd = ( (Tdouble*)theData )[ i ];
	ss = fwrite( &dd, Tsizeof( Tdouble ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( "TDataElement::writeToFile" );
	else
	  datasize += ss * Tsizeof( Tdouble );
      }
      break;

    case tTypeInt:
      Tint ii;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ii = ( (Tint*)theData )[ i ];
	ss = fwrite( &ii, Tsizeof( Tint ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( "TDataElement::writeToFile" );
	else
	  datasize += ss * Tsizeof( Tint );
      }
      break;

    case tTypeUnsignedShort:
    case tTypeShort:
    case tTypeWord:
      Tint ww;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ww = ( (TUshort*)theData )[ i ];
	ss = fwrite( &ww, Tsizeof( TUshort ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( "TDataElement::writeToFile" );
	else
	  datasize += ss * Tsizeof( TUshort );
      }
      break;


    case tTypeUnknown:
    case tTypeObject:
    case tTypeFloat:
    default:
      break;
  }

  return( (Tint)datasize );
}

Tint TDataElement::writeToSocket( TOutputObjectSocket* osocket )
{
  Tsize_t datasize = 0;
  Tsize_t ssize;
  Tstring strbuf;

  
  // write ID
  ssize = Tsizeof( theID );
  if ( send( osocket->GetServerDescriptor(), &theID, ssize, 0 ) != (Tint)ssize )
    perror( "TDataElement::writeToSocket" );
  else
    datasize += ssize;


  // write Element Type
  ssize = Tsizeof( theElementType );
  if ( send( osocket->GetServerDescriptor(), &theElementType, ssize, 0 ) != (Tint)ssize )
    perror( "TDataElement::writeToSocket" );
  else
    datasize += ssize;


  // write Number of Primitives
  ssize = Tsizeof( theNumberOfPrimitives );
  if ( send( osocket->GetServerDescriptor(), &theNumberOfPrimitives, ssize, 0 ) != (Tint)ssize )
    perror( "TDataElement::writeToSocket" );
  else
    datasize += ssize;


  switch ( theElementType ) {

    case tTypeString:
      Tchar cc;
      Tint nchar;

      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	strbuf = ( (Tstring*)theData )[ i ];
	nchar = (Tint)( strbuf.size() + 1 );
	ssize = Tsizeof( Tint );
	if ( send( osocket -> GetServerDescriptor(), &nchar, ssize, 0 ) != (Tint)ssize )
	  perror( "TDataElement::writeToSocket" );
	else
	  datasize += ssize;
	
	ssize = Tsizeof( Tchar );
	for ( Tint j = 0; j < nchar; j ++ ) {
	  if ( j == ( nchar - 1 ) )
	    cc = '\0';
	  else
	    cc = strbuf[ j ];
	  if ( send( osocket -> GetServerDescriptor(), &cc, ssize, 0 ) != (Tint)ssize )
	    perror( "TDataElement::writeToSocket" );
	  else
	    datasize += ssize;
	}
      }
      break;

    case tTypeDouble:
      Tdouble dd;
      ssize = Tsizeof( Tdouble );
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	dd = ( (Tdouble*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &dd, ssize, 0 ) != (Tint)ssize )
	  perror( "TDataElement::writeToSocket" );
	else
	  datasize += ssize;
      }
      break;

    case tTypeInt:
      Tint ii;
      ssize = Tsizeof( Tint );
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ii = ( (Tint*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ii, ssize, 0 ) != (Tint)ssize )
	  perror( "TDataElement::writeToSocket" );
	else
	  datasize += ssize;
      }
      break;

    case tTypeUnsignedShort:
    case tTypeShort:
    case tTypeWord:
      Tint ww;
      ssize = Tsizeof( TUshort );
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ww = ( (TUshort*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ww, ssize, 0 ) != (Tint)ssize )
	  perror( "TDataElement::writeToSocket" );
	else
	  datasize += ssize;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    case tTypeFloat:
    default:
      break;
  }
  
  return( (Tint)datasize );
}

Tint TDataElement::writeToSharedMemory( TOutputObjectSharedMemory* omemory )
{
  Tsize_t datasize = 0;
  // now, implement ...
  return( (Tint)datasize );
}

Tostream& operator<<( Tostream& tos, const TDataElement& right )
{
  Tstring head = Twspace + Twspace + Twspace + "* Data Element, ";

  tos << head << "ID: " << right.theID;
  tos << Twspace << "Primitives: " << right.theNumberOfPrimitives;
  tos << Twspace << "Type: " << right.theElementType;

  switch ( right.theElementType ) {

    case tTypeWord:
    case tTypeShort:
    case tTypeUnsignedShort:
      tos << Tslash << "Integer(16)" << Twspace << "Data: ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (TUshort*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << "," << Tspace;
      }
      break;

    case tTypeInt:
      tos << Tslash << "Integer(32)" << Twspace << "Data: ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tint*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << "," << Tspace;
      }
      break;

    case tTypeString:
      tos << Tslash << "String" << Twspace << "Data: ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tstring*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << "," << Tspace;
      }
      break;

    case tTypeDouble:
      tos << Tslash << "Double(64)" << Twspace << "Data: ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tdouble*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << "," << Tspace;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    case tTypeFloat:
    default:
      tos << Tslash << "Unknown" << Twspace << "Data: ";
      tos << "unknown(or undefined) data ...";
      break;
  }

  return( tos );
}

Tvoid TDataElement::freeDataSpace()
{
  if ( theData != 0 ) {
    switch ( theElementType ) {

      case tTypeWord:
      case tTypeUnsignedShort:
      case tTypeShort:
	delete [] (TUshort*)theData;
	break;

      case tTypeInt:
	delete [] (Tint*)theData;
	break;

      case tTypeString:
	delete [] (Tstring*)theData;
	break;

      case tTypeDouble:
	delete [] (Tdouble*)theData;
	break;

      case tTypeUnknown:
      case tTypeObject:
      case tTypeFloat:
      default:
	break;
    }
  }
  theData = 0;
  theNumberOfPrimitives = 0;
  return;
}

Tvoid TDataElement::allocateDataSpace( Tvoid* data )
{
  if ( data == 0 )
    return;

  switch ( theElementType ) {

    case tTypeWord:
    case tTypeUnsignedShort:
    case tTypeShort:
      theData = new TUshort[ theNumberOfPrimitives ];
      for ( TUshort i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (TUshort*)(theData) )[ i ] = *( (TUshort*)data );
	( (TUshort*)data ) ++;
      }
      break;


    case tTypeInt:
      theData = new Tint[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (Tint*)(theData) )[ i ] = *( (Tint*)data );
	( (Tint*)data ) ++;
      }
      break;

    case tTypeString:
      theData = new Tstring[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (Tstring*)(theData) )[ i ] = *( (Tstring*)data );
	( (Tstring*)data ) ++;
      }
      break;

    case tTypeDouble:
      theData = new Tdouble[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (Tdouble*)(theData) )[ i ] = *( (Tdouble*)data );
	( (Tdouble*)data ) ++;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    case tTypeFloat:
    default:
      break;
  }
  return;
}
