// ============================================================================
//  $Id$
//  $Name$
// ============================================================================
#include "TInputObjectSocket.hh"
#include "TStreamableObject.hh"
#include "TDataRecord.hh"
#include "TDataSection.hh"
#include "TDataSegment.hh"
#include "TDataElement.hh"
#include "TObjectFilter.hh"

TInputObjectSocket::TInputObjectSocket( Tint port )
  : TObjectSocket( port ), TInputObjectStream( tSocketStream ),
    theClientDescriptor( -1 )
{
  initialize();
  OpenServer();
  OpenClient();
}

TInputObjectSocket::TInputObjectSocket( TObjectFilter* filter, Tint port )
  : TObjectSocket( port ), TInputObjectStream( filter, tSocketStream ),
    theClientDescriptor( -1 )
{
  initialize();
  OpenServer();
  OpenClient();
}

TInputObjectSocket::~TInputObjectSocket()
{
  CloseClient();
  CloseServer();
}

Tint TInputObjectSocket::ReadData( TStreamableObject& object )
{
  theLastDataSize = theDataSize;
  theDataSize = 0;

  if ( HasFilter() )
    return( theObjectFilter -> Analyze( object, this ) );

  static Tobject_t objecttype = object.GetObjectType();

  switch ( objecttype ) {
    case tDataRecord:
      *( (TDataRecord*)(&object) ) = getDataRecord();
      break;
    case tDataSection:
      *( (TDataSection*)(&object) ) = getDataSection();
      break;
    case tDataSegment:
      *( (TDataSegment*)(&object) ) = getDataSegment();
      break;
    case tDataElement:
      *( (TDataElement*)(&object) ) = getDataElement();
      break;
    case tDataUnknown:
    default:
      break;
  }

  return( theDataSize );
}

TDataRecord TInputObjectSocket::getDataRecord()
{
  Tint id;
  Tint entries;
  readProperties( id, entries );

  TDataRecord record( id );
  if ( theDataSize == 0 )
    return( record );

  for ( Tint i = 0; i < entries; i ++ )
    record.push_back( getDataSection() );

  return( record );
}

TDataSection TInputObjectSocket::getDataSection()
{
  Tint id;
  Tint entries;
  readProperties( id, entries );

  TDataSection section( id );
  if ( theDataSize == 0 )
    return( section );

  for ( Tint i = 0; i < entries; i ++ )
    section.push_back( getDataSegment() );

  return( section );
}

TDataSegment TInputObjectSocket::getDataSegment()
{
  Tint id;
  Tint entries;
  readProperties( id, entries );

  TDataSegment segment( id );
  if ( theDataSize == 0 )
    return( segment );

  for ( Tint i = 0; i < entries; i ++ )
    segment.push_back( getDataElement() );

  return( segment );
}

TDataElement TInputObjectSocket::getDataElement()
{
  Tint id;
  Telement_t type;
  Tint nprvs;
  TDataElement element;


  // read id
  if ( recv( theClientDescriptor, &id, Tsizeof( id ), 0 ) != (Tint)Tsizeof( id ) ) {
    theDataSize = 0;
    return( element );
  } else {
    theDataSize += Tsizeof( id );
    element.SetID( id );
  }


  // read element type
  if ( recv( theClientDescriptor, &type, Tsizeof( type ), 0 ) != (Tint)Tsizeof( type ) ) {
    theDataSize = 0;
    return( element );
  } else {
    theDataSize += Tsizeof( type );
    element.SetElementType( type );
  }


  // read number of primitives
  if ( recv( theClientDescriptor, &nprvs, Tsizeof( nprvs ), 0 ) != (Tint)Tsizeof( nprvs ) ) {
    theDataSize = 0;
    return( element );
  } else {
    theDataSize += Tsizeof( nprvs );
    element.SetNumberOfPrimitives( nprvs );
  }


  Tstring* strbuf = 0;
  Tchar* cc = 0;
  Tint* ii = 0;
  Tdouble* dd = 0;
  TUshort* ww = 0;
  Tsize_t rsize;


  switch ( type ) {

    case tTypeString:
      strbuf = new Tstring[ nprvs ];
      for ( Tint i = 0; i < nprvs; i ++ ) {
	Tint nchar;
	rsize = Tsizeof( Tint );
	if ( recv( theClientDescriptor, &nchar, rsize, 0 ) != (Tint)rsize ) {
	  theDataSize = 0;
	  return( element );
	} else {
	  theDataSize += (Tint)rsize;
	  cc = new Tchar[ nchar ];
	}
	rsize = Tsizeof( Tchar );
	for ( Tint j = 0; j < nchar; j ++ ) {
	  if ( recv( theClientDescriptor, &cc[ j ], rsize, 0 ) != (Tint)rsize ) {
	    theDataSize = 0;
	    return( element );
	  } else {
	    theDataSize += (Tint)rsize;
	  }
	}
	strbuf[ i ] = cc;
	delete [] cc;
      }
      element.FillData( strbuf, nprvs );
      delete [] strbuf;
      break;

    case tTypeInt:
      ii = new Tint[ nprvs ];
      rsize = Tsizeof( Tint );
      for ( Tint i = 0; i < nprvs; i ++ ) {
	if ( recv( theClientDescriptor, &ii[ i ], rsize, 0 ) != (Tint)rsize ) {
	  theDataSize = 0;
	  return( element );
	} else {
	  theDataSize += (Tint)rsize;
	}
      }
      element.FillData( ii, nprvs );
      delete [] ii;
      break;

    case tTypeDouble:
      dd = new Tdouble[ nprvs ];
      rsize = Tsizeof( Tdouble );
      for ( Tint i = 0; i < nprvs; i ++ ) {
	if ( recv( theClientDescriptor, &dd[ i ], rsize, 0 ) != (Tint)rsize ) {
	  theDataSize = 0;
	  return( element );
	} else {
	  theDataSize += (Tint)rsize;
	}
      }
      element.FillData( dd, nprvs );
      delete [] dd;
      break;

    case tTypeWord:
    case tTypeUnsignedShort:
    case tTypeShort:
      ww = new TUshort[ nprvs ];
      rsize = Tsizeof( TUshort );
      for ( Tint i = 0; i < nprvs; i ++ ) {
	if ( recv( theClientDescriptor, &ww[ i ], rsize, 0 ) != (Tint)rsize ) {
	  theDataSize = 0;
	  return( element );
	} else {
	  theDataSize += (Tint)rsize;
	}
      }
      element.FillData( ww, nprvs );
      delete [] ww;
      break;


    case tTypeUnknown:
    case tTypeObject:
    case tTypeFloat:
    default:
      break;
  }


  return( element );
}

Tint TInputObjectSocket::OpenClient()
{
  Tsocklen_t server_len = (Tsocklen_t)Tsizeof( theAddress );
  if ( bind( theServerDescriptor, (struct sockaddr *)&theAddress, server_len ) == -1 ) {
    perror( "TInputObjectSocket::OpenClient" );
    exit( -errno );
  }

  if ( listen( theServerDescriptor, tDefaultBackLog ) == -1 ) {
    perror( "TInputObjectSocket::OpenClient" );
    exit( -errno );
  }

  struct sockaddr_in client_address;
  Tsocklen_t client_len;
  theClientDescriptor = accept( theServerDescriptor, (struct sockaddr *)&client_address, &client_len );
  if ( theClientDescriptor == -1 )
    perror( "TInputObjectSocket::OpenClient" );
  return( theClientDescriptor );
}

Tint TInputObjectSocket::CloseClient()
{
  if ( shutdown( theClientDescriptor, 2 ) == -1 ) {
    perror( "TInputObjectSocket::CloseClient" );
    return( -1 );
  }

  if ( close( theClientDescriptor ) == -1 ) {
    perror( "TInputObjectSocket::CloseClient" );
    return( -1 );
  }
  theClientDescriptor = -1;
  return( 0 );
}

Tvoid TInputObjectSocket::readProperties( Tint& id, Tint& entries )
{
  static const Tsize_t idsize = Tsizeof( Tint );
  static const Tsize_t entrysize = Tsizeof( Tint );

  if ( recv( theClientDescriptor, &id, idsize, 0 ) != (Tint)idsize ) {
    theDataSize = 0;
    return;
  } else {
    theDataSize += idsize;
  }

  if ( recv( theClientDescriptor, &entries, entrysize, 0 ) != (Tint)entrysize ) {
    theDataSize = 0;
    return;
  } else {
    theDataSize += idsize;
  }

  return;
}

Tvoid TInputObjectSocket::initialize()
{
  theAddress.sin_family = PF_INET;
  theAddress.sin_port = htons( (TUshort)thePortNumber );
  theAddress.sin_addr.s_addr = htonl( INADDR_ANY );
  //signal( SIGCHLD, SIG_IGN );
  //signal( SIGPIPE, SIG_IGN );
  return;
}
