// ============================================================================
//  $Id: TVmeMpx201aModule.hh,v 1.1.1.1 2002/12/04 23:47:39 iwai Exp $
//  $Name:  $
// ============================================================================
#ifndef __TVMEMPX201AMODULE_HH
#define __TVMEMPX201AMODULE_HH

#include "TGlobals.hh"
#include "TVmeModule.hh"

class TDataSegment;
class TDataElement;
class TVmeDaughterBoardMemoryModule;

class TVmeMpx201aModule
  : public TVmeModule
{

  private:
    TVmeDaughterBoardMemoryModule* theMemoryModule;

  private:
    enum
    {
      ID = 0x00,
      DEVICE_TYPE = 0x02,
      CONTROL_STATUS = 0x04,
      VME_OFFSET_ADDRESS = 0x06,
      PAGE_SET = 0x08,
      RESERVED = 0x0a,
      TEST_MODE = 0x0c,
      TEST_DATA = 0x0e,
      INTERRUPT = 0x10,
      BLOCK_COUNT = 0x12,
      DAQ_POINTER = 0x14,
      SAMPLE_RATE = 0x16,
      FRONT_PORCH = 0x18,
      FRAME_LENGTH = 0x1a,
      ROW_LENGTH = 0x1c
    };

  public:
    enum Tsync_t
    {
      tScannerMode,
      tInternalSynchronisationMode,
      tExternalSynchronisationMode,
      tModeUnknown = -1
    };

    enum TdataCapture_t
    {
      tTransientMode,
      tContinuousMode
    };

    enum { tBlock = 2048 };

  public:
    TVmeMpx201aModule( Tint baseaddress, Tint nblock, Tint mapsize = 0x1000 );
    TVmeMpx201aModule( const TVmeMpx201aModule& right );
    ~TVmeMpx201aModule();

  public:
    const TVmeMpx201aModule& operator=( const TVmeMpx201aModule& right );
    Tbool operator==( const TVmeMpx201aModule& right ) const;
    Tbool operator!=( const TVmeMpx201aModule& right ) const;

  public:
    Tint Clear();
    Tint Update();
    Tint Initialize();
    Tvoid FillData( const TDataSegment& segment );
    Tvoid FillData( const TDataElement& element );

  public:
    const TVmeDaughterBoardMemoryModule* GetMemoryModule() const;


  public:
    TUshort GetProductID() const;
    TUshort GetDeviceType() const;
    Tbool IsMemoryBlockFull() const;
    Tbool IsDataOverFlow() const;
    Tsync_t GetEventSynchronisationMode() const;
    TUlong GetOffsetAddressForDaughterBoard() const;
    Tint GetPage() const;
    Tint GetNumberOfBlock() const;


    Tvoid EnableAccessToDaughterBoard();
    Tvoid DisableAccessToDaughterBoard();
    Tvoid SetBufferOrientation( Tbit bit );
    Tvoid SetScanCompleteFlag( Tbit bit );
    Tvoid SetMemoryBlockFullFlag( Tbit bit );
    Tvoid SetDataOverFlowFlag( Tbit bit );
    Tvoid SetEventSynchronisationMode( Tsync_t mode );
    Tvoid SetDataCaptureMode( TdataCapture_t mode );
    Tvoid EnableAcquisition();
    Tvoid DisableAcquisition();
    Tvoid EnableSystemFailure();
    Tvoid DisableSystemFailure();
    Tvoid Reset();
    Tvoid SetOffsetAddressForDaughterBoard( TUshort offset );
    Tvoid SetPage( Tint id );
    Tvoid SetSamplingRate( Tint nsample );
    Tvoid SetNumberOfBlock( Tint nblock );

    //Interrupt Configuration Register

};

inline Tint TVmeMpx201aModule::GetNumberOfBlock() const
{
  // 1block = 2048sample
  TUshort retval = *( (TUshort*)( theModuleBaseAddress + BLOCK_COUNT ) );
  retval &= 0x7fff;
  return( (Tint)retval );
}

inline const TVmeDaughterBoardMemoryModule* TVmeMpx201aModule::GetMemoryModule() const
{
  return( theMemoryModule );
}

inline Tvoid TVmeMpx201aModule::SetNumberOfBlock( Tint nblock )
{
  if ( nblock < 0 || nblock > 0x7fff ) {
    Tcerr << "TVmeMpx201aModule::SetNumberOfBlock: invalid number" << Tendl;
    return;
  }
  *( (TUshort*)( theModuleBaseAddress + BLOCK_COUNT ) ) = nblock;
  return;
}

inline Tvoid TVmeMpx201aModule::SetSamplingRate( Tint nsample )
{
  if ( nsample < 0 || nsample > 0x00ff ) {
    Tcerr << "TVmeMpx201aModule::SetSamplingRate: invalid number" << Tendl;
    return;
  }
  *( (TUshort*)( theModuleBaseAddress + SAMPLE_RATE ) ) = nsample;
  return;
}

inline TUshort TVmeMpx201aModule::GetProductID() const
{
  return( *( (TUshort*)( theModuleBaseAddress + ID ) ) );
}

inline TUshort TVmeMpx201aModule::GetDeviceType() const
{
  return( *( (TUshort*)( theModuleBaseAddress + DEVICE_TYPE ) ) );
}

inline Tbool TVmeMpx201aModule::IsMemoryBlockFull() const
{
  return( getBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 9 ) );
}

inline Tbool TVmeMpx201aModule::IsDataOverFlow() const
{
  return( getBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 8 ) );
}

inline TVmeMpx201aModule::Tsync_t TVmeMpx201aModule::GetEventSynchronisationMode() const
{
  Tbit d07 = getBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 7 );
  Tbit d06 = getBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 6 );

  if ( d07 == 0 )
    return( tScannerMode );
  else if ( d07 == 1 && d06 == 1 )
    return( tInternalSynchronisationMode );
  else if ( d07 == 1 && d06 == 0 )
    return( tExternalSynchronisationMode );
  else
    return( tModeUnknown );
}

inline TUlong TVmeMpx201aModule::GetOffsetAddressForDaughterBoard() const
{
  TUshort regval = *( (TUshort*)( theModuleBaseAddress + VME_OFFSET_ADDRESS ) );
  regval &= 0xff80;
  TUlong retval = regval;
  retval = retval << 16;
  return( retval );
}

inline Tint TVmeMpx201aModule::GetPage() const
{
  TUshort regval = *( (TUshort*)( theModuleBaseAddress + PAGE_SET ) );
  regval &= 0x0007;
  return( (Tint)regval );
}

inline Tvoid TVmeMpx201aModule::EnableAccessToDaughterBoard()
{
  setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 15, 1 );
  return;
}

inline Tvoid TVmeMpx201aModule::DisableAccessToDaughterBoard()
{
  setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 15, 0 );
  return;
}

inline Tvoid TVmeMpx201aModule::SetBufferOrientation( Tbit bit )
{
  setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 11, bit );
  return;
}

inline Tvoid TVmeMpx201aModule::SetScanCompleteFlag( Tbit bit )
{
  setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 10, bit );
  return;
}

inline Tvoid TVmeMpx201aModule::SetMemoryBlockFullFlag( Tbit bit )
{
  setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 9, bit );
  return;
}

inline Tvoid TVmeMpx201aModule::SetDataOverFlowFlag( Tbit bit )
{
  setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 8, bit );
  return;
}

inline Tvoid TVmeMpx201aModule::SetEventSynchronisationMode( Tsync_t mode )
{
  switch ( mode ) {
    case tScannerMode:
      setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 7, 0 );
      break;
    case tInternalSynchronisationMode:
      setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 7, 1 );
      setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 6, 1 );
      break;
    case tExternalSynchronisationMode:
      setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 7, 1 );
      setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 6, 0 );
      break;
    default:
      Tcerr << "TVmeMpx201aModule::SetEventSynchronisationMode: invalid mode" << Tendl;
      break;
  }

  return;
}

inline Tvoid TVmeMpx201aModule::SetDataCaptureMode( TdataCapture_t mode )
{
  switch ( mode ) {
    case tTransientMode:
      setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 5, 1 );
      break;
    case tContinuousMode:
      setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 5, 0 );
      break;
    default:
      Tcerr << "TVmeMpx201aModule::SetDataCaptureMode: invalid mode" << Tendl;
      break;
  }
  return;
}

inline Tvoid TVmeMpx201aModule::EnableAcquisition()
{
  setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 4, 0 );
  return;
}

inline Tvoid TVmeMpx201aModule::DisableAcquisition()
{
  setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 4, 1 );
  return;
}

inline Tvoid TVmeMpx201aModule::EnableSystemFailure()
{
  setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 1, 0 );
  return;
}

inline Tvoid TVmeMpx201aModule::DisableSystemFailure()
{
  setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 1, 1 );
}

inline Tvoid TVmeMpx201aModule::Reset()
{
  setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 0, 1 );
}

inline Tvoid TVmeMpx201aModule::SetOffsetAddressForDaughterBoard( TUshort offset )
{
  // A31 to A23
  // 0x00800000 - 0xff800000
  //offset = offset & 0xff8
  //setBit( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ), 0, 1 );
  //ϲѤ
  *( (TUshort*)( theModuleBaseAddress + VME_OFFSET_ADDRESS ) ) = offset;
  return;
}

inline Tvoid TVmeMpx201aModule::SetPage( Tint id )
{
  if ( id < 0 || id > 7 ) {
    Tcerr << "TVmeMpx201aModule::SetPage: invalid ID" << Tendl;
    return;
  }
  *( (TUshort*)( theModuleBaseAddress + CONTROL_STATUS ) ) |= id;
  return;
}

#endif
