// ============================================================================
//  $Id: TRunManager.cc,v 1.4 2002/12/08 13:27:53 iwai Exp $
//  $Name:  $
// ============================================================================
#include "TRunManager.hh"
#include "TEventManager.hh"
#include "TAnalysisManager.hh"
#include "TRunAction.hh"
#include "TUserInterface.hh"
#include "TSystemTimer.hh"

#include "TCommand.hh"

#include "TExitCommand.hh"
#include "TQuitCommand.hh"
#include "TListCommand.hh"
#include "TChangeDirectoryCommand.hh"
#include "TPrintCurrentWorkingDirectoryCommand.hh"
#include "TDateCommand.hh"

#include "TShowNetworkStatusCommand.hh"
#include "TReadFileCommand.hh"
#include "TShellCommand.hh"
#include "TExecuteCommand.hh"

#include "TRunStartCommand.hh"
#include "TRunStopCommand.hh"
#include "TRunSuspendCommand.hh"
#include "TRunResumeCommand.hh"
#include "TRunShutdownCommand.hh"
#include "TShowRunManagerCommand.hh"

#include "TAnalysisStartCommand.hh"
#include "TAnalysisStopCommand.hh"
#include "TAnalysisSuspendCommand.hh"
#include "TAnalysisResumeCommand.hh"
#include "TShowAnalysisManagerCommand.hh"

#include "TSetStackSizeCommand.hh"
#include "TShowEventManagerCommand.hh"

#include "TSetRunInformationModuleCommand.hh"
#include "TAddRunInformationModuleCommand.hh"
#include "TClearRunInformationModuleCommand.hh"
#include "TInitializeRunInformationModuleCommand.hh"
#include "TShowRunInformationModuleCommand.hh"
#include "TSoftwareRunInformationModule.hh"

#include "THelpCommand.hh"
#include "TPopupLogoCommand.hh"
#include "TPopdownLogoCommand.hh"
#include "TPrintAuthorInformationCommand.hh"
#include "TSleepCommand.hh"
#include "splash.h"

TRunManager* TRunManager::theRunManager = 0;

static Display* logoDisplay = 0;
static Window logoWindow = 0;
static Pixmap logoPixmap = 0;
static const TUint showtime = 4;

TRunManager::TRunManager( TUserInterface* ui )
  : theStatus( tStatusDead ),
    theRun(), theRunAction( 0 ), theEventManager( 0 ), 
    theUserInterface( ui ), theThreadOfEventLoop( 0 ),
    theRunInformationModule( 0 ), theAnalysisManager( 0 ),
    theRunTimer( 0 )
{
  if ( theRunManager ) {
    Tcerr << "TRunManager::TRunManager: RunManager constructed twice." << Tendl;
    exit( -EFAULT );
  }
  theRunManager = this;
  theEventManager = new TEventManager();

  if ( theUserInterface ) {
    installDefaultCommand();
  }

  Tthread_t thread;
  pthread_create( &thread, 0, TRunManager::showLogo, (Tvoid*)(&showtime) );

  PrintAuthorInformation();

  pthread_join( thread, 0 );
  pthread_detach( thread );

  theRunTimer = new TSystemTimer();

  theStatus = tStatusStandby;
}

TRunManager::~TRunManager()
{
  Tstring head = "TRunManager::~TRunManager: ";
  delete theEventManager;
  theEventManager = 0;
  Tcout << head << "EventManager was deleted." << Tendl;
  if ( theUserInterface ) {
    delete theUserInterface;
    theUserInterface = 0;
    Tcout << head << "UserInterface was deleted." << Tendl;
  }
  if ( theRunAction ) {
    delete theRunAction;
    theRunAction = 0;
    Tcout << head << "RunAction was deleted." << Tendl;
  }
  if ( theAnalysisManager ) {
    delete theAnalysisManager;
    theAnalysisManager = 0;
    Tcout << head << "AnalysisManager was deleted." << Tendl;
  }
  if ( theRunTimer ) {
    delete theRunTimer;
    theRunTimer = 0;
    Tcout << head << "RunTimer was deleted." << Tendl;
  }
  theStatus = tStatusDead;
  Tcout << head << "RunManager was deleted." << Tendl;
  theRunManager = 0;
}

Tvoid TRunManager::SessionStart()
{
  while ( theUserInterface -> AcceptCommand() ) {

    Tstring input = theUserInterface -> GetInputCommand();
    if ( input.empty() )
      continue;

    TstringList args = getArguments( input );
    Tstring comname = args[ 0 ];
    args.erase( args.begin() );

    TCommand* command = 0;
    if ( ( command = theUserInterface -> FindCommand( comname ) ) ) {
      command -> Execute( args );



      // status check after execution
      if ( theStatus == tStatusDead ) {
	break;
      } else if ( theStatus == tStatusReady ) {
	GoEventLoop();
      } else if ( theStatus == tStatusWaitingReady ) {
	WaitReturnFromEventLoop();
      } else if ( theStatus == tStatusIdle ) {
	;// nothing to do
      } else if ( theStatus == tStatusStandby ) {
	;// nothing to do
      } else if ( theStatus == tStatusBusy ) {
	;// nothing to do
      } else {
	break;
      }




    } else if ( !( comname.empty() ) ) {
      theUserInterface -> NotFoundCommand( comname );
    } else {
      continue;
    }
  }

  return;
}

Tint TRunManager::SetUserCommand( TCommand* command ) const
{
  if ( theUserInterface == 0 ) {
    Tcerr << "TRunManager::SetUserCommand: unset user interface." << Tendl;
    return( -EFAULT );
  }
  return( theUserInterface -> AddCommand( command ) );
}

Tvoid TRunManager::ShowStatus() const
{
  Tcout << "Status: ";
  Tstring msg;
  switch ( theStatus ) {
    case tStatusStandby:
      msg = "Standby - [/run/start] or [/run/shutdown] command is acceptable.";
      break;
    case tStatusReady:
      msg = "Ready - waiting for event(trigger), [/run/suspend] command is acceptable.";
      break;
    case tStatusBusy:
      msg = "Busy - just taking data from HW/SW.";
      break;
    case tStatusIdle:
      msg = "Idle - if input [/run/resume] command, the status is going to transit Ready. and then, [/run/stop] command is also acceptable.";
      break;
    case tStatusWaitingReady:
      msg = "WaitingReady - waiting for transition to status Ready.";
      break;
    case tStatusDead:
      msg = "Dead - not allocate, or some trouble ?";
      break;
    case tStatusUnknown:
    default:
      msg = "Unknown - unknown status.";
      break;
  }
  Tcout << msg << Tendl;
  Tcout << "Run ID: " << theRun.GetRunID() << Tendl;
  return;
}

Tvoid TRunManager::ShutdownRun()
{
  if ( theStatus != tStatusStandby ) {
    ShowStatus();
    return;
  }
  if ( theRunAction )
    theRunAction -> FinalOfRunAction( theRun );
  theStatus = tStatusDead;
  return;
}

Tvoid TRunManager::StartRun()
{
  if ( theStatus != tStatusStandby ) {
    ShowStatus();
    return;
  }
  theRunTimer -> Start();
  Tcout << "Run #" << theRun.GetRunID() << " start." << Tendl;
  theRun.Clear();
  if ( theRunAction )
    theRunAction -> BeginOfRunAction( theRun );
  theStatus = tStatusReady;
  return;
}

Tvoid TRunManager::StartRun( Tint runid )
{
  if ( theStatus != tStatusStandby ) {
    ShowStatus();
    return;
  }
  theRunTimer -> Start();
  Tcout << "Run #" << runid << " start." << Tendl;
  theRun.Clear();
  theRun.SetRunID( runid );
  if ( theRunAction )
    theRunAction -> BeginOfRunAction( theRun );
  theStatus = tStatusReady;
  return;
}

Tvoid TRunManager::StopRun()
{
  if ( theStatus != tStatusIdle ) {
    ShowStatus();
    return;
  }
  if ( theRunAction )
    theRunAction -> EndOfRunAction( theRun );
  Tint runid = theRun.GetRunID();

  Tint neventsbuf = theEventManager -> GetNumberOfEvents();
  Tint runidbuf = runid;

  theRun.SetRunID( ++ runid );
  theEventManager -> Clear();
  theStatus = tStatusStandby;
  theRunTimer -> Stop();

  Tcout << *theRunTimer << Tendl;
  Tcout << "Run #" << runidbuf << ": ";
  Tcout << neventsbuf << " events have been generated." << Tendl;

  return;
}

Tvoid TRunManager::SuspendRun()
{
  if ( theStatus == tStatusBusy ) {
    theStatus = tStatusWaitingReady;
    return;
  } else if ( theStatus != tStatusReady ) {
    ShowStatus();
    return;
  }
  theStatus = tStatusIdle;
  return;
}

Tvoid TRunManager::ResumeRun()
{
  if ( theStatus != tStatusIdle ) {
    ShowStatus();
    return;
  }
  theStatus = tStatusReady;
  return;
}

Tvoid* TRunManager::doEventLoop( Tvoid* arguments )
{
  if ( TRunManager::GetRunManager() -> GetStatus() != tStatusReady ) {
    TRunManager::GetRunManager() -> ShowStatus();
    return( 0 );
  }
  while ( TRunManager::GetRunManager() -> GetStatus() != tStatusIdle ) {
    if ( TRunManager::GetRunManager() -> GetStatus() == tStatusWaitingReady ) {
      TRunManager::GetRunManager() -> SetStatus( tStatusIdle );
      break;
    } else {
      TRunManager::GetRunManager() -> SetStatus( tStatusBusy );
    }
    TRunManager::GetRunManager() -> GetEventManager() -> TakeEvent();
    TRunManager::GetRunManager() -> GetEventManager() -> RecordEvent();
    if ( TRunManager::GetRunManager() -> GetStatus() == tStatusWaitingReady ) {
      TRunManager::GetRunManager() -> SetStatus( tStatusIdle );
      break;
    } else {
      TRunManager::GetRunManager() -> SetStatus( tStatusReady );
    }
  }
  return( 0 );
}

Tvoid* TRunManager::showLogo( Tvoid* arguments )
{
  TRunManager::GetRunManager() -> PopupLogo();
  sleep( *( (TUint*)arguments ) );
  TRunManager::GetRunManager() -> PopdownLogo();
  return( 0 );
}

Tvoid TRunManager::GoEventLoop()
{
  pthread_create( &theThreadOfEventLoop, 0, TRunManager::doEventLoop, 0 );
  return;
}

Tvoid TRunManager::WaitReturnFromEventLoop()
{
  pthread_join( theThreadOfEventLoop, 0 );
  pthread_detach( theThreadOfEventLoop );
  return;
}

Tvoid TRunManager::PopupLogo() const
{
  if ( logoPixmap )
    return;

  logoDisplay = XOpenDisplay( "" );
  if ( !logoDisplay )
    return;

  Tint screen = DefaultScreen( logoDisplay );
  TUlong back = WhitePixel( logoDisplay, screen );
  TUlong fore = BlackPixel( logoDisplay, screen );
  logoWindow =
    XCreateSimpleWindow( logoDisplay, DefaultRootWindow( logoDisplay ),
			 -100, -100, 50, 50, 0, fore, back );
  Tint depth = PlanesOfScreen( XDefaultScreenOfDisplay( logoDisplay ) );

  if ( depth > 1 ) {
    XWindowAttributes win_attr;
    XGetWindowAttributes( logoDisplay, logoWindow, &win_attr );

    XpmAttributes attr;
    attr.valuemask = XpmVisual | XpmColormap | XpmDepth;
    attr.visual = win_attr.visual;
    attr.colormap = win_attr.colormap;
    attr.depth = win_attr.depth;
    attr.valuemask |= XpmColorKey;

    if ( depth > 4 )
      attr.color_key = XPM_COLOR;
    else if ( depth > 2 )
      attr.color_key = XPM_GRAY4;
    else if ( depth > 1 )
      attr.color_key = XPM_GRAY;
    else if ( depth == 1 )
      attr.color_key = XPM_MONO;
    else
      attr.valuemask &= ~XpmColorKey;

    XpmCreatePixmapFromData( logoDisplay, logoWindow,
			     (char**)Tsplash, &logoPixmap,
			     (Pixmap *)0, &attr );
    XpmFreeAttributes( &attr );
  }

  Window root;
  Tint x;
  Tint y;
  TUint w;
  TUint h;
  TUint bw;
  TUint dep;
  XGetGeometry( logoDisplay, logoPixmap, &root, &x, &y, &w, &h, &bw, &dep );
  Screen* xscreen = XDefaultScreenOfDisplay( logoDisplay );
  x = ( WidthOfScreen( xscreen ) - w ) / 2;
  y = ( HeightOfScreen( xscreen ) - h ) / 2;
  XMoveResizeWindow( logoDisplay, logoWindow, x, y, w, h );
  XSync( logoDisplay, False );

  TUlong valmask;
  XSetWindowAttributes xswa;
  valmask = CWBackPixmap | CWOverrideRedirect;
  xswa.background_pixmap = logoPixmap;
  xswa.override_redirect = True;
  XChangeWindowAttributes( logoDisplay, logoWindow, valmask, &xswa );

  XMapRaised( logoDisplay, logoWindow );
  XSync( logoDisplay, False );

  return;
}

Tvoid TRunManager::PopdownLogo() const
{
  if ( logoWindow ) {
    XUnmapWindow( logoDisplay, logoWindow );
    XDestroyWindow( logoDisplay, logoWindow );
    logoWindow = 0;
  }
  if ( logoPixmap ) {
    XFreePixmap( logoDisplay, logoPixmap );
    logoPixmap = 0;
  }
  if ( logoDisplay ) {
    XSync( logoDisplay, False );
    XCloseDisplay( logoDisplay );
    logoDisplay = 0;
  }
  return;
}

Tvoid TRunManager::PrintAuthorInformation( Tostream& tos ) const
{
  //terminfo ȤäƤƤ⤦礤쥬Ȥ
  tos << "================================================================================" << Tendl;
  tos << " * Project: " << Tproject << "                                                           * " << Tendl;
  tos << " * Version: " << Tversion << "                                                             * " << Tendl;
  tos << " * 8050 Ikarashi 2-no-chou, Niigata City, 950-2181, Japan                     * " << Tendl;
  tos << " * Phone & Fax: +81-25-262-6138                                               * " << Tendl;
  tos << " * Go IWAI iwai@hep.sc.niigata-u.ac.jp                                        * " << Tendl;
  tos << "================================================================================" << Tendl;
  return;
}

Tvoid TRunManager::SetEventManager( TEventManager* manager )
{
  if ( theEventManager ) {
    Tstring head = "TRunManager::SetEventManager: ";
    delete theEventManager;
    Tcout << head << "EventManager was deleted." << Tendl;
    theEventManager = 0;
  }
  theEventManager = manager;
  return;
}

Tvoid TRunManager::SetUserInterface( TUserInterface* ui )
{
  if ( theUserInterface ) {
    Tstring head = "TRunManager::SetUserInterface: ";
    delete theUserInterface;
    Tcout << head << "UserInterface was deleted." << Tendl;
    theUserInterface = 0;
  }
  theUserInterface = ui;
  installDefaultCommand();
  return;
}

Tvoid TRunManager::SetRunAction( TRunAction* action )
{
  if ( theRunAction ) {
    Tstring head = "TRunManager::SetRunAction: ";
    delete theRunAction;
    Tcout << head << "RunAction was deleted." << Tendl;
    theRunAction = 0;
  }
  theRunAction = action;
  return;
}

Tvoid TRunManager::SetEventAction( TEventAction* action )
{
  theEventManager -> SetEventAction( action );
  return;
}

Tvoid TRunManager::SetRunInformationModule( TSoftwareRunInformationModule* module )
{
  if ( theUserInterface ) {
    SetUserCommand( new TSetRunInformationModuleCommand( this, module ) );
    SetUserCommand( new TAddRunInformationModuleCommand( this, module ) );
    SetUserCommand( new TClearRunInformationModuleCommand( this, module ) );
    SetUserCommand( new TInitializeRunInformationModuleCommand( this, module ) );
    SetUserCommand( new TShowRunInformationModuleCommand( this, module ) );
  }
  theRunInformationModule = module;
  return;
}

Tvoid TRunManager::SetAnalysisManager( TAnalysisManager* manager )
{
  if ( theUserInterface ) {
    SetUserCommand( new TAnalysisStartCommand( this, manager ) );
    SetUserCommand( new TAnalysisStopCommand( this, manager ) );
    SetUserCommand( new TAnalysisSuspendCommand( this, manager ) );
    SetUserCommand( new TAnalysisResumeCommand( this, manager ) );
    SetUserCommand( new TShowAnalysisManagerCommand( this, manager ) );
  }
  if ( theAnalysisManager ) {
    Tstring head = "TRunManager::SetAnalysisManager: ";
    delete theAnalysisManager;
    theAnalysisManager = 0;
    Tcout << head << "AnalysisManager was deleted." << Tendl;
  }
  theAnalysisManager = manager;
  return;
}

Tvoid TRunManager::SetRunTimer( TSystemTimer* timer )
{
  if ( theRunTimer ) {
    Tstring head = "TRunManager::SetRunTimer: ";
    delete theRunTimer;
    theRunTimer = 0;
    Tcout << head << "RunTimer was deleted." << Tendl;
  }
  theRunTimer = timer;
  return;
}

TstringList TRunManager::getArguments( const Tstring& input ) const
{
  Tsize_t begin = 0;
  Tsize_t end = 0;
  TstringList strlist;
  Tsize_t lastpos = input.rfind( Tspace ) + 1;
  do {
    begin = input.find_first_not_of( Tspace, end );
    end = input.find( Tspace, begin );
    if ( end > input.size() )
      end = input.size();
    strlist.push_back( input.substr( begin, end - begin ) );
  } while ( begin != lastpos );
  return( strlist );
}

Tvoid TRunManager::installDefaultCommand()
{
  // built-in
  SetUserCommand( new TExitCommand( this ) );
  SetUserCommand( new TQuitCommand( this ) );
  SetUserCommand( new TListCommand( this ) );
  SetUserCommand( new TChangeDirectoryCommand( this ) );
  SetUserCommand( new TPrintCurrentWorkingDirectoryCommand( this ) );
  SetUserCommand( new THelpCommand( this ) );
  //system
  SetUserCommand( new TDateCommand( this ) );
  SetUserCommand( new TShowNetworkStatusCommand( this ) );
  SetUserCommand( new TExecuteCommand( this ) );
  SetUserCommand( new TSleepCommand( this ) );
  SetUserCommand( new TShellCommand( this ) );
  // option
  SetUserCommand( new TReadFileCommand( this ) );
  SetUserCommand( new TPopupLogoCommand( this ) );
  SetUserCommand( new TPopdownLogoCommand( this ) );
  SetUserCommand( new TPrintAuthorInformationCommand( this ) );
  // event
  SetUserCommand( new TSetStackSizeCommand( this ) );
  SetUserCommand( new TShowEventManagerCommand( this ) );
  // run
  SetUserCommand( new TRunStartCommand( this ) );
  SetUserCommand( new TRunStopCommand( this ) );
  SetUserCommand( new TRunSuspendCommand( this ) );
  SetUserCommand( new TRunResumeCommand( this ) );
  SetUserCommand( new TRunShutdownCommand( this ) );
  SetUserCommand( new TShowRunManagerCommand( this ) );
  return;
}
