// ============================================================================
//  $Id: TSoftwareTimerModule.cc,v 1.1.1.1 2003/01/04 08:07:14 iwai Exp $
//  $Name:  $
// ============================================================================
#include "TSoftwareTimerModule.hh"
#include "TDataSegment.hh"
#include "TDataElement.hh"

TSoftwareTimerModule::TSoftwareTimerModule( const Tstring& unit, Tint nchannel )
  : TSoftwareModule( nchannel ), 
    theIntervalTime( 0.0 ), theCurrentTime( 0.0 ), thePreviousTime( 0.0 )
{
  static const Tstring head = "TSoftwareTimerModule::TSoftwareTimerModule: ";
  if ( unit == Tsec || unit == Tmsec || unit == Tusec ) {
    theUnit = unit;
  } else {
    Tcerr << head << "invalid unit" << Tendl;
    Tcerr << head << "[" << Tmsec << "]" << " use" << Tendl;
    theUnit = Tmsec;
  }

  if ( nchannel > tNumberOfChannels ) {
    Tcerr << head << "invalid channels" << Tendl;
    Tcerr << head << "revised " << tNumberOfChannels << Tendl;
    theNumberOfChannels = tNumberOfChannels;
  }
  setCurrentTime();
  Update();
}

TSoftwareTimerModule::TSoftwareTimerModule( const TSoftwareTimerModule& right )
  : TSoftwareModule( right ), 
    theIntervalTime( right.theIntervalTime ),
    theCurrentTime( right.theCurrentTime ),
    thePreviousTime( right.thePreviousTime ),
    theUnit( right.theUnit )
{;}

TSoftwareTimerModule::~TSoftwareTimerModule()
{;}

Tvoid TSoftwareTimerModule::FillData( TDataElement& element, Tint channel )
{
  Update();

  if ( channel < 0 || channel >= theNumberOfChannels ) {
    Tcerr << "TSoftwareScalerModule::FillData: invalid ID " << channel << Tendl;
    theStatus = -EFAULT;
    element.FillData( &theStatus, tTypeInt, 1 );
  } else if ( channel == tIntervalTime ) {
    element.FillData( &theIntervalTime, tTypeDouble, 1 );
  } else if ( channel == tCurrentTime ) {
    element.FillData( &theCurrentTime, tTypeDouble, 1 );
  } else if ( channel == tPreviousTime ) {
    element.FillData( &thePreviousTime, tTypeDouble, 1 );
  } else if ( channel == tUnit ) {
    element.FillData( &theUnit, tTypeString, 1 );
  } else {
    Tcerr << "TSoftwareScalerModule::FillData: invalid ID " << channel << Tendl;
    theStatus = -EFAULT;
    element.FillData( &theStatus, tTypeInt, 1 );
  }

  return;
}

const TSoftwareTimerModule& TSoftwareTimerModule::operator=( const TSoftwareTimerModule& right )
{
  *( (TSoftwareModule*)this ) = *( (TSoftwareModule*)(&right) );
  theIntervalTime = right.theIntervalTime;
  theCurrentTime = right.theCurrentTime;
  thePreviousTime = right.thePreviousTime;
  theUnit = right.theUnit;
  return( *this );
}

Tbool TSoftwareTimerModule::operator==( const TSoftwareTimerModule& right ) const
{
  Tbool ret = Ttrue;
  ret &= ( *( (TSoftwareModule*)this ) == *( (TSoftwareModule*)(&right) ) );
  ret &= ( theUnit == right.theUnit );
  return( ret );
}

Tbool TSoftwareTimerModule::operator!=( const TSoftwareTimerModule& right ) const
{
  Tbool ret = Tfalse;
  ret |= ( *( (TSoftwareModule*)this ) != *( (TSoftwareModule*)(&right) ) );
  ret |= ( theUnit != right.theUnit );
  return( ret );
}

Tvoid TSoftwareTimerModule::setCurrentTime()
{
  thePreviousTime = theCurrentTime;

  struct timeval now;
  gettimeofday( &now, 0 );

  Tdouble tv_sec = (Tdouble)( now.tv_sec );
  Tdouble tv_usec = (Tdouble)( now.tv_usec );

  Tdouble usec = ( tv_sec * 1.0e+6 ) + tv_usec;

  if ( theUnit == Tsec )
    theCurrentTime = usec * 1.0e-6;
  else if ( theUnit == Tmsec )
    theCurrentTime = usec * 1.0e-3;
  else if ( theUnit == Tusec )
    theCurrentTime = usec;

  theIntervalTime = theCurrentTime - thePreviousTime;
  if ( theIntervalTime < 0 )
    theIntervalTime = (Tdouble)( -EFAULT );

  return;
}
