%-*-Mode:erlang;coding:utf-8;tab-width:4;c-basic-offset:4;indent-tabs-mode:()-*-
% ex: set ft=erlang fenc=utf-8 sts=4 ts=4 sw=4 et nomod:
%%%
%%%------------------------------------------------------------------------
%%% @doc
%%% ==CloudI Statistics==
%%% Calculate statistics using the online algorithm.
%%%
%%% Philippe Pébay, Timothy B. Terriberry, Hemanth Kolla, Janine Bennett.
%%% Formulas for the Computation of Higher-Order Central Moments.
%%% Technical Report SAND2014-17343J, Sandia National Laboratories, 2014.
%%%
%%% Pébay, Philippe.  Formulas for Robust, One-Pass Parallel Computation of
%%% Covariances and Arbitrary-Order Statistical Moments.
%%% Technical Report SAND2008-6212, Sandia National Laboratories, 2008.
%%%
%%% Welford, B. P.. Note on a method for calculating corrected sums of
%%% squares and products. Technometrics vol. 4, no. 3, pp. 419–420, 1962.
%%% @end
%%%
%%% MIT License
%%%
%%% Copyright (c) 2022 Michael Truog <mjtruog at protonmail dot com>
%%%
%%% Permission is hereby granted, free of charge, to any person obtaining a
%%% copy of this software and associated documentation files (the "Software"),
%%% to deal in the Software without restriction, including without limitation
%%% the rights to use, copy, modify, merge, publish, distribute, sublicense,
%%% and/or sell copies of the Software, and to permit persons to whom the
%%% Software is furnished to do so, subject to the following conditions:
%%%
%%% The above copyright notice and this permission notice shall be included in
%%% all copies or substantial portions of the Software.
%%%
%%% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
%%% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
%%% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
%%% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
%%% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
%%% FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
%%% DEALINGS IN THE SOFTWARE.
%%%
%%% @author Michael Truog <mjtruog at protonmail dot com>
%%% @copyright 2022 Michael Truog
%%% @version 2.0.5 {@date} {@time}
%%%------------------------------------------------------------------------

-module(cloudi_statistics).
-author('mjtruog at protonmail dot com').

%% external interface
-export([add/2,
         add_from_list/2,
         add_from_list/3,
         count/1,
         describe_distribution/1,
         describe_kurtosis/1,
         describe_skewness/1,
         kurtosis/1,
         mean/1,
         merge/2,
         new/0,
         normal_from_log_normal/1,
         normal_to_log_normal/1,
         skewness/1,
         standard_deviation/1,
         maximum/1,
         minimum/1,
         variance/1]).

-include("cloudi_core_i_constants.hrl").

-record(statistics,
    {
        n = 0 :: non_neg_integer(),
        mean = 0.0 :: float(),
        m2 = 0.0 :: float(),
        m3 = 0.0 :: float(),
        m4 = 0.0 :: float(),
        minimum = ?DBL_MAX :: float(),
        maximum = -?DBL_MAX :: float(),
        cached_kurtosis = undefined :: float() | undefined,
        cached_skewness = undefined :: float() | undefined,
        cached_stddev = undefined :: float() | undefined,
        cached_variance = undefined :: float() | undefined
    }).

%%%------------------------------------------------------------------------
%%% External interface functions
%%%------------------------------------------------------------------------

-type describe_distribution() ::
    normal | uniform | logistic | exponential |
    gamma_family | log_normal_family | undefined.
-type describe_kurtosis() ::
    leptokurtic | platykurtic | mesokurtic | undefined.
-type describe_skewness() ::
    highly_skewed | moderately_skewed | approximately_symmetric | undefined.
-type state() :: #statistics{}.
-export_type([describe_distribution/0,
              describe_kurtosis/0,
              describe_skewness/0,
              state/0]).

%%-------------------------------------------------------------------------
%% @doc
%% ===Add a sample for computing statistics.===
%% @end
%%-------------------------------------------------------------------------

-spec add(X :: number(),
          State :: state()) ->
    state().

add(X, #statistics{n = NOld,
                   mean = Mean,
                   m2 = M2,
                   m3 = M3,
                   m4 = M4,
                   minimum = Minimum,
                   maximum = Maximum} = State)
    when is_number(X) ->
    XAdd = float(X),
    N = NOld + 1,
    Delta = XAdd - Mean,
    DeltaN = Delta / N,
    DeltaN2 = DeltaN * DeltaN,
    Term1 = Delta * DeltaN * NOld,
    MeanNew = Mean + DeltaN,
    M4New = M4 +
            Term1 * DeltaN2 * (N * N - 3 * N + 3) +
            6 * DeltaN2 * M2 -
            4 * DeltaN * M3,
    M3New = M3 +
            Term1 * DeltaN * (N - 2) -
            3 * DeltaN * M2,
    M2New = M2 + Term1,
    MinimumNew = erlang:min(Minimum, XAdd),
    MaximumNew = erlang:max(Maximum, XAdd),
    State#statistics{n = N,
                     mean = MeanNew,
                     m2 = M2New,
                     m3 = M3New,
                     m4 = M4New,
                     minimum = MinimumNew,
                     maximum = MaximumNew,
                     cached_kurtosis = undefined,
                     cached_skewness = undefined,
                     cached_stddev = undefined,
                     cached_variance = undefined}.

%%-------------------------------------------------------------------------
%% @doc
%% ===Add samples from a list for computing statistics.===
%% @end
%%-------------------------------------------------------------------------

-spec add_from_list(L :: list(number()),
                    State :: state()) ->
    state().

add_from_list([], State) ->
    State;
add_from_list([X | L], State) ->
    add_from_list(L, add(X, State)).

%%-------------------------------------------------------------------------
%% @doc
%% ===Add samples from a function mapped on a list for computing statistics.===
%% @end
%%-------------------------------------------------------------------------

-spec add_from_list(F :: fun(),
                    L :: list(number()),
                    State :: state()) ->
    state().

add_from_list(F, L, State) ->
    if
        is_function(F, 1) ->
            add_from_list_1(L, State, F);
        is_function(F, 2) ->
            add_from_list_2(L, State, F);
        is_function(F, 3) ->
            add_from_list_3(L, State, F)
    end.

%%-------------------------------------------------------------------------
%% @doc
%% ===Count of samples previously added.===
%% @end
%%-------------------------------------------------------------------------

-spec count(State :: state()) ->
    non_neg_integer().

count(#statistics{n = N}) ->
    N.

%%-------------------------------------------------------------------------
%% @doc
%% ===Describe the distribution with a guess about its shape.===
%% Only use if the sample count is large enough to cover the distribution.
%% @end
%%-------------------------------------------------------------------------

-spec describe_distribution(State :: state()) ->
    {describe_distribution(), state()}.

describe_distribution(State) ->
    case kurtosis(State) of
        {undefined, _} = Result ->
            Result;
        {Kurtosis, StateNext} ->
            case skewness(StateNext) of
                {undefined, _} = Result ->
                    Result;
                {Skewness, StateNew} ->
                    SkewnessDelta = if
                        Skewness =< -4 orelse Skewness >= 4 ->
                            math:sqrt(abs(Skewness)) * 0.5;
                        true ->
                            0.25
                    end,
                    KurtosisDelta = if
                        Kurtosis >= 9 ->
                            math:sqrt(Kurtosis) * 0.5;
                        true ->
                            0.175
                    end,
                    describe_distribution_0(Skewness,
                                            Skewness - SkewnessDelta,
                                            Skewness + SkewnessDelta,
                                            Kurtosis,
                                            Kurtosis - KurtosisDelta,
                                            Kurtosis + KurtosisDelta,
                                            StateNew)
            end
    end.

%%-------------------------------------------------------------------------
%% @doc
%% ===Describe the excess kurtosis value.===
%% Only use if the sample count is large enough to cover the distribution.
%% @end
%%-------------------------------------------------------------------------

-spec describe_kurtosis(State :: state()) ->
    {describe_kurtosis(), state()}.

describe_kurtosis(State) ->
    case kurtosis(State) of
        {undefined, _} = Result ->
            Result;
        {Kurtosis, StateNew} ->
            KurtosisX10 = round(Kurtosis * 10),
            if
                KurtosisX10 > 0 ->
                    {leptokurtic, StateNew};
                KurtosisX10 < 0 ->
                    {platykurtic, StateNew};
                KurtosisX10 == 0 ->
                    {mesokurtic, StateNew}
            end
    end.

%%-------------------------------------------------------------------------
%% @doc
%% ===Describe the skewness value.===
%% Only use if the sample count is large enough to cover the distribution.
%%
%% (based on)
%% Bulmer, M. G..
%% Principles of Statistics. Dover Publications, 1979.
%% @end
%%-------------------------------------------------------------------------

-spec describe_skewness(State :: state()) ->
    {describe_skewness(), state()}.

describe_skewness(State) ->
    case skewness(State) of
        {undefined, _} = Result ->
            Result;
        {Skewness, StateNew} ->
            if
                Skewness < -1.0 orelse Skewness > 1.0 ->
                    {highly_skewed, StateNew};
                Skewness =< -0.5 orelse Skewness >= 0.5 ->
                    {moderately_skewed, StateNew};
                Skewness > -0.5 andalso Skewness < 0.5 ->
                    {approximately_symmetric, StateNew}
            end
    end.

%%-------------------------------------------------------------------------
%% @doc
%% ===Excess kurtosis of samples previously added.===
%% The minimum value returned is -2.0 (Fisher’s definition).
%% @end
%%-------------------------------------------------------------------------

-spec kurtosis(State :: state()) ->
    {float() | undefined, state()}.

kurtosis(#statistics{n = N,
                     m2 = M2,
                     m4 = M4,
                     cached_kurtosis = undefined} = State) ->
    Denominator = M2 * M2,
    if
        Denominator /= 0.0 ->
            Kurtosis = N * M4 / Denominator - 3.0,
            {Kurtosis, State#statistics{cached_kurtosis = Kurtosis}};
        true ->
            {undefined, State}
    end;
kurtosis(#statistics{cached_kurtosis = Kurtosis} = State) ->
    {Kurtosis, State}.

%%-------------------------------------------------------------------------
%% @doc
%% ===Maximum of samples previously added.===
%% @end
%%-------------------------------------------------------------------------

-spec maximum(State :: state()) ->
    float().

maximum(#statistics{maximum = Maximum}) ->
    Maximum.

%%-------------------------------------------------------------------------
%% @doc
%% ===Mean of samples previously added.===
%% The value returned is the arithmetic mean.
%% @end
%%-------------------------------------------------------------------------

-spec mean(State :: state()) ->
    float().

mean(#statistics{mean = Mean}) ->
    Mean.

%%-------------------------------------------------------------------------
%% @doc
%% ===Merge statistics state.===
%% @end
%%-------------------------------------------------------------------------

-spec merge(StateA :: state(),
            StateB :: state()) ->
    state().

merge(#statistics{} = StateA,
      #statistics{n = 0}) ->
    StateA;
merge(#statistics{n = 0},
      #statistics{} = StateB) ->
    StateB;
merge(#statistics{n = NA,
                  mean = MeanA,
                  m2 = M2A,
                  m3 = M3A,
                  m4 = M4A,
                  minimum = MinimumA,
                  maximum = MaximumA},
      #statistics{n = NB,
                  mean = MeanB,
                  m2 = M2B,
                  m3 = M3B,
                  m4 = M4B,
                  minimum = MinimumB,
                  maximum = MaximumB}) ->
    N = NA + NB,
    Delta = MeanA - MeanB,
    DeltaN = Delta / N,
    DeltaN2 = DeltaN * DeltaN,
    NA2 = NA * NA,
    NB2 = NB * NB,
    N_2 = NA * NB,
    M4 = M4A + M4B +
         DeltaN2 * DeltaN2 * N_2 * (NB * NB2 + NA * NA2) +
         6.0 * (NB2 * M2A + NA2 * M2B) * DeltaN2 +
         4.0 * (NB * M3A - NA * M3B) * DeltaN,
    M3 = M3A + M3B +
         N_2 * (NB - NA) * Delta * DeltaN2 +
         3.0 * (NB * M2A - NA * M2B) * DeltaN,
    M2 = M2A + M2B +
         N_2 * Delta * DeltaN,
    Mean = MeanB + NA * DeltaN,
    Minimum = erlang:min(MinimumA, MinimumB),
    Maximum = erlang:max(MaximumA, MaximumB),
    #statistics{n = N,
                mean = Mean,
                m2 = M2,
                m3 = M3,
                m4 = M4,
                minimum = Minimum,
                maximum = Maximum}.

%%-------------------------------------------------------------------------
%% @doc
%% ===Minimum of samples previously added.===
%% @end
%%-------------------------------------------------------------------------

-spec minimum(State :: state()) ->
    float().

minimum(#statistics{minimum = Minimum}) ->
    Minimum.

%%-------------------------------------------------------------------------
%% @doc
%% ===Create statistics state.===
%% @end
%%-------------------------------------------------------------------------

-spec new() ->
    state().

new() ->
    #statistics{}.

%%-------------------------------------------------------------------------
%% @doc
%% ===Convert the Log-normal distribution mean and standard deviation to Normal distribution mean and standard deviation.===
%% @end
%%-------------------------------------------------------------------------

-spec normal_from_log_normal(State :: state()) ->
    {{NormalMean :: float(),
      NormalStdDev :: float()} | undefined, state()}.

normal_from_log_normal(State) ->
    case variance(State) of
        {undefined, _} = Result ->
            Result;
        {Variance, #statistics{mean = Mean} = StateNew} ->
            NormalMean = math:exp(Mean + Variance / 2),
            NormalStandardDeviation = math:sqrt((math:exp(Variance) - 1) *
                                                math:exp(2 * Mean + Variance)),
            {{NormalMean, NormalStandardDeviation}, StateNew}
    end.

%%-------------------------------------------------------------------------
%% @doc
%% ===Convert the Normal distribution mean and standard deviation to Log-normal distribution mean and standard deviation.===
%% @end
%%-------------------------------------------------------------------------

-spec normal_to_log_normal(State :: state()) ->
    {{LogNormalMean :: float(),
      LogNormalStdDev :: float()} | undefined, state()}.

normal_to_log_normal(State) ->
    case variance(State) of
        {undefined, _} = Result ->
            Result;
        {Variance, #statistics{mean = Mean} = StateNew} ->
            Mean2 = Mean * Mean,
            Sum = Mean2 + Variance,
            LogNormalMean = math:log(Mean2 / math:sqrt(Sum)),
            LogNormalStandardDeviation = math:sqrt(math:log(Sum / Mean2)),
            {{LogNormalMean, LogNormalStandardDeviation}, StateNew}
    end.

%%-------------------------------------------------------------------------
%% @doc
%% ===Sample skewness of samples previously added.===
%% The value returned is the Fisher-Pearson coefficient of skewness.
%% @end
%%-------------------------------------------------------------------------

-spec skewness(State :: state()) ->
    {float() | undefined, state()}.

skewness(#statistics{n = N,
                     m2 = M2,
                     m3 = M3,
                     cached_skewness = undefined} = State) ->
    Denominator = math:pow(M2, 1.5),
    if
        Denominator /= 0.0 ->
            Skewness = math:sqrt(N) * M3 / Denominator,
            {Skewness, State#statistics{cached_skewness = Skewness}};
        true ->
            {undefined, State}
    end;
skewness(#statistics{cached_skewness = Skewness} = State) ->
    {Skewness, State}.

%%-------------------------------------------------------------------------
%% @doc
%% ===Sample standard deviation of samples previously added.===
%% @end
%%-------------------------------------------------------------------------

-spec standard_deviation(State :: state()) ->
    {float() | undefined, state()}.

standard_deviation(#statistics{cached_stddev = undefined} = State) ->
    case variance(State) of
        {undefined, _} = Result ->
            Result;
        {Variance, StateNew} ->
            StandardDeviation = math:sqrt(Variance),
            {StandardDeviation,
             StateNew#statistics{cached_stddev = StandardDeviation}}
    end;
standard_deviation(#statistics{cached_stddev = StandardDeviation} = State) ->
    {StandardDeviation, State}.

%%-------------------------------------------------------------------------
%% @doc
%% ===Sample variance of samples previously added.===
%% @end
%%-------------------------------------------------------------------------

-spec variance(State :: state()) ->
    {float() | undefined, state()}.

variance(#statistics{n = N,
                     m2 = M2,
                     cached_variance = undefined} = State) ->
    if
        N > 1 ->
            Variance = M2 / (N - 1.0),
            {Variance, State#statistics{cached_variance = Variance}};
        true ->
            {undefined, State}
    end;
variance(#statistics{cached_variance = Variance} = State) ->
    {Variance, State}.

%%%------------------------------------------------------------------------
%%% Private functions
%%%------------------------------------------------------------------------

add_from_list_1([X0 | L], State, F) ->
    add_from_list_1(L, add(F(X0), State), F);
add_from_list_1([], State, _) ->
    State.

add_from_list_2([X0, X1 | L], State, F) ->
    add_from_list_2(L, add(F(X0, X1), State), F);
add_from_list_2(_, State, _) ->
    State.

add_from_list_3([X0, X1, X2 | L], State, F) ->
    add_from_list_3(L, add(F(X0, X1, X2), State), F);
add_from_list_3(_, State, _) ->
    State.

describe_distribution_0(Skewness, SkewnessLow, SkewnessHigh,
                        Kurtosis, KurtosisLow, KurtosisHigh, State)
    when SkewnessLow < 0 andalso SkewnessHigh > 0 ->
    if
        KurtosisLow < 0 andalso KurtosisHigh > 0 ->
            {normal, State};
        KurtosisLow < -1.2 andalso KurtosisHigh > -1.2 ->
            {uniform, State};
        KurtosisLow < 1.2 andalso KurtosisHigh > 1.2 ->
            {logistic, State};
        true ->
            describe_distribution_1(Skewness, SkewnessLow, SkewnessHigh,
                                    Kurtosis, KurtosisLow, KurtosisHigh, State)
    end;
describe_distribution_0(Skewness, SkewnessLow, SkewnessHigh,
                        Kurtosis, KurtosisLow, KurtosisHigh, State)
    when SkewnessLow < 2 andalso SkewnessHigh > 2 ->
    if
        KurtosisLow < 6 andalso KurtosisHigh > 6 ->
            {exponential, State};
        true ->
            describe_distribution_1(Skewness, SkewnessLow, SkewnessHigh,
                                    Kurtosis, KurtosisLow, KurtosisHigh, State)
    end;
describe_distribution_0(Skewness, SkewnessLow, SkewnessHigh,
                        Kurtosis, KurtosisLow, KurtosisHigh, State) ->
    describe_distribution_1(Skewness, SkewnessLow, SkewnessHigh,
                            Kurtosis, KurtosisLow, KurtosisHigh, State).

describe_distribution_1(Skewness, SkewnessLow, SkewnessHigh,
                        Kurtosis, KurtosisLow, KurtosisHigh, State) ->
    TestL0 = [],
    TestL1 = skewness_check(TestL0,
                            gamma_family, fun gamma_kurtosis/1,
                            Skewness, SkewnessLow, SkewnessHigh,
                            Kurtosis, KurtosisLow, KurtosisHigh),
    TestLN = skewness_check(TestL1,
                            log_normal_family, fun log_normal_kurtosis/1,
                            Skewness, SkewnessLow, SkewnessHigh,
                            Kurtosis, KurtosisLow, KurtosisHigh),
    case TestLN of
        [{_, TestType} | _] ->
            {TestType, State};
        [] ->
            {undefined, State}
    end.

skewness_check(TestL, TestType, TestF,
               Skewness, SkewnessLow, SkewnessHigh,
               Kurtosis, KurtosisLow, KurtosisHigh)
    when is_function(TestF, 1) ->
    TestKurtosis = TestF(Skewness),
    Passed = if
        KurtosisLow < TestKurtosis andalso
        KurtosisHigh > TestKurtosis ->
            true;
        true ->
            TestKurtosisLow = TestF(SkewnessLow),
            TestKurtosisHigh = TestF(SkewnessHigh),
            TestKurtosisLow < Kurtosis andalso TestKurtosisHigh > Kurtosis
    end,
    if
        Passed =:= true ->
            lists:keymerge(1, TestL,
                           [{abs(TestKurtosis - Kurtosis), TestType}]);
        Passed =:= false ->
            TestL
    end.

gamma_kurtosis(Skewness) ->
    X = 2 / Skewness,
    6 / (X * X).

log_normal_kurtosis(0.0) ->
    0.0;
log_normal_kurtosis(Skewness) ->
    A = log_normal_skewness_a(Skewness),
    A2 = A * A,
    A2 * A2 + 2 * A2 * A + 3 * A2 - 6.

log_normal_skewness_a(Skewness) ->
    % use Halley's method to determine A based on Log-normal skewness
    % A = e^(NormalStandardDeviation^2)
    % (A + 2)^2 * (A - 1) = Skewness^2
    % B = A - 1
    % (B + 3)^2 * B = Skewness^2
    Skewness2 = Skewness * Skewness,
    log_normal_skewness_b(math:pow(Skewness2, 1 / 3) - 1, Skewness2) + 1.

log_normal_skewness_b(B, Skewness2) ->
    B2 = B * B,
    F0B = B2 * B + 6 * B2 + 9 * B - Skewness2,
    F1B = 3 * B2 + 12 * B + 9,
    F2B = 6 * B + 12,
    BE = B - (2 * F0B * F1B) / ((2 * F1B * F1B) - (F0B * F2B)),
    if
        abs(BE - B) > 0.0001 ->
            log_normal_skewness_b(BE, Skewness2);
        true ->
            BE
    end.

-ifdef(TEST).
-include_lib("eunit/include/eunit.hrl").

-include("cloudi_core_i_test.hrl").

module_test_() ->
    {timeout, ?TEST_TIMEOUT, [
        {"random value tests", ?_assertOk(t_random_value())}
    ]}.

% 1000000 random floats in [0.0 .. 1.0] within a constant list
%
% Created with:
% file:write_file("random_value.txt", io_lib:format("~80p", [lists:foldl(fun(_, L) -> [cloudi_x_quickrand:strong_float() | L] end, [], lists:seq(1, 1000000))])).
-define(RANDOM_VALUE_LIST,
[0.7779093128442551,0.5602545451350247,0.7766852376495581,0.9456642438377318,
 0.5990680481881936,0.047728379072267346,0.6586301304420381,
 0.012254783431982719,0.043885545402663206,0.6954762542392063,
 0.6166421525865262,0.08520025290046163,0.32141882605947714,
 0.9491578130339128,0.7676909159419413,0.5978960396563437,0.6586190704559778,
 0.5420784849708833,0.04963360638083392,0.018027253608207117,
 0.1455830378177625,0.739503834577978,0.41401933401844115,0.469807458374774,
 0.1707761434686763,0.5300297684613087,0.31700184862168646,0.556650914981896,
 0.1593775953149703,0.5234266161384629,0.09497197842604355,0.7577930121361439,
 0.8886013869049368,0.4412648984842056,0.9375472907328632,0.10637635755949204,
 0.14426795030317674,0.9823135065282288,0.6029382152183882,0.2810693108319612,
 0.43998476653608765,0.9720183422937151,0.5149035795910012,0.5311192106119298,
 0.5124394032004924,0.6978845298598773,0.3934114310425142,0.5373092586638265,
 0.059783358190530156,0.07756884285044352,0.9957386314231177,
 0.882570998517984,0.5078797318377497,0.8412208077238221,0.4871749049412779,
 0.9995592174447343,0.1568791828252939,0.8628787822911164,0.3415477516597967,
 0.4032277859695923,0.06411823600977717,0.23901126561601682,
 0.3105824516745782,0.6902515751833761,0.7074114037928012,
 0.025794563251372193,0.20619880508770516,0.5923734953402301,
 0.2840757197100313,0.26427712019886473,0.26426316531354443,
 0.5911079984555643,0.017691758494883025,0.4397289571082156,
 0.7570412979766362,0.3775823133273025,0.6361766793648878,0.6670318781988545,
 0.5314225027835107,0.03192846288429796,0.22651278211799175,0.877892965188197,
 0.8726194463387068,0.9123648316112792,0.2768066672235937,0.595341809201966,
 0.2740653058948471,0.03185435785033153,0.37646445900707837,
 0.5349924328326392,0.6741608248476372,0.03616114080831212,0.6092361546918373,
 0.7022250431043517,0.13743850662345802,0.2135675106927023,0.4356453020996648,
 0.6779143434744073,0.3782823870773727,0.32273846640238,0.7679845132019063,
 0.6855852498357049,0.8944193210050789,0.1773360908568562,0.5969208231996094,
 0.7674196813119952,0.8612327554665106,0.9633454231273411,0.2748163940909649,
 0.7697372046826931,0.9670235279831384,0.5770775718718574,
 0.010142738577649468,0.5387892927515198,0.3643238038124109,
 0.8953932201440729,0.44928333323771896,0.3011190797743677,0.7239069761404414,
 0.6301102140745577,0.4640811311441474,0.05919057243840842,
 0.12207872677295406,0.6287918643796546,0.44509409540335276,
 0.36430493010974174,0.18413443348974612,0.9930757471720493,
 0.46421101446686597,0.7797238971278565,0.8621170751071993,0.2395193350246726,
 0.466691721139358,0.9103540418392103,0.23286919729705724,0.932498907871863,
 0.1960346011056231,0.21061446705496956,0.6886362994062323,
 0.20500350033901027,0.5981142711433677,0.9277983913165752,0.8131556221308534,
 0.11766656714939816,0.1318576602081727,0.14621162829439682,
 0.11527930120149432,0.530637493708844,0.38716292917046957,0.4789674105160763,
 0.24420240777032576,0.7784796329281315,0.9715516764866958,
 0.36349060097611163,0.8969167722152707,0.8138422831128489,0.2062317416619549,
 0.7694109773742664,0.922873550903463,0.3527185032846507,0.4126519700291644,
 0.3083532919579509,0.09744427575949932,0.7362834004398989,0.6064814090655164,
 0.7978911118231204,0.8730007860063538,0.03681357993925993,0.8272019251617722,
 0.386988537875818,0.7833710177810391,0.228882004181023,0.09170599980792749,
 0.9587688999271146,0.8280582846971708,0.7515103872127389,0.8655797153742544,
 0.3168758122063825,0.3351959462353694,0.27475771999921017,
 0.10358393694108203,0.03619725665459217,0.9316269468846785,0.818778651141109,
 0.09078932523747041,0.07064145769330277,0.3307245115440781,
 0.1700914199534277,0.2882416257373821,0.5879867883027484,0.3877852266508376,
 0.3313567527596514,0.7282388787526225,0.46483386023131457,0.8613662592701393,
 0.2583608508026922,0.29859328849679767,0.6118891191619168,0.8405205744719131,
 0.3024082278136223,0.13578173724645604,0.31745303362541866,
 0.3650313767195261,0.8416720728328855,0.31250800425274206,0.8454713522747658,
 0.9971870960785074,0.25032613752909416,0.01519416999290557,
 0.18864290501229042,0.9833320575052352,0.44201561665046263,
 0.11021225282366687,0.5700695456390248,0.5527393783717125,0.7402681555288787,
 0.8264798718475758,0.7310631982720819,0.6574595273931094,0.32978422216309555,
 0.04067368967271934,0.04718418623062781,0.6532939410109981,
 0.27077429999318325,0.7755880344103784,0.9733201496239007,0.9094073971025736,
 0.5877791226783844,0.970753419689494,0.018216987962047848,
 0.19746225714231125,0.6778441348163704,0.22735423530880028,
 0.048856521376948625,0.6887645599366226,0.237554389139647,0.2614498261215603,
 0.37637492853414944,0.7601154821055566,0.4010689383320222,0.6000792127324515,
 0.07594190998505668,0.46532155331194924,0.28515870006832833,
 0.09616741396652484,0.05618488126243082,0.31969058992938604,
 0.8684754781653755,0.14934755659666132,0.2203289361028946,0.662795462833963,
 0.342589194977165,0.467192117471113,0.03826683687495802,0.9978827198430674,
 0.7895293996115098,0.478966587496386,0.302027584374843,0.18978613862938998,
 0.13830887853127305,0.934820935278554,0.5016683161725651,0.8379888001964212,
 0.037245619156541565,0.7813917770216804,0.7437868592626928,
 0.5441648052091017,0.08622491840825952,0.8512574003228086,0.9166289643180581,
 0.483066511567411,0.3777704704490774,0.37346613282605257,0.571082997026451,
 0.9673459368689632,0.06894458316427277,0.23929136461337452,
 0.5598853618293599,0.6998892906732472,0.5114689381231587,
 0.026557901673122664,0.5347524622040307,0.3081547643936239,
 0.015435769276428757,0.8176329476266588,0.16538596615669932,
 0.788800452778979,0.8435499155832945,0.46932389768843963,0.25220914424293295,
 0.8138977530616877,0.9459917604366024,0.5737336151981971,0.831983971007256,
 0.7015703756596665,0.3099207624632956,0.08377915446702155,0.3957115217354138,
 0.7587230277572056,0.8588723683564433,0.4362002843605508,0.560017684769747,
 0.827455861906196,0.18768269011986216,0.39624003502810723,
 0.15748282536828861,0.0058733514631624795,0.5643726321890155,
 0.478476121456901,0.6022303105744987,0.17925856541737883,0.4229564787910388,
 0.9745069024142103,0.1938436082623738,0.16348127968108628,0.9820337649814694,
 0.8392368701085365,0.13844452323779588,0.7633504652823117,
 0.26054070662262574,0.4148660713027561,0.7486668568019589,
 0.17729494601538154,0.17400799764158825,0.18171810678205436,
 0.6156756816479004,0.10004143565940704,0.7407021866921751,0.7730193531847235,
 0.3076018046260778,0.8200732607379663,0.6373358111215879,0.3722483098832885,
 0.8393789657843775,0.8629752467912734,0.305608197284047,0.7643129548849076,
 0.4429090661626871,0.7211462476943138,0.8982840558991267,0.01674945908704273,
 0.6536862480895199,0.37373059951131316,0.37807263062163565,
 0.7317259650685073,0.7809586344944642,0.8571279763677611,0.8220997098747821,
 0.031119652454269287,0.020070990732322325,0.3194471360557759,
 0.05506185965774513,0.1236525172850177,0.9545618299318319,
 0.46690313698988173,0.25698014286210336,0.9610196546355358,
 0.3874568768816562,0.045401977962794104,0.7681485336438904,
 0.9659306672301029,0.37779327985867794,0.8860932743409962,0.5169405239635763,
 0.5909254836060864,0.8102188388249656,0.455416619397623,0.5903458146757283,
 0.5440826185945151,0.5948652674751491,0.6193389925229914,0.6987834260988844,
 0.03523086759491867,0.2445412772481309,0.4327673577720367,0.6375166336141491,
 0.35684496531337306,0.15730861662617712,0.8686860491291636,
 0.46394111871201793,0.3580266410511994,0.5611664887451181,0.7123976197338082,
 0.6712819804935639,0.779695200081259,0.9724466567324861,0.771516349015384,
 0.05374843653745809,0.4166358393102464,0.15749781215480396,
 0.08968217484599061,0.4328470245997075,0.6385186627895689,0.9984964460020409,
 0.7665683295338049,0.21743158978172772,0.5713726885251693,0.9512820205946881,
 0.07754432343812112,0.7573680662544326,0.3680737842911843,0.2891063965091112,
 0.06668799416696714,0.875383796511694,0.2503696906231163,0.28249302817643984,
 0.572422345394765,0.9244277852777384,0.8364729147811466,0.28384227036086385,
 0.25937487664164927,0.4274546275732638,0.3870880299930155,0.4724927062135922,
 0.8850948980252675,0.38454507689734685,0.15730778131590828,
 0.05108747329853314,0.9448166900655603,0.21264307022421192,
 0.31701918632668147,0.6901722479281813,0.7243608042055061,0.558582980112106,
 0.9373698833822872,0.22473689637696803,0.4922819595517075,0.4190292338592455,
 0.20920899749402067,0.7729932674806571,0.9719335802911272,0.3847382206064909,
 0.22928923733834516,0.9486724031452457,0.9615759426353356,0.515688200421343,
 0.7768338209951724,0.22079526939332828,0.13447247876993507,
 0.7643870700681306,0.9047062215422603,0.2856717802185622,
 0.027105250824591698,0.18859277338990188,0.47272527568116207,
 0.23487522969140784,0.9957600443160906,0.7135361153999399,0.5492216574120293,
 0.240680148341319,0.5572667411597647,0.5253535472262646,0.5816220875535566,
 0.37227875379961517,0.7876379623014501,0.031204885802302185,
 0.8050541868865623,0.4291388692055135,0.8637403959307876,0.48549398560934753,
 0.9615345361245718,0.859969889553833,0.26458718630827593,0.30094889043201967,
 0.11483624228971157,0.9521592006178067,0.2054109974109355,0.7709353386724744,
 0.7461814670064821,0.4663212620226148,0.4763671390155414,0.21742226093636852,
 0.13973043369184301,0.6725726651403687,0.07357017396464016,
 0.48275805152640505,0.5813982302061522,0.24797851445893793,
 0.7813846486393201,0.14221661259557816,0.24136417876508076,
 0.2002109891110896,0.49016727593759346,0.2778452064031317,
 0.19457807256429438,0.3491888773256847,0.38218848522605875,
 0.9549940624880234,0.9717048669400216,0.12034476806126815,
 0.08615730427819035,0.24334769726620198,0.7284964523837341,
 0.19142444285842164,0.5350055468049097,0.5837981756185618,0.7295283721280964,
 0.41999124131373444,0.07621042411734835,0.6211509925818592,
 0.6062113755230106,0.26278073265417456,0.19339210840225063,
 0.8489809258897852,0.9471293132300513,0.12853929677260356,0.3344589555665173,
 0.23146122897465715,0.7200281723902621,0.7375942371632741,0.8881175326847696,
 0.42814704582091545,0.3707398811869339,0.216238975336018,0.23837392108737576,
 0.47393540309877413,0.6555120210460692,0.7075280938725397,0.6748226870257378,
 0.9245975912774983,0.28548922145871947,0.0288748213619876,0.6332052719019968,
 0.8388304525901324,0.02464830135485796,0.9901385272658668,0.6323058055882025,
 0.8835343941264885,0.4154005725908726,0.45108084665874526,0.5045984322104448,
 0.5577521329137958,0.7558956568384306,0.02681323197409813,0.5844884839269325,
 0.38345407896348904,0.9518036906831304,0.9528638660782389,0.6197093210104874,
 0.5298184762999387,0.22385550926639397,0.859980328454069,0.907703121074487,
 0.9789926596170858,0.03390910519786161,0.1244026228253049,
 0.11637127882626519,0.2737380521144652,0.951589305044545,0.13185577827320005,
 0.46865819953390575,0.252747307552983,0.5289337251419455,0.9475559655285013,
 0.7518316014759677,0.45356698542079854,0.49856308579369213,
 0.49269923195047904,0.07310747665829342,0.05502614051775545,
 0.5705862016635596,0.4920183617780901,0.4561217792510248,0.5824568089653174,
 0.04397102917125084,0.0026421432870232797,0.311327976175641,
 0.35884765914811745,0.22148416045886543,0.31401876987538746,
 0.45469213422558397,0.22598834781091515,0.508118249075337,0.9899102697753464,
 0.30131203486996216,0.9692388915181548,0.7985875649980483,0.889993212523197,
 0.46269249648336763,0.510781091239705,0.4606350115105783,0.1392328428503289,
 0.2752819087087478,0.7017714699172755,0.055899862557731717,
 0.5408597407869726,0.7021437275900243,0.0019557728665539065,
 0.477164837285999,0.06888059466560115,0.911385740668767,0.14356497830725845,
 0.635458319830524,0.31222185229523247,0.5592212016871735,0.6520566466761588,
 0.3569589039110904,0.32876433369824276,0.9988232670854504,0.5747484775689,
 0.8273606949427831,0.6855601899963771,0.5711296397239065,0.8480884855345735,
 0.06980520353282893,0.16808316428811942,0.5817776115119724,
 0.7791620800825529,0.2569368919053039,0.6295717466189346,0.5649410848139454,
 0.7200672739003795,0.05677267487674498,0.03122895600294684,
 0.024192843479205584,0.6721595281604287,0.3273045869255181,
 0.20256221476372316,0.5991661539628248,0.6321413394043097,
 0.24421305379942393,0.8777046184777069,0.3003007306260469,
 0.13206552424454165,0.5723811757144556,0.9977293386845103,0.4190883502507413,
 0.2640471544404881,0.757488071154678,0.23579037482703746,0.38956841594704184,
 0.10221163167551417,0.2746280755887006,0.27538636867905086,
 0.6434659437589569,0.9334270675749022,0.3309971247065777,0.09316958959320121,
 0.1275738657012514,0.8074401470812508,0.28930078485367317,0.0612754216812782,
 0.6008086726900178,0.6735649467936372,0.9311301483305477,0.08040126672543679,
 0.900251809510872,0.395236828872633,0.6165264694617786,0.6822724051527947,
 0.6068053153028141,0.027479258797889372,0.5601795834767612,
 0.26397367491271495,0.7424173688023118,0.8184497870085488,0.31491412114492,
 0.06471176888735697,0.3712645231043624,0.3976179453558052,
 0.33526703357312493,0.14266908721858396,0.47805977181903014,
 0.041374032701516295,0.7823222279418407,0.16615746030061485,
 0.9671442672129735,0.9762055989648175,0.7751708156402647,0.4197960997219863,
 0.21759014401574572,0.5736695448211884,0.4935593963516385,0.7690688318747332,
 0.3323648982847939,0.4578400420004236,0.47546151030572503,0.867331507860872,
 0.7389901153705589,0.8113869851341071,0.6648912574971944,0.5707057805821518,
 0.9698365927428777,0.11885071743642128,0.3071633782836951,0.6178226149182954,
 0.16240370783779745,0.2636729481056638,0.5463867464111821,0.9028482196730931,
 0.8057807023677902,0.09324970906030805,0.18575866242907912,
 0.6237233024965487,0.44438563195051317,0.8041742907887637,0.6945770270923544,
 0.31895128062164846,0.17068696018932772,0.6914255348796193,
 0.9757158182480141,0.3507480007970615,0.03279833747980998,0.7613525338017969,
 0.16377342485979773,0.4449321018769693,0.845424368501244,0.176315303680169,
 0.15154599198641072,0.6124969879945604,0.010591645030471719,
 0.7346065634774468,0.8137671815364957,0.8234976616134946,0.7796126966800342,
 0.6650581098873795,0.2853388077066169,0.585900699577364,0.43584695484239866,
 0.047370159927473954,0.5334825246345734,0.10563080858045293,
 0.3855341918983144,0.4087767754075724,0.9813967871664679,0.43872317215940104,
 0.5739666205274148,0.7651814728705828,0.41530338630341024,0.5455824933153264,
 0.7189472544196674,0.8570116608349926,0.9864204655347489,0.42640135440868177,
 0.8276445546703158,0.115298086106977,0.38180348574702183,0.3211728063784579,
 0.0507543577254973,0.29271631976622414,0.31990444254930805,
 0.10122444386415808,0.21881078932711684,0.8859742955516599,
 0.5846227919877796,0.84323561334968,0.4954101936076657,0.24890062778993327,
 0.07096726013956434,0.3530775474674195,0.47320367120508666,0.152989559684499,
 0.5966278261104119,0.4797280186744718,0.24317451764652487,0.6335988962505429,
 0.042276667477992524,0.9928409279759757,0.48702732268406945,
 0.006228379031011144,0.9251638929167998,0.2913496452181883,
 0.5507772900712877,0.6460926246895344,0.2950501731727063,0.08435179202632215,
 0.8687462087677577,0.7349412057570573,0.8383976310670018,0.513641158186451,
 0.9923767988652126,0.806787079102118,0.9650943811449292,0.5530476005485969,
 0.04782989685206118,0.9536091685126198,0.20443876999768174,
 0.4021464694689565,0.8218922979477794,0.2909313714791132,0.6638596272233316,
 0.23723912167792094,0.09585323553945713,0.8478158109626398,
 0.5906598493194324,0.3396903594817354,0.5236503181008577,0.3212225810707705,
 0.0638030488234147,0.4243920285206113,0.8944723291240168,0.7796926398242939,
 0.9532769157690344,0.70878530088474,0.7625516167614249,0.9788051582350602,
 0.6115397675415152,0.4943722163272012,0.1656865923100186,0.09720384622037137,
 0.149799272964525,0.45602824906820205,0.486995689542449,0.588392494240631,
 0.4543201593482066,0.2784816096599134,0.5949394826250636,0.5163751497733523,
 0.518359028185517,0.6338591600840829,0.3014132512144967,0.7148981173643411,
 0.07955287880359496,0.9515406384063506,0.10858144981244588,0.680835568860436,
 0.3487594939415706,0.27634315220705297,0.6358103335879448,0.8524471024693426,
 0.3586994194678982,0.8288519777833181,0.7413790112144583,0.7431924805191463,
 0.09264697075906636,0.5260822539631013,0.8297381172114878,0.901110333957049,
 0.3750479470517639,0.6886448128150137,0.6529232260890083,
 0.0018868201811931273,0.1261011918189624,0.23962883727747797,
 0.023405794310624728,0.45452728487372474,0.8947120907862829,
 0.3476190005229616,0.46981413703314223,0.7317206541845931,
 0.19721551394146752,0.6452457832254627,0.6235171061773492,0.8787411263935191,
 0.157958905362235,0.6612476333883462,0.4873984230000591,0.8686487706827396,
 0.8058638895365183,0.5765827292501102,0.939809341305232,0.6626910353121783,
 0.2943996927883994,0.9655300962466723,0.7841144299474956,0.5159808260586856,
 0.9540788565211015,0.13386529623863586,0.07142060502454461,
 0.6055997687629389,0.3020748650356164,0.4781251758955667,0.20393939339954537,
 0.929975452062013,0.21646620814761686,0.21649507986216587,0.6072705059928907,
 0.06297449022942603,0.37320106248097906,0.2316435690630797,
 0.5673907711946367,0.6006828606473985,0.05112725236638804,
 0.020768261028182633,9.94053707468856e-4,0.6244010574270137,
 0.14578491355595646,0.536768414951264,0.757544693488143,0.6342642011902204,
 0.6782348555192259,0.18551663054072753,0.9445592959217064,0.748457685227941,
 0.7323963090818224,0.35737277498476927,0.9196115893328662,0.2498707918897094,
 0.5930490118831917,0.0351982827196583,0.36286281057427705,0.9912110474766476,
 0.0030169604380727577,0.3385955343899869,0.07807787322849091,
 0.44907992014339915,0.8255932084239621,0.20612220726064168,
 0.6781232762543777,0.03671555110965208,0.10962010753611906,
 0.17241126265096962,0.3892841223719813,0.8844725232971709,
 0.003467647739658908,0.20560727647539478,0.6022575898992675,
 0.6715159704569288,0.8477526260161963,0.9154003956830503,0.9375540548759855,
 0.20141602224316746,0.3952240266543666,0.21857856367729767,
 0.6490937831411787,0.9023833871095334,0.9911007413984475,0.8116074578906841,
 0.7394536279519529,0.285468298052313,0.9214788418659381,0.750382800366806,
 0.3428018589569616,0.9281158328057499,0.8373976162981219,0.6986129303734261,
 0.09781663665706164,0.14359678887680793,0.7679713484883304,
 0.5112687892025679,0.606326632441829,0.8536547318295957,0.10110809507619423,
 0.8676141385831574,0.6932152498980273,0.7391687087626958,
 0.055915891648776084,0.1606579845143773,0.6228876965955136,
 0.15607783261260455,0.5779942788242578,0.6808735751632727,
 0.16353645903213154,0.5481649808114368,0.44747313453113646,
 0.10033519267868629,0.10665549718591016,0.8459762170314429,
 0.9865584210127847,0.16581028803392472,0.9668692661455909,0.2649661894579649,
 0.8046901980780264,0.26432244907459335,0.7063300750299039,0.5471410061549109,
 0.18510965965331794,0.8588605122815733,0.27364246803438375,
 0.14779237316170324,0.8501656067598508,0.115185179985141,
 0.013613499451959887,0.8729288906742408,0.8871748719034388,
 0.22037405214283745,0.5634353669994517,0.6845572955452786,
 0.05051706325517702,0.6458274185085086,0.6637348744702057,
 0.05141618788613256,0.4834459159493607,0.15323853380243435,
 0.19513931440766985,0.4532794350787699,0.8278433924435309,
 0.05515578755341355,0.7902725241127535,0.8109409157228799,0.3489173976069907,
 0.34599920471313794,0.7355986831770749,0.686108879431305,0.4779756864625069,
 0.37315886918157526,0.1940643904806727,0.5225724387948694,
 0.05856966159288124,0.6539692961037067,0.6033600147971789,0.8488145308102437,
 0.8049098223368509,0.4493571588647747,0.31544612742641165,
 0.08776372944773025,0.7569813869161343,0.5908365047976087,0.6436045012586948,
 0.103998193598274,0.029923769746480033,0.49093007681808254,
 0.2912092577923101,0.9484281626214606,0.5021108536416353,0.21022301638849272,
 0.26498087397418957,0.5634253776676318,0.22672949383535412,
 0.8640771890205218,0.3960593230248095,0.6885769436925127,0.8036828809739355,
 0.4544167951899575,0.3382680837467278,0.9812182706363769,0.6222797462688938,
 0.3379465902131278,0.798843912660186,0.21107572549914044,0.40148629574259875,
 0.9137096479932836,0.7957149947903427,0.5212181376518503,0.211081791665379,
 0.5291506663003157,0.43036114664625547,0.025619562237734095,
 0.050472751188047615,0.1186935714887064,0.30160746155756646,
 0.6721155367631197,0.17562173028751893,0.22663863685283658,
 0.7165964567455767,0.9987546758798767,0.7430767550350268,0.5727281147416832,
 0.7376686422469576,0.25727966369870425,0.5398505957175307,
 0.04106983753600035,0.8226657138296488,0.3641268647174484,
 0.14621275469148054,0.9635794811634867,0.9634804930246444,0.3681740199314171,
 0.5555852819622458,0.08524304613882316,0.7408972413507147,0.3815729552606968,
 0.051059776438092674,0.17390412109478992,0.9068713906645455,
 0.45694950717418625,0.9493116847627263,0.7995638365867842,0.6103469410626869,
 0.5924977724075032,0.9634105652375011,0.5489961111420579,0.379324605453929,
 0.8208776841923914,0.49052106308839394,0.7419078020903264,0.4412364327488788,
 0.823828648697139,0.806466251044994,0.1326456790246201,0.721379603579134,
 0.3469041473588259,0.010760467562469955,0.5043499044987514,
 7.525345640151393e-4,0.16001077485885706,0.7733301490366833,
 0.4583942067365989,0.9466104984871719,0.2551260426149913,0.688103499729111,
 0.2706557615444608,0.11949540081340548,0.6942990943527723,0.8975468504913916,
 0.9940652271261702,0.6265873962700157,0.28385583216491495,0.7398646963568634,
 0.7507124756314899,0.7749856978968037,0.3856469761134701,0.4936916507604121,
 0.5641295501483776,0.6643824974595858,0.8109985396746092,0.12348576250209964,
 0.9987268533625112,0.08576926171116994,0.02121950680274276,
 0.15484571740770892,0.0307785570146909,0.05807318830436148,0.501918686212289,
 0.6993226441897976,0.004355587644476744,0.5965381781331631,
 0.2895356881020974,0.6710703222536237,0.8331707400254326,0.9607194410829106,
 0.4104956094539369,0.6755239336954146,0.20733427159461215,0.8665655488799919,
 0.6100664442873068,0.41083626478081325,0.3168951492730674,0.5855309206580754,
 0.8995129739008008,0.8945739639072353,0.29341609098752774,0.9997293989675918,
 0.9540508488001961,0.3016175830070318,0.9850533798192905,0.5231233676218567,
 0.7075156459328605,0.7843468658358385,0.9010180516224525,0.12174267093776459,
 0.7461072883509304,0.18147974360922237,0.5454123053779415,
 0.030585542276174715,0.7442866481559038,0.17080272153333287,
 0.07296641618903532,0.4792339575722726,0.841621460737098,0.795464968060157,
 0.7198731961369452,0.5435531321178169,0.7319735173574093,0.2989947016603701,
 0.4550953451502242,0.6008768915023704,0.04200418442640452,0.5843949827610233,
 0.42129218467180296,0.5764853929712875,0.3871724430428479,0.8612642197301007,
 0.37683099444392565,0.5976253632739572,0.7032561982924292,0.71424448558794,
 0.6537348026012515,0.018740942661213178,0.39732711570898516,
 0.07019099635190507,0.014166376672794856,0.4686893637871966,
 0.3532105284331025,0.5391557982127245,0.7494139280974715,0.6219477874303655,
 0.17445174332679392,0.8433112523490784,0.6630826963685711,0.7059319323237369,
 0.6841625386981861,0.4398433028987857,0.5593715719537324,0.46383734983986746,
 0.9859523239494771,0.7604385303652919,0.7090171464033427,0.8879698506472081,
 0.769115494691775,0.6143209861699195,0.42617905822598945,0.4937125429964154,
 0.5675403387867627,0.5374389424660906,0.6081530221637509,0.17387858733188066,
 0.8033314535881187,0.17467128589050618,0.8316168205289641,0.0831828280046153,
 0.9051293935769665,0.3588748490531225,0.9021456393118439,0.9534608121303281,
 0.2860222681531964,0.34367982572381717,0.6097583103246945,0.5742652946387965,
 0.7258950221689326,0.4827641844352524,0.992138963511122,0.017621442199686532,
 0.8878304429999541,0.4873820870346174,0.9915863686553426,0.6359549850517575,
 0.49584117086053725,0.7956727865959854,0.1414946724042454,0.5736958026104827,
 0.9691314416360928,0.7632523035328421,0.48100625938731534,
 0.07332250651907013,0.43044434338441206,0.9671808020418259,
 0.1535472073801133,0.379105676813568,0.04108897392308686,
 2.920664261868433e-4,0.4929902932590259,0.7707097143823393,
 0.7847927730610124,0.18890689834720087,0.453010234612246,0.5510738871331069,
 0.08806089135308326,0.1894888467433028,0.4635098666745239,0.9728537236118414,
 0.4407604747141468,0.7453880774205458,0.3731933188500123,
 0.015652666820231254,0.8918925510985632,0.8783504989692895,
 0.8480117363620363,0.8610620025892506,0.055680500381397824,
 0.06467230565254611,0.65672644376092,0.5385822357769781,0.3983588119788345,
 0.8697710477381027,0.6180070411256926,0.18005094436239266,0.9884419420573,
 0.4247293915346386,0.468558846071236,0.6176737442727543,0.24519455724986394,
 0.3552644336852676,0.25132267377144335,0.8835889893473201,0.2718752786433396,
 0.13191769437468537,0.8754288391017506,0.4437988749776437,
 0.06285055175179699,0.42078006250586664,0.682677213940549,0.7852694492618408,
 0.9387403537307311,0.5859238802285657,0.3038245743043525,0.980372693654247,
 0.6735189112491416,0.9043582665675116,0.6941870400809352,0.10170923323449221,
 0.5343424245981768,0.35508060369373584,0.05545621846021276,
 0.24167530046962438,0.3772412211439723,0.21222439391712933,
 0.3723450190061891,0.7881119182916032,0.17473815605719034,
 0.22696937896087865,0.5943813740892386,0.3159203677067438,
 0.45959136589614036,0.8935675651921827,0.6368549627271737,0.7157115530547021,
 0.008488211998451622,0.8094610158206532,0.393497045060724,
 0.03943849121919685,0.40848740361886793,0.1422842910265134,
 0.33109472022270536,0.3753425046648531,0.5143002741931523,0.8551669306648206,
 0.9444021997933366,0.7279000028615066,0.6960467755282864,0.4210901914069065,
 0.08498066672518123,0.786388816547995,0.02689797081810652,0.3933525260175228,
 0.8344604929287476,0.7677509267012419,8.429991214800037e-4,0.435490508111929,
 0.9863022578397176,0.5063787566925213,0.20726384140607912,0.7564853424281196,
 0.006169254602819141,0.9000021326000632,0.9263973892048618,
 0.22767578710886127,0.7655049041992315,0.03500373644347532,
 0.8641310084793347,0.9935321244768598,0.10776473093784078,
 8.512282964486806e-4,0.8349018696640352,0.8190504691171808,
 0.4566437182079218,0.7715156409924202,0.4356322838595511,0.01279735836747875,
 0.8515713427721363,0.1497781176809826,0.6024455786898839,0.6705845542585263,
 0.6595872869111178,0.6022518638962839,0.23710727247456764,
 0.10328033998209485,0.889608865508509,0.31812009239775074,0.8578365773956439,
 0.6186386873149928,0.4775369026318891,0.35752745143026954,
 0.22829364542673647,0.746749753747096,0.5664214434930293,0.17556087138287935,
 0.17635162516719127,0.09871349589929368,0.7930884921772832,
 0.052981568283753555,0.10583298854002532,0.6014094454027349,0.21379450744007,
 0.9186791146804473,0.41177765951428924,0.1395811015671291,0.4363768662505496,
 0.04427958416046396,0.48303628663510256,0.2901692562305086,
 0.9181088867643079,0.053466593939544005,0.8040249290832288,
 0.8889209312485242,0.5720780589424596,0.8147985672922493,0.3471309679509865,
 0.08527642011148373,0.18007064892356106,0.9277381454937957,
 0.7580620176076848,0.8287657114276108,0.3815265502313264,0.7497418445195475,
 0.5200617703507956,0.39585566625516144,0.5007734364645483,0.26285652997832,
 0.020548341845790907,0.011295360286374678,0.39448182172485813,
 0.3421210350356044,0.22755160725336343,0.11368585358907135,
 0.33597352963012284,0.17809192245197047,0.5117116921496607,
 0.3872085058024578,0.9567724511100578,0.02115973909958946,0.9781026329865204,
 0.14253518972980983,0.8388811972126193,0.04724105055140981,
 0.2849223286536442,0.7316491420532323,0.11838896434804247,0.5775033405622301,
 0.6818241125158087,0.13964075159162048,0.18975770775253864,
 0.7189179746055938,0.8470332595423777,0.6273731411037554,0.5518775493561305,
 0.9029062727947822,0.5960973208893239,0.9493079168335312,0.38190971751266534,
 0.18832215814014075,0.07912106371860428,0.9191732084792708,
 0.21507615326758733,0.9474896190594143,0.9530616249723565,
 0.12574415798202454,0.48215893265417975,0.6911877866835384,
 0.7419278617319175,0.47592043255616445,0.7233190699111015,0.6879998938580374,
 0.10522474487593447,0.5897217403801615,0.3000929055620465,0.8819114367628147,
 0.018792133076280026,0.6813721174384181,0.7257916822104032,
 0.6457028819341669,0.057276455143848826,0.3654768472606331,
 0.0024558119069859607,0.9246643571640947,0.6358179920747714,
 0.08723068537891843,0.9838363913332069,0.40379929795249003,0.531420250385664,
 0.43640010449146693,0.8126552618199121,0.6700561400427962,
 0.16418001615829136,0.4077454702823753,0.0010146086114249941,
 0.03648120507132968,0.6977405506160161,0.6976106257356571,0.9131565867923694,
 0.03815051719577145,0.9987872279082263,0.8152345891051509,0.9330921246679796,
 0.5795637368084495,0.9469578816541832,0.779062993818723,0.34403433281949547,
 0.5168601669293446,0.5246967157370633,0.47876221868599333,0.6827814143926125,
 0.25862211810131663,0.9295818173499949,0.37539259653572343,
 0.8883301159749187,0.1796477362989095,0.513560788369691,0.46334811538953924,
 0.12702103501971662,0.9093530678002574,0.48184287406679316,
 0.7995944883166151,0.41509454880857677,0.09342892445946227,
 0.6518364600678122,0.8998970272170104,0.041574769149342106,
 0.4666666255533225,0.37434583132178756,0.40248591247556964,
 0.9972704694578859,0.4657407136401145,0.019329959898991333,
 0.48964090783007386,0.6224953903367471,0.23751991200068723,0.213635261370822,
 0.694410844704603,0.48224781983347187,0.04446104490154257,0.5301130552596461,
 0.8975015687177504,0.19347464942555292,0.3723663850493615,0.2566548503287439,
 0.4508112907508701,0.459750330040674,0.5422315859884818,0.39053927201002536,
 0.08877495991930717,0.7419324644992674,0.9113615201484268,0.1061356152469749,
 0.7670685414353696,0.41502432600449624,0.5611912572450383,0.6199649462711897,
 0.7504155539281793,0.7690059169676007,0.8143009096367982,0.7054267461237046,
 0.7235650600358827,0.8324962219493797,0.9740530145488449,0.28328034133251945,
 0.9158491053305233,0.18463331394541072,0.2919693998849888,
 0.33283485331486484,0.5815334458819587,0.6974846927809027,0.9661553190432514,
 0.09962887728054104,0.43514973768547394,0.9367695121640605,
 0.48510072640258495,0.3450505866563862,0.11079997083409043,
 0.8022102413811222,0.3204433026589151,0.1372081485915677,0.06102647000231143,
 0.38044382381046693,0.013793586902258848,0.22927363363419828,
 0.3560013626921469,0.03893645742388019,0.5142748303558826,0.7195736221966792,
 0.6686820296331873,0.020052158871865733,0.8486273542273313,
 0.17716597200615092,0.13798287170800505,0.803907959027804,
 0.07001598180298785,0.35241540914221803,0.26390182733724565,
 0.16723722107626104,0.968310601706549,0.9878736576302678,0.5768446968414367,
 0.6242081986875458,0.6812171255545745,0.5790783434267662,0.5033114679557409,
 0.6795513674394913,0.25503013791406803,0.09865251190606317,
 0.9609297711065476,0.5739422677247988,0.7253892560533214,0.7426617255042343,
 0.6528492663175403,0.5255648651579299,0.11410718756268656,0.1528369639018562,
 0.8339596462568182,0.9759693358508277,0.434814160225396,0.3470904716858588,
 0.040941752127884645,0.838959600452702,0.21905296073523184,
 0.036601993330612315,0.1426057221214564,0.7550141049576864,
 0.49162285659453064,0.10631309694873847,0.6144135182301561,
 0.8236012536089881,0.9604750648468062,0.9613611637323352,0.8258541972549446,
 0.978617015329941,0.47056795103977633,0.6113880587112385,0.5808649386315724,
 0.7551943696300535,0.16129494325519678,0.9227332879736766,0.4098286612218389,
 0.1704859366692576,0.5938654791098591,0.8074264386344456,0.9960757215849649,
 0.6188279155704172,0.7259423451760653,0.5219820989242233,0.6751185017482823,
 0.7585341028407585,0.15340024870828517,0.6218438382927364,0.6638285105830418,
 0.7602130058069642,0.3201403375789972,0.9506657816089811,0.9113351454076455,
 0.7292082237912826,0.2571318592532287,0.5050229387174838,0.1776473488229413,
 0.7207360933443279,0.762772949689119,0.4414768097821632,0.9370979253919172,
 0.9309617695292679,0.6206054463667002,0.6026330479267391,0.37479384702341967,
 0.2835552367668728,0.023808306996965345,0.8260843563347893,0.291638135319239,
 0.6769661541965754,0.5325983455429751,0.44185729774084725,
 0.41717618390886924,0.4288838273649417,0.06782742460419511,
 0.5991186410784612,0.3760471238339448,0.3252315676237799,0.5145379997654018,
 0.5832980315304753,0.40154345992290486,0.30783110778897504,
 0.04625606557797546,0.025886885478120902,0.08037880031459421,
 0.5281206780351962,0.6730149507683237,0.6438937520345729,0.4737862545298931,
 0.00914740011854398,0.15821352390014798,0.8187307358436705,
 0.34953504408354497,0.2584217714031021,0.844894655897748,0.38615558477633793,
 0.04526173243374232,0.7379842143827997,0.7792674978556235,0.4476033004715281,
 0.975633867658629,0.9663254730096563,0.21135730928114327,0.8168389833998378,
 0.6686407852434257,0.3479816458590854,0.6835955577328057,0.6039238644393805,
 0.6264566443269334,0.17227455045802798,0.8518791430173275,0.6457826638354178,
 0.8087583145884226,0.07297266020057824,0.7587031805749251,0.8220954216043141,
 0.7070255594967559,0.6067201765865586,0.2766654048669419,0.7169544441863591,
 0.6070936675266119,0.5311600542178823,0.7313716032168975,0.7131282150682523,
 0.03083522256925686,0.9170384115938454,0.254515874416112,0.09036349958634249,
 0.6850793481948678,0.8604524762941329,0.7526854215329832,0.7250398450922778,
 0.7301561146718515,0.06777511931422608,0.31969201489045374,
 0.5512617777260067,0.5243419595253687,0.7921341216688057,0.8131684254193842,
 0.9016874014099894,0.9597466933603229,0.025600107494485824,
 0.7999430038342167,0.5829340405435217,0.6574327271204577,0.06420004871793905,
 0.6057671158000745,0.7957641691175071,0.47715112608003196,0.7877847667312861,
 0.20000863663354274,0.22346527151181672,0.5363989603742038,
 0.6989425374186939,0.9355903690676451,0.18836944596115102,0.8042207270173041,
 0.14368893250446457,0.5208620550557922,0.49936180843975786,
 0.5245832838877759,0.7687522348256486,0.9544112203065394,0.7957870020172794,
 0.5712208212601881,0.34632771656948635,0.7257026952349337,0.9716026853085048,
 0.9558884581017879,0.019825573045095246,0.6041653711958561,
 0.6786904753401971,0.5441764643111152,0.9088875751899845,
 0.024994735865109696,0.2391238920165637,0.33545335326937165,
 0.02782908959413144,0.9948891171206676,0.35676518843265925,
 0.4558161239539862,0.02038230500718119,0.4713585264753858,
 0.32469497098566646,0.5488380087877213,0.34960992864807416,
 0.5098895282992439,0.01644027185159358,0.342860788808916,0.8407278320008614,
 0.4489599953353627,0.9640545345904742,0.23193816846964954,
 0.39464465264188686,0.9973610236504163,0.7528010785966818,
 0.10370272914310519,0.9215582899061239,0.5997561893416254,0.9854952965731901,
 0.7204171764799409,0.99550223514225,0.04925847673895378,0.5736309699877404,
 0.3034438150012502,0.291529166749379,0.4671666566166349,0.6908312084793077,
 0.5608258088518278,0.7108169586066981,0.6050976235314944,
 0.0032151438978981206,0.8935783041813422,0.7737805073781577,
 0.14771664363056702,0.567675838242837,0.5903410582221648,0.20566172960618256,
 0.9228774556385145,0.8487085820298406,0.9127396098349877,0.9362455550263253,
 0.9241826214541493,0.8722961557884295,0.8400821252724612,0.7628788828524932,
 0.5525450408440619,0.39062495068212455,0.3173681182163599,0.7199802609513716,
 0.9209334445016206,0.9763538034129741,0.7992343291121753,0.9000649783061865,
 0.154230443327513,0.019667245587794646,0.4256861031834611,0.510362571891485,
 0.6412702210289545,0.8064978388695978,0.6894830442226446,0.32112157117493534,
 0.06294967459279999,0.9152630406447976,0.6672669373412644,0.3991484460009087,
 0.6231294618901773,0.15926403126089284,0.9103097647027452,0.512309810931601,
 0.4118141417549366,0.8315765336488192,0.2541334149285789,0.9455193984573614,
 0.2685756502171476,0.49320532190510624,0.10560737052579361,
 0.46763087720903473,0.18266859817111325,0.8532541505719812,
 0.14433261730472124,0.27038601002651297,0.7052173199978622,
 0.16713132879049875,0.45512203029724263,0.6516338154431807,
 0.22144550691571063,0.23319337004285612,0.34259058777605855,
 0.06232945282024793,0.20791690181702882,0.9584233750482634,
 0.3544331450496436,0.7209692583692587,0.7448628356050275,0.5097350377339781,
 0.9883829791325452,0.7815492785085819,0.3939151331364814,0.0470935508158149,
 0.09044024989314114,0.7546309589889639,0.08317912579772013,
 0.34539093246776176,0.09939566962710311,0.9629366911739325,
 0.3785630238638231,0.9929873702518154,0.40799107837339377,
 0.14357543875435785,0.6031254543233688,0.11525781770221988,
 0.016877483841810093,0.7774779734131906,0.8565894733955206,0.612515795031164,
 0.9534675301585925,0.724956157594881,0.9997441073181189,0.5853250706087559,
 0.027248583173835716,0.5177846316740661,0.8952827048247685,
 0.5470464176046346,0.3571112192548581,0.18016509884461973,0.9196041032582006,
 0.032798615484347815,0.7069585217283894,0.5532457850683499,
 0.5917656227123884,0.9220907795476233,0.07256908212211666,
 0.11015729304106348,0.9956515508562779,0.845362975690883,0.23906295741316042,
 0.8317205479579415,0.6779112180674098,0.010829223473371341,
 0.4466747014386716,0.5814849759171203,0.21660706532210583,0.6339474333496202,
 0.41130839623677784,0.972903136783346,0.7727901267338961,0.6343515009309364,
 0.804151571930018,0.027777027369294216,0.23089283712403308,
 0.6438518792738966,0.818355289674798,0.38016193701891776,0.2314517831628936,
 0.9366142671799157,0.15312432412565968,0.7105607058691008,0.5575233267277645,
 0.7032567282560928,0.23640397753311737,0.3480324239233161,
 0.016270789971403965,0.046735079850371175,0.7502122545998315,
 0.45457121621892027,0.004941995490791995,0.9916456765483307,
 0.8117464223572796,0.00466046210327109,0.5083564306241314,0.9278518054857294,
 0.1843104588574307,0.3779065971916704,0.3399877080310949,0.1284323187379276,
 0.24485672212981546,0.40127314517602464,0.38074191353721165,
 0.7834343200137324,0.21220480498386285,0.614513217003862,0.9581914231227916,
 0.27228058945889644,0.9454348784209357,0.030756618077907638,
 0.7372691592788352,0.13325078731755757,0.07680431775659757,
 0.07492391264028309,0.20005247745827182,0.9184146561772689,
 0.5457371069944377,0.3153781635019943,0.1948431270985017,0.06119576914717406,
 0.9770124254812357,0.10517292425434821,0.21055081350903226,
 0.2885453508545772,0.5022123668723274,0.7039582565000567,0.8423605264769483,
 0.0074775867470103785,0.007409517975296542,0.9925303800081756,
 0.9151981301123753,0.18523393457873194,0.7783975231851518,0.0881360802122163,
 0.37633784878183285,0.4533030672351195,0.3729634664212822,
 0.24959386187897303,0.6751384920768769,0.6223046002622433,0.9314940707400056,
 0.5217172746786541,0.08204778504844634,0.1679379152811925,0.2032163657424232,
 0.847972083266893,0.27126198907694354,0.977583798592346,0.6777418102656126,
 0.3200567396585524,0.058359538664811694,0.6545044814638421,
 0.5446271025171745,0.6853945890892901,0.13202231934763997,0.7778661759001853,
 0.9529099878656437,0.21115534854353424,0.352980666039894,0.23351926280749757,
 0.21592726913293714,0.9721810126222349,0.07410260386736622,
 0.12118454909987264,0.46625276367134283,0.6732771596119175,
 0.5201417537661861,0.04015592129864898,0.6976841391813021,
 0.29633264501074574,0.13798841301709963,0.38672263989732125,
 0.021907780246142594,0.4266728871460673,0.2841820455818599,0.834540915072657,
 0.9933298748994545,0.5242525645873817,0.3615496290939245,0.960071243805096,
 0.14300249647150898,0.6507148885599088,0.05731833662443242,
 0.9261038265128251,0.7528621565762424,0.8098868902679714,0.4330544043050336,
 0.03666477732422613,0.4196015768079775,0.4198460392891581,0.282089574390087,
 0.9891606870121845,0.42342634085387687,0.6236171273178529,0.7662078171562846,
 0.5837260197997827,0.9046755682653281,0.5592051971448853,0.5620860630828025,
 0.5400623371981111,0.7497457808491296,0.7177408355706955,0.6373313586534993,
 0.3160843147230865,0.22229238753316127,0.6844447047268658,0.3411916397595144,
 0.9332066355505761,0.7436691377680177,0.5515475250034276,0.8678346489683822,
 0.9035003444691823,0.9785671389164025,0.29426760040192734,0.6662861270322016,
 0.6396428818834713,0.03770224155352053,0.4027685160554346,0.7420329654979363,
 0.7757416124234976,0.8104222390825169,0.9761275872029473,0.8627806795675131,
 0.020738257076408084,0.9734014360173391,0.49117400616401385,
 0.9013021055222725,0.8664712323233266,0.29766950165840966,0.7691269943287481,
 0.10822265112058682,0.5472683783296761,0.8820880407442566,0.5499625230850104,
 0.26956137405557534,0.1274950908314787,0.48971996978739807,
 0.2412744936939486,0.39155268709547886,0.470783702550973,0.2197940101082434,
 0.2806319251703192,0.4293473513848066,0.09187900601330445,
 0.09295481385204074,0.18652220689944832,0.3784173895257299,
 0.012854726389769588,0.9255601340493468,0.08095946367216367,
 0.42375542118171394,0.6000829185208062,0.03499116687045656,
 0.03112017734967376,0.42246074773476694,0.2036104103245694,
 0.9215011080198333,0.8635042236575143,0.9359489825331774,0.6427440036120085,
 0.6729501974913803,0.6140816588322904,0.20726902281297033,0.6059160043079947,
 0.382122132144213,0.6744719472462287,0.6242736947590654,0.9769283398775628,
 0.7200227326675788,0.37425968891338246,0.39793229831613486,
 0.9303596227172981,0.3288750069991321,0.8890784259795868,0.7558581328281109,
 0.39981894529791817,0.13603880927329337,0.8594571669047485,
 0.7418044140779914,0.6241368990914464,0.8998087806931162,0.10260506404223979,
 0.816199925671256,0.6429374282710572,0.31635484459958607,0.7335158558281711,
 0.11687913992636145,0.2464556621478754,0.7089934540562827,0.8526496208895837,
 0.6966968520755423,0.5889363781572805,0.9989769915988836,0.44344122893778093,
 0.6556135958441663,0.9649396988417023,0.21239961902310522,0.2034040450548561,
 0.7759850021614135,0.3409216850102518,0.40701951869406894,
 0.41256569882841165,0.08781975608054149,0.5281896379756544,
 0.39547843500938196,0.8165415355517404,0.6885910948730973,
 0.08284713616864614,0.10134828109172744,0.4360432402343235,
 0.16015308506301384,0.3536365581870582,0.04634858514377316,
 0.6943855584413359,0.2884171100214862,0.8112470957288411,0.3863625923314852,
 0.6617000736868592,0.8160709119183399,0.300594011938633,0.3918629392042009,
 0.9814885643534238,0.2964173610142208,0.022485152567660482,
 0.7328511255456817,0.18094741476535214,0.12059079593295208,
 0.3238772525615059,0.40951887961407096,0.1230215643676289,0.6708162949518065,
 0.9533790412839649,0.0568034612728332,0.112955207618893,0.5657688469407329,
 0.8219399045647682,0.5591176310294451,0.4663109788628039,0.23013459789713242,
 0.2552542706152977,0.2969843076396609,0.8535703201745689,
 0.018127324859936955,0.4048703312019789,0.37229478147719386,
 0.6832357760602952,0.6698961235294625,0.5817544890844186,0.44587921832634503,
 0.6547052010862756,8.649733163273998e-4,0.3964571486218196,
 0.8153330084354782,0.09283330129589051,0.48661203083652727,
 0.2986145283558883,0.39589694367512984,0.23709111024752583,
 0.007611556172853473,0.8711986013974005,0.6755908747866137,
 0.20743127889159596,0.10489297273964304,0.5926136430095368,
 0.3487412742410897,0.6378630481377363,0.4737352626358401,0.8073373206019302,
 0.841550783098928,0.9193424652177868,0.9043985443753672,0.4716707732049914,
 0.8538485555248653,0.9670513032479796,0.7979915374845981,0.8300093304579368,
 0.6086645013604423,0.8861726513887661,0.279924105197471,0.9031343513185343,
 0.7356401832231386,0.2982306618131336,0.6257256122953037,0.9179769106125624,
 0.5172159970121337,0.3851900497722178,0.32632532407207737,0.5998463002557546,
 0.17017275084064254,0.5031439271873663,0.2885468359305433,0.2799400754470449,
 0.3403794780010887,0.13171318165819657,0.35055182569972176,
 0.5397460816525753,0.2078313894471826,0.8115662678433647,0.8867376785548451,
 0.08969090724041417,0.19016120019824967,0.9493912246151717,
 0.3304362314189001,0.5070387206142284,0.19617651275802606,
 0.18172416648199885,0.5688181719615892,0.5283917203189294,0.0671484602023451,
 0.4316997026430822,0.17922289460311913,0.007938323354670818,
 0.3860578794692746,0.3765610694970519,0.3450393778765082,0.7482838088519699,
 0.6592291971664127,0.5205792304676322,0.22313655366414276,0.9849415960208562,
 0.3820867624185873,0.14889748421749072,0.43653803705331795,
 0.3508714976223132,0.7590150550819215,0.6737828890473678,0.3457522258247766,
 0.03001569537327231,0.260905190366465,0.9983892913345255,0.6165707162820767,
 0.37179735967958705,0.1960696496509955,0.9802691480811884,0.3326227913657359,
 0.3100822276807912,0.8328284497006708,0.9706630047099408,0.6884930120440305,
 0.438839846534804,0.5111252544344402,0.5347825794528509,0.716981914138431,
 0.20450651094703376,0.08670306464825428,0.13045055934879446,
 0.8589331539318354,0.602881694806983,0.19117975553453836,0.7221824033793449,
 0.17644167410039513,0.3482535978633041,0.20018910593855443,
 0.25481160060402697,0.056458276012027575,0.13439784731329651,
 0.672242343315165,0.011591372834297897,0.9842064123107367,
 0.06877437114826501,0.5477076770498434,0.8194152890147893,0.2153476436240913,
 0.9911111825285313,0.5157663320538464,0.4263670155163275,0.8769606975920408,
 0.2264623813077028,0.6221707114411923,0.41212770715016267,0.5385121728570478,
 0.43843663179953885,0.7529690162163737,0.5818238230899324,0.9664520594856311,
 0.1840362518969444,0.3341082055658744,0.08076432196940608,
 0.08799608847470286,0.23619754000919746,0.6854600778917385,
 0.5601968317308996,0.9858021563935975,0.45293111227870975,0.1856978328582105,
 0.7431139263413069,0.1540961316653532,0.0023431757806433584,
 0.25760815217078703,0.8157901585415694,0.7689934854770747,0.3575997757929811,
 0.41846489284809574,0.23430318063973865,0.15326142519774077,
 0.3394881773456454,0.7490529924661297,0.22667437723338346,
 0.05617721933534425,0.6575596090166258,0.7507071549142659,0.8390161358164112,
 0.29063404054685515,0.9407594483732383,0.29040204126379354,
 0.3550491079973077,0.13589408737908182,0.15026688371465613,
 0.9140468790509653,0.050791345488381934,0.6198465469640928,
 0.28291625601305814,0.552218553299532,0.33852522217095016,0.744271451989498,
 0.03860745725719428,0.01699630309929323,0.5235361033042365,
 0.7795386730846134,0.05504876833501149,0.8158786683186068,0.7398632698203733,
 0.6279967798720958,0.07720395267355507,0.33126441093899384,
 0.5509724258237259,0.7058606309447716,0.0038444254510754394,
 0.17483316980432406,0.6874203348731668,0.05661990823247265,
 0.35648765903251345,0.11211126705351604,0.9630573834792194,
 0.30012159224408175,0.34286591533485933,0.692990687285003,0.9997838194404254,
 0.8379733928506948,0.7648290121436033,0.23129275381517256,
 0.17119287401630556,0.3820584910225293,0.5144642802948789,0.3671231519532714,
 0.6060701292101008,0.032227979377729254,0.9917597458109961,
 0.18924430212689713,0.9323245783766935,0.6174300354619596,0.7849162714178695,
 0.04040895437316028,0.4725586622349357,0.3183892363912565,
 0.12493304554386331,0.017000076425080612,0.22168250543026413,
 0.9439314717788909,0.11037885661553426,0.16639347258758097,
 0.20931121281674459,0.9719749900151389,0.41423596947509633,
 0.30472949097236846,0.4564313624634292,0.002096640390237381,
 0.6065501132640276,0.05533673049533949,0.3257019652442671,
 0.19553593742418363,0.7595998633848644,0.4140862278276439,0.3196382663542009,
 0.8765908508779148,0.2281961598090696,0.11731521958876712,0.9120823819461915,
 0.7359930086749294,0.6329394420222862,0.6928936967185885,0.9208140350130647,
 0.5607180331802119,0.9945327169805386,0.6770967079450825,0.7211571206531375,
 0.7172723658542137,0.1065947464153072,0.8564831623747675,0.35665792985121814,
 0.6111750457698424,0.006941902214812057,0.315962726064422,0.3013245022401362,
 0.583124244449284,0.05036759490452658,0.4048883186949813,0.643164869321184,
 0.4014695439804826,0.48312764597566327,0.9463297462295215,
 0.19917788952241267,0.22667754892548464,0.8742674323470612,
 0.13637689093114924,0.18332472274849532,0.11553125414415932,
 0.4337109147569159,0.8315079930855563,0.11521607143068968,0.7588704341341694,
 0.33769209994301064,0.4667398568673755,0.3607391660115903,0.7767227466011567,
 0.03454779927322771,0.5689430935072196,0.042777242248162706,
 0.5309971168856022,0.34393402020625274,0.19211728797984962,
 0.9339511850760653,0.9018565182737428,0.23706574482241527,0.2826534760808731,
 0.20362915624106992,0.11290268651844937,0.9455731413199782,
 0.7871668162772908,0.8339820315139113,0.7039346193957822,0.4362431071724213,
 0.8591654112090991,0.6660470785002319,0.19182092699373454,
 0.28163077064607844,0.10669963475678634,0.22071175649674024,
 0.11908599284834898,0.7233692882807387,0.4336844649003355,0.6485063082922821,
 0.14265229543204694,0.7649187480864771,0.8419744428919077,0.8268182311023361,
 0.567746776707968,0.8097815713936996,0.4926421694527997,0.3543747111939981,
 0.5779056726452712,0.08949770937772183,0.4530690921884434,0.7761761900023283,
 0.10314318089367402,0.547737165966974,0.47439655612421594,0.6695375448611103,
 0.7913892372663967,0.5011746761514285,0.04680364197797826,
 0.31871066200734555,0.9971604935946415,0.6492525553479217,
 0.45523305527302693,0.5834013128151025,0.9343987484651937,0.8749233895606258,
 0.3150530652307266,0.8237612524325901,0.8596192385953161,0.9253258359371561,
 0.9857158881733712,0.6163946156438852,0.9213809761032031,0.5340040528636455,
 0.03990559575924901,0.8127940801532734,0.09074167519939136,
 0.04710268616896707,0.18584706481635616,0.428927018784544,0.6022097295884511,
 0.10448513317839703,0.8326039511193819,0.27231595948494913,
 0.31293466131795933,0.863952416427829,0.04205622136244935,0.638116575644706,
 0.4601712170371336,0.6899092932685942,0.5434534311862039,0.12896156169988804,
 0.7616379676821524,0.9943084842905732,0.019839459709631857,
 0.10941857410189082,0.8541654915596034,0.05438333332598477,
 0.6912451629569816,0.675960666525421,0.19591026954383806,0.2988304324095089,
 0.7756171399032276,0.1551782493574001,0.3736206596354166,0.728145874504948,
 0.2503690959974286,0.9464735323102239,0.01810595335142018,0.7114050900091735,
 0.7638501496901232,0.7060431841480836,0.3194610519913451,0.5318798668830277,
 0.8367892683949569,0.5339024214910957,0.13921515958609432,
 0.06554040438447628,0.7258282321762235,0.2005069665310163,0.5510736957824083,
 0.8602209452568581,0.8550724519514056,0.91103432554188,0.9062487061092559,
 0.922511225009721,0.37564428679812056,0.6705831180547435,0.31331876063108455,
 0.19891643290275385,0.16665648893989826,0.3267508968466156,
 0.9087786616716319,0.3098826876525942,0.23385283280301394,
 0.23883390008375727,0.9273363067719653,0.3518277785382776,0.8221333383264001,
 0.5332602687506414,0.926044712650368,0.33821836837695785,0.6512506915899844,
 0.27984479362556236,0.11083482660424326,0.7764130748573187,
 0.6449376892072478,0.645348158412356,0.009604301935521087,0.6355992611123343,
 0.9430553286820071,0.18335103872179415,0.9359935911176023,0.7489829385515346,
 0.7142578269808593,0.24293734544577483,0.6552943479485627,0.1885059620524273,
 0.019766673346484742,0.5227274090001754,0.6902145457431105,
 0.0015729832527069387,0.1734147296199735,0.3134295979593146,
 0.7111163108650027,0.35183885012547156,0.39447628736246454,
 0.8027350890379147,0.7366618507438997,0.812811610642552,0.8733388162703993,
 0.564394715465732,0.45338292374186673,0.19106254885084084,
 0.00376833602758464,0.03350538579566553,0.4782706138717764,
 0.05744403832401479,0.5844224529976595,0.08762806464065376,
 0.13222843124284955,0.2930408794199785,0.8578528520154979,0.3727189901444373,
 0.39645025530438727,0.5899177435752744,0.2653472868988538,0.8959930714651121,
 0.220609744440606,6.579861371647233e-5,0.3580883291720782,
 0.20035874941488596,0.6794764737831595,0.0072585436290872885,
 0.44485808225294854,0.249505707656151,0.08986416899818461,0.5780719177419347,
 0.7948811318231629,0.6624538388460292,0.061704098991303136,
 0.8459507313625779,0.049351268075042865,0.7770300717512976,
 0.3712810395269336,0.2650383434187189,0.34847659558597055,
 0.09938077547029567,0.4722595981923082,0.15647877607472194,
 0.8501294178880712,0.7372282033746157,0.7440630254643963,0.3495815582927827,
 0.026097242960336553,0.6937971857847575,0.10623351908530343,
 0.746567255659414,0.6841331239335433,0.21520771711993913,0.7396818946670755,
 0.22610016334456873,0.8105958468993572,0.6997741535849135,
 0.006739652829522491,0.9982463591513746,0.41248154109140245,
 0.7675798592003605,0.323849758078561,0.5500246703133198,0.950426085691789,
 0.8327441132592158,0.7442905989270325,0.8177640975192357,0.11155830857982951,
 0.09589329586385975,0.6821457761301051,0.2047056013936347,0.1468671905440625,
 0.3846117122767634,0.4828642965389086,0.29287585749947953,
 0.23057247733223518,0.030915015885528474,0.9526019117981177,
 0.9721512612735678,0.7270601503744476,0.5779676640435352,0.5120542456387464,
 0.3765731282378888,0.2889923253737239,0.32588899475930244,0.7748117979304039,
 0.27151941135601365,0.40484454540296366,0.9822252901424263,
 0.36667830478353713,0.9745804396811243,0.37794854755732166,
 0.26216732755683014,0.40642044711135916,0.190033464702277,0.3598624153804374,
 0.39893676211222595,0.7107717999494979,0.41236659909887274,0.348567701407548,
 0.31565623085133654,0.4240220757107295,0.8157108849667567,0.8337277091135253,
 0.6688710007993558,0.6975079322469299,0.13662465820504577,0.6311856774992621,
 0.17646473956794695,0.17138259410612344,0.056304962365085975,
 0.12687033652368318,0.36287214046571425,0.5443781651151649,
 0.6556107928203551,0.17384834239971092,0.05581263750160581,
 0.6177872254992721,0.18996844583434447,0.23603538071545838,
 0.027289213933171963,0.9159311822588824,0.30275859722669063,
 0.36111742755017806,0.13995486232432275,0.8569742847634382,
 0.29926908022020726,0.9746586233507953,0.23873468108219542,0.320351067803044,
 0.08469489396617758,0.16789943284129072,0.7656884776161446,
 0.9176185708260663,0.7070049194525115,0.7340785773372156,0.9315986937753564,
 0.6215124867615285,0.8656065317351289,0.15545003023307613,
 0.43780654197253766,0.649645588496129,0.43235677344939893,0.6041540056332135,
 0.4927230603038818,0.9291562993913459,0.33805042230096993,0.4858549109609698,
 0.06333476799649307,0.6628073788541007,0.2361886352536522,0.4334472653087956,
 0.31065303554851387,0.17493766611928174,0.5333895795840491,
 0.9005597093700183,0.021127435932123295,0.3996440582441292,
 0.08915835428202701,0.472336493062127,0.9967300000382893,0.35707639571113636,
 0.5288108545895454,0.4980544340482187,0.9744611551036975,0.17066034400528696,
 0.42050767749639295,0.902838674073392,0.159457183920251,0.376693145603198,
 0.48452993312632253,0.13841821327775283,0.0442646850422741,
 0.7791465125934954,0.12806750238988696,0.14230450310427933,
 0.6303555293344237,0.2864070244594594,0.3851606820974972,0.23543606120046312,
 0.3083133151040426,0.032125781707642465,0.6303067475029104,
 0.15837740670025124,0.4049547561520781,0.039519456873495784,
 0.2287219212938072,0.9785222532968866,0.3788731258220477,0.3513594812636238,
 0.24639876284259055,0.12641431520700053,0.38192693673655154,
 0.2778607974876679,0.5507022436770137,0.5895323192650807,0.9961198989733577,
 0.7347764205258434,0.45927182178736203,0.0805424321072642,0.7153494115563802,
 0.9626507984799801,0.7545491157608397,0.36927275782351865,
 0.10857411933143923,0.8287501360767129,0.22776834659320666,
 0.10658375469457726,0.9872202209213653,0.24520135163925338,
 0.21057892071006334,0.3251619936674873,0.3621866976974082,
 0.24576355110214332,0.9079472004483843,0.7157368996083482,0.601033573063202,
 0.04126595290547319,0.8744593145759589,0.1730956298696551,0.8091476469116184,
 0.6956324677546164,0.9721331539019268,0.9647981509579777,0.1206958211586956,
 0.7290279122380703,0.45302294746624383,0.9380665569787257,0.668127832948704,
 0.2513107287693439,0.28328474421036354,0.08818378978768326,
 0.5873897621834436,0.23855610447427666,0.7956195526866012,0.8643904115255541,
 0.8327352408958782,0.841365293957653,0.0032305772759257945,0.500083478605955,
 0.0861290426255592,0.465165857976125,0.3536197219140127,0.9242248809868677,
 0.7329651411724175,0.16912988692178377,0.7980207207532275,
 0.11532689372458205,0.03290547628546858,0.6838961754618107,
 0.6493461036528166,0.8547313457734846,0.42145868400298725,0.5321036902230364,
 0.8695604519682669,0.6033605337679268,0.13741899493441523,
 0.11670701994019239,0.9840174536031518,0.809553319560356,0.6659749498743528,
 0.477701983912582,0.38917940188571165,0.23162313727094508,0.5438102551710486,
 0.8791069641778703,0.4905815395612968,0.058947026042658,0.32345689491743923,
 0.17195243274860927,0.7927503930074905,0.014450009049632673,
 0.8114058578546114,0.2255791634812666,0.872398744756585,0.9149296896505179,
 0.4125867450273819,0.7075738659727204,0.7305767812902619,0.8556209681728064,
 0.5004180416194912,0.9714343609040395,0.6265967174990801,0.5776720752472971,
 0.0749393750661832,0.9818821548454155,0.694963184105218,0.08608926945172835,
 0.8218705130391653,0.6554445052619016,0.803714428194674,0.7391088123788248,
 0.6197208444981092,0.8862864568692383,0.7821137186461299,0.31628503128858954,
 0.6936096330114387,0.41952765069818965,0.6458311887655711,
 0.14317509113655436,0.9434020180472942,0.8820281174483793,0.9614979161503961,
 0.7481997597952031,0.0934244594637379,0.4530025918790713,0.9538094698007222,
 0.6719760893861366,0.4344646949562555,0.2598573305198729,0.2935733728626424,
 0.18126319316055517,0.9881269952402265,0.30740066035680325,
 0.3990520659288296,0.39819206426758313,0.9655043145598095,0.7102720235716745,
 0.6915414460334975,0.11260014388988915,0.7856483665272255,0.8323900957355678,
 0.17147704067351632,0.6534001995094061,0.7551258517169875,0.5001205401961452,
 0.6244017674087851,0.855261844822344,0.8909467348507862,0.1344071185362935,
 0.7385351573514314,0.30432072076195016,0.058030160409709874,
 0.6548950543511418,0.8300490371961177,0.7364529896032543,0.09396081732963213,
 0.013013441527101377,0.09873333388792427,0.62517470152893,0.655660019471064,
 0.35818829415921605,0.45272274360528253,0.12208439113931835,
 0.5926572261359283,0.4026890766106803,0.39377994010733763,0.4788404787557943,
 0.5795482131438202,0.17361055962986505,0.3839539089674918,0.9339261224706007,
 0.29827943618096975,0.38839269070033455,0.39850269818488593,
 0.49517500228306166,0.5060250406286054,0.1862842950992628,
 0.27361968356232924,0.34394655162596466,0.5436254450108948,
 0.10954896421585414,0.45958313723118904,0.15954911872495847,
 0.4008076907394117,0.871386534554289,0.17166171176694744,0.9309819952773842,
 0.0952207108113815,0.6066605980968869,0.38741716757262523,
 0.46456615171390747,0.2581042976705732,0.9601512057516712,0.6739470047109718,
 0.7746521194251182,0.13717335960103527,0.49611681126632967,
 0.027545494177424912,0.630200673210962,0.7258666669092655,0.6598304910774645,
 0.5920070057359863,0.03838718988067258,0.9474599030913071,0.6734421567189217,
 0.4100808530391552,0.9480242633926209,0.3774763038033767,0.02600152289482416,
 0.5370611462484872,0.5565743422821051,0.6103348002789637,0.1396684196757999,
 0.5083230530551499,0.4513023087557476,0.024478780086875762,
 0.20608553710776356,0.9594885948219382,0.7795943056830067,0.7134007285152594,
 0.7306208274100683,0.9076977816371666,0.7272808334893823,0.719945444444273,
 0.2662325261388453,0.6297319501927104,0.8868083593965636,0.8080561502940435,
 0.9176333268958452,0.29379416492448773,0.5590565820093705,0.21508801429432,
 0.4520459664881531,0.6592484521029471,0.13724860865771005,0.5794606381769729,
 0.1430657687542063,0.08433578594597746,0.04428392235826506,
 0.25024514880825655,0.4665599252755336,0.9086596366825616,
 0.27249049033180905,0.3936230726140688,0.34063777183202004,
 0.5556865280938924,0.06638755975145927,0.7007121534160712,0.7835753319986141,
 0.33780821034548936,0.47486377283722414,0.49227374210432495,
 0.4881272835373004,0.5516191160563726,0.8004951847818264,0.862425705851655,
 0.19934489663519506,0.37837928163487267,0.7098335185575447,
 0.7926293439129569,0.9159536946410965,0.2523916234989292,0.1756090322499918,
 0.9125019644612112,0.7734638620176352,0.15807773628670718,
 0.30011597490158526,0.9686739145300158,0.2885271731714142,
 0.44438861866053536,0.04491344809220443,0.33834999822811906,
 0.7319800929783798,0.24563179390905787,0.9139528249742505,
 0.23062208401781914,0.7385896844249538,0.8863071273363602,0.6725749969174561,
 0.020604732243917567,0.9436240073962465,0.920691518451417,0.8247798178442364,
 0.7363912071528301,0.1927448018503447,0.9355949268610543,0.5370432240458964,
 0.4321260152509666,0.972279669359825,0.8932376482203817,0.0956691388998544,
 0.35291186585585577,0.04352783811482819,0.01992598943474755,
 0.2127109774358793,0.34575430305643373,0.01240338455214296,
 0.07171979042461141,0.3194492976747748,0.21112298497856263,
 0.08946653175210995,0.4248103795669902,0.4651996548538673,0.7591769921735316,
 0.24963736300709005,0.4794451527505359,0.5304872072950404,0.9128542857888364,
 0.6606974705797889,0.7276001828170157,0.19186203065548668,
 0.08524821951227546,0.46145941065284835,0.13940924609946648,
 0.3084131158510699,0.4727700321516466,0.8795420320314821,0.5571599668692973,
 0.7753608805434007,0.19106360072134132,0.951009123019258,0.20747805569300892,
 0.11309297309676725,0.21963370685696304,0.49630045578912985,
 0.7191867765010069,0.3953276275237283,0.7818425798978658,0.8433547512660171,
 0.734309891247528,0.3673469933232555,0.23732752914953392,0.8837352073245048,
 0.20169340569679983,0.517990518421118,0.3581410575972206,0.7597042443711884,
 0.16098022955186186,0.5416615600609579,0.5502041259473432,0.6172084726345216,
 0.9738824277570857,0.011887476526427854,0.667572930314826,0.5922463581885723,
 0.7472984257176069,0.12156365046331119,0.08598485314714732,0.9315877667823,
 0.915849117008555,0.9169646791091779,0.30067592571272883,0.6725509218138138,
 0.9075362316767851,0.16829810852471372,0.4891291723408664,0.828521166264389,
 0.8496205406719571,0.3177272090841329,0.7649463135602709,0.6533043205062575,
 0.4277516883049477,0.004044093262660642,0.8239415098031981,
 0.7029314037531852,0.6035426512452554,0.8010754953607157,0.22151508052561253,
 0.577916681518159,0.2226619221252888,0.037591643134692654,0.8286465927471808,
 0.19511690230075607,0.4988304441226129,0.5530594039319044,0.9003736069935857,
 0.7131992509696587,0.8052461034951823,0.3055289441541156,0.11392498752250613,
 0.7459809707138617,0.34612749915366425,0.1354619911838313,0.2918923285204762,
 0.6163717592549102,0.6004549839150126,0.3144171222002239,0.9549159978829581,
 0.7974620292722999,0.7049098818789631,0.6976055273751214,0.5980703694954201,
 0.9595131440879145,0.23532571650076828,0.3528729385753727,
 0.24465017183996673,0.8679844000820327,0.8104450631949424,0.4723391693918326,
 0.8038736820653755,0.7368213612365191,0.17009434936904977,
 0.16063938752582796,0.14677891619382544,0.6713145152345779,
 0.44953600325000365,0.08261412518496991,0.512650833495431,
 0.49399496624847883,0.14001379952222714,0.7756796055800356,0.861570591247869,
 0.6584850814505928,0.10020521185231823,0.24435627040243535,
 0.23348704649288388,0.37532909656471813,0.9335101347607196,
 0.49289761626677564,0.4512716808648486,0.44956968669751507,
 0.23990606766930533,0.9203723765467876,0.719169841519078,0.4939154538918502,
 0.3552337319248574,0.6779077177545371,0.7317441860566316,0.6054909560466989,
 0.8625576315941831,0.2592799918440778,0.38387436821542875,0.861783960050611,
 0.7379036770456958,0.8141320658891021,0.6119210241250459,0.9975337706449783,
 0.4818439335657819,0.8501826289045339,0.11356086416549094,0.4272015280275505,
 0.7674838525152856,0.007081267296088489,0.5405293290663025,
 0.4955623483217477,0.21040288172267463,0.4628605271716698,0.5674771225868473,
 0.2645477553875756,0.08836456748111776,0.5267069140372425,0.8954387984455574,
 0.7036844818921403,0.5501592864140995,0.7692970952881715,0.40773334284544216,
 0.7369273257507624,0.2035352696326902,0.264017680676077,0.9588461307420321,
 0.7635311169705442,0.26797897420789407,0.18766816471159786,
 0.8278349912949042,0.21027739204890394,0.8872255821192981,
 0.44639905053592377,0.5796743214420275,0.9045168844372503,0.1317987624456034,
 0.007051142344185579,0.08716057025048995,0.6395116821822462,
 0.48689433152347106,0.14392727263578553,0.8263241777886553,
 0.7765499860649703,0.3007736729249467,0.46979957759865487,
 0.27525138967779317,0.04670760157930842,0.33830178240040154,
 0.704567366162089,0.07298700761843646,0.4889730330797408,0.46017814134133883,
 0.9331694897895565,0.6582911466265684,0.06720765581410115,
 0.15566603310348337,0.693037336106916,0.8965511915226406,
 0.024607866179908844,0.19430162363936299,0.4128556555406607,
 0.33079367464454523,0.6676603048886633,0.5899591759153401,0.7025101896993055,
 0.3649001280134079,0.8250316731389056,0.9117510966691946,0.0916051829784994,
 0.49586148805225994,0.9605467583517336,0.9910066937674564,0.4713742865363899,
 0.4928608737740444,0.49788286485545474,0.2708580772964946,
 0.35382580065903624,0.5641274810075524,0.27647023291010653,
 0.17444043805957532,0.17649535294784124,0.3588991950218644,0.403994603372387,
 0.7040956475541108,0.023929579274753254,0.08260331025997447,
 0.5097290400725313,0.14262946902034723,0.40572247724924226,
 0.34512376338789896,0.6997049459084469,0.07081413325301911,
 0.32162427046709996,0.45719257527225554,0.8267074082079459,0.1343313522481,
 0.35891076132069233,0.2835395326325183,0.2862323107816349,0.7708426794015086,
 0.8827471378213844,0.10492627182905567,0.6471253762205134,0.6711299052023781,
 0.6972677810727762,0.4706154137850582,0.31098859844790594,0.0315941405723974,
 0.042665737443925766,0.5960839155225597,0.43821151884825493,
 0.16262836399926883,0.3985921704642552,0.4141992319630464,
 0.18427074698942536,0.21677101048780634,0.013190589580910794,
 0.8599392342209501,0.9346246952665019,0.29891429857402485,0.6435754472964716,
 0.10506272932004446,0.23061912917542982,0.3559922750446781,0.960697289267297,
 0.4073229390454177,0.5649410599424409,0.17669730891712965,0.7917778415802253,
 0.5869545172429939,0.39752890500167404,0.640406279435714,0.12955943572597406,
 0.8673410030155481,0.7669127134887149,0.04404890643874726,0.3355994871234681,
 0.25586823020326055,0.737831770796416,0.71532616909167,0.11517786958930842,
 0.31962749106712673,0.14940850514467174,0.9395652149197793,
 0.45237230247637317,0.6571790422028497,0.08132010035811221,
 0.3572030390702645,0.8108981448612234,0.9915409947993001,0.553104198009595,
 0.8123615454148623,0.015751507375793672,0.5037558850425935,
 0.10392487966908548,0.192241578072193,0.3330686292889955,0.11532828818102914,
 0.219752064615527,0.44204645093738293,0.15082590098505133,0.4842352825219387,
 0.02323926597785453,0.19944734266796293,0.5220608607541647,
 0.3736633863742804,0.456634913599387,0.6387236569000653,0.5764479875258126,
 0.22413759137764044,0.9473769916649726,0.8208550130478933,0.7334752880768836,
 0.08695638508610182,0.26984669298816777,0.5483333287780904,
 0.7449564923912461,0.7891027706598972,0.2840050612636933,0.5621334227352801,
 0.730064708850798,0.6560752706116121,0.3094779945669224,0.4920905111681594,
 0.8072982909751848,0.9644803144269422,0.49635903694428574,0.4454229935216638,
 0.07226021203792932,4.65809414305296e-4,0.19382688223823097,
 0.3192934244573238,0.21892879918651786,0.690748442252657,0.8951249779777257,
 0.766173954675615,0.5620705930760218,0.16260082504831275,0.7778979771665144,
 0.6581258651085495,0.09673076654730683,0.5650794236312957,0.4233912460481193,
 0.9996566104094757,0.18832750785298302,0.8375652472361726,0.8622182318889362,
 0.07280296675778697,0.5406487907629696,0.34915165964329664,
 0.20434584096703678,0.31271053064021326,0.4859022539856669,
 0.8491316518192161,0.9507069017103102,0.7280076556910354,0.09001749608197251,
 0.6222861195718211,0.08294264711632171,0.3883780519310853,0.4727352347293058,
 0.06203002159122495,0.7498543359492503,0.24230725596797076,
 0.3726903273568678,0.7841663922885808,0.561329115547922,0.5425762311451253,
 0.40540033831980404,0.8184868331439333,0.28798925323363045,
 0.3256969762764521,0.4726672990409152,0.6248254130829477,0.5816835241913152,
 0.08298930261476711,0.04916268553589154,0.32281517482295996,
 0.39235476528479296,0.9301324923892342,0.01997744878060681,
 0.7614070035039535,0.5287423909658101,0.226409099173273,0.8508853896427093,
 0.8000295722806822,0.5452235007527826,0.8695431457998298,0.2223138369979648,
 0.4103447998329719,0.2522501116038637,0.7094256587735797,0.7147794926968011,
 0.9131835539134912,0.7479403988462477,0.9139152489866723,0.5832255713996157,
 0.886020909895096,0.5872620167208608,0.9336084623117673,0.7156786512833925,
 0.7058404731253618,0.8427750277605722,0.4211605306285101,0.8728008157073648,
 0.5460248077137221,0.09534573903659749,0.787599888313862,0.8149512960647024,
 0.9234547808914806,0.9984392550599808,0.7978641235953763,0.19136102444462044,
 0.4457040711992033,0.7762716326745837,0.4819267021110972,0.9290589824476815,
 0.5945223717139696,0.5454812056014512,0.5760087250963806,0.5962657676420249,
 0.1565613208227542,0.1377263669302502,0.8636378805592436,0.331139927402336,
 0.1721958571555996,0.32923316789598345,0.10677138649358331,
 0.19083720470132048,0.8224474821436877,0.10276437818457707,
 0.7180898543116023,0.6447863928564129,0.2045092844251193,0.9709588009519793,
 0.3636394514290858,0.5588988782392471,0.9096810472584718,
 0.057962695336645775,0.11990752025983775,0.8280764073700799,
 0.5591211787273692,0.49715363744548935,0.2558725824617357,
 0.39827160081536006,0.1369264102120349,0.5140300000659245,0.6453405927470278,
 0.38424595233070724,0.1383154534243658,0.13777096432966918,
 0.4043604851412308,0.25620918438818696,0.4503764552265094,0.6897583788684557,
 0.5304895042101578,0.4903874199894185,0.8632187487666328,0.9877340268311442,
 0.808769774782938,0.3481374537066093,0.8439746435955117,0.6749275411306823,
 0.5027350677250675,0.4232275240218385,0.954214351642756,0.42856915243858174,
 0.4663493079180706,0.39644299026562835,0.2741988760200855,
 0.07896996379714849,0.38930239222614893,0.8877930737813448,
 0.5250389888681467,0.3283237282212461,0.8718669053216159,0.2883855864530471,
 0.4962891528387734,0.8148545094438313,0.6482289378952463,0.8401863059218374,
 0.9883434598081928,0.06576506319472464,0.9306172488509001,0.6202809689801478,
 0.5468696675066698,0.6333642437982562,0.9105485932412097,0.3150215097786495,
 0.15911349665065244,0.7275285886028248,0.5295433413731352,0.9087967264922466,
 0.5427556134540654,0.4096105967173276,0.31432690652138906,0.414438043976617,
 0.9131705360538411,0.2300982925502315,0.8471025465300086,0.7406309305403763,
 0.8107812698606441,0.6818559922358151,0.10467036285674347,
 0.32717388682342974,0.18046260588274998,0.6354896993356138,
 0.8984834008271426,0.31209418331568517,0.18747824326641416,
 0.6400153815827138,0.5386785013895229,0.19407707164333454,0.2857926146892428,
 0.06157394429075891,0.1508482268536172,0.7432256494958371,0.8577147075101247,
 0.3096967980644484,0.5165673331000108,0.3409223120025483,0.5118259234492644,
 0.7397622610762087,0.14122359320614508,0.08452235251714146,
 0.05718094652212646,0.5407043870411883,0.8413738054789475,0.2825481565812913,
 0.29187113200087955,0.30417897730831533,0.7759390144475341,
 0.6392059217197416,0.15732638310533353,0.9882262757393249,0.4295666405736218,
 0.032867491162021256,0.4640257050002782,0.8464541240512388,
 0.4384953678981558,0.013655020665798934,0.08292385202153651,
 0.2088733745903537,0.3671478504661898,0.9310756492833745,0.9599799231996318,
 0.3630469400307711,0.2158089322409782,0.8633880127827905,0.5132986049368007,
 0.5299446845513609,0.7717421362559748,0.7774164368484292,0.03525328300551556,
 0.6208540280021086,0.5570967920146207,0.4811527763618437,0.08125068969468086,
 0.6558360448877751,0.6452166819237002,0.6406715414727563,0.02189203120230576,
 0.42410414223389503,0.654953679629377,0.22902030262802764,0.2867149800720501,
 0.004510370732105473,0.6012618595364843,0.7609909448448868,
 0.2998893344590827,0.12688309540716136,0.3089575069228625,0.2457267727731527,
 0.4407208135699443,0.21764035476991828,0.43534499217947864,
 0.23797159514976984,0.9507086763374579,0.713999153096236,
 0.024783606648586254,0.28642381816165985,0.6487960758320869,
 0.5600238524345209,0.5363353638566727,0.9490089175270586,0.9549923868627973,
 0.3358485981355791,0.6401494318989254,0.6153417532812278,0.8321982181596665,
 0.08234555705023316,0.06475915951538558,0.5765653137384603,
 0.7048298999398995,0.15716368228698385,0.4260978071031023,0.1844050336524895,
 0.4749652677346847,0.08400280162093776,0.6621005022216986,0.9634722572061373,
 0.11896748687470149,0.5830477878255063,0.7651965054957024,0.7044675302153052,
 0.09390243421008948,0.2105452502976961,0.42046612028163544,
 0.3712702231995598,0.470283423853173,0.66680227785849,0.6982872568010565,
 0.11241122795440772,0.9544098819524448,0.7137900530013788,0.513694826644035,
 0.47127988808595933,0.9269684893416602,0.6248307876560558,0.7105248058390143,
 0.1726934106056882,0.2837166769183648,0.9417596515059702,0.52807573043828,
 0.8971587472310958,0.05885851602072589,0.3375444042474097,0.6379562625875375,
 0.49586941884007074,0.5347426838190887,0.27033115306806876,
 0.9103054596190716,0.5033895733668613,0.19043413635511186,0.5118836743874817,
 0.04506582698109929,0.5988091069614653,0.9756435072223689,0.7209218019992778,
 0.5997268722200779,0.7015632346363928,0.6085319935786688,0.8983075231619221,
 0.35964923861772746,0.7336468086641434,0.5027018429520558,0.3404719943126062,
 0.3415907404014261,0.23225470762223,0.6514246785953114,0.5282240147805538,
 0.03212787575682896,0.893716730583382,0.41104807036329716,0.8458505053711173,
 0.20017068631651092,0.14618990022214984,0.09671647655901927,
 0.8137085103972895,0.3379044491862119,0.6067302699169604,0.8991040453831236,
 0.16803059133398568,0.34417591009545256,0.6167222419963935,
 0.26926478531303066,0.13504601797778304,0.3105507282002692,0.647911878884688,
 0.25348193823888177,0.9502557819735518,0.47126383046617937,
 0.7427156708272664,0.10894493608919875,0.29015923740333793,
 0.5347855040279762,0.8839239910569602,0.14372135655882412,
 0.10682865721933221,0.48536352956964235,0.4900998541216183,
 0.9139764506252628,0.3219616825207551,0.5876979705313227,0.5058171675115595,
 0.004713931294495821,0.4992627328063177,0.6616890020609982,
 0.27728790782231716,0.7750405946143688,0.5976195415622674,0.9099785726290394,
 0.37546925515917473,0.3014748512440657,0.5409847150729058,0.7385814150591454,
 0.9163666180215608,0.5443750594744154,0.2863016933595527,0.990757538756628,
 0.2685471870960924,0.01881980202680833,0.22968618546803776,
 0.9658072272235996,0.23016372208242097,0.7904353965338559,
 0.45622180369632026,0.7952479214348762,0.3943931549589148,0.4516419329316026,
 0.015805014250405036,0.9187983028296325,0.730749950436265,
 0.030271462361925905,0.25691719036333494,0.008435291212675944,
 0.6929154176463986,0.9435423710674338,0.7829210259920207,0.9207196398852087,
 0.41518574880629344,0.7736304237337603,0.48905141226390836,
 0.01938197719416146,0.8059539837489015,0.9468124652723563,0.409300120501326,
 0.9965737546067236,0.3604255178215934,0.695831435603904,0.864039531606452,
 0.03641593432930268,0.00396442820687537,0.6799990090948304,
 0.08583924059256809,0.6117432551988705,0.030573819706928185,
 0.06506870011784271,0.034520139093371816,0.4076500292385825,
 0.4540354168055539,0.389509040319756,0.9952423633225315,0.7646889706431669,
 0.5809670015255772,0.20707259633546815,0.36417362387982677,
 0.7651639807741117,0.2879592899882073,0.6350107068149958,0.7531131776028541,
 0.04228861184065891,0.2871158380772578,0.5530385998486921,0.2781548129850444,
 0.1487771423282066,0.9713423439289658,0.19467716036099525,0.921588493516336,
 0.9008298222234309,0.16156675647144347,0.5384983497290589,
 0.07923475840161098,0.24107546680070757,0.7426697949704342,
 0.4738998162440632,0.46537005152044186,0.33140114897120476,0.756550897516361,
 0.5737946873765943,0.7005649978656426,0.7944303230033893,0.9808317114023615,
 0.6208021847259743,0.30446502312716284,0.583215921430407,0.04098932315430126,
 0.6244313062789696,0.36975529776334515,0.8613981780893936,
 0.07545756468666498,0.6332939286967987,0.6589776144277549,0.8794322414162209,
 0.45224848713616084,0.8195890767057662,0.33701966893791313,
 0.3003314666834894,0.572856349931244,0.9358382873857078,0.8066067198655772,
 0.7574466089473565,0.8446429191655019,0.2540781997624709,0.30686233818955955,
 0.7529932864994096,0.8046276732511848,0.17014196528471937,0.6089041881599805,
 0.8360361770220658,0.16461150235676492,0.2722102236035775,
 0.08839165672014082,0.8323209285003748,0.5553874836383939,
 0.33245183124909805,0.10919719259559335,0.16846336239765225,
 0.9061538939738549,0.1202188082146145,0.1846691785893484,0.06931134090615976,
 0.46027545169755046,0.4790878409342352,0.5271906616082322,
 0.42997177132049136,0.8815754297206773,0.5316218574827788,0.8059800538052833,
 0.7552432949479378,0.30816381746870447,0.5306942742399937,
 0.23538571715861623,0.3790935948483606,0.21440389175731867,
 0.7864136185724525,0.11805102976225534,0.41192340833913743,
 0.6215954891663598,0.10880445858161758,0.34310787522905384,
 0.8859151302153813,0.7587360997630076,0.8070614816186099,0.9379513357979657,
 0.4872394572744366,0.7232395790610724,0.24516849300875965,0.281909000428607,
 0.07704541854201441,0.8591832972011985,0.40193097060898086,
 0.9718469792803051,0.9860430300121031,0.058059489108190476,
 0.2281720753983627,0.13227084045012794,0.7169254700852722,0.6687808383180819,
 0.5460420319956816,0.3082657491571943,0.6736377759496304,0.28783964825476405,
 0.2969286808618865,0.46810173437049285,0.5132072819131521,
 0.06910491215476267,0.41721907583724815,0.7735006163007079,
 0.8027005461658839,0.2693353979165466,0.727560986276669,0.19628121580513935,
 0.530396903542559,0.026196250628327422,0.04180367792379136,
 0.8713152417235708,0.6149331633250339,0.21502286600813925,0.8744590773101796,
 0.8239078352478281,0.7124102978651674,0.6419906685317804,0.38872060433862365,
 0.3916954722522181,0.7771495484851535,0.0874031203459823,0.6320870971928675,
 0.8369365925069053,0.26756433376744815,0.19612348030685234,
 0.13435860726413829,0.3456490555598407,0.702701891384239,0.8247134454901838,
 0.7620719475625303,0.7077691893989005,0.45296389388638403,0.6338969644829745,
 0.30620979234099277,0.3595042458715275,0.21388067114023346,
 0.6732539187807066,0.8379472001119969,0.3257792683132228,0.978860488884369,
 0.0035985565478876547,0.7556684119684128,0.23214892802208686,
 0.5757954175121524,0.19676651424385028,0.7990511727607296,0.7202149754438945,
 0.8771866092805364,0.35990423459291054,0.3475621622354509,0.8486551616906343,
 0.2287322882271451,0.3381943533514481,0.08366468782589365,0.5670373776910713,
 0.2638263101311342,0.35608160210939965,0.8010201304331463,0.6050718323458576,
 0.7381458844227297,0.9675106910702037,0.5150990815082893,0.5127476551445465,
 0.27829156418555356,0.8397000169819482,0.8941444486243434,
 0.25686503776291414,0.5980485823894495,0.6251415817106595,0.477722499983921,
 0.8350791268523413,0.9411939612321514,0.6486208063329146,0.05351783336777949,
 0.40058120109242557,0.6477705661935873,0.6820004001362235,
 5.683643905374236e-4,0.38021447851308865,0.5485623043835677,
 0.6811409830750532,0.8370605766719721,0.40915572333173134,0.6380934962027668,
 0.23555516275018462,0.8299927188623564,0.32675693050899335,
 0.15733763313138338,0.1812677423203949,0.8017683000729706,0.7054814196295273,
 0.526884599097003,0.9111628893848718,0.4899992514545669,0.35482794114877125,
 0.2867735536719611,0.607309531617044,0.2865641857237411,0.5734357135670453,
 0.8930089902739373,0.9256706647952198,0.6475743301246388,0.18252615863857136,
 0.0404504098688635,0.6498646583703179,0.0772633747815914,0.17335803300164443,
 0.8413147462486813,0.9227861214822164,0.5832871975709342,0.48608860996139724,
 0.8428677881988956,0.35375570211120155,0.32003792933093833,
 0.16267278015097075,0.14082861732039598,0.9390344870803894,
 0.42147956575453893,0.5051361345953145,0.9719528252875643,0.8835016105161004,
 0.456964477016795,0.49477627888925724,0.4465903428410961,0.2382898908267389,
 0.7849531128363308,0.5344159641037838,0.19933968238613942,0.1235054207858649,
 0.23119905330613288,0.9229580686247276,0.6914507612543103,
 0.47861501778456805,0.22851720755024707,0.5072890142301996,
 0.27251845024596855,0.8488273384225162,0.10696778181563915,
 0.8454260965855201,0.16854798732372844,0.40473483816376365,
 0.6746023559673188,0.2494938040903426,0.64286207693421,0.6061436023731457,
 0.6199268486292507,0.6713330463106763,0.25519507865389246,0.8760461992307216,
 0.7699661823447744,0.8191779402243946,0.31375052682272064,
 0.38083618857502766,0.9763049472506156,0.21018000667862968,
 0.35298532968975704,0.8494674668642346,0.8066233075340675,0.34671127465859,
 0.9371183242365081,0.5340041460221108,0.9936961543384022,0.8819404258366892,
 0.590254066737193,0.7105384187705447,0.11336901672198885,0.772398585956904,
 0.3163344160426429,0.33199052477081603,0.11398501290722707,
 0.5772220516162556,0.0025046574824966195,0.17006556276141005,
 0.5948426717199761,0.4490811613761255,0.4125258016917538,0.4051689407915461,
 0.826913412444153,0.5839443770452614,0.7219558166932448,0.8155730572151512,
 0.5403695276605508,0.08465597298383176,0.02270374391244845,
 0.06499321295343186,0.3645023222996888,0.9255788883922157,0.988617195928424,
 0.4425598173590535,0.2998266882503967,0.6136427103897049,0.9935688020510848,
 0.37778685405270873,0.46068219826530077,0.22751128271319543,
 0.3734365768136022,0.36278323528805656,0.6266539068734444,
 0.45756953589963656,0.9435612142773535,0.14397811129715454,
 0.1437740066217803,0.7680907219110383,0.555986536156396,0.2478856162858316,
 0.9747447425639123,0.3638671428856225,0.3447868070452227,0.4755971891717775,
 0.2846278197682812,0.7354636676022311,0.8285453778126955,0.7240126167211838,
 0.9173090920168034,0.9232012934820901,0.9686432699128653,0.8631874532366469,
 0.43559229812294853,0.5792568869393486,0.6632953570044379,
 0.44685976571347696,0.5540544710695932,0.5287604816965045,0.2855595340934991,
 0.7110926073169367,0.24158691273842248,0.3094771268713026,0.0833532055226317,
 0.13184454308026206,0.9542900118570156,0.04531047890559936,
 0.9634259405424475,0.49013607809554793,0.3797264226942132,0.9187445740094743,
 0.33462645373883804,0.16253116985965566,0.7242552586761659,
 0.26827878963113283,0.8511138093948043,0.8555193285047645,0.4420058048274974,
 0.36416698020897975,0.3496965389788921,0.6321153519118755,0.8691915380394344,
 0.36408921943444317,0.6969036561231354,0.37177513701609355,
 0.7093000985426068,0.8468802286033214,0.3236676134384612,
 0.007012598396875069,0.10085350211039057,0.2825937486379766,
 0.6051980323964333,0.5521135208768585,0.5139704615439317,0.7733135591775046,
 0.6726442326324583,0.5611344170097906,0.7201620154282233,0.09249109085226837,
 0.1607160603621426,0.425478885729204,0.8968417824039749,0.5986147878411311,
 0.6890210140476851,0.9566666177752237,0.3138305630301407,0.48226349441590777,
 0.4652603932844873,0.6750712717342775,0.3195793427700183,0.7084696536776102,
 0.21086052588839088,0.859152345061275,0.7330072327598746,0.8008387227651437,
 0.22820950258921624,0.9056978619775896,0.5623560610610338,0.8071432077021016,
 0.822448195418025,0.1275701741215276,0.9159166716862533,0.9433910994158018,
 0.8592965381959625,0.24571057432442722,0.8473129280438836,
 0.017754474746118465,0.4084503860998826,0.2728248699568375,
 0.8942909942753557,0.4315967375041819,0.9287090632039253,0.4097975754770673,
 0.7551302798806772,0.6064513305977132,0.6688704468716836,0.658395188471139,
 0.1662266286415358,0.04217874673209587,0.15406290809170298,
 0.04099649650958215,0.6887935566456914,0.37327156581620147,
 0.14268234586016715,0.5680741337623799,0.3299162962675607,0.3643230812396412,
 0.4691109861167886,0.8350046815540251,0.3904620208268831,0.7788547560603857,
 0.5244362280218071,0.6479896765214854,0.4179580229105998,0.9521264962972071,
 0.23428342897446552,0.17571754428847053,0.9891116975917681,
 0.7938918012077344,0.24867717043181647,0.5557888371510172,0.5512957689812366,
 0.9136097921114753,0.5469665626096365,0.6498869894077104,0.7360144012274499,
 0.051165817709597494,0.10983786599186118,0.509009034830039,
 0.7000351611217586,0.1399918284112821,0.6727765515243493,0.45626891931812896,
 0.7866281460649148,0.033328236628694,0.023042874171439576,0.4734452129555746,
 0.2542611158810968,0.9297448050701946,0.8708650070647956,0.9148633900522006,
 0.5870865585035302,0.7505049658799291,0.5694740213564443,0.507773476852106,
 0.7007046223771107,0.32882180571425657,0.213566271563573,0.9745970115262504,
 0.15957209303878583,0.7723639172481087,0.876202104573395,0.3410634022445762,
 0.11244370091618694,0.28073203310662964,0.26746023201203417,
 0.247017391729907,0.695964679454364,0.9069061868064947,0.5019890484485342,
 0.7704400763193996,0.6502445163001402,0.519519084039249,0.5970497408643961,
 0.3791842991479413,0.4374143543826575,0.8788512292959532,0.8625139282871223,
 0.0077533896322002915,0.27040455200650304,0.8278846771391158,
 0.312170984993839,0.740946719432606,0.22482075245220934,0.5459270468415238,
 0.5500027880261512,0.5799123358096878,0.9572246292969069,0.3489634778057654,
 0.4049447570293577,0.42371928072208564,0.7480119488853773,0.7412693910754032,
 0.7372561157123229,0.420323346792006,0.6787948542693631,0.3035782748810052,
 0.014602986773433635,0.7861710590390901,0.6994512335666838,
 0.4346271304356276,0.7314070322467169,0.20103780479186673,0.9981102635967269,
 0.011115451246419106,0.08187884122423206,0.5378700970898126,
 0.811124465008924,0.5029594890766684,0.19182926883652718,0.9056766796644171,
 0.9042342033162432,0.4183727624690219,0.09546960150108819,
 0.23927976368907322,0.6849197457436651,0.16987879449249854,
 0.8205044643581364,0.5028845284561417,0.16362814754513189,0.4398476471615316,
 0.5163413430430291,0.05610688086558546,0.23655239184061527,
 0.8007344589088906,0.7259639350947718,0.3567500902362615,0.08265164173016026,
 0.031197270542085986,0.008228584418430884,0.8358320965766777,
 0.6502944315043832,0.5305540896498917,0.25148809078956647,
 0.12774794490960184,0.48051907630474056,0.9476679449492954,
 0.07246686818354242,0.6497908288972215,0.8804510558762164,0.9468356787022015,
 0.5280355037827438,0.7731417723464084,0.41438886702482225,0.8051251400462793,
 0.6682386200081855,0.6573836884567061,0.6952649514876768,
 0.006559644128201869,0.6474280490610681,0.7271528915776988,
 0.18076502246710335,0.9513820475360112,0.2011331675593151,0.949404251250138,
 0.6996486703252369,0.7006466795732619,0.10280872655380335,0.5357434717815169,
 0.7801402637533456,0.6807999169203507,0.37144258942389796,0.3442435730993485,
 0.44310521002579617,0.026083466353620577,0.24202225544445088,
 0.8471652610527658,0.020587267739690995,0.8328586318144038,
 0.9991346721849986,0.37817778017698456,0.0728880288226661,0.8299371578015621,
 0.6731798859131528,0.13244192512166908,0.8467113631998444,0.9444470102426366,
 0.14075050223848595,0.6400402616120254,0.7739839524090971,
 0.23815663650870178,0.16607085508621178,0.8324671027081344,
 0.8399197742347304,0.5454449369482307,0.18616086686665612,0.7727637975662084,
 0.5831558509676839,0.5755835962397673,0.7617421666333997,0.18869609284686972,
 0.21239550156879072,0.9513864968442322,0.1809970222027284,0.5691474782815771,
 0.28020167209994196,0.9077943464377021,0.45908213802884545,
 0.9594265562599821,0.43020399885151883,0.6736126034638318,0.8740579434882491,
 0.1754400484647114,0.3041274338102219,0.07099322701690336,
 0.26205599012510417,0.08289781567299481,0.12062542132265297,
 0.04168511636225136,0.6305794940050669,0.25599204723544944,
 0.7826115749949317,0.6988787621133887,0.4400298874898494,0.6768235102523611,
 0.8477819522630018,0.10024593780915958,0.8540568386896045,0.604042876513811,
 0.9169725406247536,0.47135276548691907,0.6328958464668178,0.9909893468630656,
 0.3130236811403606,0.15194705109713436,0.8760825798633246,
 0.11693831219696527,0.016491333822278853,0.8703396143628147,
 0.08312480734457639,0.34946856233286816,0.7629178300058904,
 0.40753301343819526,0.8124063319834489,0.993886064975929,0.32363036944000056,
 0.3326127548799015,0.026510675496428493,0.9870072954761663,
 0.3499262335132498,0.151363288669222,0.24690343936091463,0.09374446062357533,
 0.2194635838387805,0.3587495267650822,0.8315617187178267,0.8775114209003015,
 0.7996499807142697,0.7269908482973307,0.7840760699671792,0.6991625518642004,
 0.5006524772778185,0.971623205109087,0.17382452235601376,0.4613966569036383,
 0.32640510611111884,0.7444961207300322,0.4856343409583296,
 0.05896512192590775,0.911876178256367,0.1013378491784458,0.6084421183388007,
 0.9354387755244142,0.40422457975530746,0.523398357652179,0.6684699195673129,
 0.43386607920112563,0.406373365199472,0.6417171282039593,0.6180125133908111,
 0.5378702527758555,0.4115209360838519,0.6448527190564816,0.9162684684270772,
 0.3710955459415699,0.6835183410655147,0.24071562975467486,
 0.01956149588645062,0.1692829741446492,0.7377632126745123,0.8407554545979723,
 0.3723826194766299,0.7603793862635135,0.7542339199609399,0.4313100213442276,
 0.4856675152197488,0.4879805418025497,0.5208369839770209,0.13710558998449207,
 0.9635434290720276,0.29563397582802287,0.5434270103229466,0.7560915320085096,
 0.1488078533819549,0.4176629689689031,0.549314790233106,0.15973801315691705,
 0.7177228166237601,0.8635475474011898,0.6089280344398645,0.2936634946319956,
 0.23752237292093337,0.6987021427262685,0.260153123316735,0.7874543041233536,
 0.04028652998353521,0.9072662489629593,0.11159104449224777,
 0.6095003156041643,0.44124444221410386,0.3142112590896037,0.1349542397125535,
 0.6504837573341098,0.6273087419280601,0.4695383389909853,0.7886404143655114,
 0.6214453045605318,0.6159133354292773,0.9284994482919005,0.911954087758321,
 0.8292831337564803,0.7662178957039447,0.7449515521071395,0.4548037963923969,
 0.4955589294182101,0.41496370538541816,0.7069839294064926,0.2602933220884628,
 0.7356851811187809,0.6464238201461356,0.25629368474284464,
 0.12036492115003838,0.7038626207322376,0.3179090928913436,0.9443113105946851,
 0.5329522575408765,0.7802461551757541,0.8278911647066899,0.9347961519341313,
 0.040671946127226355,0.6801243463854966,0.49911804379840696,
 0.6956090536717912,0.13078361204303746,0.3090045718721284,0.6522929646822407,
 0.7159151186190496,0.08669334489874381,0.9936879250156601,
 0.42701215665993053,0.05983555238489369,0.4194818904337295,
 0.08660057987195269,0.679029040186478,0.6157538703806442,0.44366432981033765,
 0.295956811980035,0.019785080260784293,0.40442882018330417,
 0.6193956806934603,0.533446271378161,0.0025639891076180144,
 0.1840231148732583,0.48086456281311973,0.13531990352001588,
 0.7917950148770694,0.20575185754054537,0.4416265552992499,
 0.49508781433893145,0.08166537397285434,0.5885149128250536,
 0.18859508342821218,0.06068016608147642,0.8794274576875447,
 0.5355482402305257,0.4990060936785884,0.5135557880666464,0.6471003623982017,
 0.9165326402634396,0.01935764521411898,0.5438726197429576,
 0.08138558745154745,0.6307380486095886,0.18519481413410688,
 0.3523315529007708,0.38008282492010326,0.9482342367057847,
 0.17077194840466192,0.008182580832022501,0.8734356141330831,
 0.5207449524214091,0.8323124827539814,0.6998881651293057,0.3692498630884222,
 0.716185754791109,0.4862695388662608,0.9334207651042101,0.2100926333989659,
 0.32091389861646014,0.008008281898103098,0.9146200970055269,
 0.21001363965421227,0.0436318385572072,0.689800626788359,0.6338944538674006,
 0.5075083468409478,0.4160548805752561,0.9202825276243458,0.3522888142386249,
 0.7172939303785953,0.27155457168633546,0.2165364449224989,0.8995129893601267,
 0.22045133223503477,0.5476057280763965,0.0669685137373448,0.518271134878353,
 0.20128273044715483,0.26412872644386753,0.3770100001543635,
 0.1667958672115918,0.05426411992531044,0.2391267343401341,
 0.24669660368309154,0.4850165197513139,0.27894728015575265,
 0.6708318376380207,0.6251564689666858,0.9680600545225817,0.7628514477168408,
 0.1501248891443111,0.36448797359304896,0.4341887154798634,
 0.17787911672510792,0.9696912871881775,0.2278069400343643,
 0.06302451260533604,0.7396225522158065,0.4052703461415649,0.490076157298604,
 0.8285494551343475,0.11846850111033347,0.8935909335960461,0.5312633208710207,
 0.7940652253778984,0.6359260440349019,0.5087421188923263,0.42296450791576345,
 0.6940516985101373,0.127021791461556,0.04345955790799849,0.3263200323496329,
 0.5315836911888867,0.9606479047691848,0.2254667073465605,0.4543890950309861,
 0.5799352669103855,0.8117555587256154,0.6930343466755085,0.9884473789670852,
 0.7624419660155115,0.12443167676428002,0.9902678346960626,0.592733811658496,
 0.9230565295757904,0.6702933073401796,0.4149240366522281,0.6693580311725403,
 0.11125851482322935,0.429660122482901,0.8415392866757155,0.2545548343653262,
 0.5558567198509556,0.2688766718617229,0.6894572002345525,0.9951392812284328,
 0.7900158432055766,0.9984720670947697,0.7829450474003197,0.6461086696536963,
 0.44934177766500427,0.315279812331802,0.12400790805756401,0.2042514494019615,
 0.6755851052965616,0.7924419400786298,0.1529753619096238,0.0470542494318158,
 0.784924444662997,0.9114947378694196,0.9723809859629231,0.7687775677295257,
 0.1340263210475755,0.03908406452518143,0.050912348975700406,
 0.5594746043223743,0.7113906869309328,0.45171532687051286,
 0.39948348740155015,0.06666805376562823,0.08233625291672131,
 0.8279392698960663,0.30374993914839366,0.23759909154993442,
 0.7754382219130117,0.2910888532603091,0.12197816795863192,
 0.42858927559351456,0.41555292858214854,0.6888332052436215,
 0.1586035622184816,0.5152596915631984,0.29717211354368633,
 0.11602373956047929,0.801473120037576,0.6038713087095451,
 0.004811376906558307,0.667908410049542,0.3755611717569334,0.4853172224401907,
 0.7096671588388719,0.9760965340304261,0.4217886778560376,0.27989027001297706,
 0.8463222183443365,0.8908788658278347,0.9236261527491267,0.09965096579228594,
 0.8581883129226361,0.05920808818003054,0.5620725831921726,
 0.18905190692851181,0.954761457738769,0.11194854939655074,0.5215314618662265,
 0.7230261203629887,0.2654715687175304,0.6471008186778628,0.7615182633615147,
 0.13874626854609518,0.6992709336270385,0.9562608557456082,
 0.23356207207443613,0.9178836822221523,0.16397118796855636,
 0.8795948449099054,0.8168992038069574,0.16175313271365754,
 0.43178294024058506,0.777388786480127,0.5724634701557662,0.6375486893212992,
 0.08692544320849271,0.14140126496862104,0.2551155064586291,
 0.47885404275250865,0.865627819528105,0.1244902634045465,0.8175807164721627,
 0.3994796761888073,0.9181797000597062,0.029207368215875107,
 0.7203321954187974,0.42694208769330955,0.33285641430313617,
 0.03046439036414883,0.12778256510191977,0.6676270609971978,0.615730504524764,
 0.5209614395763812,0.5110471276816763,0.537310679516019,0.02500744039608238,
 0.3821937478800371,0.9193312082745777,0.24275581821089653,0.6531888811030574,
 0.781839636752343,0.23615358768246808,0.867963692225722,0.8477526275634982,
 0.7447271906291463,0.9751500283652744,0.10822490793316109,
 0.16492818742932036,0.660580303534209,0.15428541245086724,0.5419752269327364,
 0.023451651545517205,0.07407311578755693,0.9343903776851407,
 0.32545481876502336,0.6588032279110163,0.024503680243847192,
 0.929363762201973,0.0697959193978609,0.43493162801360163,0.546878512667014,
 0.9869070059348721,0.6535782191017052,0.41158044693354934,0.8012101583012314,
 0.6023196590983841,0.07955846021206836,0.007713388038819047,
 2.5654300455291157e-4,0.36105210196642756,0.7157190860015333,
 0.24433140730227543,0.3313974209606799,0.8911349537337245,0.5104702908045996,
 0.9140060639504677,0.9542288531208752,0.20392991686914075,0.9133432484869379,
 0.10781217371359375,0.26158493570707453,0.31856004526668447,
 0.8501123097447029,0.2826594050667256,0.16661237645626992,
 0.039652960541909255,0.4987744406868405,0.700365540127404,0.923030744999183,
 0.5941817363533772,0.2925271967066593,0.04859517926836843,0.9816865892514705,
 0.48233690321579603,0.5467651179969061,0.4974946896337463,0.727582112976097,
 0.5670814403037582,0.06421648795156287,0.022070500063239407,
 0.8208516466507411,0.4153866934221787,0.31498583244214273,
 0.41831792555917124,0.5741359684307911,0.1525951139644508,0.1785346870145228,
 0.911771162468272,0.6940940194196396,0.4597394040878502,0.022275900950877836,
 0.8315589791809258,0.38636359618848015,0.13803371094490013,
 0.7239216078481902,0.9876134556942893,0.849618745339103,0.9816679341382737,
 0.8945927478878118,0.7440691902002328,0.012615116039219165,
 0.9830115428389571,0.12645041216318909,0.7149434452799373,0.1889744029080963,
 0.29030260759701176,0.3219276611221037,0.08362294008761917,
 0.21671218311555962,0.29108887285102236,0.32922729481903557,
 0.30873543033704165,0.5456388228482034,0.6305092378869284,
 0.07802395713959731,0.39954422767854925,0.29403520486999435,
 0.18600331234184553,0.8329381541774364,0.2548068845082764,0.3843075722948964,
 0.4209744065394848,0.9813294221159506,0.8094800171445027,
 0.0011296493224581816,0.5452140452514835,0.39241462711055264,
 0.15652059025422294,0.14788813136638668,0.8891672524830402,
 0.037417453088320674,0.2251234327704682,0.936149765572017,0.7040756410652146,
 0.4789559920077028,0.49835244553220825,0.7019578883195805,0.9009662206542357,
 0.2006033399500352,0.13778260666247844,0.8238558869298174,0.6852924827937336,
 0.05559831979002561,0.371272908835787,0.39210378135537416,0.2885820045554567,
 0.7468856438645267,0.764196640081782,0.41815698969707704,0.9702104933145121,
 0.17100367960208573,0.8512151546964409,0.6587820703712128,
 0.38994558281786285,0.9994159106993611,0.9585158834104772,
 0.039834362657831135,0.10433743130180317,0.9010906578192901,
 0.4819697515352629,0.4867792986806919,0.9148530549121539,0.07597838072287755,
 0.9539562673341025,0.43115083803703325,0.019032989256220656,
 0.25053660215070517,0.15501945773547077,0.7213772819992073,
 0.9261651609132997,0.21623183073613084,0.2368062898317912,0.5313996089498575,
 0.5529995730048459,0.013590988034392182,0.7976108604897696,
 0.23860888426672577,0.48975666742634705,0.600846511688801,
 0.18274116153733522,0.5736081836393981,0.28498671533396025,
 0.9723331718564988,0.8855552971281098,0.8951386438450498,0.6640870550019867,
 0.7810360625965848,0.15970812337297657,0.6845853909018437,0.8119588693835282,
 0.5887711050540095,0.7885111123847877,0.33513165378589616,0.5048431719067039,
 0.2916612412419066,0.1693770771742219,0.1738178745189719,0.5976199180975708,
 0.9557585140007021,0.4180900777588231,0.365692743183905,0.8065706858464778,
 0.22140132404090795,0.9158169540597921,0.3227579662788227,0.5107195252115003,
 0.695481406075323,0.7476112174756686,0.9630451142884341,0.9534530774688409,
 0.4586852488951838,0.7006823820074152,0.17962392688735462,0.3363597800449757,
 0.28045126900489203,0.18468150649750237,0.6160361221566394,
 0.5708183819873794,0.7739745904212245,0.7910584729052076,0.9054100906960201,
 0.17142010431318355,0.6731520402475517,0.7660524832722981,0.3663096064573189,
 0.8525924996134143,0.5051981823812606,0.750186143227214,0.02091725590760951,
 0.19287761560908212,0.6652638898864178,0.5520444941892656,0.3549384628160538,
 0.9787045822361542,0.03339450827125623,0.5674951615507575,
 0.26884829013093237,0.30988889636548245,0.4561609599424583,
 0.5936547766680218,0.9665876861148902,0.1437713527439084,0.4420144064413868,
 0.8478261923536945,0.44825531005511954,0.555339879303898,0.2556211837887462,
 0.5142569070612567,0.5826248984211769,0.23677671171358416,0.6839297928906255,
 0.6792376180790757,0.48867542968394373,0.12956346101044147,
 0.08943788139919095,0.7281826977402198,0.9556864515790581,0.8963284819899887,
 0.8534359080359519,0.503606970954687,0.2761302311723842,0.19388909529178266,
 0.5797640283323359,0.9912526300892289,0.7461328333579553,0.8996995881831229,
 0.5860648562241298,0.4808632002013401,0.6183723735307642,0.7527851093225718,
 0.012293831839801261,0.25748931936010944,0.6383679489167905,
 0.6837971875061764,0.25396092219008937,0.566878411833074,0.18532043706917445,
 0.7108837806086468,0.44173583708476616,0.5865789903493522,0.975941431717753,
 0.5659905510062606,0.07727525199587804,0.30581077943408796,
 0.23892046801352929,0.8166241492328203,0.9716467109217934,0.8781426598687557,
 0.2090806708003392,0.1692782721107835,0.6315679521102158,0.08364580182444581,
 0.4111624141413086,0.3215932145159339,0.7820292145066268,0.6549637987294287,
 0.7203392615918988,0.683320592663031,0.5066460237442718,0.466899959299098,
 0.9230373924171994,0.2643993221841052,0.8927408636564494,0.5732665281525355,
 0.844904796198067,0.8727579922109706,0.5004282722754777,0.18823126655437206,
 0.6935013347299769,0.8096450740418306,0.7217727468260383,0.9332563307299189,
 0.7545407172004703,0.8513113755662386,0.9189140356262362,0.5964189511860066,
 0.2911919067444255,0.5320214517672763,0.905176584447483,0.1355505367881854,
 0.720551356365626,0.7239193837475325,0.5199479762914136,0.5083761525772281,
 0.5455617237844427,0.49879658402546345,0.48238536479109095,
 0.47667583228421395,0.09011587354009465,0.14073049890252365,
 0.09177918715322164,0.43253429786652076,0.49501965650353374,
 0.4391415135928237,0.34353284482299207,0.6732640191808804,0.9026418132691904,
 0.9515054267964139,0.5719290981832231,0.024589206960877408,
 0.22397467849346508,0.5429969275926665,0.745619047424789,0.22008962283454414,
 0.6783790064459386,0.8622199725922399,0.9773823895823108,0.02019571077195559,
 0.4632910789438698,0.9074218351629006,0.7497526912225221,
 0.044249176528055045,0.17798801343446702,0.5149615024816186,
 0.2828000368621463,0.9521619553315025,0.6505909296098165,
 0.060634527939952276,0.01645952895830427,0.8912885724796344,
 0.4183078586346396,0.0022681982002484835,0.14278689764119823,
 0.9243311128423576,0.906697108022911,0.002858945329919149,
 0.07688323784633133,0.558522277714252,0.2980155893074057,0.5652033924885176,
 0.19828179988529016,0.9938739879260491,0.9417912265077197,0.3310508519831439,
 0.9393581663130168,0.33993995855470693,0.2917798922887054,0.282051392813613,
 0.47550220070992455,0.32610956378927114,0.02390737638172702,
 0.8476501594329208,0.17681623468044472,0.07176227890368969,
 0.4918257048561251,0.2751100482430333,0.7412805077778766,0.77456545495177,
 0.6270632659522963,0.45500934335590215,0.08766793978272125,
 0.9760339466471262,0.33977751175088833,0.46531922677133486,
 0.15637904886471177,0.37462732428236867,0.0340918695355622,
 0.6977909285084126,0.9465178233215201,0.7669689024911978,0.09180078410496073,
 0.4935826089693337,0.2273982031959242,0.7893134750529617,0.7508183918697521,
 0.9203330348307054,0.9499539307803542,0.3079050935658386,0.6274924424780642,
 0.8603340286796897,0.37185552270535804,0.42702922005313904,
 0.36586351612964496,0.4788854491214366,0.24963414435914877,
 0.03581245552157897,0.4701234537524006,0.08106890425703228,
 0.7491450837758217,0.14596929537086334,0.568848908449136,0.5379573473757991,
 0.1741128680612971,0.017096447929017877,0.37450472200800466,
 0.07576128443367669,0.5791819426398127,0.04789098139271497,
 0.28417379870660775,0.2832971688852972,0.8805598655837473,0.6041894498767241,
 0.389400693562427,0.4548544464293627,0.6165288791187546,0.023913363820766587,
 0.2668280514325869,0.42659467566310194,0.8207776996363477,0.5735274153038815,
 0.9574975800225167,0.24488167224365176,0.4075276175670377,0.7368399459764118,
 0.5443247205195519,0.32886695988879333,0.5651177092090486,0.7825609527130372,
 0.15909221152871278,0.05543954826654407,0.25487467784232076,
 0.22477478196007006,0.06647230073536325,0.9594028338823971,
 0.4369689021479882,0.18765485598816667,0.41630011369390874,
 0.9187665906182703,0.8472447627485672,0.851355234138174,0.7096089272995707,
 0.6453217013263575,0.44772027275257065,0.3790044461789177,
 0.11539014833964178,0.4696993261633139,0.6078961044709115,
 0.45693977428134736,0.5702929300815102,0.13634214226164287,
 0.9348737931658203,0.08205950778477722,0.6807214031252296,0.5201570798775197,
 0.43055457913658357,0.5075877668249692,0.28148218550455617,
 0.9362336374497058,0.8347893643948043,0.6145575499983081,0.9549613459615347,
 0.08454732706359602,0.6350500957371326,0.3725618599985949,0.5492473429648514,
 0.18489512787280193,0.8644778509609272,0.4149009816262371,
 0.03621746002549975,0.14770166084853076,0.9884090623009836,
 0.3016391067818449,0.7462345468761172,0.38087535387142657,
 0.008030064768528522,0.5351500533560378,0.8752672279427941,
 0.7578798213511024,0.5573042152528003,0.3022472357793142,0.9390928965357918,
 0.16720949853675138,0.10212767393261568,0.5658251453893298,
 0.9045584494944938,0.9978918841279208,0.006871589515392906,
 0.28668175975226484,0.35644154423892316,0.9362187718116067,
 0.9363431334536012,0.19739925274708003,0.4072721359925595,0.4091645564874281,
 0.8875540966695286,0.11520451914001217,0.15889225536307494,
 0.8547857541406068,0.9832619946404234,0.6109512062857957,0.3097280860775973,
 0.28190414410060494,0.7224652574297266,0.5768450228252415,
 0.16483117848988948,0.4128368965664083,0.27082228291655774,
 0.12105065405768223,0.35506484207974076,0.5209770875210117,
 0.9567084473954378,0.06347642580902846,0.4833348622218494,0.8906261690281018,
 0.6944929526497574,0.8769848743642118,0.27891485942577354,0.9808260652797157,
 0.9893691395315799,0.2777818411273929,0.16042562397513127,0.6858403204605492,
 0.35375113446361983,0.08241788915796722,0.8682317076368586,
 0.5790081130970031,0.0651877439174483,0.0985694446969344,
 0.037905222143460815,0.5223622926205352,0.0025910357116301075,
 0.6828069718839422,0.8906406458751246,0.08142927714097692,0.8039860237255196,
 0.7315368366831866,0.9268287710885719,0.25289073733837575,0.6996149527027685,
 0.22750925779652376,0.3490128996539835,0.6127582149100713,
 0.20389336692651694,0.7627628724565002,0.22147482483160752,
 0.1105344902583596,0.9004798131982232,0.5023152434716449,0.5378049785759461,
 0.5305131757423207,0.4780587474934783,0.35521256919516064,0.4385396294357836,
 0.5878794789068087,0.7131045816680641,0.48758197328362074,0.8351381744704653,
 0.2048081607582668,0.9791340475752981,0.9269359759671839,0.17098407928934223,
 0.14125021605132404,0.00583703001549285,0.3369445044940905,
 0.46134421304025064,0.00481464749889271,0.4786366862757795,
 0.1475428258135928,0.13264964592492978,0.23355436463990864,
 0.013845768242927492,0.7334817206665601,0.49828443774380804,
 0.18830542083824509,0.2734360016876699,0.6553047012723934,0.1589187887826501,
 0.2739217152582647,0.5770118858377297,0.14479295509566192,0.79185946796732,
 0.6705865670818797,0.913334245384173,0.6834108529477941,0.7716030463486133,
 0.8296354571092138,0.4509233306970776,0.4331821023665424,0.7599274780849784,
 0.9400885810946497,0.3295258983484043,0.6828483834060214,0.40869318686324063,
 0.7915153580715949,0.42650232174860403,0.5884920824687803,
 0.31161633630014396,0.9529536505387668,0.6375526893974214,0.7597565543759847,
 0.8954782840107752,0.0498929650864155,0.42082502337249283,
 0.006340178891385162,0.7275345555292715,0.24443786963325376,
 0.1959973758050444,0.42127676039682826,0.12500398219124653,
 0.6455156266623491,0.3891134495036237,0.5495517135619665,0.6921698674476208,
 0.6169388964110514,0.9715416177717819,0.6839260799996232,0.6650018850140738,
 0.8665726493905125,0.4434028558057096,0.055139207477740926,
 0.8837700552286507,0.18779661868881992,0.13074184006676137,
 0.5268572514692489,0.5216101242801631,0.6387929585116565,0.3689119711113793,
 0.7616375967088119,0.37294399509370024,0.3594095820377248,0.8325682330179873,
 0.18777126282201473,0.30364202261272843,0.161999907409052,0.9657879462190463,
 0.09076478738303417,0.1794226571243812,0.9974659888002927,
 0.007697819354890001,0.23532314717089597,0.041771219470496335,
 0.3633663251177932,0.39568410137816257,0.3224166795552018,0.6619068843351738,
 0.23555008167920277,0.9884100627130534,0.22768766785653205,
 0.0498180821392592,0.19583039644462719,0.3286251892267016,0.6405446158155963,
 0.9388938670126052,0.13675653247100705,0.8715132961649847,0.9400266181720491,
 0.24995498711465614,0.798845320086263,0.6477106494116606,0.14361920162861297,
 0.7632474674774335,0.9951248866280494,0.8827110161144059,0.649634608384739,
 0.3469277457064185,0.8156079675501555,0.09360128775090715,
 0.06914777327815769,0.8031209656730965,0.7671630623070856,0.6268679694555935,
 0.25627124801806467,0.7236680863011201,0.9458783084764089,0.853833854126963,
 0.40582736829059407,0.4447308605971342,0.8090569090752165,0.9124093382881301,
 0.9424474411749455,0.8613275134490853,0.8931772081564704,0.8536662377060701,
 0.6841333446386709,0.5877131089183375,0.10142049387803276,
 0.06455645353509365,0.13372527515636656,0.8209444157719877,
 0.9491799163540531,0.040353992702765296,0.031946746850430974,
 0.4828017275467892,0.6830185901763242,0.03192598111587264,
 0.33691940230139406,0.08664686150409451,0.21123247344320517,
 0.2117175567895513,0.0041996073158429015,0.9889933658025882,
 0.7341880358629131,0.5077800582256332,0.9762372165325588,0.19828833438681226,
 0.7765976415411298,0.9835883685272326,0.060953369797748236,
 0.41397948595589096,0.707621369571588,0.6415279576805306,0.8216020957040199,
 0.8395947222319738,0.957006629982421,0.9530084941102502,0.7972561888607069,
 0.2327837154376038,0.6442160579220103,0.029682036291711178,
 0.9840176052614856,0.41664160471205014,0.45642926706377895,0.633912062385457,
 0.5211124637511967,0.5051140085136855,0.6979283789282382,0.816889544973809,
 0.16506774747519304,0.8741826199314442,0.5587674704487067,0.7140019674394271,
 0.5101651296008675,0.16774641757859388,0.008009860520870271,
 0.40756555491705226,0.056700160278860934,0.6044337165905712,
 0.1669795220242709,0.4250431536634225,0.23423622335298577,0.7795920725844598,
 0.5359665743588747,0.2915032938695542,0.8922684899137338,0.9238939083092021,
 0.4872292960085033,0.7872695645635347,0.1470944482061911,0.6840466053393265,
 0.09998159429288445,0.06863588088039507,0.5318831176225501,
 0.8467927982134866,0.7574176512121178,0.7552181571268,0.561521808160026,
 0.9062119620968097,0.3366463583055954,0.19167376144532977,0.9436420194289793,
 0.7287790907843886,0.41579006211728287,0.5484679622022658,
 0.19548210956129808,0.8583793253197541,0.7633218228653899,0.5490014513945295,
 0.4936217775080828,0.7382642986738158,0.3517843831134443,0.09453283113090682,
 0.07345409711696294,0.19476005665745533,0.8060449357570956,
 0.2690619249552927,0.3150053447113631,0.05341453893069592,0.646906687272928,
 0.5225867040201049,0.7430392599153877,0.24706861270669167,0.1200460121450021,
 0.036864852779704416,0.5460540001099844,0.2942780464784758,
 0.8413863188021972,0.7083212114632282,0.4589535683516882,0.10048601396192702,
 0.7503449607063092,0.1433389451328677,0.007232710443316526,
 0.3557302700419903,0.7731782670199094,0.8273810542841711,0.15407112625193642,
 0.43357361230201763,0.8601649372071142,0.8979468789856786,0.6443341057610594,
 0.9901302031989323,0.8367633715479672,0.9623708262990892,0.34549443766277854,
 0.32243686071286304,0.6195200241784119,0.29776219316282215,
 0.8034097777645707,0.11981396231529151,0.6901505089643468,0.7427013944843243,
 0.6190381809257711,0.26110947079951996,0.4125838208252407,
 0.47357363595176194,0.4372127775524738,0.39203465272091975,
 0.7506211680509651,0.4470112176590876,0.41617430130792477,0.4448685489889195,
 0.1485135698776563,0.01652437807825191,0.6859246868602771,0.5816344796800045,
 0.6244765594001543,0.9931262745144595,0.26445566042441293,0.5874318182624045,
 0.9344071629950785,0.025730275440645878,0.2541025706922998,
 0.5898356189052812,0.977175673807795,0.3221654570397474,0.059580446268132414,
 0.30804608349988716,0.09564669942900672,0.25437283202159255,
 0.44559099450654327,0.7733814393478726,0.3708326197494144,0.4665932556312865,
 0.12269300704036934,0.6603208892433119,0.08379902717067178,
 0.44617217763993655,0.30780663199241975,0.5377365041096007,
 0.1310754127317737,0.3286031785045479,0.553633436106604,0.2758685682154475,
 0.30169239235275125,0.3230225606274887,0.2024073806593356,
 0.33901598268917843,0.7711732329600329,0.5121519732270642,
 0.13525677644351197,0.7998543786586412,0.3583578375635682,0.6207202485879237,
 0.926107864811466,0.43557142608747723,0.13190628117872427,
 0.39171734101861067,0.9598445952330179,0.3084043363221368,
 0.32317196260062275,0.24447003167371706,0.9276151331007919,
 0.5918228447515274,0.7845754152813456,0.9426346280822087,0.49910572292909994,
 0.5985302385891673,0.47482058701526364,0.5975115836137422,0.2767189371281077,
 0.9919298186005409,0.576571098246117,0.20086142515188032,0.16952041149020747,
 0.7737290670898471,0.6388987629206078,0.13559306605139465,0.4107599808586304,
 0.30748499340447566,0.7667877477726855,0.14813776950812363,
 0.04252339128648097,0.6923452068782461,0.7309186453260764,
 0.22335134384755273,0.4809838096119614,0.673710086874822,0.9939731762300026,
 0.1783654646559505,0.5702991107027832,0.603770036353261,0.4079572423069394,
 0.5889904567959678,0.8631094729542135,0.007836917839364688,
 0.027224353594193174,0.4958799716345518,0.12979209451889262,
 0.36583720637369754,0.3812194252225236,0.13337749519049313,0.551765955619644,
 0.5453273152962576,0.9480335503783597,0.13147898753263598,0.0862941055976516,
 0.41738235034074267,0.5088826218298619,0.5855120505431689,0.411622841603526,
 0.488079534319681,0.6337737468933538,0.5659345768895,0.31641029267684906,
 0.7697717151661831,0.14512944886272972,0.604461445532741,0.5828010874852112,
 0.3627272883216861,0.27503949385089455,0.6699219383001505,0.4480564376365056,
 0.5914107334156822,0.1906894917904336,0.30518983823326495,0.5924068180733845,
 0.06290007706763523,0.9997455819919717,0.2347405428895134,0.8948174963670221,
 0.3098989583787338,0.25504107381655516,0.30265786116329374,
 0.8158172313548514,0.8918522280374729,0.7701165818879315,0.07912012495380838,
 0.07004438818999958,0.3014700229723557,0.9150635065295419,
 0.23440549644524578,0.6513636041772122,0.7851172963006651,
 0.13464947329679444,0.600625963913562,0.1098810267865662,0.6076038913891278,
 0.07348456466559461,0.6027807874592218,0.13956659028139584,
 0.37203298082880043,0.6688852911291651,0.3499047851048457,0.357919463717415,
 0.08082282590226642,0.0889516560461664,0.0068724864303981725,
 0.4826227633245689,0.3966878617208962,0.8716588507084908,0.8441631856560993,
 0.4702773115332176,0.7086691669533235,0.07209283298729174,0.4772708382924308,
 0.20320867726881986,0.3632013860454978,0.7268798015584751,0.8514886640891954,
 0.4303979249623777,0.3081604450546296,0.776046507699325,0.4581107381005928,
 0.5633947519389152,0.47664493637279937,0.4988563245548632,0.8442444849518793,
 0.6501022892239237,0.8560156957319676,0.8452928758991586,0.7322624432339379,
 0.08035476329522051,0.9448870671626417,0.4941933453352365,
 0.16354555611463895,0.2852227272840071,0.6239382878644537,0.8839094475949469,
 0.8248945110937603,0.30298512545496303,0.057671002638585156,
 0.44746450409514194,0.9156969441629366,0.029695449721899192,
 0.06395945263897196,0.08174899230454524,0.5071934435365535,
 0.9663672843758747,0.22526634500206988,0.9899802656916614,
 0.09220017209203046,0.7451482355425733,0.32075531272158353,
 0.6526117182535996,0.06482973522084023,0.43279113088366994,
 0.44179812048253697,0.07714039277285578,0.11537382327882084,
 0.6549696240574091,0.9885210868503201,0.945203029409546,0.20780802881160576,
 0.7198443466900291,0.06131795388271133,0.21712434346575682,
 0.6986441288404958,0.0673813511524598,0.9332025925047333,0.496107883116656,
 0.5044266189069836,0.22510767883562877,0.6696461438653631,0.8360019433502409,
 0.8045866839006665,0.0035844848037195343,0.4563255341954986,
 0.5086980552878546,0.6602643508489644,0.6988827255176782,0.1798208982030136,
 0.6591567184780404,0.9777036005995811,0.39988028174045076,0.5218148089335214,
 0.5114776406145916,0.30856900827259637,0.6022597879022478,0.8702276288274059,
 0.22896484382774773,0.27170489190412705,0.030514415458936184,
 0.29820832377216266,0.7891051349109548,0.7589048416354613,0.9357656019292809,
 0.8955382882661687,0.8430996430365281,0.7974186388702434,0.5964686772376199,
 0.023570851814286864,0.6396111315613536,0.1540624588416355,
 0.8947348326669801,0.7745069768912259,0.24751063426619369,
 0.23672623137036197,0.8869826392547988,0.1466282847351308,0.9762244175995005,
 0.7780307574836997,0.5234509455345547,0.995461691053188,0.05535889803218197,
 0.8868795222108189,0.6671616879613357,2.863398693270236e-4,
 0.5051754382173527,0.4400684994091254,0.5669086376552194,0.3440457054631466,
 0.44229999179975876,0.8622894173771497,0.39440546661448095,
 0.8495752024749903,0.49327076900542766,0.7092219652561387,0.1073549842032262,
 0.42770082423652844,0.0537416025054126,0.4964707109152712,0.877645966157417,
 0.8564574400152419,0.4605255709795598,0.9086871611469589,0.5110064996616542,
 0.1664274181550679,0.8086452519548363,0.9866641856855785,0.3313884744852701,
 0.07546786180472942,0.09050245632902554,0.2193956024498266,
 0.19587000719968184,0.38696471031880264,0.992444456563536,0.750738358715334,
 0.4000713773832848,0.15901542181826356,0.11128039290896774,
 0.9301702820314908,0.8181843570582621,0.12926071487978286,
 0.19432496494738483,0.22440496375327812,0.8672901009425876,
 0.46782957482886145,0.026193131007438253,0.21344314272171205,
 0.7474111487771927,0.370924342021103,0.8068482640442606,0.7218126488912076,
 0.8579081690416401,0.006794367065609208,0.37295216924104835,
 0.6621676752054982,0.4241325806709053,0.9555318233129309,0.8266591536112112,
 0.35369300834100614,0.4475625871783421,0.1514313303029754,0.9152795932700982,
 0.9980858712926849,0.4298240514229993,0.015757353937513208,
 0.7109334667356124,0.8558941864387447,0.05863356848110801,
 0.01473528183606565,0.8800207342823423,0.20053576758965053,
 0.22251774425716775,0.7925171330295958,0.5857784863676667,
 0.09189918945299014,0.4271830310605311,0.8512924799332342,0.2594979432736023,
 0.3396913655657495,0.966814942243634,0.5143706614265717,0.30583844512029146,
 0.5969702009788435,0.29350397302168973,0.8040011020232397,
 0.08945031015978921,0.7146422610277545,0.4030150792481958,
 0.42241381860459415,0.47133122290794716,0.045317585579898956,
 0.14651303080944378,0.2853815955019312,0.7749569085184691,0.8667942508844779,
 0.13947291845819443,0.9184650689082824,0.2690139080910299,
 0.12348793488916021,0.8672257894910457,0.32122389813043295,
 0.7571377441408979,0.9107416560411297,0.30121813102958284,0.4953571025133823,
 0.42783020862728194,0.047252607377762046,0.4042469797238655,
 0.8668184352513608,0.6386673403002499,0.005917803012569611,
 0.6901948692915036,0.4174288883894782,0.044477822077709006,
 0.13437853577596304,0.8398691949540791,0.8364001458051246,
 0.38397822266892856,0.05171415989199135,0.504082035336068,0.7922103199392979,
 0.7125872840806503,0.5934132878588642,0.24520858650622968,
 0.026036666654417395,0.866696380672187,0.022906148414230887,
 0.5439875425615732,0.6550826619067712,0.642104113140884,0.14882419185579632,
 0.502332964222242,0.7039567767644049,0.7973187253452046,0.2843093785119245,
 0.6388358299070522,0.7559692876023366,0.7151073456485911,0.5334284007491753,
 0.446050300148885,0.21540356743879607,0.2879921582337529,0.11512902645904655,
 0.4256610746376924,0.8799032040081465,0.863906797523197,0.9744785004995942,
 0.6359759560934332,0.9618994986511222,0.35823778080808566,0.9278644008871171,
 0.5277737706009383,0.3768121884970207,0.5991599586075883,0.3756718245406252,
 0.11334119824623712,0.2355231789772143,0.27837304043859146,0.997812586139012,
 0.9834631529856696,0.8730828837828284,0.22485765083590392,0.823124183131164,
 0.7885765358382593,0.067153190316109,0.7553179559856406,0.8922413686585692,
 0.8746897827184896,0.42702104515280126,0.9322198884415417,0.542971607925687,
 0.6155894304966427,0.926056071135386,0.7248615262280274,0.5074140580170476,
 0.1999573072994908,0.14986040019428903,0.138245555288804,0.745031493641984,
 0.4099751249278284,0.042804473054934844,0.9822157934883212,
 0.39848745996673574,0.8891038106075079,0.4060930465560161,
 0.32197488115929795,0.6119046636073523,0.015474438728620976,
 0.32504014612941734,0.9216071279903755,0.9143077121133295,0.2402547012645191,
 0.603369863649823,0.7648101636608672,0.940201037910167,0.6466983724990346,
 0.41916620380989744,0.09690639711073523,0.16774732274173554,
 0.6736470137351173,0.812970872973021,0.6150811507305018,0.5831665966336569,
 0.23674494700421334,0.843940776741795,0.4628659262132748,0.25904880243361517,
 0.7821655452797331,0.3594409286039779,0.5131558848421837,0.35448153159639884,
 0.5906765982344587,0.17962422668610023,0.8471978940942979,0.3289798054734083,
 0.63951019480419,0.9387337686373082,0.6357416297726495,0.5582858770412495,
 0.794605864431353,0.4000935309470022,0.5460763680411156,
 0.0035553134484174187,0.17716631525142035,0.7874934822565185,
 0.7142120418460689,0.6312036085061623,0.49293333972162034,
 0.033525188830661024,0.08491301177795518,0.616576594588079,
 0.4951869677347531,0.12716781809169553,0.9385509756908249,0.600483282252885,
 0.6286149480346135,0.14226020340631873,0.5990840568656636,0.9346487788463669,
 0.45795535299664636,0.5473360414914,0.6545305401268464,0.9494843230031511,
 0.8446928647187067,0.49211610883323553,0.3532286607709897,0.6885896086415364,
 0.13527576590561463,0.5963217126142216,0.4168223612126941,0.7637007558785367,
 0.8401498754652498,0.3710164501989007,0.45113916632967965,0.798214903817608,
 0.40464029011793956,0.8479649908981599,0.29579205829344923,
 0.4020470445458243,0.16378911927637807,0.4210093988714779,0.6538671131863094,
 0.6183792456691661,0.3841579199784353,0.5609779907706329,0.9922048322076472,
 0.5415984240732713,0.837667168965503,0.7476058514391296,0.6080886189298809,
 0.32661805683766953,0.28399502380742037,0.720371902120231,0.8773044764516456,
 0.5939421636756987,0.5378219729948299,0.4975408482913185,0.21127912347124067,
 0.6052323535083585,0.20875011358695905,0.6993639034579567,0.9876750667884774,
 0.5120712741229548,0.49808628914886466,0.15019869932813468,
 0.12415991170617102,0.5371535759950957,0.13165311017970716,
 0.8442989511234799,0.05268450152802073,0.5082313283826624,
 0.17971645774657607,0.3062838669645802,0.2538777998285877,0.8478829168504045,
 0.021885041185013554,0.6099551623257842,0.28375506416382656,
 0.8362284147083598,0.6975521245876384,0.8370817946424788,0.8575403201508026,
 0.08478179291821619,0.050613524156227996,0.2379665061519154,
 0.42257543536242514,0.5499509137500471,0.6974717613285116,0.6240988003117849,
 0.9889263190829902,0.9740344180737194,0.588283359852978,0.7389566998446205,
 0.6584823243426869,0.89294153843488,0.16492494916150036,0.1880276228937492,
 0.9516030740501997,0.5071858016705592,0.5994151090289809,0.1925420637577211,
 0.4011192272952109,0.677038339372195,0.1726494612408076,0.5346728257409253,
 0.4091043193255781,0.012088986407965185,0.778996025392637,0.7217214232050523,
 0.25571547231819314,0.15255051960817045,0.8334072228916017,
 0.8202554279725519,0.35434174836943666,0.71537764135758,0.1362719113067158,
 0.8962589102490429,0.9859169279943232,0.8974906571640258,0.9155350924295139,
 0.7116443543234946,0.38451746113075314,0.010060956892744666,
 0.07719617819532487,0.49108524187717084,0.5633893000280094,
 0.47494961018308113,0.6652426166440422,0.38372426777696034,
 0.5234889692272315,0.6180920847085561,0.6024007574053825,0.8430023227160192,
 0.10522098360848287,0.29422491855280186,0.7850911116569159,
 0.19976363631728877,0.5174902421332999,0.425159500964856,0.37919935981960695,
 0.415156127612217,0.06965812584769338,0.3074232099518318,
 0.048787172290547076,0.6014384677537634,0.5280458005759526,
 0.09331256522604015,0.8131151614505336,0.49035584282300293,
 0.43688447163481314,0.1094264106703201,0.786280034057925,0.3336235822758231,
 0.45818416177244004,0.3737125584553803,0.8962202571042198,0.1375913340244408,
 0.022829506959367363,0.1705201487257062,0.9498492619328537,
 0.13281842374869657,0.3454851828944778,0.5754053140353218,
 0.42996523574019685,0.25228997362179995,0.9751293290323142,
 0.5931619996034334,0.3083022118755143,0.2498482493756794,
 0.010605581778624917,0.6632467757797075,0.009380282053932487,
 0.22966145966184448,0.8797984656756188,0.8559192130343855,0.45916291078342,
 0.6478605470308875,0.48782005297770825,0.17787618132595706,
 0.20322044187969535,0.37623790967291504,0.6099535339221303,
 0.10505426427702202,0.5852522252889241,0.7056691979628608,
 0.16542845422623875,0.2149679935778559,0.9447951705823464,0.6383267720809765,
 0.80770573980258,0.08289470354675144,0.623869100618336,0.12808992809137798,
 0.16489623504816509,0.1897749842337999,0.1993343076740991,0.5743086795026822,
 7.754884206671031e-4,0.8506431332683031,0.5836140133601426,
 0.35160826739134343,0.2801608687506343,0.7535089759091654,0.9177870056433411,
 0.8263815807203706,0.4327879791077751,0.6128713631612303,0.2797949494156172,
 0.7392805018933153,0.3918569896174183,0.4976258952435141,0.8363250897269066,
 0.2894823980157848,0.11520171961015957,0.3107952361788304,
 0.0060132759462259155,0.5333066682738723,0.007851426731952826,
 0.7818428016939232,0.6801530965981694,0.46510713028892037,0.9379258187501516,
 0.9618471625912596,0.30271110920415234,0.9682913839807018,0.9077451152866332,
 0.7665322129316187,0.8943251539620998,0.8193126478531735,0.3385434925711862,
 0.19819281877952644,0.2635788209718132,0.20485481415154594,
 0.20506957993759134,0.9440492952851913,0.8704757465252887,0.8821502436212992,
 0.4444507250335863,0.480950271808781,0.16198847718387488,0.7132759562879558,
 0.5120534453434927,0.4515563682684164,0.9158199973574448,0.21407922302354865,
 0.14866205229454488,0.19567778055038887,0.7315839836862921,
 0.04454824125584522,0.978870713922872,0.2587799631921722,0.42489544787978717,
 0.5026404445452154,0.20416899668622934,0.6750071608845604,0.6047606715795074,
 0.8778709706140986,0.24357891863298475,0.06735988204449361,
 0.08444102180118951,0.7568729637140181,0.48890448293567745,
 0.004170091483139293,0.5982646018043448,0.5544337610478443,
 0.6102748382867716,0.4299498296995634,0.036862591144013024,
 0.8230699028452886,0.9050019038772721,0.6173808757844351,0.918079493786787,
 0.35140593520189356,0.7191743363932206,0.29260860355945895,
 0.2332945850487539,0.8992030108575488,0.011181445139113677,
 0.3895453446677374,0.03400896423401367,0.7801507681545639,0.7764322734975195,
 0.3382789950545855,0.4767057257471492,0.283350238728587,0.906969189774131,
 0.4822693546324007,0.9570779095456274,0.35960011359219046,0.2017061636983608,
 0.27743028636994893,0.9809121994986227,0.8174526902507077,0.8264806189143833,
 0.4917459266988048,0.8138988485525289,0.324401967694686,0.3119486791840941,
 0.06037655172280554,0.33345338416537085,0.5819740047328984,
 0.17812378353797564,0.6977642717409356,0.44372341382544844,
 0.41161398378773884,0.7922739756745698,0.7768262808851547,
 0.08562113655497106,0.9665930097715165,0.22777782473864028,
 0.8544667547869628,0.12383228029116855,0.5440092889960881,0.5683225843824261,
 0.9459311948063813,0.3649382067306922,0.1943891720821449,0.06532030240966369,
 0.25756399307161215,0.783197420224908,0.6933239470489359,0.8310134514460396,
 0.19495272871406055,0.728922737718036,0.8849636195479371,0.6507588349840939,
 0.11337715222766886,0.573936595396837,0.9733589842562411,0.592994854507133,
 0.9406958375305983,0.5369093275356817,0.7902382224845484,0.7906984729140207,
 0.8920503700740784,0.8456066338784404,0.29338059876601985,0.9098267180598392,
 0.6330211214812266,0.547134056373753,0.6169019103610704,0.31879698581390814,
 0.48664328486154584,0.49704372897363824,0.8945392947042515,
 0.47042193644560193,0.33671321678500776,0.8709731733205208,
 0.6038761898063578,0.3078212719159279,0.14651276185717044,0.7641239456089498,
 0.3987377970027637,0.07464432078413186,0.9810142731073818,
 0.18865978704161113,0.13598611026183705,0.005117667224866729,
 0.008665562891668421,0.6063827574704326,0.4925412901261089,
 0.3625233674388544,0.3934425036478837,0.6360726868414731,0.4620894363301653,
 0.0684968150718247,0.6903548213089036,0.7379099268342856,0.9582177117571994,
 0.07817407440455948,0.08178067393457245,0.5575621028050751,
 0.2231193975645126,0.5329622051101865,0.7849584147518812,0.33930185009676095,
 0.10545767732054756,0.8968805894464708,0.6934267294700964,0.5873811348594363,
 0.1830719358491406,0.8535308444598924,0.5920206624906943,0.19983218053588925,
 0.12226065044102818,0.3294267612120092,0.5530379284383568,
 0.36665270679270756,0.0801977368419361,0.16213047754072885,
 0.7608890733729433,0.3548038768191686,0.2438377711382732,0.7661232584702811,
 0.33954308132377997,0.05751326468449591,0.8227657333599748,
 0.04209536069979314,0.8336592830213295,0.7486113662373007,0.8577803128198652,
 0.31077283455194704,0.8455723628047874,0.7421968980390891,0.8267613718709593,
 0.2792542200016548,0.4972521770335111,0.4907218328283943,0.31335363678158,
 0.5304897031150501,0.8615749307872401,0.23862992868944732,0.8608124144880946,
 0.7009660619609509,0.6596307344520214,0.03242439028721866,0.688176476022011,
 0.5949685452892953,0.7241199433247575,0.3818352255129114,0.780486835454041,
 0.7180535222668251,0.755277191559135,0.10182946330071052,
 0.016409952973779784,0.25878606664105297,0.5745519255003236,
 0.4618458953092608,0.22312190442103985,0.780385248947404,0.5476753337090939,
 0.9522358123942172,0.18352998659604725,0.8840663128191215,0.5540948821622353,
 0.5440589648061528,0.270637918792933,0.00986752500333532,0.21238906301639127,
 0.1297633058814115,0.04863543187426289,0.3799170441173597,0.79379148031147,
 0.7157400789522236,0.6653485005613787,0.27138977535517217,0.0761806055259091,
 0.9512152620656729,0.11383719731711073,0.578062550644103,0.13914079012213698,
 0.3555105688119321,0.1350488036832651,0.47021245476460616,
 0.40974747909323006,0.3441993031863474,0.8433716408397288,0.7026141500026404,
 0.2198708170113186,0.0060435087330755755,0.2913416418752721,
 0.7043631063420966,0.5907163257195004,0.9844380273758279,0.6088525716978729,
 0.5184794437751166,0.03747122998294483,0.9397664685849749,0.7673612227927643,
 0.5537937890749993,0.201280552070348,0.5607621328981479,0.8239609885155014,
 0.8267789235665804,0.02511836737060158,0.4455299251406699,
 0.06670677680884585,0.21697301582668216,0.6553792521797537,
 0.3588603506386451,0.18567000346282725,0.20355044849313397,
 0.9087471942836569,0.5281008302814759,0.7378529721652137,0.2056585524251694,
 0.3714920316501824,0.9546584947948533,0.5240616790431446,0.6134367377059095,
 0.698930460646313,0.08044955825768418,0.886310874135723,0.6683961235547666,
 0.11057113666689633,0.38608307316388113,0.05912115522076378,
 0.9209119760263135,0.856625608438508,0.3105033772446536,0.47754945070142296,
 0.372745589534365,0.7327179086060092,0.552804892331486,0.4656289820321823,
 0.9617038701193877,0.7320707977446078,0.4924676636088533,0.10777833889790311,
 0.30723073341370255,0.7415834098791924,0.12126989984975567,
 0.25607389570249783,0.588815009243351,0.8922789635010016,0.7724309841107168,
 0.5814784301055281,0.8559922743841312,0.5139467438573564,0.39945930195305535,
 0.24183642825281204,0.22386232517768523,0.6217266281710453,
 0.9801707133462292,0.26917101916667907,0.18305064771703972,0.512715617240708,
 0.04772041197678978,0.7931938016515921,0.4193670777131695,
 0.04227161950055136,0.38391420798405884,0.2781403347219249,
 0.007602030240302149,0.47715459007163197,0.7657616385073072,
 0.4310857398559913,0.031455427472038555,0.7151418436090468,
 0.6161155527446295,0.7641447039785917,0.806529666801035,0.24509438856319943,
 0.12529762708757086,0.9557246492557971,0.5613227184303468,
 0.07427100374056872,0.2145840553924001,0.7073387935502804,0.6938609112414996,
 0.07693360036640562,0.5375542742449961,0.8458502217944357,0.6082979603573974,
 0.8596943816273567,0.9254926339259713,0.38778127959606024,0.6503173197669289,
 0.06785915180342195,0.686273686373587,0.1686268912736224,0.5914529668120698,
 0.39965346145234315,0.16694658366392368,0.8393055288813984,
 0.9364207467675407,0.7905392527052209,0.7564685115825337,0.7922158462661403,
 0.5546114233425943,0.1803826091564964,0.6092138450089887,0.1400738159473408,
 0.27795714379244596,0.11533462016544915,0.07446438681073475,
 0.8298816101688625,0.1134723543926418,0.9990830687467738,0.4759356385469431,
 0.8753673714196977,0.5326271769992054,0.9581017037996863,0.9883261103917691,
 0.9215065540587591,0.8991358926934221,0.6100977761010912,0.40048202605097283,
 0.6413376585040315,0.2544153603318575,0.09839339503783684,0.9953898238018941,
 0.23603609914781631,0.27632360823590973,0.06422265672926875,
 0.601061041441343,0.4024298714352089,0.34677656308197047,0.9639485602916577,
 0.5900163481556614,0.7732934524038168,0.33010010335847095,0.9499502264597969,
 0.9999916926740395,0.38447921631599136,0.1299974700543466,0.1445566865327662,
 0.23426086170154237,0.2668047807494388,0.36817663363945297,
 0.7195352239313956,0.3330652693415521,0.5691393522890243,0.6655211021738304,
 0.5764216802720915,0.736947154415421,0.7574750701611345,0.9973026984371346,
 0.16263320524578695,0.8087757207977386,0.5277913888777742,
 0.0070979845759849836,0.12151994882538808,0.9138823147392742,
 0.6058244863007274,0.4606285739535356,0.644512922280141,0.06162317685797691,
 0.6484719352434742,0.9672687437671091,0.2052633215542261,0.8734981863242705,
 0.6927861072102827,0.502134931253533,0.8594294522514117,0.1652095048502441,
 0.24697869533570982,0.8210824438474776,0.8443537857216318,0.8630433340512671,
 0.729670240889184,0.49364637519744303,0.0628328825783826,0.3855588010115433,
 0.8196650297393008,0.43394609326207967,0.2712368556870183,0.8118138308477139,
 0.5588318497074739,0.8717349359210392,0.07182506683072276,0.8509162789431113,
 0.8671485456498815,0.7005896138051164,0.7249960220216749,0.2845265593271695,
 0.8830286561014261,0.2657473486288434,0.5003788678016995,0.9600655479886546,
 0.7650873963207268,0.9049490715496952,0.6072957147425828,0.3593042410462256,
 0.8101831179489742,0.7152307536270978,0.27070204302284717,
 0.05483551585344337,0.7292308871985422,0.3099231056339634,0.4977098803793669,
 0.8536450042061269,0.8678716819406503,0.7686548200521894,0.08586321629008409,
 0.14200540785310267,0.8104386799124644,0.9733209614383311,0.9290476251399123,
 0.8970599605860322,0.00757917832284849,0.24259308311775674,
 0.6334812392176695,0.889719161603576,0.08780476109620194,0.5226257252189285,
 0.3138811736166224,0.5321808032618285,0.7164805514971595,0.42127679111043304,
 0.509772014418904,0.1954060576917258,0.20497063845022478,0.23009751102642317,
 0.41279964957915904,0.6246871133329605,0.02914714829250975,
 0.7756385468203033,0.42643252436768686,0.4366653811251586,0.8476400995638508,
 0.23988685407343313,0.4201630179070912,0.9556668160597918,0.6572948944825714,
 0.4669162237262664,0.7519647378918731,0.25189014514490626,0.6716637301427859,
 0.28526240945222037,0.6952756325087768,0.9439530787299029,
 0.06305290178434275,0.42689407203952856,0.5384023286183457,
 0.6162068090053345,0.8317648916413666,0.5521891035596662,0.22468199765407104,
 0.31079299167593466,0.49168240838962496,0.25458809339182253,
 0.7398202151131024,0.7151728920343693,0.3668831636053518,0.8707976103579158,
 0.5975140486663179,0.12632478772149658,0.6613729229813278,0.594278166931686,
 0.8715674045982317,0.7578712879443031,0.23845919699830576,0.6206517263491221,
 0.06333138789073534,0.17371272413787198,0.6959522416322295,
 0.49587045106722105,0.7108015716689167,0.5884570375801865,
 0.04379903679958852,0.6655889431002329,0.8134214058010215,0.6083458268796152,
 0.23098698903588866,0.6235472633281155,0.25614754917104166,
 0.5690745293932358,0.07343342244571383,0.9893856563930192,
 0.14505971132665252,0.9600650975027434,0.900592774590862,0.7092189807791036,
 0.3628790938758166,0.342601564894478,0.06956978599269892,0.04872449226791464,
 0.7894452091553397,0.4214608806853126,0.3505732327602389,0.7017742730209933,
 0.5141826572143766,0.121743194500337,0.031216923159983145,
 0.39086035128218066,0.5537281676113435,0.7989205906824952,
 0.31446844328511836,0.9660843555823634,0.3465240767635912,0.7379601364740856,
 0.4741325158750461,0.508991395045793,0.9867458170786269,0.5926837567004292,
 0.8736044616736774,0.08006232083497056,0.2185326039207819,0.6500617679320674,
 0.364840680800224,0.5495992748951608,0.35800287432850386,0.980248953448789,
 0.2028285354253172,0.19173681121441533,0.4336866363538251,0.6005757423704098,
 0.9936537994636818,0.6630758909541669,0.8951962189579666,0.7620513041017124,
 0.09579235237613515,0.3260770081996125,0.9267461959215197,0.8464084693687122,
 0.2567831590971824,0.8266094918475178,0.809227627199107,0.5597835137139553,
 0.6238397963491985,0.3458818649086607,0.19641269637190895,0.538482022923802,
 0.6273692023837522,0.824711320481048,0.1494348784754851,0.9972250865042239,
 0.6888427419648991,0.9376717911359722,0.957264372832727,0.26689208933798414,
 0.4805750409043499,0.5002917766647464,0.5485766444230441,0.4361442679116422,
 0.896020173308965,0.7739903443636584,0.6026427197456979,0.5380627243525145,
 0.8286407146121165,0.97671928387173,0.0980351745711383,0.7678047567935317,
 0.3864434727664362,0.7972656811030461,0.8175694909898625,0.9221443824036035,
 0.5871739395322825,0.5757800608330957,0.5365000064868073,0.5775550529034058,
 0.49150446180254814,0.1156943376838594,0.10825892674501925,
 0.1751869046666803,0.9775016692656324,0.27638203803713846,
 0.44050256323016435,0.4676175656435897,0.46187926072188323,
 0.5049679069919574,0.624477579171251,0.27113561148664755,
 0.024143629329147154,0.2669063460702805,0.8459065835646388,
 0.7945545343227983,0.7541662346727713,0.5477570159394293,0.6824411352279206,
 0.8677541268138568,0.26759611009217266,0.1751228855664515,
 0.36608470756208644,0.23024620363758785,0.09092803897719515,
 0.43281271051840475,0.9455415935773934,0.42004538784312606,0.856829424011166,
 0.3922787489746361,0.3422471395635167,0.44297117556963106,0.5024628667778609,
 0.008550852589371716,0.9445844475698953,0.40714703288061205,
 0.49102557630154875,0.20696968928677717,0.1192661308346562,
 0.04690581973698105,0.6776140673789786,0.5737746664835633,0.9970245126692406,
 0.06014347579254853,0.6308455348486928,0.015000395806200606,
 0.22196554078515485,0.7661459760195684,0.49424370892378267,
 0.9458356887883129,0.1996600316734013,0.5343851098417036,
 0.019771087789436703,0.35298595361511365,0.07831328337607302,
 0.07126539452929848,0.5327761697331923,0.2828145826824443,0.4487156875584859,
 0.3989538890851454,0.9734741160781539,0.6505847501782127,0.6200338424067242,
 0.7973485772310072,0.18323232077442053,0.20108472740970895,
 0.9264218890591475,0.8532522466336672,0.15594128043487399,0.6364101942820317,
 0.2603147796812102,0.8761964822682741,0.12204727600173837,0.4545232039598359,
 0.26439830962985134,0.38073012199431977,0.18755720768735606,
 0.598182985614038,0.2186793274865947,0.7665899413412087,0.33931067008343985,
 0.7023381850506898,0.493806814168387,0.20791517831501571,0.39026518286653067,
 0.6600981693847734,0.8535635959947314,0.6507835566653885,0.359682375547532,
 0.9550807536741609,0.8253813314428686,0.25691037733765665,0.8936925692000166,
 0.8518717929314121,0.33304164050055984,0.8738928905088473,0.379320620076974,
 0.5510061362814371,0.13969911695270587,0.39159929258987203,
 0.5495139683323642,0.2779651811096404,0.20275160550023585,
 0.15676250491220356,0.2695680059158584,0.12165296402417591,0.512049458316046,
 0.8347303354028446,0.07659223322061459,0.8488076105765658,0.715532220289038,
 0.05997627119191418,0.1098294803405343,0.7852069162520436,0.9137097766110849,
 0.2545531395537528,0.383882189369652,0.15960641229459005,0.31721019808571804,
 0.07056858797756793,0.7906805988874688,0.5917638134073253,
 0.08332135433575305,0.505685959906908,0.15026048145046345,0.7155727165588459,
 0.42198685358235544,0.5266947564532601,0.63631427099525,0.6722607634493393,
 0.9393130046877898,0.8628945758089582,0.15450960440637673,0.9644460225078814,
 0.11503248919717646,0.17975176841584595,0.11077552800430923,
 0.23398647321152344,0.8952922293057158,0.39256065704415755,
 0.2973996704332995,0.949771378033942,0.9625347372155246,0.4555366946130879,
 0.27071006986051427,0.09549947547100823,0.9885325187754739,
 0.0640796422161174,0.5254459836645508,0.1500219786649043,0.8785734413105338,
 0.040539175360803203,0.5510206452716311,0.49915721270755053,
 0.5542872712136139,0.14124186550822126,0.92728855941562,0.8788216145925866,
 0.09943482849060103,0.5830117685387202,0.8020242229571865,0.2119081672239892,
 0.9312016595722403,0.890968333565311,0.7951897119911534,0.4370720030324522,
 0.08607958929048598,0.0017987733095757852,0.28291229996568446,
 0.8210925636244515,0.37367692712987266,0.019548650160225334,
 0.6203679208308688,0.8179818681349874,0.5098077267682556,0.9130047256377621,
 0.6263302137007197,0.6903018346562607,0.1981563513390574,0.8919429243132877,
 0.6628593763074909,0.21710346971419892,0.5939115727225648,0.8708030288999014,
 0.7408886440253807,0.9713725436092505,0.2965666338741595,0.6927813042095583,
 0.7224230943077072,0.5044435380274676,0.8100665019999516,0.9729878808169585,
 0.2925087876417115,0.44468342483411827,0.5210798503126075,
 0.05041830176027662,0.5672382511755191,0.49299474443659264,
 0.40140691409901696,0.19193337160334267,0.6906183866012929,
 0.20934310332123418,0.5377434419498057,0.9426744144851079,0.7125813127649213,
 0.05354506348115118,0.894135764596956,0.7826817205351162,0.6416262295741976,
 0.7289379051093337,0.8413316438490717,0.982980895740192,0.16347234305477398,
 0.5544503460373365,0.13556559703922655,0.46016473490857845,
 0.0942696745154441,0.4204129011718667,0.6604603809763748,0.4803461984070999,
 0.014747570076317107,0.13754582379758185,0.22044157349840343,
 0.9373773166070746,0.5385532647502865,0.8559621869239428,0.5264094914167258,
 0.6946769252525721,0.4855869121254559,0.361691919614505,0.4955708007464168,
 0.3184618494605239,0.22562196034239868,0.19753396216830643,
 0.7145117907630946,0.1495691060212735,0.6372382722590944,0.41494450661075877,
 0.4953657614727379,0.8620880249529232,0.7790828972205336,0.246963163064163,
 0.4442666093241949,0.5576203033419357,0.1501240208561998,0.7112170267376655,
 0.032546110774244186,0.2548231412836026,0.8909930117328376,
 0.04737163387812271,0.40858298773655655,0.7193985460334279,
 0.8488695893055921,0.018080155844534596,0.551816054702659,0.1869203241713685,
 0.22413627389285595,0.022130620892083064,0.7024115805437575,
 0.592193692333476,0.17656612087466483,0.18119991371620414,0.2801085006765859,
 0.31860009202607986,0.3403125513955909,0.653111960736307,0.18131488250655026,
 0.16942950925210598,0.9512565633580385,0.26597550822878746,
 0.9569977449566369,0.5811878093140574,0.9418192115572048,0.6609490370176995,
 0.4128341891089029,0.7479193135331433,0.30162573252642866,0.926712242942798,
 0.3213498838730017,0.9945658077696397,0.027166338882034458,
 0.08559357430663384,0.7518089217127714,0.2328335225495184,0.8063927642259232,
 0.16719698112738524,0.6785221155320561,0.06450855918879017,
 0.7592035844350247,0.33632362109086034,0.8086770260444216,0.8745957842069311,
 0.5154816659930339,0.29053841337351094,0.8407017057337457,0.6965797267241488,
 0.06450610476915886,0.788241020151549,0.6804330283970059,0.6311906506685114,
 0.1601365601120186,0.42049322265910694,0.7874144413722601,
 0.27649983652322385,0.9970882484276526,0.7060371747515157,0.5570563132376455,
 0.1295493372910601,0.8515654744532224,0.7349052154032799,0.08227579020503417,
 0.3948820149954847,0.4249187883583172,0.9200659622141393,0.8952506542175552,
 0.9188414600741494,0.293649999473669,0.7265764416588246,0.0404460964395138,
 0.2812345778830734,0.16679356241537602,0.7015875612432305,0.3674996291090491,
 0.18951622704568583,0.834735315690712,0.4261103676186794,0.9799269143747663,
 0.909253315206353,0.7592874146687082,0.12987696861646159,0.4634068462482861,
 0.35025714188451806,0.8232679134514567,0.8274508325297677,0.5039642757431334,
 0.6135650102967715,0.6727016765244979,0.29244154474204853,0.2621545701063258,
 0.5275027409263526,0.38700381691174834,0.7146899451613563,0.5755228323448485,
 0.3487699987570765,0.6069998902775645,0.566360280624341,0.502202562262371,
 0.17084346242851456,0.7792803519531891,0.5913896766678773,0.7453955638601756,
 0.24156343946844383,0.10379790669556088,0.4485210680375643,
 0.9035749024199636,0.9884664085124519,0.3104270960735487,0.49095263717971505,
 0.7930163228966965,0.04759611718809853,0.17257168167833392,
 0.18142429399733417,0.4968630958680249,0.8389044661913743,0.662247694382637,
 0.9665421765731615,0.03732052274623232,0.4436824580782015,
 0.21797298604703097,0.9041247057463027,0.30307794283432776,
 0.0036752856690398517,0.6653004935655792,0.558846412967447,
 0.11499645040611028,0.6440974739080351,0.3462295875275073,0.712670544783409,
 0.6408801739806438,0.9615744615328895,0.7851951164810757,0.2909261020740117,
 0.8818653365902872,0.3416325302596994,0.36411563383011547,0.4248417122061394,
 0.3774096773012581,0.9185427631447558,0.45086890095628407,
 0.013942191185379427,0.6605531081178572,0.4955552598049755,
 0.1983136084336703,0.1348374400435478,0.25232048463954126,0.8933853873043404,
 0.6944314431394208,0.23106863458779447,0.5387610891719781,
 0.06598520076893422,0.13479118157674574,0.6843273487195175,
 0.37780553735783695,0.14651207770194496,0.8652234464756808,
 0.8820188255846418,0.5990892472382762,0.0011139829538403534,
 0.9954859042821961,0.7732813387162805,0.32666974736995813,0.1514718923480528,
 0.015802328097400564,0.6233119816827075,0.7597019425199141,
 0.7410343060238469,0.8780413705164594,0.07725722192327666,0.3529865221603231,
 0.927103746651539,0.8866219147250213,0.3655594699836181,0.005997130225304126,
 0.20622215782656672,0.7095218718358882,0.6257377855524263,0.5600996983955847,
 0.16915757722669633,0.845119306650089,0.2645362196149019,0.09400585910234593,
 0.41961185411074553,0.9206042955791909,0.8588430760751199,0.9968684722688633,
 0.6116802475387383,0.4360736054196648,0.02016090302944462,0.7046339501380634,
 0.6866663949019872,0.4231889318066444,0.8002457336282461,0.7578143995561841,
 0.6955355960702552,0.25790121465083793,0.8916529973407396,0.7750069591687087,
 0.6653608731763089,0.08323757668441767,0.26323122307992874,
 0.02522674274645531,0.8276605904739043,0.18596441644898742,
 0.10699141675457213,0.2022782480343024,0.2549735262991052,0.4304092819756138,
 0.7547962979965055,0.4611331148173822,0.4024555415533345,0.5337062704573088,
 0.12694958882521623,0.4860877010188248,0.5346983683133114,
 0.09825302097506361,0.4094304847180016,0.237237679313253,0.9829869086136301,
 0.19705479427103767,0.7231784893984964,0.2918145911488893,
 0.14240869837504522,0.7833653210911694,0.022931598760945415,
 0.8928448359791819,0.34457366530906897,0.8155544382980785,0.6129462500074768,
 0.7578850371090435,0.611684861355614,0.13685292086864354,0.9550160904118045,
 0.776307525629008,0.6671515244671874,0.1091182959517204,0.5618797668252377,
 0.10736301419822014,0.6971515687537785,0.09225589545655699,
 0.6316848100704975,0.4681382819128742,0.8743102663558239,0.448632137840548,
 0.16459732177976105,0.558053994379135,0.3834646998282112,0.4712994157919691,
 0.21756782790295648,0.23410660509924963,0.2545839955975008,
 0.2692277493106684,0.9230392574769406,0.9759747909421166,0.20415784841569629,
 0.011819687197930029,0.41011146677242516,0.6682835461916087,
 0.2666825425491781,0.7681851437269809,0.45553306707193086,0.1874522128926852,
 0.734572895973377,0.08515917975795972,0.0014258210551946204,
 0.14773678967139636,0.17235843475854173,0.40739180989541157,
 0.5931739109749481,0.0040147142452359175,0.9936240872223461,
 0.2676079692224287,0.16366341758964353,0.32873686595501295,
 0.6689790326596067,0.18041270414145094,0.4892667614907671,
 0.19522541215014844,0.8697101453195213,0.3923996655510975,
 0.23217399463830846,0.892880003890137,0.24648238858040972,
 0.30425388176246626,0.6615492036393504,0.6772021640269155,
 0.27391734395950695,0.6767525538422566,0.09370044484465623,
 0.6092781779197701,0.5317793904327589,0.800095562274286,0.1979687278857104,
 0.20701983967997883,0.5409679258624437,0.6516654899398707,
 0.29279187289583186,0.9263100328881185,0.6784967205242053,0.6809715428721153,
 0.5055440576038185,0.14813908659756148,0.32337355896836906,
 0.34967380420081096,0.15918397794325168,0.037907377482871873,
 0.9130146905811999,0.8496508418958563,0.06216341733591113,
 0.26942219183723104,0.8675329400002348,0.19276934490878783,
 0.5447015938826238,0.354238102161024,0.2624117980121149,0.28080662408470225,
 0.760035659212792,0.2954809253285976,0.6784123389730118,0.5687932355419103,
 0.054595779496389496,0.33285956902020775,0.5969366829576311,
 0.8874320797572244,0.5475040326775115,0.5180441880804117,0.8376544432978555,
 0.4586651756534331,0.5833739021656612,0.788553177890179,0.8913366330146396,
 0.3835609310473338,0.3115334896543601,0.9732227477552664,0.8659153622792657,
 0.7955799476018138,0.30963043274835933,0.15692936662108226,
 0.5275512559253094,0.3668930715741199,0.6168200393346724,0.7363183713960854,
 0.08968679249129052,0.5247107236101799,0.19040111757160616,
 0.6186269449860304,0.29929758249074656,0.4288088620092283,0.6954364665332224,
 0.5936365292368468,0.8592361097778303,0.5458955233094019,0.7611425928157098,
 0.7050750523001017,0.915408141467335,0.07014115837960178,0.6025167541687063,
 0.5648835941743428,0.8946795229034407,0.6422546042116153,0.9774853543999604,
 0.7435863978717137,0.20901866904412403,0.7663528685773022,0.8617384649652597,
 0.8742296784694749,0.628082181679329,0.021183752015249913,0.6360966461642978,
 0.18838856596218556,0.020939739328838525,0.19282900498613365,
 0.4478111743757376,0.4920573712143749,0.8725552999523349,0.06953719122751323,
 0.9555689901206903,0.37714761266320107,0.16132594992650795,
 0.8373042256480934,0.6300688004256031,0.09412297196866948,0.7036858722673628,
 0.5563899605168894,0.1640145388609212,0.4374288711322951,0.9236885419931091,
 0.7804218864508671,0.010509332271210536,0.408907734585217,
 0.42627743153159803,0.8939300931692269,0.3989798722543171,0.3606969797140548,
 0.5659477803686134,0.16369235260588566,0.3495838232409696,
 0.36527682810698425,0.6037337308522931,0.2037207728948922,0.2226500637114155,
 0.8217377868831797,0.21544309724962518,0.3268582833881758,0.354779337896444,
 0.5103494772050607,0.6450267007275924,0.17140720272394683,0.8491932031401139,
 0.8213905925091094,0.539527673627986,0.8223591832988643,0.447554922661451,
 0.8343587026886304,0.8662584875702152,0.769991996621294,0.8661098448617964,
 0.8735851967237701,0.5925904538488296,0.5169341946784864,0.8355800671865686,
 0.655590027993568,0.21218465296935263,0.2226877383285195,0.2949856545994194,
 0.9695124822204801,0.9623738418668114,0.20877406977449076,0.473298152700083,
 0.8481035294074258,0.8223870718327604,0.6386779800109331,0.08376776953526943,
 0.4890007165410243,0.8892690099982338,0.5030685826565556,0.269663019804842,
 0.9460871625510117,0.07605420586433065,0.8942577715408181,0.5235116271365261,
 0.5969413092670355,0.5481045078189963,0.6714972296292862,0.01538153919849461,
 0.742469286469983,0.647183459459084,0.917631354567028,0.9341658462158334,
 0.3049293757685656,0.5047497060927211,0.5173606186842806,0.10081442632462467,
 0.32062823261608353,0.0925601949254854,0.18808693186851044,0.200910238741835,
 0.2107195184851567,0.1584743691466921,0.3096337900409839,0.7173221135846962,
 0.6321526578505698,0.33406192746025254,0.405984172119388,0.913369463237101,
 0.004315372021568065,0.4162420363991082,0.576866310236143,0.8982239042530394,
 0.1487237079348659,0.581050622318397,0.3427171176917231,0.6356441822757282,
 0.462617529722646,0.6967002127287046,0.6993377728156122,0.08087092929004436,
 0.019749053740846678,0.8734661149120632,0.8153913986381166,
 0.2933185249304757,0.5685179217729345,0.9407228710341929,0.153566326896864,
 0.27067400104018013,0.717955397571329,0.22257310292282362,0.8789541799889743,
 0.7770627349705748,0.5647039571555099,0.8866562621721179,
 0.003258749858087162,0.7295647275763801,0.01903531309630957,
 0.2118598249637842,0.34965949889141335,0.0601849079564476,0.2894419778166203,
 0.5346636403786303,0.30271986376934923,0.5168910564917135,0.5575569678548121,
 0.15024336730482168,0.2245779273458708,0.6475914607208704,0.6099864759504603,
 0.7954432236854508,0.9569344038480164,0.6596046982373838,0.3384070042711447,
 0.5057912471086967,0.8920156789073922,0.5901928481154908,0.9414827538111481,
 0.4346912053540649,0.8060869878286449,0.05271514856848336,
 0.009098187524888113,0.8092803283240235,0.12786222012272952,
 0.0574986936232319,0.1260155469109936,0.7890556860055694,0.5951824570551052,
 0.6011407265328116,0.5625195924090778,0.6299340800610811,0.48946177350271136,
 0.8761398492845427,0.6530139291819131,0.4207909506271612,0.7878074475843859,
 0.934595442256632,0.7777565909595334,0.24239020763995,0.02448425396708842,
 0.0012362394616872718,0.6351039789317272,0.6019390142384531,
 0.37282615372979644,0.8238257861243216,0.28055580134835945,
 0.23611153152934372,0.7841897230064921,0.7093866161199522,0.9292574526927663,
 0.39939881813673894,0.22838577242084623,0.12450116512210818,
 0.791356548829838,0.9685468649700867,0.10927202507350053,0.8795274319666425,
 0.11767618103529442,0.569063647981085,0.8992996894132138,0.6575742154843226,
 0.02437753336029136,0.0742277729853772,0.6465719894517855,
 0.18289193920887492,0.7264061155430507,0.9217480664699625,0.8049053092749563,
 0.5261220099001074,0.9391958247093479,0.09704353970167656,0.9058311928348427,
 0.8123531511219713,0.16340076744547205,0.10000179583885249,
 0.5371829121309512,0.27862103082755807,0.9717331728105015,0.5865561794349727,
 0.4356800740107113,0.31574683733875264,0.8430240602620237,0.8715679020168903,
 0.34928323422523244,0.34642250137255237,0.5365797239314237,
 0.36873339971399965,0.6253759629089921,0.2083485435325274,
 0.42873969110496823,0.9705374794861706,0.29888891156321007,
 0.10506264685278455,0.9661731308457339,0.1824312278267607,0.9588580482023459,
 0.21822455141044672,0.7270827845536263,0.6208175258167645,0.9349774062189784,
 0.7315346052615054,0.1154384799122441,0.0648981225084806,0.04855351611684122,
 0.19586346574651559,0.2655061801980978,0.9806642454933118,0.0915884038870779,
 0.21332681710591384,0.5953278985845152,0.3290661864204558,0.7021387952098752,
 0.20858618993747646,0.7062661796205986,0.9075869748252081,
 0.35848091604173815,0.3761397679044707,0.14036718486941147,
 0.31037548148932204,0.8822631071855973,0.850634092775195,0.5977168863280256,
 0.6440131898495722,0.8241302615631054,0.9245216335977008,0.03565238022029571,
 0.42350199692433876,0.9819653127320045,0.9200632374592409,
 0.006251093508519934,0.03481764350478789,0.9184094985384933,
 0.6010114301206779,0.022696513366029625,0.1936279385855728,
 0.0459852253376799,0.3575380853774264,0.4434868161128275,0.7616352276603966,
 0.6039612495561413,0.6129448430890534,0.36936095478104736,0.3972325817223634,
 0.6491513665900938,0.9772632771245333,0.07154940849402036,0.5962034117226085,
 0.09981868101089075,0.7924186184322024,0.9380275022120639,
 0.17008340691191437,0.401504626655526,0.1434723414110488,0.5214838465418327,
 0.4497054700030988,0.29047754176556895,0.9608540731757934,0.6336197758666647,
 0.8751752543042356,0.7943869848291543,0.7139186140063359,0.7168064865313447,
 0.3318593785874131,0.769299245066073,0.6602413173961016,0.12271279481995678,
 0.4422027816436739,0.8692741906132834,0.11961608915022004,0.8571908440565398,
 0.9065348048411769,0.3336322090496536,0.6369103608257775,0.693275911621473,
 0.46041386148257113,0.05844539542604921,0.5360675014712227,
 0.7815052371245245,0.4895260690943011,0.6227422237891167,0.7458931402092271,
 0.3751355474941158,0.8405381599506043,0.330869698268775,0.04785444122748517,
 0.06286973406554686,0.42954483586674896,0.5959841335371723,
 0.22472531663116535,0.4099882505581107,0.33171819044950623,
 0.9291592380715799,0.3759150208817629,0.14624445614242876,0.551168256346829,
 0.7384891967941962,0.41600350610860903,0.20142530749963683,
 0.9324663926140664,0.7761505320264374,0.6756190807474013,0.3312920830904368,
 0.6086355895424869,0.75155738007631,0.42576250534779314,0.8978275324295895,
 0.6745301856863843,0.9584744473268584,0.8610071129044236,0.5798133213806597,
 0.8456080671660681,0.4637399307819994,0.03419074006096967,0.7018118786648148,
 0.1442505948164684,0.3632623407831175,0.8083150720227363,0.8024745141529016,
 0.04705519018824156,0.035694506168167606,0.3432092063094062,
 0.9269218139238476,0.7832541112497894,0.7501536814483569,0.19686135920581516,
 0.8715364682349981,0.6822094070457281,0.23314123592975966,0.9735186921454929,
 0.6426392715498076,0.6508339247647014,0.5058485695079251,0.18020593822859,
 0.8972789042357062,0.5999952557194176,0.5608066593721558,0.8844722294827091,
 0.4264395875458191,0.20024383522500921,0.0727140053780867,
 0.28841031166254705,0.834509987745232,0.5412637386567818,0.28768285060416765,
 0.7131542795295902,0.9539653692543463,0.7187644067601879,0.4354111182189554,
 0.16718518116566228,0.23498630969002188,0.9926083915151842,0.941823955843286,
 0.043119190639560045,0.09977801936560471,0.4738303065628058,
 0.0656831652572839,0.03726492713998397,0.8550282059128615,
 0.05439283552652263,0.7532410153089301,0.061672296840458496,
 0.533228111729247,0.46255302335370974,0.49408027643302355,
 0.23889155867910783,0.15709681641409778,0.7094029023334733,
 0.3440593612898699,0.19271518709222213,0.262373980027524,0.712239116522504,
 0.5149083686002064,0.09198079611546217,0.016667761040493057,
 0.1269480575862626,0.5380134653298418,0.2861704378343439,0.8867589821302797,
 0.19651224590493854,0.23888505324107268,0.10951178282158336,
 0.006086673606780502,0.5625390220247722,0.734617609950473,0.5317267348698768,
 0.8940610370030178,0.21437934905268308,0.10659464574195565,
 0.2800166599377705,0.7643433679384355,0.8792630095053058,0.7253530654478336,
 0.8127205181591932,0.4397895428850266,0.39070439090845077,0.6989081983169247,
 0.047939403122004176,0.652256776239083,0.16747584737780097,
 0.027600504334118425,0.281154447083407,0.7514202158427289,0.36882381654774,
 0.5187593851013403,0.4581204528101034,0.30036175147537725,0.4183406011283888,
 0.024405717884007583,0.5343822234461294,0.6482668623451574,
 0.44967509882516987,0.42442779505531225,0.04247593045034792,
 0.37337229459427046,0.04961333619729347,0.04357912902163985,
 0.18964337770828676,0.9371509265209974,0.3549109060665451,0.6552608792961039,
 0.02477270212290761,0.6519770326601628,0.5513175583011676,0.4856157642850537,
 0.7569564075427713,0.26479168721314295,0.32696455044723083,
 0.7913129471176878,0.7481315484271447,0.3453827105647902,0.5611409399350855,
 0.1263842850907958,0.6164542061369981,0.6417153377506248,0.8362674410507329,
 0.4321937737647238,0.09743355468570547,0.39947100619949805,
 0.6908042465430364,0.6600428749709224,0.569554466450135,0.6678082691795107,
 0.35644127895615074,0.2739968120013748,0.4162145726702373,
 0.10092741425536977,0.34342722490083355,0.7283768102417728,
 0.2681466734663217,0.7990864305045271,0.4267543013541806,0.5975678359895774,
 0.8068240446947005,0.45731075902815355,0.6425015415156675,
 0.46337984321685166,0.6386310581585761,0.027867051155354705,
 0.5962598386336112,0.7583494653722155,0.3390065326371723,0.32626370592839926,
 0.5779844937331413,0.15630512147746223,0.45850120406110795,
 0.7548146793264935,0.2647678627595248,0.6595929879556743,0.8433293533784686,
 0.7015725794164077,0.4404244804919092,0.15977996318850085,
 0.09023234117390821,0.5069307994487797,0.6882367315198575,0.6311928674867061,
 0.9882936862672356,0.19611648564502104,0.0637834685211568,
 0.005504834652723556,0.09094459484437134,0.49293528791701324,
 0.6638234322702536,0.570787937930838,0.336702314675404,0.15027166400021152,
 0.6499400145012669,0.6355067025653064,0.2941268861302204,0.6613759328946716,
 0.6137063621595836,0.20637553382105067,0.6958384559333746,0.7396196827091347,
 0.6257238170332498,0.5238061715242663,0.1237075486199618,0.5405335112831807,
 0.6540576983119861,0.6137973167573968,0.19700815364280122,0.8228251388626351,
 0.07705722484266364,0.6060542042292653,0.1256874089940887,
 0.24019408421212718,0.5423111401814782,0.15131373756503708,
 0.9734778253303085,0.16821580257501123,0.6537803914509638,0.4824570779851086,
 0.32535650755516465,0.9073934826967403,0.46287272407201185,
 0.39517052299041566,0.5294898056840052,0.680531278665697,0.9939752066610478,
 0.34002745448079175,0.041353347594438206,0.7723617892465676,
 0.9444837600471067,0.6883450438587735,0.018594083866795508,
 0.04668275539040456,0.7867737046874546,0.15017005666843442,
 0.08530479614086173,0.6425527598910189,0.42654528728376606,
 0.5777295313802991,0.2787597688428276,0.13076018454907268,0.5066369169833753,
 0.12845263176023136,0.8126742666566178,0.9693626912639476,0.4586593407235685,
 0.03668667467872644,0.19135954994207938,0.8684115950951001,
 0.6459792476387627,0.0874541602015464,0.6414918518900902,0.2030216668542949,
 0.4825676559033655,0.8897978809385378,0.6502605954528199,0.03358876695070245,
 0.4846601928143617,0.6712912711349607,0.42374271920617557,
 0.010520262261783042,0.08603951616535088,0.5832059638669804,
 0.5429768232262376,0.3114362513382083,0.5807345098791962,0.6270001989737146,
 0.01738590367925774,0.8482044885819932,0.37253380298313066,
 0.9589220845625133,0.23746150709312008,0.46068506756052663,
 0.15090625703172889,0.46538491423498485,0.06447765099789926,
 0.5267031228802304,0.5187704587296782,0.3677430269033545,0.700288851506701,
 0.9592603859245776,0.8022879618301311,0.1686610341121315,
 0.016748036129227595,0.852831612955802,0.5472383219837158,0.9965012543137507,
 0.33621120306002206,0.32580364177357435,0.12201029414434217,
 0.4484049778608683,0.9604927315875519,0.47108327879038925,0.4511067015601854,
 0.4806389552963546,0.20888438238046525,0.42031331453670884,
 0.9268986487176041,0.010691692515190976,0.5084569134541558,
 0.4141973883625806,0.4761430112748949,0.43310928686894434,
 0.07255390753753632,0.5251522938711073,0.4872229724002112,
 0.01463308216697845,0.8359951368782662,0.9274814087005008,
 0.16741087855170667,0.19219869859212402,0.17231429471054271,
 0.19037264863817227,0.8611716528372644,0.6985724510811491,0.689546475737508,
 0.35624640568031984,0.33004475490413787,0.7290517799637846,
 0.42669449594965314,0.226269440630962,0.8855551664513024,0.5890750677832941,
 0.2776634369354749,0.18166461516994015,0.25272002105269786,
 0.5869429494727794,0.31113832854766077,0.4278242450424632,0.6996970982508615,
 0.0812671051062781,0.1830147118225982,0.1568208473572259,0.9539790862679183,
 0.2261085753225478,0.34893026211977385,0.31488612888451717,
 0.8129947096640793,0.4802884055361393,0.7862512563826088,0.26489393783879955,
 0.8688334738573241,0.33596396397921147,0.1948711866784999,0.5186360864082159,
 0.09597444559557078,0.8138326938566849,0.14462968927307773,
 0.6599925748610197,0.7559187697195431,0.9785300817336593,0.15656182913957306,
 0.8287729513840734,0.8526969023980417,0.3433754532880483,0.6688763907154032,
 0.8783587231796813,0.7614745980885762,0.8032427940048172,0.4474775809685423,
 0.05056457404462045,0.9591218485389558,0.28567764219187486,
 0.24085122137643789,0.38885808375610265,0.1879962854346532,
 0.25723103807846937,0.25794727453435795,0.4340948514696096,
 0.6164045485840578,0.21639003467302864,0.6030785105473465,0.7622994038073082,
 0.32158025428976034,0.7147893010916696,0.25028613821325785,
 0.21561417429063812,0.747896546072924,0.06306193317283249,0.3064177094976197,
 0.5830938066641318,0.7603643737012733,0.3268365926642489,0.9645773671503635,
 0.6101820698029439,0.9277488951562147,0.279707476572677,0.30903351882835806,
 0.051148762821288773,0.395648732007027,0.7601259125952123,
 0.007528766784583141,0.12716249910919308,0.9755170439106192,
 0.42293241400790527,0.8524489747325003,0.4129687087984374,
 0.20957725926142723,0.21799353132157517,0.8489664286127503,
 0.8913570441634124,0.2641415331028292,0.44033439154330944,
 0.15505697240125194,0.32120617503245563,0.7915899242428706,
 0.7053106771128236,0.9107425075537742,0.019702974317381994,
 0.38914717354135975,0.06987714710460857,0.32784429418635397,
 0.2181398598482467,0.6477503072581482,0.1987010494291075,0.09060514780552631,
 0.8521234213587671,0.3334370071701175,0.4053736483538276,0.09128174637985786,
 0.8549170618181856,0.8123669229122197,0.1841186966193119,0.3736356042213318,
 0.21806586640820746,0.37161181530421894,0.8413667395546948,
 0.08527611516829059,0.5089812790718039,0.7203074401450223,0.8147728886952487,
 0.4386337965500443,0.36738439102696674,0.3054479835517684,0.2621380014598502,
 0.2960536835396176,0.12643661407559903,0.016934628815766573,
 0.9148893212153631,0.8049899806114568,0.06409216498549508,
 0.21835329804873904,0.5552604789696045,0.14621651619684306,
 0.9277345567464557,0.27024196537216694,0.9510060603330126,
 0.22260523551552902,0.6713141419987012,0.8717089385843324,0.8883821616730878,
 0.49537747278846,0.8135417412407034,0.823919148872738,0.9064895936405443,
 0.9975220892476055,0.30855403689171834,0.22492952573675562,
 0.4970804597998765,0.7797676287680846,0.7397625555339098,0.9752449635679777,
 0.11317996791092233,0.516795331579706,0.28087759511924415,0.7443108650061437,
 0.8609069745540054,0.9836319782890454,0.9150732705820027,0.6790746733326464,
 0.7298486533186547,0.2997273571220419,0.12266306687900619,0.2994901850811832,
 0.5904654312655355,0.3154145402945545,0.6857211067145114,0.05595876574711334,
 0.23828000576598252,0.17890314618356062,0.7158399677325008,
 0.5940061219403511,0.9085686153825036,0.6819299910238931,0.36992411405173264,
 0.36361960082374567,0.09816583209272178,0.4675874280898049,
 0.3812489129895942,0.21109758856025174,0.16017470906220443,
 0.11280376589940344,0.6821698135120609,0.2498094194460443,0.5396093358415077,
 0.40336904109020055,0.5503490876421987,0.6508655865640848,0.0681181852304692,
 0.931374480172946,0.0035796066295356033,0.3058385795925136,
 0.24764369820409937,0.7902195007445278,0.2285177106819194,0.732618592635991,
 0.18290411105001314,0.7875430076668322,0.335317772243401,0.09982411185821627,
 0.7495630573214779,0.5722283188892018,0.6533131663115437,0.1830879414893205,
 0.13885610661847025,0.9000673487747911,0.9541634902275187,0.7748474174320882,
 0.2037689996437314,0.7650817989552383,0.8859813833674437,0.37680275490386905,
 0.12822352856450248,0.13351815047354132,0.11630532404997518,
 0.919601010489677,0.3015805617804973,0.6197245418099211,0.5592268919446476,
 0.5351599798922632,0.2554781961221324,0.41976983070164664,0.9057431029765048,
 0.2016081076801186,0.008006659281463935,0.31569079169173875,
 0.10597315017469888,0.9440799208979831,0.7094821907591892,0.8826674737396103,
 0.5894766335565561,0.5498478871877931,0.7413490207704408,0.42554302497850494,
 0.739261399441988,0.2186774271638231,0.9370480666950909,0.06497566779518982,
 0.9335490392447864,0.10040769117767079,0.841252285008988,0.5715993549959033,
 0.7946996206734351,0.5960083817890256,0.18239390009495604,
 0.48415352040477033,0.3980256242412017,0.498233223828046,0.2890666305038311,
 0.8635959656470593,0.6392949863998711,0.9377546110042418,0.5544273821843047,
 0.09668289467725122,0.12817033384259968,0.8813266337473366,
 0.9596245539674799,0.37219594868502526,0.2970006189944151,0.6133769256688012,
 0.4547247145975096,0.6723292075814595,0.776554040390709,0.356445237518242,
 0.8667142051396335,0.9477025443048505,0.8479667783357195,0.08853358777702691,
 0.08006623326134288,0.9217921332018406,0.3670617391009263,
 0.07195467254620258,0.8590771837247225,0.46189656622660935,
 0.33266848764414125,0.00609843560301937,0.0837513513951108,
 0.6535279873340372,0.3904915764766701,0.7718909631579952,0.1362915025025937,
 0.05747043021104192,0.267970944253792,0.5561661812156212,0.20523040800277303,
 0.2450239209624676,0.6008386485746872,0.17196840193691165,
 0.08132872405937264,0.7432729201416817,0.4242747064046847,0.9126769916831429,
 0.372334713711702,0.3169259615572504,0.19168770980005911,0.1467131544431155,
 0.04099198666990744,0.35493850068175414,0.5860151117240886,
 0.4120461116479188,0.6241623380135332,0.5376878329378866,0.5435426123324387,
 0.656478262386722,0.13335034369973453,0.7328405714041516,0.7333956951621731,
 0.7536116929984461,0.6074282447885114,0.9080638926791127,0.36099016168165876,
 0.3323833109916837,0.6070003407921766,0.60614224962536,0.42427232392325165,
 0.5896660960430545,0.6234986895874495,0.6705153986162335,0.09224494084252932,
 0.6628222943356055,0.33936386155573683,0.05482843406133364,0.635790759673235,
 0.0022613423055676263,0.47174826374521595,0.5512184600389972,
 0.6538789399344742,0.9191988621850662,0.8542109544633119,0.7364894927873614,
 0.004823839537526076,0.4380123630419792,0.42262656165887547,
 0.8237089614516157,0.6913063323747622,0.9135281671697383,0.19582072806243755,
 0.809006222405844,0.3588291362563947,0.9070827102203032,0.40734107597669467,
 0.14735170124513375,0.23011024078649034,0.1161832627490531,
 0.49169780741636127,0.9947662811454236,0.9007161864364079,
 0.42358998086278843,0.3073423326586058,0.02115713366352001,0.506347319528583,
 0.34299457663353083,0.1669191264540908,0.5437446852169654,
 0.42991697340769175,0.8447115061761089,0.011124146282082026,
 0.10355314212734268,0.10225300194885278,0.5730735743896642,
 0.7081212540617774,0.11302873609715147,0.3997842495310069,
 0.17494865252381275,0.6417413618591187,0.3348442664009267,
 0.27046704823271883,0.025977854585671478,0.7881491108930584,
 0.44217011457641553,0.5522882158742057,0.30768675906042164,
 0.5180654362602458,0.1303355394112643,0.7941663889236412,0.7924198998048279,
 0.6328572063286722,0.6823486961964009,0.6854370426228211,0.7842600578473151,
 0.8668534392709646,0.8484423538959852,0.38062987708427165,0.4607581262686503,
 0.19806050551532084,0.3999912398384866,0.4179450189065743,0.1613744806187778,
 0.01307182794164441,0.755495864928427,0.8492501577355563,0.5169547106993746,
 2.432619135905867e-6,0.5752870028287289,0.10728029419945484,
 0.9861255946334911,0.37538551312370594,0.8333443624369806,
 0.40355906858140833,0.14975074732903793,0.5074000884537295,
 0.15874189136287242,0.2512394339047973,0.1319113510468155,0.9579858873127656,
 0.9629068740528727,0.7822576146465292,0.2922333555604977,0.4401518749422746,
 0.9256015313852751,0.8544320297900586,0.2857338125092336,0.5855199782637851,
 0.6940874029913566,0.5821852892958632,0.7026247394582751,0.10957735113440903,
 0.9746664986084717,0.4997536939974504,0.3969630243714012,0.18484471192459795,
 0.9866676280450872,0.8266603847906344,0.5286766540137355,0.6069389016469745,
 0.7759980967402464,0.6581463612403973,0.2827212977439506,0.33121004030586965,
 0.6732224754173622,0.8383806682423617,0.956659904816952,0.1784756957909902,
 0.8133505057678072,0.33220525956511504,0.5665076816255217,0.4962365276474763,
 0.6170602427594654,0.0279560057820466,0.2546474541019701,0.03227369953783443,
 0.6739872165544492,0.8800081660809926,0.6404820258618922,0.8970355906981645,
 0.33063851018055146,0.03712903144818336,0.8585014122493336,
 0.7916440304816205,0.12291776916931885,0.22380322499160965,
 0.8330599209603565,0.1989385648176074,0.8178036527555116,0.04435149263226956,
 0.05305412440743751,0.16734478018687826,0.005111354856768502,
 0.029656630442325804,0.058864570023438256,0.6302676202681347,
 0.6367556297145595,0.8511381895640002,0.46297375434704613,
 0.12418745706169276,0.7902092562734595,0.5634751526785434,0.7350666502177766,
 0.13749529060128174,0.14310950934083244,0.844365431019869,0.7520900680410089,
 0.22647620734808827,0.6146765635954167,0.41931861102455903,
 0.3927754366479692,0.7938571223985391,0.7180048495351895,
 0.040582138754648445,0.06669566860605436,0.39789582171410665,
 0.6557823184417615,0.9602818809570343,0.3519251511790652,0.9467951932815649,
 0.08776620737770413,0.14096571691971194,0.3983961599198268,0.694074576561289,
 0.519746538505417,0.7573839981203125,0.5557738977553712,0.9592307863851508,
 0.8429781382724734,0.8666936636543289,0.300601908848001,0.27410004337023575,
 0.8466282848995693,0.075449113165169,0.9890816347803869,0.1571482553460587,
 0.3708500620394193,0.14489837050842092,0.9161642037178691,
 0.05426264799089231,0.37436988318019404,0.166537663450265,0.3720702883945084,
 0.3938988505298573,0.3540188158330455,0.12090051606323693,
 0.43199821427048857,0.07416870474556869,0.8579974959948515,
 0.7509145667933851,0.8654148797801631,0.8756757204121184,0.7143316050662141,
 0.14064981116471553,0.7566164998120949,0.171645160494576,0.6550446930138482,
 0.12596728151860526,9.501931554893872e-4,0.5460299379701541,
 0.43692992411002285,0.09094795540651357,0.7057844516211548,
 0.11250405930814777,0.0026890233695220367,0.5125790747058683,
 0.3500680212877806,0.9706013681854826,0.003046741447259582,
 0.45588439687447035,0.6380098511481117,0.9440209978693375,0.6861389555998898,
 0.9911419667727793,0.5964713414700652,0.15953742627648893,
 0.16278441590362025,0.616594607433826,0.5384391466569326,0.06241732397437838,
 0.6145613819906777,0.31014980913114343,0.7181044784728688,0.6856226236543709,
 0.047536368246162475,0.6679042095423513,0.2812355528104288,
 0.5551469376538719,0.6298762036212853,0.3014843908360887,0.0625552115197916,
 0.7492527414738634,0.81192814828847,0.8625314217562651,3.486075755787521e-4,
 0.737954918931636,0.656497535414064,0.5591190943107399,0.29267571827157324,
 0.6817296800287224,0.7343946435070224,0.5929165809768532,0.12347097697479237,
 0.380798399746724,0.4129329743015858,0.4166590138028501,0.4947418991384739,
 0.9339521784075403,0.865135578938457,0.7004303782228238,0.1773730326855848,
 0.2420074561454133,0.023073910586418278,0.783791946304832,0.8001951286052658,
 0.01241774471913959,0.7773808624005433,0.5467735086041168,0.6328028162127514,
 0.7566204850057947,0.8473960867607346,0.6816139960587592,0.5952370515415066,
 0.11143963158482861,0.32118185663846077,0.4105502727986351,
 0.6081944564876407,0.8455617642056571,0.26452007758554874,0.8530544529090104,
 0.21723888534549807,0.21692931891423461,0.3442695105606882,
 0.6156954908940778,0.8719925236842849,0.6349124163120422,0.8214773411297404,
 0.38768698097231236,0.8240315327394493,0.22415211295124826,
 0.6622717888440124,0.6688289467834452,0.4718362786468838,0.3877349753816788,
 0.43409617799844336,0.5372425946312699,0.11626940931363472,
 0.3302121368039601,0.943743995121846,0.983192537443683,0.5251166606522459,
 0.17893238541536072,0.6291333778224103,0.3835684039797217,
 0.15455790366742816,0.9927925518062894,0.08369095205971988,
 0.9448390913948193,0.7710928187785455,0.1265858496417659,0.1522278599022583,
 0.24818573660465448,0.2398853235166768,0.04969848482162509,
 0.4344934529878438,0.0041620576844102475,0.6892039852845211,
 0.626454449217822,0.8252044995241233,0.9187301699377219,0.257668230435359,
 0.1275755629591021,0.277579745464136,0.48623193244890406,0.6230142202228203,
 0.2916023912242657,0.27518466276458686,0.6931605048473306,0.9403786806118969,
 0.8488071796509375,0.7294148956149139,0.096405024590693,0.7646107657817306,
 0.6344872985850792,0.8064251670999004,0.9537101139055634,0.1979352693535369,
 0.7208289272794982,0.8430060841524435,0.6688996340086678,0.9008171413951086,
 0.6402614526887951,0.9703528072151043,0.9321672272152697,0.15721332606274263,
 0.12053966902103752,0.4862481738283254,0.3505614532593252,
 0.05809730188856399,0.10759809970140144,0.9009919628158511,
 0.08168272925855258,0.46183940554827163,0.821261597933283,0.3314133394044776,
 0.22863640687676234,0.627428480847243,0.4871364715500096,0.7918469315857011,
 0.34342122748982984,0.4105085309658012,0.44724764094973246,0.383194978696795,
 0.3387795064601592,0.1393568488912318,0.20121655349943446,0.7299044527284618,
 0.4002157363841383,0.8466595554863534,0.11229880021846861,0.3640425882386421,
 0.5219835802569411,0.45185650053447446,0.5765848405350896,0.5975629978059653,
 0.36989750394688237,0.5838961937584208,0.7876592327824224,0.2894655696710261,
 0.550492228725366,0.5112083411121958,0.9388280033287505,0.3678583701270225,
 0.9369422418098539,0.14777493886176485,0.4228549020168706,
 0.38800131551078476,0.8453684550125511,0.8078127233591389,0.8789369510118813,
 0.7483460947708686,0.2916585807060238,0.8352701013521635,0.9395022174480446,
 0.13923724373762547,0.8728792861763947,0.029720344674688093,
 0.3959958253239061,0.2910770233829557,0.8661515304579529,0.6875643952412462,
 0.4808326032536263,0.09939544166281622,0.8453629156434366,
 0.35867462119184657,0.2914612703746883,0.26658706229630813,
 0.5134354918081392,0.6934845244085549,0.002380718776516799,
 0.5844415732013324,0.19628055918937315,0.43677195368235844,
 0.36778916461292965,0.12125933018892321,0.10160209222708794,
 0.7877633453266991,0.5043782313901342,0.14645649904646552,0.5522170800268875,
 0.38202437365684894,0.22502578509696658,0.49308307725991796,
 0.8472890602109494,0.9046112444407985,0.8722818757099688,0.5033604404256476,
 0.35565919139511026,0.7035682613200183,0.7675234928185376,0.6257375905138532,
 0.8741020761281254,0.8859082300528831,0.9920455876987851,0.7662826421776973,
 0.4110526492507991,0.2237720961370071,0.21030376286885866,
 0.24126768544759247,0.28084959582498226,0.480112976008241,
 0.36322797855692535,0.3135971781306496,0.9964894453608604,0.5714181790933799,
 0.6666406057322702,0.42242125562250565,0.067903345942534,0.42150814015489857,
 0.25812027428447293,0.6431311196554389,0.8146609069192605,0.140779840264723,
 0.464743591384174,0.09380353419391463,0.4162815691520153,0.9958672589060865,
 0.5103779404515048,0.8205145919774265,0.8848163419671735,0.9195316741833762,
 0.8549253597620701,0.3994955488223413,0.5678682714518914,0.3372688922250132,
 0.4808658748453867,0.9534984234783885,0.2348753635032853,0.03884553452702166,
 0.1801510997089687,0.06896832874060377,0.3994632781540378,0.9147815761757003,
 0.26651201189897944,0.33853710060545794,0.18704272747720685,
 0.6139704000448128,0.8407964694979555,0.21668235333102126,
 0.45123312876308885,0.7691339657293492,0.4908340943500068,
 0.17905455153710992,0.6755539806909707,0.3603202776833965,0.548446338991441,
 0.11184768967959846,0.3367662818447875,0.47344781377323153,
 0.4825843601884595,0.875036230032604,0.8743300773091042,0.679525341860573,
 0.5581649111177607,0.8006859485492345,0.5996243645733815,0.5434199268055908,
 0.20419018514338805,0.045415100361151706,0.4629841643541991,
 0.9062202381179055,0.13900310623447254,0.4433044536231776,0.8235534945076387,
 0.5797137532454785,0.660604495301314,0.7626955274253003,0.0546271501606983,
 0.9814005912861824,0.5131572243479181,0.2881460584485658,0.8792752040295583,
 0.36154792166679084,0.6162428594804931,0.46952429498411463,
 0.8790753033330533,0.22008322841204941,0.063600610826965,0.4615806936164839,
 0.15654689767507235,0.20360453151092772,0.8664399667465085,
 0.7092958300875961,0.20716701221938494,0.6169524879140947,0.7811606425530804,
 0.9610275653151837,0.06858493217363093,0.7890201109440268,
 0.17010289640642695,0.6197553852271305,0.3874719070528534,0.3859605011219769,
 0.21280776185484918,0.07302157575230495,0.8182482519298387,
 0.7676604525000296,0.25464104921838715,0.35901300199735786,
 0.6598305140917994,0.9379448760766835,0.18080247274678873,
 0.13700471313030693,0.0022122748728391484,0.32527458037169954,
 0.966344083869603,0.9923934267927226,0.7562149929464681,0.29568494876022133,
 0.12889737160935744,0.2401697151393427,0.7157497930572174,
 0.050961691194317815,0.9692675671042256,0.7980301519149356,
 0.39790632413695337,0.761399291753199,0.4936603056977449,0.5914979478972537,
 0.48715041241360746,0.3003908361829428,0.6953764529190425,0.9531513690881793,
 0.40018976086240565,0.8263679656699939,0.3513886567679436,
 0.47764504114866546,0.5705938388637147,0.530395005357931,0.1646892904832129,
 0.7058960012660593,0.9201246267784481,0.963493898460741,0.6011571563923787,
 0.7191511625729307,0.6830853930061807,0.3080383799141907,0.2998507638239528,
 0.42257088363893636,0.936898874978653,0.9561076373699404,0.17671292299004315,
 0.8217490381055461,0.027390274035634543,0.5482178747148581,
 0.11033733681143543,0.4661843328129498,0.542001313451784,0.5002611649791071,
 0.9541006901647422,0.5733132259417651,0.7144332717309845,0.08312906725276858,
 0.4834383645878295,0.30057672257147916,0.85571811541641,0.9275131876071374,
 0.2038823756819147,0.7767850396616948,0.3060751516336181,0.21576665902056058,
 0.33344412793991474,0.8131776319571211,0.9153347213091404,
 0.27755204093058417,0.8257151302697988,0.2933015442384759,0.6695529636167337,
 0.5003813520610886,0.10668204235213685,0.22112132518094296,
 0.9879311094261005,0.9973564886726398,0.9488512866677866,0.9103307017624263,
 0.9011362793290781,0.9287451468971002,0.6497095653121495,0.3080114893281748,
 0.1709196650983178,0.7620075582405685,0.37994166398607676,0.7129983615673776,
 0.9539183255517466,0.594198950126219,0.43244736617427937,0.11072957545797224,
 0.7668803074797617,0.9114357301820363,0.49927697269327276,
 0.08461478424255264,0.0837143048136132,0.6150244184313641,0.5653470378266576,
 0.3490104948911743,0.639233924831403,0.354457007506475,0.7659454444564825,
 0.4136317517163717,0.017698615056532696,0.13463931779502525,
 0.4135334259158957,0.8279063698665157,0.5419914685217716,0.3643205349318648,
 0.23907741597478138,0.8132455286790791,0.5264401402827992,
 0.28402040199328005,0.4802179519086529,0.9927221983221941,
 0.09560998146336364,0.30750288677424564,0.41885790556717695,
 0.1427085850719637,0.3942353399065798,0.3346922438255243,0.7482600055417361,
 0.8832399439707019,0.2678475192380877,0.5952299670433466,0.3190382711112475,
 0.22105811454205215,0.6150412674466715,0.57257893048055,0.46357715788681964,
 0.832789629494012,0.010329700734533653,0.8201518865439157,0.621890322393774,
 0.5379773935716217,0.45577043440016174,0.8131805546802678,0.7122863016173148,
 0.4127761927247974,0.7593170453767317,0.6592218203510807,0.2566986616599123,
 0.7732404501402863,0.6415890233170808,0.5168981687425621,0.1799717014322707,
 0.7326773858236963,0.39337508470075155,0.6586815727656946,0.6533406687302413,
 0.4224133666908071,0.5625399083920296,0.012921588657653471,
 0.5464570445639041,0.29724415565969786,0.5627789928798161,0.5944903347249726,
 0.05310417039300186,0.22028924086028046,0.2264313215255097,
 0.5384534532061733,0.27951486691380734,0.7105038635585552,
 0.03364392549774131,0.8503432923077139,0.8459251376115197,0.7640962971444905,
 0.9063941322888197,0.14897362350464727,0.318905560531224,0.20244881947412074,
 0.8432098673578937,0.5172362221559026,0.11981352235169596,0.5925974508889171,
 0.8506252957284183,0.5916706511389415,0.07551393771670545,
 0.40966589511611595,0.2308356407555764,0.7975764768440008,0.6612644882165272,
 0.8378910650165466,0.6933285109001276,0.15106002853989886,0.975578553769544,
 0.10630338620326374,0.6074030820453401,0.5936305347177102,
 0.05269961169989423,0.9821746913149204,0.28218801742291033,
 0.41111024014675623,0.46792652121207523,0.9624973012270749,
 0.3182184235987733,0.6306844795709077,0.7892995101473318,0.8531945138235776,
 0.4361059960841853,0.8171011820243893,0.8592658879510511,0.3626509138995456,
 0.7273236864954612,0.5641828218232492,0.7719955369800684,0.16774739699488184,
 0.9867125202018765,0.9430021665238407,0.30144600459604376,
 0.03645164623167818,0.6575056175461435,0.3455891175974093,0.7664828017274503,
 0.7097379295321793,0.14595773018789737,0.37859828577212873,
 0.6778972498982699,0.17184182324739716,0.5231503594853741,
 0.30290229847646233,0.9803497149253952,0.9291592528290287,0.7645177484017054,
 0.42006192653878116,0.019644348678153345,0.943035920233009,
 0.6223293395587155,0.47719624019747864,0.1010522600999414,0.4786749799015382,
 0.4985827550631675,0.24152230168773137,0.051004670980922984,
 0.702416335342947,0.23296604116277664,0.6723865979898146,0.2644149009449419,
 0.38723829063105764,0.7675039999106068,0.7892537849695306,
 0.08279416391210881,0.7121911478152361,0.048011888590501206,
 0.8196479037782853,0.1593794687378165,0.6946411323471231,
 0.029168753005064052,0.4070199880018893,0.9611731848024175,0.802414975128697,
 0.4776778128112895,0.7144909435873734,0.6273356306170343,0.34483327919663165,
 0.5902003419348515,0.10668360550782574,0.4566787507954535,0.4747688461788302,
 0.8481657822935119,0.02871535965575267,0.1251065462579719,0.9903829863177641,
 0.8407133129099857,0.027613590703814972,0.3355652035446187,
 0.6759279979944434,0.5565885785531506,0.8780840495232995,0.6005554170648059,
 0.7254423385205079,0.051503967166735665,0.6659180992855938,
 0.2038057186875868,0.6419172304071469,0.36231834180979205,0.4569437140753515,
 0.34766020554563637,0.8357452562046567,0.6486358057742955,
 0.28720152405943256,0.27499363437428204,0.488400142113786,0.5153198041853262,
 0.1539839269383526,0.17053681722822955,0.44918381859375855,
 0.46549038157417166,0.41605543270982026,0.2965596233549196,
 0.25071301523111134,0.9831292664502219,0.4009376165339327,
 0.08913554952853597,0.9080895596400658,0.9190690174607331,
 0.38098478007177683,0.19153068806556683,0.5012410737643832,
 0.5737684107490528,0.3996618158351879,0.8751383260996035,0.14693716088708464,
 0.5356112572963699,0.27089032891898757,0.6287603335834518,0.9715753847279434,
 0.5137659383256857,0.7715524094267944,0.4805113279241079,0.6473912098572245,
 0.6934884338665158,0.030303396887997436,0.03640097622378091,
 0.4403465228463823,0.45870308789851055,0.7621563304458189,0.8568287310675452,
 0.7241826935161813,0.8620179720330919,0.47505811424038336,0.6444501006607778,
 0.41173877045708696,0.6605803546329666,0.7674803994652802,0.6242818909946657,
 0.4802756561071997,0.6092510597058423,0.8331531888531318,0.31843636219289395,
 0.1939658086907946,0.39241865638223294,0.660358264331488,0.8275657198734759,
 0.19550231611407576,0.6402723575877742,0.5985614643202902,
 0.19014674378260388,0.6167046333202478,0.928680689692373,0.8058902644000304,
 0.7233477663933673,0.9092560580135312,0.5760225366112663,0.5574574402560223,
 0.9403055732356066,0.788927804659394,0.23024188113466737,0.3084124916039048,
 0.0022885775179959378,0.6837454022782776,0.2629538421129721,
 0.04289143829598818,0.47103842515155203,0.171526863960612,0.5084907498016071,
 0.04981960305533606,0.7367318948083253,0.9484753038413501,0.7378942984511423,
 0.9696969287158983,0.046679567626999874,0.39406988984928304,
 0.287764173673543,0.6666606233500165,0.003683260367886576,0.7100725181033934,
 0.2426534353550499,0.2585196797467252,0.40705099726021876,0.0712610898123387,
 0.05388961124351577,0.46043002978548264,0.47605037050090726,
 0.705127597196261,0.1514275211176237,0.4576475711228354,0.7536694356137784,
 0.7432960486327285,0.26472771373443027,0.18244800294245367,
 0.6218662516837755,0.19065436960480675,0.06212294607521629,
 0.45111696410402924,0.8722099186580573,0.022136090071650072,
 0.217455253007382,0.5744437746659614,0.4680530240828429,0.22176983603112976,
 0.48286834103587406,0.48055746329093485,0.007444840068676517,
 0.1782787131346396,0.6019466789099697,0.5773898331642983,0.3813157685691131,
 0.35039995407895774,0.7767438256438635,0.17948956495555757,0.90512275237482,
 0.55154963810188,0.8272567680971831,0.6290707205114457,0.8733034049869499,
 0.3520413538454824,0.6478315990420106,0.9824813297698428,0.7309267096680183,
 0.5102552410955452,0.7106288085326251,0.718381568594058,0.6599315976478001,
 0.93046326075531,0.622386631935699,0.3000610022622302,0.4817114539435676,
 0.6994013771767669,0.6818993949028425,0.2939640796113654,0.7926159832355102,
 0.8654501930834213,0.10124151284313632,0.8892999235080292,
 0.23306755129641954,0.6497129573491298,0.6494670657524212,0.5276738330249616,
 0.03117978220300155,0.04669887019732277,0.5057814406411428,
 0.16198360429710024,0.8552775942735007,0.9923880782542971,0.7578632102394157,
 0.8656277931611585,0.4767804004183517,0.6425560013071135,0.941887349405629,
 0.8594349904911279,0.477520375358348,0.27068686412113185,0.6126118615296542,
 0.24347606655807852,0.47961270720223603,0.5675960904888027,0.206500954535227,
 0.5455723304890892,0.504362160263029,0.993324690772777,0.02198822286029689,
 0.052231135868914036,0.4791185080977314,0.5442778456079314,
 0.7751962027332255,0.29574379329737577,0.6777073291787931,0.37523991532388,
 0.2277399615904867,0.6656753233519707,0.5139315120502489,0.6019038848855324,
 0.7125800524531921,0.4461645850006434,0.5384826141022244,0.4948640194222259,
 0.7950530527278299,0.03390970590878739,0.21637707855795207,
 0.7932548497985725,0.9822235150914139,0.9581369562211314,0.00887001759379591,
 0.0888953142219786,0.4703696615440588,0.9482880524983721,0.29784683932826994,
 0.9413702416434312,0.6082399655765326,0.5458260491456959,0.5071918213757852,
 0.3641743532870777,0.3225708405736052,0.8014144240853455,
 0.024981993925116797,0.9934303625456163,0.876384715251493,0.5853671442139995,
 0.5451604185462918,0.06544442911714365,0.5255746493745563,0.79404437395662,
 0.8783127941302952,0.36775061006802456,0.6685362261399045,0.6365780887424032,
 0.6291632139578731,0.7646701605841025,0.586882270310271,0.12508098729110273,
 0.5707604475632637,0.03210575578829333,0.4107948954895695,
 0.27930196900457416,0.9499187510077717,0.9590071829448805,
 0.060704635025879305,0.5748137642766374,0.8593810912398049,
 0.24031813319802897,0.26189353305925933,0.3573948993777698,
 0.5019598657457607,0.6218511086196644,0.8267667673441303,0.19997676691314425,
 0.6206979417343855,0.06086142380640447,0.8528109436776967,
 0.13860240054237394,0.6199706289248117,0.5558424821147204,0.9172247208910509,
 0.5063399552881577,0.26007407280593964,0.9956796044701883,0.7988449125765126,
 0.6642394048266445,0.2601111941463089,0.9964061631900829,0.8836943270607559,
 0.6097603739128876,0.6472377134985688,0.2892056816316676,0.7567576852187846,
 0.9126907798918009,0.04278705319809295,0.9532161604797219,0.5242702340503922,
 0.9757158372401227,0.5407914516112192,0.8963735711590954,0.45715301807620945,
 0.26710785755536814,0.7198862417759008,0.835684387900887,0.4252133449318526,
 0.5042980076338593,0.08320557549198582,0.5269680787591632,0.7352653417194324,
 0.07048848262456386,0.4328660119619029,0.3924311446853238,0.9378630343721283,
 0.47083011008853903,0.8769156455357566,0.39144719791344873,
 0.8932216319596539,0.4892272858599912,0.4742222221699902,0.10089917224263134,
 0.6905426974812389,0.28521224548759694,0.9715832505586041,
 0.04200188001881844,0.4893164502981078,0.8148046582865555,0.7951974169388736,
 0.9158491494379691,0.29709678089195535,0.7086718090775755,0.7697574276559336,
 0.8235960338496932,0.6453497487049425,0.1983818408733069,0.47999995006400964,
 0.5960866702573092,0.7375908321375472,0.3217940352280094,0.2027645134055729,
 0.7971524245907976,0.7508168617593837,0.9431866943137491,0.5033420703435099,
 0.8164209647279526,0.1520095101751876,0.46556675536335457,
 0.031800111285682275,0.37601528656810423,0.29955621283390343,
 0.4556360644140234,0.6023023613818279,0.6949669749439186,0.21683937782123242,
 0.8122004616145156,0.7892473435531191,0.3026681741806738,0.1870867926532339,
 0.37234238974124145,0.16412668190195823,0.14976934932515185,
 0.9343479579163761,0.48756216657752083,0.08620940686743173,
 0.5687457797653426,0.429309254573111,0.03841808789097734,0.9215716685415277,
 0.8077606384613919,0.4537152780773429,0.793894837283653,0.007243545761859704,
 0.9274796214681389,0.09513205486321985,0.3941170815467574,0.1186916468325191,
 0.8383942276732528,0.4429476702549371,0.6451363814054866,0.20987763817458416,
 0.44252516720052193,0.9253520749757822,0.904596043134768,0.21760859882820482,
 0.38885542590734323,0.948458797565222,0.21346390061772402,0.8769891483747838,
 0.187645302436667,0.850403921390386,0.3750946317062156,0.9941983753850714,
 0.7592924080194781,0.008564193437656353,0.66589786513475,0.03625104455669759,
 0.8222235825227729,0.3594239569480383,0.3825697421473703,0.22833576646324993,
 0.2547630593506828,0.8196130625930957,0.754534677103584,0.9621939986330135,
 0.5662421561964273,0.7011293768920273,0.09994343029615271,0.8381445542996178,
 0.5342660579328676,0.5897507386086396,0.6555024154864464,0.7328827389210186,
 0.10674869695067168,0.9150227276165428,0.8281124912604056,
 0.004076366896079819,0.9385917103748486,0.8492519891092475,
 0.46370540948410144,0.2939897070738374,0.5028350391744649,
 0.24620034020280712,0.3058909290213575,0.6369880967888144,0.6999273302332666,
 0.023475462166852035,0.4876809674768976,0.3460947882016697,
 0.06112452252130829,0.17937381475630387,0.8357358201373268,
 0.9267834823261125,0.7891072726663172,0.19090532954557948,0.86052830753218,
 0.9513616077239372,0.6422296356626418,0.7939211775372695,0.6823366397551215,
 0.6713345955439302,0.42645006889535586,0.022637752807577294,
 0.7368842256487736,0.06381752626809167,0.23095200000344318,
 0.7600669350562067,0.5351401253280247,0.7238992502698571,0.9961336899359066,
 0.9082524958980206,0.3182137949807383,0.9406266910186424,0.2804307912762568,
 0.6430470433145471,0.3456700146648911,0.3479844341102625,0.37847588038289426,
 0.2101608891350888,0.3417721589792969,0.6467025666311558,0.3525356993124207,
 0.7519476981002329,0.8228171986547979,0.7180477618710945,0.2210829447594509,
 0.780379269144933,0.7953160276597298,0.943434143758667,0.20383653612423658,
 0.7021506319554237,0.7679871810784014,0.6266970780173636,0.16165678824885876,
 0.5078251958779806,0.07716413557938939,0.9515816469207562,0.6442451994375611,
 0.7665599611252459,0.9719198949180164,0.09040215968390386,0.7111294729545807,
 0.3462084980714767,0.99687496561102,0.04561837772995592,0.25265397665002975,
 0.8996257042464677,0.9719824897855786,0.37909765131805084,0.1607825035249565,
 0.28703058770456635,0.39148317960630363,0.46688847815028656,
 0.4069352863436989,0.5578604271489611,0.39950358705458655,
 0.32315263128601057,0.724743179432203,0.15883582257477624,0.1879865125690885,
 0.05404088427488529,0.8316937478416349,0.09493701471966831,
 0.2880711845449154,0.19697758118258524,0.8653649182570582,0.2850105010763777,
 0.5537535732852334,0.1261453085067492,0.04758798544747311,0.5978076661125694,
 0.7078988177310549,0.9608831063291517,0.6440742587474783,0.16777234916713757,
 0.35529726880632095,0.9647947156784155,0.03463187694153935,
 0.9069118568331356,0.014774902699596915,0.5640988122421863,
 0.7557591985543273,0.5722640163190078,0.874595668944872,0.7437005446946962,
 0.6075196398718491,0.07274470351146423,0.6041324235104946,
 0.034426720609829764,0.7846657883545631,0.45270246446823403,
 0.7499862785550283,0.5508081492372725,0.24129200627898162,0.3073879129637386,
 0.5336502644317087,0.7513116041480393,0.8402312521212515,0.9183813212848208,
 0.9369560095948531,0.1933135174628292,0.7284614997058974,0.8390985473068716,
 0.037249825442976725,0.28222792875244385,0.6344656061082022,
 0.3447966239933684,0.4487100121571397,0.5763439330311204,0.3066127041306762,
 0.26771660911420037,0.15403485139220152,0.9849028195616588,
 0.26363947849505653,0.8386943708723391,0.8188391078641272,0.5581863666339824,
 0.05502522295842471,0.9299029809425293,0.5977929585535695,0.5502522292683724,
 0.4261565131472834,0.2461938193074823,0.7284741845505943,0.5325350786041555,
 0.14878694109554524,0.29511373742575997,0.49948326808008603,
 0.8844612590642696,0.23734044903884022,0.02619854722882531,
 0.5058600249874127,0.8766101820441862,0.45590499217454517,
 0.02041936011855705,0.9617050914035952,0.5487891131883583,
 0.45149209107457633,0.557965396861092,0.4245949757859473,0.12382511034512922,
 0.24566022327285275,0.6301818301912371,0.4860428631270054,0.4678707717011934,
 0.4463229425410058,0.4056490452422572,0.2309628998777139,0.965734820594417,
 0.31504185977969623,0.8255938440528476,0.5923621889881484,0.8909154847260198,
 0.5727815992352161,0.3654427843395681,0.31348654004347865,0.5031334173220345,
 0.7305027921846553,0.28253602016301627,0.11313958237881538,
 0.9801129571806418,0.4670678259587199,0.42587507957819903,0.6757298872027216,
 0.8708872087028324,0.13652139196066426,0.8998640928425544,0.3532769517018375,
 0.002662430530914306,0.5665545097882247,0.028844524372152014,
 0.2522235918165733,0.6239590706318224,0.7690603368938469,0.9859301178146594,
 0.1373193346782975,0.9821264056884182,0.36050055724930175,0.6267252145874133,
 0.05444761614072313,0.5204803630989553,0.14304377778200528,
 0.42922239487667135,0.8831353678890725,0.6780180739917686,0.2872511800970369,
 0.45265525260163186,0.4335170844710946,0.7896951107857387,0.6789530106305028,
 0.8492577897682559,0.30501314670032376,0.9800928655448193,0.1664421556891177,
 0.3412419678467512,0.8454179287117226,0.6804473154918044,0.48180040034594973,
 0.402749265008165,0.6243125089437801,0.9316504091278628,0.9173987825436278,
 0.7337983590768795,0.532335480456387,0.9778431148320613,0.6697916884464713,
 0.3617990570588816,0.6398267065431436,0.463475610472881,0.32289843219031533,
 0.011520379933831926,0.9955550944206919,0.8508818034834054,
 0.8746440484894734,0.5256020722794759,0.05133446515536355,0.7310962238035295,
 0.843133066765061,0.8390142623001626,0.24047723120509346,0.33287074490310564,
 0.37546641068424624,0.6575003857837126,0.620917084878929,0.49636342597814564,
 0.8767100397044135,0.9326298932809379,0.6321408782867818,0.11529681646804002,
 0.4330222850856493,0.8088943142607892,0.6551341319721441,0.6303350987349953,
 0.4566851150610447,0.7330507578380688,0.9434770096097359,0.7410750123766755,
 0.5532522408635211,0.18987954458213174,0.411766267825434,0.5294411984174582,
 0.678222166348974,0.9300245960775468,0.7496592212382559,0.17692922087406138,
 0.4004430366926851,0.11501195706094614,0.047526057502933705,
 0.6546785218464691,0.8119727341645043,0.4493931331344163,0.9886983898763713,
 0.8170389015482294,0.8227822776132946,0.14940606495756048,0.3920273252277341,
 0.42714764553670925,0.6889585891089046,0.06877632228096087,
 0.9210608793081081,0.3742890356429276,0.5390049102119239,0.9402932291579371,
 0.5852594131530351,0.38182241482727164,0.4345657556169866,0.8034569233837114,
 0.8158077854334261,0.40780909523549524,0.955262901497433,0.1529692520613669,
 0.7180005746296484,0.18893712450114208,0.27968592162987627,
 0.4891437890774578,0.8046182936264241,0.5763430661597828,0.3699038244156513,
 0.13301966554701705,0.7617899848118399,0.8008612905525998,0.5589841285620479,
 0.07040737227931115,0.9846916951858987,0.8530387069776618,
 0.41406240005703354,0.8948194799182799,0.4645615293907839,
 0.030291679232541724,0.15681053285769986,0.8321696581927136,
 0.6550248799008188,0.5764752538088375,0.014346484247514502,
 0.7409794814477461,0.7924211562936245,0.833802629322013,0.13802271608404093,
 0.29057494639740755,0.3694155094284517,0.47561526014726674,
 0.5653325383119787,0.5717899730724157,0.14385158626654904,0.7687666539678437,
 0.17994427415377046,0.7865236591564948,0.09196791678056448,
 0.031827647429876316,0.1730129903185006,0.5147170859102626,
 0.0718143115439629,0.3300139843298394,0.35908868981378095,0.8867811430251078,
 0.6069828796392556,0.8267522116407865,0.3033676615347697,0.1730038500976785,
 0.607538669331587,0.0603263147085642,0.6491687644507615,0.7951762444942317,
 0.8777729385509494,0.15982085264486945,0.15258739622862594,
 0.9128971012130761,0.07205512631470856,0.7536365727843626,0.6312377870634436,
 0.3224451752257671,0.8873182591141351,0.28421214108471293,0.7430299786915199,
 0.7419621069426209,0.004246113470695478,0.9157410699173781,
 0.7183179833531061,0.3178539522538222,0.17134985258768864,
 0.16031626518794828,0.7272856256060826,0.8024615364296225,0.9143770752004221,
 0.37240531831606294,0.5040683468837477,0.11295023547035188,0.225231339502317,
 0.6630646223839001,0.07845301040991781,0.37136345876503185,
 0.4198411404874719,0.6485723942927366,0.723270485193244,0.10589222001323861,
 0.5727219344182919,0.31652102602746734,0.4788980026248383,0.5301870843295096,
 0.598383777241813,0.06622013112486402,0.3519374376993738,0.5232862969510611,
 0.6544433065520481,0.8595274991794114,0.32034286439219084,
 0.28146654455994236,0.6329339201679473,0.2765190113902678,0.6176516365711744,
 0.7951705675269857,0.17077911515496658,0.2012282381260938,0.3630076551133121,
 0.019366386778273403,0.639456682616539,0.6174213030306059,
 0.07672534398244668,0.36274865552752455,0.8762894664610344,
 0.9253293303719659,0.553842676295246,0.3929587113704228,0.06870932291984222,
 0.9617166104872962,0.3462158659494875,0.7948390683475187,0.10575969142618868,
 0.49232610380625796,0.5416850915431501,0.6665151397487307,0.6090839786974003,
 0.9615005336273676,0.6722158098588293,0.4600125873763711,0.18557499734058813,
 0.3458212555205695,0.18076467543622932,0.002203869493665911,
 0.45399134254628737,0.5429475049328408,0.6946669741641449,0.9400059148370011,
 0.43780270574271174,0.349825067967824,0.5830260775613133,0.561263800614358,
 0.5798172962190501,0.4623847152169184,0.41260163159284546,
 0.10441388770733495,0.49723637526800024,0.973836597031617,0.2753548677309202,
 0.9989204442446423,0.4297246925998712,0.4314444864454371,0.33255823990218814,
 0.7137712397082867,0.6127620571240542,0.3000094680998504,0.43235998564369105,
 0.3195382555471863,0.004230994047897618,0.6889770213131455,
 0.8576573685833366,0.5428219889735392,0.9839000037195218,0.4555394107924393,
 0.7722110440331935,0.28874097063371995,0.21159420607727397,
 0.25559223401309794,0.5089225867329817,0.2808093617128793,
 0.059993434593562545,0.36514877574986304,0.688752440068806,
 0.5270391893581057,0.5593369512195278,0.14133986701680734,0.8968842202709008,
 0.6831122599497192,0.40432804449833915,0.2569651614467009,0.84396765148379,
 0.06099814158040273,0.15426925320090645,0.356771374587722,0.6378266383547351,
 0.7000866762407959,0.7903003313400869,0.6748373831052933,0.25536232769593736,
 0.4811121598055774,0.8357448412985375,0.1401126673457166,0.5229388736425359,
 0.34404742564962976,0.10790552517013519,0.3811392291054915,
 0.7316730813121829,0.2692845560512648,0.32436065415077786,0.3811833903167611,
 0.5239040388041135,0.32846359045758877,0.6438982848453396,
 0.36295173543230685,0.035448908158859926,0.47310630490651406,
 0.05552753375624753,0.01843912084625654,0.6618046423913763,
 0.3066382090507972,0.03807108269405812,0.31229210166522303,
 0.9283517705322846,0.3883177831320378,0.7009142933989908,0.6549755340123671,
 0.3220732505828695,0.6075097216908781,0.01556360191699635,0.3354982139130014,
 0.5532070892309227,0.6875785584328888,0.062450446668968596,
 0.34331656390887566,0.4711202228139102,0.6476532279668071,0.4507619277333119,
 0.20259575588812262,0.24441224398507333,0.5803304982211117,
 0.7158232333843372,0.47201720765700395,0.6090544632480773,
 0.41885409433816534,0.5247710179660875,0.034612461634341174,
 0.9844293956974587,0.6630147718613351,0.40792395332997333,
 0.025306110894306344,0.18447792000103946,0.763119530167055,
 0.21671094505510013,0.7379452088683558,0.6183927938731711,0.3957000909452849,
 0.8440423174550846,0.9870588686368179,0.2410027151017149,0.2563621535335804,
 0.8910196695062957,0.10681511840506197,0.12337045844172212,
 0.23403383657636911,0.8806210334292226,0.7650420092609476,0.4491879472073096,
 0.8788534720276402,0.6877922144700456,0.37541175192658016,0.5472441748076765,
 0.5679123976967702,0.581137621309103,0.38999742142129157,0.45465480202225617,
 0.43843119042552914,0.4174715908034393,0.43421383331891894,
 0.2569412873068855,0.37995499584344805,0.9070903099034022,
 0.10403733001166804,0.7315441034672614,0.14091933032914172,
 0.4286660508516853,0.39929100780392035,0.009231293717010036,
 0.4569048306163149,0.2854890768003764,0.2073521539545815,0.22897491755379673,
 0.4186872457091021,0.09943780844450456,0.10812655933880977,
 0.8036105972161994,0.6686328722650875,0.44763258619213775,0.3167005892963126,
 0.5008628023657895,0.3551407040180664,0.43440907044891264,0.723143307115311,
 0.14822870996492565,0.5890243044058445,0.6337468306276344,0.2571727669884043,
 0.31824760918476747,0.5894984677369992,0.8179673411066433,0.9125179058156835,
 0.49865651551365076,0.9373439620001868,0.11355171427383925,
 0.08832653030646453,0.9166132311811485,0.7796685767014081,0.3375725938859059,
 0.5682861538090496,0.873386061296914,0.8667645491711478,0.23304693009143995,
 0.7490506332723013,0.1393565720350588,0.737336765390061,0.1209732149629631,
 0.06876876711084623,0.48905883359659785,0.03935544995716589,
 0.021863048731306067,0.9591314709688805,0.4117233337344095,
 0.0790366462910741,0.6297259026955201,0.9464892844361572,0.7747955212776061,
 0.754207250866218,0.49223378468201584,0.8027624758252098,0.24605516672862027,
 0.1578403026555557,0.7370973815615667,0.2591090215040118,0.9716990939581506,
 0.5784126674260242,0.6820693521572507,0.6630315433013926,0.4327769597639829,
 0.8015469272570108,0.02488932149393097,0.8032670698677755,
 0.16751364698651572,0.002633798288899758,0.549911766540257,
 0.4036173668167451,0.6840272872661408,0.3850100409984337,0.6645188034764242,
 0.7126917255651417,0.3538140224095032,0.15176520696754292,0.9786390393797048,
 0.9158197949904894,0.4121775651570986,0.7452078967517807,0.36040816670191067,
 0.09519172615272253,0.37397613442212274,0.3458663627767611,
 0.6710232440777668,0.9320964018394539,0.05690630577130684,0.9639051059503193,
 0.8563475044586313,0.7404355112933693,0.4647213811047365,0.6760511030106275,
 0.2387669530663944,0.00861922974438356,0.7766769551785693,0.545581660086796,
 0.3229790639239155,0.21116638301253043,0.8862170438638894,
 0.18634397784527457,0.8481920692470549,0.6345782379662247,0.9989684181948761,
 0.939510080044934,0.13235252890991078,0.21599207183513913,0.8794108744502989,
 0.550889732008131,0.2552210074917891,0.42185184994686453,0.7647746781646511,
 0.8524617034185226,0.4812549399271,0.36786829047859504,0.8886430747408772,
 0.759806554410857,0.5964838203320862,0.8671330441536093,0.8202714863840448,
 0.11657093999421508,0.39442190128011234,0.32753229946587437,
 0.7443515796891879,0.5525662777645162,0.7145841934852225,0.4635100653858938,
 0.5888688400018838,0.32860244912725356,0.13737846886111837,
 0.2918338313427018,0.27438849169118895,0.4505749615860808,0.8458061032084214,
 0.19787939198866278,0.04438341117625366,0.8414844770816085,
 0.4032825656485083,0.926403178887184,0.32939495118057593,0.8621161783855954,
 0.6834583255018493,0.8991525326810823,0.2673383886089058,0.7614177283636944,
 0.9690825452726586,0.4352469345185509,0.5937528009568251,0.5314384914237819,
 0.13249570739351635,0.4859348761141856,0.2974074082995235,0.2885722923090722,
 0.08660026560721856,0.44650584345592403,0.4516050453313857,
 0.7068991817366757,0.013582049677471564,0.09048112217701976,
 0.932085359101508,0.5487180520271647,0.3226497571463155,0.018798622103398555,
 0.041009853222345516,0.3905865063822399,0.9737471072989127,
 0.3476053471568513,0.973477975097076,0.4046055702067033,0.6953128838107232,
 0.47879393773368495,0.0032071993391609155,0.1402257949417034,
 0.5762036583166129,0.7536258222438438,0.556915950547027,0.38288011135344635,
 0.6430067429459465,0.4083361909685097,0.672608345256605,0.7521068754022612,
 0.5801003319310947,0.5196339708880024,0.39214624631135864,0.9395977905614972,
 0.6087868605241363,0.81062306518962,0.8741851311379922,0.7686372005286238,
 0.7353443314504626,0.6666353346139857,0.12421146479073042,
 0.026559124674467416,0.14552863854175602,0.2604886762076203,
 0.12563109711961695,0.7552413696650596,0.7889645597871181,
 0.06453254055321889,0.6199904332298452,0.20042141955988257,
 0.7376843202544187,0.7986648911721465,0.8952118395573374,0.9683479822706498,
 0.45154067569663203,0.008098339974476176,0.8053877270864639,
 0.9798293988085003,0.10708343077156068,0.11821672437331732,
 0.9939345140917825,0.6395592379194852,0.8772117490784965,0.07039583063856236,
 0.28005749922552137,0.9866025508476154,0.271041274045833,0.6876821506849438,
 0.03130646495995326,0.1738846190012977,0.01265718657163728,
 0.5368359443879972,0.7465571302133337,0.41847573434879226,0.7577345565321427,
 0.9335163006489524,0.9074844939329575,0.36693758982665436,
 0.10724079198139036,0.050112800892767484,0.5667854369037447,
 0.9782943492592162,0.8851562073845359,0.7526011594456025,0.9149142739676469,
 0.4191071715435192,0.4517298280942442,0.016497161796697357,
 0.8461776426709295,0.8251420853980497,0.2487272989542666,0.4857917734263416,
 0.2117340027523439,0.853675416674835,0.6322616777240916,0.5120034583100275,
 0.41272937325086045,0.9258156178238148,0.8493537362449134,
 0.15684616960505238,0.49365120515159755,0.8732607262866549,0.14626657204353,
 0.7899973637956085,0.4157159465416652,0.956601318973016,0.7418145439959394,
 0.8724194999457392,0.40789198350798517,0.526506102255775,0.7796751835793442,
 0.6547738029441273,0.47537192673720097,0.48436050991744684,
 0.8011571328700765,0.9068139258375689,0.19023752415112816,0.8004843376520149,
 0.5054050949540873,0.6314449684543186,0.886011497496401,0.8756076092455103,
 0.5637716675413462,0.8817330378042433,0.8304357428217004,0.10656230157191049,
 0.38971725059462503,0.17784903070884683,0.8047844030272207,0.813557832758421,
 0.4229707326879082,0.5909565018148072,0.9258727638179894,0.14999742494171853,
 0.49725163612520995,0.10730732310163249,0.5829158663654841,
 0.34355915342404986,0.985528446065471,0.7890804189999507,0.14820032705010833,
 0.828482305311007,0.2910888823486081,0.9072104397328017,0.5399510744400008,
 0.7069142283791826,0.23954359062371577,0.28903507654089255,0.443942935491485,
 0.7883380840080033,0.35938883494552976,0.10387871181311725,
 0.43087215912348364,0.0757771755888973,0.9170996304560094,
 0.15156308213418523,0.23166675020381522,0.25850218205784015,
 0.5208641003494382,0.8997665053361406,0.2487759297804668,0.6245777043660353,
 0.8676424332173583,0.06693943482426701,0.5754708669980149,
 0.17936124022662991,0.5017887976777804,0.8958270238326397,0.947619942805473,
 0.8169120724371176,0.9923938486860098,0.26401196452778253,
 0.036930075077001745,0.009789278016776093,0.7928892512082234,
 0.2976778713820586,0.01528789075808068,0.13997722347624109,0.729262350506461,
 0.34225921086040645,0.0834825291634751,0.7005580548102927,0.5476980827387818,
 0.0931761934540023,0.4906420182084844,0.9870342001135838,0.8643321771191144,
 0.10319525757071712,0.8072953586325806,0.6275273333489098,0.2613757580720666,
 0.7431454503040391,0.01805107766008407,7.585066543671637e-4,
 0.5480071552475118,0.48381850052801734,0.3222398112817375,0.4871836101373711,
 0.5735449064783668,0.14359360674197286,0.09422019271133153,
 0.6821453688129094,0.5175156237445208,0.7470472803539779,0.8811714837658925,
 0.15480203710720009,0.2178388568251831,0.027880646682791665,
 0.9005607784146579,0.9301182826264228,0.6328343495217409,0.43992836717891526,
 0.2285239653449911,0.18111398468856155,0.4308927533182664,0.3408791039896264,
 0.13362581923142802,0.3339156605252308,0.7428698302101419,0.5175542174055122,
 0.5687868819044699,0.885978588316505,0.6822644336958508,0.9636433987621801,
 0.7057463702089487,0.0034992176836209365,0.39271816032410767,
 0.8294317948706406,0.24268250246731304,0.9500735446680645,0.2455235123170656,
 0.9034105728996208,0.4590353266008145,0.15747145494634363,
 0.10318335258398881,0.47615738126572893,0.29828918012313066,
 0.36603576371203084,0.41959871053819664,0.9321102117736041,
 0.17750871251025324,0.8140426591256708,0.4343868498812756,0.7418981288609364,
 0.2840658302999628,0.2658558066982446,0.5299538261794232,0.8150202882012991,
 0.126581415657247,0.7363025945920556,0.7121952402016788,0.9890734113907984,
 0.8715929929302156,0.761166434483166,0.6813230136830554,0.8899876835147583,
 0.13177736853261868,0.636303177462655,0.16257913621291775,
 0.41796771304032254,0.998050440008271,0.43728776310295303,0.9488150528609453,
 0.5803151485014202,0.07236919496426086,0.06267254640967124,
 0.5680946370457765,0.19512567552503923,0.08578396988306147,
 0.9856019872963157,0.3678988592950614,0.806901150785693,0.08768972309762613,
 0.2207048660412575,0.35205405260240064,0.8146464318920731,0.6435077548703214,
 0.6840596369893807,0.11440163293138172,0.4074356788377891,0.8048534143689893,
 0.9804616986258753,0.05752369194689799,0.053118377239525394,
 0.8480742552923816,0.23177667173839323,0.9207347554282475,0.6474140532302853,
 0.7952947137551916,0.4543622549790127,0.1500003543897631,0.13729222233095717,
 0.31984409864941,0.9450201319461711,0.41215701433126806,0.9543209226153563,
 0.5253550316675838,0.8962057171619496,0.7387165223946155,0.4117612097172453,
 0.8926081784833931,0.36770405078798696,0.2956810963740192,
 0.34463891523833545,0.07829767185953918,0.2113407367064456,
 0.3517480494877495,0.5026741922266303,0.5142019554336119,0.6268310313347718,
 0.3840465449513627,0.923771076275219,0.7196554922241536,0.5957432931120903,
 0.5070121579810469,0.8006185387094616,0.7444776049307417,0.12026196631235242,
 0.12871747289761715,0.7982620251154184,0.5396153442370084,0.808451240373261,
 0.4024656758244344,0.3472032386732754,0.9973010359890623,0.06629931115885634,
 0.6523344160413894,0.3019632329144316,0.1494951989510045,0.4551571807565271,
 0.27046902398257866,0.6391265299951229,0.6455598699446404,0.7860623634591348,
 0.737215625501409,0.34194942976119724,0.8491604228685218,0.11324960586809008,
 0.3360476401404088,0.4659813094103755,0.2576045283829739,0.8646406829993374,
 0.8903454252481304,0.9071512353302803,0.8895923929780688,0.5343880466550475,
 0.15033433785062722,0.7455099509451311,0.8627622169200233,0.5469543369197487,
 0.3257623087928839,0.8976117091238872,0.8197401575982259,0.9154479116490563,
 0.5676027667274062,0.9345275673237238,0.8534468026756182,
 0.015703572807013177,0.6009840923139569,0.7437040708611967,
 0.5722005331693661,0.8050880380552848,0.2349461488397,0.38410998599540913,
 0.40342444507694786,0.39691534915536064,0.18150256461807834,
 0.8569426653374141,0.8029916116412132,0.3359513889669623,0.7386627356187754,
 0.9581269281908583,0.6498454355792769,0.9483865802812489,0.1533898348955468,
 0.08220752151440192,0.18567972615066153,0.7051873633667223,
 0.9811022214817046,0.7673055981133614,0.5481390366109397,0.3985349718370116,
 0.9904667628820599,0.2706785410469069,0.5455002745086807,0.8504165744854449,
 0.8107859362616336,0.008851174439866272,0.8684943586540264,
 0.5341973534277378,0.994389860774057,0.11561782739859994,0.8825248160726349,
 0.11590527456803223,0.9366085567010851,0.8861590270738408,0.8643017646320493,
 0.3499582079156808,0.25096960583829375,0.49505614333044223,
 0.4526062717488435,0.4654741822331414,0.14167590049126033,
 0.36601818410563447,0.7905170142543352,0.7075177836735246,
 0.37655495693428365,0.6861558373471455,0.4907427109923612,
 0.008767700198397788,0.79628079496561,0.3494022790907212,0.22143523076697746,
 0.09676655306234139,0.9868742019765777,0.27133960009116975,
 0.2095660324955707,0.5010927681448073,0.4140431778906708,0.9173820930365921,
 0.5521353027114216,0.23517403653429447,0.811802168872475,0.9885788058732019,
 0.7403522557488403,0.026717161893783148,0.3080664741000272,
 0.5959802208890238,0.7806364925783296,0.22445765229324155,0.8466508529638127,
 0.26772844920268,0.9307938681787563,0.046888912017102324,0.2539052007472916,
 0.7366625518077607,0.06813105223736815,0.9823063745387327,
 0.20385549640516443,0.04141832977073012,0.6533221311126352,
 0.5086389625466214,0.31904107331325104,0.48752140526698096,
 0.041812483230043274,0.3677953360179147,0.3220004170915167,
 0.13413243947433906,0.11960997788542438,0.9498447802228481,
 0.34529034009785675,0.7377133798618084,0.3631082605980469,
 0.31119907109415423,0.7092931889605204,0.18086210683554027,
 0.7807659944560138,0.7766418313956175,0.8618518706616586,0.40068066371509126,
 0.33728979977546436,0.9055609457336358,0.7927593809870856,0.9037197250824661,
 0.8893032889905798,0.8756729303427969,0.15986626492370182,0.4107402256721686,
 0.9620378656470061,0.33008601598679777,0.45937415619225097,
 0.2751359640824551,0.8673562446192996,0.014660338330041545,
 0.021761944990362103,0.09722307356397819,0.20264383136644848,
 0.7476765934586205,0.4131402342709243,0.007703789534183492,
 0.10569308502322461,0.31431218823251805,0.3501273659709535,
 0.11652915737124192,0.11535311965212869,0.05345114416635666,
 0.16121837647791004,0.35553282686424437,0.03773752071447689,
 0.9674968967751252,0.47878570147613,0.11826276260679791,0.022710349215093295,
 0.7003520473686314,0.7024847533863273,0.21107709374817207,0.8596167228832872,
 0.5029365936427335,0.3818266265175153,0.9962728577340226,
 0.030218182544035743,0.08287682271784524,0.2679968001566554,
 0.4500098950742807,0.6905447144351332,0.17130265656297905,
 0.09367392651138673,0.1681872263727615,0.07215696289897011,
 0.4806904892801148,0.3751956064637466,0.8674616037935707,0.7756967499726052,
 0.24204415767536158,0.8255664340310224,0.9043270837633722,0.6298828212627872,
 0.2796399566956177,0.05702581682493313,0.6405211411396504,0.6445754528696539,
 0.777745660059327,0.11308843261740043,0.21437176553131432,
 0.10083992045237733,0.9889874146022593,0.4284064353334113,
 0.46383384810691775,0.5143126185124588,0.23700686091438938,0.775212698830336,
 0.4215139605632586,0.8265977422127412,0.31891170777584377,0.8322330505638691,
 0.8684423119638276,0.30588133648327653,0.04284762195563396,0.974659318716442,
 0.6376681758826858,0.7874555318600879,0.3581350151070326,0.16852190462478678,
 0.7037344710338509,0.598474046234985,0.06815680624402709,0.06815627771272847,
 0.011716899555748506,0.18496814389260585,0.19642039951146117,
 0.9320015755994959,0.05107736612105129,0.33957411217401157,
 0.9835528101893762,0.954219183540792,0.8184098051991504,0.0764745140332268,
 0.9779297046838248,0.9020277289826837,0.21218688284549458,
 0.13344787942148384,0.8362459715272369,0.9947917407883897,0.2718389658024716,
 0.04648875113878115,0.7451939749507385,0.9607592151809972,0.9264955887332335,
 0.6775830483672528,0.06847221233604683,0.35525898486240326,
 0.8405966323080915,0.7086524492008515,0.03282791017624609,0.4525791831739838,
 0.33059782015158745,0.5308901126146819,0.6632658058487922,
 0.17233531795618962,0.21817849966705338,0.5063429838101586,
 0.7488961918971455,0.35835677299934543,0.39339695320721324,
 0.5804714381444371,0.761897034487393,0.6179538478176065,0.38808132078656465,
 0.11491963336214595,0.6135023351652071,0.025911071195830027,
 0.5702367213748265,0.9075337493697534,0.31823103950799436,0.4902553318031936,
 0.2831769024398064,0.7284405779308295,0.9482551827758727,0.5108058221671664,
 0.9934206880650749,0.3491142031010651,0.8390947128096055,0.6125745810300861,
 0.37145639363453997,0.6584737208725863,0.5220968787377569,
 0.21597291477548752,0.17578107763406614,0.21875843533043526,
 0.42056536004563205,0.28402773182708296,0.8866083923106116,
 0.17217545635671916,0.04510473611347965,0.7811758738977016,
 0.9318616581072693,0.37993324523154537,0.2263985544301047,
 0.22852993666302168,0.875331762961752,0.151249952072677,0.20084127980395483,
 0.2397086510240729,0.9517677021153912,0.5463219048009494,0.6479703681327441,
 0.7805230429251335,0.3518852230929924,0.18589590269739587,0.1436207953288805,
 0.03721052069419284,0.16054332904720903,0.44349867225405515,
 0.2559378617257858,0.10126706687580422,0.6823145700910885,
 0.11012120131662173,0.5585146403713536,0.0659432048275157,
 0.28793871352193856,0.16247033823962242,0.5615474108768786,
 0.0983060866463612,0.7429901054472494,0.8300847611017256,0.5027283190640951,
 0.9731231132553277,0.7636021045720536,0.29135228170361427,0.5365706665804308,
 0.4926751011402146,0.8796114263938114,0.1995228728517663,0.32472681293607675,
 0.7556092941324449,0.01872207392721592,0.5219744865393979,
 0.32770595809031233,0.6878459462687596,0.5365315778523865,
 0.08447917936088489,0.3161863944864137,0.8097580233945684,
 0.06231508124015417,0.43190304135322244,0.6380429137184918,
 0.2262898660772551,0.7764225236829355,0.28343432941476776,0.6608672621085584,
 0.10190920051019114,0.843314889008514,0.13229226491371537,0.6268364068957982,
 0.22488488127048267,0.08333332601300847,0.19076517782158797,
 0.510183602928617,0.46472898762131265,0.03882600359400379,0.7646776585897189,
 0.9910497813874619,0.11796503126271951,0.8053775173587083,0.7395165321268117,
 0.3903917691750828,0.11537713422392426,0.14193435988093683,
 0.13283763687229855,0.4754877252320674,0.7573094640391981,0.9570682382991866,
 0.666300854255749,0.013323329370422576,0.6261737693934927,0.6287511628025235,
 0.8495071346395008,0.04959146321698105,0.9536882533965072,0.1658122561840185,
 0.006510128841651808,0.3574249580624873,0.7851966350930198,
 0.2567604469388185,0.6331632089692797,0.9562798643742797,0.4059520146347356,
 0.9720784301104752,0.1208182881115275,0.6305088132063155,0.7242616985575733,
 0.20785764486301384,0.5509347650496978,0.62714884233611,0.41700467897563454,
 0.5083763463588724,0.7710087796031205,0.3357486651431192,0.3672621667196082,
 0.20585728377938362,0.9650088433936739,0.03545598447303089,0.181935748837733,
 0.8195929158473232,0.3537108918130454,0.48850452247516085,
 0.42770882091599616,0.7369266253287728,0.9211304410514973,0.9758459274324441,
 0.4024195503777347,0.7910516029536897,0.9360159545180036,0.37660453337611444,
 0.06099917267077426,0.8195237912882589,0.46414983538274324,0.623985323747532,
 0.4629577648131752,0.011526748518447594,0.8423596425562939,
 0.4482414565025351,0.4266743787934061,0.11245987670773439,
 0.02966700498873487,0.01756171584027677,0.9195706676569387,
 0.2659234241384735,0.9356060923616624,0.613097379705284,0.6485801857522312,
 0.20963313177099352,0.5462669211366004,0.8390264563850889,0.4657319376222496,
 0.5543520889950241,0.6448273807995503,0.8368164352951397,0.4230019198626327,
 0.46446951579687235,0.6717546004219079,0.07859655725546433,
 0.23161506957394684,0.5912610107867916,0.6508954933576694,0.6500205811563482,
 0.35335535246815397,0.40066888471337914,0.3860581135554866,
 0.30422923236739585,0.33177364651383834,0.37059984308711535,
 0.24823559630306158,0.6467337950285404,0.9250765606069684,0.8584088401450088,
 0.21897094463332367,0.6203007520316144,0.8363812962219768,0.1567351583962483,
 0.04368960623615181,0.43872730329398935,0.16579913065031715,
 0.6787520974155495,0.01793550401979027,0.2531039422476101,0.4375765645227486,
 0.23839427701744198,0.9170987343560274,0.6185810021217045,0.5441211432821089,
 0.6571613319404765,0.423682163282904,0.20821683803685043,0.1816327153310694,
 0.183596602307864,0.14805076093880432,0.18790334972393008,0.7334298172662392,
 0.34856896358378175,0.3369783669711861,0.5998206979463852,0.2441777344087045,
 0.39703535406393775,0.4557320153805109,0.7408788760621211,0.7273485982184669,
 0.3075426294361039,0.27957977719535476,0.785624996868856,0.868258177066545,
 0.4299460689679143,0.7536273417326205,0.9562638249136896,0.9711912265417062,
 0.24458792356481862,0.49531679918897076,0.9364959375972586,
 0.5175830442661149,0.6562197669882748,0.1880247120992109,0.9641734850806574,
 0.5672960746287921,0.6386584174874038,0.4310331840993681,0.6594782449254339,
 0.7855581631040395,0.5061851990869326,0.6052229505182812,0.8733078819591658,
 0.42096672974368143,0.479595603695839,0.8777486740023273,
 0.004055778703068946,0.9644298208791602,0.2945514683112236,
 0.6231918816389542,0.4532920646904459,0.05103806583703374,
 0.19546725619348038,0.20762066625453002,0.27625497843632907,
 0.8856518693404429,0.6763572861095382,0.5987113410051542,
 0.019124461942966242,0.08455530826405311,0.04674493634456911,
 0.7077442700921206,0.5640296349039964,0.4237585025347206,0.2952664877680281,
 0.8719770007723774,0.6952661513578097,0.259340010350152,0.40957618720238487,
 0.12549263225279716,0.41726187133187953,0.41165178597604546,
 0.8200658163962391,0.09558349607487693,0.5323899870174871,0.9730139438657477,
 0.8720482314709884,0.6828264258506528,0.2963902367326563,0.43852929894089865,
 0.6235430122080551,0.12677821609041684,0.3821684304608445,0.5080572257039777,
 0.93617042359619,0.7694953661659979,0.35922236227307025,0.96782545532291,
 0.5952481988405838,0.327472265909582,0.8903929029234254,0.8820229727992634,
 0.3951335573785133,0.9019523084571041,0.46400888606729274,
 0.09874243638932678,0.3613318521216602,0.791772910645109,0.6297395202847779,
 0.906941924608042,0.43546728241929156,0.3962992940021063,0.9345442900139671,
 0.04036546334915847,0.3107702674699915,0.1489728209708775,
 0.002592775406880965,0.967323656472523,0.2238173455550434,
 0.21352047473530966,0.6178850481971288,0.7530354329012816,0.9905590463478906,
 0.13135879440937848,0.5510481257926872,0.9014653273025882,0.7390239938801103,
 0.575658605054157,0.4723401429128793,0.5906751316696593,0.8030638623893844,
 0.07382678496764083,0.38883235367949975,0.7004998070163772,
 0.05199555077682383,0.8664645479296945,0.663033831845665,0.6283211205168957,
 0.7639183432886545,0.04017264255908648,0.6913948735030576,0.3021959391567799,
 0.3003621541397178,0.8855523325059106,0.9126464619464295,0.9443526986569689,
 0.5118866646988548,0.8242149937518866,0.8730969089811825,0.500088574001001,
 0.7551053344570045,0.963982386204234,0.2848745187377577,0.9047782275937143,
 0.2013740600719255,0.0322973757260826,0.0975760157714789,0.1029346663167574,
 0.8468087009549718,0.5453158585319559,0.31057008060923197,0.7883223778290692,
 0.1627598109395203,0.5199519076075809,0.23536692519956592,0.3535618911170565,
 0.6483141733392784,0.5005507662523205,0.4426535719510619,0.2064639783402632,
 0.42619713183398056,0.5889542944799255,0.6757365950285733,0.2532371958626717,
 0.7436821624013422,0.03647578495668091,0.9872589355679582,0.9269566610987199,
 0.08154330941243015,0.8341578315772517,0.3629689980405031,0.2843433155977757,
 0.9496359307883142,0.6533787615702603,0.8127356050092363,0.12164379587596597,
 0.47501208991012267,0.48687408619403216,0.9605351636860958,
 0.08630749911426172,0.7310093810999788,0.36690501157859945,
 0.5160498087536264,0.8027072867481652,0.18815488637564748,
 0.009522584092140685,0.2156135374639251,0.22073955554622127,
 0.6358936151036998,0.8840582623421704,0.58846798472066,0.5167158821705221,
 0.8586604903525732,0.48286713474144094,0.9229533619483801,
 0.26300195216038236,0.5850187167994869,0.9276877993889836,
 0.38132010489337886,0.5552838609368348,0.2276919164619884,0.3429069595833477,
 0.21251684252624758,0.7998431289281857,0.5615577829554637,0.5974089112361055,
 0.9642026279417761,0.9063355832338847,0.4162978758275384,0.15921000463662682,
 0.8471620049611408,0.225393185193835,0.4665656944737123,0.7096870551888396,
 0.7619668055740815,0.7176755361346449,0.7874577788103917,0.946676314220017,
 0.8805925904348044,0.5576560923776279,0.3619802124403103,0.23292228339253518,
 0.539457455397625,0.16642683728822072,0.8581513031674167,0.5069744535553761,
 0.19799017164948973,0.902250684009468,0.5887312981767516,0.969497132932015,
 0.6573619149349652,0.004275797670607817,0.49338644499854967,
 0.9337441294393004,0.6684789929709927,0.28637759683346675,0.8171358917500917,
 0.690318035330669,0.16607324280833236,0.7551858680902034,0.759881334335238,
 0.9030355569460332,0.08876537700093567,0.3083091939120045,0.5785175673784066,
 0.8746473228657932,0.31267102586448914,0.25090011960826164,
 0.8227073074276989,0.7288035836979986,0.2040658026405059,0.211794268771474,
 0.25302153896225466,0.5502517518837012,0.5852110647122418,0.880860299700002,
 0.9634097823043003,0.37380053790266876,0.051624161925549306,
 0.2262391533685565,0.681904880218034,0.5545565552281597,0.3360230825609978,
 0.9494143743311442,0.895816583664,0.5699985077027756,0.2879573345649603,
 0.7330962790538275,0.7418228697446019,0.8088302387070843,0.20252470703391168,
 0.023355125223186723,0.1348645770465018,0.5173154871025406,0.946572381441253,
 0.8222858571159114,0.4561301415536887,0.42657072503629523,0.9984074636577095,
 0.4325895403743172,0.6032379019336007,0.880062181697847,0.43137201921270096,
 0.7453011453181972,0.3946290610588735,0.6037393985226894,0.45350417368625084,
 0.5987879851402131,0.229377475713737,0.862652260434091,0.7994163538840158,
 0.6709871701780116,0.6929032528023624,0.8439339574457154,0.5053797259627772,
 0.049163553819093986,0.5996525709193969,0.9968612627112609,
 0.47836591200061274,0.5678766486935177,0.944251784774407,0.30975102856864356,
 0.5581284810034663,0.28248386183846896,0.8361485886615713,
 0.33123053732905305,0.6479662096899,0.5694184535382095,0.7552192751318222,
 0.7228387488152845,0.369617582145665,0.7763652505669173,0.13498416981108308,
 0.43142271211818084,0.06177655341537991,0.25837983752210214,
 0.914740953415018,0.9709453527410699,0.38960979768540493,0.3738118900408759,
 0.8243638909428211,0.1935156382455343,0.89687238103581,0.0319672230685103,
 0.027373103634407148,0.6275140762819932,0.7227961362650726,
 0.3758652599324992,0.3972724381727252,0.8516828958514724,
 0.011611315281342738,0.7268848822078607,0.9268505885118462,
 0.8206734529272258,0.2416135324963311,0.9031698002736074,0.6933912296366688,
 0.2899736223357934,0.7014868228717202,0.023847808624058886,
 0.4573478919342361,0.5194503136326789,0.8484733224911012,0.6567207904843184,
 0.18721549445053198,0.03536558172743032,0.14656790648587747,
 0.11239995008016823,0.4909796455369505,0.10328551546759601,
 0.9743390863554464,0.36427177802076427,0.6911089346409599,0.4412156484166243,
 0.7727763492557411,0.3593825739130928,0.8394788207267426,0.4184876803918067,
 0.8410083030273628,0.7380496502560896,0.5312426901837329,0.20017892179797336,
 0.7726136539934975,0.3933059888232152,0.5536160484087252,0.8048490573800493,
 0.7323160021004536,0.28999440830654755,0.021819068170817135,
 0.6266721702712577,0.065998688908384,0.109969524508292,0.11004597449713982,
 0.4613968872382439,0.46999356519771185,0.24273161769687446,
 0.5128518744203403,0.09039971378360934,0.48377210895015954,
 0.5682494566221689,0.45666685546868946,0.07071544380123229,
 0.9923472185756502,0.7474518193638117,0.5204700113716004,0.15631248811064669,
 0.7026690167894222,0.2299861241966049,0.5374044002514987,
 0.028755391425421717,0.7442571119395807,0.9969417672117444,
 0.6253260952317093,0.5068308757340761,0.6450365039300069,0.6659434272824892,
 0.44140906758053167,0.11668781179047383,0.5937500008721149,
 0.28716505645558166,0.9056582389111806,0.24165502605455824,
 0.18044628144426533,0.5110787112485438,0.027074363637048182,
 0.4345439646309345,0.2955318726883134,0.9383643712454136,0.4050564456472756,
 0.9434018324818123,0.9048662439822369,0.5078006909737979,0.37044260953798036,
 0.5525954258185789,0.604824317859051,0.9890194659965678,0.9805094358034037,
 0.43989934458149427,0.259033301587252,0.778109088168089,0.5537532728941974,
 0.5566527260917512,0.8082807231003694,0.3043540297158984,0.8692094979473626,
 0.9909662580077888,0.17217664336737648,0.13632443173881925,
 0.1563603040066296,0.35172848702831905,0.18267538007178785,
 0.48413283396682005,0.6005592687204072,0.10335345822236086,
 0.03365164201217019,0.5661301411651668,0.1522696385894705,0.5950283454415999,
 0.7333176765475036,0.042624964792929654,0.8663466820216779,
 0.8658612246831747,0.296790628499376,0.16811210903823348,0.7375260063141664,
 0.44689385821138994,0.5748528279468976,0.9192966860898727,0.6466436587535589,
 0.8600314975132937,0.015821733547889516,0.3287124403272037,0.925017933125028,
 0.6594351322214763,0.5029758244720375,0.0836928564991033,0.42549261673120076,
 0.15820492777085615,0.48686597076193183,0.076413385636799,0.6549824002019325,
 0.3895382000447124,0.039823121067251055,0.5803791105248947,
 0.5858185853518951,0.1044594734157468,0.879369084503854,0.4278697995797158,
 0.6722001021859312,0.5944990456837298,0.930860300638342,0.3851693392129186,
 0.665919315416097,0.805702529171028,0.3383669997333745,0.5608795688376365,
 0.42706794464180575,0.43044432418956424,0.12852649612260025,
 0.07600309875472122,0.5141518148503501,0.9075577585930193,0.592584396783617,
 0.6691123263821995,0.7484959959124041,0.8686720107201233,0.9323907144158613,
 0.21034114197379827,0.31706450074264203,0.06179379022930476,
 0.919132107731959,0.7708222771746654,0.8474883228676261,0.20440063401393538,
 0.8654707622574744,0.9685643261992514,0.26195936689666344,
 0.40791082534294765,0.18095898264633192,0.7353993848926561,
 0.6461510262217878,0.4276434510458753,0.9225055059794207,0.08935489769572136,
 0.8513526115000283,0.7416240944226491,0.8040241277244362,0.6655174211515011,
 0.24087055711381544,0.6433000283411467,0.3553195878214592,
 0.46540499854160244,0.25485442680030557,0.3689141001462707,
 0.7600142012077324,0.7304302882587995,0.8093183036793198,0.8905132230398758,
 0.33421062047162875,0.25066928729076343,0.9529343059546865,
 0.8531664050802432,0.754438498675666,0.055790941565513896,
 0.09526933161739337,0.33664305081436385,0.048007858791075386,
 0.25237475491094097,0.2891210241375465,0.5292449005556459,0.7252587004034328,
 0.9868515198500173,0.05818385269094151,0.7449174536298335,0.6333216597109014,
 0.3951019733801613,0.7142159123963571,0.36122460805310974,0.7442290532526119,
 0.20372291078260862,0.7819316850498811,0.30919076394301037,
 0.7743952943437692,0.28503016983867047,0.7641185618586129,0.6813659770986837,
 0.3557906699346398,0.17459039243111008,0.7309863531116205,
 0.24751157523004175,0.38258187487781503,0.4426258656647932,
 0.07440706544015452,0.1610599842769347,0.04246561289626849,
 0.8392526708784579,0.005116939449011482,0.4367971813712309,
 0.9583970742528821,0.2371769991703273,0.1813296900018625,0.22607900497249278,
 0.4675829229241717,0.06859713800507294,0.7586611191956905,0.689878611232573,
 0.6574866648678911,0.03136509548120414,0.4246868103385001,0.260496131064776,
 0.24259089132567202,0.26937715078275604,0.8687577313196024,
 0.8600995060849245,0.49956597005764225,0.744126936925882,0.15847380941099187,
 0.22242759539400392,0.11332771203494696,0.7492530652854278,
 0.43076609125260523,0.37914885011105537,0.3410449563254685,
 0.47993085477058395,0.035383238249234505,0.2895331115323393,
 0.8474120928533612,0.9291790327009903,0.8457464198218855,0.42807741973476066,
 0.6877575943021739,0.025874776860820514,0.1906072397541917,
 0.30917405509374185,0.8681425665084709,0.31097523898049606,
 0.6303126712864608,0.3172132321612232,0.9012088816475673,0.8979757975513668,
 0.1653579250546081,0.6821880614732824,0.6311447225798759,0.7419274334647871,
 0.9918621746285344,0.2639009219191637,0.4585400026287223,0.4232199740633531,
 0.5127964193293068,0.11229817813264675,0.0427908419414601,
 0.46994132628591667,0.33472385210185773,0.6665828378227369,
 0.9386699998458957,0.2830809593335488,0.5805620431267862,0.7251848811471382,
 0.29668311655570223,0.2447756526616649,0.9611577875077931,
 0.08855987563939405,0.4317527043277902,0.5308724580698846,0.9532722229502034,
 0.13022837473674143,0.2737453578923137,0.4108874195918565,
 0.43920375578146176,0.13906449801963594,0.33234209938080184,
 0.460838838518115,0.32999078134063875,0.7550058784065198,0.3467858022872542,
 0.9561901956802966,0.76862722758302,0.9915872482781318,0.949424963465953,
 0.5363886647917098,0.2845479585948374,0.5154411237726879,0.4599639338482312,
 0.5358564733497839,0.9194913357095803,0.6582999667079432,0.9818255838573917,
 0.6247943410473726,0.3090994913216485,0.816899338925038,0.22675067908606228,
 0.9655657841993868,0.9299034887510041,0.4742219872739353,0.8971945486838192,
 0.7634945740491864,0.23689113554380414,0.254270753774702,0.5148585365817191,
 0.3398086743274459,0.6370045964962757,0.6420614921757314,0.20505559309339183,
 0.7908958947021111,0.5219144497093411,0.4818392237021011,0.42254827577367327,
 0.2615191407248515,0.8793865801441093,0.0224931053320111,0.17160024063356127,
 0.8671851504544,0.48114212592440364,0.9551429879991122,0.493857260730887,
 0.5986153211193255,0.9638029248026768,0.893146337217949,0.8143392976061626,
 0.2930945021702841,0.024221664499395,0.42967211917720816,0.6846425191433503,
 0.37513009290811994,0.7312862500051077,0.4767589120958421,
 0.001978758367185418,0.02508718678191557,0.6202391794458926,
 0.4167159302142309,0.02447818941631197,0.35276293725923713,
 0.39517952018029046,0.8790362825426232,0.3530068884451504,0.5873488362153548,
 0.7863765650508675,0.22434626785666734,0.4651041964359469,0.7651231666310712,
 0.25245265982374676,0.7366285501254681,0.4561219021208548,
 0.45650877293768843,0.13446851315261943,0.043097230376691886,
 0.9278219895261712,0.06883619063838509,0.5126560142108203,0.9898335644941461,
 0.21309017011868958,0.4990820247544959,0.8311631595135117,0.654046634826397,
 0.4589517109823765,0.6987027827174179,0.28676221390821566,
 0.15910692306508611,0.7781886814007616,0.9924546519244154,0.2744470538238707,
 0.4149174807646395,0.456298262955536,0.5896622723708588,0.07542523579846228,
 0.6917907953487071,0.4887008228137638,0.4141622899155383,0.7115693069615303,
 0.8469515270632105,0.4418154462034105,0.08985011901772433,0.669178611698145,
 0.23441158569066223,0.5442959945241705,0.5927311464367901,0.7358799834378874,
 0.24560981899841838,0.54931935899791,0.5461251808712018,0.995022795789979,
 0.016175503379529443,0.8138655579904079,0.9203361889514979,
 0.15884549735769038,0.3597913945299247,0.4740181533666369,0.5428252650607245,
 0.8374129233208778,0.5156768700479549,0.7114232517030702,0.6538190886874032,
 0.31402450946157456,0.19553323133091904,0.7013374433882683,
 0.5110003892921672,0.15772039120885362,0.6161752024324204,
 0.24417246963080275,0.3752172523157561,0.48893080986472814,
 0.4549533508109477,0.7802989176001295,0.1642435328883335,0.7075952669190566,
 0.607658525075203,0.9083188107860858,0.5240668364068504,0.39224813489755184,
 0.8159738566135408,0.864998384245072,0.15801456127042268,0.3258011237368591,
 0.09495996020062603,0.09834080309096682,0.5437366300652163,
 0.5006205496667612,0.6728337631080065,0.8737674060143807,0.864021218684235,
 0.17083778721496035,0.18088937615508838,0.6565529282151863,
 0.3336990171039881,0.017339649570613624,0.37389616327749187,
 0.8751862649130739,0.8553845604886842,0.5639288556538402,0.16241318522240045,
 0.6307984774041675,0.946742453419915,0.0037355213949493704,
 0.1580231189792497,0.788182523759988,0.16585756096286597,0.6279257794204861,
 0.11571699316560558,0.819200967772115,0.9880134058819863,0.10019227541134557,
 0.5963273801546191,0.8222174143390152,0.2769109879693896,0.7463876375690601,
 0.5249094028319731,0.6297187708223828,0.040682275962498404,
 0.8485492321505802,0.4135734359597453,0.6511530884841289,0.39506277035079906,
 0.3965425542936928,0.09908615824838929,0.3970215762382294,0.3192215401736579,
 0.5473459873444992,0.7338026358021268,0.5011243054546175,0.9666337676003605,
 0.3831095499110344,0.9220493274224191,0.19891549793967056,0.5615704490044376,
 0.7613543725086112,0.4221563626576358,0.5773349037876625,0.10854743337215089,
 0.942927381534022,0.5845788084134742,0.21447771690699768,0.3604481926662009,
 0.7594698949063469,0.43257891777254565,0.09183883533764681,
 0.08775257766061861,0.015419644858046921,0.15618855275553245,
 0.9466699340637573,0.7447180921235566,0.9743823753009898,0.4800643269004363,
 0.47846286172049735,0.5298498374197163,0.9022410265048126,
 0.21467656381938327,0.48185615372969637,0.2748415084119036,
 0.9544413321197199,0.9501723902604188,0.9157383347744112,0.5532059786996573,
 0.7222033465307931,0.0137671282920373,0.9503532737856955,0.7756272388966203,
 0.4252196948422591,0.8661970748207298,0.585739267989145,0.9401268190588844,
 0.12536507378498962,0.3988062251867448,0.9218801953706544,0.4472323632682669,
 0.8557829017261953,0.5280041148646573,0.7444411238655224,0.10865550815785485,
 0.5086707582717169,0.6952294616426562,0.2226870408668612,0.44026914110350346,
 0.8863884602461072,0.3843247879540279,0.7055967851148838,0.9499021185774792,
 0.025895865840622423,0.4355776826294303,0.6209908723459634,
 0.7503080610821341,0.6082528273465926,0.4035592319742308,0.15098808437996025,
 0.025259512264734663,0.001453685337129751,0.019006405200493015,
 0.0898476734492537,0.4570153413146334,0.041712394070515746,
 0.9291021870485143,0.15957873715728566,0.43655168612089057,
 0.6428040129182593,0.14412090742891304,0.49104474814544263,
 0.03991412985060028,0.16765689940315687,0.7247578558984695,
 0.8359225337477276,0.12721612563219498,0.18660522219094666,
 0.11920047252368693,0.6511289173571903,0.04491693270786379,
 0.7798478438161143,0.42559088128412703,0.33567210611686504,
 0.10272513857875154,0.5615253225943694,0.1301181638205018,0.9295399905676592,
 0.6738542607596028,0.5783110021496469,0.5836207818976986,0.2911686276476372,
 0.4959285267811774,0.2726518238940199,0.8837227789022469,0.5994897861484587,
 0.6246902294740401,0.8482067992762442,0.20369648092486103,0.749574044421897,
 0.5004350814649773,0.4425102816260612,0.47852993227296636,0.8761250705928851,
 0.9870763621953358,0.9436254811985093,0.5972906985350553,0.9052626553392048,
 0.854901693284112,0.03583199120189906,0.4793084628563974,
 0.016914368278982295,0.3192687452514189,0.5922239435240195,
 0.7479031411443955,0.913238908519029,0.6778011460242015,0.2842816331658591,
 0.2954437602988458,0.44737100531222695,0.6148171946361065,
 0.22605722563791708,0.0779760332937679,0.5478393330178957,0.9727375059926124,
 0.29885645064956035,0.7511912191753378,0.7536494674970087,
 0.020696025555346997,0.6005334188276146,0.3644396375769663,
 0.28124495373314273,0.8292256456908486,0.22684950553766003,
 0.8668594582206943,0.06988733666374825,0.7208363602052994,0.1993001326166014,
 0.07666284979418825,0.43802779512911627,0.0341381916580078,
 0.004866701798105666,0.5887241978403468,0.9300852439952617,0.854498429043331,
 0.8087066775109374,0.5903131414001307,0.6752024375288229,0.64170515113252,
 0.3534920787057798,0.7142310237862332,0.11110314853877656,0.6734036479235798,
 0.1978273613572713,0.5589714203368189,0.8720781343945435,0.7438301906441013,
 0.4805731605562429,0.45850567428450173,0.7177832852157223,0.9102140592330519,
 0.3626062127344203,0.3648927310493628,0.7065095488012588,0.341936206943388,
 0.20936557421780733,0.3704410799357617,0.5632130335999185,
 0.17374347135385015,0.9349461353090261,0.9050147291975748,
 0.24244805732254548,0.6431057448544227,0.5612961246199856,
 0.056273509821667345,0.1716733548709748,0.7220590531884697,
 0.4703940552772492,0.9106225100893383,0.029479408089822412,
 0.5461210584879426,0.4306073776209014,0.21928538167915623,
 0.33939523313486464,0.492313643582727,0.044538690867982345,
 0.44347255705915123,0.5488577951727116,0.37195988058501994,
 0.8296217797522519,0.17906974223810745,0.20340141274751367,
 0.4191488941997421,0.3600500080918916,0.9371013455385104,0.35803829462312164,
 0.7775081167230982,0.2120958476186957,0.9180943253960273,
 0.004932881764424524,0.21001159675568382,0.44866392827867374,
 0.44320437081157826,0.15773587828215896,0.007289781749902358,
 0.8298794961678324,1.9025339455902568e-4,0.06640190325148376,
 0.5933389850022034,0.6129334327001069,0.7900181890442977,0.4451163891627766,
 0.25399022816185457,0.007757505976397794,0.8853245724624375,
 0.7975258907416747,0.5236683573504486,0.18160356014548762,0.7529740928067128,
 0.6516030264705837,0.8236710522631856,0.7443205422823896,0.8748455550558174,
 0.5280209532715964,0.7672336084576811,0.752459569522768,0.8665158393463394,
 0.5287375197168336,0.21190960880572152,0.25171120083645726,
 0.8865732569878965,0.3387104643927443,0.2744877124667797,0.12039875091238439,
 0.6804270808736035,0.22664935851129941,0.12295541189455661,
 0.6943602766394719,0.8399381749842695,0.6569176055873943,0.5314718659088009,
 0.02763203234885747,0.39477511728917225,0.030543991629039424,
 0.6508831264326607,0.028675007991075585,0.8844601310113029,
 0.4828055128885599,0.7844753773245614,0.18015930952118075,
 0.08010324511194822,0.4196163942370599,0.9815464012290697,
 0.09236220949212426,0.7604556139428302,0.9406357353804131,0.4810002406407178,
 0.5122137077225503,0.7303327044514066,0.6212100400103385,0.4306615686623668,
 0.07191524320844245,0.1187675690202985,0.865249019853984,0.7803335256640309,
 0.1195499421473647,0.2653479066550237,0.6444005091545966,0.7590126633508761,
 0.19601921170364534,0.9361358307790856,0.6199481589608915,0.571020159326968,
 0.2795101640560451,0.9324693796798262,0.31595386030959416,
 0.09666150615849034,0.39445937707943746,0.5830951275690388,
 0.7452936601656681,0.4539125197116576,0.12675066186719464,0.3035047463074606,
 0.9969489104253894,0.8635433098981298,0.7168544978416224,0.6642141288866401,
 0.09892148071586337,0.08499129751543955,0.6316449474296859,
 0.5363218267622065,0.04561342307786609,0.9531141856083879,0.8216049945344805,
 0.8587518072778259,0.003903909703701247,0.45475587374442705,
 0.6743225234694589,0.15220434117061787,0.8650069682734213,0.8457313517571458,
 0.025211226439547985,0.7883269490326912,0.5915654915884916,
 0.2985341937003798,0.9095424757090187,0.973359404697658,0.5830307178572887,
 0.6993268321306233,0.39755972116204663,0.7024714331665528,0.7484637996471467,
 0.7058959301562686,0.3679861731146774,0.175040961283621,0.25218645762512637,
 0.570652498413943,0.12315770878311671,0.030486712278882555,
 0.48876471296256285,0.04619088102058411,0.9493133550943258,
 0.9964635492872083,0.009770148010388335,0.37895888092936225,
 0.014688787308736795,0.6745693743063016,0.7422851275888025,
 0.49221011521460245,0.656888761644375,0.7638023922512999,0.43940291601565074,
 0.4896769590823906,0.5493784986858075,0.5188120034883633,0.541482631869486,
 0.21968893518892174,0.7782853369399854,0.47509317370635895,
 0.11055273656418585,0.3032837853638314,0.6072602383962425,0.6746687221892161,
 0.3361478549500624,0.8324461606001132,0.8413838173602642,0.7335032740295414,
 0.14488693818738418,0.10007856709165419,0.8106357850520358,0.658543944387717,
 0.9620729232419847,0.42622200579983394,0.33624772095058053,
 0.7617134556641975,0.9562662570909632,0.9883949493047156,0.5176688484151358,
 0.8608009049356632,0.23709780165833716,0.6578193560777721,0.8905878266259447,
 0.6853455782528821,0.9285454837199706,0.7259274611720185,0.8871398390874105,
 0.3021230895415947,0.9486579645258657,0.6368665845013876,0.8073526791683181,
 0.23936029586016772,0.7688972313571755,0.5552038286767729,
 0.02900607168445779,0.10129164151667058,0.39175677285722343,
 0.8841534346703019,0.041779483582604415,0.9700308155981445,
 0.3479760917374537,0.7873028983411889,0.3796911292317303,0.2154057908311362,
 0.5094360557192117,0.10486177946363995,0.007664592751879362,
 0.20895899370968318,0.1681539330522629,0.5517637649230916,
 0.23809783778624372,0.29486217086812727,0.8796630060084799,
 0.21343930963021607,0.3286336320917437,0.8557850955934043,
 0.19507271373551494,0.901362847529644,0.7530664212016228,0.1612020688624458,
 0.9115275406811297,0.7789342041601723,0.8324923739492769,0.824249224273053,
 0.34978517362843264,0.8362459277047998,0.32664111288352926,
 0.11218118877964589,0.9639937592277301,0.4154028668361781,0.5187537282459319,
 0.25106606632342254,0.6881183338605347,0.20402143018962682,
 0.02765236308244312,0.1714676569792516,0.025001910013912942,
 0.8051218942360998,0.361670208089536,0.7392702949386616,0.6534142408561224,
 0.3111200096724144,0.5391065463530283,0.5252233047909645,0.1700306870245678,
 0.9605589037238338,0.049362675736643546,0.665286735413566,0.9020095835454149,
 0.0012109052913441554,0.34508518465136395,0.9612059442340272,
 0.020396001763614557,0.42303270875131516,0.7265003330481079,
 0.6326911839837353,0.6671459633361787,0.04004516092796939,0.5391421664650288,
 0.9690721708834582,0.24980107634211823,0.20300062771337468,
 0.5569800547345082,0.9684554269368089,0.5962346480491104,0.9458090341226888,
 0.11399427782051175,0.3064970485728109,0.505819823196425,0.31909601878931115,
 0.9963860171594495,0.028417675254264085,0.7972389354251324,
 0.9512220609348427,0.9498717170519568,0.4875807425829709,0.31038069971291105,
 0.8112122257103409,0.7710351024230273,0.6701263900195532,0.2910436269419654,
 0.1186132616386133,0.7604606625640107,0.6605063623379447,0.26106189912360345,
 0.6508191079262977,0.6274128669066628,0.8611648595286174,0.33616929792922046,
 0.36676707970883193,0.883156758560988,0.789110082295038,0.19011608075362152,
 0.9849518419537805,0.29164262546967024,0.40211358460301416,
 0.5476062504714546,0.6248173481071191,0.8203258053080631,0.5198423589260808,
 0.7031312555900515,0.8040860326994677,0.9670052578059326,0.4489509466277021,
 0.25319133488028167,0.36074693377411404,0.2658849945453913,
 0.2664576202939214,0.3688435358965788,0.6195758134688111,0.594446090991497,
 0.32090126431064847,0.8423674205954579,0.0032261045941885946,
 0.30271579145509164,0.841124937140769,0.050945605501028446,
 0.7424379471669936,0.5657783952651089,0.13472555142532283,0.6833891078788802,
 0.3425232884432139,0.047621143328821725,0.6609813151225288,
 0.17300570964669737,0.6476880953946126,0.7162494639314959,0.5771262164557409,
 0.08378493012617605,0.3854324830408292,0.28907238744528163,
 0.9801001979138375,0.2017734201056346,0.24092461337900473,
 0.07338530853630154,0.26752018887797524,0.6858588494081326,
 0.5472022840631859,0.5564697864046546,0.9868369059876214,0.6303034278804256,
 0.3157117260326683,0.5514424835605104,0.2814920034210624,0.660914053627557,
 0.8931589873909433,0.2553553016438348,0.3197230941745274,0.6179733378954217,
 0.8009014787748362,0.020633349371272924,0.3934524105621896,
 0.5128548484200345,0.73389724501251,0.7249883208464888,0.43003759684170817,
 0.23201781177939007,0.7165337753280573,0.3563081166671843,
 0.33485826477964986,0.8541726679123255,0.04457099285156385,
 0.1616138538772135,0.7377896232361605,0.40288787308742646,
 0.19095247917240366,0.34553187010094877,0.15931739344769624,
 0.410730057821304,0.8350964910818958,0.9367508516101183,0.9470138021794512,
 0.2572076342710792,0.5523544407885508,0.32218335157100697,0.7575179048097694,
 0.22164593695461654,0.34196711759388543,0.7551520744302651,
 0.4487019033835976,0.9244982829137963,0.03389475002521636,0.4996762102471862,
 0.17923710615101585,0.7590444949647754,0.5294006511597192,
 0.12926244701014733,0.4837792777762441,0.5358932820983551,
 0.25072291144475545,0.8638676894023083,0.9695043196895513,0.5690769071957179,
 0.12343255497088257,0.5473306972688795,0.7581987675155664,0.6953276359630173,
 0.676502606473148,0.5054064298993064,0.35909210455460105,0.8518724985601392,
 0.6324681379696794,0.6639832291158984,0.7812491769636076,0.10298552641066316,
 0.6162346947604862,0.9387485710590532,0.43001750790049065,
 0.01207249796146359,0.2389203676887699,0.8917694206462644,
 0.24673457808186872,0.23451080269638303,0.21095027240277442,
 0.5341540463861081,0.6340084027394438,0.42914281672273535,
 0.10036061074923297,0.32741057574167964,0.38881510471131575,
 0.21393609604369312,0.20190844897765114,0.4711457803766361,
 0.1826905558238281,0.15951830932364242,0.27957629677916407,
 0.14703369849820347,0.04701497532557486,0.8960570235903086,
 0.025562124364670535,0.0463783342074473,0.4840577188551183,
 0.29970149105186095,0.3402162426270694,0.1320414909764136,
 0.39906973577078364,0.09186148353632162,0.12673797850546986,
 0.2660561572594009,0.27716642610926134,0.11900148390204024,
 0.5149284322012824,0.9410880884755237,0.849987340691093,0.8552111770556317,
 0.44333888976732205,0.012251644007388318,0.33235939436481265,
 0.10042522225363681,0.9498797655444285,0.8045187611573569,
 0.02213433253092434,0.1961735851369314,0.516246614302269,0.2963538864341374,
 0.6018658631508601,0.8031873416976153,0.7798656732247315,0.5591728633268424,
 0.4514831502709824,0.6665313768252874,0.10387379820195952,
 0.45732917391687633,0.290031620889423,0.1559492750507252,0.2908158449748448,
 0.3674814194002908,0.4892802419125004,0.2091124672095912,0.6403469033221282,
 0.5195061490588407,0.9266821198080015,0.7521027718449773,0.4943389526017332,
 0.6542969640761386,0.6087889616499642,0.9314953618155299,0.9278739668199317,
 0.5566410177171023,0.40857695509766034,0.762755321057566,
 0.024835737322548734,0.8623040759418678,0.2757640603283691,0.328812357362985,
 0.12667795019449868,0.5543589579293099,0.427412633535143,0.3305725339905826,
 0.9923554025764499,0.5548832558971769,0.5703042392801162,0.733464607417842,
 0.8563346458392396,0.2200253889505427,0.6660038755544586,0.6451235447422788,
 0.8601851559222442,0.37300589356756575,0.47603357639511035,
 0.29700005432814514,0.4534892634161716,0.9581431080370443,0.865746907260288,
 0.9347317672655672,0.9487985647966118,0.5273501288104496,0.657128052738248,
 0.9211850377411526,0.3819729023812519,0.18949902563911092,0.1990662137849729,
 0.3582670991766921,0.15776970208149776,0.4511269037439336,
 0.022732799153187933,0.38208579479745264,0.17668353710964146,
 0.5294308753588469,0.73512244675029,0.7868843540408224,0.5686462914437515,
 0.4316457798550535,0.8956403132121245,0.9381641085137485,0.15307979024772167,
 0.20820548330010658,0.5587868240501874,0.9327941453763186,
 0.08407637358688758,0.5526434715193222,0.7665090124068447,0.9740342937959087,
 0.05378175516923722,0.7472045835206893,0.7210348439773998,0.3386057074650476,
 0.7000675517592966,0.8986290770343832,0.11598174046624932,
 0.18354682159619584,0.5953675108738753,0.9966529854128283,0.6697381479495955,
 0.5929857060791702,0.7711675000211562,0.35321990025673733,
 0.20682609033706412,0.7462893968692754,0.5805061807172892,0.3473574586561301,
 0.003328861666445526,0.4665546220888792,0.18095363709801682,
 0.7211779788905868,0.11340716740960588,0.14605132834603318,
 0.09067302775784092,0.9966738366952951,0.15662645664391917,
 0.20709552615054727,0.6044737077609346,0.9738568011615251,0.5776764646843612,
 0.5069297312044284,0.8440948940554599,0.16657378222697217,
 0.043048004903000536,0.13132540321367714,0.17150171608709486,
 0.7331787259345509,0.5741269939560677,0.5462213293079651,0.2020838313089377,
 0.7752258617186991,0.7042911058336507,0.8809035137498552,0.16339954121271072,
 0.4217722611195436,0.5610626891681151,0.07465572927768227,0.3162729733978539,
 0.2600781271202004,0.7985549405767574,0.27295335030156287,0.8810613645774639,
 0.28701821814971995,0.8919917122422092,0.9865227844992457,0.3318804753828717,
 0.6577312008952966,0.23451272778812082,0.5920492028504772,0.8762111327801932,
 0.20234152762700885,0.8474737911559671,0.8973131147199968,0.9437121236938778,
 0.09207384916286654,0.6663095886742286,0.020402274891053396,
 0.21949945384646619,0.7605686929460709,0.8880741649284587,0.4982145675203664,
 0.8470056966269607,0.9657747802095048,0.7721214717615931,0.13900051035039518,
 0.4465880262059385,0.6603071285303937,0.2987953050955451,0.977253684981555,
 0.5855336391473165,0.9867705643552676,0.0030525704480875993,
 0.955330919138782,0.3189161674064017,0.8507304345318031,0.6287094181452696,
 0.1029210721254823,0.3538766152244457,0.2128560144429853,0.4277175757639301,
 0.32331304572839437,0.6759115059483806,0.47071720284777196,
 0.2940201829183219,0.518026807967453,0.7970674797033226,0.38870811562395313,
 0.9791181659094474,0.34568333034096876,0.6761765585324198,0.3006811526550105,
 0.6514838555756455,0.6018607573649221,0.36825522907923514,0.6125467787288303,
 0.7961083579160269,0.29528768083248425,0.9065636328127232,0.741243418427081,
 0.5330203682398502,0.5236426628952953,0.32776815985092145,0.6571076607928487,
 0.16868456818421484,0.5862519876716675,0.6456730200662643,0.5877728381136437,
 2.4611301296728616e-4,0.21701782842533335,0.6498813258518289,
 0.009190264873851661,0.13761583468978278,0.15136140807319287,
 0.6390775719580463,0.677924929868629,0.8770728528412227,0.33079030586940317,
 0.8794588829879645,0.889870793240528,0.07655387181823892,0.05911289089202798,
 0.7361440753808124,0.07321930078427974,0.08329292547978462,
 0.9068412464433785,0.4067261855378268,0.9364683545364194,0.878674063455881,
 0.09517848200744095,0.9842260569461112,0.24847557454699887,
 0.7955909404800374,0.23297980876590074,0.10466719491778664,0.975001160712734,
 0.5329049021137142,0.5384307128686341,0.6810000986054356,0.6721376174375497,
 0.9487033426999318,0.2321956881428775,0.8710413335189002,0.5822856626604286,
 0.2706929923763385,0.5553335746242426,0.8842815494799998,0.3031465293135389,
 0.8734043695184598,0.12297067294279396,0.7323618560377109,
 0.028806335283110385,0.21356369472207226,0.4909619133960078,
 0.9533358007648919,0.20283732443636204,0.6201044886870053,0.6534575616063936,
 0.39825215296620053,0.5194720610736288,0.4286327229496979,0.5448278464060275,
 0.6391184064418957,0.7786620660837115,0.81574136698573,0.3845888289976115,
 0.16275265666452832,0.9821229962181607,0.22175004379358487,
 0.8281281838396672,0.8223363601159533,0.11981552549002783,0.1251794404951092,
 0.2123407757836201,0.8875190704904384,0.07992828284803022,0.7471484852844992,
 0.7190464079481466,0.9486135055380871,0.32212923387347236,0.981978893686699,
 0.8550159570515211,0.44016437270518427,0.27709659517633667,
 0.5869001602062509,0.08462923625576024,0.008402458945012903,
 0.9107246748204723,0.0919350945560139,0.03068866961805372,0.681158299577769,
 0.5246004271644992,0.9256645142010379,0.2608776440595062,0.8330830418164507,
 0.6537196348140454,0.6825525594439209,0.15601607750672442,0.7094300461433782,
 0.36037906588743196,0.9968209277628456,0.35914426404423805,
 0.09542887184397608,0.6944657107039964,0.6214121208775896,
 0.18690371200969258,0.6820890192993356,0.8513705558845817,0.6671377678280658,
 0.44549108493781375,0.8234852733894162,0.5703715608930372,0.6152035697962988,
 0.6123369171134605,0.9025925521583681,0.43855384355040183,
 0.27310576055799285,0.5765192929409005,0.8093655913782482,0.5652334210805147,
 0.12351316147316627,0.8283331005690737,0.7501407394115853,0.8389418260291188,
 0.29576241132555436,0.5192497084446989,0.8902036242416888,0.7793185964681373,
 0.864665478666445,0.9502834656736251,0.7651998570730278,0.41732646549433583,
 0.17625100706644892,0.1895093263637302,0.14545801134108838,
 0.6225341073790811,0.7708220011407598,0.9340317112458019,0.35718299777274765,
 0.7593901259664957,0.41317175652757454,0.9142194001234136,
 0.17165770386351475,0.5373846188405661,0.4819642125296878,
 0.19330761677358543,0.7075075265088924,0.07948667128877196,
 0.13459074089297174,0.481715979417101,0.8767658024869395,0.5228893753136681,
 0.147402724892026,0.8039416011689313,0.5745021745274865,0.955023390537702,
 0.04463672785773021,0.05026641535829668,0.14853260475133356,
 0.8317386003447466,0.33864150812357174,0.971699664812858,0.18827585399283453,
 0.6664929151247562,0.9813213098599023,0.6979510804252825,0.4008844825092428,
 0.19079481481176852,0.033770387496670606,0.9714844689582338,
 0.27515023464374033,0.024933264223271134,0.32183185968537165,
 0.10601035654876179,0.3414361305820628,0.08658954077354963,
 0.9845266209343069,0.7002987990989381,0.04665152323263222,0.7675508521881027,
 0.7945268965839406,0.33497049589705286,0.09887290261641402,
 0.9752712473956059,0.9462224844347055,0.8135008898211868,0.6025184756579284,
 0.06631442860875092,0.5855555141235321,0.6601065975979951,0.7293028813792298,
 0.8379382026902554,0.5571960953427463,0.4018140305924708,0.9471897916363895,
 0.4148172078597542,0.27298079457968927,0.15613613390101644,
 0.05112080964912791,0.7291553615355294,0.280516921186832,0.6964576901553943,
 0.7869753190204988,0.26308827769095877,0.7991494375564362,0.3175715139208324,
 0.04347950694709368,0.7712867395305165,0.7944213878651966,0.8787521798548222,
 0.387003093317313,0.6723164616340489,0.36753661624200107,0.9959284376876428,
 0.4822820193371752,0.6471478377758018,0.1990705114867276,0.46912205744338975,
 0.09755640147125344,0.8148737484970651,0.15926386459090847,
 0.4589604737229127,0.5763271568812567,0.13766988967250515,0.7365862292738385,
 0.3402958029662565,0.6396867205657046,0.7138995247277017,0.7381337114142478,
 0.529283451279187,0.7145856264410925,0.16597408026382443,0.917593158952625,
 0.42984328834238206,0.1997334820371014,0.874499014681174,0.7537143534903954,
 0.6210690961648184,0.938077983249669,0.496673578535757,0.5792641571537361,
 0.5685883167315822,0.605676211875274,0.2773028463086584,0.5450268481643996,
 0.7864784476069037,0.4726846215888516,0.03758319483591577,0.5317361328606325,
 0.6077895745271236,0.9096278022599369,0.018637765789408012,
 0.7975314033569226,0.6595944344345595,0.14525134811371898,0.5048637776785266,
 0.9159229854290611,0.009171445829259994,0.18156980727129213,
 0.4776664020018859,0.11281075134204788,0.266085924860068,0.5521512051177612,
 0.7555546800381211,0.6038341789949075,0.6353394503172637,0.36375283293753613,
 0.13982495796681726,0.9336074559752767,0.528455422407484,0.5772339268782696,
 0.8692013153860622,0.6550626341646553,0.6395479723025572,0.5528634186931077,
 0.3069576389592157,0.34054791284965336,0.011736448813676637,
 0.22130301694955656,0.7064666388375638,0.8853593094767113,
 0.21812094539799498,0.6955291103196319,0.08078993513696808,
 0.9035929154357659,0.9773998527666131,0.2670677832247643,0.9506684615807922,
 0.606239966834092,0.7340945847650144,0.9088449770555782,0.21040449070194467,
 0.46435627108541744,0.7912761253398709,0.8221384357827347,0.6306030170223116,
 0.09041019339931033,0.989930463309706,0.5941933029058798,0.5463756389453005,
 0.3716838667085334,0.9822437652093277,0.1276303850498346,0.3473143652527637,
 0.11553633395830121,0.4379808542110679,0.01046216768555519,
 0.7884987474707934,0.2875904171823117,0.8912771525381457,0.8129506343511675,
 0.3323672950683392,0.7769661442636064,0.7118363772662966,0.7475187903435224,
 0.6123793367153321,0.2578601074191522,0.3879982673495397,0.37942295530892556,
 0.1289602759816173,0.7085188796800862,0.8735859603539496,0.05151236261174241,
 0.31124610891906546,0.4635598691709062,0.5615429226370632,
 0.36066219772747254,0.3085071641188377,0.3922066306440477,
 0.18032818805700834,0.2836816061177805,0.4915301184006471,0.7599493922751352,
 0.41067434775574285,0.13359391615761462,0.09701778993776011,
 0.17585104617363623,0.5358304742863398,0.5985935164742113,0.4666247050816239,
 0.37429176463089375,0.11363456173348518,0.7788266380652362,
 0.9798906774185169,0.5008382299146419,0.4873463625266341,0.1004049815822955,
 0.9027671557075135,0.6405469461558826,0.7563712022764532,0.6583128828621805,
 0.3974340771498236,0.08684061351243777,0.7177347332819649,0.3294646641496042,
 0.7797568485860843,0.6095783790861186,0.1767206699677698,0.3538073820443488,
 0.30731686764955846,0.6703778788081043,0.05595023212057071,
 0.8811927355364624,0.9307496746043862,0.8645143802033691,0.6968831943481791,
 0.9077523471523993,0.4001279265539901,0.18183250700791698,
 0.45116209339785607,0.859487642861011,0.4897897165135363,0.577136824879868,
 0.9162966310781568,0.8970200239703554,0.10015337072997355,0.9383791455220919,
 0.6112047841183141,0.16891286179220832,0.3681666700409698,0.1543484977281444,
 0.2734625339534731,0.46012608204212047,0.31578849375035856,
 0.7875843907598528,0.828691500514964,0.18622799841357784,0.9948580457199739,
 0.3299149506252621,0.8265834317127236,0.8199064327963846,0.455417673921656,
 0.9958383726605526,0.8317286656489262,0.9388721034393548,0.27059330944551196,
 0.1369351500424285,0.8008749780946496,0.10247197448751977,0.541261324099499,
 0.5775497650238955,0.4484089848550159,0.6288882870107035,0.36652638455145026,
 0.7229089633114679,0.2770115871572203,0.8582496398330249,0.3039477545926864,
 0.11710154371791814,0.45455780881369434,0.19860151794418246,
 0.6875339971386062,0.45734158747551656,0.26612470181986814,0.650150823790505,
 0.289371065051765,0.585675947224543,0.03918486724516246,0.39719136004787214,
 0.023905754736690987,0.12584371191304933,0.43332923961889325,
 0.6843646712446017,0.3409115606304074,0.5165630481455837,0.3391997774676111,
 0.6660290576573932,0.4380455419584598,0.5408050076643597,0.12392658805118106,
 0.10513940475529104,0.3214213997368023,0.6433049104742263,0.8484225828434053,
 0.294863081637053,0.876766343846744,0.17816444470012327,0.9738684961929909,
 0.026201463742307474,0.06469227007628842,0.18714387229680862,
 0.7116149078100821,0.2468919592932839,0.8718525686582369,0.3612369848461525,
 0.7556724602722262,0.0061485407761859845,0.3870645423092126,
 0.11456669151426657,0.9105321826799372,0.047871051366280226,
 0.7897755780779477,0.13239475560650027,0.7652702617477167,0.5908333579324963,
 0.40457661581043713,0.82980890647778,0.3095059631118291,0.7182797848154328,
 0.1348002963839301,0.8793648465492991,0.6779147256128712,0.16992486009648677,
 0.023121856443331756,0.3568731059771014,0.13521180153650425,
 0.3975300961458965,0.9296270406978356,0.3221886920644814,0.3988722145056587,
 0.9122381728519569,0.49168693533854546,0.8538533829512729,0.6113584109983788,
 0.49466127406724036,0.07595589751438869,0.4538365835115393,
 0.10164423441711468,0.27021800507821525,0.26973719703767407,
 0.9340206186482635,0.5393351349610435,0.44347595662513994,0.977925946673283,
 0.4065432038110235,0.6655352641255786,0.621695345714417,0.4788003539346718,
 0.9042679131129179,0.38297347826132144,0.0755232602644218,0.7193499956316068,
 0.33944582668049617,0.1867099754720244,0.6812038904940166,
 0.07717941046028243,0.47828478299512234,0.1744117134432701,0.359448881998259,
 0.49736538949793085,0.5094340396831969,0.12318968206562841,
 0.10861733076868674,0.5511808740511036,0.49042710069281625,
 0.2864247324443083,0.5947521241480567,0.296650227661156,0.2777161869640121,
 0.39954859078389715,0.3201045540277566,0.24729552873201954,
 0.4070287835658387,0.3609869327689872,0.5505148377881014,0.3871141092017817,
 0.8140766801732455,0.8760642595657099,0.777549058139404,0.7290872210761606,
 0.39266198620015746,0.635048056740068,0.13355216432504557,
 0.24205920933208636,0.040895206207179347,0.26645319502631026,
 0.9622556059786009,0.5444966652755675,0.21547773279058924,0.8614838191486013,
 0.34432788055695707,0.5659652637557386,0.762244995313234,0.4035550760800146,
 0.6314426829411915,0.11997014416900331,0.13837130936713093,
 0.19221772093229506,0.5536770057810281,0.8773312739983583,0.7265823386005512,
 0.5015606576522004,0.2947300741550052,0.4798393853389694,0.5345118029558086,
 0.30747317564626786,0.13648270195930923,0.08560258321069991,
 0.835142600194823,0.16926317825535397,0.7786118611541748,0.23559854834985494,
 0.656933005932918,0.04772189200423016,0.8865798841178893,0.2012140213393243,
 0.228855028321425,0.9821023735675563,0.4023992573015899,0.09223805983018418,
 0.039278292044677476,0.8136514626085528,0.9943860546377852,
 0.2838619405672679,0.12506719322686632,0.8069539872008493,0.7926284339485813,
 0.930988542454402,0.8695590433989606,0.29219342340620025,0.5597889317629824,
 0.17261542959592935,0.57491078367658,0.23332438331620797,0.2210062945301562,
 0.2110670359705097,0.7486272205454846,0.343983320141663,0.7032233395819028,
 0.7581905103440557,0.659962908066225,0.5738699003438861,0.1657397487342912,
 0.45598373493773936,0.12494897742832356,0.8034273237582529,
 0.9130311892276213,0.07459895039954845,0.41015741466551336,
 0.3495580999865745,0.48336789906856914,0.09429280829722542,
 0.2831213114962603,0.3740520165012915,0.7086033482818364,
 0.0044486901130738055,0.7491891180175348,0.6441498402457465,
 0.4100970125558264,0.8395718224296554,0.5703640060288618,0.2865644669380022,
 0.09706274234594436,0.613187313480747,0.4142119610091033,0.7395298382985935,
 0.5193187774023914,0.8480822997825483,0.75853259781446,0.9709093685846782,
 0.408128421304777,0.39242138567806306,0.4880631324005956,0.459506101995393,
 0.7113813548776466,0.33961745527435017,0.23867542230638583,
 0.6417449615812231,0.12684058055760838,0.9282346296442839,
 0.18975919623263626,0.9363948940823735,0.7242021234354158,0.791253257879686,
 0.09365369404775159,0.28202543011790415,0.19657722384772258,
 0.6124068222710146,0.5124476779047493,0.9922800352281238,0.7086864921508588,
 0.5728352177396057,0.5701458232592589,0.4308880491461261,
 0.043487585004297835,0.5148761979436406,0.11860142780769356,
 0.7632586557451284,0.7556637182825285,0.8710494216079743,0.30694007216504315,
 0.6800485319792483,0.48826203838958193,0.4114523826324369,
 0.43713600580616396,0.010053209766771376,0.07876374748232706,
 0.07726673825961283,0.38845569434895777,0.8260668944257393,0.857520505115909,
 0.1264074103626447,0.7061864748731883,0.3288596072649298,0.6537574244252553,
 0.8796630635465529,0.4046417170132788,0.2625838485076787,0.36918457452991815,
 0.7714796716497947,0.3159092494281529,0.6803505279302707,0.1758578928657183,
 0.38442151621306175,0.6565997450570633,0.6980532574074967,0.6711341104981953,
 0.22996646685440425,0.9334492842646765,0.4823068785256962,0.2847141698854111,
 0.8798357413007181,0.4708158450993072,0.3761371334612921,0.07777224447977982,
 0.7920680626018158,0.6669906514631897,0.6821935203983216,0.20257846243085864,
 0.5800080113239079,0.3700481837388714,0.679642468092876,0.6737855863575684,
 0.906129070945455,0.15599429756902916,0.8890916844794726,0.09444015489254709,
 0.09000339669713464,0.32357969672338704,0.5607223872163156,
 0.16640471304055948,0.5964017303823107,0.2996505267923225,0.4108071307892014,
 0.3418627693101516,0.5207556048292735,0.09378414137716862,0.9605328255609957,
 0.20373772558769887,0.461342337080502,0.4903085792813596,0.5225372832011993,
 0.2927759962525961,0.6984494429838278,0.05686898117215433,0.0486959904386971,
 0.4296116994216277,0.5616072712553544,0.07403832651800824,
 0.43773569932548184,0.7973786162839218,0.7567665781454544,0.7681222680398497,
 0.8399641938910569,0.08472047551319484,0.20766371239488324,
 0.7273124845477857,0.047701559632361334,0.3566659747046277,
 0.6739344905052223,0.5231250675191366,0.6465339633495059,0.6487009295412336,
 0.15014819150887804,0.3119136966269873,0.32356529673248735,
 0.36408796388506515,0.11113734110231976,0.3890340844521759,
 0.39470913448987366,0.7419349990623758,0.033181763216940885,
 0.8944625993467081,0.09805824913798078,0.06536131820191493,
 0.06204243976581991,0.3838460786111155,0.756307741808861,0.7856205659600778,
 0.010204122011054029,0.490322546908322,0.6923854404568603,0.8966846496234543,
 0.9402872573196194,0.8075999600231507,0.5996233401249864,0.17998217551400997,
 0.5941163465689436,0.10705206817529933,0.5431500136028623,0.6421696557164728,
 0.7963658348673869,0.37634735337197855,0.5557503587783924,0.711787576999953,
 0.7486956187900539,0.5654245366534808,0.15323578260960213,0.6813296021346549,
 0.5758800507514835,0.9548045699209464,0.9029718224576176,0.9529239076621643,
 0.8225831278648147,0.9095500802148532,0.6189574164303113,0.07153673950784645,
 0.002739456874260804,0.8666220263565099,0.7920043094613691,
 0.17946971564929293,0.9859577122976179,0.5426650440824989,0.9467591291451579,
 0.7327210894065427,0.07962312657061044,0.3767334695824601,0.7378609567664342,
 0.22914822020318482,0.1646565706665496,0.454894307219491,0.17193912081572493,
 0.22278424182287349,0.06437487094728689,0.42289943572328326,
 0.7190707394809175,0.15944629332814475,0.03395557236984881,
 0.11067136473311912,0.581692445089366,0.4273659912582991,0.24465559375343326,
 0.7409067006700153,0.8073975191446806,0.642251163072103,0.08959911364694784,
 0.7157735957358379,0.9749715801805188,0.24897256572086768,0.682558456156613,
 0.16593213559137754,0.9877894895491419,0.39281198660616334,0.957232656575728,
 0.20223953638562342,0.8624233604359381,0.3978106016513515,0.9509524012386397,
 0.6323473110358202,0.33529939211839044,0.2009873017309981,0.2865996363164873,
 0.5101478135146337,0.9048633877045005,0.16637297323516098,0.8996395760443714,
 0.5746918557763254,0.03854631394504171,0.7523789703915849,0.6747944327497597,
 0.8015714261506326,0.7477546384556872,0.18474426755557782,0.7646194979053663,
 0.3545888536937183,0.44958475688100874,0.3054410442988885,0.900367736582323,
 0.4752973400216097,0.5151857899574419,0.4798720994073137,0.37885652546352677,
 0.4290342070950026,0.5031446690578961,0.9670664981520256,0.3720000743261518,
 0.40540702737177514,0.47925691130795434,0.40410953063353927,
 0.1053208776502963,0.011134911085386778,0.25700007473572295,
 0.15252514406292494,0.9954584371922056,0.7589665931070885,0.6855602858461842,
 0.4187421310217644,0.9998847263860858,0.86666395171963,0.24846227777661978,
 0.23580655821273155,0.7975356184756066,0.09653863745515312,
 0.5140446396925907,0.7840267860668405,0.40081179899705144,0.6320988416510878,
 0.1065488924971345,0.279038228271154,0.11407229277307318,0.46623384114059285,
 0.9287538841542793,0.301610481750044,0.6159428493240329,0.13143527725634108,
 0.25824815601505857,0.8796340984328761,0.8209980967313684,0.9761665723333374,
 0.45404364950597986,0.6935507018601914,0.9648235520323316,
 0.10989287146050908,0.16099032648421985,0.3421715637463464,
 0.3258722137694411,0.9175720333371237,0.8903731741184916,0.9381995185709028,
 0.6322501741140142,0.8822670331315371,0.27858812831374014,0.3318807722996413,
 0.9945739094529237,0.42290667507058866,0.6072112653824178,0.8047673606349481,
 0.37483942260857084,0.9335662334549282,0.34345335519512543,
 0.38271484753951657,0.7263414987147834,0.3463232619454656,
 0.44883288953062705,0.11194307770678247,0.9665650644231418,
 0.4901765292443503,0.9742036904957403,0.035415649412288586,
 0.06540937887021248,0.09703222038136583,0.44424451888568994,
 0.5482568317674145,0.3438127560352645,0.6489181673141493,0.9080816614955773,
 0.23554280852076948,0.6809325971757787,0.02715801392280992,
 0.5386641948615679,0.36142294504137007,0.8022860989629496,0.7672874086969139,
 0.4690417438124468,0.1912111780017599,0.7128414686453995,0.7928613045097,
 0.8360184420303018,0.7386079305314704,0.22741368660182337,0.5491926262191823,
 0.6462261449366199,0.6061875509424198,0.14136062521856108,
 0.16233783245388378,0.7953823040585518,0.466026314766823,0.10154030275999926,
 0.3970256758213154,0.24080671086976757,0.9467099321536854,0.3956078946914747,
 0.4985037988655503,0.35913994944965455,0.09082722039835223,
 0.9498516129046604,0.9619681369398879,0.6180077994021481,0.20377940367929814,
 0.6830663840628018,0.5793394476475644,0.7696752256903842,0.05904891906912457,
 0.765033594369374,0.5185873859417658,0.5505817659015338,0.6945932902784422,
 0.04870775980826936,0.5936222922493204,0.3330341489255587,0.8366701940417994,
 0.9046943956694059,0.5157801535515716,0.08464813392253101,
 0.12072441784826915,0.5987220605825789,0.951924520973981,0.5690013834338968,
 0.17698347602866138,0.5352006815065932,0.8424402514725734,0.3939885283609187,
 0.817335035232584,0.9819439699748141,0.6376635163980959,0.5813499554189062,
 0.7254868054066185,0.6005135823629535,0.07850398496592914,0.9449157396874989,
 0.041623134485936575,0.2508593396054726,0.9868087061482296,
 0.40243910945841865,0.6379999524561767,0.8937357441555961,0.6487615866764531,
 0.8755352353053738,0.7922118316008357,0.4879207640926798,0.9958983395522869,
 0.30858530821374996,0.7786545502810098,0.6363925939051821,0.6187330746306837,
 0.5435331725950522,0.37951868402605504,0.37659973574888617,
 0.31079356563382043,0.6320668215376524,0.03718082286666058,
 0.7142612579270996,0.23741044341160367,0.5721724472575788,
 0.15408244424923978,0.2373889548766055,0.8044602313593952,0.6153923083857571,
 0.014209091464141288,0.05769245599897377,0.39122799401042396,
 0.5779240875796718,0.8125557862765076,0.5247986588813951,0.20165103390979322,
 0.47134900424474346,0.5584716194633195,0.3974054068321309,0.9864638907179607,
 0.07223674157411708,0.9576709017365385,0.45763209298951835,
 0.41939750312500534,0.21234473430767498,0.4074933753856541,
 0.9194374298989741,0.013231274021408224,0.9255150605244701,
 0.48000677797518776,0.5990903408324423,0.6775949459834655,0.5122988117847269,
 0.9409272057094892,0.07977861521678997,0.6443280931122689,0.5703319916957552,
 0.1986949216303343,0.30915459671601586,0.503129941480477,0.8925017704223989,
 0.5297479450145386,0.7991337359974082,0.3767802403285726,
 0.009153987000118602,0.6966394573231032,0.07859495973241237,
 0.11713791315900346,0.9113741093585411,0.7795873106864448,0.2411488963197389,
 0.06388006802085178,0.0703174596953905,0.1842804807727645,
 0.005758362053379984,0.09672192209943231,0.5748258982104727,
 0.8551540072833732,0.28074532194626,0.535905923074423,0.3402281150144566,
 0.8193490267178655,0.2902040993779189,0.103614497109351,0.7666553376662285,
 0.1744199243375255,0.5395374298402881,0.35501726855618754,
 0.00856628113685376,0.6207868544829437,0.22049226008446665,
 0.36064669524005855,0.9583822124777397,0.3012138819660576,0.8018751405554905,
 0.7520866223966198,0.10462638966476234,0.5472388930780461,0.5649570850651743,
 0.6837117781214983,0.27933659614426465,0.5513851618721839,
 0.29380404921836945,0.018752613571470844,0.7840817435558735,
 0.1041008240728245,0.9307343296008004,0.3519903363736737,0.01051605764669361,
 0.08420846346691424,0.8773340889375966,0.49414713250625664,
 0.3686270091677003,0.34606225738128815,0.6305100131999943,0.7416872019800443,
 0.5246642318666382,0.058426987033657674,0.6901488031527152,
 0.13856172280397483,0.22968079556773802,0.9038476754154717,
 0.15035762701114774,0.39696542397961265,0.3983692740093926,
 0.1317748610379128,0.35438144980357666,0.3204263388368611,
 0.17137720694942316,0.4562510003040433,0.63625675293027,0.02472544775375063,
 0.21180895238072273,0.8940363832189178,0.16811724871222467,
 0.7903843954951055,0.5832678819592436,0.6414936500585968,0.6588698515538364,
 0.8370414179095441,0.9888171508914045,0.6010890361832245,0.09090847448181827,
 0.3998377684460527,0.5525888580680091,0.6874073574767372,0.5934178571627674,
 0.9416244072469051,0.768851823724728,0.531766384160572,0.6150092168828042,
 0.15297888460247544,0.9438612798273398,0.36300624897782396,
 0.011709066383926658,0.6947888429024102,0.3980487023744219,
 0.7970718290256936,0.5364783259777224,0.2563540052865543,0.4654686707572434,
 0.6803259198747137,0.14231962874673565,0.3935203824621345,
 0.17057071051050898,0.5213286112708263,0.8525920248480493,
 0.34702831814005497,0.03530940271194494,0.49940960844280236,
 0.04898444379033162,0.9994510674406196,0.04129809288267783,0.930525058058688,
 0.6612951492592631,0.36949103294303853,0.2580488771601479,0.6733859572470278,
 0.78743778587757,0.8648945843521886,0.42627008772188935,0.5736098558684322,
 0.4961634402102295,0.35238414573203214,0.05222560038415058,0.367416368921611,
 0.24522980327003296,0.5182378543831413,0.8060878692846978,
 0.02037495781059806,0.939932388418081,0.8085898089473676,0.738074943097314,
 0.6948685872497206,0.11533605914179235,0.7007869125188814,0.7887842370764916,
 0.07176923862956275,0.4605129128203648,0.9642374384665008,
 0.18737949351739736,0.15773914605306705,0.1657715582792706,
 0.5830169599397941,0.0992937149834553,0.9742874144070124,0.497178464635718,
 0.7205981773498763,0.40295473605400167,0.07487240849258503,
 0.9274104916903367,0.40642327274532886,0.6828104995997364,
 0.25794935731153346,0.6063302565455065,0.48805801408337457,
 0.9940845270020724,0.8502885091639232,0.43906105954960006,0.7664786708848355,
 0.5067462563959153,0.30472602593569953,0.28525951659290816,
 0.4545664968701628,0.06872783037728836,0.643406615415058,0.5109919857923081,
 0.3655578381412069,0.09378154427095986,0.8732875671487634,0.5060376561050891,
 0.31040675621644986,0.2679324320749683,0.28686548831896663,
 0.15866701544971085,0.008685099113772798,0.5987041796604323,
 0.14068823747459613,0.15520506160980674,0.3281242990916923,
 0.9234460493725458,0.7902131384209338,0.06516347203668815,0.2972222573594854,
 0.602456899171487,0.6705743168342452,0.25084984185069714,0.6724049343076336,
 0.8575703149168645,0.8982361053592165,0.020086473388412074,
 0.8901242561284292,0.9236867926351928,0.2012188653268734,0.3977751197579643,
 0.4973886688950454,0.4788296697877372,0.390511580937807,0.10933750599417469,
 0.8900245653105502,0.617050617234376,0.8610043937827107,0.23449311146196994,
 0.22349472505629286,0.7057141614218492,0.9426148528486569,
 0.17700254736628906,0.16923888078751081,0.5413078970853831,
 0.40047154498824944,0.4178600251202236,0.058730978986829,0.8774171917380515,
 0.9442353066726601,0.6579160017076645,0.8871569497183065,0.3595979027878352,
 0.9099975017808087,0.6752571154748062,0.6301739546694792,0.7514026203327634,
 0.4249376077915158,0.6803575806665686,0.7641722215405585,0.04129983627726108,
 0.7566785168694032,0.8656451703546011,0.990967532969418,0.004859621433218164,
 0.10273518613185639,0.04285534455933582,0.7872550422770832,
 0.7433823028330652,0.007906340763495168,0.04427233206151271,
 0.3013297963225151,0.19915286980955904,0.6053995478951676,0.7943095427381839,
 0.6694711907267611,0.8442370101656594,0.4606997549819426,0.7546470871117359,
 0.559244547311879,0.8866433916016547,0.5293061405728396,0.7931786864535593,
 0.7442083001937093,0.26357630064633053,0.347334427198432,0.45431103854979094,
 0.5072259009963783,0.7112202996668077,0.879763321417752,0.3431699124015567,
 0.13852862640845987,0.9588002985522387,0.8222028928718904,0.5095453455558914,
 0.8031555801944791,0.18288218055623873,0.9708499286807385,0.8793304452481165,
 0.40203622747918977,0.6896937664550479,0.6516441447603788,0.786570165589156,
 0.14865642116753996,0.35919814032831776,0.9052807413297498,
 0.39510697246628235,0.792988402209575,0.8453714454115818,0.9996857183646541,
 0.059152023931819286,0.5263502193107588,0.30987305307963264,
 0.3704343323944772,0.9595680796984994,0.7571931339856163,0.2720150417765319,
 0.18124004767552104,0.2693263647036509,0.6151089082204053,
 0.16829645780686042,0.926527146040606,0.061232702156331587,
 0.4523501959224546,0.6786649925642015,0.09530697133805399,
 0.16621682495564316,0.336296707638465,0.9068881637056686,0.848595764916119,
 0.012974748609183018,0.09993549712965599,0.14522336712632444,
 0.5251089499189213,0.7970925921346382,0.9469144699776367,0.6524849073469408,
 0.9116910557334948,0.1987673180120636,0.007179445777029447,
 0.6809650490034869,0.4572319297600511,0.6867566648795123,0.38281378716356784,
 0.935701630881502,0.8579355647052265,0.09289106715465567,0.2973923789823022,
 0.01418548671321107,0.407782499589164,0.1473200841747021,0.21945728525160624,
 0.2240767062101262,0.3393465436455855,0.513417755837756,0.9760951324433702,
 0.34436771767396945,0.9588805988980794,0.6017808914384718,0.9867198466847698,
 0.9443269642536907,0.6575947296804316,0.8686662170828359,0.50210935085822,
 0.17686990948413261,0.34698219700277644,0.7143992215252977,
 0.5346280503467773,0.5913446591381819,0.6632842666412723,0.5448207521087864,
 0.8216531502837794,0.6494489402909245,0.9654222456027775,0.626872795272414,
 0.47887404876270623,0.21669211020694534,0.8850006851809593,
 0.03659186562972894,0.007945955532817073,0.6117976328939382,
 0.9533395939711112,0.45931449086888543,0.2826303501844536,0.6262470001723593,
 0.8412571926605003,0.5076290124929869,0.7263055561352851,0.2007586276604767,
 0.028941198581295735,0.7375647592027837,0.5927250179142673,
 0.9797470828208297,0.9302278711426591,0.029127741072637492,
 0.9588909874017786,0.6018223013305374,0.727589235939118,0.698292344768489,
 0.9265005742647185,0.3741385687284209,0.23105964890548192,
 0.39372845185316985,0.6751485297602547,0.44632308947454846,
 0.7808392673240133,0.5514728635336459,0.947095336101635,0.8455517490037823,
 0.8666654867352497,0.9657917101253821,0.45784487221595205,0.9278932472434535,
 0.49521803603679415,0.18781493005493854,0.8547508582036935,
 0.48796685863986633,0.6222855026020788,0.8211597030178519,0.6473491087969797,
 0.5097490880793845,0.679260377880459,0.8148504066858848,0.970290673511324,
 0.1291238522552467,0.36828041305381665,0.022252721811568454,
 0.5647681797185384,0.7477642562136053,0.6883443691394814,0.7335364481693508,
 0.5780092883734077,0.6157408892377373,0.8973567327307675,0.06883373724383823,
 0.07405492392792357,0.23800278773984795,0.3621116710336699,
 0.06645790766087833,0.20535139314503748,0.37150420292230146,
 0.7105934261651121,0.40003692059519547,0.4167860136227184,0.9628568157190872,
 0.13075699493962345,0.8541409945445564,0.4214743867403765,0.9343115145971193,
 0.1640141109271691,0.6372260896051992,0.4821708475136868,0.4647992275340962,
 0.43711572949377264,0.8352291010886976,0.4519903701066823,0.6319819919242,
 0.40191917803662136,0.49938713998802997,0.8680844305557959,
 0.6222070138790227,0.3886232737650638,0.8074051881726413,0.8207753800933159,
 0.3192859223985268,0.029537864797849478,0.33496268795628914,
 0.28158383592024805,0.4640287162426677,0.9541964309220816,
 0.35223638977404914,0.9857411896654298,0.8709379158236322,0.5065354992188605,
 0.8160546044515503,0.707820836602538,0.18304300653463745,0.08214708510109359,
 0.5217420834136348,0.9115500752688594,0.02326605327884468,
 0.32185564396487465,0.26851844579217143,0.9700521201426269,
 0.14368220970347867,0.8317822585497584,0.6489039157384324,
 0.33777862518883517,0.8327950666979229,0.4246828528484423,0.6040660702042223,
 0.13533878696604962,0.627155163800432,0.6832038135559705,0.6620062146149289,
 0.4178158794419342,0.033009167857493193,0.8245686465373485,
 0.1316216178528633,0.6285018559759548,0.3895822797980375,0.31986761425247157,
 0.7180780266479263,0.5939824441406515,0.44476809510314774,0.8234209577324411,
 0.9753491472331755,0.6300922177711489,0.08089513806471216,0.9951686404152243,
 0.08952861818326063,0.27753812891328833,0.7834922336622661,
 0.8473423726964575,0.6408950814929918,0.752495040431412,0.7057208254953303,
 0.9088199429592808,0.7669848420476311,0.8098104291705028,0.6822560497483471,
 0.43329599421744336,0.9724211784065045,0.6664343677122392,0.1687917979474095,
 0.47640978474113793,0.8949023581925067,0.3547559269744196,0.6254292833194812,
 0.6330310939405898,0.9311942331068565,0.3187213384079641,0.9797455683174283,
 0.6287991267633147,0.5180037356748775,0.3430152257091291,0.7295882973314258,
 0.0027650228539131128,0.21056986823369128,0.6317343961009814,
 0.5454606866842553,0.7408111478419165,0.9180407051366283,0.13763288406656116,
 0.9432945503635133,0.8856233186651549,0.9121088945920522,0.49757786476721255,
 0.07631729255789643,0.947430852657007,0.3269164683427518,0.4741617121208589,
 0.8526940211197231,0.16786329575356607,0.7613684063816128,0.8117777508272827,
 0.4868113633393314,0.3714040083105741,0.5230137164546149,0.1944758977626082,
 0.6710640347146096,0.22010784466492683,0.7258392434036631,0.933138206390317,
 0.5914437991536886,0.5618547302891683,0.9904951346940707,0.04977060821330748,
 0.5951408531031468,0.6226839812169446,0.7347167934259529,0.6489154762394792,
 0.7443031611823432,0.5580539611132755,0.7931803760853863,0.7546958126320737,
 0.825493359416688,0.1024818523295169,0.4405156766637661,0.37322731168764434,
 0.08270597464085505,0.14328656962462638,0.20928879108771115,
 0.8480007029036509,0.5759230153323547,0.6569470327186451,0.9685300777335449,
 0.10358622552582153,0.7639350001680046,0.17352779417134334,0.910824727169868,
 0.6973377231490739,0.7633499600825773,0.17868364501440048,0.9307886466316612,
 0.3144451576251327,0.6748970393995313,0.0059207183268302765,
 0.48080604388872916,0.33785272271876343,0.6556416064968664,
 0.7657085058053593,0.20393762763318346,0.30258896247952083,
 0.16738334195213245,0.7884671167314614,0.7926499927090044,0.8972325423766958,
 0.8888800692436604,0.2541860312825527,0.06383697943870081,0.7528650019402323,
 0.8005526341256862,0.8317736486714642,0.007349729512744041,
 0.9930214550589651,0.23413977862742907,0.2904533135040729,0.5961435094879864,
 0.9520128987740868,0.8633367288916329,0.03227854657352791,
 0.21325860143990527,0.5993474392690206,0.2896677344883083,0.8713434860577617,
 0.7235070914233781,0.2359374777008003,0.866844579158316,0.49067721920911345,
 0.9606407619588826,0.7946261875460624,0.6557901886227144,0.43769417007246114,
 0.412748110823281,0.2505067051677129,0.9686321607180449,0.9710507998812887,
 0.60628383517836,0.3673092422968727,0.3403933906208386,0.003470589593309703,
 0.20410559915312443,0.7883579750087575,0.2573171616949208,
 0.036743112256046984,0.7812379315742686,0.1301332252888766,
 0.1470086527999973,0.03848354372455254,0.7122047862343528,
 0.039020255425966655,0.5195246840983949,0.6001987648202814,
 0.5087727313521492,0.8363565204032083,0.6135936560777759,0.12018403524950194,
 0.8085697231210868,0.2747949485229342,0.40079835966602073,0.6588527599123691,
 0.08291898392280994,0.6401684452308717,0.044055712661157376,
 0.5528252358440997,0.6964168518102645,0.5193653098112438,0.5985179814214712,
 0.7884997291261838,0.2249150529815369,0.13874003049484374,0.8751889823312156,
 0.6701947194940161,0.1860999750619724,0.3257271818478742,0.9485944401857056,
 0.13672410959471315,0.5662238274170461,0.427795468532571,0.5989308618380321,
 0.04192786503469126,0.08802540165593986,0.6806150755389021,
 0.9458850334740053,0.8475465516334487,0.6991253927168506,
 0.004799620718717312,0.17753099617392332,0.14442857214479443,
 0.1874309601071189,0.8594703985748162,0.8936770150468537,0.10540614305522,
 0.942105109309733,0.3586106513460534,0.9493817609581077,0.22009280189464198,
 0.24237441590144804,0.5476368967919176,0.42482349810954356,0.335267768358246,
 0.6100842519006688,0.2183862343559363,0.3377735669406996,0.27484294087558325,
 0.4940992281832999,0.8257300175162026,0.8856358289225467,0.3671868746830027,
 0.5777272816898628,0.770127857973648,0.20294897490895647,0.14415600648262328,
 0.3536859651282457,0.9598636695651152,0.46537975584956204,0.6865420686219644,
 0.9991219747092825,0.0813216753010142,0.797041521418918,0.20844800104631744,
 0.924211167397059,0.1669054842066181,0.7010238142820043,0.5863379042198934,
 0.6093827058598549,0.12275070970316404,0.9284471860537136,0.742016438304933,
 0.4039295076686953,0.01601449686941281,0.19820671196119788,
 0.9668804492709532,0.0676465115281003,0.6427293793402159,0.8334303171124153,
 0.5438509320631497,0.13202007585136344,0.5835101675718187,
 0.24505568501257857,0.2546534875765737,0.1745630693472051,0.8091746226546718,
 0.254958682103246,0.16645810202197397,0.6236503764392686,0.638750817281512,
 0.8961211638555037,0.9455878618344583,0.9999309552001433,0.6636362487795477,
 0.7208424620337774,0.9496696850643733,0.6704499209868487,0.7088932857527637,
 0.1322605870926984,0.8895676697307038,0.7860092804543476,0.37279976485315514,
 0.9118558563355978,0.7456035239305403,0.1361167262501055,0.24478071562759007,
 0.20921505188806344,0.7088129405417098,0.9313657606861825,0.5439306654923781,
 0.3009816249935534,0.1779722211501723,0.9521259699635471,0.9658455785487727,
 0.4180425465027352,0.9908658431780554,0.3315270612999254,0.6694455058775225,
 0.6352214030737373,0.0795337201191052,0.39061836398121075,0.3861232270946403,
 0.1533575550417393,0.15041077302582095,0.4818438830270355,0.3846782932922864,
 0.39172249632031075,0.04530651750998227,0.692836113756766,0.6207177176133316,
 0.877455165548116,0.209286296108251,0.3022822976317119,0.837751847094951,
 0.34716568601345144,0.7772581368448662,0.6381702158756044,0.5819578390618727,
 0.23534595896556587,0.38646890904725584,0.36494205834691684,
 0.4809136275360616,0.869938724675034,0.011748505311445223,0.5383174735418725,
 0.684637609989991,0.6795496223852245,0.9784310538505813,0.32861684132467894,
 0.6718120617770478,0.15060321239012642,0.8231277822552066,0.8207707518895108,
 0.9289708961812193,0.8385682005630263,0.7576115374447029,0.7040965899063761,
 0.028169539812191613,0.494918151030324,0.9645053945931813,0.2656377782494269,
 0.8654205542917186,0.37922348713866094,0.004730172546308586,
 0.9575119187623278,0.7308416342116196,0.13296427059480065,0.9253753854045599,
 0.8544948261478335,0.5201884046507569,0.6901509682375544,0.725962000556935,
 0.9559114553582198,0.7799866429120172,0.6692620464412521,0.8734930090832878,
 0.8824232731283296,0.680655112991079,0.28958580428948477,0.6878984730143543,
 0.7576830538668725,0.12816297608627647,0.21023059984674497,
 0.8974501532389512,0.6075665082336549,0.8910370647375948,0.8330956638535021,
 0.41943189183813556,0.9307501557621165,0.8441564443630558,0.5169194836450511,
 0.04809267605275647,0.148256563619933,0.27480680911142985,0.8119862434091664,
 0.5375112343086142,0.9602608670836783,0.6437119993676785,0.06691890789238375,
 0.97102276689572,0.901357462932352,0.6150041827488797,0.858479085225003,
 0.7182774407042987,0.8087013263214184,0.530462609390734,0.18656276343846256,
 0.9424514999422651,0.8647249938100019,0.32641980553934347,
 0.31418799388937235,0.6131941115828918,0.8958315062177747,0.7747973598784563,
 0.4169895311504588,0.9729695650393577,0.08684704228397921,
 0.24203433473370461,0.3789919478420265,0.6251301013008103,0.8136304272603067,
 0.6109372805225151,0.06065106471069537,0.017733080821696,0.07579343042568609,
 0.31034596438047735,0.8688763410273893,0.6926391369923293,
 0.06597437364245773,0.7202108122437695,0.84807706585079,0.3967637409986051,
 0.04747147268392127,0.3934441483823723,0.5868591414529073,
 0.10553800779663591,0.025588829921529,0.639421868978652,0.28147750506199,
 0.2765131930526453,0.6037688001160957,0.5015742146354578,0.5677591559207492,
 0.6461121252331459,0.3201727634328082,0.7161842926973615,0.3869341588993387,
 0.3893759396491443,0.04043756928394604,0.981335508345011,0.910619834458012,
 0.39671469161864625,0.20957498207274594,0.6173812323259267,
 0.6893654802829189,0.7669918239441548,0.4018773106271536,
 0.044493887588957226,0.7281978310107475,0.9201612922813821,
 0.3706229269571185,0.8014268155583799,0.5112242764724505,0.6693757162754217,
 0.39451158347533555,0.723502268049441,0.48906536952840773,0.5021339936706836,
 0.8340662964338091,0.6396275904739767,0.10853001249386263,
 0.48959503223648315,0.009556679255686595,0.30091992423374747,
 0.5582835187429067,0.5420086324911378,0.7682054933431098,
 0.040950828383598536,0.47622114407971905,0.8067383972862716,
 0.42871570763669475,0.2311649858391116,0.8059129344935714,0.3091608219432388,
 0.09154897921131522,0.7757335775265124,0.2573977636805984,
 0.06810083592560467,0.5496590949698044,0.17581368241257644,
 0.9914682172795855,0.9323702803025467,0.8172003282661405,0.36253188920307067,
 0.4241785327474529,0.6606077395238865,0.8350679444823078,0.30322870500426113,
 0.2594819787783438,0.3327510166570965,0.07121149985168751,0.2762690210322307,
 0.09228391431849015,0.6780661552601374,0.20676219039495314,
 0.8654508470905696,0.017903733377296116,0.4590653952997934,0.9335917494949,
 0.5783901001204392,0.6499749323955514,0.9454907086271783,0.363737063420014,
 0.9419601526336099,0.19921736843984017,0.9150429759569064,0.8569479830600855,
 0.7523382594687533,0.6449130099814262,0.545734065411826,0.8335549643494159,
 0.2150580984825654,0.3607754913794059,0.9715439750900543,0.3621902498883389,
 0.2982528349125485,0.4341991811657425,0.05258054743931739,0.126846524116059,
 0.6840525727283475,0.28013027395054646,0.5996778882348566,0.6965077048722338,
 0.2879626907492123,0.7865864807984373,0.8600259620325108,0.7801495025079436,
 0.8797037435433479,0.7299830399553056,0.6381737507472618,0.3595198847428662,
 0.5265611858203407,0.5911127489097575,0.5458620751034189,0.6814406808615946,
 0.39287279534429886,0.7441913196649236,0.8032739079050529,
 0.26772502344151095,0.76035705003439,0.5017406422139806,0.48818438299616174,
 0.6756404406077745,0.5835684637399535,0.3554620752667915,0.47910959453166135,
 0.016133164268471667,0.9861320212203227,0.17606694017820346,
 0.9355622008031087,0.9949848642073961,0.5322725643088547,0.6749222030967998,
 0.8191107669094693,0.5473941384009954,0.4516365184823221,0.02721521094457502,
 0.7249092119321149,0.24245250785048444,0.33063712677754786,
 0.6838329684690414,0.8731365903191616,0.8482653820438224,0.2559246241574158,
 0.06283822960409069,0.7778507437724765,0.5470073873798773,0.1475848395459195,
 0.968398927303465,0.28066254318401307,0.2448048581703144,0.5586287094092381,
 0.9232565859510021,0.2622005750474733,0.658058418727779,0.5137199013752263,
 0.8545407409756829,0.21207251901978574,0.5885051119991785,
 0.19843547738317946,0.7066405266883918,0.32726638120615303,
 0.3872366539624603,0.6519155790213894,0.6882112570939324,0.9476258479571461,
 0.5822394440462795,0.45535270434808783,0.20764195813249342,
 0.0364142875256287,0.7302331764022928,0.2701365909664951,0.3267127283331023,
 0.1330336470785648,0.5234274076754626,0.9226189398622715,0.9148620615892109,
 0.32779706174319645,0.39991730800717695,0.6137457865630331,
 0.0376037912560111,0.19484519765185948,0.730069400112106,0.421141908693591,
 0.06979646756037117,0.40187681421973387,0.10364368301839832,
 0.12479575561339862,0.5408907975594973,0.5106240646874911,
 0.17676334239033809,0.964467999646513,0.9100024976566496,0.218738392611613,
 0.5667674561075112,0.32176431026119934,0.7564321269192464,0.4046346708587778,
 0.7672339820516493,0.012964580871150932,0.4488397754274648,0.359346655400803,
 0.6519353981397101,0.36804827129200457,0.06563304593908292,
 0.5369400989561631,0.17089383289576265,0.46441532768578997,
 0.027368516997140935,0.42705108011937,0.6440936877227266,0.03118443880688815,
 0.39222541067693106,0.0511988385486315,0.020153901457646062,
 0.6465209487618716,0.782517065889046,0.011816723852805633,
 0.33467063187840707,0.025855655452938997,0.38387562594587166,
 0.4842757748936688,0.3965379707432072,0.7271274677103013,0.9998680513608277,
 0.009582985047618786,0.144779224233621,0.3758411921727731,0.434195354510282,
 0.028176320690299406,0.500738616713785,0.5705677330067982,
 0.04184210344694439,0.3406805507310312,0.9767524961836856,0.761540725044402,
 0.5674387454879354,0.977395001656172,0.6670221691073904,0.45107267524253414,
 0.08633604108712412,0.2798215833763048,0.9758789557826608,0.7417337719818554,
 0.5829044403925213,0.0348903191559633,0.8657126375582291,0.17266105293815814,
 0.49555474884580175,0.43175846773975113,0.9043665077711293,
 0.28559716425476245,0.07907054788028622,0.8258278144639256,
 0.6361139450883345,0.2577366838471451,0.5129628746096614,0.23995330455590935,
 0.28799305243144413,0.586067563859333,0.5692959942972233,0.23365190835995753,
 0.4205450605742894,0.9285488206469247,0.13221469873244907,0.6581167620856888,
 0.9776950355506902,0.004685250466214708,0.8845558314078907,
 0.21599992323735462,0.9636043479059025,0.456196171465288,0.4936777424486688,
 0.5394184445145296,0.32420808935615,0.1347165495305982,0.9083951926237561,
 0.43892863866740117,0.527155235902365,0.3766572697274384,0.8477385321270713,
 0.006325687306790506,0.7135949210719615,0.1842578463307153,0.946423652593076,
 0.9049376156461919,7.824885007390314e-6,0.7035634337355628,
 0.9224704451086096,0.4867459100682999,0.08830068329155849,0.8052459181293089,
 0.2565562152275489,0.6066850962454953,0.6695913047465304,0.6951478898091061,
 0.7434775028174925,0.32040463931223273,0.08325143234376609,
 0.6640449771511575,0.09697466198033311,0.6484777892749848,0.8323470691860818,
 0.5133917562034681,0.35357129497506257,0.2216440456777341,
 0.24199480431548792,0.994974230737922,0.6697467203745593,0.07684178614914994,
 0.8888066562646182,0.06976397639678211,0.14534630750800004,
 0.3546550687808472,0.9276885230286895,0.296214254781645,0.9454543818055552,
 0.61212899178779,0.44539114388720913,0.882734765396665,0.41621906440676026,
 0.8737049958478867,0.436700938270171,0.5842402934971771,0.5381746388356048,
 0.2826375556972135,0.633838813599812,0.1344399304628684,0.4433243333878857,
 0.3223155492881322,0.5446342808133512,0.89519900322931,0.39894285446962285,
 0.5027638369449907,0.5574561522960851,0.027052807014942726,
 0.9380548735259245,0.7359639736279355,0.8032266663379732,0.8423833893711296,
 0.4138072029773999,0.6713664870808014,0.20919610504751196,
 0.12845332929823727,0.5426297232052857,0.36720535487237416,
 0.051589273417233295,0.5820893035723049,0.05761841374689847,
 0.7644593953260916,0.7933624795323166,0.37756329700505264,
 0.18859341028808374,0.8179780966246515,0.10253166158398841,
 0.10689274413347583,0.8382823898709704,0.3853420569314364,
 0.07897796051236083,0.3854073848628876,0.7762294813744198,0.7756053079012883,
 0.5619986624148097,0.8474667312311253,0.40245893720757553,
 0.18718793125851874,0.6510273149433584,0.4368367155608549,0.7196489704067773,
 0.4123111014485724,0.8227193315650944,0.40939185965430247,0.4512905227816897,
 0.757022924878642,0.5695608128311863,0.4813412694374717,0.014424274521360858,
 0.21695796475521956,0.6564878140180795,0.8141693669999139,
 0.22262854545579036,0.42171332236055104,0.9158483004169273,
 0.25455281679167174,0.3040493461185997,0.5475487873964452,0.4088186159030741,
 0.7476670320096158,0.2974996431687035,0.41513444778762376,0.2605707444022496,
 0.03393265505595666,0.03639572335765551,0.6247047882026348,
 0.2775439907423428,0.8777445255545195,0.6266515553103327,0.6064371458500734,
 0.154116861408102,0.6408297858618363,0.4189220624194465,0.7611974951622948,
 0.8251229295928508,0.12100221416961354,0.19331214202162617,
 0.5885153562097755,0.8237538804039272,0.2955187664741886,0.9376524526967915,
 0.471385331879132,0.3752995041123599,0.9266664026627687,0.9853232793489539,
 0.9950183880363237,0.36773053573036285,0.16751839247176759,
 0.042413506096710685,0.8590585697477939,0.49416112586671834,
 0.1867194512469036,0.07032646617107707,0.9400843512773347,0.5425710384833398,
 0.815059689945637,0.8598711638052408,0.5372535802336769,0.09439439168533847,
 0.9350342902367976,0.4919867041544158,0.4484243241295215,0.424018265337866,
 0.2837186636279877,0.807646214920054,0.5511080944765656,0.2742719489731936,
 0.6860171050132985,0.17233940581656193,0.8337631449640497,0.7719384402390889,
 0.018014840884764793,0.39103408066624035,0.6880970215089726,
 0.4471298507542294,0.694398804829962,0.5370147624419058,0.9333272113868414,
 0.08174269534133638,0.6978787903948906,0.19753076949089232,
 0.4738273570587598,0.8121710152520291,0.17334392015249767,
 0.07217617206471116,0.20030346654193432,0.18464835843149185,
 0.8479351717950999,0.8278648100114938,0.2459721877398291,0.7025974849563643,
 0.29179577301899196,0.5969192553810301,0.4026744961388885,0.7525766394353726,
 0.4586726086398699,0.6107715042705492,0.7064790596006473,0.8669351574493493,
 0.4128844701601936,0.18313439436310175,0.6089552798165916,0.3037194036251497,
 0.16251193386545792,0.45763140720495776,0.02343275100686759,
 0.5945085020095215,0.717850849981488,0.8434421582180724,0.5359135319174829,
 0.07531581746878946,0.31983243468857125,0.8139993994024004,
 0.21138342047617276,0.9738084861827967,0.3253760676878883,
 0.37473353995149405,0.001199138543675149,0.37009802917183177,
 0.8652698853602373,0.12246322799375176,0.09209354851265295,
 0.044818961793087086,0.6303360070969924,0.021288511824488432,
 0.05063478952109579,0.64323395436884,0.5674816775703567,0.0460354653451619,
 0.15225891057559804,0.5799375299412006,0.04706147823113249,
 0.5320624882113941,0.5735928270290115,0.6630787436270332,0.32822412539959245,
 0.1517825666154623,0.2459143902794595,0.1759308135864206,0.5570111438839953,
 0.1362709911121952,0.5489226144990272,0.762775751772368,0.5690946059061383,
 0.37768797173370294,0.36948011267854586,0.561460759419578,0.4059116736965954,
 0.02239419015620625,0.8289641898212637,0.7097953271272786,
 0.04063486520371795,0.428635140174894,0.9918701814184735,0.8020840657770831,
 0.6656985467209132,0.47284452309087655,0.7153137793854323,0.9457248231769914,
 0.8391594207488827,0.8950397733774033,0.19836570168853374,
 0.44655226694442973,0.2997611431460425,0.481687436207889,0.3739166932544308,
 0.8964797203526979,0.23131004791971443,0.9789353686465501,
 0.045924843262111104,0.23549776311382964,0.3050135499478869,
 0.8214630361464598,0.2514143463249623,0.13687644246689346,0.2514136404508841,
 0.054180634011565365,0.024226333498519548,0.11629300318469382,
 0.30642970602706376,0.18569187759004968,0.21400147547141024,
 0.01148916382776477,0.21003352820683907,0.6716757754627077,
 0.2033995572782893,0.22032467123761346,0.8252451354015004,0.2929046953350072,
 0.5392439983903792,0.2237989029918248,0.07207726428078332,0.7707236681899661,
 0.40798953480933386,0.7158082961344157,0.5257058821603218,0.8930141075103831,
 0.24943525109515996,0.5918691616929769,0.6332449798145229,
 0.42715142905403525,0.4154738220489237,0.38088961738420357,
 0.06455482620424169,0.8965023225358864,0.6114324213588187,0.8899524308896513,
 0.7676285092070884,0.3473887059234749,0.17469540088831093,0.6527306424499435,
 0.8626117438756473,0.7802074291029973,0.519929354270386,0.9859429708046618,
 0.8316710430662732,0.23004312390649984,0.5076813399916247,0.7105018022560494,
 0.9280847096345883,0.23100576165912012,0.5160929959090451,
 3.18446892429769e-4,0.7907244047610299,0.08044326825183878,
 0.6169721951160456,0.4731711011089156,0.6346876877019717,0.9910752647398906,
 0.29663173734050974,0.7763806798451951,0.6325491852669072,0.7553053846052434,
 0.8893214271210226,0.25417643633840636,0.7701545094240165,0.7238131618135305,
 0.060907045974521234,0.3391044955542363,0.38313726825960803,
 0.34239701825326985,0.37375319160429976,0.9907243127435109,
 0.6926536260237597,0.47112513787067434,0.6918328664222752,0.7873869703699234,
 0.6176739738112202,0.449319132418571,0.14235603924407692,0.6687941027223022,
 0.7716298760447294,0.18046822117967343,0.9435783131350007,
 0.22531876900544479,0.5009038121927328,0.6764238266135674,0.9072526995138467,
 0.8269465344975846,0.29727741633625804,0.39296411384232166,
 0.5849127286135445,0.5503610595372547,0.17351585154146154,0.6674894573779572,
 0.8412395448637562,0.3177674235681537,0.13032919583569202,
 0.46533222292794363,0.013349402695801427,0.6403894842646446,
 0.3940233665922176,0.30134730946516086,0.11381291868131005,
 0.04336208637835037,0.13160807902048788,0.4762269151619618,
 0.7749193460114343,0.31112382815921014,0.45238561333693883,
 0.7535983866182158,0.9774640662338774,0.8198461367426185,0.8702800411248895,
 0.1374586724533723,0.015319674997742783,0.463353055994857,0.3704482736390202,
 0.09158146379653676,0.15141752076615644,0.08019664606573396,
 0.34396008273616974,0.24023037733789598,0.17104938826712834,
 0.6131170850054838,0.08643753586772351,0.5867274724910356,0.2787218422476241,
 0.9694288146123992,0.6940048164432362,0.5030481958649307,0.3290167600644869,
 0.2437863314559685,0.7594427471530077,0.8002568088975986,0.27042441444071386,
 0.4531349171979987,0.4366300218392615,0.33699916890809745,0.4072930466856959,
 0.5977622440670771,0.1853844308677377,0.9385347881283175,0.32784932180062876,
 0.23999950945305992,0.8993133975310934,0.7795732030909084,0.0893882183531074,
 0.05957360823630076,0.6762976463144884,0.42869598057600566,
 0.8380127933617617,0.4405436638720579,0.005180604254238719,
 0.09036719421314576,0.16720065488673663,0.2668257798716829,
 0.3476101821802309,0.6310018200675983,0.17412263340767764,
 0.30758657215455965,0.45381375222038034,0.40675197485348236,
 0.8950214468279556,0.04388759745788329,0.3884783686845853,
 0.14502050232420016,0.26478690522643455,0.5070386906146582,
 0.5887842281171296,0.9819697756807819,0.7197371592644255,0.7534572526627463,
 0.07853188727149396,0.3070272921294509,0.4858629467263854,
 0.18555430424440122,0.30785287429787644,0.76828551072777,0.6975530812552321,
 0.4748497135175678,0.6522509150915702,0.7721954767201582,0.2654698647131565,
 0.7671492377188258,0.9087010261737044,0.3540672846191444,0.5059066793380765,
 0.6983287741778593,0.7803203005781971,0.8397936507437184,0.08579870001475942,
 0.7773354711424802,0.6765354922268322,0.7909156928135409,0.4581542576261266,
 0.14851210266908577,0.4465976366755018,0.7089187034486296,
 0.33784449545653406,0.5989633201794775,0.9281559079729076,0.9543514383144361,
 0.8315391435949051,0.42892473844350854,0.6440677299660633,0.4030947043900325,
 0.5851836975136133,0.7738530673955466,0.7833130331918459,0.5220356531820612,
 0.841441849972314,0.6189748241344843,0.8875927312688613,0.5238452998373233,
 0.3144667079104483,0.12441401637433291,0.5429267388509591,0.0827321705424644,
 0.7474015164249658,0.8748835254307892,0.2511687094149999,0.12324481712534774,
 0.8502213847214986,0.9257597368652687,0.008303088572522932,
 0.4378471417566795,0.9405751716567429,0.20362055709159854,0.2373858574677059,
 0.8203325548800958,0.4267200182413585,0.9822410851724933,0.19688544479688486,
 0.545341035759244,0.8716858680244263,0.5256816068341521,0.15547481712104794,
 0.1863175893458534,0.26124733853532156,0.8526618838692941,0.4981272764198996,
 0.11048779026402045,0.6838113280213938,0.6225733185431087,
 0.21310345945390818,0.020443729132429644,0.2814021084925201,
 0.41316690870549067,0.2533094960964194,0.3676360068132445,0.3250870181486324,
 0.3517319920239932,0.2898057257108193,0.15506522401645062,0.6349805318945366,
 0.7796251982610171,0.6528012838019228,0.008199296007233903,
 0.41226677899991315,0.9972982148353893,0.39078384495198526,
 0.6759786271443181,0.5897632126778021,0.7401540635026731,0.8649791939934246,
 0.49406898626370055,0.3260895902768802,0.29762133779651545,
 0.4824076951454783,0.7584607242129037,0.23457567841459204,
 0.33840163062108164,0.06465911692292925,0.4554273316128441,
 0.1813434491971524,0.0044541812111206225,0.07807904517973208,
 0.2763506565014151,0.570737876732275,0.27042159604512306,0.5437888508406125,
 0.06098954976695459,0.786965894847977,0.8834864746335208,0.4574177098084734,
 0.39304317058717764,0.38987695532890543,0.11603314821702904,
 0.7031136037771832,0.5510683319118866,0.9191067948673174,0.929888109631876,
 0.7276942836870117,0.6144030274838719,0.07047506212664412,0.8720053145940523,
 0.004899759591827535,0.5576161724177998,0.2773449284570497,
 0.3027456394014346,0.979987168393874,0.19344175783003326,0.4640066629795687,
 0.0369130027224579,0.1628105416543677,0.9754643766570695,0.429029266287308,
 0.11132250844476366,0.05945282810272179,0.5852772758279543,
 0.5381819898813244,0.5498163139290535,0.9673613836064724,0.7856168427201325,
 0.7767074259879202,0.9609088236494224,0.04976759659366026,0.6816878796520522,
 0.8961287626879935,0.5782320361585033,0.18796982999846046,
 0.06588224512042551,0.8375407651180546,0.1788919306319121,0.0785291133721212,
 0.5634353875289192,0.42539847608730175,0.7064219755401963,0.8909739515756129,
 0.09747264008199064,0.29263753795372416,0.35428337787786357,
 0.7205138077124054,0.8375090118044135,0.5424090542766562,0.7810124782102775,
 0.49888147648659864,0.13539753563688894,0.22853050354497195,
 0.7831079676484871,0.6906723741425671,0.922280813555351,0.8384121579392358,
 0.38873549262715934,0.2110242032579881,0.7260709718135351,
 0.40656219521794446,0.5446924678636047,0.3033527156173065,0.6388195766391473,
 0.670612834574714,0.6378832028059085,0.7392839322014856,0.11880698062313733,
 0.70432643340187,0.5913410029004313,0.7966194959417788,0.37673343769825085,
 0.1716028110447324,0.21650055917332767,0.9504098360066398,
 0.11076811821234722,0.8578836012361909,0.794683222110967,0.0609432804841642,
 0.7487484082653255,0.4919973223514025,0.3725602536563265,0.4939113717906465,
 0.2182673330806827,0.41446755818323056,0.7685873273950562,0.1289795529149329,
 0.7295775599663584,0.9743685113453864,0.5368636302689498,0.657556189091886,
 0.9973089748600378,0.8106079204662707,0.6360144148703759,0.19739243805691076,
 0.3011508276274011,0.6050571532347996,0.45604824182762593,
 0.47942701188218795,0.3511981563097045,0.66183256336122,0.7448246156052867,
 0.005484603644233643,0.053453978953208914,0.9614051915299071,
 0.628150525142926,0.4677071837215414,0.5614445339451185,0.9691501946294214,
 0.39589929722167483,0.6394462015686155,0.6964922904275034,0.5074517343708015,
 0.3064003102535684,0.03345809118772436,0.53024744845379,0.9463698293175871,
 0.9942068933935416,0.31564290158282005,0.027328990959685218,
 0.7553760659296382,0.1317802328671196,0.12759977524713584,0.4494205403727126,
 0.23066719265911884,0.4729181577763385,0.6712439350559152,0.6058167609703633,
 0.17892032331646024,0.41891905838728094,0.6579099178688553,
 0.18607096545186064,0.794846374630564,0.5543550307783354,0.10865867028809595,
 0.27521182030819635,0.47916188682820116,0.34973004640618366,
 0.8569825203297203,0.4968694998225356,0.7394947448400185,0.5756671228431548,
 0.8760816343218661,0.31122507343657224,0.0732921010001687,
 0.17279753082928317,0.4296110185924431,0.7704599031465541,0.9277708467961245,
 0.3450063425470733,0.021008668563617605,0.5893462130937078,0.371544959866691,
 0.2360414253310834,0.6334955839464119,0.18668703165324274,
 0.13756809223448496,0.6727750585824095,0.7409649860759396,0.794826523484331,
 0.3163632929305019,0.3486528431682302,0.33366168134308416,0.6105660332442261,
 0.6034305246640705,0.4616752661518594,0.7621522194654138,0.7936586967572589,
 0.3898876329956926,0.9597341430221668,0.028391255409115135,
 0.9666477512185176,0.3066674027019646,0.3575131405692872,0.7983252257657458,
 0.9132260205948539,0.4046524509712984,0.11553156543741028,
 0.19204186569407866,0.03700124834153895,0.63939713326159,0.2521284131390006,
 0.5701831977913091,0.7877955205131109,0.40966791807632297,0.7042012653103735,
 0.9952336867644056,0.6736154400597923,0.30113574609161586,
 0.007672087579852382,0.2891206167023197,0.4620205852003988,
 0.8488135020297028,0.5564795898614731,0.7573088907458401,0.8740916219886842,
 0.8763717963583968,0.003338046854420962,0.4033201546580222,
 0.2567349650433087,0.2853311126630702,0.09581856838497327,0.7886254593027054,
 0.5534645513492891,0.907289381014345,0.9199576812959027,0.990654105860268,
 0.504347483848516,0.7887184044240777,0.6300467515177595,0.0578584582620747,
 0.5511294411942055,0.7388947600829874,0.6062006515092104,0.7915450010844998,
 0.12427738327337845,0.45125753808068236,0.9896665695129486,
 0.4176385999605836,0.10020764161870666,0.5565252685650904,0.8104617327089313,
 0.7750720865751697,0.9742376589679312,0.47724809715103234,0.5874951773275168,
 0.12507081214660842,0.8648684149651497,0.3009683812767475,0.4150293293321884,
 0.4910234998021963,0.7206843738041392,0.4389447673282463,0.07982306020093344,
 0.20081593000683517,0.31854665446784747,0.9266136085968464,0.32046537484794,
 0.32497230094003904,0.6734652441526824,0.803032586680475,0.1420829289156854,
 0.5194227679048622,0.7609235854322838,0.6839521954744671,0.04354377534907572,
 0.8071608029831988,0.7705107659676336,0.7883634133260194,0.7818162948254254,
 0.284128743676604,0.3892374998439193,0.8120345121928473,0.6101375718549625,
 0.7074423857095097,0.2919761796696049,0.3645770818148387,0.03169871487512266,
 0.5889255747318474,0.8883048996444338,0.2531777704109548,0.8290148957203765,
 0.3972528937565991,0.6617103246000896,0.19112453540466434,0.2331728184953663,
 0.47537251082844134,0.32123477636765174,0.925348063841943,0.8778056048524997,
 0.9961470391533866,0.8151758547476703,0.1312236127000549,0.7150971728698906,
 0.5825232401308029,0.9330498430012222,0.13240077827085328,0.7813354539029902,
 0.8952839977208641,0.3116432341816393,0.1283700644418172,0.1355729447324291,
 0.5666269834146406,0.8653784223412953,0.1936463484112314,0.17301480761675758,
 0.6680914093850248,0.15998542088121048,0.8761857408376128,0.6744771198692923,
 0.4973194951538026,0.5771464926328271,0.8695022644420133,0.6653281147179828,
 0.9169031864453511,0.7841539647743968,0.934719018921111,0.04759969236257522,
 0.6271992901264677,0.6989683330968414,0.7898962008482403,0.7436088563310278,
 0.2870349086008802,0.5706874608125677,0.02348340561234341,
 0.028566632762648703,0.744359894359805,0.18137543416514712,0.629367675767202,
 0.44822424923350457,0.30234763416069366,0.1762501076374292,
 0.8542646536232148,0.4154329168417674,0.47524314188893924,0.8875703885896293,
 0.44030667667626067,0.8194725275533467,0.4919453604874776,0.5460573448075063,
 0.029652585768478068,0.6722697545721171,0.42859033268619096,
 0.9686267498804009,0.27243613022568125,0.7874812877889128,
 0.47643808041900504,0.5587444540632308,0.09943157956443938,
 0.13976416523175805,0.29573013841156015,0.9694523111703487,
 0.17491180126349204,0.7589439624111495,0.5593009276047485,
 0.36417180677957495,0.5154638517942566,0.8444543828640093,0.923801445810219,
 0.8341811109014292,0.08510078357534367,0.8042350609366214,0.3083848683199709,
 0.08605007687915733,0.5468784846716298,0.9376875528761289,
 0.00878945611205506,0.13192744592922667,0.4994694737878582,
 0.08973650774639963,0.058421117781058074,0.8133416680750304,
 0.9429841888990501,0.17196034079907419,0.4926989022120779,0.711101123013925,
 0.8495243224606687,0.4857129532039376,0.857779032153844,0.3665575397733787,
 0.345608363776136,0.5042568519288353,0.6656660144345399,0.3493766310800247,
 0.3177410437281992,0.6426420487653647,0.9471885953931749,0.2984174144889048,
 0.05420849619238144,0.26047941336200475,0.3201681354807837,
 0.046165474644056625,0.2978040296921789,0.6600387719094206,
 0.2801132334411235,0.9410970475986675,0.5754298714420686,0.6551687614820113,
 0.9991590619177575,0.4588215787094433,0.1244409521616876,0.02725689289273414,
 0.42820140178808286,0.10870478711831044,0.717662840902525,
 0.22159149714731807,0.6605778359673143,0.71252570417177,0.269616164191817,
 0.917592436615218,0.13676288406850334,0.10887324177676472,0.830576216753269,
 0.04613205867204351,0.3100767322916238,0.331883762652458,0.5907364122529306,
 0.2942856478817335,0.41916864016805344,0.33858570643046215,0.827744998906629,
 0.5505584192719646,0.6421299604059161,0.610846197127636,0.6337426033267072,
 0.08179332563945751,0.4539863615688957,0.37000968303923876,
 0.15146365569767029,0.5320239557020827,0.25158201851297257,
 0.09343093551184767,0.45956159184719814,0.19761993606198502,
 0.13966596304327117,0.3692472484994598,0.5230803672414647,
 0.005800484400579675,0.7980895399206153,0.7669389874483525,
 0.2735879984431583,0.940919437893381,0.5443506428987571,0.1290571318633218,
 0.11508161125057303,0.16058291766736132,0.6689270288996249,
 0.14226999587084377,0.40012426019718983,0.3881202525814741,
 0.7358797044214264,0.14684703130814303,0.035868060228218646,
 0.12787287909251033,0.660766264316671,0.5921326384990312,0.353753362194845,
 0.6930435117650269,0.8003748719258971,0.3417889079345918,0.7645330767092297,
 0.10735124452569078,0.1938756790298073,0.31504373020548915,0.78552113714575,
 0.9614204268559038,0.3603974273492693,0.8818603303277501,
 0.042236625095987135,0.36505461063299793,0.8208876430711264,
 0.1389840202052286,0.5165025444773617,0.9142040272623935,0.142806281452018,
 0.997481063560888,0.5175655053395508,0.44075548015392096,0.4495132040778709,
 0.811096243879056,0.2754606336345511,0.969250553029916,0.517711030232716,
 0.9034249829421004,0.011659134079475275,0.44812459994732223,
 0.5372451612759841,0.7383979891929644,0.24695550641444952,0.5279318926325738,
 0.17876288477978308,0.9516839445261736,0.3513000171837558,0.8756387165814187,
 0.4995916661196306,0.8140476076835963,0.747645208382376,0.13003459068069145,
 0.3160530739901145,0.6952435921854319,0.3887354346095666,0.24346788999325286,
 0.9541605952236676,0.22017702862681554,0.8644600216981178,0.2494178369578537,
 0.3440683889382272,0.7787094311564654,0.9698672611107048,0.42225531460931287,
 0.6818023187394744,0.7043849703217658,0.811835122235032,0.7319324901515121,
 0.3952580866053387,0.6347864473345061,0.37050161576859386,
 0.024258210418522386,0.26018414339468965,0.5458779952392814,
 0.3557832189828133,0.24029074937160255,0.515587141039492,0.3546795603814892,
 0.5813585663422166,0.8016630807418157,0.2691558334560161,0.09211885538355769,
 0.679094702182268,0.27430813034157875,0.8270239220287778,0.40218111995740324,
 0.48994521760736043,0.7849696294018684,0.7837790697023989,0.9880041376094489,
 0.89488954433642,0.09946791955333711,0.5678402455711251,0.7552008943910348,
 0.33072948032975646,0.8208023835114177,0.10684390591623893,
 0.7611798439739786,0.5089779861940384,0.624895515753347,0.6801624064104344,
 0.6344732584513146,0.07623928832776916,0.9322853380787988,0.435424249026627,
 0.8204421584194762,0.5080927466809083,0.6949240206084645,0.30586062336000175,
 0.6860817746899672,0.18279544794739533,0.49912619060610885,
 0.4145677542952685,0.8663203027990825,0.23231996298130242,0.6998333361267961,
 0.48634878338030485,0.9381318546754881,0.2353312718348165,0.6000649362625277,
 0.1281128371119059,0.07550282528013619,0.19722968828771592,
 0.7997440142113577,0.538718045532786,0.6482304438831893,0.5753763599139202,
 0.7203972179623755,0.4777646026261002,0.8903190961724949,0.8137216869378207,
 0.031684459809555254,0.8022374716491821,0.3830162609063418,
 0.04628529645835655,0.5745440698786626,0.34415010592632145,0.647906476312505,
 0.11057059049110474,0.22479466676697335,0.3308737302062441,
 0.30424115459584355,0.8616995142338107,0.7382544076724817,
 0.33533786861932124,0.6818736199577994,0.9982184865114927,0.5117624406459459,
 0.9093303259106373,0.934789480617692,0.5454376565927936,0.5094139548183177,
 0.4594148884985646,0.21252647459509777,0.3436330530226065,0.6232770468455266,
 0.3113662567526516,0.30985732227847684,0.8712913394010816,
 0.18892595216695207,0.737792076348655,0.7129993145314752,0.2905646572319136,
 0.30599191936375025,0.9142578824099096,0.8713851182459894,
 0.14089699448229276,0.40585814363397177,0.9441508285667068,
 0.4523658095353006,0.1138800900527579,0.7060323592819664,0.48846802622038654,
 0.32039771235949865,0.3654981938548624,0.987958960731575,0.3094584839191009,
 0.03809139845976828,0.05946727622993353,0.8807714970586226,
 0.16155982008125058,0.5314144658627636,0.6304995974922354,
 0.06159796614150248,0.858699482459759,0.5268487881655421,0.8760270302746739,
 0.6145458146156413,0.2930889211678336,0.5356927824773059,0.2212766613639422,
 0.8312677105568748,0.34709365843358453,0.7680810228761814,0.282949248274331,
 0.9740922610572472,0.06782191766059975,0.5049905142527096,0.8670989504594607,
 0.05736856311925942,0.8895452124400943,0.4887969177864686,
 0.34630244650402064,0.24100395130654095,0.471175725106456,
 0.35840804416442185,0.7564274852403882,0.30531409105953455,
 0.9840135624682026,0.3212942874520812,0.5712587159550297,0.5556958394932331,
 0.3726325521342777,0.4948311352479312,0.5065332804595833,0.7149848648793262,
 0.3616952523130883,0.9717291044467135,0.5666706426397785,0.5771909180709351,
 0.5363201960260586,0.5513299723541194,0.8793460025279191,0.7909680400014578,
 0.3566633981747257,0.3881350190969849,0.07172423684408635,0.9031509338430755,
 0.8902405225948598,0.3696047976039739,0.812562728499475,0.29204656672257345,
 0.19176508936694203,0.725408576594175,0.40328579436718204,0.7249764792520511,
 0.021501798514399706,0.6448232127741929,0.54594453071019,0.6610064453217669,
 0.3800802723230352,0.5921310855617902,0.38467778127456576,
 0.21278826155881359,0.8929184952226915,0.3934756516652831,0.7400934191629284,
 0.6241311127069511,0.27707628102838733,0.13327030192670686,
 0.37225859872561795,0.43974863986929313,0.6531192402849715,
 0.5258082565306706,0.7212767143336635,0.3605616168447858,0.14838619845957046,
 0.8766848075995538,0.4783799742635565,0.6746171376857415,
 0.058615613376811426,0.9826846997801857,0.5566331446929195,
 0.22945273358107532,0.11294060729621946,0.30264696091178456,
 0.3969115623155173,0.14085803094065974,0.2580537161005667,0.3132454733063369,
 0.7566861468429521,0.5820017027296716,0.4305819813282392,0.397149149302942,
 0.7141997056002511,0.12010918666696535,0.9684268744014232,
 0.012791602857960549,0.03206504965507351,0.6901076018496342,
 0.7867544096317409,0.2054611288165914,0.10091299138538179,0.9756605780844977,
 0.5761867488322466,0.5482487063036645,0.20166176642500822,
 0.016906300953665188,0.6342867237469214,0.5998224850028128,0.213458932048184,
 0.38975452534839317,0.6015861481475215,0.45260702317677626,
 0.27056939048634776,0.4812170563227759,0.0069923593394563,
 0.00395419526712637,0.8396217045223677,0.12165172092997456,
 0.6667972588212318,0.6439781291908281,0.857584930652251,0.3194890590920897,
 0.3124569550870612,0.12142012007041203,0.6492541200863565,
 0.43333842089344876,0.012345786818200644,0.514065774171588,0.573343691668558,
 0.19871268828265598,0.7984116725128947,0.6648041387886532,0.6061616564331062,
 0.9881062568768301,0.4803541143348625,0.31195290695676636,0.0715613152564556,
 0.45077122047531437,0.7713441607913338,0.6710980850774647,
 0.43464490887767115,0.9896752166657135,0.6646858153202196,
 0.24404742840065174,0.6953034796531424,0.45034502997028014,
 0.6439087896214546,0.4299888458209644,0.7333093081234177,0.28551147740720906,
 0.607066187448319,0.49093493805329325,0.539933925181132,0.22845747993101784,
 0.6812921282961021,0.8471652310923872,0.34875913555403015,0.3747727825060846,
 0.8466471824086694,0.6923233838202857,0.6106202369672956,0.5177214795911537,
 0.0031290160395082145,0.9285606789653288,0.7568027703084429,
 0.8302562961757768,0.041705961124381186,0.057750281685872706,
 0.5888541076815752,0.13869247740955526,0.8842289778535386,0.7340237064093281,
 0.18535611909474448,0.5422378133708532,0.08341939435938939,
 0.7240227316358113,0.16408586580515283,0.665475907092475,0.6410509645184582,
 0.7000153165578359,0.6706833488914584,0.7618895074406324,
 0.021259287144128836,0.3121845898873693,0.6188470090236234,
 0.9343460461773276,0.4089129483363637,0.3219217818642187,0.5998697267041799,
 0.47449850782758174,0.4563077657087016,0.5017038392193045,0.8611427390485715,
 0.8568607665389716,0.6245167590000242,0.1317742402429608,0.9404133240326168,
 0.6376592809482756,0.06725688787000017,0.49332468113922734,
 0.6836295788699203,0.4276677965947636,0.391207635442836,0.10508027754633664,
 0.6458018720937181,0.6514214771775753,0.5936772899152601,0.48963304336193325,
 0.4017472132080232,0.30415388729383863,0.4969128106567051,
 0.41119693708484484,0.045975988389130196,0.2684268474988688,
 0.05405972351228472,0.6066635022712284,0.44745351276874057,
 0.8311942813466272,0.2081853392137104,0.7369939615015079,0.17846414126639776,
 0.35625312878355675,0.4628236493549829,0.9976267319582274,
 0.11052161084273338,0.8673906861171118,0.9154869769950416,0.9546321477656992,
 0.33993145650150214,0.21265315800361018,0.6388746584564432,
 0.5902714132826709,0.9688223690742329,0.587555258869065,0.7558808822273181,
 0.3217474425203145,0.886779841948028,0.671127435744865,0.44982078282529003,
 0.42252315519629624,0.8749871334106293,0.8753874250006222,0.6296771022210622,
 0.23441206233098155,0.8133546897099141,0.5138421858442801,0.5030606207991944,
 0.40888777008043264,0.7352172685585738,0.5261345660388491,
 0.44754314390042926,0.6282309929021678,0.18032998148108603,0.654538987260735,
 0.5371172153982448,0.632143521943637,0.743008332413049,0.43230559012818315,
 0.23346598030124788,0.291496164755471,0.27961428252615383,0.5368281778407687,
 0.028539886563257788,0.3581340276198227,0.9518678099098925,
 0.5410208686319748,0.7340257576762552,0.30031419170127605,
 0.31286495799894065,0.7250461760667511,0.4390239955435322,0.4967334777269756,
 0.6801857174864487,0.43017168009570395,0.47881418443167356,
 0.7624441517619478,0.01852767847914949,0.7866223734811821,0.9353512470633143,
 0.0795603874338453,0.5561014582524239,0.9867944700762016,0.36149075603070435,
 0.4286020916947013,0.2671112438229889,0.9076173115272728,0.7532591941446035,
 0.18831118351723997,0.28054098138502903,0.5236999310943273,
 0.7316646598257096,0.7312436331799191,0.024264383421819025,
 0.8612086644804267,0.009469614575468355,0.77404902701504,0.3548512932694009,
 0.5320489464348253,0.9107918115543655,0.5277275128035493,0.997791812826832,
 0.9287624647794671,0.8257344788334806,0.1446556179467221,0.5337220021855993,
 0.6615676312634255,0.2810381886251344,0.9310237326431148,0.9623373345363334,
 0.7301807586242306,0.2491886312907452,0.4555351896740283,0.5190285638400393,
 0.43408601099225375,0.8605933936105782,0.9405746412585003,
 0.11334499711049473,0.7067027562139443,0.44048481823197205,
 0.6396323054759674,0.11958136371708117,0.687044660387571,0.9295478189008345,
 0.16653894800411262,0.7853622443375022,0.2622972051441248,0.3970289525285964,
 0.5479074907529549,0.2837417120060153,0.717837663200565,0.2889058459995021,
 0.18181679119344063,0.2816487506160327,0.5296389528993267,
 0.15389083582836516,0.6728887510603436,0.6012069868275156,0.8697935945674893,
 0.9173382710675879,0.7995411199465857,0.4440935962077466,0.6469934381217312,
 0.04865132657244897,0.4814367079902432,0.7826890696737162,0.8454925081019826,
 0.40001873029903723,0.8016213325153513,0.996197090264627,0.36034429783480126,
 0.9787758726350237,0.12315222658833802,0.3708876426568619,0.8400542263636965,
 0.4272601915007783,0.6105560160284025,0.1103579276139861,0.49605938514370906,
 0.5129788170790496,0.740925316939932,0.1438649370608286,0.6885482622200108,
 0.2663986768304515,0.30405059865833695,0.37769903445706265,
 0.09042737990220284,0.821829571366499,0.9251409110819416,0.9960877779799783,
 0.42569303203333453,0.800399498834227,0.2887277182930006,0.7558452189477561,
 0.056543254591806424,0.5215066704339336,0.5613336950795924,
 0.8129312997473471,0.057724921669467855,0.17962617452915508,
 0.7317898686204526,0.4520628566484767,0.5237055188053096,0.09220538117338883,
 0.9325989043105132,0.5756273974811146,0.17456142277504327,0.1019344839823475,
 0.38867220495804045,0.9628391732127668,0.3625813474492341,0.7229854516673635,
 0.7400366122513861,0.1798319052092111,0.06748869425673554,
 0.21592094848987808,0.17165546017182642,0.03182858178071113,
 0.31316414819098715,0.055909247182799326,0.6891638307469435,
 0.19727574274840787,0.49763948860008755,0.03723948568597957,
 0.8994972614349691,0.9699117828266811,0.9709021006530967,0.21109310130958636,
 0.9158667433122808,0.8908214811564015,0.8670747692018472,0.9209554000663269,
 0.6016350775396818,0.22387482931266145,0.2540556154215087,0.7600428954809818,
 0.49056295232406233,0.49515520611707475,0.24772796810489606,
 0.4294916127044429,0.555727564033949,0.9515344765211766,0.4318968708166696,
 0.6460979411379034,0.814647529238871,0.21787743963417683,0.38119808748536943,
 0.3072093021267387,0.20050350584200405,0.4155835131929657,0.991411135580796,
 0.0598309466221848,0.6126827338535235,0.8443732034237637,0.34190132060518785,
 0.13151976115694908,0.22832147466546904,0.11603966241281805,
 0.5454381275252183,0.29727661075579626,0.8135464436397326,
 0.44324606369305086,0.7110568775717134,0.6260408929386286,
 0.33048314448203864,0.1292439563018084,0.9712192334982399,0.6547025639238514,
 0.24831241891514944,0.49209130080927654,0.16109366089031296,
 0.8755780188815816,0.5736091781651926,0.11846509372515512,0.9294460681193634,
 0.2814390073457641,0.8244397080237856,0.6815863931962445,0.09647376895501503,
 0.17754055306046224,0.3662599695154418,0.09622763965186854,
 0.9102791236535217,0.4031437904024592,0.5774416627772166,0.670396169062768,
 0.09645537709322494,0.4646033110164547,0.21190829915919418,
 0.5939699038650041,0.48896443986089455,0.12231237885337543,
 0.22158107248588543,0.01837956273891228,0.3694598547720265,
 0.49037909364555954,0.0926317636267785,0.6122336973059626,
 0.06723507199789636,0.6362490648978945,0.022156785580029426,
 0.2792462616003666,0.7117689706030855,0.9367384257763026,0.13793418825226544,
 0.4157389576942213,0.7794438469905556,0.6449112968607879,0.280317622403662,
 0.16400952271417013,0.8574263858254587,0.18919442884289384,
 0.5936973367774077,0.5628579379860573,0.5762553493566062,0.9234482449607889,
 0.44497358084832284,0.010410385448129889,0.23886639047646618,
 0.23229928414507095,0.06663099949933637,0.4645403811410489,
 0.08347974602038954,0.8904736252481336,0.740151289386281,0.8550978710722925,
 0.28687944447412805,0.751489639487731,0.6687375035439379,0.6948366972267306,
 0.9892882231183081,0.9639067010936283,0.7626981826879768,0.7689703889971461,
 0.5114549781233626,0.7527155328389215,0.9601342858327566,
 0.007035313812717603,0.3221099769503645,0.6282777376184892,
 0.3269133837818271,0.35748951705822807,0.9501206429724506,
 0.43194683827314106,0.3424075004138557,0.6252813438308914,0.7715684336765538,
 0.5836005553021062,0.2635496245914005,0.15536385807845443,0.9428105190828762,
 0.45002781847491646,0.9988162243510675,0.49056214391038144,
 0.42179795967350564,0.4679810059462328,0.9575873098568572,
 0.42456437929584623,0.5311763980248062,0.9711302997889073,0.3604615835466386,
 0.4755148141044947,0.2837685529310917,0.36824626314737874,0.620852286019048,
 0.46888399887167265,0.9280059431013361,0.7321283764400742,0.9389447039177187,
 0.2017496619213217,0.9374899663005797,0.7786922632025804,0.5518687506941886,
 0.9273599080655441,0.5192127557078728,0.5421922244347135,0.1249489654741679,
 0.7315180176677267,0.1901653666966806,0.23887718671052605,0.8448812534252654,
 0.7758640685499492,0.4307904066410918,0.4528066428922327,0.9599427718901136,
 0.23547682218317612,0.5859337157979456,0.013044154822056453,
 0.35646135695538805,0.6815383095320112,0.6746295949050023,0.8697007199478474,
 0.7179837639561438,0.7990660966512274,0.6534458709170718,0.14992598777237065,
 0.8996355078579404,0.5009292672821842,0.8329072370625874,0.49014985113078435,
 0.5536283598033849,0.14581776349788544,0.18298808654556087,
 0.48553159329126105,0.4545374810270095,0.5790266887889162,
 0.029068150143208182,0.42247676713475124,0.7153882012729497,
 0.8906792432694924,0.40465547245177647,0.057964460035290766,
 0.7125187480912692,0.16676039590616742,0.45584908449727624,
 0.14238728446940818,0.18653351648048488,0.5311139236370683,
 0.9794220895684623,0.45503454008269706,0.8362544814739866,0.194355698361935,
 0.6575631975520375,0.4282536755906573,0.41499818316758097,0.8258349295801609,
 0.3658974884854558,0.5732245331495003,0.3376046368863276,0.586580841083494,
 0.18842514850300118,0.8179607857587088,0.28457431480770945,
 0.6255265089436151,0.7360716435841763,0.5503077799785462,0.8719220837181934,
 0.7370262134489607,0.8267394782586905,0.12842276988790624,
 0.11322020131700294,0.15670537421112685,0.42033489859268736,
 0.7772558900840391,0.17359402412739044,0.2660811020540025,0.5188130793792421,
 0.07376634937085558,0.7937396940086163,0.14596106214175797,
 0.8677992307058549,0.9670930625548977,0.5248416206602413,0.7495155007136715,
 0.8658524336819378,0.7160668500445333,0.13979844523567397,0.8510437934351481,
 0.9788888030494941,0.04900240481343554,0.3655702845944162,0.560296334229367,
 0.03402598145995028,0.9200380145293507,0.18878376563281019,
 0.10319825694924578,0.645460891977377,0.22935945993386153,
 0.04202594235177459,0.4543250999695979,0.598350358058444,0.806006973643876,
 0.7444031387977106,0.39869058296705273,0.6356075653448016,
 0.15669468251099483,0.6389381491200136,0.35173671774555226,
 0.4619248307574514,0.6353396269228817,0.22285371186282865,
 0.17140845058233434,0.9482273537676476,0.4547665011264006,
 0.08454077407754901,0.6466201509718259,0.902264070325603,0.23286019952088655,
 0.16356502040230547,0.8263115453119435,0.0077204485916094745,
 0.7355414352204008,0.31677894561117725,0.2885795752547876,0.7827125511276818,
 0.6150720397309386,0.263183710220085,0.13968525139813603,0.5746052555723249,
 0.29151406242793054,0.5316727626221249,0.009772090106532683,
 0.0308285962000191,0.10392348234536919,0.9627539440384187,0.7082248349252729,
 0.7253266644758919,0.09163003028972971,0.6739887863833934,0.3960152457426287,
 0.15968985985228723,0.48648734982651365,0.9429621417711235,
 0.2243583691874783,0.9271315238851879,0.506146330247073,0.7252920403965615,
 0.4130521999605543,0.05685182476438544,0.27055314900960015,
 0.6552593204588905,0.9560285097383433,0.8176457891256346,0.8080579849115525,
 0.4971461760506807,0.2935959517295337,0.3294828723328711,0.036343180543984,
 0.46818306615198724,0.9747784637658075,0.5954978887200615,
 0.06943712833639071,0.5252576006803168,0.29452890792892183,
 0.9134703373186116,0.4269399973767687,0.21102895798458532,0.4282580739093521,
 0.023001606925976747,0.3778465980785838,0.4767999204311201,0.689085430460922,
 0.27033644497568854,0.9602544744341117,0.2493119281433449,0.2598728491121266,
 0.20856417198414667,0.4842620288250328,0.8913542846140016,0.7592282433716439,
 0.001614567370525677,0.7593830678744762,0.1598132684644583,
 0.6698569559143815,0.40877132140819306,0.031642325863182275,
 0.8429871898170245,0.001067830158132388,0.3247461289078546,0.228408333604041,
 0.773548738208229,0.26132595441158435,0.49163076887430235,
 0.14235199178008995,0.9327757071874774,0.4852358817340806,
 0.42370456711322135,0.7156735062237018,0.42981068766068076,
 0.9159390190163114,0.1711529840467454,0.577413969749264,0.035562511848503675,
 0.06940209285279764,0.3326042838653699,0.10664870946495919,
 0.6494104099188638,0.40858914979140837,0.4421733369779347,
 0.09724610854655635,0.42465457808790474,0.30775068813885964,
 0.6645801241773953,0.8493149151284456,0.16125540310618502,
 0.08334233665063517,0.3733799612764428,0.2295115201313903,
 0.28645372386869306,0.2890179042163732,0.9001267887505586,0.4755538975207123,
 0.9669559869548834,0.3200456950648668,0.5965749464355585,0.07892020936540334,
 0.5596473636543786,0.8182671888871785,0.6104249836929214,0.9751438712120961,
 0.8492005869442655,0.6676285718065682,0.172986724801244,0.2010501405918207,
 0.608961215910967,0.11048526819425741,0.12074339234709686,0.8435484878410319,
 0.20700848023949447,0.472103908172701,0.261550899029655,0.5004403558875729,
 0.5432617027311166,0.9041936990116866,0.8506305342046095,0.7549351172217981,
 0.7218739551545699,0.9310867193117457,0.8550311595086613,0.4130995465939762,
 0.7200266595709475,0.3839588420054091,0.6509360466023587,0.63348873348413,
 0.7709245381049298,0.14216407084206695,0.9413312204503478,0.2120503932007619,
 0.40213770201337984,0.26807988088039636,0.27746057137471003,
 0.7700164608566107,0.49480746565477773,0.8388281382594841,0.8874118550306466,
 0.5102956794383703,0.01652905585657527,0.9870548424574794,0.5801435187945263,
 0.5993888520619367,0.43130138716668776,0.4842699156939847,0.6939500459815723,
 0.7724787232236942,0.5535919996763281,0.23067163912254784,
 0.24187849291431784,0.49883959756418716,0.09482657666267968,
 0.04122036282023678,0.9663961820135194,0.23349086073586034,
 0.2556779808335671,0.8118749417091652,0.28263630162383835,0.9885623224413366,
 0.6011178429838167,0.7347403911910333,0.9470059942572066,0.9993954738836445,
 0.871645978107443,0.17588021224549166,0.7589520622012893,0.5160597820894576,
 0.20238306435691678,0.1479843452936439,0.5399677451604553,0.4678531220917066,
 0.46381513319757295,0.2853057011773006,0.3983236548014425,0.9183570320850434,
 0.49767229027311266,0.9789197305726082,0.6727375293598078,0.7623948507375016,
 0.35864813612736646,0.7306577274651294,0.9952664588879011,0.7873150783058411,
 0.8313976710848185,0.8715446347997265,0.7765078575093908,0.5704458657527837,
 0.7427417159978346,0.722330837085632,0.1368796426063521,0.0203059342953561,
 0.18309303450954362,0.4729637810047955,0.9292853963657183,
 0.11966993129177717,0.7510357119256894,0.11615832951712257,
 0.12695100216367405,0.7304575387357864,0.22897572509882602,
 0.20809956367493887,0.11360218331264282,0.4968024697132699,
 0.49908158706834205,0.2739055648620974,0.2267898056075347,
 0.17174428876608272,0.548695911667422,0.7740045499572984,0.6719483742031622,
 0.2324343547031551,0.5139800317949977,0.14572803527531353,
 0.13677206937624453,0.011097705400535895,0.353705925047447,
 0.3936002877347008,0.7052747774904201,0.3626736893095055,0.5351664549221417,
 0.5669856472413749,0.2986508837406132,0.14761490258980137,0.254175744072908,
 0.6526359962069582,0.4162720212406411,0.6957611529793292,0.10840275868435445,
 0.5887557557317862,0.910931673911879,0.1792349992322062,0.6919649919552393,
 0.06084189060907419,0.19807230159183564,0.8605592324228524,
 0.6765449709545316,0.6936749570932236,0.5029242900119618,0.7452681345573543,
 0.5864311252386649,0.7717710750597805,0.14661739198623758,
 0.34769875140543116,0.3638305412794187,6.846493590700886e-4,
 0.7128791951178796,0.4826576105581477,0.23690989980828825,0.6182722662723702,
 0.6645354033845171,0.5278128733354546,0.18911662040740918,
 0.23323598072893514,0.9927281307273534,0.6464368310861184,0.9304285666914582,
 0.35442135538727404,0.9921127653961921,0.6937457106649525,0.7237069035476935,
 0.5046089558804134,0.31145794562306994,0.4060730613610436,0.6309631874147128,
 0.0771230936195284,0.5952053780743964,0.611324447022544,0.331182872296907,
 0.4657122333078826,0.3286855086382674,0.9979056681246052,0.3895944521970104,
 0.20375296607343107,0.5160530544376397,0.5666723730188741,
 0.21113521361260723,0.17364201536023316,0.26648467046092605,
 0.21352493317073296,0.28552914638742055,0.628650065662109,
 0.012330209048618479,0.9904236183545738,0.7932536974600263,
 0.7114356300440375,0.7872204260979572,0.8044419235919634,0.8089396046804151,
 0.034504793604349215,0.10761062205719618,0.881425740423241,
 0.05915335511893982,0.3337143014967128,0.9124465365875386,0.3753754342430887,
 0.7118280936310064,0.30143291486872714,0.08456116183319129,
 0.9910372134762692,0.8682796335013185,0.2005497729373069,0.09816493255046765,
 0.5242288153630714,0.019529740091334902,0.3060598597522898,
 0.32713538689143606,0.17386949993575995,0.6661643549560453,
 0.48630429412831755,0.7012777153447077,0.663855343106309,0.12508170659205353,
 0.939372222746973,0.39602494792046317,0.1150578641769806,0.4584792866113875,
 0.4213468837959007,0.06793753974132011,0.658667869799588,0.36998667821811504,
 0.9092701377498875,0.13645405744277705,0.3924297279288068,
 0.005042609989545532,0.3034537681208568,0.08986934753412101,
 0.40649961187009365,0.5568917824504911,0.40110557736061503,
 0.12857409749804216,0.5370163998950257,0.44495858816962097,
 0.4443550268046147,0.594630125961784,0.7372857892270704,0.4010428473996872,
 0.7250266786497311,0.39894784560615515,0.7295197409869068,0.7539894878110766,
 0.5165515843194535,0.14976048582208545,0.7858696100608683,
 0.20227550591861565,0.682283705977568,0.3463526799427872,0.03191020460200511,
 0.3621712326857296,0.46963900922335966,0.976471375793542,0.8631871936188498,
 0.8681422365287432,0.6450770497156469,0.07485148764147453,
 0.06137144655664306,0.5204213447015691,0.2920467106788971,0.2277947732693163,
 0.09192191700945473,0.826364998768587,0.40423065268824643,0.917446993240138,
 0.48462760064721944,0.7103959454759389,0.04951327698244479,
 0.012975638485477115,0.9750527305842284,0.601642565104431,0.965005740921701,
 0.1783402867261954,0.1559579384176869,0.9355588878519101,0.4907289665238106,
 0.9185516775053394,0.028719755763859545,0.1882202184848668,
 0.8690990423548389,0.509999678323571,0.042754541723429185,0.46184634871536,
 0.7402132753171414,0.10939375114604222,0.5079795337328818,0.6210232442160553,
 0.8318570946290629,0.9089760399247125,0.8093393131860018,0.615210340285522,
 0.9406393545774037,0.6861330330070317,0.24721434511158613,
 0.24398712043462434,0.7763244173415956,0.7610735495915811,0.571886601886057,
 0.8209286361145999,0.2584304436732693,0.8610892494697965,0.9828142820747227,
 0.463992066118482,0.5909921687703588,0.6072645550682871,0.42188394439352794,
 0.39067196876652865,0.6845891007857989,0.699772180857641,0.6601660229668831,
 0.1784847592660448,0.668287568443067,0.4510603373395502,0.7237210861680992,
 0.5821330782428167,0.3128859117569931,0.8953048256258648,0.6991443847462132,
 0.31073691387752644,0.17144221141214844,0.38718101445422515,
 0.9529341113821708,0.09157184998469359,0.19328808397217523,
 0.6460028293113392,0.7115582913626273,0.3494138906761529,0.6794103311231643,
 0.24356235612200272,0.875817460616376,0.8808846209976675,0.6106341225041565,
 0.41742620083436754,0.524810675418263,0.627655689907871,0.3337443276452714,
 0.5145564986813123,0.47642478979007863,0.09994085192573798,
 0.4720324799887081,0.6112235595239455,0.49446611330875645,0.8266549924230002,
 0.841365841026201,0.3148406388498538,0.7006050735925525,0.26762960961769666,
 0.022573418764006092,2.47083477222132e-4,0.8003174063861258,
 0.3152026019408536,0.749337593441036,0.2571393287912308,0.9755546337929203,
 0.6482265220690159,0.06640280500854667,0.34212865791883307,
 0.3066354024846818,0.20514888713754476,0.4441091703954585,0.1742267503933015,
 0.5631339788273401,0.07128477548048562,0.9402525594483485,
 0.12685470696399126,0.8216529857604975,0.8983404937916617,0.4013035401495242,
 0.8542952088670928,0.98847399594084,0.09960000289900017,0.885869486539722,
 0.11614011563688553,0.6114376189396493,0.049918792777041254,
 0.625568970442012,0.06044525209555951,0.9265290023645045,0.826738456495966,
 0.9761483652989766,0.7199529785584596,0.4345305216704942,0.7692530146857652,
 0.12078584051125663,0.17395887615567474,0.8140564906949738,
 0.4623202423196797,0.38826669517457446,0.9119432581779203,
 0.19291399301757595,0.30257243266717704,0.613395116288273,0.6463405349187376,
 0.8559488801952355,0.23793941809586905,0.46607823388336567,
 0.8787161760390975,0.9410973831356056,0.8974015528958339,0.4318160580913367,
 0.5772978408875417,0.33877929231364423,0.5476803876109856,
 0.22295723868814354,0.9744115811270538,0.43036675381123146,
 0.8406610038297386,0.2335884588538466,0.21190386612205436,0.9833257193806743,
 0.6377429160374855,0.8078242693253814,0.6266457262240719,0.16200634995705665,
 0.7724628413698548,0.3046620479811488,0.04248172420641738,
 0.29650814285440874,0.3267215053716044,0.8667876141734198,0.2360057526878666,
 0.30399288820145765,0.27480371469788034,0.23927678424602505,
 0.30741443354354525,0.24173497585034454,0.5370239598930384,
 0.01831141345107634,0.6836951326261422,0.8661752060881192,
 0.17540466888374606,0.5840030759021186,0.8179437615932201,
 0.04460954809828077,0.3275621699083957,0.4155841348286834,0.7883036991552489,
 0.8934830734384517,0.9957864973201682,0.8094385416489155,0.9449482597435647,
 0.5947533429783713,0.03999458585195348,0.47856503204073475,
 0.12098505820519112,0.7110441458753246,0.8712879766847719,0.5480534608251583,
 0.624243491642733,0.5509749570945403,0.17326779758163469,0.9987992733536104,
 0.1950728323399753,0.7142058284381404,0.9719338672612382,0.12799838551064335,
 0.2131791578415645,0.9544502565624895,0.052324292706569686,
 0.3124991851082384,0.7818157272907145,0.8218652229528461,0.28010587497862305,
 0.42941167388674706,0.3348365931840338,0.7644008377439288,0.462175585673429,
 0.4625741647471512,0.7289169494521228,0.6036904204505753,0.9805267430578003,
 0.7870196456850665,0.40570051309500155,0.27883004580189985,0.579100129417392,
 0.19212657060403437,0.653253549845108,0.43915495141708727,
 0.19440535023334715,0.2582414700851964,0.7497581851763967,0.9631425295494574,
 0.8544768851024744,0.3004123000124941,0.2662612300443612,0.6793979969333351,
 0.9514209255354088,0.6417068519740109,0.3732343329292064,0.9715803821212521,
 0.5766522327313215,0.6087808011200667,0.978403315065644,0.03242021892054903,
 0.028806649995164246,0.7400641085782316,0.39801149930285484,
 0.7518280590945821,0.3932881067422984,0.7052563151345426,0.15972363079387053,
 0.81370062145623,0.1972650655916287,0.47167496150276966,0.3339170945805341,
 0.010087639945258742,0.380200002942203,0.8424588114303663,
 0.30475627567308206,0.3534638714256708,0.09037128276613204,
 0.24604861808141365,0.3441916787643644,0.4812685822976426,0.1010353613789522,
 0.6786833788734183,0.005050778097978048,0.028379334515694743,
 0.42306709161231304,0.8050698196701389,0.82767077951193,0.011142065413356272,
 0.13357811021837596,0.781062322019253,0.3282291801660615,0.6874044339207425,
 0.2677230665341036,0.4070413879395727,0.017520240711578983,
 0.05163704485515641,0.5861695456187427,0.8318805695586691,0.5163196379946364,
 0.04174249898862337,0.22011379243869855,0.8229131849170772,
 0.6390481491727366,0.39487042907526015,0.4680391056247045,0.8212212346891139,
 0.37359569145228144,0.9124013569990076,0.9333116012977988,0.740155410127668,
 0.5737307190190505,0.10117778720401638,0.5045109620721858,0.6897085305779924,
 0.08706882329515686,0.5758343979225747,0.21761314921207608,
 0.18135888234671438,0.36467445262992493,0.3833486040239804,
 0.8452448950549104,0.24946981692367542,0.9245009471669711,0.2949678248974088,
 0.1993679423325,0.18753949685898996,0.14148124311169352,0.5423997498620476,
 0.33783125654396684,0.12946836857142163,0.25688427628022337,
 0.7620574615816424,0.6050215666994053,0.3358438678436454,0.22777435646570077,
 0.1719578452695133,0.36667361880820226,0.9586338118010324,0.6832236305919483,
 0.2640169160170164,0.5683406530965331,0.39965464130186845,0.8601145089377491,
 0.7247420578821453,0.7365736644952868,0.3634972352117979,0.33290441448075747,
 0.17168017283599213,0.07636865843254859,0.6065455112335865,
 0.5130729357002446,0.4910274930474806,0.004825379659273765,
 0.11948915942339156,0.6533242412875586,0.562600834934165,0.4280904180928312,
 0.4277853378172134,0.530085741873758,0.8338628857209062,0.756684843593806,
 0.2422655433749753,0.17092913222221773,0.01159154819133601,
 0.1709585471898175,0.08881595429000322,0.08143142981913587,
 0.06446193513175258,0.7518500997482926,0.5710808068754074,0.6226127254429246,
 0.22265100453702957,0.21664455148394812,0.24240528743794587,
 0.8613097539838827,0.8999555202356868,0.2673349844662325,0.8111702622000602,
 0.15898412572147969,0.9492403779035621,0.6446775745250931,0.2858534505967707,
 0.2607611619999306,0.36704325720318387,0.0759962842131624,
 0.06637884110275638,0.48843558628879225,0.22581928733092826,
 0.6598060139286074,0.9193091295735889,0.2535101129833225,0.829042640976951,
 0.5627896703833877,0.7297247349725255,0.8269967973375254,0.8815456683137403,
 0.4521662471135951,0.19319123837604013,0.737673463392826,0.6112308320774629,
 0.7072506139589751,0.5712124158535165,0.2959234913728428,0.5575174343689068,
 0.2826591233277773,0.7092748629055845,0.6330441686535013,0.6705091815999441,
 0.37443365003004003,0.015850805549350033,0.4142696206405072,
 0.8538536867946025,0.9895696261777086,0.7986301126210199,0.8314275157171815,
 0.2216906489027416,0.9374821065078157,0.05648333405914996,0.8295865166817199,
 0.5714326359257071,0.37024723825609895,0.6803812904542667,0.4019185213400779,
 0.05415247765480635,0.9788485928835279,0.39730617002932245,
 0.5615796957132129,0.035812747694102476,0.6808427905550282,
 0.02079725518079434,0.5565672234864065,0.7820530021754436,0.4415144275841961,
 0.9540689455116551,0.5885905017312091,0.07978953848875847,
 0.050768698713380966,0.953657904582951,0.8405242496498877,0.9578426678215848,
 0.8183855361495265,0.11901573806860144,0.3756796159158645,
 0.36586384520732396,0.9122669037153778,0.5735711999974219,0.72031936475093,
 0.5318375415459096,0.9153314601976952,0.8090940059914619,0.2971368941717921,
 0.24217310661987002,0.4023516048829653,0.2234107517627658,0.935311974250588,
 0.41252180292823726,0.08134098707830228,0.14964043422653017,
 0.8598263897226596,0.027955938289568993,0.1081396280072866,
 0.7755754389705749,0.9482616042198868,0.11602736816037162,0.7290231515527419,
 0.7605797318689231,0.05585401526458755,0.4384789185012513,0.7308620374747848,
 0.1320188537457233,0.4642247891472865,0.09029986200641016,0.9394642611636297,
 0.772280213898052,0.603010886146468,0.9819879544000198,0.5108923197505608,
 0.5993042733017102,0.9548214577957419,0.10901210987584886,0.3328437497368786,
 0.3961590098247182,0.14900647722478988,0.559927735113215,0.27648891709593637,
 0.908405795975406,0.11506790819904822,0.1764523133708913,0.09266393559179875,
 0.038213249600467014,0.4574743766600389,0.6598302885228932,
 0.3976577662753492,0.9573448583966525,0.7486109360868167,0.5617417692130408,
 0.3375381045052256,0.07470065966869044,0.3279353289668999,0.8836892800461149,
 0.6338772175292544,0.5470623020698828,0.5964176038550723,0.16558043558654179,
 0.06336384634695957,0.8291693736436159,0.6554378433102247,
 0.44885204637042275,0.20994914612151172,0.3351901720424353,
 0.20394020662100087,0.724166252630425,0.8864815979069468,0.8560890503780468,
 0.9681808083265723,0.6464903557310616,0.647627453279591,0.8347749904642311,
 0.11240268231281203,0.9429836661153874,0.79200246157858,0.6069553504349307,
 0.3402446759386071,0.6336433091122464,0.11873763426558426,
 0.12557404021267649,6.0918523397957e-5,0.13100799016245246,0.888723751619861,
 0.7535622902270676,0.8081116482112641,0.009889161532074975,
 0.6349044777705352,0.527801247766767,0.21640887304825818,0.808231524311139,
 0.8910094896163576,0.21783992487503712,0.3573535714390691,0.7371487493410849,
 0.986813956546897,0.36902747588566975,0.24371373282456155,0.4436277756238669,
 0.5223228969207364,0.4576402562602616,0.22090414409356618,
 0.042919239412496646,0.4732838994339966,0.4619034461962418,
 0.11269671742130705,0.5123160301279492,0.38863971441917744,
 0.3670238623935803,0.004421662139294846,0.9546133412341861,
 0.2665607178316698,0.9771776020746425,0.5630612703214113,0.4680113826458775,
 0.4147206391159417,0.10157195690720788,0.6491551124768637,
 0.12173398531383117,0.6688813128682501,0.03263749132792837,
 0.8960207405012032,0.9015707042830019,0.4397289020368449,0.17931021870936492,
 0.3799193053846994,0.5700634445360623,0.08543512259339603,0.6720907542018575,
 0.4264821365566164,0.006934154449425134,0.09216482317811048,
 0.3626815442516076,0.6026094011423009,0.20273713636817647,0.654026570031686,
 0.08724354114111976,0.7140816291057674,0.896538828313417,0.5956478943722923,
 0.3964565195991554,0.3864220789709867,0.7145206996038334,0.30410822900309153,
 0.8362122367090817,0.19850813966452385,0.6848856507866492,
 0.23780555275513937,0.6121742996268486,0.2804026906869317,
 0.26629686870060654,0.8304744574546656,0.24217010625873714,0.616297545972502,
 0.2968157539490759,0.42961887101151996,0.181804404556547,0.857724629035075,
 0.24518096941912593,0.7507995576517764,0.1158044902003097,0.5579202167791801,
 0.6145138020881605,0.6620076831811472,0.2291966992327905,0.9712519806757459,
 0.5364284389751874,0.19187013224559868,0.16694123641251069,
 0.1286946052240463,0.7536966159043061,0.027263683887229773,
 0.2265932990004551,0.20259671259558054,0.8555278549357255,0.3889973956304411,
 0.1290546112474238,0.014322176811404796,0.8364066232390245,0.880494786826213,
 0.45383041540981295,0.5752421916578059,0.38794358695841713,
 0.33735644484967575,0.8419735320984596,0.06089398734812357,
 0.03472840263543808,0.3995735097439146,0.553494398909824,0.158694278660222,
 0.48693662035628205,0.24508093763859895,0.20198811783924464,
 0.9193863111097524,0.873097705126723,0.8522298741188198,0.32772663713766526,
 0.2031014032359354,0.8228536602780819,0.8754295129199127,0.11061669207511393,
 0.865999706143608,0.5625759574183989,0.0018078640064730367,
 0.6404772584136074,0.7625022545034816,0.8180782216415179,0.1407472498721507,
 0.5625509533082486,0.9386597066924018,0.9434351082241441,0.03971520161188202,
 0.495186487440868,0.5201252920255788,0.32039126111778027,0.382768556426595,
 0.6974715044713686,0.9739806549639354,0.35522088859112655,0.7537259008824105,
 0.9250799278310486,0.23875529721736888,0.24319191288383357,
 0.13718810628980138,0.07888137150314267,0.3370023349798903,
 0.5036365267713687,0.35170410994970114,0.02583100874156341,
 0.9311806156516059,0.9700201418993615,0.024844290653839862,
 0.6354591778652915,0.41762899790411767,0.8061849703532833,
 0.25948435089702315,0.7573970543508616,0.3340303404282188,0.5823165021083682,
 0.753247676738312,0.5484438465249293,0.23321417620360874,0.08087295771799552,
 0.04837395438754921,0.45149070925129,0.7948261501729357,0.8264637660683414,
 0.30580354309520064,0.09018506140286953,0.028896512113767647,
 0.0569296484357521,0.7682004405556022,0.36568056670707827,0.5760755035830205,
 0.056703741235594074,0.9001145059699999,0.3755045479970188,
 0.5090096325862228,0.8972502815281748,0.6857255513032346,0.5542198625172607,
 0.13532805152430694,0.928127040071472,0.8057868495126561,
 0.008394516565339516,0.575876661736574,0.5673109025199842,0.1441818515747696,
 0.048726932450814275,0.8635949792246365,0.49318496571445014,
 0.252341558360492,0.6623651364341678,0.8514449804187773,0.35980331898069984,
 0.6203340937470581,0.5965802359790066,0.39688963319502435,0.339566819713253,
 0.8467795148313789,0.011897300369641206,0.8818661303123322,
 0.6452297474818719,0.1624782272902885,0.5181100754420724,
 0.002943782575117737,0.5165249622563919,0.3172851375817448,
 0.5495277526354752,0.6206315263615686,0.6956685790313432,0.08105238465312481,
 0.8290338211075085,0.4110782497780061,0.8110063429837954,
 2.349292657264357e-4,0.14744410394718432,0.2855390708347635,
 0.08607121339604185,0.16566425381657401,0.067695893097246,0.3333505908592237,
 0.04224041698564107,0.9792275097432185,0.44077455971537904,
 0.12537021701568118,0.33547588698585207,0.37237934897927916,
 0.2699308339680401,0.07514670493389952,0.8988965783725581,0.4699335527153766,
 0.13527595125370573,0.5205289023393128,0.6621745555578606,
 0.21017535156495426,0.7712619021885203,0.059966847718830985,
 0.4602238126593221,0.714774332284691,0.6212432932925454,0.5215364994090106,
 0.5519737331525907,0.12464353448428422,0.5992636444201263,0.4993696708062003,
 0.0772779833571896,0.25926707391219295,0.4696139842352576,0.4772486013333074,
 0.9284084362269184,0.9366307541658513,0.6078800854500067,0.7162967963452588,
 0.6578883914606956,0.037122362900365524,0.3815984130394984,
 0.2685390165754864,0.5110138303831909,0.5764147137653003,0.558797639712165,
 0.6014608553163421,0.14781925606426094,0.7831649015750913,
 0.16961043803522446,0.3601464151274807,0.3941440644351373,0.4204088328995562,
 0.3080607752303258,0.5929035928468894,0.32626954648036877,0.1891736560942351,
 0.7378145150857905,0.21327367786630946,0.4398581475812453,
 0.09722318570647825,0.5110155207799456,0.22155007028855833,
 0.6178554783658525,0.7672043322586722,0.481233045733801,0.44925979474215383,
 0.5053856253104532,0.5799159269110017,0.9171351559384234,0.21653375864890112,
 0.2407913477531064,0.7763121135882045,0.7799778179666509,0.1846738523521404,
 0.8140357027933757,0.5443397893575812,0.3309698732736901,0.08933967753781635,
 0.11626511646039828,0.19688358159604347,0.516594868909682,0.5022210520766299,
 0.20984766152551182,0.10867641489134494,0.30365202049689866,
 0.056874553744012735,0.7049647464231124,0.29525307777545706,
 0.5803887498860105,0.8733952557101333,0.5444050590259045,0.11780728739782786,
 0.6284421823664295,0.12482890900709132,0.412545937410427,0.21548588326740437,
 0.37701696899488124,0.2838682122204902,0.22438551818900787,
 0.5588952243690449,0.5034160192174048,0.7170215360717732,0.31453508241319916,
 0.58335603473292,0.1540810064502971,0.08099835057467952,0.7980264684845416,
 0.9308978983167917,0.8145589359559126,0.5573400146669676,0.876509001931094,
 0.9716892801417224,0.5748090081819645,0.7864500742891786,0.979429147380063,
 0.994786798363607,0.8502901708732999,0.3162184111648767,0.3509478549948237,
 0.20207400872666403,0.383766303715418,0.034725130836004925,
 0.0217688023948458,0.009441024017826094,0.32459011072202315,
 0.6770091478758669,0.6239973885794342,0.5380675762313063,0.586353465746815,
 0.8998722248871043,0.2100189931612041,0.09188557705087852,0.9993730298250525,
 0.8080406128681396,0.45421664886671465,0.02518966622048846,
 0.08602900072320097,0.7111646290119555,0.9530492413522168,0.7101964551036484,
 0.2409948427258245,0.45366125746726504,0.8536091103090796,0.7272557296266905,
 0.33234414096920695,0.08201971832514443,0.2243785880263679,
 0.9623136279366016,0.002142480321852158,0.5336762925896785,
 0.43885329945417284,0.04735728115013027,0.47411604295351006,
 0.046186659094731186,0.7661547472756974,0.0536938527787818,
 0.5716336865556861,0.4705185133878974,0.9456909188318088,0.4076930981222089,
 0.461640258899622,0.8369373003265711,0.5585534376448709,0.7878927482178204,
 0.2364910145880572,0.008318368244202734,0.9371246751121662,
 0.8021196285852292,0.3861661953047738,0.4777762921599056,
 0.028020634752922824,0.7048713177713369,0.9327386149950677,
 0.3706055992901357,0.7622670186623247,0.28422992822504567,0.8320340235491175,
 0.4356751377858177,0.9057728405122807,0.05531219228881401,0.9773307433724329,
 0.3598605537800381,0.8621553641324063,0.5802151523945716,0.6611240218590417,
 0.8169369410346959,0.8158005799597935,0.8792008970635231,0.5082929149814149,
 0.12228777596217832,0.12483308540574911,0.5169935649739961,
 0.7869473713138039,0.753132935756166,0.6571256456479372,0.8461114796434349,
 0.6439453170249738,0.9237867248673298,0.6864562499442812,0.9362335426978273,
 0.38961798927149216,0.6640520838708233,0.45731944043123507,
 0.8135535782682431,0.933855453392035,0.09442821206298668,0.21264183931420888,
 0.10015062961926402,0.6775742782810982,0.3898770695313143,0.6720487351176114,
 0.28741615945262167,0.007877974148911115,0.7981731183396973,
 0.7375505950468892,0.15371951797475392,0.7581124486883982,
 0.43579178787547834,0.900587291862362,0.9536925247011249,0.10753878290025576,
 0.3467447333432494,0.4008193502038131,0.3467591435131233,0.5206641035784937,
 0.8658087255351599,0.08202983185987289,0.3690055122930159,0.7145250560843702,
 0.47718629253762357,0.8271206339296893,0.4663997321122999,0.3030341492316584,
 0.5779358260347042,0.6974784437573401,0.7024715555111998,0.9683966271487042,
 0.7726716323278644,0.6533693315724557,0.8871318471826072,0.29533824719040014,
 0.39888550259483013,0.20938759910729898,0.12729562367330471,
 0.8355429840759528,0.3369864221967467,0.6668884796225558,0.2708036820357802,
 0.07907646118228973,0.8085076259109624,0.1720647584815722,0.8827715126277299,
 0.4846803238919488,0.7228394569582489,0.08508629498916798,
 0.33305593568799197,0.060937894736387554,0.06545276485671137,
 0.28835312050426076,0.49941109120902893,0.777611279514477,0.7656847254310476,
 0.22563040474792206,0.9567755432315448,0.41161258770319387,
 0.22770392260368144,0.26405902442081797,0.49577821658938903,
 0.7967136287323415,0.4240399285414602,0.871117665993567,0.53195694395407,
 0.6365604178402454,0.7391994588740004,0.5147842145495751,0.7357771254136226,
 0.8731396688109275,0.22695951042669527,0.2373244164912678,0.6396470579999527,
 0.9439911167592205,0.7562862184509747,0.9136261388984137,0.47008295812210754,
 0.5903089277098832,0.6801604699035726,0.7837390336921997,0.9667930842402385,
 0.46308430723949356,0.8353273257110284,0.334969526087448,0.2746108612583411,
 0.525430420674285,0.9937476621289317,0.5491793891565782,0.30580109733102,
 0.9749335835319831,0.8772745795814584,0.752248597991379,0.862097129859121,
 0.20361323651227314,0.8902045260965282,0.8981560431152278,0.5549641124476362,
 0.19984028984611524,0.5395397545519356,0.15020356982260252,
 0.12811455130217353,0.46198134324333384,0.5462019439851927,
 0.5374665356374275,0.7991936331825582,0.7113146369816665,0.232514933408851,
 0.48087229820500776,0.08276887673691258,0.9968481018366364,
 0.20652009101999325,0.14171320458571235,0.9021852480822627,
 0.03643093394872732,0.39073445244158556,0.04005650080871881,
 0.3000259394268676,0.08076510859501385,0.9194235960247823,
 0.21618078624974824,0.7737050894825649,0.22315375990426112,
 0.40665825199184635,0.8201825631320157,0.17617387279855468,0.55367326882251,
 0.3471468215518957,0.7515769515943326,0.9600124696748654,0.7685947014385569,
 0.8199078710277679,0.07580385293862546,0.23671946724336823,
 0.6296355674415388,0.5596215687542252,0.728931585102754,0.8919584886128228,
 0.22899888865516094,0.10767044562429662,0.6413227042040007,
 0.5713391816586967,0.02939028746020378,0.9886261170885136,0.1336117464974962,
 0.9405967458367077,0.8482137629165432,0.08242281090137815,0.3166094028062677,
 0.5301194198740369,0.4210287773610225,0.7419996712522999,0.8773218805500753,
 0.416841420083922,0.7242854178422846,0.930825427341906,0.9683498948199534,
 0.9347654705665078,0.125042888086632,0.09786415186615971,0.8598551211360925,
 0.42676378113265034,0.7485234766400378,0.33761933514969744,
 0.14513286229727584,0.7432232485334113,0.4374133466908682,
 0.011713210176043765,0.8718756132977276,0.20432450592060558,
 0.23562064276477723,0.9425698872540959,0.3820673861317545,
 0.41002144699385534,0.5782016397264326,0.1971497084684064,0.8692676741162586,
 0.774888242098604,0.2828151060866644,0.01395988012701499,0.698785838688289,
 0.36480322597261294,0.8434492470751769,0.2087244298206632,0.3042533634980866,
 6.485911437230651e-4,0.16731943945828087,0.022246366655381045,
 0.5104836518327185,0.014934556326722137,0.02202477466648145,
 0.37902875862050345,0.8536795425213485,0.3837248885838034,
 0.28922546837997454,0.407387721006056,0.2415971719276926,0.7975365117989859,
 0.2937063634198318,0.7548272329357962,0.25954569680796136,
 0.18505236980484674,0.2156029273162725,0.2936657738717824,
 0.018338873825929913,0.5266358558365019,0.41203912294191836,
 0.36079066214778666,0.7572769036877799,0.5816939163356797,0.8503038245075116,
 0.609133603596836,0.92844940635332,0.2790116806932087,0.31690200819615144,
 0.2339129863737257,0.4068357673402814,0.16994245982976197,0.7257658666410101,
 0.8568707260590747,0.34660325685541205,0.1866243898159582,
 0.08716535366178957,0.15936433010034934,0.5052069652986566,
 0.3664524192321771,0.8219146266295786,0.38384837256595594,0.2174872909861858,
 0.7685292551367159,0.9413861814714304,0.504560618181169,0.9876643083919945,
 0.4153727187780163,0.08436982611839328,0.5119848771790764,0.2729094942377893,
 0.4669866216076536,0.5826272402950228,0.2761568568943581,0.5333822904380869,
 0.15509995985596658,0.6975363800516402,0.9436842415463166,
 0.32904898611766653,0.6417915198434905,0.11849601539965615,
 0.24724571465294654,0.10116304590816638,0.7879989594497232,
 0.5089493354646201,0.42025345833673666,0.3581872762033963,0.2943126096673171,
 0.03740814854636265,0.005966917297037355,0.43253946976398283,
 0.5883558155324174,0.7381034669029036,0.6021300474998074,0.9712620899396064,
 0.4691924028103698,0.48741848386689834,0.6313104339570169,
 0.17615226429569386,0.2894926250708689,0.04209580506997168,
 0.1645502928711856,0.8821775657342993,0.8571350574096995,0.6811836196689685,
 0.7608988547127793,0.841117171995601,0.30102720582858533,0.510238562104824,
 0.45751647156601183,0.32972680178521774,0.29873222756401985,
 0.349378973561256,0.06865337136325411,0.8117314054872763,0.54538722470623,
 0.34201607398122547,0.9405370356454393,0.5385363216724921,
 0.45063173936104384,0.9282718086689123,0.4417915249933295,0.7480714537948793,
 0.6135300987835566,0.5626919032009403,0.7006820276465582,0.22114260122224783,
 0.16982011564465116,0.27283977809059856,0.5236139201899948,
 0.7914549392738827,0.8916054263186245,0.2530321459476855,0.42213719886657963,
 0.08929680935605233,0.8775734351970477,0.6628011825652906,0.5276239982237505,
 0.2732242220265596,0.5420055185308899,0.1981769963691571,0.5342107515363171,
 0.4257063547896349,0.6045033322203659,0.927872323316811,0.6475839639110785,
 0.7249154505775907,0.37777999730486933,0.28474134938543083,0.512053821434111,
 0.32134063288757697,0.857007004955527,0.893659512880295,0.6889960276098684,
 0.018688132521797907,0.9933499342737022,0.2516185692106544,
 0.7411670597149682,0.36114506728423235,0.022944039287777107,
 0.802175419381522,0.359439674748242,0.886048645318792,0.7673027728711939,
 0.5562974513304667,0.18016989144628281,0.30150233455324393,0.2629649723006,
 0.9262801604919935,0.5554358687259973,0.7740629392461595,0.7914511667682383,
 0.9799095529992015,0.24679124821169163,0.2774456922725924,
 0.09622162226049535,0.9134250122039664,0.12672388584732197,
 0.44442213693440824,0.9855395583996219,0.2839173616246976,
 0.28064757657496575,0.4871965840000362,0.24355443720818692,
 0.3218653463893214,0.3918284684411426,0.38169856169647787,0.9574368934470089,
 0.4626654740180456,0.28053657712104685,0.8457379202903014,
 0.03339841928760956,0.05766370339175142,0.506686551667659,
 0.13976065090403123,0.05030768654558038,0.26876607399985264,
 0.6133884654699372,0.10551123470231571,0.6051883573606992,
 0.08956076595534712,0.40246105089793405,0.7898041205344464,
 0.3355808197632646,0.10586668657020948,0.7073043200135855,0.9716782178208434,
 0.9334544069807796,0.31020758119458425,0.3795245467963676,0.3828427750271366,
 0.27908481561335796,0.2811218564074959,0.33604366268169295,
 0.23526269434483527,0.40021601379705873,0.10996131962482703,
 0.17134751912784674,0.31402909771762366,0.8386868554867968,
 0.8005093454049781,0.750391274728783,0.2999920640660946,0.42658490525091397,
 0.42397729168871356,0.9682539292581022,0.46407487349627496,
 0.27344296827447345,0.7559007300437619,0.27402755568576365,
 0.46691012400442156,0.2897401032504703,0.5578390496232492,0.1842859182762968,
 0.6894989231694656,0.989427574625029,0.055521158828650274,0.2861108957741635,
 0.17075163641439528,0.5048662146257464,0.3336695066497407,
 0.37828740797577853,0.9775045712922321,0.7606546481050106,0.912664579636762,
 0.4325304444869821,0.5537829206995167,0.6537545726679215,0.18176642884421979,
 0.48311202600741054,0.26726609455371075,0.3248815971552048,
 0.5726158936577554,0.9840991062428255,0.18138786682900232,0.3160495116888944,
 0.864719951907631,0.9326401773769496,0.6747005206513275,0.007590031630372085,
 0.49411319613784965,0.45158849736859297,0.44154968262488403,
 0.6678341293796043,0.039001271006426586,0.9718925115560987,
 0.3454513618766406,0.4496365930770423,0.1398526373252239,0.2753147201468622,
 0.6589314453384323,0.5179692557152065,0.7034638216892778,0.4220419531611638,
 0.6019385314263814,0.2808920078805881,0.2557641788984377,0.72039866561569,
 0.8192822832182387,0.8602312102292596,0.3548277657699296,0.9261280148593163,
 0.9432979987349948,0.8049690711666687,0.2468220254808745,0.8823724364196093,
 0.45510987764794686,0.1463815914216734,0.016997689000062155,
 0.8518290205390422,0.5342290813341545,0.5226226160595261,0.29261754178899424,
 0.8539605214284363,0.4215135988351958,0.19688170946200667,0.9179022519394068,
 0.6442385759365103,0.7431224911422708,0.15839551593864765,0.3584636540675873,
 0.8939304985715294,0.6933163386800267,0.6456744857736826,0.2632383724769899,
 0.6940209722619725,0.653583279229373,0.6899145056317522,0.9745513464539369,
 0.0508325481825872,0.7816412328285366,0.34000464963838906,0.528071727634809,
 0.6037613394438801,0.20470726339640666,0.7712022926290543,
 0.07713856893669069,0.5659380599306897,0.5169075673362719,0.6370005124018994,
 0.8390254079375654,0.8924903017137321,0.8490107057075525,0.9454086452740738,
 0.7494490094395895,0.7558222228650274,0.9628568364626171,0.12357324232314693,
 0.23493436187010275,0.6781211262173501,0.5833186187129532,0.5077684184278488,
 0.8754543788195149,0.3839620066364723,0.042283828585018,0.4996400925069805,
 0.0671463629673481,0.5075280081530807,0.886752365359032,0.715270884345826,
 0.6391889655308985,0.7349702064126397,0.38436357756793404,0.6193484805608022,
 0.9842098278240105,0.4194207696481681,0.6556231139467215,0.28177514364638256,
 0.3552199778954396,0.7597875966039837,0.4105454441811581,0.3189730610005991,
 0.34434677364642363,0.7093917829838707,0.9634793730235757,0.5884930165102955,
 0.12822429620016718,0.46239611974029393,0.19993475397097793,
 0.8572693085268837,0.26791970804716336,0.46273772028283955,
 0.04882694524478515,0.13706756793636155,0.15708777213379332,
 0.9124614909425409,0.905997166231598,0.6794093737061471,0.3845395941979841,
 0.6809772943501067,0.7501091643852359,0.05645269843243916,0.5172403460924416,
 0.11573309254200015,0.7270637096049101,0.8628595663955677,
 0.44828437001291455,0.7012896677735889,0.41590472142340074,
 0.43909915862177906,0.03440196092264125,0.3892999991988202,
 0.17425028209426507,0.07234474858644235,0.22572084400325032,
 0.003537887400013151,0.4008902594581013,0.31546598767681033,
 0.8358450842748827,0.21205056879542972,0.7058825165815334,0.8344709011645327,
 0.8288823913232989,0.4364025601170275,0.6174662678526024,0.05267415445172774,
 0.08596919617243348,0.5252426208363379,0.888296576574353,0.41110385802303384,
 0.8406189857421826,0.6649215390303389,0.7554849977026268,0.2490350724039322,
 0.18960913322892858,0.29663166394370344,0.7380357053869914,
 0.7812501469989325,0.4459918269534877,0.6790163788817164,0.6389133218585725,
 0.9405014003144705,0.43013881019588474,0.4134314368643358,
 0.09480459518312612,0.16866176395726318,0.2524554044826389,
 0.7623516915017068,0.32965336413799196,0.9057451125792036,0.914136106300233,
 0.21722849754742635,0.8996705601382752,0.4809587252829971,0.8910820886427854,
 0.624139540626887,0.21039260219849376,0.4490812960676942,0.55798504656059,
 0.8139591042256423,0.7490967222279351,0.3465483416491473,0.11407671291359645,
 0.4734156979808888,0.6397333901502679,0.6140132867070038,0.5285936483014015,
 0.6484985998732435,0.8881710623012382,0.11908882735712667,
 0.04114954562922146,0.6650219119318652,0.6204500348088077,0.3289395347494858,
 0.27825023369449264,0.1953250805786647,0.11609812896406502,0.422937623410376,
 0.40847507897333346,0.9354423938606506,0.5151505600833375,
 0.06370006038774312,0.30311972876188276,0.935707365973533,
 0.43961961481471046,0.8968512897813735,0.29440572715499924,
 0.6952439748485099,0.383296966836834,0.9694256203843473,0.26547675301458507,
 0.48498297597791884,0.0609548094040836,0.771861318067803,0.23910119205903146,
 0.4038164486075275,0.18645156955119435,0.9356915663449883,
 0.00509997708547949,0.22636575259849678,0.8228969778796087,0.766004545881613,
 0.661856993572266,0.9876956413999107,0.9861128453025901,0.2830699349857282,
 0.6974797029762002,0.27754217470797315,0.39006161441541776,
 0.5999210417888244,0.3469248072226261,0.13352500268700718,0.5415078613172838,
 0.012857530987074206,0.5881623482561894,0.7153978052054878,
 0.8815125042316913,0.36725381753419917,0.3736234518963125,0.7619261761709091,
 0.9959184756462169,0.207284980082597,0.3608151623548439,0.12685403748073976,
 0.28727936812942123,0.19813767522671022,0.7968552536330349,
 0.7864388592844984,0.2538386549843472,0.180042754101139,0.8820017081485,
 0.9750973291746835,0.09663906985932436,0.7717436569776757,0.599881526715884,
 0.7171896397808285,0.8866396830312817,0.3524674527082362,0.48990255778920866,
 0.3673041375709266,0.975615215849841,0.7023835807927046,0.06859617043705424,
 0.4868456191830798,0.7078245422059847,0.4627658670929302,0.30056613814553834,
 0.6497672310634278,0.3680633160479754,0.7204905342979143,0.02378524526247272,
 0.8191196197414625,0.1296326991647868,0.8050285719101872,
 0.0015369599396163336,0.9501822295150052,0.8263380311215175,
 0.6316420037336327,0.5630038072799893,0.7810616844624145,0.15449345307237372,
 0.6995961107934573,0.8778801394308834,0.61046789832845,0.9242953129890994,
 0.2512024731958621,0.6704911526977458,0.49417391159202095,0.3377756416397746,
 0.7623359128269767,0.021589698275582436,0.6827021002331655,
 0.7311669396685692,0.561846359213013,0.8760345654679584,0.7937196755573809,
 0.47626542966539553,0.15713471140652702,0.7385207157099581,
 0.9504741998163146,0.5685918530409373,0.14861548063870889,
 0.04952079983818858,0.3728082572774305,0.9275104064449237,
 0.11302772483376666,0.10198562479738182,0.7672504440173796,
 0.6880909861333805,0.175120841326189,0.5310733079379447,0.6912511470648538,
 0.7992480436167754,0.5882228195528199,0.4628606599970465,0.11501412372550601,
 0.8457600756389904,0.5596793698897935,0.36221894351275974,
 0.14864617060074226,0.8297018945935279,0.9028786995524191,
 0.43559725304238583,0.8006904289250548,0.7370458633495661,0.7718500582272406,
 0.1663440936650794,0.1197930292427446,0.28377170342915836,0.3482747278882504,
 0.3982982877743946,0.09162916586726477,0.14473102798260884,
 0.41109573087925444,0.16240799564080421,0.4874780683911957,
 0.28278231698279366,0.3672721468399598,0.7759581001492877,0.7660911879065541,
 0.7448089207090507,0.7596913074540044,0.298466625711423,0.2935573517634962,
 0.5325651112400462,0.19727023667962373,0.21568682106772152,
 0.14069000446428592,0.054059340518916255,0.9110238029096863,
 0.9655458256258832,0.6290395035820184,0.5549406369213261,0.8589081130779433,
 0.9650128541976073,0.1379177714397961,0.7704327112374763,0.1095349966191499,
 0.6518622080237201,0.13398028907736392,0.6987268068011124,0.9618648688342025,
 0.7354935047893945,0.38239196284300525,0.8833254087758159,0.8654420102834972,
 0.4575797992165843,0.8505155556089278,0.3110566663224278,0.4801138288933783,
 0.7318607189671508,0.9446620699639303,0.3085689130395136,0.7949640524833602,
 0.20288317993678107,0.7468202946923973,0.15326979042286726,0.860252608905522,
 0.386426701248897,0.9819464699300544,0.712554145622888,0.05809424436819988,
 0.7898275268482342,0.17230626380233738,0.4538840600238514,0.9412213254952698,
 0.8671387839248843,0.02270587467547258,0.44840278558553714,
 0.011457220756434627,0.7080310082615004,0.7017873745682526,
 0.8933911583796122,0.6317423312978633,0.5703537852178895,0.6579581315320812,
 0.8421440892501965,0.2804100432835479,0.7833735720044783,0.3997460727741592,
 0.053333421946447856,0.03584127205497456,0.06953262657491177,
 0.66155966004141,0.952980844817596,0.9273029988761902,0.2747799148770078,
 0.8915851674212758,0.6341084760351666,0.49016097305047346,0.8287022209717374,
 0.5709260545210247,0.390409178890628,0.7757430627847544,0.7397619890410815,
 0.5687073519570721,0.7275873271498516,0.11338638259311873,0.442871141187807,
 0.30253399638080347,0.9270630103027051,0.9044674233543392,0.3893150171981321,
 0.10303328421775748,0.8673962730972323,0.466576460347512,0.9884110779582368,
 0.7130703878596689,0.6441038948699174,0.02305607547756583,0.2515525242526887,
 0.9224645778669098,0.026683782137746248,0.6430930073422861,
 0.7126331482426213,0.808430189895093,0.812980519460625,0.06959711512816968,
 0.32165343182010553,0.9001297400386672,0.2511284955665216,0.2518801727533525,
 0.5408743010184636,0.07740447679356599,0.7315099946288904,0.9236116944706002,
 0.17841884042138723,0.11167828934401147,0.17571487273360498,
 0.4789072856418699,0.5408805356711602,0.07499370639500735,0.4840354592805075,
 0.6410611189831128,0.9259287277947739,0.4713969602288549,0.12502504765622924,
 0.19253198945930983,0.7616350487801539,0.3898028512174908,
 0.08288456267720523,0.42690818732514635,0.4050949722553784,
 0.9013363265445379,0.8355959955817043,0.0739319053055093,0.2960581969051085,
 0.3152490816479879,0.6388632935616255,0.8270502855725465,0.4264283100874552,
 0.6031340177458488,0.8072608310642714,0.38722873369260846,0.9049354855881078,
 0.5106752698573804,0.8586284639620287,0.900333665914981,0.6063207772158947,
 0.8603109049022031,0.7502188162152658,0.9811213338008171,0.508442745198599,
 0.32466652181628053,0.03436756933437779,0.30464338187535567,
 0.35729005604735176,0.029560940246338263,0.6108363503513453,
 0.3038595054245774,0.3955443831512746,0.9967263282532539,0.3431039533362815,
 0.4228396894849128,0.9326219216975656,0.7865994544381067,0.7252290726078255,
 0.1901986339341144,0.2840318837174731,0.9674715266534525,0.16359110282997547,
 0.20241532616369384,0.9710888214131586,0.13786701321381556,0.777795193392706,
 0.3046366621201969,0.3575364210293225,0.9994811213111482,0.5492785154466406,
 0.18946394421512924,0.5163776263790437,0.9765188237287113,
 0.36129239770532173,0.5568020725563868,0.6830979638551884,0.7195471860903048,
 0.899218800293383,0.9109265290259166,0.8957501081105852,0.7818354111136615,
 0.24357838022568445,0.3458740231834867,0.6726119249923311,
 0.22736248896060351,0.43288069206286917,0.30990310207007443,
 0.07962472783248065,0.49869349053294665,0.9619040927496474,
 0.2622378944493515,0.6876143889659079,0.6526434886048328,0.2677380368286508,
 0.41935840065023244,0.6090372653936809,0.8993417423007833,0.231507077652215,
 0.2312257539588397,0.4438256055278984,0.8958319281177473,0.5363935982508327,
 0.22061190037226885,0.7633449623405625,0.35953336696834426,
 0.9455354867762943,0.09612402550838717,0.5393748454389462,0.3471389158910073,
 0.8000534800540962,0.7708995607063938,0.9103203149452056,0.4316231657199383,
 0.4458524021120437,0.1462463175018719,0.1834221101915191,0.8267874216197172,
 0.5669041228812667,0.37188752644397727,0.6760981831079829,
 0.28720284854989075,0.6354898294172076,0.9844994582995813,0.9251658758469089,
 0.456397283461759,0.3909401664514818,0.49539602319728193,0.4680885382315655,
 0.6963529217253969,0.3541172964646103,0.5975414533753917,0.01166858303306173,
 0.9506904423216739,0.5556431241778749,0.4293608855722659,0.08682694765384658,
 0.8682411381694497,0.5355892015900836,0.7757928989279629,0.38986995102426103,
 0.9912121711291363,0.07505618766010425,0.8125923470413733,0.6770962050682386,
 0.590105631002031,0.5198354012080153,0.5718113689046794,0.1102607838613453,
 0.12897687189248763,0.9492977306750168,0.7524413437975925,0.2732588020105996,
 0.7734503961460296,0.18696383444474307,0.12122977744030061,
 0.7646686414403634,0.38715094521399174,0.06321746132056161,
 0.16316637599056472,0.5954964666304396,0.46065307370061404,
 0.4921794631729215,0.1166268772807697,0.38710343601875397,
 0.45867661994750786,0.24856872689996423,0.9084314516448321,0.806591555851516,
 0.4131680513359419,0.8045583199123856,0.534503550641779,0.769357292607737,
 0.42775203712286713,0.846438994627598,0.5129114431938757,0.1787720574767243,
 0.6154149787872102,0.4186133048667092,0.5303084540819509,0.6063340884298587,
 0.9882104339075344,0.5357894576315256,0.9731760156566704,0.8293808262919147,
 0.37157287914581016,0.761771759171358,0.7988459646649314,0.8823778981629321,
 0.3779377508930266,0.45153242577767494,0.773018011647564,0.546180532430006,
 0.024055665940134086,0.4616083907143741,0.12003882750779549,
 0.6644319687966769,0.773618108571794,0.7681167879124551,0.2191689330025659,
 0.3487906169007716,0.48591005704978685,0.6297345958667613,0.8799972014530588,
 0.9773101779348798,0.2562151979863745,0.8240841403482018,0.5170039513633767,
 0.1096636352016197,0.5180690938489114,0.8735032907316596,0.957827868735639,
 0.5174301859819018,0.6335348430177309,0.7673268438536407,0.2874351646823675,
 0.37116902549266617,0.5639409661597439,0.4528925548306476,0.9608472542492905,
 0.9520835674192851,0.351896919437486,0.43195474270700873,
 0.028703139154666157,0.3351858277152294,0.2774296511643688,
 0.6162540373902475,0.07422851031927213,0.2124933985875631,
 0.14145344648045743,0.24124686129646689,0.8396700796400207,
 0.7689396679023671,0.8416273784387072,0.38787497416094696,0.1485702182644767,
 0.8893715734036161,0.6452163362441684,0.27996211575056096,0.7361102608391092,
 0.511289682553956,0.283580300772278,0.29365278114280857,0.5249032009837828,
 0.6767822259013704,0.8215674978059172,0.8116447021269337,0.24485531047277465,
 0.549646788351359,0.7677970744517565,0.4344248998252903,0.4183272604018401,
 0.14857366722804055,0.9075755251961797,0.9710365879456838,
 0.41248133240190377,0.7508845104289251,0.6238105102293084,
 0.07792804692138355,0.7119736015823055,0.9596360735550896,
 0.24929466557391777,0.6447852031943377,0.6502534859049851,0.8714098277443312,
 0.3260152212418015,0.24485328930425776,0.7970982900009668,
 0.045290326656936886,0.5319747260967089,0.14551472639352003,
 0.13227268759136812,0.4048158937425286,0.243069853016313,0.5969504137081064,
 0.6737607997131586,0.24314303468602483,0.9478673506349169,0.6586346612363707,
 0.4174102815788352,0.16949952219002695,0.4779974179767824,
 0.33832250758545557,0.9075642398205928,0.041754256104603926,
 0.9896676075751334,0.010370092701529776,0.8218889163645167,
 0.6321317890969294,0.6071212596389258,0.9847012557960595,0.8532990269655096,
 0.18258122750057704,0.6008827549025,0.8906966310663292,0.8934869713256631,
 0.8395518481106874,0.9435527165844425,0.50748389899134,0.8149096615702905,
 0.6759368103137603,0.8855687014199348,0.37631983954259507,0.7016832480247792,
 0.8791544537037669,0.44365033191591896,0.5470987146188145,0.8671016604782357,
 0.4926491569154179,0.9211694157246066,0.35776689316839005,
 0.010179254422986506,0.1390611781881037,0.2683856539748457,
 0.016633323098829367,0.18813012118603767,0.7265940385765437,
 0.08227239844345957,0.6042371355731705,0.8317853040034845,0.8941629270754113,
 0.8144019420626759,0.21967573181925182,0.33265684018198016,
 0.4765604196600236,0.571458976352194,0.6699333312943179,0.9902296115361795,
 0.6931568690203056,0.7170706431030168,0.6230636620982819,0.3181781046463734,
 0.24819620211043192,0.2382532233191248,0.3004472763896243,0.9818043626242948,
 0.6106081869297793,0.5839931139263104,0.3795978174820789,0.4105403097248147,
 0.32246652171813617,0.7124354856232754,0.3739959347931483,
 0.20019175018821445,0.5981912263895911,0.347408222682256,0.6876982935120527,
 0.0828377325816454,0.5615952709443411,0.7705993710121015,0.9135351883869867,
 0.5893000925429498,0.33279922339292023,0.967838133563944,0.18513765787598335,
 0.08277004875071026,0.2247773148748483,0.08728492712713554,
 0.36248327330926233,0.02903072525548933,0.4257703822210811,
 0.8561798841007082,0.5267282817193135,0.13498027785841626,
 0.29366286414540466,0.141872696655579,0.48485524126487356,
 0.29732754014115115,0.1592365939826097,0.033795640667916516,
 0.07880083206789512,0.5387000750243336,0.6843730851062341,0.8161398642673671,
 0.46707200853815556,0.3667703816154345,0.14497358761372792,0.32022502658632,
 0.750978520442577,0.7580700976659648,0.39722928582933026,0.28851409356873503,
 0.2775907704157774,0.555122203155037,0.283931440600934,0.27780189792232646,
 0.2480638019790875,0.3107917225960607,0.10880043691872665,
 0.07471602914013453,0.14956548413486115,0.5787414782182212,
 0.7403660466481757,0.22006247713613447,0.7632140436595223,0.6266467970351303,
 0.5349095182403666,0.1256203803677418,0.79688474080425,0.22024124992247407,
 0.08811940199098556,0.9608536447374746,0.38350604371575503,
 0.5771713245405228,0.14172171394641586,0.584460005791066,0.8545698068378212,
 0.9850769301400243,0.3157242510069498,0.6692187186186369,0.708319765538341,
 0.3231471275375918,0.4759185269904461,0.7743926734291963,0.5028518503897575,
 0.0962361199617413,0.44779120130477623,0.41711622502834134,
 0.4657350215883448,0.6397993182822477,0.5260689985058805,0.9382023082301253,
 0.6443328576145366,0.31743132825530096,0.20440113841760843,
 0.3694275128359372,0.5873675489238713,0.7481413366271014,
 0.016570954812483718,0.09684827515853667,0.8706929985019621,
 0.920720095753604,0.21314866081613448,0.24507162843650887,
 0.46827106149640507,0.5259251853382919,0.47302036554058857,
 0.7383140345808147,0.7194844694991782,0.11256701273395742,
 0.46355131688182705,0.7504547643379262,0.9187183071085109,0.8533910229590081,
 0.10715049880508487,0.04367024516673246,0.08725155668207718,
 0.7087130592534729,0.3836013722975483,0.9633499241168657,
 0.032442682971478454,0.4851704648744709,0.2656941869113286,
 0.7188345892396367,0.4078568665759331,0.05926181116037055,0.5016514796066883,
 0.9643169208395362,0.06435677008358842,0.7485432804332403,0.2953907197641539,
 0.18742775402492085,0.875545417884572,0.5784433606857923,0.5367728638864329,
 0.7646176905568404,0.6049195534363462,0.34140411057858633,0.6231228174869374,
 0.8112489948525353,0.21293319096886298,0.735811818132978,0.7098858944865015,
 0.15133631435580353,0.3754357492010053,0.2605673816323787,
 0.031517824718331644,0.08678961776428906,0.11666950095312989,
 0.6242807091207444,0.36454371679952435,0.03868459936089441,
 0.2575755600256212,0.5854583902557355,0.09595355993630772,0.9939219974276415,
 0.4887985387448962,0.8408868480783859,0.3677871030752815,0.09906689332097063,
 0.06935215638464565,0.7424926181481897,0.2646536643301649,
 0.007133419985386635,0.7523634350826613,0.1412664433400771,
 0.6711728020984226,0.28894917880436166,0.2178101693564639,
 0.39111793396004146,0.7481520254762685,0.009983290715974702,
 0.7942762641987624,0.5827790462775572,0.7363788598393762,0.7262370209241193,
 0.6809659937009488,0.49638594569164485,0.7313257587041989,0.5711481881503658,
 0.695758343056506,0.4941358401796856,0.25667098496596963,0.19814673437740438,
 0.9841021886041528,0.6734790700264407,0.18733795810081577,0.5276905700373897,
 0.25293348461421783,0.23298687389758932,0.9318666816064912,
 0.23000782394939412,0.3371373322060385,0.3590174919944126,
 0.09165682703093503,0.033892802194183336,0.29895093411170304,
 0.6400025753718285,0.12228969755070773,0.3586062432360255,
 0.46046433397187014,0.7174035054438024,0.6462706124063051,0.6801611309973875,
 0.4574952600093286,0.7969997076420171,0.3974104469102181,0.6732797169735308,
 0.569869805586366,0.5482937785928337,0.6860909336585361,0.8617297711765608,
 0.6599155425961892,0.2730467595392494,0.7434055000746584,0.6590861803064385,
 0.9066957394550565,0.9547905973694326,0.27009158238304554,
 0.03155422942902342,0.4799797351049532,0.023504403815093222,
 0.03952298086734696,0.8057727132287404,0.42094809934549793,
 0.2020254406180465,0.8586581593363469,0.7829016035859724,0.8012781322317458,
 0.07351176957248351,0.9580373435780171,0.47247433911254866,
 0.37578562702850393,0.8554612450467279,0.19251482101456774,
 0.22597784161821455,0.9787962197621046,0.545772231805338,0.1253160263825024,
 0.6390564471690537,0.2287622832793318,0.6998719600903534,0.6165697472718952,
 0.5854054076952692,0.3259056324013704,0.6864484059017597,0.3115848368711467,
 0.006107043764865039,0.9941843370770925,0.7080595569393723,
 0.5274480330645731,0.11237325640496698,0.7924140253453159,0.4486966520157688,
 0.6478687081486768,0.25392469604904655,0.6749387210303497,
 0.18630428641787733,0.15886983815294664,0.846150742631658,
 0.15430929766130008,0.9351501486946984,0.8709542281218121,0.1942050919071875,
 0.7267148930686634,0.1960475984836857,0.9944139491656733,0.1946203378054443,
 0.11841340527919597,0.41851161520309177,0.904161670798866,
 0.24555952949144977,0.10415601671732944,0.3730687606899149,
 0.23245748122313836,0.672165425579855,0.7048616740830307,0.5028269404031727,
 0.8115974361699401,0.911312606051972,0.04926149287653703,0.6268207758107013,
 0.1858903776236317,0.7564159371269927,0.12273071152472259,
 0.13222402930231103,0.6169351196645174,0.38952818820255053,
 0.2510103376470558,0.691923192242172,0.6429098137317385,0.7496775408821665,
 0.6355551842451275,0.32425038707968334,0.2508063376097531,0.9948972066333105,
 0.024258172191460314,0.730584587069033,0.9871054719070281,0.9012858623703764,
 0.3391820950194767,0.22320914131033787,0.4732663093110989,
 0.12236137915689049,0.0820908211051602,0.3416066629167086,
 0.49304329937251123,0.035595418234975806,0.5626940515775471,
 0.41335257048310914,0.1893263571470407,0.03543480892116557,
 0.7201423110655918,0.8307774196114117,0.16416963846687016,
 0.02307017722954019,0.7416263549731656,0.9809646876622548,
 0.36604411376115453,0.24287058831841368,0.05518460206026854,
 0.6719498269686679,0.37748078456363265,0.17514762682026574,
 0.23676004818291707,0.40907865741819527,0.5072907835746954,
 0.36000830490047475,0.6494316229917044,0.4375104104753831,0.9060445136193659,
 0.6648311952527909,0.6678547742259876,0.1952000168713217,0.04824733837093331,
 0.1804902869733146,0.6936506321421501,0.6790246639701943,0.6336209760136293,
 0.9471016702151045,0.007072524634916921,0.5157081794684599,
 0.028369114120728733,0.11100284181190978,0.2861431138096101,
 0.30847174464015237,0.13527531484439737,0.30842875340994247,
 0.17176755858958714,0.4746171162142728,0.5644762332805839,0.2072384519470366,
 0.811350144253264,0.5758118368992763,0.27200752258179606,0.35314352142395944,
 0.951854721929864,0.5391248152296644,0.6781897624327665,0.564722966203549,
 0.6131409518795816,0.8029139413002759,0.9046588788134512,0.5319882342685043,
 0.6511646349697805,0.7165496734937626,0.6155863235742014,0.8138236738349961,
 0.26932784089333617,0.11260147587823832,0.288010131184823,0.3496534043920697,
 0.2530086009025644,0.17857313400107744,0.7426768882075298,
 0.37046051743133157,0.4138255924910974,0.8022977876778264,0.7603817957566126,
 0.3828930481800672,0.7041924082620549,0.327858826070351,0.14282456882036176,
 0.6044966447494956,0.823884892482378,0.03443326647838374,0.3596735468009268,
 0.6047072026245537,0.16930857682699096,0.9303123801832417,0.9034044407860996,
 0.64181562698883,0.02091109366419075,0.6501675990493366,0.43354565958187097,
 0.7353538322736989,0.007339507515242261,0.7164978538000984,
 0.8631912049225132,0.44681001081066274,0.6122708241844003,0.6180965305083692,
 0.18086327628614052,0.5411070292224358,0.5313265102880202,0.3922298773606724,
 0.04639669976952876,0.40089736020814704,0.5210400963741332,
 0.9079236700553462,0.1270973390921275,0.996958312432393,0.25856318535925316,
 0.9539628982122099,0.8644133721593775,0.30425725166429995,
 0.24758116807322306,0.5077917532658013,0.2661394101387652,0.5977413856943635,
 0.7391540266185725,0.3385684303503428,0.6448884973614412,0.4002743982506448,
 0.57786144597194,0.9996031367609403,0.9711245827440047,0.7289803199419305,
 0.9981930372750332,0.9407893762336579,0.45004322087212034,0.7423628563198988,
 0.2976354427705695,0.9542303186970515,0.13942781284214878,0.2853784235027501,
 0.8063221046531158,0.9394218842233583,0.6230029647963908,
 0.040947466767563556,0.12545954162305017,0.41799403050227746,
 0.07207822902193484,0.8160350431686743,0.8099457343561383,0.598598059516613,
 0.8064269979128266,0.3015656278182798,0.7306054474055478,0.3217880161061719,
 0.8520015683702294,0.7683737502981912,0.4704466614177548,0.08278706831008242,
 0.2841636854012545,0.8499550924121809,0.8873264061956277,0.6671354613681834,
 0.6531457310151179,0.6569989440507026,0.8245232104667783,0.02026482245026051,
 0.505575118176097,0.1824854376866082,0.20148581268380317,0.3057673561219503,
 0.6764009415489985,0.18919408787840097,0.5746904622654836,
 0.21010134806823821,0.7111900597731698,0.06011360485847095,
 0.9384810147962568,0.5945835312489818,0.13825116846501906,0.5629456063067353,
 0.4327395015163643,0.5192206868817758,0.7928173454946165,0.835103839515323,
 0.0578134461144959,0.8306386026492618,0.13122140308034713,0.6551953695462933,
 0.7502409791881923,0.862810633317014,0.34817819717626375,0.7534818863768075,
 0.6593847150457982,0.3928026035267017,0.7349611532732718,0.19335117553720416,
 0.5144556745343115,0.7655388658407467,0.8528405881391191,
 0.013700259137440773,0.515509658478804,0.22538221800266756,
 0.8318689334716578,0.7802841301961244,0.0353428485961913,0.5612032643039393,
 0.36310581299152833,0.37101699773065744,0.8738208768508055,
 0.3273395809260765,0.014426722394638603,0.527238378814168,0.6918667949433115,
 0.34133841462918646,0.5551416038183543,0.9148215527174228,0.3203014803637758,
 0.3044451253294441,0.443350263776634,0.6512483140756644,0.9828142862129816,
 0.6123921534294864,0.2681866856082785,0.5670831711292089,0.452635553434912,
 0.4891469913232672,0.18037643968095796,0.2510974323664873,0.2634865717992926,
 0.74903159518956,0.2949671126934662,0.07338430307578125,0.06522324095293497,
 0.4784628013906277,0.7502529428267625,0.4897261298921378,0.5544285243908232,
 0.545045112440894,0.171536393074754,0.7360486385257593,0.8859253189376081,
 0.7213024595844513,0.6657241803438886,0.9217937639601268,0.7674914050839067,
 0.6279712427569548,0.5204096097043628,0.5567623678163319,0.7024730932107205,
 0.43780802018605813,0.4162686162778959,0.429188958102397,0.0902665217435612,
 0.9381681105158217,0.3604942728786833,0.12505113056483808,0.7106927379057961,
 0.009795330393192025,0.031052986635703328,0.4270790691561466,
 0.5476596876161116,0.3607693241218687,0.6948752101665444,0.5657349778653444,
 0.6849389046778683,0.7635779261359145,0.044746161747137636,
 0.10226236716039983,0.638554077985928,0.6924264317738303,0.4420051992328815,
 0.8424663961739444,0.9376186581863242,0.7720313809761412,0.8895290253767653,
 0.44647451441049657,0.30878654902026037,0.49934359013943286,
 0.4624245448330846,0.9127677279220592,0.6046003535049763,0.39092078587010604,
 0.993852979012283,0.435206358721432,0.06884984873715472,0.43619270104905516,
 0.7871245033058772,0.2800257505055619,0.1097641185424777,0.9573583262852648,
 0.9289080094963548,0.2736366113607869,0.9781469072304791,0.964670569728161,
 0.3746362989517229,0.4194968685080184,0.9986576633808595,0.4068577912222665,
 0.9664744159320728,0.37065031779319324,0.13480726014688493,
 0.08487321844875118,0.33822407801917176,0.8021944005234278,0.504802396223657,
 0.9907354964664553,0.2736767993333482,0.038922696054945805,
 0.5208684665887741,0.14184403861442596,0.5623151358016304,0.8016161258654282,
 0.7337587333298503,0.19808264283145116,0.11715966154299828,
 0.4541431730901173,0.9519598357183878,0.9198852314955751,0.36536516076368486,
 0.45888413527694594,0.3228647961069959,0.6449844478488675,0.8858117820074104,
 0.15794352138560785,0.25438359546004774,0.26132028318151634,
 0.9387391423298935,0.04815731305900661,0.10091439368421284,
 0.6699223478005526,0.4322273192354815,0.3339593281335328,0.04099393277242047,
 0.48745600780892306,0.5706306512580416,0.5439854096847072,
 0.46171508629149194,0.734606303360663,0.5215578906793281,0.8075273379400276,
 0.27040624030963945,0.9285969598414958,0.16450250121815368,
 0.08799033004028955,0.5929529283331847,0.03430255715108821,0.967409295479945,
 0.29035393184709324,0.1563744314821025,0.36475204614950907,
 0.9943388371180781,0.032569612911440005,0.08798450328019236,
 0.08429345451061054,0.9164140966779974,0.5189986907929917,0.9283173766338944,
 0.14875937453493826,0.283350694302118,0.4623297378675786,0.3684734995599539,
 0.6400390534026164,0.6549200144872435,0.36215146410841936,
 0.19860651011316854,0.9460951134750201,0.6460894771459238,
 0.11381298819114372,0.44172872719401723,0.2609355942083649,
 0.18809708686578708,0.424303928007559,0.6806858750536242,0.14762041561420736,
 0.06608274842764028,0.48740370643515196,0.7502739849651126,
 0.2617034988426368,0.1612984786869075,0.4861917713077768,0.2384473278219248,
 0.17386377303961054,0.09650321426078823,0.940581369855798,0.4200395871935386,
 0.3802671196057741,0.5201950681050551,0.39837240241828986,0.8050007477763533,
 0.6744963685241412,0.6662053119709699,0.7304133072631614,0.04711853594680859,
 0.6794859837338301,0.6276275324198471,0.015760120140903555,
 0.038255482439429556,0.429348527194202,0.8729950766361757,0.7167271874816208,
 0.45453311417630327,0.8028978321772597,0.9323988138642887,0.5616809842330112,
 0.08932943306190144,0.3126712818672187,0.5751922369457361,0.2214200330928675,
 0.8782642287969636,0.11030927184916517,0.6304701380264677,0.9769196418860815,
 0.5640892649860948,0.2925492585103312,0.8551530322990482,0.6812876403017925,
 0.0051606713295279105,0.4436099872761422,0.21021057039385926,
 0.001674854502818679,0.1252694847768373,0.1381819860866237,
 0.32713476908729633,0.7894123584635002,0.6615859477437214,
 0.06363221932324692,0.44075515250340225,0.6691887752065667,
 0.6240114904637866,0.0592364002920609,0.4912662441087312,0.4767570593009899,
 0.7925224569241532,0.19516129166118013,0.7299044942489792,0.4885999809908702,
 0.7602546647750846,0.7347508858858244,0.43875720409858,0.6596472046803032,
 0.7875488276077436,0.11807832448376587,0.689765092696105,
 0.009151724979473475,0.8720378748225383,0.4903520720557638,
 0.21736989437327425,6.455314984810112e-4,0.46671239587502034,
 0.26330083100977697,0.8835801594820704,0.9081069554369517,0.9685173598075916,
 0.9124568070938504,0.2101078143679257,0.27871295813683594,0.6938203110493074,
 0.5719240596545702,0.533560746809708,0.20172612388369748,0.9130832409321581,
 0.6009053942566849,0.7904530579778458,0.7211783708310224,0.10093214056951783,
 0.986329376069014,0.39299733261007763,0.16120458239360114,0.7693717097830118,
 0.40254314027218574,0.3755290502653603,0.9133656960470702,0.5173804560132484,
 0.24943429837740405,0.8668253114572518,0.8566009803214435,0.1398393364597068,
 0.07870692432309456,0.27562962502729815,0.3053656969091878,
 0.9965630281510329,0.6718521416955479,0.5631346885672182,0.20853159511534447,
 0.5944905266219171,0.31043406912987814,0.9896029339932936,0.3587421250743823,
 0.36886501869332733,0.9933406322017084,0.8947475245603183,0.7088339556316889,
 0.7196515947516963,0.3632805485644214,0.1503530728411604,0.43701332628229306,
 0.4631202459940936,0.5796349603612271,0.5792643452500149,0.21043317897575164,
 0.04505208818036044,0.5829834828197997,0.8268194079625478,
 0.16429696547410455,0.3181282636118774,0.37577022242752556,
 0.8882515319435714,0.43825596494511,0.4452757484033758,0.6739613462303082,
 0.9621839168032302,0.9937937909229427,0.23927972844471523,0.7711282405650336,
 0.7498752974824089,0.7955994886294869,0.6837732251127097,0.3190803139351487,
 0.3358153302343858,0.3364503009679515,0.4347708907423261,0.6094887622502496,
 0.555962815220561,0.11926085879189874,0.5451940019108011,0.9639451890197592,
 0.6986769669040882,0.5351428957335052,0.021059074276257417,
 0.9379046689264862,0.6196898390716542,0.14417870028594593,0.4207326222730201,
 0.6018470784643621,0.4710271161202103,0.39271764017526956,0.6521929405267608,
 0.7171605468648017,0.39195933479743805,0.40226473328348167,
 0.8965374590349529,0.0013135967220375688,0.15280286228602402,
 0.13969006644246396,0.541436323224215,0.4321057416227658,0.4259287986801671,
 0.16157377167774345,0.4975577089234615,0.8144628211229501,0.7140806337766338,
 0.22491237264705255,0.10570992999196416,0.9969362085991169,
 0.3960507514569025,0.44499535236557053,0.628449844817991,0.3053402450935828,
 0.41683700753875663,0.014522464533232915,0.9026125311866183,
 0.828211383121487,0.8065043897676677,0.300196977647594,0.8376661166667896,
 0.5301325940001028,0.4820664094042397,0.3735441364019839,0.04861682400757139,
 0.8851293772273041,0.3317498784321913,0.14978177093639766,
 0.20756676540361307,0.34380480086310483,0.9705124954166425,
 0.9762896319223421,0.1540461786424916,0.6367106164740937,0.6130309568231149,
 0.1968424326764523,0.6365101472968846,0.1280090566309804,0.585633841744818,
 0.8335799983515213,0.2752964091807478,0.39748065274688626,0.5628254062143433,
 0.6583187771901832,0.23506374720151202,0.691207489674513,0.8964044752916084,
 0.23534892748391922,0.38660505340700213,0.8051055149131351,
 0.17312000740536748,0.1758160586142845,0.9343141186485455,0.8715068646034266,
 0.9716716050101972,0.2631769338234524,0.800124892355804,0.7422971797598809,
 0.0630139161793346,0.06743616440754685,0.10509194921371112,
 0.047523065998457015,0.684763760683909,0.09316850264965326,
 0.8995150512741341,0.05792651352245637,0.098202858702516,0.948058269138523,
 0.35414228866577235,0.05409580114677459,0.32669710017404696,
 0.16551313759646635,0.3030475842912118,0.07031123193390809,
 0.11622046193446045,0.43560222181414465,0.6867146070447291,
 0.5400993076434861,0.13493984304261164,0.44648933978024674,
 0.0748037564612406,0.40481040963283155,0.09180701476667841,
 0.06451414929755894,0.1065633168731831,0.9547997281424442,
 0.01738921550880912,0.7877747120866958,0.23225226337802818,
 0.8226370402940132,0.9884578947909909,0.5806994458325223,0.4693767661734931,
 0.46851198120118054,0.8518849615649706,0.6571164140850676,
 0.18450618744654057,0.7037555542246798,0.04966332112021177,
 0.018477988086824437,0.8389861224320894,0.9301971059502885,
 0.46065477822571244,0.912794487428892,0.5651059129972263,0.6052437248754442,
 0.9432082037552313,0.9239208825986562,0.6684015164017286,0.8792152393843192,
 0.19463954541565154,0.5542612001636975,0.3800349953612734,
 0.03012556068188732,0.7404868062869941,0.7630510413496981,0.7155410848441159,
 0.3724986040569803,0.721376239235662,0.5692529848355916,0.3928771067695741,
 0.4033363195600391,0.7444436858926727,0.4184826284054124,0.7986749229459918,
 0.726021304639803,0.2511916535944042,0.11467310171360356,0.31887463993206744,
 0.8453029479920258,0.7230487520564718,0.8983416643662495,0.9902881391780842,
 0.3403259568850354,0.866329738476625,0.902517132649501,0.09847559707083142,
 0.5085942036325957,0.7423048759186358,0.5464182804990649,0.49401591117271004,
 0.6337057517389194,0.6627267698350093,0.05757736148950621,0.8895180766423251,
 0.40443012302367853,0.9993278482644051,0.027279385618663166,
 0.20874423594703917,0.6122583917061117,0.059243328159754305,
 0.45490211557934335,0.14737254828262825,0.6819524641757184,
 0.7318815329954499,0.13920133383609679,0.8300243107558356,
 0.48194900334437063,0.9534435827594855,0.15885532638125854,
 0.9743221988990789,0.6036044568854579,0.06642472792082421,0.468685787470725,
 0.1547279947488127,0.6828879283720296,0.020393616751656296,
 0.8424751712510689,0.46240611847674384,0.13751889549320495,
 0.6208074815535567,0.7243114096040935,0.9367578873256377,0.11113782623326174,
 0.4052395717292201,0.5704499314455284,0.052137962175764385,
 0.8806399189802234,0.015216041541646685,0.4593305418821223,
 0.0657080383857338,0.20499657585386688,0.578424054485092,0.9811688262915073,
 0.059110329355806956,0.9022635445302547,0.010949799462771637,
 0.7548557065094745,0.7889974012165267,0.0690745808257125,0.36116312868463907,
 0.349631064590775,0.16378107416394627,0.34014115158128466,0.8956400457779523,
 0.5783547424917669,0.10000129611534181,0.012494057311686624,
 0.8347168194360743,0.5906800110438624,0.8434540437698105,0.8961225978305387,
 0.14497740574601958,0.7727167150322538,0.0355126129004828,
 0.19563585081141244,0.40195029415220307,0.38188526293676994,
 0.34957791590636744,0.4868787644616922,0.8151627190631764,0.1368272675654606,
 0.9080673694940228,0.9801504560411491,0.8128651746605573,0.9225769486278245,
 0.9886797953130518,0.5159067417422829,0.46009508610611083,
 0.09393607222670886,0.9031030766805481,0.47110121468533284,
 0.5663029876730656,0.11932971713135632,0.9180432037138807,0.496858996472872,
 0.7329766988652194,0.5369491174506681,0.7727798830812556,0.5644731439372479,
 0.9683327562237147,0.047493652695022925,0.814675539627714,0.4592023884741374,
 0.41609347158438725,0.6526866423293936,0.27339458769162317,
 0.6784029151455094,0.1462561754460957,0.8272233580742507,
 0.017160236205832402,0.2682601547628657,0.19433088274867494,
 0.9636081747581352,0.3296639909234742,0.0716892319353214,0.19102164938086008,
 0.8828227555363716,0.45215795539411374,0.938223869223248,0.9962769379583452,
 0.9101190372865916,0.18100037160483085,0.5570417219771812,0.8602171369942759,
 0.6842496064194983,0.8753241556936385,0.7999012900297287,0.15211590904039973,
 0.589474275896693,0.2054081881230433,0.19297018994432014,0.6421182222732112,
 0.34873933554666803,0.32022851949139397,0.3590772348858705,
 0.2038440932384772,0.3693219804328087,0.9120840508657356,0.7478900791558664,
 0.3618540959416302,0.39543067423918055,0.8975492125682707,0.9644556388428631,
 0.11225670197570392,0.717897349331908,0.7511633818102691,0.20047265436503958,
 0.27995370672831266,0.5375941792403005,0.9398094409668727,0.3641580765850232,
 0.4933825905132867,0.5101942992800897,0.4749001941380314,0.5197967736718615,
 0.14613334344380124,0.742483028779986,0.1404677005295295,0.40775818382175233,
 0.9477513330739644,0.23291656004129724,0.5702644158962394,0.5855629140571897,
 0.6879629486532708,0.7335901755931844,0.9369635702634156,0.5817059339919082,
 0.5267272769846789,0.8144812806417926,0.30720936681238886,0.8640149797804951,
 0.09043353931695886,0.75550165596349,0.14687445198717008,0.4020425994111825,
 0.9045642304246762,0.13643466852369235,0.43819048884488876,
 0.01568716475422416,0.4345836883788431,0.22849600349567067,
 0.11687925058176984,0.5664996480887401,0.6696260144645575,
 0.09873803327360098,0.1941755961992211,0.3821138127742302,0.5897003686000651,
 0.6199031723763659,0.13271716367460928,0.2935531341250953,0.8785614201746358,
 0.678635279619509,0.6064474499178627,0.3428916990175572,0.286390954095879,
 0.01738984121503251,0.5280904849935693,0.05008148316281702,
 0.9564126633563735,0.8070225875065725,0.6130979710635781,0.37172504313105903,
 0.09947029617217618,0.8638680895058262,0.5495958211623301,
 0.41628309663091956,0.5694936796632082,0.3652817004750488,
 0.05127818435895459,0.6787807611880569,0.44319951281462155,
 0.9630342703518179,0.816333029889593,0.17682202865480867,0.8866615885722632,
 0.19868480161110147,0.6688917646524919,0.3466565276220207,
 0.17988809375246095,0.7810040097177513,0.35458373600657767,
 0.17358860567952428,0.040700065669641905,0.06848718059717718,
 0.251515487196586,0.8128439033684156,0.5284783257825414,0.7274268751393833,
 0.7746592841516197,0.8220274109952225,0.06498632667720106,0.9465669703948849,
 0.4848289968188835,0.41360625875340984,0.1928743084392449,0.2508662825645325,
 0.014553935615547409,0.5809150432578931,0.18775634750778092,
 0.8616266202441187,0.12825567996887666,0.4890808922065698,0.8274546802823071,
 0.2001617019966878,0.4902507934164325,0.5234560331989699,0.04282068717546672,
 0.9030017995983411,0.8246426590463165,0.03165999477933534,0.6042221951817682,
 0.5985184105567947,0.9853217150806431,0.6334907061295973,0.09166801611947306,
 0.15369406572087096,0.42149955410519924,0.8975276413613203,
 0.43670894538438276,0.09586334509294914,0.9793561806623217,
 0.2439355493408163,0.7427552429221086,0.7479563883082971,0.48939758001302536,
 0.20494498993631016,0.323496711021645,0.978511995877745,0.023692656810937573,
 0.5868060197234203,0.13348250410763196,0.6845589139633141,
 0.31367014915924574,0.7736148631307854,0.7798550133378181,0.3008222661086387,
 0.18603476642250172,0.27299873937248065,0.33137392231443763,
 0.4044988790314581,0.31950185543815923,0.6754186296414441,0.9701064515026839,
 0.6038576958622364,0.9268730300880469,0.24465546511071257,0.3869289949789496,
 0.9121341100185526,0.7179141590924378,0.714952765323691,0.7572617512892538,
 0.17456029532134176,0.03691373073647619,0.3355318962310656,
 0.20947124360554614,0.7990904312259876,0.996842488527057,0.8856314929344644,
 0.3324264831074282,0.8157707920691203,0.5409150947798587,0.398035263501197,
 0.8056911230320861,0.3877426158482955,0.951551117144185,0.11248075227680909,
 0.7548150849953891,0.3832429208507101,0.5014290003710355,0.33214021485308887,
 0.16061919577609174,0.04152672744784924,0.5373947495279274,0.824992015028472,
 0.1667681662620173,0.6402978379831752,0.5734516736596682,0.10395792983739616,
 0.06574443763330495,0.7089960925551618,0.21995269645171822,
 0.2716384027689964,0.21690978045135312,0.9737571468015854,0.7051998427242108,
 0.43621291855660616,0.5021598063724486,0.7076115328020599,0.4877856839758724,
 0.8393355602941874,0.5397162805961656,0.4682203842707061,0.7878228468961649,
 0.3850107845831572,0.968879616353101,0.12685638570892155,0.7612840265115968,
 0.23440229192461703,0.3424692460826383,0.4938999525038137,0.6455908214519358,
 0.8149164921370035,0.17128619244820986,0.446092639112338,0.8362631113777295,
 0.013404670202000757,0.13328757595489438,0.1833988457116652,
 0.38282178223434027,0.4838662538573252,0.9926818685868289,
 0.18829329013114016,0.505916869600256,0.2060028577634696,0.35410036671615863,
 0.20720358662661575,0.12793893961861258,0.0562289395445813,
 0.7208585980529685,0.436808531164519,0.4093578348608925,0.3668405547844301,
 0.16980840027382005,0.28821328112815225,0.6502177033303096,
 0.3346373552438433,0.7468423044617188,0.1385020450587351,
 0.009492174276197085,0.3499565511442584,0.5552223785307743,0.88424285794315,
 0.9428772427220907,0.31622314961943865,0.943507081731581,0.45593371618245226,
 0.21932693758408506,0.8515668808432244,0.42017765353209025,
 0.4255202369169818,0.03200377217341077,0.5407070887732013,
 0.18357903319431101,0.014428478761025043,0.9366216628197745,
 0.6532363578926509,0.775051466305728,0.8757072600912347,0.041089274530969666,
 0.5292333385246184,0.8671771470901588,0.6326731257410768,0.8139338425626375,
 0.3138973006181497,0.5528132950228621,0.25290216681814304,
 0.20827061375409117,0.48497363812033,0.09088933874990579,0.4082132546161551,
 0.4574220960383377,0.1340370115131796,0.9940710341131529,
 0.026353847077608794,0.9133902558752731,0.7955028536427511,
 0.13169835191970558,0.5190430843339778,0.6765155148909607,
 0.12219601373747258,0.5654590327538023,0.4523441714049692,0.6072486179939102,
 0.893381556322909,0.2907177349940604,0.553382142883314,0.21161444875987012,
 0.21120266633661777,0.5062968770118217,0.5080353482658426,0.1368496249078126,
 0.8331067272824532,0.5218133570349822,0.9271249528450521,0.7323089097945847,
 0.7809957635033509,0.4901506591260153,0.1629315660559656,0.5251139536174937,
 0.39255110356453127,0.9611120772267979,0.8924159923820252,0.6195128087461282,
 0.41301115465287797,0.3686985221718123,0.09621619234173051,
 0.15688883360744832,0.3984852068725311,0.7912613089195506,0.3112940746662002,
 0.9703604130051035,0.9741888281087292,0.5472338379564109,0.5704095370425508,
 0.6926939326385915,0.9294995803990042,0.5907909774549431,0.25483661362631993,
 0.5725034286293464,0.8441585246744809,0.0890831198033778,0.12758033123872503,
 0.7597694808898154,0.036221979107788016,0.9215617518432312,
 0.7930856268044111,0.1760024972353792,0.6892029400309959,0.2372595971908803,
 0.8177322202140674,0.1432409561239345,0.18144962904267892,
 0.40514898174505054,0.2128684171850601,0.25421216804311264,
 0.7492766867046632,0.4346209334403923,0.12711710177412572,
 0.015850561763740223,0.19077946641303767,0.938817637749814,
 0.1647245588459968,0.23573617587489348,0.433987568154504,0.2354064877426323,
 0.9209619827503344,0.8125010194345064,0.060387175439341245,0.630559544797884,
 0.9041534792554035,0.8840230129901544,0.8204203774460856,0.8380719796399629,
 0.019646835763400317,0.6422299412288545,0.42885168264904694,
 0.7971513314698853,0.2649016699642137,0.1933748136908603,0.0704351651192675,
 0.12283633889852974,0.11404229234126406,0.7960387290411891,
 0.3479069524214087,0.8034632435370123,0.42860493347143025,0.0817001750632047,
 0.9657875201867748,0.11352283715405254,0.23237621661046226,
 0.4942213000978487,0.5230463309279408,0.7769243718392684,0.9161442183280674,
 0.8600739655412792,0.5973877445151835,0.27853095260396477,0.6032538046531569,
 0.15899182621809438,0.6618752842721186,0.6296820840559724,
 0.27337491156506166,0.9835698852208868,0.27553438016522225,
 0.4286112224529598,0.15289115156621125,0.480600947336457,0.42456556351359087,
 0.5670533756661379,0.9988495714704264,0.3883454633644753,0.11374485538195256,
 0.029240947546623097,0.727999272839999,0.16284551274095016,
 0.8679317040336025,0.2506302036368383,0.5640547960582907,0.9244253771159995,
 0.46647430546251556,0.34405534477390864,0.9702614722654788,0.870451670253926,
 0.08758044970766488,0.55550264579523,0.3345188584072518,0.6480168615225591,
 0.0949328591736569,0.19704615999409425,0.3383781966216137,0.7099214363806944,
 0.7798776016269577,0.9209565797743808,0.8481606295360239,0.6774876580912857,
 0.3862812870411342,0.4271444118937392,0.6204768982305977,0.22196469273904973,
 0.5928308994744557,0.48009631977210276,0.17243447527209466,
 0.7748370680400033,0.3521878259472365,0.3175067094900228,0.8297932671708442,
 0.8241793547531306,0.21320951513778208,0.46649779769568744,
 0.3229823510356786,0.33442433870072064,0.18778746081015796,
 0.5885178951160435,0.06408471677811389,0.5836168147127951,
 0.026381149433614803,0.8484112661584745,0.23424133505935807,
 0.6176447646310103,0.8943163081680663,0.3016692190206911,0.2234255539947576,
 0.8974859896068584,0.6504207423994391,0.5872348741241725,0.16472342701717257,
 0.9207348670728511,0.8672125695526477,0.1617335393688638,0.03470014819310707,
 0.5709568482295069,0.4503764170438662,0.019657900004976803,
 0.28030391979930314,0.662506119093824,0.47399301415786765,0.6663714897827795,
 0.9416251447138801,0.5908381071533179,0.786855947365739,0.06562034463026112,
 0.14870999716379862,0.18417749710551012,0.9191492515003257,
 0.12312418838414518,0.8050153101453186,0.575219444414626,0.9922894470415885,
 0.1409290341175582,0.5857194970632658,0.6200591476605529,0.5297436829964315,
 0.15986949268287287,0.5400865807615792,0.28490914563768643,
 0.5728927753603447,0.28048650516377893,0.7890764841177561,0.9365896017653504,
 0.0964909135421399,0.6521090085155612,0.17407882202858505,0.5568770605740974,
 0.8539085184202493,0.008878918993014184,0.7435227706502786,
 0.04104555892091599,0.05605707369952673,0.9681162427066506,0.893243530910611,
 0.005552418377211099,0.3933355229673786,0.052127066762009955,
 0.7840496635302301,0.811856962294678,0.3202589389866922,0.6150531930912845,
 0.9881533868678279,0.6003595648457292,0.8786113005488253,0.15453121660205416,
 0.5952794084166617,0.6114823572432377,0.1206602976024399,0.459100115718341,
 0.18229253844814552,0.8480862660110173,0.13488979985375238,
 0.4657348552378213,0.1392481186519915,0.4056712075470237,0.9162395887195698,
 0.3069899493036511,0.005858017976362984,0.2466287874341928,
 0.2678064947034634,0.05230182146900464,0.5308141676550294,0.9442214876886518,
 0.16680382020551365,0.1318402791240948,0.5999640580709746,0.6236343571999097,
 0.125044635308326,0.1949651122251549,0.410842365432746,0.258986972767845,
 0.5836743216355205,0.8445638920593406,0.710937970876655,0.38438617441198375,
 0.4388087183586673,0.5671835678420293,0.6025792616457806,0.35876337193369645,
 0.9696796314660481,0.7846913396486102,0.9080272200080611,0.7720420720676031,
 0.41483031880281895,0.8408068459963788,0.9153090264780339,
 0.15086396225509047,0.6005518181208792,0.7002950744348029,
 0.07674453485448529,0.15699891858885995,0.4560806227108092,
 0.6381687351983144,0.4017105461506155,0.0763815288164379,0.5875695819353364,
 0.8158635775466192,0.27231761895853224,0.6131217595306487,0.8110946382423371,
 0.024322253536464444,0.3996589819947285,0.7490609222984178,
 0.7254264251619398,0.994646636573979,0.7033838806105529,0.17302449070073678,
 0.9466026182977842,0.9452217632570986,0.13643627234494093,0.8001932072229838,
 0.03838052689751026,0.9865578031117338,0.8366157527840052,0.9216651581463519,
 0.6997183786338979,0.06189380874892336,0.18292861905683178,
 0.4777649108809985,0.4320436364675212,0.1552887408929643,0.7656119089478292,
 0.08547134417298707,0.1051899673747877,0.8155685221929185,0.1017501705552527,
 0.20445483193303315,0.3837707162610605,0.16895509940413544,
 0.6236797162329953,0.48315287200776535,0.5715593539859575,
 0.04472186596101835,0.47185060475710483,0.04952860100760503,
 0.19159133921649052,0.26531824966630346,0.20368491369090602,
 0.12927124719285965,0.11159195249080023,0.4032629485420972,
 0.6708850377689005,0.29126837083932866,0.6548689302653464,0.6289675385409029,
 0.9490157089639434,0.3867945004574683,0.04689866409632404,0.6317269324977743,
 0.9503517898261458,0.978226909060235,0.9658402178588094,0.24227710590128282,
 0.4682883706518345,0.26375990660788373,0.540463409401862,0.7093255664225935,
 0.5635919552971229,0.633250940152284,0.5739286426008077,0.4406355783049989,
 0.14709351757713507,0.5259576442632669,0.6325480473020202,0.0037720140828017,
 0.623856845865971,0.04375822943973884,0.19901700667350053,
 0.07042702736610151,0.3907019818429641,0.4646385625524416,0.5568662948070258,
 0.4304408043760485,0.44496144303515306,0.7117549593026459,
 0.05941516225669685,0.1067370515403494,0.09928266142853415,
 0.1025999281052059,0.2173404657435032,0.03646205575227812,0.2619103129670042,
 0.791686560113017,0.21949600131445524,0.4210553823762888,0.17489172277263243,
 0.42451193243212704,0.9958110926186229,0.402455477677072,0.5823350436642734,
 0.0022966658067753443,0.27012562143650976,0.7417655416217493,
 0.4195760032580892,0.8616720667996821,0.6604542012777336,0.6717720712981647,
 0.9196853331334446,0.9430226079724952,0.9405036418709688,0.34222248444907677,
 0.32058694517419206,0.5073835414086139,0.7880685325836726,
 0.32109997908448973,0.8092461172230854,0.29255310203107754,
 0.7582072555325444,0.9525767621869907,0.11504758894244083,0.695919991358074,
 0.8382640698905675,0.4801244536108311,0.7302692899680019,0.3506043271815156,
 0.17129452772004183,0.2425485660001233,0.11277980425036227,0.405230962810459,
 0.8136127872187188,0.35611060356062285,0.16394224625643283,
 0.9415404548874984,0.937682329415528,0.6432929118363754,0.828846931400264,
 0.6547427808175419,0.2950092532757598,0.2793460076234138,0.6217102939321388,
 0.9342979757247732,0.8075975654229456,0.020257010613625615,
 0.9233470296071838,0.8672075967211516,0.907271137996918,0.32651494455062613,
 0.3377913093280617,0.5867210994769476,0.9844824989821046,0.4371786181991886,
 0.8288608242512412,0.5036600029758612,0.8670045623314412,0.09497347764890463,
 0.7511169692172704,0.5114696220246541,0.638340482858345,0.2840001051759091,
 0.2038488914164207,0.25021132990987627,0.8235111834463128,
 0.45909457238074336,0.45688228577381584,0.41855525668282156,
 0.862106203219798,0.4062741882986006,0.23360394806759566,0.6334785312900237,
 0.3575945742813046,0.9869544613266857,0.1727133908512425,0.5011507072976431,
 0.9180832559001076,0.13484986378312092,0.5684952264351314,0.6624189590875095,
 0.10386893353822213,0.03318580884807287,0.1317737632600262,
 0.5380947676940171,0.6384284542511931,0.07666892887139798,0.9130420379916746,
 0.28243182984147663,0.24623559221882174,0.23023256531625336,
 0.4508802355615177,0.4835855201119553,0.9683029845847823,0.22521326317965362,
 0.8223646351937471,0.23564499432062058,0.05973851416354348,
 0.1211243971481617,0.798138965552005,0.31999472750496605,0.6865214330441465,
 0.9813582394222494,0.23005701192414918,0.442509700808125,
 0.029896057417036048,0.6953218287776625,0.15861316360829214,
 0.5394589495687321,0.05338595381732214,0.8445631982448819,0.4135785117560842,
 0.18750528892779217,0.020239928588712153,0.8427071595924196,
 0.8621971175821319,0.7930554112208554,0.3018798053160878,0.1818407639494698,
 0.7550639047038377,0.2727813879991373,0.6525575623581656,0.11708108241860993,
 0.3216149868949518,0.0014953487333765336,0.5281833411035946,
 0.5569718287471858,0.3642412259148363,0.25051328752576085,
 0.32176491803197393,0.007749719536585542,0.6879527316549985,
 0.6304876214539056,0.3671086421988763,0.7458967760283888,0.815361772512551,
 0.22218612348285782,0.005219976980973984,0.18534952744747224,
 0.48412861343175495,0.49563575455201925,0.17043037798489225,
 0.3736112526203358,0.09603226898347728,0.8402865823308386,0.3059957622761321,
 0.44665217860737383,0.9965905626166512,0.15676827045830088,
 0.14292684629041275,0.8754781452617764,0.2791791297741404,0.8676980206596819,
 0.018776145341399042,0.5293607351156191,0.6080930219522968,
 1.491032213635224e-4,0.1729582077228632,0.9011631033101615,
 0.4364293360052992,0.7517822565536003,0.7652735051580002,0.4399424605487545,
 0.9493033761428681,0.6778404446774619,0.11145470930254076,0.5961502352045815,
 0.4970804050060055,0.46675690696752825,0.3468370142533972,
 0.06184616658000952,0.7485222987950836,0.05917618614577924,0.419255802724516,
 0.8820384534752725,0.36129584971192064,0.14227936807853214,
 0.4391055630402424,0.3028752574841227,0.6617207960024957,0.23529201478523187,
 0.004640262013095975,0.1853714341156415,0.2759875670163803,
 0.9205853235197162,0.6142289687899861,0.3597948745056584,0.11799027164012654,
 0.39722900930313665,0.9960583604368739,0.5346852646070079,
 0.24991543672150873,0.4515932483631,0.1320884992396919,0.010044723025723457,
 0.8608800727600642,0.9906978716531747,0.048367731674193615,
 0.13130066223931924,0.34457906857009557,0.7119864838338648,
 0.0019699148844680714,0.075687911307318,0.08635615124468954,
 0.18809116702096007,0.7380847117893968,0.19162840360676459,0.690123426104232,
 0.9438235413286618,0.9695188580389591,0.7334770115871948,0.662062559138793,
 0.41293530951611146,0.5318670447153684,0.5127729681950157,0.8349614292410703,
 0.7488592431135261,0.11098194410008588,0.510270131692563,0.5831404416164736,
 0.5350731042009752,0.8928386530993013,0.7503780137007848,0.05435760979813686,
 0.07236888763935412,0.9285723103128575,0.01348149325591319,
 0.8567833768331861,0.4737203984645959,0.21016563582600523,0.952921340930616,
 0.8116048013234289,0.1277384221055683,0.6253105116609414,0.9918768569900972,
 0.36209054568544274,0.7822089262631757,0.32125084491674494,
 0.8275917653590421,0.589390527117734,0.5175218005638924,0.5949926326568108,
 0.5993697397435812,0.5047427757932739,0.6566635881349695,0.1619303699796769,
 0.47386838825075617,0.6420620280553034,0.2217578051704988,
 0.17147596825457467,0.9275123922975956,0.7092281884447641,0.5736239383707109,
 0.10865826526474287,0.481075535508127,0.40102589482520923,
 0.12383761812751104,0.13267887056762007,0.14520923505867678,
 0.7986770876040487,0.47034557265425314,0.3855638579838908,
 0.35431047182859177,0.5517431793243425,0.15936219974831578,
 0.10898285350880854,0.5433117816201887,0.578600170744048,0.5458047642086399,
 0.7618848118209709,0.7145641877124563,0.03774842960242186,0.7951532420332811,
 0.9925462730847407,0.34689677083102577,0.342942268213539,0.05653066899169046,
 0.40776610171633776,0.898640735964855,0.4746740070863671,0.9427522800491052,
 0.040877980979777906,0.445458855750575,0.6867809119187407,0.2939236548915769,
 0.18338332867008345,0.7945321712026844,0.8873654744071577,
 0.45267866895906206,0.22717387385351917,0.8699498863287857,
 0.22128541533439783,0.936568443851157,0.9364266610310207,0.8802311021945411,
 0.8203188117923609,0.8752187696014062,0.09595130394770635,0.7406436202577343,
 0.9209381759449936,0.28142013557388856,0.9345061327596236,0.6986085498204241,
 0.6580244352239243,0.6869895617206436,0.40712481629787756,
 0.24102688626486046,0.007993918470351469,0.2673215804420557,
 0.6593666256707942,0.9255918794448849,0.31734853128893314,0.7755568752670506,
 0.04385605793799929,0.6394853839753819,0.5884389143153159,0.210030922843215,
 0.19306104557153336,0.6966267639610487,0.05013598823370258,
 0.2843086892495821,0.5050278020495395,0.8099176539598002,0.7448943666380721,
 0.9141592768608134,0.03056438458041466,0.8126726789563492,0.8481557402687022,
 0.8727947328057093,0.0398391310110332,0.9771075921396071,0.8204691493897147,
 0.6002583846688321,0.6236430085700987,0.43032516831565504,
 0.19362510701133984,0.8871813106465142,0.36258348506225635,
 0.4962857329148854,0.10737684342316645,0.42402422481579216,
 0.41728530471552827,0.9136336115126886,0.5091091287872254,0.9692294785310881,
 0.7722184003322907,0.3370140012210886,0.03075107074527772,0.58159249577783,
 0.8626934121799097,0.5961174187697434,0.9220578001880472,0.7109774165129114,
 0.8887087539225198,0.7882154894937986,0.7989097923850704,0.284808078284569,
 0.5884689025493959,0.304565524234954,0.6185760785895831,0.2818021282405003,
 0.9519013041305214,0.44835641542804405,0.18945617860811648,0.762529569479147,
 0.8843252292527746,0.9347467339749123,0.5725293628096615,0.9363180583944637,
 0.16601024990401347,0.38647282037375386,0.3733544299174232,
 0.4168470327743756,0.316958818972055,0.41946011335194855,0.3773044974000045,
 0.7912622365535005,0.8490265103130454,0.3931308882741609,0.5371423887179935,
 0.32588486571416075,0.7723661169026347,0.6081677149441981,0.5432096894725428,
 0.07542906887031564,0.35940085685510315,0.17779567144587827,
 0.6364961735630035,0.16704042467203772,0.9296189599297219,0.7344131945031499,
 0.16794322178897736,0.08158285923495379,0.020976571911884623,
 0.6346224958318831,0.4886926149509475,0.31059463985750413,0.5318445008559554,
 0.3192433595857388,0.8454987379015555,0.4677676785913427,0.40228362683173113,
 0.7775867727493154,0.7859193670017425,0.047695667570570865,
 0.9955168233106672,0.45967975662880445,0.5592566973267276,
 0.27376972788357357,0.9100639189779609,0.6767497306022673,0.8897688255917757,
 0.1451270421303682,0.7984220493577656,0.6478388604427333,0.09540760402146464,
 0.5009851715330638,0.33952998322593286,0.6754212280767904,
 0.18427077194134867,0.7688304731290376,0.34427153199210025,
 0.3732279301697271,0.12747908786810125,0.7433226348383877,0.5923970438095927,
 0.3968173592938067,0.8679025872763279,0.8274879979705805,0.18190989141129532,
 0.08467636427244096,0.30085154743715825,0.2433431003799501,
 0.2593328783107558,0.2985134850055676,0.3719000134576488,0.46198313882719955,
 0.9988908710682094,0.03735818510120226,0.5620206702212643,0.5122589486891672,
 0.4739558445415306,0.3804431281055104,0.6323991795953599,0.3446953262730771,
 0.06064080835456942,0.13649900196320297,0.7009095433041508,
 0.7908421759955245,0.772256042320381,0.454817989516628,0.12255907111224074,
 0.13062547101900523,0.5554867809417428,0.9915301769364696,0.8781901221657545,
 0.1053658774007118,0.9750481916653653,0.9886218032836712,0.28103625606517635,
 0.9354738565049128,0.8079245894856502,0.20674257155812092,0.2920040241073354,
 0.14395335336519088,0.020220251059371197,0.5580020940113536,
 0.8274559298964731,0.1179639926817101,0.40700569950527765,0.7489012780539471,
 0.13998008603081036,0.5938111562533316,0.5240186249681367,0.9805627293517473,
 0.5727104077969813,0.7138181156747821,0.11216371362456667,0.6658081018887035,
 0.8114049771915031,0.532571600814046,0.14603150068561133,0.3900470591789845,
 0.06632713184948469,0.19342910826271376,0.11979331022190509,
 0.4978402532029611,0.41733693583207443,0.9394410998899674,0.5915255786143541,
 0.4218243121185925,0.40769838550659065,0.15329834970535394,
 0.09169280170544714,0.8215213711140461,0.12097094664525598,
 0.43136963605610457,0.09505701475057227,0.5637335118456566,
 0.9426175678058757,0.3422853022705038,0.36939491050887985,0.7764309994297666,
 0.7193796153202751,0.6487954150070333,0.8918337704204505,0.7413636805066884,
 0.04337530634290854,0.06209487566887084,0.41385741843270807,
 0.4717273966328368,0.22672887394915264,0.3170406503450949,
 0.42372156366401614,0.2687999398443617,0.044179526627187604,
 0.19374661320723274,0.2012383198084906,0.9273701553367475,0.9021003676417978,
 0.6437733256295743,0.780804308173793,0.8580467029717644,0.7560791020025296,
 0.6013571046122945,0.7759472920813919,0.2801133420742993,0.30892050724261166,
 0.197325834623575,0.9682817999067227,0.3453963384262867,0.2516468286287501,
 0.07635892332715655,0.22258789498693754,0.08571539622164281,
 0.8177851575771118,0.5585619668122237,0.2389518517267497,0.6844956761678468,
 0.14693513826569293,0.3611904498889489,0.5277964803001257,
 0.062086623797116935,0.6555847816841465,0.9613952239819533,
 0.5695558665406731,0.06227049107283289,0.0713108993351067,
 0.023663719673750583,0.6245686048652251,0.42879580187589783,
 0.07509390525430093,0.48230370502518827,0.20714773393356078,
 0.11502091694687067,0.6240134527712771,0.7112515431212404,0.7802395837359961,
 0.891430040824161,0.8618546829770762,0.05714071318147618,0.2471118554641969,
 0.1674444079062658,0.4276547645981392,0.7752505242022325,0.471239158464708,
 0.2675246596209282,0.40520932680231525,0.5612400960656263,0.57804187544336,
 0.18843817212236613,0.6251910722657676,0.7560725998706015,
 0.09048839924182028,0.35269451914357997,0.47226717351940317,
 0.33313184379313365,0.980122252256847,0.8489235861480411,0.7287462068553536,
 0.18426584408275526,0.7174258709521271,0.6304436669747184,0.1701479433482821,
 0.4613602172226253,0.3952645114886878,0.533837836524288,0.269259414359148,
 0.09959784355102774,0.32004095749548855,0.8663433915427204,
 0.44545375373867346,0.7172711078630708,0.8070947121843235,0.2921744622031801,
 0.4231465088952616,0.1679990180485974,0.4518332456062245,0.32068950636829885,
 0.6648274778142259,0.6405936252474758,0.6268331269686068,0.588871102755855,
 0.733392403104115,0.1140833466975274,0.1097793800218071,0.36648351437608484,
 0.12979147606589514,0.8884315353265457,0.489755716102946,0.8514559162658921,
 0.9043974640106187,0.6450658345155289,0.029397583385115,0.8803514587608181,
 0.7788578779644093,0.3108339649178149,0.7745003901951519,0.4251418134146967,
 0.5783208563475117,0.25870701389391226,0.14069582265486102,
 0.48832920464926255,0.627927926851634,0.9340314357017822,0.4901159832723454,
 0.6283409569686342,0.21125630765041215,0.8028570112895635,0.7359996142529899,
 0.03051776265036854,0.7078606287290542,0.04403036638390678,
 0.4174956867365983,0.283520297411618,0.08879562029987009,0.16298035743622763,
 0.9432304124285984,0.9107400165858547,0.7192349938635584,0.6626722852913044,
 0.17724324192937013,0.7081166991140355,0.7354940715635856,
 0.27245544441178715,0.1877861324158301,0.544556224682214,0.22831940858065214,
 0.9869543692782659,0.8733088771237973,0.24915373655336537,0.7108498951318624,
 0.5809306040515806,0.3042544961911534,0.49970930696097204,0.6630983927009254,
 0.5688366959112391,0.7274848269893607,0.5096938252463497,0.24613392867420048,
 0.17277468983013378,0.5728570040997726,0.3008864594157491,
 0.44795290765855134,0.01323527213351583,0.6879892281719684,
 0.7295144400945289,0.8278373848798312,0.3902360781257359,0.7268609445044822,
 0.8732110986856129,0.03460192702961551,0.5692807803140344,0.6530701596058862,
 0.6035348107253005,0.5556024235949067,0.02405683000238068,
 0.14282924891350945,0.50083657755205,0.6264543903459232,0.06203989304880231,
 0.8082946480345582,0.46666969166770744,0.6067867102021889,0.944425987031283,
 0.019763087075347907,0.3120210423254729,0.8313957365356178,
 0.8150514831502457,0.041605663897504974,0.9517271740496807,0.908864679403211,
 0.8961521472376728,0.050563666838603916,0.6549992708259518,
 0.6108626113759721,0.14220157636180508,0.6415814998768887,0.9395878800583852,
 0.4060896344677074,0.8752428149231151,0.8981198375039811,0.4644317367294045,
 0.41132793556535496,0.08542170582315645,0.3401486738712246,0.967693398620327,
 0.37404226769226845,0.5135537726938847,0.6073080231096717,0.7583721171742464,
 0.556782568933545,0.8272261743040354,0.34664822197220924,0.6923715132935137,
 0.3031870073226639,0.44414251398094917,0.10836829257530012,
 0.5545408881469941,0.4659100434278409,0.11642836973280168,0.9577436973918747,
 0.8772143426359175,0.18188456025503885,0.10696752693279177,
 0.6814347516063611,0.38832135740213214,0.9027814276879972,
 0.17000026998261353,0.31928032314749233,0.1815186881393468,
 0.4695693331961369,0.3491055807045643,0.7871357126476335,0.7491541182647554,
 0.746724824734055,0.9090966448195357,0.7452310169657211,0.667250685225117,
 0.7794442445739789,0.3251477175463715,0.3339476127015142,0.7640451648630696,
 0.01763559734031661,0.8524644115686756,0.607990341135926,0.33534667355180336,
 0.34903256109474656,0.13133546139736474,0.4101436488816935,
 0.7215224900502484,0.5232518768307388,0.04757507905505498,0.997416923765994,
 0.24489066591062503,0.20287398977303794,0.3324024352151458,
 0.6376021622348237,0.7399386698809005,0.2918875329008267,0.8041378205577114,
 0.714988553902479,0.8887485801257389,0.7504639026352788,0.8433986230700797,
 0.6913017482451013,0.403447384133658,0.6562112095237153,0.6201413098343037,
 0.8730286985642239,0.7755368571401896,0.27817014257870365,0.7420798503473163,
 0.3486648661331764,0.6577112342185971,0.47403737079474284,0.6030650644306569,
 0.2854606730232201,0.14711499616865642,5.519428989598785e-4,
 0.32752970849147034,0.004362210810679001,0.4159178381687565,
 0.8181641079812435,0.7111720279366337,0.010504807756715917,
 0.04358266536827471,0.10817259921002242,0.26298144289557124,
 0.2045331071994675,0.19054205330898688,0.8700385138636493,0.9669196646564213,
 0.9612577991155493,0.10362254695466866,0.2089440812342196,
 0.35502374808214887,0.04655907304624085,0.18980286727284135,
 0.19041914366171298,0.5267262833189325,0.12956510269269061,
 0.37115478667127455,0.3751484592187123,0.10934346349147794,
 0.5765665169531244,0.8360520287720566,0.6634004671507587,0.5507608193446075,
 0.9674858826098814,0.4230271395127999,0.4596837295787145,0.8288686164443376,
 0.5224527249553311,0.1448793922386099,0.8396590181314073,0.24042186090973738,
 0.4130246518116787,0.561218424719033,0.0338722516423936,0.38877449913580997,
 0.6201977407827934,0.7384724620003589,0.09300209763460754,0.9844045322498571,
 0.6854070742360843,0.6309327500459705,0.5180432730507994,0.35341089697908223,
 0.2287598786926054,0.5946454952402671,0.031677421203614164,
 0.7723813407548671,0.8233543219984697,0.1932151491938695,0.22478013411318365,
 0.5720701656253881,0.5943957513054839,0.6439616416176411,0.8160090677329634,
 0.4723479016927926,0.038970150501627665,0.5806668600709537,
 0.6051637460550691,0.04952770315098887,0.46129740742742187,
 0.10129020673535472,0.5872292680619242,0.41054918543448826,
 0.9722726808828418,0.46476649403637027,0.6545703967368368,
 0.12201261413502318,0.6464835513509798,0.3604914312617863,0.8402004412181386,
 0.9268229685822363,0.0836696739238878,0.9801219438922207,0.13263975691492225,
 0.03959086049577476,0.17289563440764666,0.28160713254411174,
 0.20703293688467095,0.48161088417863696,0.613385745775435,0.523393564096232,
 0.0494766679890859,0.6646072638496835,0.8920780219439151,
 0.019671796156570553,0.842719725005637,0.8300890627262989,
 0.44598479606197405,0.579331552843338,0.2505698616080555,0.7078590170091659,
 0.36399175194483735,0.5427600682307681,0.5737926307670064,0.6784696865028927,
 0.4843082215636536,0.5705154573440695,0.8149367750538123,
 0.014401363558417923,0.38573327250305667,0.5935343830460181,
 0.18658870971369035,0.5863465227489155,0.6140434993919984,
 0.24556187572135801,0.423148649324267,0.2888887716981311,0.6734077823006654,
 0.49546141409013367,0.05141539150051999,0.09164777531780977,
 0.6500514045820818,0.47588648458790284,0.29872006309472243,
 0.8655445932342074,0.17555935483220897,0.06634572794274163,
 0.5575277358025656,0.24041511854787423,0.06962643002704849,
 0.6046193803913279,0.5203993954366491,0.5594748858685125,0.7774018360821378,
 0.5813498552124237,0.7986429888281028,0.7138269706657381,0.8894175207467164,
 0.2698282792523419,0.6898353058119979,0.044092249685635965,
 0.9213136884716331,0.5536852701575689,0.6661986387688997,0.11790193493868129,
 0.1436623389664452,0.0069593430246200905,0.5657268651287956,
 0.14804707448610743,0.8560400762690382,0.872071289716504,0.700571580370742,
 0.7507818064496667,0.026397334108649306,0.3932089384218801,
 0.5927321505039361,0.048140445779554786,0.2309520162616171,
 0.8318267346953004,0.6436742082579426,0.04392112613596333,
 0.46720095825542607,0.10715612672378161,0.8998817995621049,
 0.2665350851845807,0.9947509867143396,0.31749588568761355,0.5711340618770852,
 0.2347039672690988,0.9907719855655752,0.4432281739866424,0.8570346809291881,
 0.7830042091653415,0.828087741892217,0.3922685251662338,0.6482463036209507,
 0.5084690501113819,0.23796377242388067,0.7781816714068867,
 0.41263180608916705,0.06200869101033335,0.7145129270348223,
 0.02487240559408066,0.5129802470451527,0.22624398597529383,
 0.7329025635732724,0.6430410618312842,0.8786781328553058,0.29539904910218706,
 0.2105008719866035,0.24053121722620863,0.5797102573139343,0.796158441068534,
 0.06267296606587369,0.0038129403105765514,0.06401726635566407,
 0.6744151760202414,0.022548849529479553,0.17576831492129918,
 0.7390315874885562,0.24719192696954284,0.9365558589271936,0.6405120751371339,
 0.6486481838991117,0.6472603404204383,0.19883243743774337,0.1471373845695062,
 0.7064680327824825,0.3623588745582358,0.003900148764726996,
 0.43232652370579394,0.2979607543497804,0.32789471442146745,
 0.5752508124346113,0.8764675267070751,0.6993028502361999,0.24930781807433222,
 0.30649393716940676,0.03290083838023594,0.25316593941981547,
 0.14332558013348806,0.40127768341902104,0.04945119171259804,
 0.6449131593345262,0.2724657777847963,0.9924879465647022,0.8530657249783287,
 0.47361710434666937,0.39474967550992957,0.4472554532050036,
 0.07012521828434581,0.07158060534213861,0.16054803907649162,
 0.1749289024551729,0.2927436360259096,0.8123519054534862,0.26907800394557846,
 0.4532677058776968,0.9599246557781355,0.4505248442429616,0.6311699966848194,
 0.9197480456357793,0.9171260761564584,0.525034381122678,0.2908868913754612,
 0.13943200816546764,0.3701344353408995,0.2528622674401957,0.2752258275425339,
 0.39620983098311646,0.6056958248633024,0.539425277727335,0.8972624550975323,
 0.9221772986075769,0.6950763097521424,0.24395505951508123,
 0.17343146078732607,0.49405707932606324,0.9105581796689967,
 0.024145325488288072,0.2095242267171965,0.11291028190994779,
 0.7292092622590035,0.6013854551653376,0.9577748739004752,0.7975054729841782,
 0.06506337418367802,0.5196980454906392,0.647300245684243,0.1783373320559637,
 0.36131391606077756,0.11322156552160456,0.7431951476660916,
 0.4459489578407063,0.25799616008589776,0.10347901334840559,
 0.5644276705586304,0.055841523754815436,0.40341535375290016,
 0.5418857760434741,0.37882443140138955,0.21803865236311137,
 0.7037694132191478,0.7620828094960592,0.005691522887249945,
 0.8371196523110493,0.7284896638644737,0.07760987839507127,0.4974882656963381,
 0.7229870322613909,0.3069825356861432,0.7097738537756556,0.9125208930279217,
 0.08488080129529252,0.6843832453947527,0.44146644899950016,
 0.8224206626109448,0.13996829831751756,0.27969669008240494,
 0.3227620062233263,0.11946248708135843,0.6022205899451202,0.6427840732030994,
 0.1672210673664144,0.046292656306565094,0.3085053889535133,
 0.9349125874135423,0.70379328856854,0.0809303608842582,0.18208978783223406,
 0.830637903549125,0.22191679524178975,0.5927385983042586,0.5297162516964217,
 0.3775629909595114,0.7154552235890405,0.9141771957966344,0.7007337476606629,
 0.061964664264602676,0.19620561703388606,0.3176272257520478,
 0.5208756964085068,0.2484184370400746,0.08236147285464768,0.2992309491787277,
 0.7783374246333341,0.34294656687111125,0.38341692431231045,
 0.8565597746234914,0.04758324388450985,0.16778107204720139,
 0.6137100097542049,0.424694597301957,0.09565120290887141,0.6102872663908963,
 0.6995995337183469,0.6685741464543993,0.3076082370744915,0.2314432028635326,
 0.4030749650107358,0.731882836224372,0.932249738656138,0.7037402846988691,
 0.5040138673776855,0.8042018811852165,0.14312899957688074,0.4676937293622705,
 0.5817605650683915,0.5105768920782537,0.008810765639099327,
 0.3424360874361888,0.7716588419704093,0.06585082734227465,0.5703224648495224,
 0.5411230994965702,0.41697579376640503,0.6555236026432967,
 0.11414721519982762,0.6783889129771329,0.8582065564811977,
 0.15452862384400046,0.3829893812257291,0.1652222822727658,0.7567869572951799,
 0.92171861229501,0.9048608519986375,0.4080260424024813,0.8545510585735694,
 0.6962896626043199,0.8185494826782488,0.7427755848198104,0.8906561006230954,
 0.7621996651842253,0.2183437837195521,0.9835238093987083,
 0.015689535935107135,0.43584138592249533,0.7013179607566931,
 0.6961601646071572,0.4254280146070477,0.4941136297391241,0.13824883719926162,
 0.7927381137222512,0.43783899136036275,0.7148196985888127,0.557519989092546,
 0.10013966674264274,0.8756591164550319,0.47045493909096736,0.373449234571154,
 0.14536878737324288,0.9439725736320024,0.671903238391816,0.08136643472859417,
 0.6363083851078613,0.7318263130033232,0.6620145559969969,0.2876556512219224,
 0.3797982867118216,0.6486635749596974,0.5688961279604658,0.5101821478082854,
 0.9975768082244777,0.9176199012761307,0.4039856865078808,0.0324801982598486,
 0.3449813762955459,0.13274084383090778,0.803618969645256,0.6878470554351653,
 0.5544921876198129,0.08349565976430884,0.45535703472597644,
 0.25294723056334445,0.7208342583281897,0.2528253648907334,0.3834618758916225,
 0.7746102062644793,0.4933131230985828,0.4169205256904406,0.6108039633422773,
 0.3933675381634416,0.0375748778442081,0.5017232794098248,0.02495642720765634,
 0.0757777335759161,0.761668551278761,0.32251959997161295,0.8291649040316854,
 0.36975068792558985,0.005403263578827255,0.14778523012477118,
 0.40606202707760597,0.2625587416018865,0.6397702748029036,0.611047924919876,
 0.9564473402937325,0.14407855460902097,0.9330138290044604,0.5773296272497911,
 0.35740949209361805,0.5201070264551886,0.3419156264265366,0.8621749549489587,
 0.8226083455345443,0.6467871353272818,0.04431960087245934,0.3140281320130133,
 0.14109161565620065,0.6811696038367056,0.08440959704445528,
 0.1318685969722857,0.3434924027295755,0.2407166271979323,0.31949987338127406,
 0.6508331882321331,0.6959170887206186,0.5097638442652113,0.33571438934631304,
 0.3354379510747003,0.16535425902011447,0.13929629098544394,
 0.5708176907637105,0.3642154515946273,0.2922654472646514,0.832031384095937,
 0.5839391037516387,0.6619124460685516,0.11193645995738666,0.4288534586988926,
 0.19801187029720757,0.20423205116419163,0.9681934964684226,
 0.11293254083844795,0.3277164704549559,0.06808789299339846,
 0.6209761017671365,0.992100827833031,0.4112867205033034,0.06067166910279076,
 0.839855355730169,0.8827977752772896,0.682745214198934,0.3757338689471603,
 0.5675849246758453,0.8628805968526583,0.11993304498130608,
 0.035386470653122615,0.8488349804182522,0.7793047878143678,
 0.6882465218395938,0.7030884486321676,0.22699758907562606,
 0.03909098998104843,0.20819947320831045,0.35828573609073733,
 0.43627900170283307,0.2393706706292097,0.514393085064702,0.09213495000344452,
 0.7279773873995927,0.2907454834361006,0.13235327184585488,0.8276168040985198,
 0.6392725703798364,0.808106828660056,0.2833174389937504,0.5091717757899967,
 0.2766779366782506,0.021428052011293053,0.8334884377230432,
 0.8023402618787155,0.6836041351210244,0.9061250961564121,0.728846561167182,
 0.028182416246948727,0.5390170954899934,0.039090813458992235,
 0.23067883486084173,0.0197674153943721,0.6720308206562426,0.6796489977376727,
 0.26409746969181347,0.9388999270642295,0.7898037886610008,
 0.12356470401760533,0.7623535798878543,0.45368405479359974,
 0.37994349545501505,0.5875282671895167,0.5642373324560734,
 0.04349547990358493,0.5549345634383861,0.479496828813304,0.5217108788150014,
 0.13160994046815844,0.9241888800816674,0.8942692301284116,
 0.10878435969731903,0.6069525446215553,0.051369017159173413,
 0.8295413162512576,0.0260664695103614,0.9217962554810303,0.5438228869413408,
 0.01090701857943377,0.8218222383364051,0.17939344428208193,
 0.7934994698226292,0.7134850313255188,0.2029559935883246,0.6727990424319109,
 0.022121684149936316,0.7199471800094948,0.1168658361537055,0.909121705041106,
 0.8536810616926718,0.64321689931907,0.8205291832250075,0.6874964688468767,
 0.8828160299253083,0.06406712318788921,0.202628833143499,0.6367235742423157,
 0.9142282829493703,0.8374920920097696,0.16862832074546752,
 0.12260122616928593,0.33443333313903845,0.5914587144309416,
 0.8519211490737831,0.5622997560537617,0.3556282395893575,0.20434502611208494,
 0.7123327503173245,0.3997227992794994,0.8757042817839725,0.9541887343947582,
 0.3862500390966669,0.28238604030282743,0.21355186837120432,
 0.9583314455203173,0.1671709872920678,0.3597227203538905,0.7542165113566234,
 0.7943484964884615,0.5472074601029072,0.6238020137015442,0.22257523001648172,
 0.32711036928416837,0.2639584130181539,0.4835236059322724,0.72669953729345,
 0.7712114779704906,0.4198244666563,0.5411514863310926,0.6622915370386138,
 0.12344420773745468,0.60600009685397,0.343386283855006,0.27875784242584445,
 0.389454025364279,0.09076073028859677,0.050003465481233245,
 0.004060175696992685,0.13480955468092914,0.7782545129287566,
 0.5684429118939393,0.9482546908539832,0.6377123301285039,0.2127423816995121,
 0.4383293546612541,0.3795693621981203,0.35547132927270586,0.6830391325203911,
 0.8639703365669591,0.37929522465825394,0.8134772266823485,0.0760165422960547,
 0.17379512583796697,0.044857627189490046,0.25274593282402447,
 0.23897417051849645,0.9319837349955254,0.6240885384318061,0.7057334984191159,
 0.13840238406543748,0.8033218959150523,0.47225288688003964,
 0.4810644677704099,0.3384707257203209,0.4147921094136009,0.1923781343357679,
 0.9041272972229851,0.4808746614433089,0.1171003393727269,0.1573694255854745,
 0.7678326487975571,0.21513935405208362,0.13480835325624885,0.678952435667847,
 0.31487073413276345,0.4235539694780952,0.8768171412924263,0.841593235537108,
 0.3883850283074577,0.945927711611368,0.8575854993853045,0.15606399774858393,
 0.7209339319293329,0.5204548062148623,0.9225086626326323,0.424087389891067,
 0.6291254108699241,0.0253510379085069,0.6083935505018576,0.5633786297876671,
 0.18274359686109864,0.6207355976841287,0.5981172286989762,0.3416506909773793,
 0.5669866988601853,0.7837738675826733,0.5995478275561337,0.701753415307503,
 0.0010142303106823736,0.16239321190833678,0.4212735559989661,
 0.28740954078226966,0.7338788683587807,0.5204008753189373,
 0.44398247545048675,0.3969004178315986,0.38806436317836634,
 0.8265138025001149,0.3152215921781162,0.9374386306968592,0.9612601993960445,
 0.5794671015774967,0.16078770930271025,0.9258596851851132,
 0.042316967256102234,0.6169982007803394,0.5551906732511782,
 0.32449630723909073,0.8232326177734811,0.8945825325152911,0.3862618388887382,
 0.6146880848263824,0.006549655417004807,0.19754302485599906,
 0.30570858685063895,0.6194725834314838,0.13660191732384874,
 0.8462855110550594,0.7133499698156864,0.26020732416452885,0.1546745041899592,
 0.3231037074946852,0.013451632074877096,0.69745300245435,0.865467200143387,
 0.12637312301030146,0.8666738633909506,0.1407173172294396,0.6980881982237203,
 0.2263044554613285,0.9197338974594181,0.7195764284567843,0.08005523422471705,
 0.6663806267201863,0.1771364557045665,0.46090530799513496,0.530977149671978,
 0.7606446084025951,0.2485091878654434,0.36748827859035105,0.6674960427842254,
 0.6635727275044376,0.6383236666584484,0.28956720934610714,0.4112196850461911,
 0.22713936863165796,0.45928397718044933,0.2947531708923179,0.714682588028256,
 0.5975334344004988,0.6169684802804426,0.084815727800859,0.9006662994361281,
 0.9714934661938962,0.6605895664177972,0.27173707662678426,
 0.03844667755636255,0.7956839145898037,0.6592501823289877,0.9589619857700933,
 0.7091568510016529,0.13192365016183039,0.8887175764486613,0.513026642630768,
 0.11151921228815087,0.9295828229052828,0.06279494811879305,
 0.6169689080517997,0.1408953572834537,0.9371085603571051,0.10682507261034213,
 0.45816387088540333,0.13562610307343326,0.5682675869012599,
 0.9113296849302347,0.7141431867598667,0.37741074868897595,
 0.09763939395050192,0.46606111493983726,0.7792028251996715,
 0.2845561066810308,0.6330192370332393,0.3594045511812505,0.5136276588171966,
 0.07605225839708807,0.2785442017979206,0.3159269895735417,0.6457496190427541,
 0.8166889671099817,0.8065886483371401,0.25434889580271625,
 0.12341863269103437,0.04631748436242966,0.08851619506793285,
 0.9650840888368758,0.9586669845800725,0.23767039850798655,0.5290948094217471,
 0.35019455106583697,0.3335055521277439,0.5702695505582712,0.7381442136147482,
 0.12245682067790087,0.09913816437704404,0.14457155235241004,0.54141595056396,
 0.6011142832861204,0.3510062390467946,0.3023894224822653,0.0125654107282962,
 0.1708900523605471,0.16789007691374203,0.06606237704864149,
 0.40704449814175925,0.525213938419146,0.5661663530133607,0.5156155849047941,
 0.5688705947791398,0.7145122637450426,0.9278431936040876,0.5733441325637985,
 0.05191794510124237,0.6467696410790237,0.40430421842049125,
 0.35128541913853106,0.9929442470746848,0.7236193147899352,0.5824519967741718,
 0.3535272687305242,0.18666017132627588,0.18080527950186942,
 0.29042361295423036,0.3937283095480537,0.43848476519200585,0.871280258348468,
 0.3010046503793974,0.1547782617159693,0.8708870006604378,0.5874187996854502,
 0.8477476196771829,0.13072084745670576,0.5231659586508051,0.5254880181048567,
 0.40638169733475193,0.4319657544895741,0.306145824773569,0.20430041202810667,
 0.5947869827170833,0.3082286527206075,0.45168014832791514,0.8812796896967184,
 0.21639801426779937,0.5737788101026692,0.7942794811494508,0.8128294023602562,
 0.9152432200160254,0.8923442924770593,0.4074886172378366,0.4197018340352565,
 0.90120790602624,0.15959453414152314,0.40979258363306603,0.5046138458101972,
 0.3390270492150984,0.834975759667634,0.3079890984431116,0.25466645351764994,
 0.2839565997810857,0.8514882368191414,0.6999585911838574,0.42074884538150625,
 0.6875083248215017,0.921408440509801,0.5078999186854939,0.3275016811235486,
 0.808229784979777,0.9763934439312945,0.11424931989490472,0.5431931164069248,
 0.3586915302944521,0.826540460352064,0.34314100809034864,0.4986287823207517,
 0.09412347134420762,0.9221326673633803,0.400528629166757,0.9633016701041477,
 0.285786152120185,0.4187006705584667,0.5544295769430656,0.7700152418729199,
 0.7410745961512719,0.44145258280104716,0.5909456974701092,0.5115629583513133,
 0.0721784205690631,0.4301799770586272,0.8544444099349962,0.21789541542288504,
 0.8612866051487781,0.19119072592596054,0.30147079997316417,
 0.48590728356591795,0.6893367956399584,0.7822825341791718,
 0.09436600078887958,0.6189355955432031,0.6688435704453918,
 0.22291784155620076,0.1692962345023612,0.6901894194106859,0.5628139058229099,
 0.41857247775530015,0.8633243556626603,0.24029861535014652,
 0.15669659452516727,0.731524293612872,0.5317812013990748,0.9306345198700121,
 0.5550511097572561,0.8112655872638239,0.2429911081597681,0.7560378704590187,
 0.3853551895673901,0.630688010857174,0.7472483739741372,0.09671581315324829,
 0.4774720591521724,0.49944015301304145,0.04662818888139231,
 0.13765989601729478,0.8956600208125888,0.03610849259982618,0.447984949756955,
 0.16160843885110832,0.6152898616763883,0.5760211656126978,0.5856607529487856,
 0.8598656813175943,0.457816362793276,0.878450669957198,0.8794804245316129,
 0.8061129511973111,0.3155980275133411,0.03008122026998339,
 0.25074903715325714,0.3105759671858871,0.9790662251652115,0.2839958134397166,
 0.07859934754898334,0.07819048035512921,0.4241435778829076,
 0.10937364994868792,0.5701586767848258,0.6791398213979092,0.639557252356262,
 0.8635645210354277,0.3167841004978985,0.5930008030083195,0.1628232161279647,
 0.04898620249155994,0.4908126773284198,0.9054493385659347,0.5119046829677539,
 0.184069965831688,0.9195016563783984,0.5485217272079741,0.6479890695390524,
 0.3912489226470308,0.5793739782950127,0.9596332740100213,0.575032357512762,
 0.9738235667792875,0.23537599399603115,0.4204732914689464,0.2952900119402291,
 0.06638391802351917,0.46728499795293166,0.6311116504871044,
 0.7127926363280481,0.5121210897601287,0.4290000321756138,0.6897499264185953,
 0.24709930099484056,0.04037526735953523,0.7124626141967507,
 0.3080582521827825,0.8725246537335685,0.3927257008400462,0.9612606630469335,
 0.06734580106134502,0.6460115456816146,0.8955296851721011,
 0.24343637178003785,0.02096168153766209,0.46378646520740896,
 0.11403312331839721,0.14501061460274078,0.854729212849411,
 0.16112565354351183,0.2741394622659076,0.5088600505320476,
 0.22597157069831808,0.05503286634052951,0.8949755702390313,
 0.6356491787542841,0.1865147440704945,0.1701060707985591,0.5223862072484187,
 0.6338654622100353,0.013976981367465946,0.22969824187568333,
 0.9644413712086407,0.8155624518994692,0.7340198917073151,0.06646844875203173,
 0.045455839489617955,0.2661364920426128,0.6012317763654126,
 0.9912883274625375,0.9547590229063289,0.5912529972502562,0.49609121259680644,
 0.5883676353810753,0.1421530809896262,0.4067143546524271,0.1878442490772434,
 0.4392325736317284,0.8077012313376912,0.2132126958386963,0.6858088359307033,
 0.07654899558061334,0.2426183230249208,0.25556140295724405,
 0.7614358645443587,0.9637136554335983,0.719521969337718,0.4360375715358191,
 0.8784641859636761,0.3322382734186021,0.9141034782201731,0.9337999627428659,
 0.643946226545004,0.7106120882015211,0.785276607274052,0.86743909368483,
 0.6877759883440974,0.5692151335796529,0.04735360209751249,0.9532138901218261,
 0.6751783980079723,0.5846450521288629,0.9382323496582866,0.31210300619744424,
 0.7580604568559797,0.41545920450138396,0.3277365584392522,0.4861819400404215,
 0.31503749289344485,0.27870990417090025,0.2334041326049333,
 0.4030597967517071,0.6118956576792911,0.5086132805252866,0.4620013116477334,
 0.7496054732167089,0.2069215838802677,0.2690897622923133,0.5495320942171323,
 0.036413931575621006,0.9701854503580578,0.7286820853063914,
 0.1703951697031839,0.8304787245561571,0.6035845716650801,0.45841393300998723,
 0.5173876992233526,0.20015579180437426,0.3006003566698925,0.1955672831930868,
 0.47208677602968296,0.502056746039768,0.907990249048714,0.5219636144723978,
 0.3201251083177351,0.6650855993606445,0.4771870856935707,0.0924348945591058,
 0.5681419980591594,0.8133754964686286,0.6009735398223983,0.7416433731385765,
 0.5589557768545961,0.10126107287459007,0.22068963186072443,
 0.8114155807458737,0.1725202806903644,0.5125904446969307,0.8421237128231357,
 0.09562335576784253,0.5866770507701328,0.6664601238866594,0.5166348373070185,
 0.871018494220822,0.7796986282892766,0.7231407069663536,0.9866433793355533,
 0.16437351684141832,0.17045535590848038,0.4781324510173127,
 0.8257820289521083,0.36404918269731656,0.6277938035650354,0.9580244180282723,
 0.9422056922475972,0.7666495610606834,0.8466784029429559,0.588017855293603,
 0.26448740774810675,0.6079154159921217,0.8637166242543066,
 0.23794227838512227,0.7317292079856834,0.930673087941868,0.282779398150687,
 0.36521065977470757,0.4895452474943979,0.5379222634050416,0.5373387541366278,
 0.667904771994208,0.24270473019257943,0.9632661610315958,0.7219904512013895,
 0.19690257403878075,0.14118308162338988,0.12058837130892852,
 0.462385078641367,0.4666036200068231,0.16251417517918498,0.5437341306049587,
 0.5400654356439052,0.262852600115685,0.713394068105947,0.07132296862761167,
 0.37638023679381605,0.1992315594386177,0.5036034735585684,0.4047771151588886,
 0.12153021851896939,0.979117433005669,0.6444293839940511,0.9784587990104966,
 0.5708106521446799,0.6793153570254914,0.24488487827659045,0.8943252828932491,
 0.6099194875086125,0.9753180976801152,0.07192943404822416,
 0.49543187112848164,0.523841177175363,0.20897126050804105,
 0.24434588437840044,0.22442220974482963,0.058582891059568376,
 0.7733457628335626,0.18328418330214724,0.8146902871683814,
 0.03502966872849311,0.7371020624848272,0.04751574170993067,
 0.6851942145025841,0.30364193833449626,0.7429982618052611,0.6337118848894554,
 0.9370711002325282,0.8299679454782937,0.21003004744372689,0.7901154684949361,
 0.8600024869679412,0.4343983431351812,0.5304607396915755,0.2552210394857414,
 0.5897091884421084,0.3888830159515274,0.6243013693720345,0.5338982572661722,
 0.10775510532500909,0.8323077945924258,0.42782994586361434,
 0.26275255672943165,0.34140642667609344,0.5416971157428501,
 0.4689446840350123,0.15698870281767008,0.42997313197061837,
 0.8183583406036725,0.026065299435213407,0.2010808105273566,
 0.44555639537701597,0.4122508783218507,0.7745325110285862,0.6945866933954614,
 0.9304754726086111,0.7975146902569675,0.34873853997196036,
 0.40595795441459614,0.6689929047824477,0.0519762980750722,0.8599405315959562,
 0.6637968176670497,0.7660638705103628,0.42615557402349535,0.9937016710313935,
 0.48411286242652607,0.4050712046399024,0.47246245595017744,
 0.7570839455298647,0.12169215555851076,0.968195053117713,0.553090283615086,
 0.30538666497318134,0.3589433016995689,0.793705644146018,0.27811084905201944,
 0.029025536691834053,0.5723897272077106,0.2939981477644221,
 0.5974865884709125,0.01866371094589514,0.5122332220473725,0.8964073240779686,
 0.516613501177334,0.7496327867268328,0.39663535090561586,0.28270096952926893,
 0.39726063338484563,0.4256371059941598,0.9841377868812121,0.8954925204185699,
 0.9836662404115819,0.5752379594943504,0.4981846353819219,0.03577135918584318,
 0.30919653005590275,0.74122289502954,0.1326736431866532,0.22201882911450987,
 0.38614497814524995,0.12415996448634858,0.6600921765689148,
 0.3905750063066121,0.38461533509544177,0.9560117169488442,0.2217516382125787,
 0.8165927525746766,0.15323942239238442,0.7043238325854557,
 0.39549554297851386,0.674260856855114,0.26805153405653503,0.2975116669771385,
 0.9684752004657028,0.9043814565759675,0.7249966404553182,0.9745917483682907,
 0.0918089854044759,0.3127898637486798,0.9360810458718994,0.2958597667417461,
 0.15277167191775065,0.47358116549806617,0.6760503116387352,
 0.9597492957778786,0.7816548070513168,0.8944170525779014,0.21345772345828462,
 0.16215071234452927,0.5705624771892863,0.8926022040789813,
 0.034713661257890216,0.9830707125893636,0.5671362870010517,
 0.7823262636698958,0.9184353082082303,0.12496257830706314,
 0.015697973331545656,0.27023604144059277,0.4655830952965947,
 0.08459107445245295,0.00459613950703619,0.07756690213323625,
 0.5556307338719002,0.9958978715427499,0.21397847031909456,0.3915505403227262,
 0.8231349326037388,0.21334598425242401,0.6788426806209984,0.9227816550410999,
 0.22250552809924828,0.17896159634577014,0.4214005853678229,
 0.5578025176339284,0.3283930725472617,0.5699274038627393,0.8097325735596151,
 0.3636632742161039,0.32087136870541266,0.06825656877751607,0.77450948452631,
 0.375697741325146,0.012130271588203678,0.35399785581255216,
 0.6292552757567234,0.03539966309118392,0.25934477284606683,
 0.07194450967291588,0.7111416248414877,0.022004226729234966,
 0.8194257485366572,0.25696196673462157,0.4819716614957805,0.4848763343994512,
 0.12172550435657803,0.22578077192443669,0.7964302322294399,
 0.3322377045614102,0.586279809909337,0.22967185319395333,0.3882539933541084,
 0.09869137436911024,0.7409432392400351,0.6374601556695058,0.8127691122572144,
 0.4772432932905748,0.9786329575457683,0.6091051767491467,0.4807636921086421,
 0.5657297441260674,0.7194283238334548,0.7237083163996566,0.9895968239848344,
 0.7304438088629966,0.14299937694226583,0.2374821684408941,0.9201686768695317,
 0.31999825174602103,0.5752416090265441,0.3961448372705826,0.4708263957971315,
 0.49588304120566895,0.9605223854935164,0.2015081362928608,0.4014290527822757,
 0.3916919432683714,0.3537724159028378,0.17284452011865903,
 0.35556341062941754,0.10363080434721328,0.7992491071230392,
 0.3724923079875031,0.07339216435347873,0.5671891399006228,0.2289820421947023,
 0.6793789915904227,0.36437889067587426,0.3845129413864712,0.9308384874096751,
 0.23707392171861597,0.9935932512284341,0.739136616310868,0.17853306435079497,
 0.6196969293045531,0.8106766849691991,0.9066980723082255,0.3420933996681246,
 0.5947989117575967,0.04446696745048084,0.9578975454495765,0.9847289025245768,
 0.0018637472855587145,0.12232187611084422,0.8771006980311847,
 0.2379930030137598,0.06954490660476842,0.5956091892393354,0.9298574258454392,
 0.7254602018022162,0.7845728853092065,0.6462069283907903,0.9618189349180738,
 0.6645651249906633,0.9815705908984091,0.7645562811828214,0.6072789352216914,
 0.6760479179178679,0.8172708331095605,0.6538911919812286,0.3849278541853822,
 0.008066619751008752,0.3858575770945418,0.061464046971347885,
 0.12068561587706061,0.07110601773806369,0.10948874102819173,
 0.14418280371341652,0.21185852803956762,0.9682054859378938,
 0.6522460824794475,0.6429145625378179,0.4348960176590049,
 0.028571665811723435,0.5928766533728644,0.3395591552762339,
 0.7717149371698289,0.8289435436022322,0.3417851866472207,0.14255764527249748,
 0.18349827927353501,0.5389480869994772,0.9497874846915675,0.3375457103643381,
 0.23072467925957285,0.6219067198454789,0.05457300868086945,
 0.03485849055512813,0.13648815512782952,0.5275775392433014,
 0.8409691527766053,0.947969532162493,0.7862083407272273,0.7004346891690894,
 0.21303198653771394,0.5215218444910991,0.695091587611285,0.9002094439699845,
 0.5629084288027015,0.0027017732095021563,0.9097054758062791,
 0.6626028675876666,0.5179924114344673,0.7984261362765551,0.10602877867547,
 0.3170918420928176,0.11839458241182599,0.7245869446140213,0.7101544322085359,
 0.9962818000216123,0.9736683591879368,0.3229353420586434,0.6812153851352323,
 0.05247027453320807,0.8436073925221361,0.21728150466690965,
 0.8150062446947598,0.9991833694607816,0.9251412398329304,0.6175534174628575,
 0.40059974247771857,0.5375266152628868,0.8731924408508251,0.2689597913468039,
 0.28553828746694676,0.31018674827350556,0.1235931584405191,
 0.9542082935717816,0.48393547773737045,0.8282777334445036,
 0.34626661063615816,0.7410845738389439,0.13785401380251838,
 0.3490796845126197,0.4186780132007305,0.5095346162474922,0.7153041000957613,
 0.6736848815481,0.2932612479484593,0.39766320261631427,0.5785922418177919,
 0.1957763543138864,0.8971352442300824,0.28985362286571437,0.6999925211832523,
 0.09161423667001367,0.5602337095222302,0.9201149267186733,0.4773908946693697,
 0.5028790908898371,0.8787146648100974,0.48413285733044154,0.8638485169127236,
 0.4911409951731769,0.3065941090445624,0.0019038960024192475,
 0.3264169923128123,0.35225854915258237,0.7758896686771756,
 0.09622171197400697,0.8715502336490013,0.6333928821463773,0.7934731975560879,
 0.6633787504711344,0.5706269172005344,0.9299300688474914,0.2955902970017654,
 0.8745138678570866,0.4755719558339413,0.42478167665077493,0.6334013896425894,
 0.7217411450085277,0.796175158136189,0.5078764001285287,0.47137397839634854,
 0.6690478344900538,0.23570466732128037,0.6089989544378409,0.7886349734106717,
 0.4507735784698944,0.37061508490469597,0.09827140473955787,
 0.04844326623047435,0.9317418249600605,0.9004864567727846,
 0.06833031433822712,0.35382562536739304,0.770796347336283,0.5327716815652412,
 0.050604798367032244,0.5255691881742416,0.7377881730896327,0.907798307083966,
 0.7278818472372052,0.45467930837222603,0.4212144615802331,0.7868134367676193,
 0.06583362409106164,0.012585186805164184,0.12447712342562933,
 0.5095544214311819,0.9076967190172354,0.2063982268469412,0.52131473771881,
 0.387021890516306,0.7332100415303661,0.8159790123785267,0.8270154974482838,
 0.3758606845994119,0.2609477851171592,0.4138893720579946,0.12993666172509244,
 0.661863960216448,0.3704073838262212,0.7024788020453364,0.4145086254474253,
 0.7708005064837278,0.948356357559924,0.9982171803439669,0.2624748913114435,
 0.41829892799709034,0.8963662686838562,0.4265788101526701,0.3360383041636329,
 0.9610923152581708,0.9207925471489384,0.93489996543315,0.31079251658341367,
 0.15547609968312026,0.5585149801881659,0.12096462339108371,
 0.7969622222129571,0.16373353914960376,0.9033017308361632,0.7440605136194925,
 0.5642941894419782,0.8326174821631573,0.9007758447547883,0.15206554822160712,
 0.4302981643457946,0.23060449149932905,0.8934026101538113,0.6138711417755975,
 0.8108120497446554,0.1621020519844364,0.2808101323330199,0.4523770725653069,
 0.8669378856123017,0.7179745960249987,0.33829010590658526,0.979535765181871,
 0.9248607915005638,0.8101266595299609,0.8723122636960623,0.17232073793346614,
 0.4378998677236655,0.619784582698692,0.762286808252423,0.3051300174551196,
 0.7677473677385297,0.5422967409588623,0.13569315241380842,0.9656433028339992,
 0.34919667093306817,0.2556787953801476,0.8376063881186917,
 0.19208605017754532,0.8878734463643843,0.7977634563153981,
 0.016440130279682053,0.00324734261934978,0.8644938988978952,
 0.7289754072975589,0.38816796269054465,0.12716821097206776,
 0.18582552506438377,0.39891777418394436,0.0922705174332138,
 0.3120559239711128,0.22422265370607064,0.5224987727826264,0.6464615682913977,
 0.020528178666423202,0.27967949365738953,0.14149261779096223,
 0.8203234059154016,0.7807675083780155,0.1046921846237816,0.8615542372732504,
 0.25941508091120946,0.6209677578665812,0.12451903588956359,
 0.5632566025045727,0.2573460245517645,0.22398227367003443,
 0.14932149648523785,0.5015427248735203,0.6771918502113523,
 0.023161428463427725,0.45957522281003604,0.8348753785743676,
 0.26898810396574424,0.9482154110582516,0.24239587837894128,
 0.5891238352523213,0.9144784491925658,0.6385463328491114,0.926068371392863,
 0.5193943661198228,0.3671040425848582,0.41059756573280615,0.8899668240623156,
 0.8780472111385548,0.9282286453128916,0.8778406218824041,0.25321954111513123,
 0.9091634655833509,0.11003107322876282,0.9842738855892464,
 0.24908802041316747,0.6882850217967672,0.4157947435123448,
 0.19991024865269758,0.731777996783208,0.7756064816440613,0.2638997095740394,
 0.425340230938387,0.8100378839634668,0.9312335512781047,0.0906495644310018,
 0.07956205161543628,0.22093087123891353,0.6393023481548492,
 0.1565361716924819,0.757133415822175,0.7400627179118449,0.2193127606885975,
 0.1922583387674004,0.06696241056271612,0.27081975459927077,
 0.2692799057228692,0.3238237021881629,0.9029098428600529,0.37747970731009206,
 0.4053185047319806,0.8816068096453032,0.15948205949825434,0.6895481221854594,
 0.8715553876886297,0.008645381956392084,0.9553854975359104,
 0.9514672388622563,0.33144146725369494,0.7427734588742287,0.7896351827974991,
 0.5397765970087043,0.3839067638296313,0.4385649970029236,0.34748103029398947,
 0.8896183005528717,0.3769949527994605,0.41651117529330517,0.75293201605854,
 0.12413361468601103,0.8656650113638584,0.7325771264714003,0.3030510360359693,
 0.37109801174143686,0.18100736929867278,0.07464723537913598,
 0.3831071816745163,0.2214135678502014,0.7670018072881755,0.23110079107212456,
 0.4288453556380698,0.9589503324414735,0.943787332407288,0.8016332279124276,
 0.974716190417938,0.8280325735642846,0.44251627469295607,0.3295699891436742,
 0.4260254751615027,0.4977037379796543,0.3747970410051985,0.8649297089703859,
 0.6709367895665829,0.7771765043121954,0.2429629926560406,0.7463346450523073,
 0.1966394593519033,0.7532520547830405,0.10947072111513234,0.2048892589435286,
 0.7417357462135608,0.21843343321462672,0.4120054752329373,0.7656424300413422,
 0.8165704710447049,0.7286250993333068,0.8773107768805705,0.12839890512165275,
 0.5915093588276831,0.7577091228506201,0.7091132718162092,0.129183266790996,
 0.20578763883682472,0.22606600408054223,0.22603583154245777,
 0.21759732678914168,0.749210521750258,0.47578932522751427,
 0.49274724938034753,0.791803330517369,0.6096345617927219,0.4939368410797219,
 0.5431184256392493,0.09949137914154982,0.7247007099085541,
 0.41635318254354703,0.41546662018770397,0.511524305632617,0.4884348432245249,
 0.8476066784776184,0.21293426785730296,0.14241725901224156,
 0.14925608557867842,0.16657374198886388,0.3163150849140015,
 0.5892460656598533,0.1993674751598058,0.9174191611561858,0.9726393951600647,
 0.39536983500657974,0.9615291782982147,0.19401417744816596,
 0.6825794026896577,0.5290819153173025,0.6392656542343762,0.02089564160145818,
 0.7036589922303917,0.4224523101691371,0.6428074188798285,0.7030317927462352,
 0.526522078528799,0.9151766605693208,0.5112249358745782,0.3137457047349298,
 0.018409442287083744,0.6130736397990502,0.5700567374062849,
 0.8867226343134814,0.08031351474143122,0.4912586947749974,0.7900008282108826,
 0.08604020019513958,0.6350053282732352,0.9930633654414665,0.8891162708636231,
 0.12967305893178638,0.7321409443727003,0.16355408010074868,
 0.18255003813841642,0.5702411217011755,0.7939675995075117,0.5708502719347998,
 0.5959642178790441,0.3258212679927919,0.4199429999561688,0.67610002369176,
 0.769621944837806,0.39077615498010365,0.32533069624561617,0.7274684083827377,
 0.3026976484156162,0.7025098006169863,0.013848946431917475,0.369895085084892,
 0.3434620981795067,0.5833208423867775,0.6053536788383858,0.16624399024063752,
 0.7222865307699904,0.5779986482560348,0.33441695032016516,0.9817215692624875,
 0.06663422637988603,0.846928611405078,0.9655790165960788,0.7360310369074187,
 0.11311126663620974,0.8855139778113239,0.6641435512484143,0.9542471106429453,
 0.12768601785854872,0.9645880527664165,0.03213977902351117,
 0.5741949620784216,0.3722086926096443,0.9512118294991446,0.3618276844824989,
 0.3993769393294673,0.16300497398692582,0.9834096511043736,
 0.25579287884135804,0.19444867945490218,0.7312860787605884,
 0.9327482185684862,0.6468864376107157,0.8802532394204504,0.6978556368807115,
 0.9163103941474939,0.18993559618974543,0.6880206671960379,0.878448191328871,
 0.7685294861555086,0.7867254365580534,0.1209776855639556,0.5744370919838824,
 0.4083550012127334,0.683817161591074,0.098301743055625,0.18513342696161839,
 0.6988133537459856,0.6691188624143204,0.25310770551607586,
 0.12673348960405562,0.09416026083140216,0.4302853965258332,
 0.2617069293946661,0.46100186447752334,0.4795070959567148,
 0.13768639824794882,0.5624723108924689,0.8600014902255548,0.8676152594189225,
 0.6503889726347004,0.7410911291772895,0.828663981374852,0.45958974600163083,
 0.17272930713844703,0.20731293552251517,0.7033289499941925,
 0.5671323157748996,0.09731802689096292,0.22520543064909526,0.7233985393332,
 0.3766453019075562,0.5432379585764375,0.9585053116438785,0.917196039490823,
 0.9367927320278675,0.7722557110257398,0.8314094907132817,0.11635481879981247,
 0.3141065300393425,0.6838414459463844,0.9351881912671638,0.6802053117376722,
 0.7868768765754522,0.22419816873922382,0.09788807945760014,
 0.7853986572095509,0.01945086132845575,0.8581934422174843,
 0.49081401855205464,0.4319449037438332,0.13198016705897475,
 0.4761475248282252,0.6598551401218833,0.508796419625228,0.2363080440406472,
 0.5044118543719133,0.6104250791393125,0.8256291868765981,0.6151559474401653,
 0.6150286668764547,0.9557214259141159,0.19316085281958795,0.8839897437859877,
 0.4354558883225397,0.6063842186969527,0.28278559816470494,0.744585666387547,
 0.7376393286099047,0.4165578276673382,0.15534966465984046,0.5362555215060147,
 0.008232861201640906,0.04251633346344719,0.7982711269274004,
 0.8401805987517351,0.8887435418412543,0.005432759179986646,
 0.4033880363400766,0.5943313228886549,0.5513840979192113,0.7780041323007245,
 0.8246296768356596,0.7119511757433942,0.9794310823761527,0.665003911226465,
 0.21153539760985351,0.6854990696414444,0.5606267234714841,0.8782490618451916,
 0.2313152272451393,0.45613670991602784,0.17768097626917856,
 0.1183459811000247,0.38512118935374673,0.016165397301647455,
 0.8966218972745361,0.431376526705872,0.31434724252595203,0.5127241178780206,
 0.11160506182634822,0.5932009728547653,0.4263515158041663,0.789226126795752,
 0.6098420690057924,0.7539878664605364,0.9358744584454043,0.5956389264767644,
 0.04899186080447304,0.7335814661410783,0.02085831169429697,
 0.8655650374976059,0.8046194547137944,0.5974612973957462,0.38939916156676413,
 0.08080726487861334,0.002265420421551645,0.507810592146997,
 0.20756941181020494,0.543045140350906,0.2586867226177624,0.3485515064940309,
 0.2823959869922894,0.8268978752829107,0.32864393555701477,0.0991895886225117,
 0.5626440943292091,0.1385355238530861,0.2909627895323025,0.5925228065741048,
 0.4731845091576805,0.46898033295280384,0.8931559173364816,0.8402487459203891,
 0.725049329862455,0.2028166811657226,0.9507439769001286,0.5473807667980911,
 0.1058742984047003,0.050851742584079984,0.20736027116864042,
 0.044860331635137585,0.8208410230414576,0.12057665756930613,
 0.27479436150976433,0.8782478806548858,0.06723771701448389,
 0.44499599680527757,0.6837094034703812,0.6137114364554656,0.6213648203466134,
 0.3011451273245911,0.7664652889569942,0.7428980057127833,
 0.037741872615495775,0.014596202331983221,0.6052548523897423,
 0.0018867271371501015,0.499657545422892,0.5037000135382826,
 0.5304989593417014,0.8751795881493284,0.8250753735257899,0.8956084813843068,
 0.7319572486212884,0.7226679119597653,0.6175641095040124,0.02085930332342023,
 0.7363405395075922,0.299288751360243,0.8866192381484265,0.8319244537578805,
 0.3908295046537631,0.8987018888883299,0.8521843571544225,0.9473007391710611,
 0.519134230076864,0.07197314754196782,0.654917390901103,0.80032305093159,
 0.2459807272329293,0.738891187901773,0.7989430999629688,0.6680589268580738,
 0.4832785786767986,0.6594608891279803,0.8124207857645838,0.15091965270473806,
 0.08206017228742735,0.5901618194895246,0.6606780253535088,0.2092991222531535,
 0.3631584693975164,0.915809374423524,0.12784240263366142,0.6914397118090755,
 0.6091426777621537,0.46559339194212657,0.9897556240473603,0.8045135197495846,
 0.8021078005420594,0.4160667201278898,0.669946801234534,0.14464796661645873,
 0.8819132526456266,0.2806211210319127,0.23926463010868537,
 0.024897541068081286,0.1954968910132272,0.7120486910487097,
 0.8274942602414223,0.5083273640294438,0.08523133660004789,0.2776218309259867,
 0.22085071958656455,0.21051284960882832,0.21228024020956815,
 0.8718640081847012,0.8544297234976914,0.9283537404061981,0.7644368732733956,
 0.7851751296955484,0.05687836583423378,0.7396192527651329,0.8419043232323021,
 0.5812449876711506,0.7115577624038618,0.18043011321217517,0.7575840681683832,
 0.2360650214374439,0.7207687662929042,0.01047473049589942,0.2270768911718679,
 0.5023759050957389,0.20170024549250032,0.09776243578657684,
 0.22454642483380882,0.9702491380417304,0.7050887659289338,0.4783696768248801,
 0.8337606530816368,0.2699246892339717,0.09024776833578074,0.9449143594395742,
 0.28586399828493936,0.3939469271339323,0.31998844218279765,
 0.6668534028391164,0.5027783166359745,0.4279038107537082,0.7610232767660726,
 0.7885777022223558,0.3461842814310383,0.2716302011056142,0.19022232155808128,
 0.7237596574352899,0.8971532547422132,0.5580429113074666,0.3588061061980721,
 0.34821299712372167,0.6928417808817946,0.577510754760343,0.16952324329785706,
 0.4374715478070651,0.5863101167346358,0.6736816603879197,0.22361670549272306,
 0.6830590995629117,0.3444872742770976,0.6159783603165133,0.9277558287070293,
 0.2979215078712034,0.39112988009795024,0.6382508596954195,0.355298803414443,
 0.18537307963977268,0.5257223205271466,0.32980553419634195,
 0.20222873339147607,0.05329783508364694,0.7895760841290088,
 0.7955297173655982,0.020797354889896646,0.5563502093602986,
 0.44030780361900157,0.1457277730927551,0.40234086340184394,
 0.35641283815131064,0.5334647915814114,0.4700771415563787,
 0.06538650632461507,0.8080956014573722,0.13226402065586162,
 0.5669161181356273,0.07730773893767429,0.798694914713672,0.9075181524406235,
 0.4495189544430246,0.26755874851103123,0.8994845048856972,0.5542074808335598,
 0.41080058322258994,0.6326795138230319,0.15435350573078555,
 0.6950918012084953,0.8150352922497004,0.15087840546790077,
 0.37137490750912194,0.8562893743159401,0.5040749375242426,
 0.26809376688304387,0.7010432095603241,0.9314103107764711,
 0.06528842299328252,0.837455216774678,0.7765107539442831,0.49058970409681457,
 0.5825878910151725,0.8417222100208873,0.9748893345053871,0.42568203756913636,
 0.8857585703419599,0.6104842026357001,0.5229504730675518,0.7766495104200098,
 0.8902364419770404,0.5565787972850141,0.18789872838343336,0.8028814209420813,
 0.9205377511269873,0.7777995989573075,0.7866845626296625,0.4124677495850262,
 0.4573921920482311,0.5351705530477223,0.8109939891223434,0.6027989538537273,
 0.6706916330375526,0.09712046265064633,0.49620255330325913,
 0.23658045971420816,0.7461694049420026,0.6759615090351732,0.6392287821924324,
 0.46002922192355755,0.32433606211254506,0.9958208233463052,
 0.7638756052322104,0.9720694368112965,0.024699119065851316,
 0.07575089490577913,0.04440673032496578,0.06553406279521512,
 0.7820686762275157,0.3908954379088122,0.36661093308271087,
 0.20177946812269865,0.53340959039516,0.6280200654697461,0.03539414707050714,
 0.5382589754192764,0.24326624641850725,0.858987472795341,0.9853191057410059,
 0.0028654980822747644,0.7767149466781842,0.559485561249703,
 0.7787604520856505,0.7271057744477973,0.9528944010006126,0.19659559909387692,
 0.019156775287522687,0.5311740214949545,0.2862847626654895,
 0.6542050523275188,0.9214773471635784,0.6030156539481263,0.39288015068233584,
 0.12874530723642796,0.31396818661309744,0.8997889822654725,
 0.1878478313642986,0.6513866524432111,0.403152301492659,0.322471900950533,
 0.5948016629597035,0.8137168038848877,0.03428848561550568,0.8098388497192238,
 0.3225932443402897,0.2569460060373463,0.49551852022214615,
 0.20051213359476172,0.8824142482988212,0.7835796072221323,0.9005552852699538,
 0.61270393779533,0.2627355569901716,0.9452176666661866,0.27075842982619835,
 0.037591255902248655,0.3410743902465958,0.02674694609022854,
 0.3190584337801451,0.33139409246324025,0.560704143407642,0.20116237279532168,
 0.2129032914003356,0.5193908362955649,0.5425243870756383,0.8661695849090366,
 0.7354180199643896,0.7480972064175214,0.5956916729533941,0.3472619808039893,
 0.41422430341195726,0.6364375063154326,0.30293443557047006,
 0.3700427534081786,0.13788716966469083,0.2978976224314638,0.6136324093764627,
 0.254692286135463,0.2251762155551731,0.6170918740639824,0.5131593011060156,
 0.35307011314606973,0.10863116897980796,0.10939521705957056,
 0.07628560756887237,0.5026625968870099,0.016222412040237666,
 0.052460872609022324,0.15221299967760404,0.3861638338219351,
 0.5054598259077292,0.9641309866072866,0.6305597204775607,0.5852256451515812,
 0.09301096562849365,0.10586343853171687,0.004409112412238181,
 0.12033491715251465,0.35851876701832464,0.4542631496135484,
 0.05476385832257469,0.03674750793140891,0.01794146549925324,
 0.1827608921668057,0.1757110592203539,0.7398575615568258,0.8577195703630185,
 0.87658546315813,0.8193435158822804,0.6176914405548273,0.9772885953881942,
 0.9012440953457368,0.7698142029883978,0.273768529322185,0.3684042145057347,
 0.6158194803421315,0.029215974857266902,0.6520560626219302,
 0.6910188535979701,0.919707861946972,0.3898375050531082,0.7737554317632094,
 0.10237254419071817,0.9618005008439655,0.4998907258964652,0.6532504680412454,
 0.16520635204092293,0.5235217266634229,0.6560149701233323,
 0.019056860408892984,0.04560268685435043,0.8222071137197292,
 0.21914454707276632,0.0646496616173814,0.7319031114651776,
 0.19990089991641047,0.7225106344229046,0.5415976898517687,0.2909685509510441,
 0.6972079921625417,0.1510456588265099,0.6831826229819051,0.6702582182946046,
 0.37269924018559153,0.5935985776834098,0.4516626877527048,0.4831213611517552,
 0.34398859054827424,0.08119040602860461,0.0898832002541502,0.301848231644947,
 0.49087820062318466,0.3113778205403017,0.005699087477656017,
 0.5170078753477843,0.7540243866653552,0.3642070473983863,0.9709052019136353,
 0.8314330330209568,0.305830117134948,0.911695096715274,0.48379813109236547,
 0.13686907635107615,0.25294338385882054,0.7211931469880855,
 0.1523714146450731,0.7865573080601752,0.2400360540058588,0.872856670602294,
 0.047229643027630774,0.2907137458481982,0.18327665610343946,
 0.2697346286982877,0.9710997936596688,0.35202559088775853,
 0.05611928094890495,0.47970927993824275,0.5431533025241406,
 0.7819078191956853,0.33810567913648004,0.2584921200978928,0.039372371644536,
 0.9409733262970031,0.8432309787861382,0.57379896264697,0.22481259425699884,
 0.4587096325513508,0.5960505484671526,0.9598122574238868,0.04750768322655463,
 0.2554932849074921,0.4393087608042009,0.7025686774684554,0.8990579885785979,
 0.43429567175247097,0.699612250140291,0.014229880367177206,
 0.2126684953248138,0.22069057372496426,0.6670203143778927,
 0.44094074539960526,0.9627735370031733,0.7258626745628998,
 0.20002250457161275,0.1699692929913702,0.3836910364919526,0.7597962609955525,
 0.26668038741660627,0.5496815648138718,0.3230677035046332,0.776112968100269,
 0.767129872924474,0.9544349132844813,0.014563758363480561,
 0.49665422066548304,0.15994991238277456,0.47078706036336737,
 0.5581792403229977,0.98213361755414,0.07070921615575843,0.47783045328538,
 0.9021720510031819,0.6702710759964184,0.17494530230644656,
 0.43852381406294527,0.720142373770159,0.7249384110406009,0.6264013301074582,
 0.10251125061028943,0.4682079025383564,0.5865502147848562,0.9679159325945997,
 0.08632024419796835,0.15343034095657881,0.8237325743262597,
 0.29552647812032684,0.14640061846461405,0.8776743348839472,
 0.1874518486252985,0.1053808944317719,0.9480550897890299,0.7815260163282683,
 0.7935372188574461,0.826215462584747,0.3423780441064217,0.6666084261340023,
 0.9076076117985563,0.2461960239619143,0.655357752639999,0.3840589221513624,
 0.740579316317649,0.8595701054096048,0.41450586738075923,0.4651870032182803,
 0.0700526471248476,0.6827660008834415,0.09001373997547302,0.9747842706685821,
 0.6329989155865388,0.6797285157021163,0.30919250844658386,0.5951581595826536,
 0.624072381272293,0.1540154678010297,0.14037731272106413,0.8322411392923259,
 0.7307478780522296,0.20214385812891367,0.7219020291723128,0.7214948499846625,
 0.3583176645938888,0.9478318615030538,0.4561302726113494,0.5430026111978971,
 0.2521334855867772,0.6760942935598218,0.8036901924614848,0.07534098646582854,
 0.5532751792150701,0.48566021580998786,0.9069600564309096,0.6601896305503921,
 0.1712898751850579,0.22086961810268257,0.8454699082362745,0.8551978580942698,
 0.7567745125504698,0.06157376829054162,0.031748781093823175,
 0.17066801966901135,0.4782275048645359,0.24054204011286118,
 0.06671935148584451,0.9333887781912826,0.9713867620346303,0.0939260756190532,
 0.8803569655789178,0.7559822699353995,0.7034758477360341,0.5211648524870333,
 0.8878557208375012,0.4806018302477443,0.5057706552744069,0.9691945633343293,
 0.6771218030592653,0.8697149299004129,0.17674257771823987,0.5813658888263853,
 0.8263122088841772,0.28727781200078295,0.9280440808572287,0.9856143106631884,
 0.0657128762335718,0.9069277866018839,0.07346315948122184,0.7805769070291337,
 0.12167920914231078,0.3139826015392265,0.7599951088010894,0.7273087431397683,
 0.5591573649110276,0.15659977698023497,0.20392745613333574,
 0.9080512382052257,0.10543082952588556,0.502730457986274,0.9314372675121567,
 0.6175539085221046,0.7197627727819788,0.9201268945620612,0.3975813915535824,
 0.8147980746818312,0.2829184086424614,0.5529141531633104,0.04630437702205903,
 0.33133231127449514,0.11970158435054767,0.740183005550047,0.0888100974273286,
 0.9669048717272479,0.9031168404403447,0.6221460931699749,0.9527578809703086,
 0.33254338170138464,0.9340249154969139,0.38772491928867336,
 0.37647533694918045,0.2855269675097625,0.16062098577375372,
 0.023183521957484854,0.952045434817116,0.6298754516354323,0.4497366343381156,
 0.08407703020065738,0.5138866723767171,0.300711094279841,
 0.011237398964722534,0.4854803093065475,0.9603697449156863,
 0.9534776066896024,0.46975990385544697,0.05668122973801648,
 0.48594055638270817,0.530793866466867,0.08841112331696921,0.5057569482886467,
 0.022980248208625764,0.6027203594022995,0.04133117212034798,
 0.36909137366787603,0.8404050546099369,0.21526615189095366,
 0.8123184900678548,0.704995893154075,0.2414911650766578,0.6673206355249838,
 0.1949408027506272,0.21536693770424997,0.4512481197745618,
 0.12060835785176949,0.9150471894088581,0.5843011800644845,0.7660080726810787,
 0.5480701877279057,0.5507287146088309,0.5751371576428,0.6551728949999632,
 0.15443613731221872,0.6817532516346674,0.16753631452459095,
 0.5416135220794543,0.05286117673823021,0.7730054320762659,0.3236865134978588,
 0.010492433094538067,0.5070847215546522,0.6106631361949066,
 0.3324619575865654,0.1285967418965629,0.20472425396158733,0.8565694001626317,
 0.04781591289497211,0.7251372610614694,0.7927085887690414,0.8843007026235084,
 0.22033051681326876,0.556275777716715,0.4206613462147427,0.1224922753511718,
 0.23293759521245494,0.7417021978296202,0.1129935153596705,
 0.12122676444288538,0.5071045409035697,0.6782898162181371,0.7167460443197895,
 0.44527271485276143,0.7317827547259492,0.6901487257618109,
 0.15173930319652795,0.45940388125717413,0.43878150787876025,0.30715863561042,
 0.45979865775321194,0.16558404642476243,0.1098738870765894,
 0.12018002079277379,0.346500605969154,0.3051199582549933,0.9949708646763316,
 0.7000550820778382,0.3044336996596042,0.1843799199543814,0.21601818847178023,
 0.30018010023250297,0.38754927685285556,0.5593294534509484,
 0.32618828474164685,0.7974718616973666,0.4236031163533436,0.9867921836367798,
 0.5551641959769241,0.8894238280244238,0.3596129200111151,0.220793068390029,
 0.17420577511191737,0.1997623764211346,0.28349098123984084,
 0.31324610433997846,0.4182373355513076,0.024355109427169563,
 0.8838390999488916,0.2639167554016719,0.8852060706504097,0.6532737093047034,
 0.9455143538945251,0.2658187454021945,0.20916462212524312,0.7546602157691705,
 0.7347657014085781,0.5061021667961304,0.9669612725694678,
 0.026348680219639653,0.9668502015378562,0.8774254649902758,
 0.3245733931775229,0.9864407850019684,0.23000070859036104,0.0719100832222278,
 0.8919504152810426,0.14025176445858767,0.8643260786282232,0.7458978511230016,
 0.5860579881786846,0.24314392523638806,0.36927852756646207,
 0.7942363015231552,0.6697520195227744,0.4851767577169914,0.9817321197437466,
 0.7076110578689546,0.36538005629571435,0.17488522515273286,
 0.6030784104879484,0.024185561834812908,0.7577494895024692,
 0.19770671280878227,0.2038933407262825,0.5477699228364299,0.5783743344626606,
 0.34162626121325257,0.6966148048610843,0.4343128273332748,
 0.22222513022510515,0.5434468138371892,0.048019569722565714,
 0.7313131070181368,0.9642590228837837,0.5461951772304195,0.41497489775159035,
 0.5353379248678081,0.32935871223421154,0.6219704448953157,0.747007945250118,
 0.9500241223759025,0.8458824343717654,0.009055561694837011,
 0.02380887515442931,0.9912290266698511,0.6062209725448179,
 0.32307939124349483,0.607412558518014,0.935757753941108,0.6023057544035785,
 0.011822381164818707,0.9576217054102032,0.7282885144493767,
 0.35227629640176217,0.17030150068692895,0.12471739169263107,
 0.25584088134971006,0.7486010731677326,0.3831876701969016,0.9403588020101257,
 0.31870953971341354,0.8964150072346101,0.4938011630062279,
 0.15141248704394517,0.02252141340457292,0.7864443217672115,
 0.7256039206971658,0.5292995675661412,0.9756989840502329,0.2888881353888292,
 0.2770836186866088,0.650156929571392,0.956500750257218,0.9841117641395751,
 0.39989003225963704,0.5111737401878499,0.32396849587206256,
 0.0411442287118452,0.40662774618906605,0.20044071914070138,
 0.40356513612072975,0.02907924885869073,0.9118725540230107,
 0.8015827520086709,0.024296967804353198,0.633123607488441,0.8022007226260864,
 0.8491678756498752,0.5720429661065474,0.760986893869106,0.5060506733333126,
 0.42049992747743714,0.04602489817013822,0.9732332352314249,
 0.41063914823310943,0.9873996290031588,0.605205161906095,0.7609227995380454,
 0.5447127677470844,0.1801774934621153,0.818941812240632,0.1914565179500829,
 0.09801846653754687,0.8556068320577388,0.5776186802525805,0.9093775855998869,
 0.11760493672833239,0.5928602688770306,0.21124467985820505,
 0.16668235926696107,0.9774402131525642,0.49611148014577433,
 0.9502487102573746,0.6640703430287669,0.25183468445210444,0.8542216486626041,
 0.7146738482557542,0.7197264834038635,0.018610634378033453,
 0.26652310357873654,0.1491356550530779,0.1898612644968033,0.4091219118852256,
 0.18442573502357273,0.5288860929409414,0.8325668268139785,0.7057152475787613,
 0.6507622320054988,0.7319915315515373,0.6826784081161553,0.9636338306325949,
 0.09939387162043944,0.42418354452471285,0.7663696103428231,
 0.15703502095650068,0.9970552179114354,0.5053290583363097,0.5521149540020215,
 0.6978505903592647,0.9625181351370882,0.9050455923728548,0.5145610442500504,
 0.5041341923833557,0.2156173458386691,0.7140730807704412,0.36040467210585714,
 0.8856251488741615,0.9933123546933237,0.5878364921076649,0.2228064605601775,
 0.7939884687473445,0.22324523588084277,0.8665709762352535,0.9664580158123443,
 0.21629112055910626,0.014727982419577512,0.8243404684499911,
 0.8706639388744712,0.8015027180904694,0.6309469319063233,0.400092078703988,
 0.2942034777642186,0.44815017837653515,0.35546328109949976,
 0.8558178814865788,0.988052587006135,0.7512182780763469,0.40743034425997093,
 0.3820860255055336,0.10395061702881048,0.12564458177120608,
 0.0076883834555550035,0.6146042568774622,0.21349140538593936,
 0.3553997890519256,0.6569803753377763,0.7456448228163265,0.24676012039688366,
 0.2627952545524568,0.8595687398600111,0.5967869473047099,0.6304265470327826,
 0.06345625045171577,0.056408947413454036,0.8140191108263231,
 0.1828605158581228,0.7094377070259806,0.6438467214965662,0.09911264060545122,
 0.30962857465819094,0.7932982637438923,0.26781256585560465,
 0.7041573875947603,0.23270889988571575,0.4689099030802617,0.8182323703157253,
 0.33247804927288804,0.2407749506755601,0.9001204595735118,0.5341472953068576,
 0.20826478755612254,0.6998687417824409,0.1610333750047308,0.8341806185286481,
 0.9045358815819621,0.93955367982667,0.16566760497604105,0.8978070285659271,
 0.6816151309754214,0.014169476735834019,0.8197480081009632,
 0.5658835038983919,0.07963948266424004,0.6072093112282121,0.742770836238219,
 0.07667190476306418,0.29231474173711713,0.40634287500024024,
 0.04021264725024942,0.130482153208891,0.060961740330929826,
 0.44527892895648635,0.6387218939088212,0.42262675001508754,
 0.4534133658350935,0.24524858383333314,0.1909721896279415,0.8377280827479645,
 0.8457268964082367,0.3695248597416493,0.35685582010469863,0.7819141197790377,
 0.49459414647439626,0.5144710661622585,0.6971460434251965,0.328704455372512,
 0.031048686470977915,0.9965749261741628,0.23964674356970572,
 0.16661904399829008,0.9873393878897173,0.28390317291202094,
 0.9743780483913732,0.9884449586838158,0.13936296702649553,
 0.29777410229747237,0.4202080490698745,0.12497405204849421,
 0.3067531385282408,0.30365064079658455,0.18521506805653076,
 0.1421871947575577,0.25222628941884184,0.06459412357919925,
 0.07501742308359904,0.23403225401594563,0.28442861389126295,
 0.43786370928594065,0.7234486520402281,0.22299439443849844,
 0.22027622342361908,0.7268534862502537,0.3447290960368672,0.9430519772138702,
 0.05748467061936158,0.36554733403690265,0.8700980113996425,
 0.4017260445941271,0.10718128081973044,0.37425942910730536,
 0.5011619057862329,0.303552737105399,0.7278004889063068,0.8353161823672107,
 0.128871116528106,0.004574876148928153,0.3274904273119833,0.6878583540823222,
 0.674372477405487,0.4683981321463224,0.46577908338700924,0.5835573655540214,
 0.47905847323269934,0.8255468367596961,0.9279691089119072,0.7028080528368876,
 0.40895446164061544,0.4033308403116035,0.5159774284789089,0.316284117268647,
 0.07453268225968424,0.29111415392187645,0.9650532970347048,
 0.1250124432756663,0.47791446883448563,0.24342971953089798,
 0.7595329933872766,0.9759262586454633,0.7812535858130071,0.35853221912501265,
 0.5523533562646378,0.18439985632457057,0.9658383717847588,0.6358736747897332,
 0.5461598088582708,0.685731645676772,0.07760726006980379,0.6235571179463796,
 0.12383351213064453,0.44048208002334543,0.6543378844831705,
 0.6315853304959358,0.8045266400989365,0.5364048009642095,0.8852106943314324,
 0.8095226895579973,0.2713546853936506,0.3696689508168871,0.1780549249725465,
 0.9960431756829541,0.4724042692628192,0.24830622804084546,0.2577854029820108,
 0.8234298719520972,0.6217679122399129,0.5943731925826361,0.21281986370021622,
 0.6993486692833518,0.6062253984559146,0.14683157629439558,0.8720510357486583,
 0.3678695290647881,0.5343973425539867,0.6701165714573186,0.5444977882366577,
 0.9481180269661255,0.3272812059150677,0.8668379957922774,0.0649035483154704,
 0.48958210187091233,0.8390500910195652,0.0105811071753944,
 0.40037305709571747,0.8351941152977999,0.3038186998513176,
 0.22767299597298463,0.6149962250950376,0.7126841457838977,0.4136433745392535,
 0.1241730752002893,0.9652305775132111,0.9648233737238963,0.3791700605163877,
 0.046692154027783705,0.8478677490271735,0.5287146098123374,
 0.20323068243312403,0.6001182089291864,0.2593514734497564,
 0.38401737562533445,0.8095056573778175,0.05872611153687857,
 0.6021923914806947,0.30301181162115254,0.32556464156327125,
 0.8580477430510546,0.8346652007186367,0.8367023567029902,0.2676193578212548,
 0.6864228738178937,0.45160474897610114,0.6273968226572267,0.7289303761928494,
 0.17814217757305573,0.22982180072079894,0.862662014148444,
 0.48698750249779044,0.5705829537754894,0.5277294392575458,0.4695790571471149,
 0.7344398490475358,0.5271420469481636,0.0581689988652041,
 0.033180400062787885,0.6875343570525494,8.960019989754375e-4,
 0.5054762348463504,0.8836176321916025,0.48121301840754116,
 0.49479095268953643,0.6782944701769383,0.6729995830585364,
 0.058575805788597335,0.621261998152094,0.4890705354414807,
 4.017786255480216e-4,0.09979455692659489,0.07842403064402237,
 0.7929600838723121,0.3964856766183832,0.573951622964379,0.7374101201013616,
 0.5769921240119674,0.7550249985469919,0.8146350583431082,0.6208180428371134,
 0.6855921588599299,0.6798834770037588,0.8360033543788737,
 0.035463435608105276,0.11531970868239905,0.8879278733035766,
 0.11808120333682859,0.8253242215390626,0.3086564046419509,
 0.10791317883380613,0.9935634616290682,0.817627877916811,0.5832112150524027,
 0.8060934367292233,0.69515696628894,0.6096435183206235,0.12386983026027754,
 0.09728248468198808,0.7141264705177715,0.8230689188559166,0.9953220182911038,
 0.5919245675855254,0.39750922146591183,0.043902514814840754,
 0.21817505191289888,0.5372844829538838,0.6061739742749468,0.7572802634621081,
 0.5221326431821334,0.6421536557287412,0.9194291374177093,0.4038895272991804,
 0.5184649577923682,0.6728445582766074,0.6500000158755702,0.07656923234741564,
 0.9038272393026949,0.5554504966109923,0.407691513276322,0.4873455082153393,
 0.1765668204342763,0.1604511397044891,0.052697702769411436,
 0.6745124664665076,0.07129963787139626,0.08674975950799313,
 0.3318692737532085,0.837407553031424,0.15954591175712773,0.4981904831318491,
 0.153514218342387,0.4683659289307517,0.011842383738660867,
 0.41787754510839026,0.9235738321056739,0.8002440894249528,0.5594294070951257,
 0.241082340289276,0.669762848871272,0.3459225379066473,0.8772822801474937,
 0.5349165370785216,0.7073127695056288,0.01577963921424863,
 0.37391979605777037,0.12262541588195375,0.9130581354604366,
 0.40846848294580396,0.9691456862889044,0.8586923421945031,
 0.08377943432878598,0.7063430659710964,0.7297371607250378,
 0.26584437786704307,0.7546770106924221,0.07965154046840162,
 0.0543082961424739,0.3112963702184883,0.1719490006427723,0.8239496117842822,
 0.8491784985703176,0.6299836443819979,0.45935016030797127,
 0.06170192212798897,0.3642842503704222,0.3939973846974326,0.8003242097801385,
 0.0918214400351689,0.0948583387270544,0.26203622046323327,
 0.16603377491595017,0.046977367577027196,0.1779949703458794,
 0.3548425864584265,0.28882608447079505,0.5784350553205481,0.7921224562078281,
 0.388904780152621,0.3642147013950622,0.32681654559100815,0.6752107172394519,
 0.07873432126263935,0.42083033308517204,0.26137697143824523,
 0.2741581322006196,0.5003948419078438,0.3000379208922367,0.555895446632902,
 0.7796293390641151,0.9998173865924994,0.5356859653223428,0.21981748408012058,
 0.9028805869695246,0.7145081133744317,0.06013350475178081,
 0.14928203825094155,0.537575907321806,0.0712745252108351,0.30813687121132427,
 0.209753562427774,0.8247635141432257,0.7815350242100944,0.32143120345976495,
 0.1893759256570201,0.3885952985234612,0.6083325602571925,0.7218766922105577,
 0.21180394779120992,0.20039330225594731,0.34162578047583547,
 0.23833732824665788,0.5451616332333025,0.8650233800120607,0.5117222166516566,
 0.7667398751084289,0.9455816156896095,0.019477095896651697,
 0.20797262529080973,0.05889066757109962,0.44702912941256734,
 0.7526395530890422,0.332283667052118,0.24585025952797446,0.2082110038856113,
 0.8162474614997953,0.3633067674459538,0.30147519838646664,0.6802075333742634,
 0.22763635643842206,0.3565563317178273,0.28624280280270864,
 0.053780951080200756,0.7721589073490033,0.8317015810344046,
 0.1117566144446307,0.6137224502251718,0.34839382435805644,0.4767262904476709,
 0.4745380613105029,0.33433980423585064,0.4719593401574208,
 0.20963074282955685,0.906122161801868,0.052535649116759564,
 0.5151327409449111,0.7403028508352254,0.9573039111519374,0.37215191662750957,
 0.12189710421655553,0.10917299005910264,0.9545621575435387,
 0.09850734695910657,0.016326253355367748,0.6074467824100634,
 0.6397012595859761,0.4329303189825735,0.42520478364387615,
 0.41476165285838795,0.7975897344006455,0.8531599646948635,0.2530009666009777,
 0.24501801990884686,0.0024155488307032735,0.6953631447986159,
 0.7373322597315195,0.30925635262760676,0.027898988592755303,
 0.9667381479452926,0.5335268505310484,0.09049524389035857,0.9270916070532647,
 0.5216905041600706,0.33610835672479455,0.30213600233118876,
 0.9261038118641889,0.9366203595769352,0.22293955726505532,0.4378403617009422,
 0.938907995250102,0.9255755454322094,0.7481897568605809,0.7753985069268033,
 0.050709193274627085,0.8160298583827006,0.9621502077755355,
 0.9883373202477467,0.12285307114760036,0.7558062341888097,0.8759387687958999,
 0.7445716981953154,0.1978791220732139,0.4064627219125212,0.03129718157678696,
 0.46019318217144356,0.9900890090880288,0.20752115804570204,0.950614120500083,
 0.6737423743931006,0.33155714585839413,0.26051371901640963,
 0.10664624819078516,0.36590922360356726,0.7662329676362941,
 0.7000249701693344,0.6609378184376488,0.6764454631934105,0.8359146138396869,
 0.6751603307216242,0.3518644183883465,0.037660187534016565,
 0.8917136148843566,0.9441035092365434,0.046875254950833,0.6270225427403141,
 0.9008356315693746,0.3352184954689511,0.8715533300739502,0.7555704377935909,
 0.2631944197610284,0.8879893280858551,0.8678035740587718,0.1811905446450376,
 0.94274329757693,0.02613632194213078,0.81443809453164,0.9086734585258153,
 0.036319366486648685,0.23333047560166043,0.48928605707492856,
 0.8740184529439369,0.9814057848037949,0.08482438477122667,0.7201780558283889,
 0.25466911195739206,0.134270460732555,0.5452537681661818,0.30954984738010727,
 0.6772594861282278,0.3660414337097373,0.6587451590323568,0.535023848793803,
 0.8470040076722272,0.16034363731216672,0.9742364214831467,0.6959037652937987,
 0.7094339705786132,0.47898447164437485,0.4492852639314182,0.523816209260234,
 0.17779995094643064,0.7942237858289574,0.8595079241532277,0.3038916129234306,
 0.13653397610634688,0.19502575109344922,0.8044405830175577,
 0.3433361413172923,0.6354919762057554,0.6626462282701807,0.1523685200593039,
 0.24083970614103278,0.2113317393147809,0.23053059537725007,
 0.7209583123042439,0.6690331443091587,0.024874086434781617,
 0.32675297478750687,0.5630488330618347,0.6215602975444062,0.9395889772039766,
 0.2171966005826964,0.427142258131312,0.8733867644885788,0.6052175148844164,
 0.342962004523802,0.9389319048311545,0.32269269377206444,0.0959628493256387,
 0.951940491824331,0.6612798349471745,0.9967766064889072,0.6982282623931615,
 0.28212659471622925,0.8355888243469666,0.009664656100466629,
 0.13866052650261773,0.006423935343029985,0.864145565278062,
 0.8954921940991882,0.03231624675347855,0.7863609049116933,0.7905636056477808,
 0.30433715002131867,0.2058355002720541,0.8026448707089661,0.2599180552548156,
 0.6886078819831233,0.3859037594398521,0.7756223244191243,0.14760968791845408,
 0.4576944298174579,0.2047812119845157,0.542393740893385,0.04941470837247974,
 0.6624505126637141,0.8953721839700494,0.8241575088034923,0.15332060688655558,
 0.6765945331714978,0.39335851895163465,0.8815784808952738,0.4573318630824261,
 0.10887606509226988,0.42711122422656245,0.6618647824555095,
 0.014816878254917176,0.6451691764822536,0.23339751467377157,
 0.9051033082691813,0.9356696738485684,0.031965724059441425,
 0.6463823510088947,0.3549417224397178,0.2906260727412022,0.471936883858723,
 0.8550387058104789,0.15412135181324693,0.30492645530172047,
 0.16231796434569157,0.8324121619950828,0.21065286257873728,
 0.7330474360176418,0.4156093158623797,0.029354784145203983,
 0.2672849516509367,0.2080596104447786,0.6639165834588755,0.9863082345689813,
 0.3708455460558514,0.22206345349102197,0.11418714389585416,
 0.22228729973336592,0.24682393320501916,0.20410585505601542,
 0.3341829624031797,0.9860095095188395,0.640371443666662,0.21941295684439954,
 0.8832339739279732,0.3563731238818705,0.7443713864280784,0.4027190687953116,
 0.8365992181432248,0.36493258715215005,0.5688629985450658,0.3767182242816225,
 0.748638647264099,0.8686478841647622,0.2636584564218478,0.73672434818317,
 0.1364951258674556,0.32402284803936354,0.25207196561614675,
 0.6844764785009879,0.9617586397625075,0.6941411758352202,0.1969782615096476,
 0.3841187847294153,0.8749886592076952,0.5521589253489722,0.5970395998711339,
 0.35126824242377086,0.12723113057123903,0.2231541018162394,
 0.8303078969436161,0.019098087778944506,0.38687350358985273,
 0.40368144947913287,0.9190533605206106,0.8485815417455765,0.716440595905282,
 0.4482028632994266,0.4009899424939264,0.18920276308053008,0.70493338398775,
 0.7309798937381548,0.6288808740244164,0.5051437205884622,0.9891814355094208,
 0.9739259549619155,0.9983057349444537,0.5347694326137545,0.5518436795512345,
 0.7293608461256312,0.9372730964921967,0.029340953169437523,
 0.05224589094640564,0.9602444024098242,0.5496629623174613,0.3678913553097891,
 0.7284462121689697,0.8790321825085748,0.03774141190887026,
 0.43147569922658113,0.19458546400416044,0.9522493788415108,
 0.22672067436566057,0.19810996372556522,0.16114036469193604,
 0.5785453048573028,0.49545543528433433,0.671920989692746,0.39859773118429564,
 0.9758360033855487,0.2512323931587812,0.31712414485689,0.933534684933124,
 0.29804501840966,0.9986690342254928,0.04825182842545539,0.6520894425614633,
 0.14191717021751504,0.5421787522233399,0.7513330041923312,0.7484388429256827,
 0.06370623488151284,0.012562977539128828,0.5132901717445725,
 0.1530063427683066,0.3473881719138109,0.03687354473843529,
 0.37519090742259076,0.6915813670865872,0.3208756987882896,0.9841989371569715,
 0.18889063004002815,0.14730620623803925,0.8939854662858356,
 0.36654841783317815,0.2813428425351494,0.9980129735917047,0.7185242938681076,
 0.1181084608206937,0.713221754112156,0.9207547824925593,0.15659328081235735,
 0.9365066734423186,0.24757832710271066,0.08040427947171636,
 0.4359094420912679,0.6675107754380627,0.9931274320263976,0.2989213178407635,
 0.8417521818647808,0.5800272518525309,0.056045410514948935,
 0.5308169355195159,0.8950682517692563,0.4436714408178305,0.04183310815025476,
 0.8779315830080836,0.22815863517520618,0.3982437306816219,0.4403587172367147,
 0.09963265698847668,0.16347604704202612,0.06849490883214626,
 0.9449062974582655,0.17455046362754856,0.5451737276794719,
 0.39000792291055075,0.5277828938497713,0.975263553546994,0.3850340153681028,
 0.707502040731439,0.6952542605508586,0.6614787445010535,0.45324724575184827,
 0.210678728720047,0.1407145238859484,0.8600479023532811,0.06852657216651181,
 0.840115348207187,0.9832731823580947,0.35370156926485197,0.7594811193146541,
 0.5860976295676996,0.1519411303968744,0.23887569446899481,
 0.10794460199698097,0.3743410913963692,0.4700933386089108,0.3601277075967385,
 0.2446294203599666,0.42674546916619194,0.14225853517547893,
 0.06691444028633142,0.6813062406005764,0.5518451695801684,0.9005647501386684,
 0.5573897551597418,0.5456314038622438,0.7705915769074572,0.13645500188334203,
 0.7868094912972535,0.4299972700319271,0.015423176190703658,
 0.5791239574113018,0.2720039181123548,0.21022875493457605,0.8944788627074575,
 0.6699651478226819,0.41167780240730234,0.11946672774662115,
 0.10995081865416434,0.88427295909225,0.6038917367882755,0.769410147565617,
 0.26462622514026946,0.8400709702115632,0.7711216027084168,0.4020324851114465,
 0.5874928177893133,0.7372070583979664,0.0011508123759771394,
 0.30264585584164616,0.6408007540666204,0.8041786076436347,
 0.11575501147324885,0.3628038004167655,0.8962065416189495,0.6665104067788952,
 0.25931398053800614,0.5921494713804666,0.7864043315811224,0.7759038798746818,
 0.160801075144147,0.6822144000301208,0.20526720434307888,0.52495976188145,
 0.9640031113641515,0.3552106314243936,0.48467710479708426,
 0.014198904491986686,0.6118415335516602,0.6251635347156559,0.682472222326645,
 0.4517338093354927,0.019393174193760743,0.6951979539438712,
 0.6940737033206391,0.22248459968984624,0.39832123793763596,
 0.8239303891322426,0.9246854991879715,0.20599574415376865,0.7058098459335161,
 0.6954561921034141,0.4145751262813878,0.33349632184570455,0.904159472395729,
 0.3592609431528361,0.9211364456422207,0.720004316256946,0.5916742369526059,
 0.7848934482463255,0.6468122703752521,0.30382946866255545,
 0.18516038262149614,0.8349208693536874,0.6765656210114857,
 0.17071754675058126,0.5235842868293592,0.9469625196368229,0.6221720748992049,
 0.3699565714921885,0.4337065855033545,0.5609983583280618,0.8409955535741842,
 0.7130542153537088,0.5147691168220152,0.3437491623355041,0.161137355624264,
 0.802869270890821,0.5866059178835024,0.425775142742865,0.9782808857925279,
 0.18310044591153696,0.7646019111958507,0.7311960950119671,0.8921751436635291,
 0.7563693214033275,0.8077878056240846,0.46624568660169086,0.3641859891857344,
 0.36237198237544366,0.2298021065768746,0.1745529561703807,0.5257485989653565,
 0.5464231787887033,0.13163172529555056,0.06214408927220061,
 0.9928709975784473,0.48872112369251086,0.5945401639132025,0.7364750081722174,
 0.9368026457540749,0.5699901566434173,0.7789407197668454,0.05463360674926532,
 0.6028172602405983,0.5040042297820952,0.620954290684811,0.29479765411264813,
 0.22073018754806784,0.9073176849693654,0.4085134915399362,
 0.18139418967134646,0.28561572083922504,0.20888448070226306,
 0.3909850379990947,0.6754785142807862,0.2896099746987394,0.957098244918311,
 0.07978856197874173,0.611871527046978,0.9221812325755201,0.7295911281261035,
 0.5904093352690769,0.5327473729685895,0.4759410701589404,0.9266914857354255,
 0.17928096324420062,0.0739427252996605,0.8837120197496128,0.5406068726644739,
 0.40179941944618447,0.42337442312585316,0.15888938251291496,
 0.9082033655214621,0.21901432521459874,0.5987094078986914,0.5942257949193627,
 0.08590458926305067,0.4281267535357123,0.5484822379815599,0.9171207893107027,
 0.7156502021838247,0.234204555345642,0.3370436438994222,0.9487256307161664,
 0.7288260864286998,0.4877770572327653,0.29837882784513503,0.6574676384377789,
 0.9520283072962318,0.19141194793917726,0.23324516898464487,
 0.0828477534722375,0.7064747605031705,0.03445189721349484,0.1086366240835751,
 0.7148611193737459,0.7215542978978866,0.05883133116533823,0.4886986148495214,
 0.1043449315477446,0.5327686962952454,0.7036936047119278,0.7819503085385046,
 0.5646772581963582,0.27566966133050697,0.4864161352898897,
 0.32754603659343984,0.7136629245735945,0.9720432283800918,0.087310026877504,
 0.44917936644094925,0.7755473287412759,0.12134416496760925,0.98832562552293,
 0.3839439787838248,0.7606017767593394,0.21895533001809386,0.4473473549936734,
 0.09330237680458653,0.6293031622007184,0.32221354907983435,
 0.9236583128099062,0.4204115233371978,0.8148827022517754,0.2678846993708631,
 0.42697758494152105,0.8781077968233068,0.07037644613369143,0.696312177381817,
 0.1014972786263153,0.2681846038054936,0.9806847006813715,0.35254225148452634,
 0.7397135484658405,0.11246858810366345,0.18845731190765658,
 0.14501792967455496,0.5393450635344887,0.5463224268867234,0.4156948516607363,
 0.4823166668210259,0.8858067060909497,0.594168407334919,0.9151118695452413,
 0.6588285499886556,0.7906566950891475,0.8644001820430007,0.5403076855847049,
 0.25307816692775953,0.8516243587014167,0.8220347089096998,
 0.40630687762546225,0.3156568627428057,0.5378536934422836,0.7871517176488062,
 0.4934217047919893,0.5421853361403912,0.25686172088022297,
 0.07320729104380119,0.07815787967252485,0.03032766862217595,
 0.3505661648765861,0.6896929347281223,0.5197670958302645,0.1779396943089271,
 0.6138851653901228,0.9326460202884264,0.25229084088993947,0.6242936024226415,
 0.9153303390234584,0.9075101968311237,0.6447200093164116,0.2305642833507623,
 0.6290923825615545,0.19051318318987842,0.6885995562739239,0.7522757626993567,
 0.8592940033276368,0.3393666930362862,0.42513189093720294,0.5883669858999925,
 0.2846385416231815,0.011470667785824284,0.22099728469568125,
 0.5807789021085323,0.21377143171571322,0.6534625771411242,0.9989329503390454,
 0.9717276127343112,0.5553646037696902,0.17396802120230492,0.4701759835801822,
 0.49935267693811636,0.47536860258451097,0.5295677640643934,
 0.6426758856287904,0.1201458958702718,0.7762014247628631,0.7526192539568954,
 0.6741804727178029,0.7828266021075648,0.7734198085006558,0.3254206412802604,
 0.15473094998061765,0.41254347268058333,0.49224728754159364,
 0.9676067058079754,0.745419815067358,0.11184360360718981,0.6627437918292,
 0.33167847317502097,0.44376998611712326,0.8702008321563017,
 0.5721469408193233,0.30773315904887244,0.9630209165623835,
 0.05597902402670751,0.17800802115066983,0.362776241232585,0.8122429303360003,
 0.8763930980848361,0.5792641958744299,0.5270260571307911,
 0.014487415553887328,0.3770534163924988,0.30679720148804246,
 0.13253294427607565,0.6377639649417708,0.8756673854584374,
 0.26256867781519444,0.8207947696454655,0.011687972855725315,
 0.11555171127712494,0.11522624070843412,0.7991191941882283,
 0.5424229576088321,0.5934464567347657,0.42161519268916015,
 0.23845872428214832,0.5188063118624442,0.8622989633789448,0.8820835730455715,
 0.9306471876012242,0.10029732819969006,0.11726423460917323,
 0.6901058349225062,0.37685086305493654,0.7924359430090463,0.6879284895835737,
 0.6003435512168298,0.3617080214759636,0.833250540632802,0.590778105745442,
 0.5465887104884873,0.7197835850846931,0.6916407187391898,0.781819637175673,
 0.5258847931290964,0.42968936671407376,0.860388695378435,0.9314808683517974,
 0.46878101341462763,0.7844569453116672,0.2862643626340192,
 0.27045165056184906,0.11839849049591844,0.2325550386650207,
 0.9567530207613776,0.5058559689366817,0.01728405493740648,0.7800519564312888,
 0.5139419289645769,0.3134978381012018,0.1225827473043688,0.3333348911757188,
 0.9428987259085931,0.2818976927223653,0.15597880611023962,
 0.12229336531580681,0.246674562089634,0.2969190952834727,0.3621561481358896,
 0.6107837566082522,0.22195107573389306,0.4376966969823449,
 0.35968212116162157,0.6856703078338502,0.45536228391720057,
 0.14985893327117505,0.6160707103621058,0.02072324474429399,
 0.9892694156918501,0.16983032575659862,0.3775821411663872,0.8664246289399306,
 0.9392503047578185,0.5324670863981454,0.9859083221128583,0.9971425110958316,
 0.9246771711491215,0.3376595713199174,0.5431514196355528,0.8801455447780158,
 0.4603069529987427,0.07936137304756252,0.4038400705910725,
 0.11330756737549008,0.13254160967651585,0.6285779296942853,
 0.2162305571786265,0.8685261015747572,0.16886496586690536,0.2136679698091194,
 0.3560557649356667,0.9662152342692154,0.106068885691733,0.0866273688885969,
 0.74995305330236,0.9048898739677159,0.26311875199569223,0.6668013686025721,
 0.40644891063622024,0.34368103131977945,0.10838187116119158,
 0.6492361321224142,0.7776613331571416,0.05746197427021771,0.7273910325632031,
 0.830831544555706,0.4767653899429828,0.004773632405600137,
 0.30046940135573674,0.7752020032568171,0.9538778693873259,0.7727905581457498,
 0.9114876045887107,0.3914685195551002,0.18927078569547184,0.591663277227314,
 0.6088628727987516,0.71652272143686,0.9337410911745876,0.28363826752014887,
 0.4308255977082135,0.8442272814615989,0.8206781829572041,0.6636302353440989,
 0.3973886944835654,0.5391522305586122,0.784960107224964,0.31214113409223376,
 0.1302823660401664,0.2751257590728321,0.8289889313225661,0.8529474716494411,
 0.14439080152677897,0.7381316675880001,0.3728812072229427,0.1648257637450955,
 0.8946832989536705,0.9962766991540207,0.17868534144586212,0.9797779719236237,
 0.9528382329194248,0.050323172408656136,0.31802288736759876,
 0.7037769419894588,0.963462914241456,0.5094719844914578,0.14556827947833528,
 0.7612414491129192,0.12393027687295544,0.8167702102530929,
 0.41246735197118356,0.03856934144153612,0.8834603415381328,
 0.1875011966260195,0.7363880254586195,0.7538618273740832,0.20633233203600532,
 0.3404178978069424,0.5752191394580807,0.8588080566641925,0.407251900230417,
 0.04903860419793937,0.8451892504356889,0.6694424826442604,0.7656831684882994,
 0.1814085665435895,0.2961881827156634,0.9138935460554442,
 0.042771818674918305,0.008920831380327843,9.467998785398501e-4,
 0.0032291241467432563,0.11483760288054246,0.7757618181296593,
 0.17980946855899027,0.9410988117949228,0.9437450211040644,0.8000837784455263,
 0.22242972717357767,0.13562790591293228,0.7449881722798923,
 0.6190688117448088,0.95781775288649,0.8238817174386933,0.47263159785778874,
 0.32905470341347653,0.8374528347733343,0.28753039176632045,
 0.9409187205531702,0.40580877979022545,0.9084477662363462,0.0594687194507284,
 0.9236747077523758,0.03492000671135831,0.7111468488744417,0.6344875816492099,
 0.5743516184571975,0.5346956991236667,0.42971926888181255,0.5835877734117031,
 0.08253138708778496,0.07076339405960119,0.7428873691745448,
 0.8015720680631356,0.3939207075553265,0.3900574483097229,0.7048856324067979,
 0.06602662131971959,0.20194147226109238,0.39531659467924696,
 0.879628668647572,0.8019047085466741,0.5554086783381957,0.8647040511719585,
 0.31725809640080327,0.24231068519965526,0.07433031666728518,
 0.7398108411317427,0.1941293884437022,0.26947742763839466,0.5599949325680428,
 0.3227218036126398,0.9957356207239256,0.17522387022363684,
 0.41322996189969696,0.07418313902226392,0.924893732429052,0.8873833565688564,
 0.8262463973222541,0.8498132452662053,0.9394863782401065,0.23133972039983786,
 0.37924597614993527,0.5878049689273799,0.5070652741664305,0.6415703664329745,
 0.21326686408283324,0.39133812378960964,0.2862188262147922,0.927631830439095,
 0.39187604516001795,0.7882892457128554,0.22781879301140917,
 0.3360857345425504,0.14786045216796562,0.7301877979466403,0.9216995517159076,
 0.8186528021054497,0.8820204384135263,0.23923947727871198,0.4651109102135884,
 0.7345681027765651,0.4030664493010091,0.04477676407466591,0.7990134885864798,
 0.04731202105719412,0.053158161307577556,0.29459375728784376,
 0.040045861767579405,0.7399255542865507,0.23889066905972023,
 0.42338659949485025,0.6727856620302213,0.38008668832439785,0.833738592761667,
 0.004770062684877785,0.5756967409028534,0.21919696485764517,
 0.12073435984783054,0.08544602634056542,0.882431736918288,0.4519266801320002,
 0.7268720248807304,0.22039046756858438,0.7170062141910925,0.4331080715148914,
 0.7919875198594187,0.2847434981104744,0.9492766172214991,0.826253831605033,
 0.5218895740798549,0.7057363268771611,0.9236945280928883,0.36279894067031127,
 0.7135735782065907,0.9603303726352224,0.17593465346538606,0.4298457799101002,
 0.5720531857403981,0.6006125978204456,0.09916974924674238,0.5409964637535009,
 0.381073217471108,0.8455990623927739,0.2703371612044895,0.02345655873802721,
 0.7631413177529345,0.23410518165403849,0.5023293052059381,0.7013942549983265,
 0.00292780591794517,0.9174345358273021,0.3434016121421952,0.3204973976617248,
 0.5520509911531196,0.5078598361898423,0.15331882933218,0.9300107424357732,
 0.5034860335756114,0.8267894121042912,0.12055729292494932,0.5470745037945975,
 0.7684926876200495,0.4848744664574193,0.20704640486161408,
 0.04338106347024051,0.6190370991190206,0.17132070671952537,
 0.40242597677951564,0.40247915849455196,0.5684683667583799,
 0.5273910539205764,0.7941209476393979,0.9266742878248593,0.7307656489024075,
 0.5058033535623785,0.9921346804770012,0.7789748222805521,0.9458988152605541,
 0.5113075624918355,0.9295408800841842,0.21087841293281517,0.9173635534577372,
 0.7615920629401102,0.7134056125531305,0.32486922884265745,0.4304091921505264,
 0.8047745059861009,0.11534951979269414,0.2551412283200739,0.686791838782321,
 0.6738767710965583,0.3101010579414595,0.2690724128501094,0.9328413898494338,
 0.3765882560791276,0.27812142843987,0.39949790798504403,0.21423276658446266,
 0.9412037909134557,0.819182591562607,0.6331936086636082,0.3381704493190215,
 0.13303498978464146,0.3250743821454789,0.5865064523372797,0.5094772031775551,
 0.1480560853995263,0.8793661502155714,0.3620290164208081,0.6282930734788501,
 0.9844909994914114,0.18809182086173082,0.2002828234349564,0.3921780639132596,
 0.8046661813029711,0.2538055415276953,0.642116265427286,0.10539719270492243,
 0.5198485220571758,0.1043429677911486,0.18617691492458766,0.5575482315458632,
 0.10223863791945942,0.9810709514428816,0.235627752545007,0.42212210498735103,
 0.964771654551564,0.7854659082216611,0.6497455046677776,0.1456320596468622,
 0.6210536580114014,0.9804690975221604,0.4262692041919729,0.8577146596030198,
 0.5467837624547652,0.6050717393676254,0.5889524706060598,0.11022333785370841,
 0.5940020999526805,0.9697782311074489,0.18624811243221107,0.3359059373679565,
 0.12912265943928125,0.9498627587095868,0.8873818882283575,0.284138197936855,
 0.738804334832565,0.4835645505379953,0.9093760792908496,0.4719054482247371,
 0.1896253838428099,0.9775814540162612,0.3216846720476635,0.37726448016842207,
 0.44635363126823857,0.8348135916645604,0.38615539606699767,
 0.08133674198369145,0.13262778791959406,0.4744490655139426,
 0.8943182166084839,0.01637636110627605,0.4577810477345178,
 0.42517095595024834,0.4795272600842043,0.6041766606465707,0.7800821920279424,
 0.8633841195405807,0.6835219752363477,0.30645027712847284,0.3188847919245743,
 0.7685676035273131,0.4123486881521331,0.4012020893797754,0.13412326219916215,
 0.3088012643193154,0.8724173491206613,0.41193155260462544,
 0.11819165683758981,0.8972977855074316,0.8452500774351392,0.5260002157082403,
 0.6206089040040867,0.8060207056452763,0.16764990558582393,0.6182074205699299,
 0.22486509589296522,0.33279212043195583,0.8793565904999595,
 0.3847856739582888,0.6237757589399063,0.37874713366920043,0.1940357580689125,
 0.7277681259002322,0.938472010075913,0.9028701661026285,0.8631455627661382,
 0.48000122046829297,0.7687203188086446,0.579410298607692,0.09881368984169325,
 0.6921201672048637,0.7871876495269524,0.9843304934070228,0.33742320641222334,
 0.8167904644160874,0.8006500242613039,0.18070184723056326,0.6359279518886236,
 0.7222941103089016,0.7857304711014123,0.9924634887333572,0.8999066594726967,
 0.8911644800998824,0.03001335003162997,0.6483918002916835,0.8661803172639658,
 0.9306074917781889,0.3147747960574073,0.30362189643358517,0.652628849781567,
 0.6725783071008595,0.02837456464745769,0.8643452270677161,
 0.26739204188205545,0.23769559098356907,0.3358362684450136,
 0.06748287089081939,0.7358389520979912,0.8229080381514844,
 0.04267134564024411,0.4469993807248611,0.9144863627901835,0.8036078271194302,
 0.4773568046519976,0.6933835036274715,0.8386888794588825,0.32013466017511416,
 0.2513897009302184,0.1566517708988171,0.9809297068336753,0.1953628638850219,
 0.727746219918573,0.2377824396972379,0.955860590134141,0.14042697387794967,
 0.07892173919326761,0.26066270822009674,0.11775776593811993,
 0.5226615909817238,0.7934000746814648,0.7445152678409475,0.7479711393733893,
 0.7261679645643933,0.17838962028798666,0.24745399478450725,
 0.13371144934729107,0.7595351210509613,0.25044982042133535,
 0.27043584458972103,0.20831404476469073,0.026255500984992963,
 0.6915916750527992,0.3543319766849429,0.14455509593721416,
 0.13956295108447614,0.31458587215298384,0.17256183750502307,
 0.3131925762037393,0.7365527757030758,0.35425485216510055,0.532604359484357,
 0.9396045116593903,0.1028792923526688,0.6696497189686581,0.46974013321303,
 0.6210566219280655,0.04982258196530687,0.027539622666368224,
 0.17127343798532013,0.10313532960761163,0.16696906260696143,
 0.31606433760359454,0.34997176923997353,0.45488461093057064,
 0.24451276075600803,0.6317660717074766,0.7206943185312162,0.9090849126164721,
 0.6991215388101186,0.14085009924237257,0.8951439826373958,
 0.06222689769423628,0.998447877774043,0.7963924706723287,0.4156438667871464,
 0.21840765254928807,0.14916169344424124,0.41856945109784727,
 0.5856282234132884,0.32085496600180075,0.04821250212630679,
 0.02186653638559699,0.23601624594654158,0.7558624443391778,0.362246118122011,
 0.9899803832566816,0.6037023096934924,0.843987210121223,0.4887205308745389,
 0.14625350177768004,0.1803382935237735,0.4931406827765089,
 0.24392698348323838,0.8337594807751353,0.7652727823516726,0.2091855755734212,
 0.30703461470389737,0.6508890445648243,0.1385413753299768,
 0.07799416197015852,0.877217657474178,0.8067566685687854,0.7799669438256676,
 0.08165552652456787,0.05205730065666825,0.9287267168194551,
 0.5335111000966993,0.7574868348145406,0.08287576874349467,0.6322810403778412,
 0.467251219187083,0.43759579128614423,0.16176384574078562,
 0.06141030313381113,0.24770620379962938,0.38540099282077733,
 0.5277446394249916,0.26107330123601413,0.7695393550946584,
 0.04950447693371762,0.09798209096710508,0.2600776459758396,
 0.29732967993801673,0.8974451981254852,0.8405677185630589,
 0.25465399273618283,0.48320529995396977,0.7518102755102248,
 0.7832003029584431,0.7624270995647447,0.43707881953211203,
 0.09301500046738753,0.1953474514884609,0.07613850849334503,
 0.002236394051752355,0.5879828949919942,0.4868946028932801,
 0.25073549613367196,0.3588058415536374,0.16286562391132708,
 0.8231602974287114,0.6924567182406791,0.4946007182223683,0.5673054026384621,
 0.2536218819868665,0.9841431600561144,0.9170014536565217,0.9957324148495711,
 0.7302296954804572,0.01930386204949963,0.7269900957922585,
 0.024918349581914412,0.7809732823786976,0.8663203679819167,
 0.6380175067313549,0.25377972018397665,0.6390505180014714,
 0.19600274326081146,0.48386435319996024,0.08125702895621123,
 0.3148281576560228,0.968022246336626,0.8119573711043608,0.29499793829402166,
 0.8369688087684167,0.8275153312396951,0.8772409007039476,0.5866188329289451,
 0.44093658787344214,0.893659804185719,0.3943717631385326,0.16340639765253784,
 0.1763424255115651,0.13868629221436413,0.15003275138675665,
 0.31498445761135496,0.9614963643754721,0.6434304187256757,
 0.40892009705943755,0.7096655943187282,0.5536414946182225,
 0.06005970646373904,0.810306922725508,0.42948031176278123,
 0.18818664128225748,0.345849790774199,0.9549607615689079,0.5239036793700099,
 0.7549152091247763,0.5836929544312839,0.82406467243083,0.8357025176189165,
 0.9174741411121218,0.9000041564360352,0.5321088595639475,0.5434775373686662,
 0.040973378572395136,0.19102862435499557,0.4566088837336475,
 0.01865426360251743,0.4768022367848297,0.37890331203193317,
 0.29350640413443985,0.8450943887623756,0.5960332630138166,
 0.48719565754596017,0.716863561269722,0.5482916927362672,0.2806926794726873,
 0.24751990619918074,0.381968922670185,0.9215885971636174,0.8580773063751207,
 0.07232592535355586,0.26265033304366436,0.43424226154200096,
 0.13643486288703055,0.9204553843868903,0.6013847408524085,
 0.0026304749919485726,0.8324231693020563,0.8077099840868737,
 0.3377203154379155,0.5522536803724228,0.4272089893290685,0.6213017217306245,
 0.7477862588081255,0.2773117321239901,0.35061565988117704,0.8736753422940636,
 0.4611308883965347,0.27709147929884126,0.809836872479411,0.4606225458094637,
 0.9467772955888298,0.7062701616257573,0.9123330642133654,
 0.018362423358186497,0.2853608203692395,0.2560675036371973,
 0.2866357857225347,0.6398663514339937,0.23143780798128188,0.7094550474745648,
 0.7536662381941671,0.317514994831151,0.2613559709784472,0.8555067196331522,
 0.14001383304930948,0.014377168741079194,0.030494887943377957,
 0.3511485689378582,0.6353091417573302,0.587166315567513,0.5960403263124487,
 0.1406789091855164,0.035781775830698304,0.2935656659956599,0.884243348766035,
 0.13262210437647892,0.7612476147213081,0.7524756823056378,
 0.014450080525056253,0.09431703985554252,0.5338984969708284,
 0.7635520750568532,0.48167711819968106,0.6073238014366332,0.767346629116103,
 0.22427572103428617,0.2716428389372174,0.8278167717466283,0.7248892294852806,
 0.006881216484939001,0.08596170115420476,0.6885814358078316,
 0.9470228246777974,0.5154883495730554,0.09592757692470133,0.8383705497562998,
 0.9723307051195998,0.8142860054171757,0.6575473430183894,0.5363593792852991,
 0.8796143481134359,0.3157248656595444,0.9254210992170909,0.231306143734193,
 0.15139006138991395,0.4415834039534292,0.5790246757323876,0.4409532933409849,
 0.9857678509764333,0.9910877058083628,0.6354130870457585,0.3474294436775508,
 0.7069441045752984,0.4274393170433469,0.5733520660316421,0.4581589635499195,
 0.6216458663541301,0.4179425494958757,0.36160473471179355,0.1809098783960661,
 0.40935489316438667,0.6487617826512776,0.25766632934239686,
 0.3117872230236497,0.3899127475370535,0.7406034807668493,0.34629303961537017,
 0.7260579313305189,0.19589334695039162,0.6716230908117182,0.9347670082006853,
 0.6855018655273963,0.727369306987059,0.5655368204785431,0.8706119012174124,
 0.15413381456335906,0.5976608467156518,0.6937861343826462,0.6013988473355867,
 0.18415633700311618,0.8641361204736243,0.9381486066648619,0.3412313767270906,
 0.37538174814267955,0.32879937692276895,0.23178005463960993,
 0.1330056442239116,0.9351249926231127,0.9262785967208681,0.5812875376454401,
 0.5807732275955736,0.652854373819141,0.6220085223479369,0.3008778197016557,
 0.25845063454160144,0.3904852059732934,0.5767819587547718,0.858401512006481,
 0.5838111591968143,0.06434809719959744,0.7269853101277259,0.6713886410065227,
 0.24282949162545897,0.5998613514959643,0.036775824325621986,
 0.6050984848118041,0.3599986996567327,0.872865909831746,0.7110672607933796,
 0.9761928937310393,0.7539535595825575,0.764828442906906,0.49010613112832413,
 0.6962745903521063,0.9036831672425738,0.12186089067633965,
 0.037910873554158675,0.39103832307676256,0.47829753951078,0.3895227670535121,
 0.9741022307043053,0.07819287397668517,0.29389666587452545,
 0.7813710617095103,0.7320215521219529,0.9889133839077583,0.7256104675015227,
 0.5467738645288208,0.5575495324263963,0.7934983105680544,0.5554266677693829,
 0.6068402811606296,0.3356814252891065,0.7872845784175806,0.10109186770947176,
 0.5301089925392637,0.7098085264048449,0.028841327797387795,
 0.4664269401025174,0.08066932155379392,0.5799410333385246,
 0.31376595372980565,0.31151903512509926,0.43417791460862076,
 0.6140318574996716,0.23939851390719624,0.6389107595181678,
 0.49784662206890296,0.2643945884046496,0.085235004623609,0.3402739537919496,
 0.5050364247930175,0.07273951078470609,0.9069725958471564,0.5243185553219623,
 0.49250381087471407,0.8625775051960475,0.3040012816454063,0.5353398805934975,
 0.6758945044473103,0.2320222909380506,0.3547228487227151,0.45910125450636907,
 0.32937364392363966,0.06389241313844851,0.7308174475112579,
 0.8196326348961639,0.304706810287215,0.9039425948978553,0.559102045359768,
 0.2855700689719518,0.6518871103480488,0.5812937172801256,0.0845085157607246,
 0.8735914968118953,0.3516038239286109,0.11825558420057936,0.9310079554396631,
 0.35286276761253843,0.663886873174769,0.0489594419304582,0.6541217590410384,
 0.12460550260406733,0.3054703759245536,0.15112773825830006,
 0.23297318237168618,0.5868491393550214,0.371283974606205,0.23869524812291432,
 0.07178253399396872,0.8925440783284636,0.8346021734410206,0.5930632145551961,
 0.8034058098962278,0.7955210177725489,0.4109597863062219,0.37657939712348154,
 0.10803616111958347,0.6372460704557408,0.15147405540059078,
 0.8788393450870293,0.6366477691791625,0.427685198065309,0.8877370578408933,
 0.28907985108597434,0.1916035291840703,0.8616152370187968,0.8043965242344794,
 0.6242324757398582,0.6994030098936709,0.4341910691862222,0.10070519275729783,
 0.2955285581243138,0.5340866577438373,0.7759246258504684,0.5377861199687046,
 0.9278626234189281,0.1698091650442266,0.5501077712407766,0.15726157687537556,
 0.701886575308953,0.6719789100807483,0.003492028736390873,0.7297077128423879,
 0.4776763797431789,0.7370202104579958,0.3135270622819474,0.37730612404943387,
 0.6591060412715827,0.29947136395747076,0.8143345186273502,0.8057722114387328,
 0.32770896290437024,0.34582537542408487,0.4242628169279603,
 0.9628521626521338,0.8666404999464308,0.13078273403471652,
 0.17394569624243517,0.8536012983114084,0.6494592309163884,0.321093746827826,
 0.7766110655385184,0.42006944335111474,0.06958463690954964,
 0.10609996018496637,0.29624978431807614,0.9169647210153619,
 0.0026844331054018955,0.6811860084629776,0.6690172262245497,
 0.21966061233919842,0.5328041638570821,0.5945121565573095,
 0.08075504672886014,0.2789941541735105,0.520940970700271,0.197335092748683,
 0.1295634021872517,0.3022309789790053,0.09621300476916095,
 0.30796125578592703,0.1027130912245724,0.46859322874117204,
 0.7192400793580421,0.49959858449758365,0.17917523183567863,
 0.7794184952054868,0.48373668516565316,0.9156784317987168,
 0.03522586141560513,0.3687475442821244,0.9396864857358295,0.8237898720084664,
 0.25866437693029254,0.040660714241859686,0.9971306816500469,
 0.5647490863564629,0.30402337723836625,0.890484526643561,0.8971670953488833,
 0.9253436156932436,0.14061452285657716,0.876383020112806,0.412056634783867,
 0.4928432162226436,0.05256487619681005,0.8603115839044841,0.5924890913642806,
 0.6701462095896813,0.2716767457061149,0.537098349033094,0.8298560902475135,
 0.9673151081697914,0.3505760962233586,0.7430837925232456,0.8034437841850999,
 0.3119317783412101,0.16632167234195028,0.017881489094606184,
 0.8760714714317497,0.25564673081365574,0.9879027445138208,0.5025913256228316,
 0.975011716525401,0.5578926941851534,0.864453391743663,0.378415589910745,
 0.3564473644539381,0.25382984568899614,0.5000290492291528,
 0.17564545222745842,0.842633536325901,0.33582049865913166,0.645981786491035,
 0.9424280103485474,0.03303150205228644,0.5862509926123491,0.7946830150949165,
 0.2017634493867111,0.12405317388746573,0.6269245889280134,
 0.24299952405312597,0.17732876218596116,0.06272914942261609,
 6.457229626786187e-4,0.6799102059471117,0.5239020389835095,
 0.7545602242830335,0.9096394259902933,0.6404711908633964,0.8644314979611348,
 0.30509685501973793,0.5576429988785363,0.1512151697838151,0.847460035829421,
 0.6571901700154198,0.059500281122046195,0.5019266654353123,
 0.7133898901024412,0.6111575658315,0.0682961695906521,0.8810379718607804,
 0.15313332089676268,0.5099361876280025,0.36832584625425324,
 0.41540532297565935,0.5070941094186845,0.1950366102814053,
 0.19299889580834795,0.832142807071901,0.38333701668715836,
 0.09144215408376044,0.965527943827608,0.21160797269706022,
 0.34894247726206873,0.4542970765196992,0.08755813258696821,
 0.14409182662242692,0.4465033974805196,0.6589352801866686,0.6620815850452432,
 0.02719729405787874,0.8601976318636776,0.800379145869603,0.6571563405830914,
 0.6263843383239943,0.6317700058559901,0.0674878350696777,0.9086687706992884,
 0.6828228533241539,0.9878392256459433,0.12090553838898443,0.9567288621030681,
 0.16067792958573135,0.19567389405004,0.9408099158806824,0.4817273753782838,
 0.9173674315596468,0.4558361686626695,0.2593283091578452,0.8793386053460551,
 0.8032686481270456,0.6343731458775049,0.6440308740851095,0.7952416335400706,
 0.8047595687600849,0.19185483780300505,0.00855060964182186,
 0.10674878147387112,0.0969961329502399,0.5892319276170523,
 0.32434262791494295,0.12622330717724373,0.19531511210083108,
 0.7632245096794747,0.2845896965110859,0.9615360865355288,0.774523655725078,
 0.5520067834823429,0.9049565143489993,0.5091435092519851,0.550219769209178,
 0.6598379948164452,0.22917282908794945,0.6320337292827013,
 0.22249410520774915,0.3166054631886659,0.19146497356958336,
 0.046110905647940514,0.11579103994262552,0.4601646605972949,
 0.6565265111005871,0.829531296598296,0.22021099597210703,0.6367625823989321,
 0.8106612231842852,0.40160470432251505,0.29539180641272067,
 0.1890521792431329,0.4337428232364142,0.8487497753998752,0.9499027339086139,
 0.8712811656116101,0.16203095937083645,0.8409159194067455,0.3643605451831352,
 0.9023483708068222,0.4314767221729682,0.8795751232044046,
 0.016119899042276065,0.40463646184169255,0.552136636211924,
 0.4598575499181714,0.9550191014815554,0.04050764975667365,0.8869883367636758,
 0.6589974157194931,0.7598553498866718,0.26925982165888873,0.9764795194143034,
 0.2510131151791891,0.7306832716151233,0.5917288386759698,0.46728732744324486,
 0.1573551591001624,0.9913108040921896,0.4873222116212508,0.451456761379851,
 0.43647181835598525,0.8371380971872474,0.4561106624060791,0.9821081399099592,
 0.5247406596759757,0.4595645974977468,0.7107197119060015,0.5793512324988488,
 0.11759150053966017,0.23726756612740618,0.49416390263470256,
 0.8615309306800195,0.8634389466545033,0.34015855247718996,
 0.08471031284933184,0.8528291594917305,0.757440977751462,0.9613347609554359,
 0.941103155072889,0.49000295100130764,0.8345899689154478,0.25416393359576495,
 0.40882488574980635,0.2776309910863952,0.1270530999575894,0.9380905592422596,
 0.18762198705777777,0.006738402593676307,0.17014514610510378,
 0.20652466530861247,0.3534383117081652,0.2682826348480901,
 3.489096252993207e-4,0.6454975711203791,0.5150442125062804,
 0.3678652012991156,0.7356381763544491,0.3397589630860274,0.4151908601763493,
 0.13270317356854489,0.6986330112742594,0.5434145829610416,0.140859498380432,
 0.5918865222422623,0.5192754814548824,0.4676104991261252,0.5091314598061399,
 0.7517648546421183,0.37993623772839125,0.9649591450955264,0.8248026933429164,
 0.08126550771910013,0.2716270635108916,0.4797717374304118,
 0.09408980321374816,0.9139969202171163,0.9080105278838887,
 0.17293667305564198,0.46689864048559626,0.9591128473046947,
 0.5553906816454374,0.5921161459228216,0.14599990862770706,0.2278830240688987,
 0.945080769068516,0.8910579816732811,0.3385907486018994,0.550872425788024,
 0.16633339180994,0.8594481864358026,0.29974348789440686,0.6632330112994325,
 0.0882955546619657,0.8801158730031423,0.1654206444133408,0.23645132338358565,
 0.14047459440048482,0.9176298368149067,0.5303644487734035,0.8753805609488343,
 0.3585010997750542,0.6234798643990956,0.32400001595292993,0.7504566300729273,
 0.7232135531194289,0.16091070258142015,0.017542426719041604,
 0.4595516862621055,0.5510827000772873,0.9678545130621908,0.7468984539559057,
 0.029297882315750257,0.9507980759491086,0.9107418938264635,
 0.41118249041478017,0.40791677154562267,0.8004823219373453,
 0.17446015021041028,0.7856752648794547,0.5462627130148648,
 0.44533655482536183,0.19436110963679754,0.6256356576128382,
 0.5719806959155357,0.4005133074159063,0.6033417616863956,0.8651040314567638,
 0.9615454622733128,0.5391622222677492,0.5596496063088247,0.2104448638537465,
 0.8682490865044948,0.5090092422374298,0.8323665604056141,0.7666406192882237,
 0.7529795715279648,0.6945122337619183,0.904325636782398,0.4861180725776536,
 0.0575797821228472,0.4108449194778273,0.6177280722997242,0.5329415641229981,
 0.8407903764966426,0.4313132001664134,0.13140161271424988,
 0.39850974878070666,0.7864680220489184,0.20266117900231218,
 0.9343965409545578,0.3168371103813208,0.5549955197535925,0.6817449014838659,
 0.20172049800221215,0.6304598712071983,0.6387104926339184,0.6870856756444768,
 0.9510067211072232,0.6722923952422655,0.4432791931323722,0.6776047605782956,
 0.9492343584880679,0.24752444948205232,0.6745347239707721,0.9713666058546887,
 0.5849727354610175,0.6465478369192245,0.005585934264807801,0.564387439429373,
 0.11774639042990187,0.8615163929195865,0.15589538967295036,
 0.26142064562748113,0.6396053740731023,0.23593899325973045,
 0.8269221166158073,0.3232482111566487,0.5637690316338484,0.5013255868340571,
 0.05075986466855176,0.563967386030208,0.5132570307916683,0.2836156763903316,
 0.12884403320615512,0.611080344527016,0.9458122346808986,0.8348556387642443,
 0.26282612182546994,0.391681218577669,0.4385317959778092,0.3257855640587509,
 0.5949061044621315,0.8887124115868797,0.2910437357740854,0.38340824290062503,
 0.6383332493841215,0.36521094070209825,0.23657322547576853,
 0.033830286322014125,0.9460839990297578,0.3787801771409761,
 0.4050757770456954,0.5739349095270442,0.9372760099291958,0.8827761205944,
 0.8037464930645878,0.5414261498330958,0.4705985680415604,0.14235683894064832,
 0.13552449918197063,0.8693130277923733,0.16708071932385438,
 0.8779497351148432,0.6550913764299165,0.23050732394820828,0.6126722842306974,
 0.010353371754257878,0.18552100294059948,0.8584741615948247,
 0.3908641011283325,0.8375953747295672,0.9679590163076838,0.14954382169535918,
 0.20881787120281758,0.19326793634095207,0.5113914157770545,
 0.10519149063431088,0.94050349688039,0.9445654012326387,0.6521805495424483,
 0.42968115827819475,0.010149000669775976,0.290970394640784,
 0.3084980956127743,0.9217413495140535,0.1098413307032825,0.34034902935413147,
 0.5176086221314979,0.9609956046034792,0.34619210678824996,0.5552506958819662,
 0.8904732338043032,0.3983394085416525,0.7258623620563774,0.44980442365227236,
 0.38940374959092316,0.41849032379583206,0.95472369200207,0.05906734656854318,
 0.42470635889662844,0.14333875011414,0.8425126395509389,0.5198983351396032,
 0.8593902731940148,0.6053758544382354,0.05767460241038436,
 0.008232619332106172,0.3826511456732933,0.016521984119639455,
 0.5275299505431094,0.6105064116881662,0.9858032534878057,0.39613208996977656,
 0.16107469144764808,0.20930127704017265,0.7440832114896084,
 0.2229190739490189,0.926813431951953,0.0907885327969441,0.5427300262650829,
 0.02278310753006285,0.5291381706717279,0.0520518764301634,
 0.25406186408496056,0.5250845063160978,0.6585489018525829,0.7750166921083486,
 0.4500935273104112,0.6289260885467617,0.8083627367400538,0.9776174337727971,
 0.38984627041504416,0.22055719183369293,0.21877145528269815,
 0.13209792400941311,0.8787845577216553,0.5839539931897758,0.7061645087399375,
 0.7821586670633661,0.617540271871464,0.6181581200118101,0.2043549337410675,
 0.12876001719663144,0.9315997007520916,0.5214219414400241,
 0.25270837903967247,0.5717848060859869,0.8303182595240067,0.9953574953219614,
 0.6056782100501023,0.1267686003353451,0.13336707775157242,0.2759485196685999,
 0.6741964812590561,0.047080385524369994,0.41339778851905407,
 0.7036275540964925,0.7636251889844373,0.24423663944067886,
 0.33748190237830766,0.5039328890924293,0.8246224882690255,0.6986044553845422,
 0.6466856904346718,0.34900133096398533,0.9594435591267226,
 0.35447195773851037,0.3087548349523791,0.7962845380828243,0.3957685256221902,
 0.6015029552392456,0.3951534967861725,0.1492644134531298,0.6368153042270147,
 0.8071684188668685,0.026247665504360418,0.32817724040861795,
 0.1446238388542025,0.30690072614452524,0.27014468284702864,
 0.29967563620511206,0.32768562498308507,0.9064491664544057,
 0.9239226346869951,0.061299579078209265,0.5890935401884478,
 0.5274067958329538,0.7962600401025609,0.48621472435552426,0.6518636731738656,
 0.6460949992063544,0.020208309118533663,0.6426253267305022,
 0.8569177518439758,0.9331842208449951,0.5371231441807484,0.11794539637379786,
 0.8478615101192158,0.7348298806857816,0.711243938341606,0.3656751252096052,
 0.668900210574752,0.24057210200235324,0.19272363279902682,0.7802421778569737,
 0.16221823601248586,0.8519695059818913,0.9139286014054185,
 0.10854171209281216,0.46497091154799974,0.6393355973624131,
 0.011912026276385768,0.14644341608945888,0.6657031548237425,
 0.053535969033699726,0.3498360761824201,0.14676722317132196,0.85551054169217,
 0.3779086253427436,0.3875064470168448,0.03598257024805773,0.7496207181859921,
 0.36903686140226366,0.11117921380576412,0.984906713850777,
 0.39395188433046224,0.21597009955027102,0.09754819752081245,
 0.784413591547926,0.9999587668875489,0.9486368483103746,0.7403832776765437,
 0.01846079262423428,0.3691099427870328,0.8836607019056232,0.2528362429749609,
 0.07065265106969709,0.027984943648886484,0.9512542188203251,
 0.8311374042383608,0.840603259137068,0.4417914248403707,0.24776288429864668,
 0.6817531384644832,0.8632727271242554,0.2983125179183934,0.8555598925285549,
 0.6346835815169073,0.4824630699125345,0.5681470960736068,0.7189167174274326,
 0.8713109084291818,0.11589639286272035,0.37014366602906723,
 0.40595764836487014,0.9178724279359374,0.3173070021317874,
 0.03861310491946901,0.6716664569844756,0.2814092174184192,
 0.049334467575837926,0.9650189401899895,0.3858980647638066,
 0.29989722312371536,0.5100399406397206,0.9563389104706675,0.8464872469100851,
 0.9051819189469019,0.52453711659165,0.48798287496940407,0.4282206268922739,
 0.16893263697978067,0.43172183251755925,0.9923676300507308,0.1505147770534,
 0.5941881898107674,0.7663346660201321,0.732243517485902,0.03584098893419274,
 0.9016205230393363,0.7977439027367225,0.35835069049688584,0.7413753048117377,
 0.04198904806193593,0.4045972909305898,0.03555785696317937,
 0.5586948390508393,0.993128819612359,0.03784546164349434,0.6591380993806772,
 0.5030090722507715,0.6837652274583683,0.23699551478786063,0.6509586607032284,
 0.16773794542712206,0.8926484884468481,0.9286294286290718,
 0.19507473468311531,0.6998162287033114,0.14291382023803056,
 0.8553364262348266,0.6003208019130357,0.9739175017970402,0.4206987776304769,
 0.4698827799286708,0.5884014701933555,0.18077898003123494,0.1212498031546867,
 0.048712839691491205,0.07490887685532088,0.33495999872883186,
 0.5344489519563738,0.6326602861416118,0.5444997751540488,0.6493374896208849,
 0.3079738787985147,0.7870681797947827,0.45954329386113213,0.7541588710470928,
 0.9482545578321795,0.6634493682289871,0.6487973889150584,0.6999232312779367,
 0.19885792492742782,0.04215590522191359,0.1655043255322236,
 0.4666280821957022,0.5236803123455175,0.9955445997181546,0.4247518584992065,
 0.740353169379033,0.041519818417080745,0.5889255111282923,0.7644339925629079,
 0.052338081943365644,0.8592089988550956,0.5747105775540791,
 0.5963714838302092,0.9104403877320204,0.12480978489665773,0.7212527308848967,
 0.1313394987157951,0.6632333361327571,0.5604678918710564,0.0108640271275382,
 0.9830366468132102,0.8795627297103457,0.8802862890034505,0.21280888745681148,
 0.3345203576298189,0.824908173137826,0.7349922806563809,0.7410300726230246,
 0.5060550775680435,0.33655196412887256,0.6515344621764793,
 0.19030712517510795,0.3735351050358824,0.4022076032147718,0.6720140249341878,
 0.7446080667533336,0.6235933720747042,0.43782035037601164,0.3091851385371083,
 0.3416695937780587,0.9773404659942309,0.823465952563155,0.556245543778964,
 0.13736231106132546,0.8453051078551398,0.06949395643422462,
 0.32771483835577264,0.5551241350041136,0.8987376302375676,0.7229105887060299,
 0.8768960396984544,0.4612459233539188,0.43033531372489164,
 0.30169817503738583,0.65224966598109,0.6074831855695778,0.6442599852163876,
 0.011333859241549393,0.3157388846214326,0.868426263281703,0.9617415178228621,
 0.44692045202236175,0.3027064366024428,0.8975364853793307,
 0.19442582140241693,0.7620874142199039,0.14314797906331467,
 0.5711818221763776,0.21332398484606474,0.16236489335678006,
 0.9370005304372667,0.9458721608424827,0.6716288869681862,0.2882727294842502,
 0.2655735130993828,0.3784247190682756,0.460260540783597,0.6693814462374748,
 0.10264713396040026,0.8045561813359162,0.6619681928224789,0.7759702076210445,
 0.27042825322692354,0.17925137062826668,0.4961871352000482,
 0.22399774144891682,0.32058010646304524,0.20061733511579072,
 0.8139087412378371,0.7133776010582031,0.05248982393072643,
 0.30829069068732506,0.060576559668273466,0.6190441704936105,
 0.22995056385940504,0.9337604397652467,0.5393787750432464,0.8334355638980633,
 0.9894508789947003,0.2704418262176417,0.5649173590513583,
 0.002620525328069667,0.8048445652534159,0.6646475886852318,
 0.9070079622055109,0.03969726904808735,0.5258666010667559,0.7605397485397994,
 0.028552761577821717,0.7927982103700526,0.2944595108956798,
 0.6117564845489555,0.7493665570132623,0.2560694711649981,0.613345806500211,
 0.628125311528424,0.564870907729552,0.15206676856196688,0.6308009572295931,
 0.46183449814757094,0.9618008385808259,0.3198172214178503,0.3835220341762964,
 0.9501563307159288,0.8265015534209454,0.5786973643339264,0.8051401368786795,
 0.3870154890967803,0.7826398760756426,0.0071379925823479295,
 0.2907876146124916,0.03064213946289185,0.17787830271608474,
 0.026414841463850647,0.4744969782030506,0.1095820013887484,
 0.7901769708882683,0.7931827829302837,0.9121050899704838,0.9611892777100574,
 0.5309072071943046,0.37951297407280793,0.9989716186587903,0.508492142422553,
 0.3683623765754638,0.34539949515600354,0.5592954387248359,0.5652239687382355,
 0.43803825403591856,0.9520481480195481,0.058326369306769266,
 0.7694202806192206,0.942805979862355,0.7673406260699482,0.344011082374571,
 0.10560587916422215,0.07311898656823956,0.9309000719590514,
 0.35012200926697057,0.4630536754545749,0.38092296745821197,
 0.02693052904861748,0.17284306353047652,0.13908997017273395,
 0.9847221634543328,0.19510627981337103,0.7926909387900994,0.7528631701815114,
 0.6398929346447026,0.4540403796182785,0.9295714027747212,0.40496867459960384,
 0.009429299133278612,0.241389555327312,0.9888989167003286,0.6786864185016002,
 0.19082340563778355,0.8988097994950461,0.9836944970464977,0.4099301567204765,
 0.38851519801602996,0.6294801103180773,0.5500503346093252,0.4301903038074052,
 0.5489271367253858,0.08083101455529496,0.3673611435397103,
 0.25991979880009886,0.03452498801287829,0.5739775387569709,
 0.2761523536023517,0.4450843312817806,0.8751517107787008,
 0.050395436600528254,0.10122375724552235,0.8377499221381026,
 0.7212005551339671,0.6787332668217593,0.5644595066292998,0.16979878790883096,
 0.17828880505311007,0.1819066573254593,0.9180038756005304,
 0.04997580860800088,0.32501643102135824,0.20379823059791058,
 0.4350859386241943,0.3526664789644838,0.15707499624514876,0.7548908611505089,
 0.655651180451073,0.056290729812125906,0.3174973782129563,0.8492422177513309,
 0.9623657443829371,0.5837476020989155,0.9725018198681054,0.17578579950587825,
 0.06170943398863227,0.8401125156339958,0.8798219967870596,0.8486371978702333,
 0.0560348999546717,0.05134514117500022,0.23825626548844614,
 0.37643208265290706,0.9221320914489364,0.8856517956962948,0.6023914894616669,
 0.6615847377618994,0.7631494362827214,0.6202818050839275,0.2923490846981226,
 0.09208540900506013,0.9246246042591801,0.7869944361009863,
 0.21026161812709077,0.44540196298442336,0.9712712924161914,
 0.23354880809222456,0.9348134683197852,0.5642380712917183,0.4151102212719989,
 0.9553007936058897,0.36976597786218857,0.7266395045682292,0.4133360616020134,
 0.07691793709140782,0.26538689795378323,0.36267736482514257,
 0.2414819727411478,0.038983299033017005,0.6363972749627435,
 0.12544706994712962,0.2981569026134545,0.400513107570859,0.7163000664643375,
 0.6436916589060031,0.4916066464755994,0.1564286209085901,0.5776827251715866,
 0.03925201076300555,0.13199130943015047,0.10763865457990163,
 0.08693010468793017,0.48006472480118245,0.20771405103169127,
 0.21972407306940256,0.5848323560254518,0.9650811866161974,0.5446165975337313,
 0.7002353759972066,0.30827794016500476,0.8911573732999437,0.2114578306809527,
 0.2697461064600791,0.21008878596561764,0.9571284635713861,0.61576816909601,
 0.9233678734999068,0.5355078328441345,0.057043799440629206,
 0.9061550544319844,0.7045728274087147,0.5857777401297896,0.6587573684748395,
 0.8123006163639293,0.9776558343311131,0.16139475381662005,0.9720993024463334,
 0.321386197881372,0.3167617865165674,0.3449585136402846,0.2523611422933908,
 0.7449856600594102,0.42709406224753477,0.5171610301868651,0.8022105348949441,
 0.7888234792606363,0.4552393738181414,0.246674204024392,0.3324455517513083,
 0.7073502714685469,0.22317039214368428,0.771500365522004,0.7142445702742644,
 0.3356624879358703,0.894750464215177,0.6666762883200846,0.23844454709094942,
 0.14064181324091918,0.5794017439107091,0.2217285960637263,
 0.11296434496209773,0.6538099046955552,0.004460276089750748,
 0.5870459066317871,0.45779027405144734,0.4362070897489878,
 0.055500900712186274,0.6185169543089882,0.2970220801090879,
 0.1821289577780958,0.864249769893771,0.5799454529314824,0.33992171994190035,
 0.42165945210694133,0.9299525989038591,0.0020610698089601476,
 0.6960523620126097,0.6526422625196944,0.17969805419651164,0.1411332406673541,
 0.599847676305748,0.23353302910973583,0.262195671244146,0.6144698553932448,
 0.6571899234389756,0.9328316804906822,0.13928378157130472,0.6258137386154671,
 0.08611276215650054,0.1232112177574336,0.2448068076892762,0.8813596640867268,
 0.8343161433917152,0.9821024469418439,0.4306060436617153,0.21130962013122379,
 0.18725720394106793,0.33682463655739137,0.4354433273994279,
 0.06934862576577416,0.7412067652060063,0.11109738534612423,
 0.34950109879185987,0.6506415923764445,0.8180086139081374,
 0.08573649476408718,0.773623430553786,0.047713683741322455,
 0.10957543665716807,0.1263578405217337,0.6667874919991651,0.9763243175464699,
 0.6329225921902107,0.49494146775292713,0.0702146514779628,0.8056520233525734,
 0.39772661158928524,0.5377044070200505,0.39133166262861363,
 0.8651164853777067,0.6121853592040295,0.787137183018841,0.19762889758241653,
 0.3400012956711975,0.5535654165790576,0.8245400122046469,0.8203286141931805,
 0.3764288237607679,0.16245907334374554,0.32195227559577566,
 0.9085029471656363,0.8341582394261757,0.5346440709722702,0.12969582638205612,
 0.18351080715248114,0.8927083619714717,0.13692264232412443,
 0.8289293593044696,0.5602791682449038,0.4721287716461896,0.3168076031194149,
 0.10315816390502452,0.47897496734259737,0.9574625921775647,
 0.42205673677721534,0.560661317610997,0.6959759556238204,0.3270054681191631,
 0.5570282732865139,0.17776523896879792,0.9717094683428258,0.9588403013744258,
 0.8105335154415995,0.7702100243915178,0.9497416928706583,0.6130152045788724,
 0.7211352581526306,0.42234664906355013,0.489494109878429,0.5063279687226035,
 0.9415237813922411,0.6126933749675014,0.057108592498266164,
 0.7159777184116258,0.6353487159735203,0.2819970852332263,0.11186240473543252,
 0.9467681164578469,0.30136574595720045,0.6096153237553701,0.798423230937445,
 0.8861420253081217,0.305672310273261,0.2720283492183009,0.8239852295959406,
 0.3868846533270054,0.39935231974866436,0.22062487013337428,
 0.08200994596756384,0.2680924057805345,0.49276492600640265,
 0.3893494246746153,0.4160701937908664,0.5375876465283738,0.6823185852264811,
 0.5405130976937678,0.42108308492035573,0.6425073994268629,
 0.018051137741213807,0.21774435493322508,0.7734393238299296,
 0.7880607981629935,0.911076945201507,0.9468665930091282,0.45117312891942873,
 0.5761949273309354,0.11738327600132692,0.1970090807618481,0.4814522506487251,
 0.09243886155568293,0.4070621716981433,0.7245151781810714,0.7690423391345158,
 0.9336000534385547,0.8491839137303651,0.20950932827449953,0.8587024036678264,
 0.13843289627572597,0.1326969097112427,0.9483138402057066,
 0.37711702426138094,0.7402098110502014,0.7360369429039055,
 0.05473113407631691,0.33622559137209906,0.6974953118563996,
 0.19046884926080077,0.31499632123667365,0.7086756976423074,
 0.37441320382197263,0.19177109496908995,0.4232222928224172,
 0.6985645444099959,0.09695030150259731,0.3986527976946531,0.6984910077548016,
 0.7559574589708025,0.20925398645420212,0.8089371432037802,
 0.36097447129186466,0.7283046944280455,0.2571419075367568,
 0.16833465364838396,0.6355630149404835,0.21608767876920032,
 0.5957025549313327,0.5333038339197732,0.6759081934609366,0.4037332334892807,
 0.5000989127092293,0.4941436386457506,5.184808509735017e-4,
 0.6922837128004782,0.4041944961944678,0.04372295416914307,
 0.28323940243476675,0.6157886131293926,0.32252389380659086,
 0.8874891945669274,0.16773185436100613,0.8933109032571307,0.8380351578768576,
 0.27041640421449886,0.9920386258303038,0.24566760462843196,
 0.28091182351057875,0.7452274505119802,0.25715699188596086,
 0.2502135934647014,0.6314768266072884,0.515150605462455,0.30592912130072725,
 0.4481445384156847,0.7063703570710386,0.818266229616704,0.21343219751258402,
 0.20137523054727702,0.024974342086493406,0.785446054275719,
 0.02037554049718382,0.11984976676053993,0.3696035861958863,
 0.6272466093281422,0.4665719894261755,0.23498863337505338,
 0.21446538996252773,0.5186657477176889,0.016631623439670395,
 0.04095046213410203,0.46220857732950515,0.23087335552885124,
 0.17780802336828105,0.7884432951577828,0.9076816923521484,
 0.35027822287406984,0.6187074008984492,0.51527461529293,0.11622163983543643,
 0.19612642166332328,0.4847309031909848,0.27743998042043305,
 0.5893401941422483,0.5851454500907701,0.5527156158606266,0.5217846506898755,
 0.14723215697494652,0.4937121867379467,0.5991432687084868,0.8093450957805063,
 0.08230550505139034,0.8377461036358114,0.5070310580742134,
 0.014972282668156534,0.50098001805515,0.9684739720175974,0.406407486383221,
 0.49357053783920235,0.6646166575043159,0.2016850299478401,
 0.18279995472762967,0.1821900917746584,0.3342969571108966,0.8478403453689544,
 0.7196853275735899,0.9993115142847052,0.6022781601259098,0.3891056191004504,
 0.16648259782332586,0.17169393403841282,0.6277809548284266,
 0.3606073975987679,0.20494703540790105,0.641778649123336,0.09876414560020108,
 0.7036185798696524,0.9718257794794757,0.16587252256527318,0.7614970556016611,
 0.1437242458715613,0.5227431650547557,0.10502049470626362,
 0.17802265989306376,0.3318131567465612,0.4167861063967915,0.9932322575868311,
 0.802179728446356,0.09825001583658788,0.19687688078900623,0.8888762922369255,
 0.3650143236796616,0.07432109149345534,0.8965604670590934,0.7682691666034736,
 0.662891065550677,0.3060521787651783,0.06271332161536813,0.42307292982620104,
 0.913631229635586,0.15217572004212387,0.9961379101554266,0.308441668949187,
 0.4204220505035139,0.03152354937613355,0.5519691898516048,0.7980527766211387,
 0.05594868896963856,0.4481540029495994,0.35726708612132885,
 0.7227786901723813,0.9494649034495563,0.24121727826297157,
 0.20752041748234873,0.7428713731555168,0.8977301164367234,0.5393639562008695,
 0.5537700861020446,0.8158570360433252,0.3645436619732123,0.08827267654832427,
 0.20601643547104487,0.45939912611813616,0.3267159090749092,
 0.30851411549712593,0.06216810450358834,0.11845332882332815,
 0.11699213467467917,0.7207982301470405,0.7044890994634484,0.8870103784458528,
 0.9964550554882314,0.4750542195103078,0.8325097733095236,0.7553455591970144,
 0.32442509046767354,0.3838925655592429,0.9130624073978219,0.1745757069783359,
 0.9134445403343145,0.8689516691373737,0.9403205253114602,0.8134784170847653,
 0.7611815704360667,0.7190375293148841,0.6136453317599966,0.708025816289905,
 0.6838623971070307,0.11757460076397896,0.2125338654119151,0.5217707603299756,
 0.8225331432748089,0.9955313200843608,0.7980006332750964,0.14545874406025272,
 0.8917574831376299,0.9107797998148022,0.5927953307612223,0.42383041973484614,
 0.87954977423187,0.7482895437978764,0.6930568668559786,0.0728376657205203,
 0.2368234924545839,0.27179333952600493,0.30948966108755316,
 0.7760792040848563,0.9584501657243454,0.19414620278498507,
 0.16545660476536583,0.11386690311133751,0.7479185241464815,0.305680817038548,
 0.9455777190897648,0.2058514310121491,0.9957979948411609,0.08549588425880617,
 0.3717399694421881,0.5484773209132329,0.08010547895563358,0.6807484227032824,
 0.8317310299980836,0.0953972360107056,0.1259003531513453,0.06745086313093973,
 0.6925183059855868,0.0829020820194123,0.15165892739625697,
 0.09526940907462422,0.14702445333425196,0.7863105656914378,
 0.4068954484556212,0.20891414713846956,0.5738209341549365,0.6012360111052145,
 0.1809939651673731,0.47433970197425135,0.7525701615136491,0.5040746231396642,
 0.6503283770400567,0.6787957310660044,0.8518620414234073,0.8852634430164189,
 0.18445823762944302,0.4383033169290218,0.2788889172885324,0.7582186675844487,
 0.8058155501579087,0.7689000076734188,0.23182480661706062,0.8636499529972091,
 0.9759349904559234,0.33429630259149945,0.7239590063612037,0.9069988584673105,
 0.32270757894722335,0.9073119121364073,0.1216455116389048,0.3373492481519865,
 0.01208476548874926,0.6685727559825194,0.298605650379992,0.03369788456757228,
 0.3376842553753605,0.9265698186838822,0.21181154555394555,0.5692777686246617,
 0.7090450164827383,0.2961975285905196,0.33850166135049253,0.9210221480915318,
 0.319266824530161,0.3174819342890064,0.8457901302543195,0.4557161911469546,
 9.831899413380585e-4,0.9280203082101305,0.6240574122339542,
 0.6126291247589457,0.465643138928295,0.9149195115592179,0.3931592040371469,
 0.6136071774589866,0.40768375551627134,0.7435850853300537,0.116115598211329,
 0.31125969806061804,0.6945802715582863,0.8132832878424459,
 0.07503281212011659,0.05573307105131242,0.8234312404234195,
 0.23500689855800672,0.9845710523024676,0.9196449503782139,
 0.029082561381795213,0.11855122416563302,0.34876010156155757,
 0.7518341694165751,0.1399568419610907,0.5750957921522516,0.23946516749996394,
 0.8119155949102046,0.32105830268683877,0.9547814376791349,0.8254606501397256,
 0.746778763772561,0.4208056266450192,0.10778423383168312,0.26328785971576296,
 0.2666651688588575,0.542574892961219,0.33629237883179663,0.8659438913167867,
 0.4073610660011677,0.8236448657491376,0.5153724882248057,0.16962139609131055,
 0.30143699063336504,0.952941074858175,0.9777450945729657,0.6145869528974798,
 0.95614291290832,0.44256601688272257,0.3322163940660259,0.552739226383446,
 0.8399589364273127,0.8915672074571144,0.4002223137820454,0.3643657020351462,
 0.5139588929654154,0.316422267282092,0.825796968988808,0.13205408484713876,
 0.7161052029535648,0.4283576492086638,0.7769149503263624,
 6.250197724988471e-5,0.7001546646208564,0.0037866243423931856,
 0.15661738184176432,0.2518079021759535,0.16957120254228175,
 0.6394865653338118,0.7531526563454233,0.8526130401471909,0.04358322241442403,
 0.009877659052133558,0.5287871747506174,0.7198131096228464,
 0.22773691169788624,0.895409526981544,0.35151896501519175,
 0.05615043897754868,0.33059021413461465,0.6185111725693515,
 0.8416116259467012,0.6010011607876837,0.4534951116140783,0.7673649497017022,
 0.09650092508315655,0.23661643121621456,0.9045770685568882,
 0.5420313639981325,0.05940701603524012,0.6980589258075778,0.4975464254177683,
 0.4285361844475851,0.8953797153352169,0.691993832435068,0.9014763742130586,
 0.6774451830872605,0.4528678550978834,0.1848138783418115,0.7382339486594149,
 0.3887168455303318,0.5346411908942057,0.74941127851766,0.6269929304312125,
 0.020981438157467225,0.41050372558994674,0.050565670429751686,
 0.8672941076771432,0.7004123186778117,0.4836664405129787,
 0.039887766586772955,0.972997403811743,0.30843964271016133,
 0.10375740913230602,0.898136831247732,0.3124739761378943,0.9386640401062767,
 0.7434859107682098,0.4749402938706019,0.5632781582483926,0.9265591451997982,
 0.30901994820630496,0.5873941651162723,0.7696523060216737,0.9899514122106137,
 0.18805601860128385,0.9591915935938986,0.08955114109916895,
 0.009313390505946595,0.4312520246873711,0.23055102830024243,
 0.7557869365710963,0.5754843235763615,0.782849211469825,0.9157808300772871,
 0.23952386456894614,0.7672218385704991,0.5151735590221359,0.8126504430605257,
 0.4657211569245142,0.1868363629676595,0.2901013775325182,0.5490005478400254,
 0.947884037104784,0.29565624963443093,0.25164370656942525,0.6067454710779069,
 0.6584196945931888,0.3678652990256326,0.18891896575961897,0.6114391456634956,
 0.4509284746279051,0.03914557091299875,0.6107718213360823,0.3384349025190666,
 0.3555106633907381,0.2697141689765592,0.4364451581248161,0.387532764332738,
 0.9179065119494338,0.19442118474009973,0.6875636351689861,0.394436644844957,
 0.14928099518677995,0.6285128004011824,0.8337857352707633,0.9486678627686038,
 0.7994075930598112,0.20530241209897993,0.548927298591266,0.5101673583216989,
 0.8511906872443735,0.5892370250297164,0.08148650555501902,0.7869705991739054,
 0.1056600881748857,0.30417196285375536,0.09098143403484293,
 0.7891817004027722,0.058739671430321105,0.4102385770672494,
 0.05589982533910931,0.33405992607205903,0.46704712043903207,
 0.15664184756542465,0.7179239007008039,0.4990691699236618,0.8870313831809005,
 0.09323014840636079,0.7179658370069596,0.13439037080398852,
 0.6941988209370938,0.1879587405242149,0.2196406362670793,0.6702008402829356,
 0.12222342012579657,0.8864920549500348,0.07657646076249414,0.513055442753742,
 0.4691264429630819,0.14638188861340617,0.3049938754193896,0.7040311434255118,
 0.30728346163279907,0.05730537577062922,0.59542534966901,0.5419963442675215,
 0.313502740157962,0.7516803673466993,0.32529661411889343,0.65120069342749,
 0.2588007517079116,0.1344711318978914,0.23321492093471374,0.1483702547624658,
 0.03513331707501954,0.35365006493252593,0.6193556840083695,
 0.09042771671269223,0.2277588061705521,0.8519664049127984,
 0.10935167800330592,0.6243083648870095,0.5460809310694883,
 0.22408245122359383,0.7288198610246448,0.04591395947263999,
 0.8696527606078904,0.014512323991281662,0.9400215881638674,
 0.7429501305943106,0.11391725383982909,0.9123613548319043,0.5895209303254586,
 0.48722665350655725,0.8529677099142247,0.5888855564101164,0.9251434667089548,
 0.5307750839975167,0.3401513822468112,0.6766979405131119,0.3703643980022492,
 0.8436056063832845,0.793773154361079,0.7751079359630489,0.8543889866998448,
 0.8072995206710731,0.4029912459005568,0.6708449681886122,0.9815175328084664,
 0.557221211320453,0.42407925649482103,0.23240667046311358,0.9587912284264722,
 0.8190004546183769,0.5541077482554675,0.06467943828867873,0.4630489364120881,
 0.700792847823329,0.18445566874357544,0.6214279342579468,0.8604493766412205,
 0.6084144221055549,0.05166930566240879,0.10124847096126299,
 0.26564174896280657,0.024850937926326755,0.6904246010847387,
 0.11066090041501575,0.4578593238403783,0.6542020435630671,0.5299909850029901,
 0.179931327277716,0.3880940454762726,0.12984142887386085,0.9288347272569406,
 0.7717276451656087,0.5626503519982872,0.7212412004190205,0.13043012131711507,
 0.2823200686759264,0.12965960595604065,0.6498822475618291,0.576338767756604,
 0.08835227907487853,0.8651265691628747,0.4266236483466225,0.3257489664409098,
 0.6388372886176706,0.5535575735229249,0.8066942035429079,0.620265410316676,
 0.5772151196003533,0.038243627879423925,0.4829305567718938,
 0.22066240052012265,0.2445874152596783,0.5100006373082612,0.5300136784870453,
 0.31202013552472263,0.9425964875577599,0.44737993162925327,
 0.6243229383124685,0.6700995405625613,0.645363432458471,0.16128006541902484,
 0.17099708243231537,0.04782735033078833,0.8308666048529018,
 0.5775055508218483,0.33056398187516267,0.5921248128743181,0.5912453739006667,
 0.5790448890824105,0.8826103915344123,0.052052792914551005,
 0.42636239463673253,0.9435594437207331,0.22249553945880718,
 0.5938485530072368,0.8949526296074922,0.15965651815757498,
 0.31473001804474987,0.5088541933098185,0.167819873401007,0.5153764196843862,
 0.7115968047020041,0.3074586672591192,0.6303287542889253,0.47661749134564424,
 0.8614893526284536,0.3548129791422796,0.001497713978632631,
 0.9968249144970921,0.24088928557965317,0.44613271098072005,0.706608106905782,
 0.838367154451895,0.9476170458992621,0.8037049350142974,0.10496187507989674,
 0.38345692639850737,0.2031773875326599,0.9534571854897607,0.9333447455125334,
 0.06919647535986151,0.43746339525684763,0.7402409326101229,0.342486227835479,
 0.25817326134918317,0.2151675091116232,0.704843355725625,0.03165288436771352,
 0.9252617303189298,0.9494366233136782,0.22296978537329337,0.6096977556506299,
 0.3639649421277016,0.4619485026622706,0.31446262821783566,0.2754747531258306,
 0.08405724665685321,0.0079686965052298,0.794782316945046,0.10781281763861361,
 0.13786937608906547,0.7055962895916572,0.3478972482691428,0.5461921654539521,
 0.2449875160260816,0.6083282832255984,0.9186583595833931,0.8447658937920605,
 0.001966451867851271,0.6768209641204553,0.46421110102042507,
 0.681497478710093,0.16061949635848827,0.04363488201174759,0.8602309261120858,
 0.5226433283882437,0.7065648303118134,0.10183607505417924,
 0.13426120120454854,0.36437806136427875,0.77857714488837,0.49588839989961886,
 0.5399946536381095,0.17656548479425338,0.7744204419656262,
 0.04478772798712105,0.016024566193869272,0.34953107917746806,
 0.4530673968037676,0.5309126535239036,0.23446934529015373,0.4582867366024729,
 0.044224038236473584,0.8429393296923998,0.5708896352223757,
 1.1987489729947676e-4,0.6771046036258201,0.8329073643150006,
 0.907336072736913,0.6701584149873263,0.6493199629506181,0.9717340292002211,
 0.7770722452401356,0.21945967619321027,0.6276250837574636,
 0.0031376979397913507,0.6060165724592246,0.9966123157134273,
 0.10222321072386176,0.38735936465224674,0.40458790584705107,
 0.770137844616442,0.5998885183080712,0.8881515647641195,0.5295138932978921,
 0.16176466284267632,0.1977928674303403,0.47128501221830965,
 0.6339158609752682,0.11884848523848801,0.5991788659369521,0.7963361517734056,
 0.7899094504156542,0.04781497365151299,0.5169692412896495,0.984865066033619,
 0.8921500184382913,0.21694803369636362,0.9080021031409012,
 0.14236393518396295,0.18200928797580096,0.832122005188245,
 0.05380805139708267,0.006609222131078729,0.5492661799246041,
 0.4623522528837748,0.9967982051379491,0.8661764447354396,0.8869563833053592,
 0.8114732678215193,0.8755965912108954,0.47063510538227016,0.5516989302961974,
 0.2620162368358465,0.5408310694823213,0.8338102494530024,0.12234959318605876,
 0.509550946305674,0.3638033812595233,0.31589643934636613,0.42580449166201173,
 0.36037914068756416,0.5903904953529228,0.4898305651145505,0.6346453994654145,
 0.9659142592015814,0.030301650202300023,0.2541603939147956,
 0.7309773260202833,0.22774856179861536,0.2594415523679403,0.4418270961040608,
 0.8987936318253256,0.4274716839077797,0.010239562053564022,
 0.005333624574058615,0.8650140957343384,0.5502468505862945,
 0.4375709912889675,0.004974404514277775,0.40242778656030964,
 0.6929005753356039,0.3412636690708679,0.6352534718718826,0.9859479110487299,
 0.4266477895508599,0.8364949311399096,0.7681584679721148,0.03541227271261249,
 0.1175652264034649,0.8934041580562788,0.010980787378508272,
 0.47641196872207137,0.11382959664904657,0.884166163552552,0.7924231518661663,
 0.20872012395256068,0.8528106781044854,0.025497846246733236,
 0.9342605011968899,0.9337282763861126,0.33497661769021714,0.9058529300360826,
 0.05132828745237317,0.0018558291046191222,0.1835983597410069,
 0.1611630772365047,0.7406763992846978,0.9497356310282289,0.5717109683365457,
 0.03224262565963054,0.6319515043865662,0.19052299590271216,
 0.3423258596775067,0.017823888685557288,0.2012116502486503,0.627603673411006,
 0.07760678550302968,0.01352001766010913,0.0914262937528183,0.240047005931178,
 0.2575049779650911,0.08057437609211426,0.13076840085393293,0.374615764347503,
 0.995869698273685,0.6565629397300267,0.3931824338065897,0.4143853135967426,
 0.7802758040317268,0.5946520467050435,0.4863295244732765,0.20909521714462187,
 0.47731234683060886,0.02433539961071296,0.9833988775526931,
 0.8962362373838536,0.4495440247889618,0.14238342461124365,0.6984444872725685,
 0.8663476728023874,0.21157140560162502,0.9544203040766981,0.0285219231979148,
 0.21704816322838527,0.8527054854568814,0.3224047824878401,0.597065069016578,
 0.39318075525536855,0.9295168965839972,0.08832987420251803,
 0.09758247765544548,0.9698925069887788,0.32604418148047576,
 0.7669886454559086,0.1936511393300293,0.18210065033969403,0.3966701155391842,
 0.6374356215489849,0.20795086798649842,0.6916150473864181,0.1894006244234504,
 0.9951987808944235,0.5198952562785719,0.7009445176851132,0.2701387817091746,
 0.6466923381691013,0.7422039402809466,0.020074906506711954,0.265632364181273,
 0.5061518881209224,0.24052114744913378,0.5947461842190916,
 0.39988014426250784,0.2801814152184312,0.5119779121311867,0.1586246191630465,
 0.3484549011794448,0.9571006619475639,0.6065767965679301,0.5972520577565649,
 0.42015276425590364,0.896412430327093,0.9796215010031148,0.33746532768122073,
 0.8962675542369485,0.2022229029363214,0.7171153220920377,0.6706878567411295,
 0.6609340132425956,0.7799256224946292,0.16871592602705454,
 0.42896872590548396,0.8488762164521199,0.18714564657915767,
 0.44997583888329673,0.12580809512698743,0.36788895710274594,
 0.8914192622210166,0.4566947027715681,0.20634942633952236,0.8830764770733036,
 0.531466287560428,0.6498538867780402,0.9417965534818069,0.3529990829761417,
 0.42523238099944793,0.7153679429223533,0.923509366633734,0.21288661116150565,
 0.76276593861989,0.3429704778506951,0.8539233090642218,0.6182041456168992,
 0.14027583827838896,0.6445939100835504,0.7431696416524993,0.7674703153225753,
 0.5636940538263604,0.0293423997066391,0.9988451037068822,0.681790852953387,
 0.853879535231068,0.011373084259623023,0.9087399590433611,0.2523781458042188,
 0.5186529550925325,0.06522220929106104,0.06943352668171865,0.132471229750867,
 0.7911452850677628,0.10976382746604252,7.934127609807717e-4,
 0.5140595225077016,0.9502259670471169,0.23833953590674484,0.6124838431744386,
 0.48027246357281606,0.6820165279172415,0.8903107550132259,0.7910171594733078,
 0.6549576102794887,0.7668855613977971,0.45078766615602306,0.9754736709409626,
 0.8492491454323162,0.9929139202082822,0.8609153265426932,0.20997274417717549,
 0.980644358132783,0.30603609723795977,0.7869110558560486,0.5970874656369849,
 0.36402075443350324,0.08752253718742375,0.16709342367543845,
 0.09287791816040325,0.9832225917515008,0.8756917891373148,0.5456894737994222,
 0.285671538570465,0.5717867814733842,0.4607662960846246,0.2415304930766019,
 0.698651022393984,0.7364293095298697,0.638769457045339,0.4714936385266114,
 0.3193279007452874,0.4931715514488503,0.61478582596079,0.09189930292555948,
 0.8615315115742608,0.8076444845859553,0.6951092118637056,0.822405066314581,
 0.9963003627398249,0.8379033435611501,0.41490739778042574,0.7630340305422671,
 0.4402363408568096,0.7476796821590558,0.04095609501553832,0.2772741022048042,
 0.21441902594600148,0.06529435771439052,0.5061093902921426,
 0.3792772054222424,0.37561479955357024,0.4059829608517125,0.8189435020989008,
 0.646429360883724,0.2631185849164074,0.4315217511304801,0.05012634333185528,
 0.8538286647190051,0.6004426719412138,0.6963124054025084,0.18926652804498567,
 0.08286834794321074,0.5160991034168232,0.13021930972417706,
 0.8769307831404273,0.48199061165759605,0.9159931474708483,0.3532693232444769,
 0.9686969806009138,0.47282258207833994,0.8140305989178123,0.5823656121669343,
 0.4735748377221892,0.8353508899659274,0.31404723465212525,0.6298562194143748,
 0.19316326594475597,0.5911842777210113,0.4275404831305972,0.2147436433858657,
 0.10476056338486917,0.2805414677368073,0.06803202298861943,
 0.05466524042276777,0.8300285211271314,0.9493380990814909,0.9661567603831658,
 0.4820961322598387,0.7528337905316321,0.9499236070000858,0.0408341772368751,
 0.7327078845354754,0.25098494788293857,0.7539894330735215,
 0.18752837305477843,0.1529081454528567,0.16706868943565112,
 0.40356204549559305,0.46803138465721394,0.10905598624671098,0.74285007709252,
 0.7579296410248874,0.3531866642409409,0.5106076691739972,0.1518061996520661,
 0.3950892623534915,0.3825896108699589,0.7059736714999505,0.9550524152589472,
 0.7468168870939128,0.05714350866086837,0.563439397338286,0.9559094855613354,
 0.6913303469031464,0.4460493973985885,0.04008254969126224,0.3260203973339585,
 0.02150218727985842,0.5071500045395633,0.9048898297373139,0.9757135755313201,
 0.3449010502453981,0.3219071423012678,0.8770453238355318,0.43255568471521444,
 0.2301528682542754,0.04283098546972042,0.9640177123295837,0.3465561283891334,
 0.285610533028137,0.7997321872194206,0.11695351806310661,0.21077416781176095,
 0.1883620166604455,0.6884848735823711,0.7272790712258673,0.09394630261802706,
 0.5304535802268789,0.4622502637319036,0.24650792679690314,
 0.08591858128651764,0.005710547660420895,0.12512307828423386,
 0.6539115168367586,0.2822838509360329,0.7096461853613808,0.31801295621713743,
 0.7685585268358015,0.6404427741139148,0.7015236298241099,0.6492884918703138,
 0.6344035867548798,0.33810077869837263,0.8703924518510082,0.6720822528024829,
 0.7055185247811279,0.19224896885037246,0.4893305898853493,
 0.29272561318667234,0.7252626885244569,0.2788978382224039,
 0.20627178033287585,0.736173251891426,0.9671544176543844,0.20314992562845213,
 0.7487873725953668,0.17905049313912658,0.5070725808682065,0.9250251477193702,
 0.8481259865843429,0.758974544872281,0.09656335288477058,0.7474505422856494,
 0.0894936879177678,0.8650853672192896,0.12760666158777623,0.7369214988448823,
 0.8086410114245611,0.41137518903459924,0.1535457865643084,
 0.005167639292264159,0.7905904047014337,0.42435968189572715,
 0.6283127512013816,0.9773351318290356,0.44907372091851827,0.6919148515143889,
 0.5703597528713149,0.27418788333412636,0.5859521131814939,0.7435661529901133,
 0.20779411283144777,0.1980974188312825,0.8989486263753542,
 0.034167970537919645,0.8404663158599893,0.8568914804948004,
 0.5086307505525222,0.19561751294331697,0.04112948880378353,
 0.14765627555022254,0.6611694829960427,0.15854156691827714,
 0.4137940547234771,0.3590981068314727,0.17374581345636775,0.6722433970157543,
 0.14250739029872905,8.387570447124792e-4,0.8644320173917873,
 0.6541280819564026,0.1742115837237309,0.5945375440971297,0.3247677281660526,
 0.26736011426190887,0.8036774822980821,0.7068288544690916,0.8052727304577861,
 0.9562806669401412,0.7708720595079759,0.41947722672701326,0.9732159680829039,
 0.5580731052856213,0.899901461640193,0.7377701087946027,0.2103466702318627,
 0.42515879262174183,0.08970585723078683,0.8760798624787114,
 0.12529122684930105,0.7311911346202733,0.19439539676543172,0.863007870624199,
 0.878434221484839,0.9707223359880983,0.7586782280590362,0.04246787566677468,
 0.8797724897845667,0.9294543762886179,0.6261787630063498,0.9366110912952079,
 0.21327991649814804,0.8292172587665662,0.4380564701379759,0.9125418227059932,
 0.2651256052571185,0.7621524278945088,0.22747103675080604,
 0.10339795860773493,0.7837234641083075,0.9968204586535719,0.6481270027177899,
 0.825423044329307,0.8090589564463257,0.4177355347469601,0.7756395230071037,
 0.14830193764284372,0.6319524761559543,0.9438907623642689,
 0.38491327651207874,0.46474641336691236,0.07572526018904946,
 0.5466275087290413,0.45028452203601055,0.27133372338134865,
 0.7835418020205145,0.3264504737836269,0.9168223422321916,0.9674747030704994,
 0.597592446497968,0.42330439547232923,0.2549168456864682,0.9356768025693135,
 0.30965858579930716,0.4900075758327378,0.3503454122765237,0.6930116713466761,
 0.744985105093278,0.19321964569899952,0.5631477292779646,0.7346856309073849,
 0.11255117681664006,0.5300302950883853,0.1453445761715998,0.42383893377535,
 0.42965123714886544,0.864665177560932,0.2565619089484753,0.21452007357271963,
 0.16832844059275087,0.9553326012084795,0.2299205800412224,0.7172490329555946,
 0.19800289996618048,0.9050404155531199,0.03893160370953985,
 0.16912846702025097,0.038724640119436105,0.7652344437607067,
 0.8161129104458238,0.1552368708373556,0.8293676866623197,0.7177648109805671,
 0.13353124261204463,0.15077915595268476,0.14155620474276276,
 0.5356681880708472,0.8592377152013178,0.7419757712780858,0.21064682804797086,
 0.17643140809064306,0.41800154985691473,0.8835648283394482,
 0.5163339536171773,0.604566234992701,0.8472922980972556,0.2680102744851731,
 0.04455353158267994,0.21398746090080967,0.9051081519567986,
 0.11818025928235387,0.6817201659592071,0.21806111341186774,
 0.7877090995274583,0.8482843413756105,0.8607482318993754,0.7196984393525087,
 0.7102737917726891,0.262334517856627,0.2836233652544372,0.759128387604227,
 0.29545620147597307,0.5195239427202532,0.43364300345441076,
 0.9480543322218399,0.16067304661521775,0.47885660457601964,
 0.6428949679523849,0.13021952503619705,0.13519506834993922,
 0.9038218799487309,0.343443631519885,0.4347314273910504,0.4279701930800619,
 0.5171670337980393,0.17273524929615758,0.745049283912923,0.737568212669969,
 0.5522532224997142,0.30884184834111816,0.4135491568643016,0.9815701956408563,
 0.13606669837946317,0.11230801421905567,0.5711397392753863,
 0.6847716020823947,0.11892850330224414,0.6446056266471645,0.4794852012683052,
 0.2852832910865484,0.7687200870649403,0.6965917250166136,0.733305071717204,
 0.09982178750058468,0.17956859216662713,0.10396762177555574,
 0.801150167839133,0.885011470202135,0.7697314006176507,0.9156451410785592,
 0.04535185534630837,0.436945896199484,0.733351837432501,0.2012894706314906,
 0.5806891623277518,0.23341018687020998,0.31322906662062766,
 0.8746803060189458,0.3326538347260166,0.2452886652100419,0.40663785388993856,
 0.1316424610261201,0.8552323690362194,0.5771334233905251,0.6818533114305446,
 0.6675129470769542,0.08406297558562525,0.6965039293528643,0.7126458740643833,
 0.6602549317221149,0.7020065695399363,0.9201300703170513,0.4623772014985197,
 0.9065713167585043,0.8965875999341123,0.21018609837548097,0.9275881664984293,
 0.5375525628890371,0.991434396368562,0.5198353313237273,0.05870872383417003,
 0.9113698122615512,0.7679831844881922,0.9141682148719726,0.7080948500166184,
 0.7730926288671269,0.4160726363420292,0.963925963107812,0.3982853173632557,
 0.4231411032757314,0.408590186705193,0.5586905516604592,0.35648190058669715,
 0.8471610703587401,0.19089497105118636,0.2883670849919103,
 0.20020530398209946,0.5232165281005963,0.716927163286228,0.0136995530140116,
 0.8617577361377382,0.3695336902058808,0.7220006366478093,0.5987795576294219,
 0.05001943038989909,0.40094744969716056,0.9915074278074921,
 0.49778556609681157,0.36922218079538294,0.8339375366153822,
 0.2687473355894916,0.23212715624502667,0.6786409932961823,0.7289432152976121,
 0.6784637908119977,0.19853671739477485,0.48719929515830396,
 0.7179585235868933,0.9570964320712196,0.19118203575425963,0.2518000813636905,
 0.10067884994046339,0.4463299676353958,0.7318856326965911,0.6486778266634893,
 0.8367124943660933,0.0434669851196402,0.16851840671482088,
 0.19301156405083975,0.8039337041184841,0.029940671653307827,
 0.28019083345267015,0.09452181927544767,0.328796484576223,
 0.39253164602096835,0.6151260440582847,0.747591090653515,0.5018470727424944,
 0.1939353349173284,0.027531544195133928,0.6858729620374756,
 0.8558097061240716,0.2317735767166068,0.8270090008063931,0.14982557744719893,
 0.005908110050955373,0.9385387545846707,0.10423158738159866,
 0.8415184838253436,0.854623808461352,0.6466468807717319,0.40609171320096005,
 0.5959739720914291,0.4128721301631879,0.8250328344528599,0.9983121276777611,
 0.25809029506829606,0.8193709815230465,0.5887211003150437,0.8413781509940513,
 0.007288779118341582,0.827818966463827,0.3578627146897019,0.7563602626412832,
 0.4486398117541376,0.6611515271938905,0.4137184004633664,0.4154780642635837,
 0.483249816211033,0.5558761278722307,0.41975800011143893,0.12202925147671062,
 0.8196916235412439,0.12046988933333114,0.5914565839337511,
 0.009166146648410423,0.5033823549180884,0.6726993741104044,
 0.03932095109815714,0.592489489150706,0.6616937574050623,0.8393354018428266,
 0.5658819150449566,0.9040181696250297,0.5435113610069177,0.5875528336949947,
 0.8116944358763167,0.8851581206537353,0.6537329852316173,0.34919412524536764,
 0.6825939219651199,0.5218927101553108,0.9349436178692186,
 0.029913566207829922,0.30978376857213064,0.37482103336872075,
 0.8800285334371869,0.7272069930290406,0.6535771745520987,0.1675096810605371,
 0.957559889811663,0.0374931175504295,0.8635427500727226,0.04451398754775304,
 0.5873503940612949,0.15571074214568537,0.246118465394069,0.3169619809803278,
 0.6857841322417149,0.5432521028852727,0.10173378898919794,
 0.07581632543519468,0.7433804396878221,0.8163554871634826,0.8836328923205674,
 0.7354307922809549,0.19128164698764893,0.08565631141066099,
 0.6695137565150738,0.002980326435375935,0.9527960129734195,
 0.5814170705918273,0.5296363585853787,0.5600727069097284,0.6503266124230584,
 0.6821622176169215,0.4562972354877707,0.9868543271080563,0.7219068095027825,
 0.6773028389614311,0.747401116200513,0.09298586396194275,0.6777155590510042,
 0.28304788665802105,0.06388076264835474,0.5357648046854175,
 0.10192341023808582,0.5797211729872167,0.43669140982847743,
 0.7905861276710318,0.4172089629070297,0.1704307573262489,0.8553377916793862,
 0.5409136281832785,0.8770901999851747,0.4572899551498627,0.6893139753395322,
 0.38172386289362115,0.2570421160811117,0.9932825152100809,0.6553388481857414,
 0.748521647070229,0.018509403618826803,0.1974647978825519,
 0.10918125037609827,0.2606200465480585,0.9788040072089027,0.7502175771638318,
 0.7455983127091177,0.7958236096616091,0.6649539435557494,0.34006444563100935,
 0.9253351453992257,0.8639251841928964,0.7943931896343012,0.6637419882422809,
 0.2724706891996901,0.1283388244876671,0.13698750071506394,
 0.02491623588845937,0.4095763588737823,0.07454273185436178,
 0.43629614293740593,0.5702412266251324,0.10641102323822638,
 0.9837021485473886,0.8090191029682793,0.3815348365976796,0.5324653414395473,
 0.9439346238079264,0.37868573154803253,0.42124922507225204,
 0.2508171005872739,0.1622497694853826,0.7161493252984431,0.09338206263505988,
 0.9581387830465724,0.9637733163187959,0.37253467456202294,0.9639120464329405,
 0.019650101905124284,0.13493651374910165,0.8393208845146236,
 0.4869912425775268,0.11618135502053784,0.3166274291644644,
 0.27405495680610237,0.3215200574924715,0.6665966106028112,0.861876773552009,
 0.40601435640699424,0.731261566160067,0.8147856371618433,0.3164485721331075,
 0.5881347431277358,0.6284791563783821,0.14389529415893987,0.3425624869067849,
 0.052721329318029864,0.5749499678205203,0.1838513401214178,
 0.16401342080953552,0.8794304947359287,0.3328735451583684,0.9074202151589641,
 0.8919764372380452,0.25344738455220117,0.1907846701927275,0.8916871063913158,
 0.9570806340147864,0.736639481162285,0.1565531187181738,0.5621170764873692,
 0.7954883753923382,0.3603842765968277,0.5543906274642608,0.4436328925274634,
 0.22317478110886424,0.03191799673202811,0.21381849428944943,
 0.4958168815821703,0.7853123816715191,0.060901853298397035,
 0.6707518805110757,0.889710701107702,0.40145670824163815,0.13055122589132273,
 0.35928390903368934,0.6481690325447874,0.02478360384083378,
 0.7726343808990661,0.3588578896139498,0.332881323656562,0.9964655222843443,
 0.14112469211197165,0.5905016573812005,0.5285105873062376,0.5479623984394874,
 0.32082703877758445,0.8276807744857437,0.04689698577275614,
 0.48144287931308616,0.9425093345541627,0.42840734292129257,
 0.9285547460525257,0.58880587215277,0.17683378898460667,0.2529462018627471,
 0.16633171102906985,0.6439641080723403,0.9319270809075103,0.3545331508153483,
 0.7181559211790226,0.9829226695656702,0.2423591431401676,0.5286092990409126,
 0.6139471561821055,0.7203956372936515,0.6716504662601135,0.39532615475171706,
 0.5793565669385379,0.07972636396858623,0.8041341343381313,0.8271636160590514,
 0.09990713252713224,0.7051766307499909,0.13413412937519498,
 0.3382684270954771,0.2708847790121751,0.11815420122388576,0.7670343282818223,
 0.29799682748473033,0.5961209753857732,0.6109885848912814,
 0.058702961529230935,0.7757309104034327,0.5202356912914089,
 0.6122003079106503,0.2249090368373463,0.0521624397886733,0.5453839044600767,
 0.9486623953419517,0.25481986069144313,0.3607080411556519,0.8034146686487083,
 0.08991990721429521,0.2752513998014442,0.614854322975221,0.5192933139584844,
 0.18620897793095814,0.10476979898797123,0.04744676498331302,
 0.6322097731435109,0.13439392434388897,0.7167285252310023,
 0.17031247459630128,0.313436137102442,0.56706836750306,0.6026241711157053,
 0.9184992096904386,0.7980605835422252,0.7195835751546976,0.5892780032363826,
 0.46567295258153507,0.8167621231562883,0.6743814210550682,0.7342079918499919,
 0.8077370841683478,0.177676851240814,0.2549699505902757,0.12519003969632359,
 0.7885815633052337,0.46919066415587274,0.6696413155084919,0.6218516099541848,
 0.8973382633958051,0.872330654211862,0.17239139284863647,0.6594398448520538,
 0.009337768522024725,0.26983935318179986,0.8751993667986873,
 0.7574057372469574,0.9344164440763802,0.5961503263351545,0.5341890614090179,
 0.4655003358440054,0.13510751099651286,0.4175648151811704,0.2259988597396857,
 0.6421108544970696,0.7259878463335455,0.18023938522898408,0.5368987336733796,
 0.6279559038207185,0.43487514589466125,0.8613868159212387,0.8389021808020791,
 0.3246107573223075,0.3044607656139562,0.2993144468155896,0.619038057671085,
 0.8852986913003087,0.37454322463253975,0.6165878316568706,0.7235913664722514,
 0.8476873337620915,3.0378816278153753e-4,0.40763436680100196,
 0.044083158246440046,0.5064970988562248,0.8435635285590318,0.869739271082243,
 0.9131524238608961,0.9486595441074195,0.9235549923903573,0.7551844239235568,
 0.0038116811542334705,0.8386679782688803,0.7749234252213096,
 0.2620152748157042,0.2080160509301353,0.8238111317671077,0.29272913397205824,
 0.30222013822818805,0.5007473606478925,0.9263346848050773,
 0.43947610634967926,0.7690913262228091,0.7412833089056952,0.7531352471070951,
 0.2438891704096311,0.6520120756459106,0.6918330020488866,0.4912149979706647,
 0.4940050465611385,0.6667896242645904,0.33615798261000296,0.9818519696523266,
 0.8014687291974055,0.2756297600742609,0.18113077114116993,0.4113128957287585,
 0.5256583307826541,0.3806093437719261,0.38447678054326373,
 0.08380966496800879,0.8552358338737025,0.8235976556630045,0.0722607481009252,
 0.20955239695277905,0.9665137208716588,0.003291682368620319,
 0.9581057189620023,0.5016503892153094,0.23845255560123557,0.6699776303900676,
 0.55838412070075,0.4445954552877569,0.988398330475853,0.848997693978653,
 0.4187076073515722,0.19179150621066443,0.2609113056443191,0.5207332070948589,
 0.2882415879184248,0.9744078679766954,0.1529932303970294,0.10895527583112019,
 0.9068477115275405,0.7196606517677926,0.06390983591755495,0.3698875794536708,
 0.960894535927394,0.8757680306100243,0.11637950786504103,0.5046335090745606,
 0.8681160629504313,0.6833765509364068,0.820123454445039,0.684816152068433,
 0.4660349154089216,0.7209235196158116,0.5985896419965309,0.8504244817046643,
 0.9121209324745695,0.8143880672472323,0.8741495742704355,0.3496420093338556,
 0.5591948701680063,0.3777289143179554,0.724174319474493,0.6123526857914073,
 0.23362585214901332,0.6350493437235338,0.9220730650014948,0.3194253508369881,
 0.013599909566716195,0.25954542677606407,0.077512287584389,
 0.1714687004994736,0.09416722856524518,0.5638091302163242,0.4559719123530761,
 0.8416526643666415,0.41038646695525016,0.24275937892461064,
 0.7217563855087886,0.5946028208446107,0.10457876025605428,0.5790159553164402,
 0.066819638131008,0.8278759109865982,0.8119919478697932,0.36326923500000585,
 0.037728993090728635,0.23385759414890384,0.2841230107161147,
 0.8053946789047719,0.5060162132356617,0.8401015506239363,0.29461414224874494,
 0.9438354772474501,0.7035285492002668,0.34159671636138456,0.9501863425887029,
 0.36318021232812414,0.5257296323829548,0.90414157119765,0.5234897077215425,
 0.8375712370806871,0.2540480996199007,0.9836040026858475,0.7362350760897015,
 0.48703423134361823,0.6455689404789506,0.8332940692532834,
 0.02259730897466239,0.7311139215567709,0.8272616086689921,0.9184830697408394,
 0.5084506252851737,0.49038763181895584,0.029082391254892204,
 0.4582154712790997,0.5477352121496741,0.5628519140764425,0.3826343419416296,
 0.6805797832792085,0.5356323471412012,0.7759364910298662,0.9671012478722414,
 0.648195075640008,0.35012380767824935,0.35044157772427664,0.6306747745283775,
 0.46229425547977865,0.0215536659415686,0.4511219735552944,0.2966134529731298,
 0.3106849035185203,0.26714186723774214,0.023138381559004273,
 0.7768761021959683,0.2532818369018437,0.3534474847833854,0.09781223531860905,
 0.7048937109998441,0.10682377510288543,0.8969212113573308,0.800837340090291,
 0.5811402258406051,0.12925978794670534,0.659820348029102,0.6573012794582841,
 0.586951628592647,0.28563134783782496,0.39920073347994944,0.5720495509024833,
 0.22598699120901689,0.2553465993033782,0.23857006939346947,
 0.6425319900787209,0.7849080358400945,0.31603589506414864,0.8617010300810006,
 0.4619355256810136,0.47742666255551447,0.5880829450574121,0.662537356930564,
 0.757135403912031,0.5739717320365145,0.062060294975584296,0.190286110551386,
 0.5391196652925945,0.41013565701463517,0.3187032684514509,0.8154219388333532,
 0.8377307221486894,0.1235085113244958,0.6186467548207607,0.13615362124704333,
 0.5189663535684482,0.8440326678834722,0.9593310954200484,0.578040685259359,
 0.4533350497801767,0.25272567066771345,0.7815016591112139,0.9621838670106867,
 0.9129109348876135,0.7857022055200976,0.9208056413957348,0.2963708596016539,
 0.8521334095919918,0.2833906632669998,0.17207193533074128,0.7786480148140608,
 0.7056181481190894,0.11576940537901426,0.2293583671288062,0.6863986241438673,
 0.8928410752047737,0.21933349967635862,0.3176222184428682,0.2554205235488879,
 0.5025054007089591,0.607014246239337,0.33601993539617936,0.5149322673577238,
 0.11615949653799507,0.3905729070246182,0.26694893419101495,0.631483401572078,
 0.08087100458733809,0.8410694689246938,0.8278547152744617,0.792404947373981,
 0.16981013824311653,0.09470689021954226,0.1893634981513097,
 0.8429989844048843,0.8948656698908962,0.384823718514527,0.2799639880411674,
 0.0709020184288105,0.7082300253387683,0.18120222064970004,0.2246460422500841,
 0.9840652030840791,0.4300187101061038,0.8168908750789052,0.7696751125085364,
 0.5387107189438576,0.2911731022960149,0.09515669872891397,
 0.42714195695803836,0.047050209657572895,0.8863855586289924,
 0.7610476525661336,0.8757985836134616,0.6967632790354814,0.4054194748823955,
 0.32305231974269966,0.9351260300785049,0.06667578392027462,
 0.9894669717067548,0.9232371563260368,0.8734821457484728,0.23826199754705202,
 0.9147040142427716,0.0010989995017210985,0.08134165794229375,
 0.3550770378854643,0.7363159399377434,0.45317569402901015,
 0.11855616970534344,0.3607290615025873,0.14140920037970983,
 0.9374078751699172,0.02186475547284561,0.4763697155677068,0.8758663361245731,
 0.4213367895398159,0.17123709695249267,0.2939092560580985,
 0.34281266339201855,0.5440007570957195,0.09836215539680204,
 0.8374408943294551,0.46115256615261313,0.9910338672970076,0.9294210392166765,
 0.18276843533765752,0.7583542469434703,0.5585675944585893,
 0.29114027881314464,0.2377837824062603,0.6576507326628597,0.7869920979525096,
 0.4257733983579569,0.22007275186209108,0.7487418717706461,0.9126094579573929,
 0.9811139650096208,0.6294917882150816,0.3785364536901482,0.3639444129777697,
 0.01776881477922454,0.6704286831214892,0.4630837276613068,0.8640341154408847,
 0.0339294268516247,0.7531994494945252,0.2447948040364677,
 0.0036241705881935316,0.09803558063242723,0.01872857019359897,
 0.659454170492041,0.9065032614573431,0.26462120589764004,0.350802083072718,
 0.26816029066205904,0.7944041463211692,0.052416994332191824,
 0.29943218071991706,0.5481093142968918,0.8411009817283196,
 0.19056957228656513,0.24694923206641994,0.051750469842513525,
 0.2728585273352625,0.7364097561831121,0.11469854860051187,0.9664644141553816,
 0.2466026973323049,0.24583757398790507,0.9227364381903437,0.9513960654643708,
 0.5793466837145198,0.5347314938409998,0.929830248104506,0.5020025119409518,
 0.07011264629633507,0.12586966957897272,0.40777759501852606,
 0.6296565487408143,0.17520818388651804,0.33489168509468314,
 0.4274951934433521,0.4350856219269137,0.08648908664385113,0.5266780132807524,
 0.7846581094299914,0.3108464071935839,0.12844262772737491,0.7275000556495265,
 0.4737066737767579,0.8268811722390933,0.6693624360488968,0.8083698224995937,
 0.8704534168804203,0.3520182255632226,0.36059122711557867,
 0.11585688328083354,0.32425450277041035,0.7776627237972762,0.664498916540101,
 0.817188218483745,0.45972930096177245,0.251345384270662,0.865488975905681,
 0.2476553128939425,0.22353741224053547,0.6069547565184024,
 0.027785615236830985,0.04949342153361935,0.5093478094858047,
 0.049471203880877446,0.8442399916206833,0.18733857929782782,
 0.04519495023320086,0.7779280188628084,0.8124397895224615,0.1284273356430119,
 0.8840668841171481,0.8000704606321293,0.0885301324720491,0.5356028736366438,
 0.4458194904550301,0.9165070110703494,0.02455676731744527,0.0898841961573924,
 0.6371892774056495,0.4065318132247031,0.5899816515590479,0.3060132569754147,
 0.06010092278642021,0.4666153653291023,0.18948594973198374,
 0.5565060602992086,0.40430720869089454,0.123920089210668,0.4654552067329778,
 0.24489436150001087,0.7796137180924645,0.17794873230696473,
 0.4246562133445352,0.4438330826674157,0.922966065392954,0.4803202656479991,
 0.7265431157373944,0.10974811134873541,0.750527277236566,0.9951576002762368,
 0.9863105804672074,0.9151794720526587,0.298021276271054,0.9276764936457261,
 0.021898360252036397,0.7309797604252054,0.40744276360375664,
 0.5455210748430471,0.3639247441541774,0.6805709154713643,0.9020969726651548,
 0.6733295054274894,0.7330622495751182,0.3724980517922143,0.352434417123728,
 0.18071249172017,0.9558347218138891,0.0492288554133572,0.3662208694873029,
 0.1935175119264697,0.09077172490168828,0.9119235762136633,0.899628127986452,
 0.993547009155791,0.9938227225855465,0.9181818157166602,0.08475909188386643,
 0.5530485746492438,0.3200240823590681,0.005558993149502256,
 0.3621714488460427,0.7502096051796142,0.5781926023373366,0.4461964588090134,
 0.447028298813665,0.7223946286724504,0.9068395202775228,0.19859121247777833,
 0.21614711815286336,0.8351898198174033,0.08914210118979293,
 0.5638369070498557,0.842047133375645,0.16127566701031681,0.5213030872290287,
 0.92611361589732,0.5175938370417511,0.12820077403097818,0.27233495704922006,
 0.26817607092133144,0.474474256715687,0.5288600745957659,0.46127207241287016,
 0.2673951242917749,0.7970682993164412,0.8167105822006419,0.6501227291892555,
 0.7893961769679708,0.29013903290342646,0.8205745199217561,0.6306824268579904,
 0.7205526684455771,0.6004239653808375,0.2934347578173413,0.33786662122488476,
 0.029413153810349146,0.7262121912713166,0.5903897798494312,
 0.3391268358947006,0.18830126879325815,0.5262218997545725,
 0.10173261847291948,0.04998840653600656,0.38601579905397265,
 0.8497532191836423,0.8209479838092398,0.26896551244231814,0.7421556320143118,
 0.8861669926609516,0.7618432561429697,0.8807035739340608,0.7122647158390424,
 0.19379740216170693,0.5691166728265066,0.35956957389569866,
 0.7566850525508246,0.42643355902011804,0.5781200077841067,0.9503538195068861,
 0.5950047938350878,0.8185091234485606,0.19836982394966696,0.2342853725180074,
 0.716658343383469,0.5770355306367608,0.1244895216644859,0.8823534360461612,
 0.37344685208305495,0.011341269278031585,0.7825139395745266,
 0.658018822770048,0.5932007743062172,0.3174984032204462,0.03544199013826965,
 0.03000076833508636,0.056688402637481206,0.17004050106616797,
 0.08976905316795336,0.17793958872413784,0.23081974582035947,
 0.29638386028680885,0.5273278610917552,0.9737056211383939,0.8756432201804996,
 0.04744316188846198,0.09234973021721204,0.3508497873357972,
 0.21828853540449578,0.2457034599936505,0.015931989483112474,
 0.11987827556440256,0.9493652901807623,0.9854109735396049,
 0.07191017402186439,0.6634642388848451,0.5545213695943426,0.411581016109419,
 0.840256722597985,0.7681865448502226,0.9177251009157358,0.7075909885565311,
 0.12442196752626933,0.18993565141477353,0.911712012463315,
 0.08838582539201612,0.9802713224675748,0.16096234815555,0.6951413592225191,
 0.9042816059104183,0.5964016845016622,0.1645163512380018,0.4098673192416308,
 0.7822457671092766,0.7434621544247523,0.953672751934067,0.352045927837345,
 0.47934764697789944,0.9101124528402471,0.2322676017233527,0.4947116044708655,
 0.48476191140974445,0.43025415940051026,0.5105450288986625,
 0.015818927576159547,0.8106318133356664,0.7052055772953834,
 0.19675439669678918,0.8277668143897184,0.9159481366672558,0.4843036977055012,
 0.6230493598653352,0.573309893753483,0.09189651568135349,0.3770614112364905,
 0.13040648579470548,0.8926815415903582,0.6686695361706303,0.6271708405211182,
 0.2596532091727638,0.5660452307049482,0.22188406527791693,0.505611151140314,
 0.4962084302993016,0.16237757945198694,0.722140361420909,0.7224216179624641,
 0.3064412771555598,0.5020119501631962,0.6059757711351494,0.5489878271802595,
 0.1682238857268772,0.8390554309842754,0.12010800187804194,
 0.14160160935961497,0.40943417020884953,0.9659206152897807,
 0.9740508920294958,0.9368317238482748,0.5376634123194902,0.8091017569510762,
 0.5206972090317574,0.7445906311867708,0.6706486730620379,0.7731456985165235,
 0.4735678060741494,0.3846020186011795,0.5910193914176591,0.879372728627065,
 0.6701044072835658,0.6761801316803463,0.597811163546063,0.3186465401406857,
 0.6847630777126001,0.34885193835516615,0.5715879902498434,0.5785552167547712,
 0.9847735998257463,0.9640071523157442,0.5296713149566011,
 0.044225567539348365,0.8199004569208601,0.8817794625929035,
 0.9304355778492923,0.9264903382684276,0.6254838920212075,0.4650286258372882,
 0.5915068321730877,0.7468549121316788,0.5379456983965489,0.6291370218907013,
 0.5359862232989322,0.9211502653096134,0.33621347416476,0.3244789768999434,
 0.573743478351216,0.09898859147297845,0.37843907452764014,0.7451107926575642,
 0.8115529736479503,0.9884745523864532,0.8840572466745968,0.6198856091497624,
 0.4574857245548385,0.5948610833334834,0.593570645235587,0.6158202778535169,
 0.598757038620401,0.5208457724409267,0.25775004579952665,0.45213474961093225,
 0.645085760502481,0.7192651498266185,0.24022953584189277,0.8477148253486253,
 0.41587733463541565,0.4003407081753275,0.7668940354365132,0.268729432104735,
 0.46638500894523227,0.5318604956136816,0.5479608791837904,0.6880461463328124,
 0.09772326075784976,0.09803770179044957,0.2603794527982264,0.703404738989057,
 0.9636662044083226,0.8932619254521573,0.5501771971279397,0.6489089973047582,
 0.217188738465244,0.4044582400778933,0.7398325493588672,0.1178593279912824,
 0.07503514930666932,0.8765101319046659,0.4023723049225424,0.1418363683932945,
 0.16344449902437164,0.1262279307221802,0.3036179885036535,0.9585206091882132,
 0.7423995910119482,0.8742356002883577,0.5717922893822357,0.7553831530886272,
 0.1533170349915257,0.8739787234712235,0.17381124181665208,0.8341646259875505,
 0.1483996575201334,0.0012916438080088355,0.2187841293360896,
 0.6524433150695539,0.07576362092790512,0.049997916598122916,
 0.6592866983659303,0.7725203921418669,0.5172011553606856,0.8642151964822692,
 0.8777687794377516,0.8820066177263343,0.06563830279309901,
 0.46830425126189246,0.20005505108849653,0.024106830234175036,
 0.929349191393298,0.09764272508998562,0.8834559009624529,0.0627089719117121,
 0.7957123484462788,0.9739852393706758,0.9225648370284144,0.4603117075846228,
 0.9375705173632563,0.7384085629471455,0.513965689576204,0.11090093602496687,
 0.9030460467268867,0.9763722055326564,0.5045453965835058,0.22576396576698043,
 0.15824391357411505,0.8388390106005154,0.21093514367734245,
 0.9936092469580443,0.3390617099864772,0.018792968339439553,
 0.30829816157489676,0.21471025527026155,0.42127450176762204,
 0.5149239153389437,0.09902541698943002,0.8716661265011227,0.5659796596635581,
 0.284436963393484,0.08942786251538681,0.535545879650011,0.6284854504686038,
 0.1017332954422937,0.7401725691898241,0.5050561855319194,0.6313264467445105,
 0.26512254452079675,0.15285322103208943,0.1960564354823504,
 0.3163934790994051,0.840432589399472,0.38296593129841805,0.9914914725819419,
 0.6784075011480206,0.7242334114130179,0.08846431159433099,0.579846980100397,
 0.9902317607924793,0.3140821312978095,0.5246717570024918,0.7367360631192155,
 0.38434689007118916,0.935739203676473,0.03914106492781577,0.4641809068690662,
 0.7065117439464994,0.46575393164793855,0.766485941787598,0.40718908343380844,
 0.8546754957607628,0.19381102432213737,0.16890471094447423,
 0.2063741529388553,0.1915238137774934,0.7498807879977027,0.3185734575066056,
 0.2801589278126878,0.9699560231987766,0.2416827529039406,0.6709217260066799,
 0.8965138139560344,0.39370470244313005,0.2919684682155054,
 0.31816275064240995,0.07529105069117015,0.7298381838155398,
 0.39503229071143264,0.9184134447418923,0.5055945792558636,0.8157617494157031,
 0.5809094122157287,0.014076212987960579,0.27717657511109184,
 0.8363671198600467,0.22740327417032813,0.8679165481036706,
 0.13116092745903107,0.7130298109763982,0.4603365570206466,0.9489345476126063,
 0.8206792128000542,0.5518050390090307,0.8454907829408408,0.03843467763410768,
 0.5491341931615084,0.2510850173636062,0.8643916034740091,0.9213350118205623,
 0.06792960645794777,0.8152150358938511,0.8635372571042733,
 0.03496203311785662,0.054097864441583265,0.8620573685581725,
 0.2499821024092551,0.8675725998661601,0.8471361922950558,0.8622305749552078,
 0.9982015165137492,0.8845941108600329,0.38147584063546414,0.4765646129977381,
 0.7049740442532002,0.29943648054422234,0.550350988802206,0.6839471008040681,
 0.5535174379018287,0.09360159225482534,0.5879552437751621,0.7797970099605271,
 0.9847494214814957,0.7917835274670468,0.27473264732247626,
 0.18061181153658978,0.4533276504964372,0.7936411498171801,
 0.20350558196659652,0.06251803695246783,0.7248277439592649,
 0.9932983662210133,0.7296852348740049,0.001203908103137974,
 0.6035277441159245,0.4450330913839482,0.7153125497839139,0.6706477880565935,
 0.9331205950486674,0.5861254647825823,0.037773074487559066,
 0.7508585709000176,0.3720461593235267,0.14298280004300257,0.548873300389205,
 0.9424891620026942,0.7355992026008993,0.20866396333544412,0.9135546979491735,
 0.1152310701314948,0.6063625611909765,0.6303560567164086,0.6940938794909249,
 0.18839391920161774,0.5610753293281917,0.6084143193596957,0.5607181289974333,
 0.056052996099798746,0.6512930636457137,0.90423340038217,0.8325071437373922,
 0.29728131481062126,0.7058981756133575,0.28935516536084394,
 0.7851450501714055,0.8571572732929101,0.2859851548503216,0.36892248997605204,
 0.8559953020620895,0.4531501012539043,0.7934604604778455,0.11609177897368927,
 0.15522173938179074,0.07965506118087307,0.7627971844231837,
 0.6793680484776106,0.6595335311002696,0.7788450117183011,0.8637910232501828,
 0.5595712802201951,0.32358296168510603,0.5748589735089223,0.1498979805986389,
 0.5167614980446064,0.030988511422870957,0.7476346200290198,
 0.9200449225762577,0.12845700448934383,0.9341770305482666,0.6976621910365033,
 0.24202100517432112,0.5212508823708967,0.693046602160162,0.08677180312185351,
 0.17873982048706905,0.8120008634680821,0.764414260121396,0.8802507460426227,
 0.18276460824655738,0.8311285973162238,0.6444940620293464,0.4645969286197673,
 0.31720717115967245,0.7780425420070557,0.576874125113355,0.5650267979990848,
 0.4237385505410104,0.2941378060370917,0.3435895831498388,0.9701186846938468,
 0.18053493794228492,0.3898976201465628,0.38447021975235285,0.342137003734338,
 0.545539012429133,0.8969381377302791,0.639925061182315,0.5215490425940713,
 0.8031346206611821,0.5081395138850621,0.46820513337100744,0.7065887362403976,
 0.4594875749609255,0.835358175734682,0.4249167732792172,0.6800408664182286,
 0.7206081688220769,0.8830463390908316,0.7056834612160584,0.16788541619741715,
 0.007999821498650417,0.37202164858797804,0.5145499042695592,
 0.6625011511396572,0.7897406394976507,0.4692443026022286,0.13963863822671896,
 0.3298521644708141,0.4639023520188418,0.6672082876563862,0.983000631258476,
 0.32418733864586236,0.13128024966685892,0.48068301956494386,
 0.902936629872344,0.059316194460472205,0.07192073626160866,
 0.49148363843757803,0.31210813283432326,0.18857260278838806,
 0.23577918353329308,0.11500332295998761,0.7326982721345731,
 0.11321450707645231,0.9776089570928621,0.5292723190604562,0.7463599963457563,
 0.15055974226588154,0.29916826292314214,0.8238651732868767,
 0.9458379227941113,0.522522859727177,0.9680921238322008,0.9850588319112773,
 0.3729436630648062,0.02346195667298212,0.775697966965757,0.7364782365127409,
 0.7484762960934013,0.3726933219789109,0.7178993918400952,0.5718276874189572,
 0.6052964660893966,0.42696448022815625,0.5183187221673211,0.8571764429182724,
 0.8604130481691257,0.8357416357739452,0.21114014122460412,0.581368029393836,
 0.9496596289507893,0.8223422154481239,0.4740634281606456,0.2897610986306849,
 0.16162299390897172,0.9115744841156459,0.7679273459355365,0.2979234814184306,
 0.22794686241855278,0.510672214343028,0.13022268217471666,
 0.43472765055586693,0.2376825294423034,0.8637853302717597,0.8597358768134965,
 0.12597725460891507,0.7029982530922325,0.13615945188913325,
 0.027240941150050824,0.5036517851609688,0.06766765375513761,
 0.395148785220202,0.3527996583615258,0.6533001493985896,0.10159905205804987,
 0.3872414175926614,0.09079002565507344,0.4612319084227895,
 0.27733128731759016,0.8776452784993475,0.7925923263346991,0.8661222810947836,
 0.2027551416504989,0.017241659505058293,0.09006613761808435,
 0.5552193865769949,0.35633295306380186,0.09766142004359668,
 0.6850327492754057,0.8970601793943704,0.7066538298988337,0.6388237449959084,
 0.09321233446317634,0.49042069560499735,0.6844963704278629,0.658056878316517,
 0.7215377788531984,0.5873637720505865,0.9143817950787868,0.7803995279468295,
 0.19942393633400246,0.46957821320776194,0.9231524028922419,
 0.9626753874048352,0.47560989106797913,0.3428277590513453,0.5616176092661082,
 0.44466958534881085,0.11695631758165093,0.4424009215123892,0.790652307743687,
 0.07708813460423336,0.17172311173292154,0.37336534391919163,
 0.38401712463365767,0.7699068247111135,0.6017956764130409,0.7012043822300237,
 0.06595513650995577,0.476522401714625,0.6625475518850775,0.12338207108741372,
 0.29373923388875156,0.07151795345976708,0.04135958840871512,
 0.18789636793165132,0.42403797715444524,0.17071991585702273,
 0.5868518571525383,0.822918968245868,0.5997650140213224,0.36902497179112703,
 0.04939460657606143,0.8877011242367124,0.20587787717557382,
 0.9186185243669955,0.9901183262305322,0.021450852504224294,
 0.11030114334162833,0.4545555301929539,0.5749020083857117,
 0.18237468229048148,0.11772195070082858,0.3509930547380662,
 0.9069418705884038,0.6974043683776412,0.6862177496165855,0.2343784840170905,
 0.32193641576750687,0.5653169921601359,0.6670656818459235,0.5448856731617987,
 0.27297486321074804,0.4821117062594027,0.9975262211295337,0.792312213468634,
 0.021182725795776514,0.8494190021558553,0.7988815029985374,
 0.10278568358743656,0.85879340170525,0.8322333442140795,0.3871383857413646,
 0.2532562089570799,0.31550212170964964,0.554855976004695,0.261454354547774,
 0.7430253860883377,0.2514709640157551,0.4568366013479257,
 0.009076156161706206,0.7678460175814256,0.37095911058408404,
 0.9859368451724994,0.11289385353607384,0.47488774362222896,
 0.39861334560964046,0.28371220775113426,0.9911920418917587,
 0.8070201724240509,0.5859695357351867,0.6589444200082911,0.8293769830894626,
 0.59894285721653,0.5198727486405167,0.6025916547667189,0.5704775625838376,
 0.906195800571513,0.8518639817873465,0.9993200798473018,0.9812270603292673,
 0.559197194347557,0.16163136459585525,0.7504567106399144,0.8682609985789947,
 0.71225651941595,0.6637633977388805,0.45725450731065165,0.2838418153882255,
 0.11164900231067276,0.07393219373537696,0.10892425073301581,
 0.4745399708298641,0.8528417872997339,0.6926306039699058,0.24138149303757195,
 0.3166487785901291,0.2069922170056654,0.6449664477971302,0.8772458598472838,
 0.7785335840959661,0.43953924104333675,0.5787234222470041,
 0.31977899831915635,0.6800868162007574,0.023193413080612735,
 0.021994050089255257,0.492913968884932,0.6097756281729517,0.1283700810425502,
 0.6402424148446656,0.0638968680256995,0.1871280310225576,0.23843375701677383,
 0.23132111664795219,0.7133853669601393,0.24082691855793947,
 0.7698484069980451,0.9757975592930516,0.13044285918570908,0.4226902505020953,
 0.5756818919718893,0.4466568619857655,0.058085642649575275,
 0.9594719101101481,0.8713549431621183,0.7080210603473952,0.06923184580584385,
 0.7696193978041208,0.12775726558814016,0.4922983977356924,
 0.25092929900631544,0.9424688264637299,0.0994959603754878,0.6108791665286083,
 0.8789039226357366,0.5975817996772782,0.8921060908802221,0.9531578669681168,
 0.47053842467147255,0.8095369069404612,0.6904605775837472,0.8220613897843897,
 0.17478882214651514,0.9129583771314743,0.22654462214641113,
 0.16398902744010124,0.9133709586997225,0.5837115333907031,0.8368063219253642,
 0.058856526569757595,0.3943043894196956,0.08621099835071422,
 0.714266503988821,0.8040618413321063,0.197744957966017,0.8904271018374468,
 0.4167381432874996,0.5520696788986142,0.25377322038219063,0.6327530036785619,
 0.7306868061142939,0.45815789886541813,0.7403210369583878,0.7897183393872476,
 0.1255119462476988,0.1019477233914653,0.6093639821861663,0.8974105969276028,
 0.9369998158001084,0.8923529327473054,0.3430239274212934,0.9118983784599625,
 0.47223454970811796,0.5884493927034542,0.21870041309779342,
 0.20220790583050852,0.3408683280867837,0.9937574063601164,
 0.14324640819146695,0.5276118652178372,0.819849961840564,0.8059060462805818,
 0.2975146126348843,0.007462855674271296,0.32196800844449336,
 0.6260966051248563,0.6565991769196183,0.4799520298083628,0.8967003239148683,
 0.14701593320635054,0.34031351436868884,0.7580394977697852,
 0.3577048989309971,0.48295894128892825,0.962727813015994,0.5770877720616855,
 0.5655056140011482,0.8941240733114437,0.12630984367191223,0.7080856149759381,
 0.9924611209534333,0.5044839299699915,0.0014371327561255587,
 0.6267751836899759,0.5485214806740886,0.5423345609154434,0.7374850983049058,
 0.17170743305974057,0.21614234885657257,0.19118225628028152,
 0.2136744574575168,0.18454369224327094,0.975795657498963,0.6413845762998274,
 0.33925665091665647,0.0016458335446430938,0.7865333471277638,
 0.43867882840914296,0.6880833114174855,0.09921801475837655,
 0.9497986346492221,0.7125366034609126,0.8736164170553796,0.17147680941080834,
 0.6513189091602868,0.6704182138162493,0.7121321173561889,0.9546217653589769,
 0.47514258924955555,0.35660305857424035,0.825477725383739,0.7276082737248657,
 0.47787229833254785,0.020683469480358374,0.9638492210942241,
 0.8996229690138756,0.12053816962209929,0.67448389973634,0.8871661677896201,
 0.33299889182983644,0.17700859185817697,0.2366190052299475,
 0.7180222417581287,0.569739392920438,0.8178364449564746,0.8833731706982543,
 0.9214834542571493,0.7545249673491383,0.07971155236406546,0.9632987332549121,
 0.9629006829261587,0.7888266243302081,0.5891598765137765,0.7736574189948819,
 0.35956660946533836,0.9929047506899388,0.08586346306226789,
 0.5240234786506075,0.31445603882517015,0.42368436094497186,
 0.9292233316246419,0.6259594975193742,0.9664755727421278,0.289506539942486,
 0.20759531204246773,0.8302932287424402,0.19798619457497846,
 0.5914310192809188,0.521668175575143,0.42224093728417356,0.4968782486008698,
 0.4716114350775621,0.25073636992135295,0.7268813651708905,0.5825158068864618,
 0.1341244534887539,0.6804554537619987,0.922093212120514,0.7681137109631175,
 0.014040711546240203,0.8986698029328218,0.25472292687247067,0.16017071542216,
 0.13980204790564355,0.9038243914901938,0.05287997621842366,
 0.34380995731441444,0.5675708152122455,0.607954919665018,0.5993696610577358,
 0.923874810606177,0.0458019557066931,0.05069600453445766,0.45942846951455085,
 0.47630226968133604,0.13532887337440236,0.5424021251620577,
 0.44252498118845696,0.341530766365049,0.7146096098008174,0.9130543740913198,
 0.4579715264660501,0.8596645767886097,0.23002015309726653,0.7863257478801802,
 0.22634619419404534,0.727458953813453,0.49206809190629197,
 0.26066661739186525,0.09618815631284927,0.3422370251614838,
 0.5886011182806623,0.33038245390542054,0.988732140732328,0.40985914377495425,
 0.9561289920878505,0.10138275672620667,0.831497931036578,0.45040778331983644,
 0.3612105535941603,0.7994373193404706,0.39474795283370345,
 0.007099214932306364,0.4851054905764569,0.7079947811822965,
 0.9095141140203534,0.5060547038172989,0.7576089906934874,0.6759655593945312,
 0.4194704113209837,0.5850675267987346,0.14582452661178902,0.1824873243228735,
 0.15245198275934269,0.23325303377374673,0.25657520283099355,
 0.8954397914231986,0.11379542404325449,0.13191540553389203,
 0.10454795065296929,0.8279591067197092,0.6123011491715102,
 0.18886811760448008,0.932834530518988,0.12693560104759294,
 0.17005393169057137,0.9881036320034582,0.8024401498514621,0.783019823892176,
 0.3963934265478197,0.7093394869488481,0.29682086451462364,
 0.45251857524146766,0.7134972678556015,0.33319733584707034,
 0.4650505180704195,0.6078327458073081,0.7631813477034376,0.39425027305898863,
 0.6531884670488648,0.0774049960178419,0.4933654785306213,0.18759543023437453,
 0.4768972633136743,0.5239407903205308,0.4953701927740135,0.38464863958661377,
 0.6368291688264776,0.06376639220778724,0.5011181300060717,0.9593983740714962,
 0.9235069517545365,0.4823392284510526,0.692140815762611,0.4163548479252436,
 0.6971840208800691,0.05207070999410013,0.8115869720241059,0.879620348473794,
 0.5732949538236716,0.3659341412422312,0.255953844064706,0.8663714991552851,
 0.7597008386880898,0.6576393282309603,0.6071391282744921,0.20977906060539553,
 0.6759588419176393,0.30636692586848,0.6192360794129864,0.8093540443052778,
 0.7813749054386465,0.5359216670459692,0.935018411868798,0.04897960107202404,
 0.7670303028426307,0.4392904082293838,0.013349654873870409,
 0.8662830744652883,0.7731723549948085,0.7866814145620528,0.2816921546471697,
 0.07170346608990541,0.5787247848148018,0.8942823840669223,0.4851220327195501,
 0.46045752221671754,0.3706538411988606,0.6696136425232703,0.7379637229571498,
 0.40309990351363867,0.3198123573929481,0.3332841559150451,
 0.29644321267877694,0.4965648853284551,0.29667134669564965,
 0.30773490645885737,0.12903930756576443,0.017811679434437955,
 0.5631884857220035,0.3552511015089408,0.4765463119937725,0.6006204150798448,
 0.017369777011586596,0.4095121410593511,0.05611101897887161,
 0.8029259914839582,0.9877376176549372,0.017106841917182702,
 0.8240524818107923,0.33184453288446036,0.7227884712555646,0.979305492397325,
 0.6637785827075185,0.7048725695385593,0.2799444889381283,0.23647189473177832,
 0.5583437876716105,0.9052382776588634,0.8346658394012211,0.6325370313868187,
 0.9063733730608621,0.9465126015410487,0.9354157087343375,0.8097894734975202,
 0.4030899896347896,0.16600414509264916,0.1757370738182138,0.9065454320696301,
 0.07172665366456765,0.7009760461264292,0.2639245720546106,0.7608171002966685,
 0.42034054817375777,0.06119166215588023,0.8119041014066942,
 0.3481751197573817,0.6844374196003173,0.7030998375956086,0.9179459706592589,
 0.47086834120778553,0.6979625336757572,0.6235623090884964,
 0.08789184660740412,0.5331099725625692,0.4151214170699824,
 0.31950563560085354,0.8241354963030321,0.11966023958493377,
 0.03909928495150283,0.6235317950209577,0.7558653034707415,0.3224812417308006,
 0.07385712554224255,0.3622110404969705,0.7589062841082794,0.7433296130402086,
 0.8599766424849308,0.6990097626841214,0.682785884472122,0.47966099995888234,
 0.6974845555825907,0.9146454838132452,0.4241354532710908,0.6211031304332687,
 0.2742481951333644,0.11049930371625605,0.1370416724151815,0.859871822452777,
 0.8604396953940086,0.2101979816942282,0.8626005269857006,0.557217153165395,
 0.6783342455839997,0.7159581220076746,0.10135857761401257,0.3580013056914332,
 0.9023630285007915,0.24675260431094648,0.03055876011109837,
 0.29649359181326496,0.7010963900421983,0.576660855146897,
 1.8655305795256538e-4,0.7839455810833771,0.12195145513324213,
 0.9912628437315645,0.7004598961706553,0.35830381382714627,0.6499826018196737,
 0.3246574371066462,0.714206681000206,0.24203835410300434,0.4750410738477451,
 0.9449040687524021,0.8909672497525248,0.3969565886522167,0.07330259921239557,
 0.6190752393890147,0.08348875507844511,0.9754880704631269,0.2053851622088888,
 0.9575901597287241,0.1243897444343719,0.3032842408056622,0.2616227431860233,
 0.6653146880634957,0.2725698333215316,0.19179315320282453,0.4070796886452901,
 0.3572941306064764,0.6123679041334592,0.004630818999475483,
 0.6738719326992421,0.54507095765766,0.6558781628146747,0.08392929557508311,
 0.6824285865959039,0.4649178685116604,0.5021372594116699,0.7929443850631924,
 0.9235527749515697,0.5475246041845047,0.4355744826594137,0.7359324416368481,
 0.03536387213252967,0.589370505805683,0.6656398887212385,0.986950283571648,
 0.68714555660137,0.21656318555721743,0.9978423090183548,0.4008240768961391,
 0.941573028789437,0.8820269866701825,0.10180706634399495,0.6629469448573982,
 0.934473562454756,0.3868381067708515,0.460805121379649,0.028233889972441695,
 0.6268933266775509,0.5548571545711343,0.1647158327307967,0.3925230544756576,
 0.1409468848684312,0.6371592360924957,0.4912541396199642,0.9681301230261621,
 0.7500252684475761,0.2292480842944642,0.923471000844698,0.48823067416771015,
 0.9556506007323179,0.23386391038286025,0.0275056392878944,0.561600249340307,
 0.9676363646497941,0.6160706049740704,0.369727070351889,0.7059788959102152,
 0.10952029691342713,0.28447748911269344,0.5008308103954896,
 0.8606035996362784,0.2399817265065476,0.4213390558371062,0.5962205188900292,
 0.8548674074885944,0.5179937435294778,0.9438260523216855,0.20609071651033828,
 0.9968312586714178,0.018860477950112364,0.2179510194589328,
 0.3315127527366407,0.6466131073473147,0.0717146014338017,0.49547141724624033,
 0.4882120558091363,0.9056063365307332,0.8745846097878901,0.8426197739245678,
 0.737391083403041,0.7392909049837099,0.9367266706890904,0.3662945263309182,
 0.16394961090617033,0.12232996498726201,0.9046099972334385,
 0.9486526540144152,0.10676753334238309,0.21590817623282532,
 0.09926157528014024,0.323332075634315,0.1671276718883048,
 0.059352756152270536,0.8692663446346264,0.47428241584184017,
 0.3547037049654015,0.8554439934251538,0.4008995567662941,0.12205285118669262,
 0.9196187034814145,0.9007253975762264,0.11728901461558017,0.958161026760126,
 0.6971331058010244,0.30816595688308157,0.4545765646268589,0.7970976322404288,
 0.38312835706180637,0.9916616423157538,0.849288715116076,0.6769227263667514,
 0.7441637022373061,0.8635187617880308,0.3993097642869735,0.6284552956785651,
 0.6583085479904915,0.6986066832601098,0.4322932957869453,0.6103254238382851,
 0.6588813552278628,0.3657195883445513,0.23965099726606598,
 0.10925497640091619,0.8425289822372347,0.4395703682249994,0.7953625199582406,
 0.4079483771850114,0.9183815183563928,0.8112146819613594,0.7759301011892646,
 0.6922490280844765,0.8575246324930954,0.5467207137166665,0.4221029642466041,
 0.47717817795751416,0.289770350889645,0.7204568680391674,0.3012483244653188,
 0.34407539972255097,0.8489697697259025,0.33684712229590763,
 0.17344425642702332,0.11237089878336914,0.7510817426387286,
 0.6739632758023949,0.24372413086954225,0.7660635017121081,0.8471107789918931,
 0.9516928333360273,0.7911318214955179,0.48636981558444903,0.9414524198500661,
 0.9021264913018644,0.6200263565388761,0.0865878777249991,0.43848108979043543,
 0.10162552204895892,0.4354587712752027,0.502347625957476,0.9638361336607011,
 0.597544230589446,0.5324274262598296,0.046419935621720976,0.8435757160590849,
 0.5942747020845676,0.3590388937944484,0.005505495610819144,
 0.3928570632153301,0.31816715741510027,0.9005022084393283,0.5626373109605972,
 0.7644585212913216,0.5587254157148672,0.3709088058396456,0.32741397228101976,
 6.584333612708049e-4,0.33631929895612733,0.6161179734111772,
 0.16127954456631033,0.4981192950585197,0.6825849896542181,
 0.48873640574954647,0.7465678999353572,0.6486023451350131,0.8708809381032486,
 0.13172072568614057,0.3737516111665621,0.8571612355783998,0.4342549870263698,
 0.7004551478504945,0.9908697896463357,0.3030347338427525,0.08345242400418895,
 0.5495871864359773,0.47939715679469685,0.5704885757697508,0.8951234968303448,
 0.7197852732201246,0.6048083031435462,0.24677689912706902,0.9734177469877646,
 0.8346754071776216,0.7326249038220317,0.7540815351918307,0.7439858821813304,
 0.37531116065825776,0.4955337612498145,0.045974686955150945,
 0.8160818772275981,0.0617067006555444,0.5837167064421642,0.01652553110175159,
 0.25070212410513204,0.6227357764660391,0.20865832322095845,
 0.9611963090201566,0.20293082828450715,0.2917164176267222,
 0.04865461439920871,0.1789902058980698,0.2680155844600174,0.7082699490116611,
 0.8470591645721975,0.4750256521929792,0.2968127702923685,0.2738729215059408,
 0.9064958756872795,0.15328043905479283,0.4839539865630652,0.9461887617885122,
 0.5976396618460926,0.5832620263395534,0.2430828730501715,0.9524714982360395,
 0.24451668045606456,0.2512228293999945,0.0938594144481183,0.9353586601366217,
 0.0705522505661208,0.11390112343972558,0.5594970292527895,0.9340893693739264,
 0.3889369690688589,0.9965407429029995,0.8729018617135115,0.771955827412721,
 0.9959101533806249,0.17755955352065755,0.7960947588825028,0.5562958308311962,
 0.19766391311247988,0.8481636903804751,0.816889115747851,0.8013474845953105,
 0.37876690414705483,0.012052446932622862,0.6863219462877612,
 0.06364867118596929,0.5966116631046452,0.2603652294772356,0.7173589024151364,
 0.8277546646010137,0.7680326478842305,0.9774261445347917,0.407849529555232,
 0.81089594987964,0.906973959554102,0.15130586529052825,0.2981949316052218,
 0.974700035345023,0.07318383551836571,0.3335184998097167,0.10875434778943804,
 0.31443044126745634,0.5981094171809497,0.616356167998695,0.3473053264206928,
 0.4552805208580215,0.43231231109078316,0.47934494089097235,
 0.6588341395868643,0.7088009131207128,0.8880697887604797,0.6746460373386808,
 0.864739383598225,0.7435109268458583,0.1216353913147058,0.6645928721734589,
 0.48697365660631553,0.10407926571604631,0.33157870835760506,
 0.5612767197670202,0.491079581644427,0.9018804174195773,0.5272245105111376,
 0.2980650354355018,0.9131638351535526,0.8627963593592815,0.3324761494805194,
 0.9135890504894753,0.707397588798874,0.797776024172204,0.2846329303435945,
 0.7053533767628697,0.7563935435052548,0.40474526976048286,
 0.14298315041638887,0.8813443708515188,0.3169293762529084,0.7135741531779132,
 0.6544789270107731,0.2982556353369369,0.667084605035396,0.10380420263511658,
 0.11216733717337657,0.1345685019439844,0.08379578591129166,
 0.05972586151556958,0.49039310822461446,0.6413411291636297,0.688387427016638,
 0.9060514623552108,0.4748253305899992,0.20044897931274264,0.4405586316719651,
 0.2277589153074009,0.4370604271041041,0.9262359528770368,
 0.041339230018728634,0.6516021473561974,0.8896048311128838,
 0.1948989780190281,0.9286230773992189,0.49802391855239514,0.3262545466966046,
 0.3668591241085698,0.02700812978006828,0.17395325751473933,
 0.23773883210955504,0.7593603546307749,0.9376657740673308,0.3360777319234324,
 0.9382743189735401,0.40728982915484124,0.5455560784262059,0.4056402335840643,
 0.28646675959688916,0.23852386383372315,0.6658327251389998,
 0.8149649508785306,0.24781090621419077,0.893388409576881,0.9144235533485937,
 0.5816592070216373,0.9319789506916768,0.6828883729112002,
 0.0039858493822466645,0.39396568404349697,0.12383592545486932,
 0.14606176568152718,0.7588314403020728,0.8321732254597914,0.6775771172825242,
 0.24204843972697931,0.922091351438742,0.328404427140216,0.08337361566338497,
 0.48351968324009675,0.3701936559349297,0.4484580471390305,0.1786988359187056,
 0.27243957990502665,0.8798017481932041,0.6760850565509323,0.6052926266498008,
 0.6144061358632422,0.7712396387234529,0.15597913691123977,0.7154701627328106,
 0.2254554216950977,0.1569342049708281,0.74371756777661,0.3519166897781807,
 0.6544585101775018,0.1821864421723456,0.4065137075793267,0.6978000929354738,
 0.37335218894018307,0.6924783495605672,0.05897389174984047,
 0.6980228281564843,0.09861250649765207,0.9331404266318599,0.7037618621340024,
 0.25555102887937375,0.6947748617886098,0.35645030206581696,
 0.4743421737221931,0.4077200210611037,0.9276533481578875,0.6895938846476493,
 0.042444282511795106,0.8923885449893231,0.5347211510679853,
 0.9522574692802376,0.8308940518161365,0.05734145609958008,0.6275381748349894,
 0.3691165572857861,0.3483652819566563,0.5252627192818852,0.06824510925337823,
 0.23913421016672554,0.40221492564000283,0.3922755821770101,
 0.2511393776445582,0.6065344755174039,0.6950364994263435,0.32755989736278723,
 0.8436624338250661,0.2792952933452779,0.3994456221726804,0.8757952673478983,
 0.5282287921335167,0.5623935952911883,0.9689675214089737,0.45786185261222023,
 0.3468224537125284,0.897336843132865,0.7542299490895426,0.036659859697057096,
 0.6492671818534984,0.46073748156108707,0.8284243512044197,0.7246423354580589,
 0.33376087937533394,0.17509414777251642,0.46666548236543504,
 0.5379109540111255,0.09935595545691289,0.1317862160926886,
 0.24949168121981757,0.6857469199504944,0.7063948236411706,0.7912531573002654,
 0.011005044763062966,0.9878904047007655,0.35471939195538693,
 0.5320963697493439,0.48500650504371834,0.4286911795067764,
 0.39429296842559425,0.15316385422330803,0.21871790076519404,
 0.03098971058270472,0.5733952954247313,0.10988748838299545,
 0.7233217979917755,0.10919624617138368,0.5534126124526304,0.709918482441737,
 0.3523349844306848,0.970077150450317,0.35056176437913145,0.6227396708984713,
 0.6152308138384063,0.2998292748810877,0.8807155309545713,0.518829095311958,
 0.4078818383149253,0.8386822683246355,0.2340549825769087,0.8292711766742071,
 0.33907678631395566,0.10138463568062084,0.460912052157991,0.1849229384884894,
 0.08709189132242523,0.010441799594339884,0.8983487250602162,
 0.6947879016034343,0.6136963355518387,0.6823427244899187,0.9489865136509518,
 0.7506570154499925,0.7917979571351862,0.2275346474489055,0.20397497328525072,
 0.39505532559048695,0.4583735444757222,0.375628435536422,
 0.044032784798173075,0.49070445641256455,0.16994024667702812,
 0.30686062118619817,0.7225931861749033,0.751943018060272,0.1500970829121563,
 0.10245104655131954,0.7798153114772911,0.41987732931573496,0.515334221087927,
 0.8867728168472311,0.6680213503116731,0.6329385958649493,0.4374138973352073,
 0.7820680653295079,0.37967187152378357,0.41498988317963514,
 0.16661760140370907,0.073903565972261,0.8847040783246062,0.21471739990632288,
 0.8133135034682695,0.9819140592913204,0.5115212639519358,0.6237473453667158,
 0.5749645739677783,0.431701186008817,0.993027840775522,0.12049038308690119,
 0.5135287496906472,0.04052784135990628,0.769300633504842,0.5589253596305438,
 0.6199655407674738,0.41760400384142626,0.7028467787414617,0.3269010883042561,
 0.3629492506959957,0.7611728842590907,0.5941003825330509,0.9315619242614023,
 0.3667520778990929,0.12872648325363367,0.2567127941313909,0.2049798371110313,
 0.8373058446137234,0.15865313920025803,0.29509014988937154,
 0.9100763580247537,0.35515281238508833,0.9921543699197165,
 0.22027499888032442,0.2846017595331376,0.4302570447832953,0.7418451432081697,
 0.4114567628079657,0.8846668905817607,0.9916743374644552,0.03378228492200697,
 0.962272690414912,0.2645154155565391,0.2092167732713378,0.18504494007515482,
 0.5999870995642868,0.8073917543406272,0.17644065023055888,0.6855990571595025,
 0.7414885983723081,0.7008792645182063,0.9084437603912429,0.31783103884608055,
 0.4964970851176199,0.058382724564308486,0.7751221420379092,
 0.25535964451096815,0.06343713668875783,0.8609058636635807,
 0.9010560891876457,0.31323293266191976,0.7520879856380136,0.7238308174688893,
 0.15068788147039847,0.5449558882859533,0.395663005650453,0.3470430841996236,
 0.7543592037815725,0.7944656862719666,0.8591246564922397,0.48281116969291293,
 0.9508284854553397,0.47983959384054964,0.9299589401196211,
 0.43932034768666417,0.8367758010837583,0.5744496779592271,0.6297807934615537,
 0.6012189242242486,0.755741078918454,0.7498329564695553,0.23654171196586593,
 0.3723733435016131,0.9215785594062605,0.9262069475673395,0.7268957870065714,
 0.8944626284693431,0.5261615059263692,0.4411992646978562,0.26371116971065,
 0.07265029993941352,0.4219717400657129,0.8760038140146564,
 0.15383048911056807,0.9796386451399188,0.8520748846121359,0.3328866559594733,
 0.49152307859081334,0.2217042178214309,0.8373680630450981,0.3341888181955672,
 0.9458126659980742,0.5550534498374231,0.12317741808210803,0.7116780407049982,
 0.8213850182041316,0.5899653244405572,0.31572827236244505,0.8834823171281259,
 0.39669930312515667,0.41075705079963065,0.09004837485593431,
 0.6903832493581413,0.1905913626745649,0.6813388531593326,0.9987371422451435,
 0.3472389099430815,0.3120633984794935,0.29661907009235655,0.6468602392234449,
 0.9242353055110806,0.7307921128442277,0.4270781920615868,0.4604641001546842,
 0.3254584228323524,0.0289461322421235,0.509666929788899,0.9585505086564929,
 0.13079891487644701,0.8464878657018357,0.6852933588807967,0.2923818502536675,
 0.2688680487843167,0.9579441430606143,0.16644745635359348,0.9416183663120649,
 0.419733586039295,0.7144723179978001,0.8723479517149163,0.3810285424635663,
 0.365964120425719,0.8944893114289273,0.09458577313343941,0.10258533338112497,
 0.7592707363219271,0.26752891026636094,0.15450499041642418,
 0.4121492327903542,0.25303288767225773,0.09746026790617757,
 0.28904852647703105,0.36903535500255713,0.08967114656486386,
 0.3781933331277417,0.5646534658564993,0.39024490115260513,0.802643561106105,
 0.55474591340202,0.47012145016286255,0.22804361876376733,0.5005332945904638,
 0.0935124753462343,0.42168414620264627,0.19061066988735353,
 0.7188357519319263,0.04271744811162004,0.7165287442391532,
 0.26240608552560973,0.9079300510657802,0.11970933833353725,
 0.3626579524742344,0.2739914215428726,0.08161826967849573,0.348997879611018,
 0.5587497968419453,0.21481049253181705,0.04237036016959295,
 0.07878558528946988,0.4322232369426734,0.3725548178269763,0.7114412622877373,
 0.8394319035596683,0.7556493269615935,0.10599865771096861,
 0.30114627498360547,0.9167374892501781,0.17965575431631353,
 0.7714056966641207,0.2846938964236507,0.7212704423853604,0.47811653486714945,
 0.8052048246302921,0.5316739331714264,0.6981013693413732,0.5479750117518014,
 0.13914615693221755,0.1315776061669166,0.44042510710782345,
 0.16692185656934178,0.37983785083720933,0.14996455065961312,
 0.14812471671315153,0.3718931889878486,0.31897058449154037,
 0.1433754350516956,0.8813814474758298,0.957635424896365,0.6693799963461009,
 0.006709266261259894,0.8987162694917592,0.7524397954644637,
 0.1877942069580153,0.6232332290922113,0.9042952666657968,0.9578842273206881,
 0.754442849046919,0.11604928991894903,0.32243891047597006,0.7650194403500087,
 0.2274981251292736,0.9369948831741043,0.060975863646341555,
 0.8013544723394981,0.7061532564583043,0.5431089717359191,0.07948080236660449,
 0.3096580962631934,0.27464653832885255,0.7756538777026645,0.9719577864839469,
 0.6643364921165075,0.5812707076690533,0.7355047988544059,0.32139247220964984,
 0.8571028078828462,0.8901045620427084,0.4525684499085739,0.31392650125457533,
 0.5717700622998241,0.9929051270702094,0.4740461849444202,0.07213941744909935,
 0.9121615995056581,0.5546959228443172,0.38661735066291136,
 0.22168150579668167,0.1310584646817472,0.8636798553007513,
 0.004593143847229864,0.5262385270825196,0.403149921365626,
 0.21672959589864582,0.4331405434103819,0.46859320766331225,
 0.2518655981876319,0.11216828980043758,0.5389519898482913,
 0.15060126914752103,0.6605807273862664,0.1381881029755857,
 0.15110731531232247,0.7243001051085254,0.7729453132834901,
 0.48125268443549285,0.782905027043433,0.6599699525211283,0.3296223725966255,
 0.8887087854303682,0.782736072900191,0.8002401925933594,0.7957765391464715,
 0.9893783542468723,0.0708017778324408,0.6171201105049584,0.46963190987925696,
 0.8068115435053249,0.7796263798925316,0.6598331093577399,0.20976038288641474,
 0.8861732909251441,0.40790850180770133,0.5886365250463609,0.8156751883012611,
 0.908320474592841,0.09732740064802359,0.47097294249870236,0.613953627548546,
 0.5054635050598794,0.5427257451732335,0.43365361786427703,0.8673443530492518,
 0.12464674955420274,0.6806783624350564,0.06762523730736902,
 0.9738355553135749,0.34259360468897493,0.4213345439375774,
 0.12030685359824922,0.21240273067666515,0.8353903481276664,
 0.4069627550580389,0.8422560730758041,0.4011442723410641,0.395135920162101,
 0.9671277862473402,0.25331918795165287,0.41837402782254973,
 0.9872151800532831,0.7378473660138548,0.037479424912707504,
 0.07059830158655678,0.9199955516714993,0.07257379670312003,
 0.02515381972092301,0.6854516139146096,0.2973649281600478,0.8231961161710919,
 0.41846672830445475,0.39154375561985333,0.08040634483213094,
 0.12871697630918988,0.3769146713254329,0.26273324408862375,
 0.7780742321404277,0.01726831825656916,0.00509925332588923,
 0.3144886997559495,0.778100433665613,0.2732981417460233,0.6881766123620074,
 0.15085800281180262,0.08608413304288676,0.8555734305117151,
 0.8335073330406628,0.7694701347129381,0.668960534014126,0.48544066663957797,
 0.15087923613572618,0.753881100597799,0.7723560041183541,0.25162527496043463,
 0.3524511147096018,0.12497468001630119,0.565301032045175,0.10913707995371924,
 0.5789001973894466,0.39314521504793676,0.17384508944893318,
 0.9982039672711392,0.817537743438735,0.5544278833223254,0.056382321164414995,
 0.369660077774686,0.6296287675605282,0.8608932508311746,0.6721346541113185,
 0.9272990539476065,0.9973145358348916,0.409303172371868,0.44918594456428795,
 0.269499631973274,0.8692918208456671,0.11913636285635887,0.6942333285269112,
 0.4302719100504391,0.4082578083539795,0.2823520925876035,0.9103443581319642,
 0.8168260485898172,0.44006264684138463,0.7725878881220144,
 0.24828221325269695,0.40212715711018177,0.6231882582866963,
 0.5252830067402106,0.07276547611609441,0.07502350941400315,
 0.9634507368801705,0.2639869015772843,0.0115067976978106,0.14773119626385023,
 0.6278370780943895,0.14331446390306612,0.4097910526967009,0.9873022677728958,
 0.8206127499945864,0.45414033832293954,0.42852409903350386,
 0.8320619337181504,0.392344333387149,0.6468373412011025,0.7182639693857604,
 0.4973485900855301,0.08415467744705318,0.3330053103243241,0.292271438188202,
 0.4217238146038821,0.8745400513579356,0.2233685741529915,0.18084065413800243,
 0.9969042039742664,0.6256545922470822,0.6325775577595064,0.6335745934703484,
 0.06856350296490143,0.5184261136267938,0.3797412948961618,
 0.02251580558025512,0.26566659441113405,0.29037535078400767,
 0.7090774786556434,0.6834702911426788,0.7464421593562377,0.5820986753885022,
 0.5820106867815025,0.013086666690687032,0.6560779056362317,
 0.3455201793649968,0.21381670157413257,0.851310186227344,0.5619065471152631,
 0.9752449529644343,0.6267611566604486,0.4634383888637513,0.8228296827356288,
 0.6165575595512985,0.07106492141282228,0.7729464216058279,0.2233085208891723,
 0.28906120495252174,0.8564802769051043,0.45587231550127105,
 0.6297511255677789,0.7600007111357658,0.7916842993219418,0.08843978102459193,
 0.6819655045164257,0.39810463734049706,0.16084362085515858,
 0.48616608720695986,0.6219792784534712,0.6547178685321137,0.4793658254022628,
 0.8121248911498516,0.24966653076290202,0.7174585028720301,
 0.10394053118623314,0.9427703652738173,0.3650028854748345,0.69671584925901,
 0.20445241755803978,0.4625553606667344,0.6724471431293956,0.4721267534043576,
 0.7389348109341886,0.7520323428518224,0.9825760625023943,0.12642483804968818,
 0.6795673717835402,0.46496347175161523,0.14799982388467703,
 0.32443141846885715,0.2826053713882265,0.8203764888415997,0.7726114092351896,
 0.5597003509046439,0.9740669383381949,0.6241522351109562,0.8605429802547302,
 0.9616959700494639,0.24353391829915783,0.18525832881627924,
 0.6152455752784276,0.7931117369824103,0.794302154816157,0.9460243183706478,
 0.996143897832645,0.31590147661018486,0.05712907074797291,0.9008380879939136,
 0.4413094264027313,0.3013915632262967,0.4513261117189631,0.3666260285858792,
 0.2837956304651037,0.31781889870587143,0.8548743898539107,
 0.28886855901559494,0.8889282381203074,0.10639849285120695,
 0.6017694069041334,0.8400462288171628,0.8844396568985369,0.726290701340322,
 0.6547711712630245,0.9254349138861114,0.25032603558340816,0.7451475203542545,
 0.9328352247373519,0.6880534544929314,0.12487720860962626,
 0.20422747416873943,0.30595829639306393,0.1452025114984249,
 0.24866438082423015,0.4593340782291415,0.7510894872878459,0.9131543361262175,
 0.9378682985585106,0.27652728938724724,0.4213336039009391,0.2978964937284021,
 0.4399894922928038,0.07919293185464382,0.25164887839096306,
 0.44599853123813304,0.4191089482335436,0.4813672082182322,
 0.029719974815900496,0.9947322252305374,0.17120633349501557,
 0.9622733566167548,0.8425254817653991,0.5621963287112259,0.6445040744012006,
 0.6330073054411373,0.3668037004453123,0.8584860893767748,0.29543200609531406,
 0.7743870306263434,0.3808045865399913,0.8094490023823467,0.9615748969088064,
 0.45129802628087523,0.6225538209255146,0.825860905456792,0.9754105590492034,
 0.03676485987624356,0.43046737085531284,0.6394878076725697,0.936520212881109,
 0.501136508496248,0.20499051891685005,0.6515060863100798,0.9055993701518402,
 0.6167571891482507,0.2732249795134558,0.6594535691888619,0.9468695179240183,
 0.3855105521319635,0.6962807920408912,0.6268420448217621,0.1056837125238238,
 0.3551676623077368,0.9580683518651117,0.645116731943205,0.3179068218507308,
 0.1803488973247933,0.9883393333441969,0.5878719457764824,0.5771364980086046,
 0.7616250952679559,0.2899993587845875,0.5785197296397518,0.19431193376805267,
 0.8982873941258878,0.07277172517905361,0.5778109763530923,0.241974253436555,
 0.9338144464082636,0.9586217574426887,0.6363606590679023,0.22689066186568096,
 0.44323995993281584,0.891093007749595,0.3914058664969081,0.2851789634900411,
 0.6898908665887881,0.17175465764819908,0.8267758094535436,
 0.002871922421927686,0.14821281797872687,0.35839090110547955,
 0.6633315426986763,0.2557896093442996,0.6761710921296497,0.05363741042017478,
 0.12089448656256974,0.40529856177106205,0.28267049083223783,
 0.5846093549806575,0.4362861585154375,0.3331005832165902,0.25962545400875214,
 0.6579634961025382,0.3865429284500286,0.9854601020034239,0.8371619747265662,
 0.02434435031820681,0.7412632022748538,0.17882383422969617,
 0.0742085048758574,0.8046140806390094,0.5996512301918892,0.12754356511827736,
 0.12897906037536766,0.3798328074469638,0.6152210175947886,0.0571540561782542,
 0.25269645718369294,0.07944272118154205,0.04077472803841542,
 0.5839829397326739,0.4246269011351903,0.4837188537741258,0.8337935046582416,
 0.8076997358954944,0.7444651788170009,0.8280837254868285,0.21894433634408994,
 0.37202977512524527,0.6406543803651509,0.8899971312431313,0.4876894948043795,
 0.13491029990800174,0.8900587486907185,0.13251438679340788,
 0.7800940094384118,0.5052658121842865,0.7666591335546424,0.570118531764333,
 0.6897281888988099,0.9907337211171553,0.9156100239786313,0.15851407725155087,
 0.8538454697056597,0.2209415313231211,0.9360939638842856,0.17417494200900396,
 0.7342998715798409,0.7309675717009535,0.18855562171914186,
 0.08914629294558629,0.20941396341087648,0.5557451255694752,
 0.17713663295125692,0.7035167632616344,0.20816298424522928,
 0.6015541454146548,0.7427924641563369,0.6249248036748964,0.18386346450269897,
 0.3850557003176265,0.9467503711901736,0.6284609486354817,0.4148605714899428,
 0.5011714433572475,0.3716180752663043,0.03543381062903739,0.4296512583508514,
 0.46509504694873516,0.87883056591663,0.2883093157803571,0.04438179462372638,
 0.5877215324300264,0.07492562835992689,0.9819220853000815,0.6461575049198695,
 0.2585233668677924,0.18276985822427316,0.18299571303367135,
 0.5426644109733543,0.6166133524026979,0.2804179880053269,0.8939129552550918,
 0.941114143200722,0.541816768038562,0.3038554901640653,0.7096963828454323,
 0.08281773836800477,0.9373313656290273,0.9550377755431422,
 0.29859648644507686,0.7174094363287196,0.8947134676508431,0.9763269260837556,
 0.7748458975958584,0.982603786076249,0.708039413567578,0.6832942402228063,
 0.6309366754835949,0.0228375878475362,0.9870323034787198,0.302775225996589,
 0.2234155465435469,0.20133673417490083,0.9832601936407451,0.2310957914991142,
 0.049448385481675405,0.3729729819334784,0.04998526535522374,
 0.11135635406254674,0.5475127765157294,0.8048077383177076,0.1506689667483122,
 0.6203527790847839,0.7543094042571832,0.8746674611424107,0.30896040841837125,
 0.06681388033642854,0.2766962591833453,0.6027431794030064,
 0.22761302321533083,0.5626121803139345,0.6124917446112144,0.9432401448013005,
 0.4230634643979714,0.6600765828750844,0.22586180495904606,0.8684837992572899,
 0.08615451296022747,0.0872779354315778,0.3529057335520802,
 0.30875613284568504,0.11933051659469562,0.8625429704880092,
 0.11623754394030572,0.5908013133289165,0.9498403110204697,0.6088585403090466,
 0.8865874857477651,0.2794444542317791,0.8378887165416969,0.6971088465255681,
 0.4289325986733351,0.5657545933569862,0.5904741467806384,0.6111615404791472,
 0.8793249800597193,0.45196299339284973,0.7354990152146675,0.8672863900670502,
 0.7945719530576376,0.32878538102128285,0.9017595781029671,0.5609176587166957,
 0.3736093191113524,0.3050828888326079,0.9596176184224533,0.0987023290278396,
 0.9133249007080727,0.09483290019084278,0.2697525127132504,
 0.009595815030833288,0.5480307663917464,0.6146433661207465,
 0.3706023023774273,0.6210352750631979,0.9225373543212465,0.12219362963691249,
 0.5605334601002586,0.8741919487106302,0.9282640710880679,0.9946727874332003,
 0.23621496596021319,0.9060178106704635,0.13041852022779177,
 0.7525505074627142,0.4971901188890214,0.6290789899192826,0.9020567157749856,
 0.2159517648211049,0.38809532435655714,0.966864837817578,0.3252181294707328,
 0.9524002703718356,0.5930915778458661,0.8189178851379368,0.25321180871866156,
 0.3577472409898881,0.6848169544875534,0.21103419556918923,0.6272873634107626,
 0.5198850056875752,0.6506051713601125,0.3883050683285416,0.7540132050790187,
 0.32221097990127656,0.9565170590753054,0.061651959910032805,
 0.43597162111703425,0.34376846526077764,0.530467725297582,
 0.11866884059055316,0.4708258477817421,0.5091758599630262,
 0.35116834761824245,0.9502794869297524,0.761065537132551,0.9010602029340649,
 0.03079219373704023,0.5504993907258664,0.42792119492880176,
 0.3833206256766163,0.39301374189157334,0.044833337232139225,
 0.7350157300382721,0.8825662470687459,0.7933934604747316,0.25160363671364894,
 0.8538264047415947,0.8941341298716297,0.6644725272811189,0.7946944926902886,
 0.10111178706452584,0.6359702850836171,0.878480032760141,0.8819646250928556,
 0.30267898118148084,0.25972797362716626,0.9057920801081774,
 0.5242261808811421,0.8175228591333875,0.6061782880159874,0.5871442048515654,
 0.8322331747412531,0.1706280812229728,0.35226464734408536,0.5878065363737507,
 0.4429013917470719,0.7853123127328238,0.7463084711738045,0.8626586236456317,
 0.5933177000538705,0.013296134320142428,0.42197131266444665,
 0.6338750755573622,0.2496329086517698,0.8015407646877261,0.9709497761459462,
 0.6447889161683821,0.49558326965323796,0.3060288590737009,0.7042034583076261,
 0.5142386541096471,0.24653680005793122,0.5567952248344998,
 0.05936582882816244,0.6431738767556373,0.6478795642687237,
 0.04775794577922532,0.9933232153191645,0.6864613609513728,0.3148437267265214,
 0.735882955483077,0.04614499002917882,0.35021450912603624,0.6603531977260398,
 0.846124764006878,0.66805246917774,0.5208410956258924,0.8204467701544244,
 0.019555823902819403,0.5337560558735649,0.7709999700815672,
 3.8488326688868035e-4,0.5125750373521639,0.37110607804918616,
 0.1422930748513831,0.76889695250676,0.6703460276352391,7.142063323285486e-4,
 0.2428085403540905,0.07016495882543183,0.7120342108466551,
 0.08927428784293368,0.4769848960701014,0.20788195036922585,
 0.6267375497891094,0.6847683953095254,0.5438876867102208,0.720964743451357,
 0.965361600137322,0.6357155906993295,0.12300589497579872,0.7286193147791377,
 0.3230099769012057,0.9358349942789007,0.25779742795004146,0.9167898805290126,
 0.7227015120677831,0.6997977947614841,0.3959712591652569,0.3562909767544781,
 0.2185757220418505,0.12405802656103526,0.8592515826160614,0.9709299269562305,
 0.4047163456762929,0.14513420857738168,0.15016133161965906,
 0.01049778742559937,0.628882384052912,0.20978240869597053,0.7475225685553203,
 0.6403187250238725,0.18069451463751784,0.1503787607009469,
 0.07220553493083381,0.15840821215547962,0.35089372343433,0.7276933377049882,
 0.310535099478768,0.7227184049828906,0.6052919272094918,0.5100230253934157,
 0.9639583769334078,0.46614707621622375,0.34905288455935224,
 0.03542908031828296,0.036907617683164196,0.739033612046931,
 0.4307498247000735,0.4747619853649717,0.8932848078596098,0.32381908056148545,
 0.8384928637895501,0.32428330686432505,0.9534283437781462,0.5022203377282218,
 0.5641049989953286,0.6827031484651561,0.24589725280313035,0.8782366281958031,
 0.6204471690427809,0.1105444489624503,0.6407904997025234,0.33735542302357047,
 0.38692999927036564,0.5784690511571964,0.9298177272028358,0.7324073572167082,
 0.5467923703436687,0.10403995893139328,0.9695351079751827,0.8229572338729573,
 0.5562656231619434,0.7654544119739698,0.03061356636046353,
 0.19309424319585466,0.11453517972883798,0.5991318693041435,
 0.2840578494661037,0.8680937933068594,0.001583641721685014,
 0.8300039580771865,0.6545102378784017,0.9255533461957608,0.23744414692100946,
 0.5582325124047741,0.17754466379419376,0.9198612984323277,0.4329128002152538,
 1.9575981346575677e-4,0.7373066234196641,0.3081052682677061,
 0.21609104473439755,0.37149903299328535,0.11799967908913489,
 0.6026432370885204,0.912491457605302,0.42095192475391974,0.5753024923567691,
 0.681274767200853,0.4005746951353606,0.25495147916645844,0.7137056229570456,
 0.7674294442254477,0.928363370068956,0.3179550698635062,0.2828626969285508,
 0.9370587677493075,0.30136488831531927,0.9347103154736114,0.7428735144367354,
 0.80353191542995,0.019006668883370326,0.4937384441158903,0.8137067869085991,
 0.9015061143473809,0.7138900742746264,0.48453726801319574,0.633343066095252,
 0.715330020647252,0.7581363941831872,0.32176703276308816,0.8647733590735227,
 0.28278951847219624,0.4720711519351678,0.008522605005618855,
 0.05414192213070701,0.9191890686819046,0.9574951759289959,0.8512117353817323,
 0.6086793877161009,0.20132338385808712,0.621516578946986,0.2507309622194652,
 0.2975437787750199,0.17073361753096217,0.8328140057638276,
 0.050490430880322346,0.31457398892055644,0.5527270159790446,
 0.2126445463510942,0.14968634586641882,0.35499510802230283,0.64205906777947,
 0.9498759367834093,0.5428147563703272,0.7709465959710374,0.4355799344618718,
 0.8183085443530196,0.16015745038675866,0.5869596135797203,0.8870508713763084,
 0.94650805988916,0.21581729598789667,0.842284615884986,0.928009605393246,
 0.7220529489837556,0.7568393743330797,0.731638991783414,0.24396761278048662,
 0.14088867913202396,0.5642193148753002,0.17908113071947285,
 0.15142020306335757,0.07417591075511254,0.23061794727845075,
 0.9726911226060564,0.6897278788656412,0.4483393303570744,0.8537676429158296,
 0.9535776203481711,0.24883194054904412,0.304053067838597,0.5157347990213771,
 0.6734048950798261,0.2159861307410741,0.8382530358174254,0.5154923340500208,
 0.8121377045904851,0.8330529738080873,0.1405861979754507,0.9108144287595648,
 0.9935411523895471,0.3533705004038932,0.10822202022142358,0.9932829770136633,
 0.2928590440949651,0.7963798818083747,0.2085545606823641,
 0.040409620228305965,0.8012195634589976,0.990639802928763,0.5675412493706894,
 0.28699260583862085,0.3117824997403027,0.32510117509755143,
 0.1693914883910519,0.7698976209677966,0.7265223349328258,0.5928261176420925,
 0.7865374822662412,0.5054511650208655,0.8170493912624723,0.3326253280268293,
 0.09342711167384743,0.6314225230554612,0.6084397642741237,0.7862347913632958,
 0.94092025401592,0.1743146368550521,0.8808801646931214,0.04236702599206765,
 0.22488803213443054,0.6673592049068275,0.33961417280816675,
 0.4815525517378968,0.3323370913308611,0.19811897630705644,0.7447226642968233,
 0.17472457135010422,0.941602763890028,0.03107823762629025,0.9063162297591836,
 0.7279583907493096,0.24724387971859185,0.46710711676847616,
 0.5382378839954234,0.07894904963336602,0.6849737535119751,0.881619107174064,
 0.9390006106293892,0.8504257787042651,0.13552917526355202,0.76804639206357,
 0.5632087864903895,0.8127213671111421,0.011768073250045141,
 0.7453192558109761,0.5504665955414902,0.2436308084855846,0.3985896986198516,
 0.362718954794113,0.5663488681847563,0.07419073238668239,0.7775139467389286,
 0.3309674127607125,0.5906730651368504,0.28212751583851925,0.1786613681959569,
 0.34805359665910585,0.8421948532573406,0.9780123927081298,
 0.17365356611329086,0.19361049324583912,0.4174094000824077,
 0.3759807504010967,0.8233881754530941,0.9439607752620178,0.2652135447450398,
 0.18709558267810122,0.01226955354530801,0.5256195207838319,
 0.8441485343595438,0.5768658962929073,0.2854757431683159,0.770655193289078,
 0.09825262414900515,0.35452365249638595,0.9600970041543485,
 0.5402046278024685,0.9552632373672099,0.42857038817090465,0.5473153564914923,
 0.4177865658302711,0.7849107057706223,0.5723183222023946,0.9578155047263307,
 0.4170990772038897,0.6794503253156893,0.18944270185555334,
 0.31040835168616165,0.47118789761093516,0.034390581320638214,
 0.9441359233883844,0.1639857706011838,0.4718903259315739,0.3654336034461968,
 0.5960052214002748,0.9797721949714352,0.17089792528714198,0.3388405520257084,
 0.0022304176717217183,0.6848824415986599,0.3360419645743107,
 0.7497605558963387,0.10324353605973391,0.08036450425059305,
 0.3004244709163518,0.6008488482699449,0.6530835978798025,0.39184579689923316,
 0.12433673755706509,0.15414715693187941,0.8635628904876806,
 0.042369936817412523,0.8781344384977636,0.05371948582564978,
 0.5099243250246985,0.250980886200577,0.19163151146138135,0.38837477412823607,
 0.7010668676467618,0.1538423018163848,0.6084274042118055,0.8691111387404259,
 0.7177916316889925,0.0999996570661108,0.7177576720204544,0.5665879399234685,
 0.18562308917572112,0.10760899183968353,0.2835897418077463,
 0.25642065271253356,0.2852050771553255,0.6643801285196576,0.9483233399474146,
 0.4315986018677994,0.6967285989800427,0.49770571152033305,
 0.37896756628817485,0.1538315067636844,0.34154129012663537,
 0.6190355750330255,0.4946670130140257,0.5050152835184976,0.9369956181698343,
 0.08873169314476148,0.653314228580543,0.062143030284602996,
 0.5955454059669705,0.13080253765573546,0.14224125706835888,
 0.7260227827171158,0.2032825707542888,0.17006785386528234,0.4907409794037375,
 0.6956521559121924,0.6079431007864943,0.6093579548737388,0.0185737259940727,
 0.7748492115506869,0.4111633352636388,0.8112119375457192,0.48119070167934697,
 0.23281423897894993,0.7537043889110757,0.8276153140386378,0.3966302228428614,
 0.36743390816851706,0.541475405548521,0.9859854016499393,0.980352302506339,
 0.6128769247237896,0.14945052316586416,0.6021765035519892,
 0.42299792121649404,0.47161489708395565,0.7137881373342503,
 0.07357315510564655,0.9015376671580455,0.45698238586661966,
 0.1451780801362571,0.7422187683873324,0.8025472542898929,0.7088042343713702,
 0.1319940764717944,0.40413982123426495,0.6782708887417089,
 0.27132460249240664,0.5923160288371377,0.2425737121505872,0.9122447383857677,
 0.5342727752796641,0.4591634426743195,0.4103518924573465,0.8281507224797,
 0.8584351937458098,0.16494420613665917,0.7425405164638569,0.5764358667770859,
 0.49427264282191663,0.7763410606618728,0.34531993880404155,
 0.37463508396770995,0.15906218892536272,0.9665685591064347,
 0.8338643812532274,0.6361560376096523,0.6227218915443551,0.215622049789024,
 0.9394211843424611,0.28528310853137095,0.5106908614082422,0.8468363456465192,
 0.8314897870196364,0.13777772292814428,0.531171755404046,0.06386654883522147,
 0.42469172552328005,0.9027866992354563,0.7384376251586566,0.5344207131666074,
 0.5410496869892455,0.35216436179090593,0.4162484399391593,
 0.23996162846502822,0.20580608100301379,0.18031764101264358,
 0.8676490922904483,0.4330436441307908,0.14907168719430874,0.2828699946635155,
 0.3913531638398777,0.3202457243082867,0.7014030641741766,0.3052453695597307,
 0.2190782520903075,0.45174781453986756,0.004005755410817491,
 0.27559885009582175,0.5900556916444871,0.8792021239957578,
 0.42599877412770093,0.957713351452143,0.2252180278873659,0.18451627548135696,
 0.5937079219166094,0.01489785514251063,0.5464515276116628,0.746276230466166,
 0.06138260316616806,0.34484928193212905,0.9527667726446565,
 0.9895296021721768,0.21469620060746297,0.5071336351613698,0.5114913232327505,
 0.831627244802763,0.5319813301498509,0.13019473521610447,0.3113871587356467,
 0.40667729103284433,0.4115503998803268,0.8680944999315545,
 0.34683222188242324,0.4206820275476739,0.7069317853469537,0.839236586160107,
 0.6192740829832521,0.27288244294787567,0.3200658480663706,0.441011548446003,
 0.9904690067833681,0.03796648050886142,0.9418351220768331,
 0.11524603210152518,0.9071219306710014,0.9775297789393337,0.6644506325744268,
 0.8136884797360329,0.30479577634428034,0.07631424298633305,
 0.6246164729879865,0.9348912777466056,0.8239787690021468,0.8495474671857541,
 0.02814824453498599,0.6157920649535427,0.1648080018506679,0.9024090423142382,
 0.16799237837564562,0.765227847000285,0.3545260824363684,0.5745513187661448,
 0.44993876820257095,0.43995978849831674,0.6787268451470315,
 0.11452455694096508,0.21800758249216146,0.8607742254632238,
 0.19610380244720116,0.6628995561618082,0.47650600953903055,
 0.9187895885322327,0.660797747658987,0.20804158140268503,0.614185438251157,
 0.12093918495828015,0.11645722271502934,0.7834087718023715,
 0.27136393917543566,0.9458925197732235,0.5181196361991298,0.3911851008858299,
 0.3393622320450842,0.7528901275954053,0.24235436057863502,
 0.09158420801486045,0.7694740073452703,0.9343562205775459,
 0.09615582034131165,0.22937616044616282,0.1381230655622171,
 0.07754509848741853,0.11005372668310154,0.05180804974373332,
 0.5833263521197886,0.9573896839487186,0.9680277332817175,0.5496397960271399,
 0.9578111427185326,0.8312351603293943,0.2473167549901032,0.6456524982803958,
 0.2082994598316854,0.3318918116639008,0.4021888923649929,0.5402206790701205,
 0.9129326209720163,0.5883204858428407,0.6977093973954928,0.03550977226782881,
 0.5514522138703332,0.054835372227911794,0.03113353079612191,
 0.9276879508224468,0.5001254824440264,0.2457838331324247,0.8342724019243974,
 0.171096756095931,0.17895036219557747,0.3922647268715356,0.7320007869645048,
 0.8527433890995924,0.6899947420257169,0.4851796332059891,0.3706173114492137,
 0.8378524176632943,0.6003588578220225,0.30587614294357934,0.8832698688591843,
 0.008994594053791993,0.7319042325564198,0.4480269408002434,
 0.14775793952669058,0.5880039090926595,0.7483372307561886,
 0.32624520322092776,0.6791913499445602,0.21742067003190912,
 0.33571702248289026,0.2732250681775251,0.5559804603264347,
 0.18208268605139977,0.09991395655207302,0.11860325095743074,
 0.2779821286030061,0.08628608438951824,0.34422786632229063,
 0.8049095449602465,0.896698693637434,0.4848649708144762,0.4234560674085742,
 0.6357618215386609,0.7011471447779299,0.7087760686297441,0.00440087332707495,
 0.6673514371823202,0.6308555096550251,0.6514640228003442,0.4912547389673093,
 0.484859064979663,0.49733679475320147,0.8662683697929193,0.4020541568095982,
 0.9837070213910224,0.10354166480356575,0.3234306717713099,
 0.14458087656664098,0.5580944831513747,0.04366030816028699,
 0.5921289119899334,0.3062737744711068,0.5170526972369195,0.02990946472686884,
 0.7376071201489438,0.2977508987947971,0.6208512169506094,0.2304788639673374,
 0.21823187826065948,0.1050437330123224,0.17232185285359658,
 0.20959680859588836,0.7860045142205884,0.039357776549868606,
 0.07227095017529361,0.7307630006796182,0.10196971506435182,0.897748357064323,
 0.01882265987362053,0.5764515700459211,0.7613083797016389,
 0.12586175320763504,0.7980616137236766,0.5326124067283525,0.6503086403903262,
 0.5972748262486681,0.7001333588309872,0.3090207867079853,0.2552080872262581,
 0.5279361617623974,0.017212390396987853,0.6181560064668461,
 0.5167672721158127,0.19695055247803295,0.8190162752156543,0.740998250625551,
 0.5936643725635119,0.6431578847238211,0.746369545987022,0.28313811293979696,
 0.3972224035265284,0.616856813163974,0.6734145520260219,0.6731722437488323,
 0.7018482877054919,0.7759281312490962,0.3143918601352792,0.5355869771132032,
 0.29170096287854785,0.7692028456838568,0.10989763115352058,
 0.3858412646661422,0.6039346684878668,0.8499213504616,0.2268780576616124,
 0.44938751904582175,0.20481240629744946,0.9310920383202684,
 0.02055076010895307,0.15512403200832092,0.9214510157049018,
 0.3348943204946594,0.7064440199170701,0.1059727035309026,0.17790857517456804,
 0.06331108115928386,0.24645511125509012,0.4623939842521181,0.724779501071211,
 0.9940278558981338,0.8551459965999172,0.3132208146143134,0.521660295457868,
 0.5362242458935131,0.7694968567047688,0.23425908695213893,0.8056553813001666,
 0.692241577223769,0.08196962151743636,0.22292346188844747,
 0.26839041762315574,0.61521058530545,0.9780169142599167,0.39889367458249836,
 0.027667363989297145,0.8470311172089023,0.4065629537218164,
 0.21223425253999417,0.9631987068251987,0.545939237782938,0.12472523238622524,
 0.7954369732689592,0.5174750702597493,0.9991937239273768,0.457898363122929,
 0.3942646308858968,0.5049131907908715,0.549297518909577,0.8133133798575303,
 0.22476330700379155,0.019498024976136064,0.49073765867245456,
 0.4564046781119738,0.062180229711826795,0.541979204449385,0.8982530392144907,
 0.07128259347536503,0.9404132980893029,0.19250891611520893,
 0.32479095176133876,0.47357229756425734,0.8329738743665782,
 0.3966341559297101,0.29877115628311357,0.1842329175983639,
 0.25375357861930237,0.9270731391034118,0.0026022357410954156,
 0.07130835418747461,0.9530752798953167,0.2913392740181069,
 0.011626735897798723,0.5949788023036376,0.03773017570505233,
 0.005717583142656135,0.17211297354240374,0.04072459058974065,
 0.463741225962667,0.16102822749215995,0.6872207918498244,0.9754827760631383,
 0.5126371721517063,0.3251584900960127,0.6002508769757716,0.5088551302570352,
 0.4891632151185805,0.023325376152441568,0.6117480281452721,
 0.6367436341536173,0.6331032500511399,0.6205640568386682,0.2286981206120665,
 0.22830062074613444,0.3216004188453382,0.055692445183215034,
 0.2279501687735952,0.3756746735472859,0.6311731425152176,0.7133445643814101,
 0.8305366674489003,0.9779199300975728,0.752150528343651,0.6769833765522294,
 0.5680454883411048,0.9224113900541429,0.43672130525028896,0.160051924214608,
 0.31072797691001897,0.626978459210521,0.6954760816068215,0.666718944932345,
 0.5400908840933807,0.1605892175069492,0.5480551675826011,0.12928541511659908,
 0.8989900452752525,0.45218263893318134,0.17783481416979252,
 0.9968616370774104,0.47199554541295863,0.6530354888119233,
 0.06034147488019426,0.8943483242055916,0.819073087145142,0.12985968333064313,
 0.9870123110844635,0.4951343958403406,0.12404441423968371,0.9379271327551414,
 0.601785937351345,0.7150640576608108,0.01013912585254162,0.5734722603055948,
 0.05037531264446082,0.024663750488990055,0.8572916221820956,
 0.37197829021736384,0.782388046464835,0.5014766074712922,0.8564749098725126,
 0.9569237370844876,0.3709745792381527,0.5583132768963188,0.5776615541928255,
 0.134315848982402,0.04396621793355704,0.12414984520847971,0.4880703199323928,
 0.3341235061897295,0.5356335041861311,0.21044283673068975,0.5042573148060412,
 0.21850429882319788,0.03669282328500334,0.6246477915123438,
 0.5635458612485551,0.4098126163222471,0.7518609624546057,0.9258175669184651,
 0.35824030041551147,0.9377874829279378,0.3228287895526297,0.687012654068726,
 0.08272268905222246,0.31563337819867765,0.43585052286167913,
 0.21948018687752924,0.7348251135452122,0.9623982102480136,
 0.18765282085579327,0.7268288317424909,0.6127485221579854,0.4264485504973936,
 0.06271561329158792,0.7240151673089089,0.2923760636718644,0.0815786185319679,
 0.46263366397285444,0.20241875543787613,0.7214895648729167,
 0.2021579418655528,0.8052894117995216,0.6895618006635181,0.1879053428236792,
 0.24611793128492265,0.7610543760566735,0.2993406349092552,0.6304352202503871,
 0.48020614961817454,0.6397628567248476,0.622593931286673,0.3115715060443206,
 0.5472688499343681,0.6110060289500046,0.7779741637101273,0.6592126192192447,
 0.15219779974085224,0.34543836539112527,0.6937584862138154,
 0.12494297289813083,0.7854714879491683,0.5810158141576699,
 0.47603946600534397,0.5804311017192105,0.5260342009804218,0.5681948645866809,
 0.9068433325785149,0.7486269743602902,0.2974482415994725,
 0.050729975322211684,0.6726340688183784,0.5190080502177269,
 0.8843522850185301,0.4162675412068262,0.9421744513833886,0.03340827999310714,
 0.38658494598385584,0.029264513751635635,0.8723015873660634,
 0.9032868857617826,0.6168445279253051,0.22108622533844935,
 0.46708804709876617,0.19071183496029342,0.9640923876699979,
 0.2661134677779453,0.9874946189356217,0.5139132765280386,0.9038281918913208,
 0.8254296456173383,0.6273038868039872,0.6447931893890827,0.1267099314894221,
 0.7528664123031871,0.03532842011944903,0.7632740427959096,0.2271781996983535,
 0.7832749283972429,0.7569613888411364,0.18608748737050496,
 0.22302885870870615,0.9869007892719542,0.36239609260731476,
 0.5087898807768947,0.8860500022672142,0.99603164404858,0.04631148828369458,
 0.12387976673387158,0.37717434955359286,0.8393330736434232,
 0.7059606375225793,0.7851500231876724,0.9472931293449623,0.8592495832039729,
 0.04856398678836549,0.054493876186013424,0.5261815661850937,
 0.7280352358042158,0.16707430148581348,0.25472578929927947,
 0.3180057703905681,0.8926896784019145,0.15536578708037496,0.7593605171844187,
 0.6250938391367206,0.9561447287364472,0.15684018398297817,0.2706407127280962,
 0.45443981571350955,0.300789943494973,0.5386272135374501,0.9822939359886516,
 0.39687458848648194,0.3264255790234366,0.827966889989852,0.9048531939452772,
 0.27537987814076315,0.6358576621628275,0.025351744211746463,
 0.13749062079155994,0.15426553508117336,0.18769654858159812,
 0.0581623998279831,0.9669310373307921,0.7096459827840137,0.14255967844132988,
 0.1660374668342418,0.45976769343354384,0.7396105418043055,0.9808313539181875,
 0.5921108719727787,0.6674291019650874,0.5163529450197126,0.3614596622459947,
 0.38072920341646843,0.36652104022508025,0.9396025535659307,
 0.7731773517468171,0.25155812893813545,0.370516896411453,0.1096586365002239,
 0.6507580944255663,0.6865625149268912,0.07040511237739377,0.9177532103435669,
 0.702444919983071,0.8929357037436144,0.43635302915709917,0.518011369524952,
 0.13005940737510424,0.2504927342205482,0.5887684512666518,
 0.26613688605062047,0.7057895889447332,0.13897381843184753,
 0.8941006001858657,0.7480231813790477,0.7020147210752791,0.22816051182775043,
 0.03810304859679503,0.12731138189089308,0.31211357422874897,
 4.386300446250768e-4,0.039577746386369794,0.6456930269891042,
 0.3971914860329704,0.945673242346615,0.5876531263884528,0.9798612759951094,
 0.49772929624680673,0.5472938195340112,0.2743464732637231,0.8131920095249056,
 0.8740629713220037,0.0428450002640729,0.666169090972523,0.07746795733023237,
 0.738699618464133,0.15799229787253488,0.9980709742433912,0.3633454569463882,
 0.15516204330004868,0.2908962429329064,0.16950038504954168,
 0.26579743504012354,0.6535033882607291,0.655600613943617,0.9375220481553441,
 0.7740820241144575,0.11800101530671503,0.3070970650747942,0.2349730701895345,
 0.4821115041333016,0.19582521573920486,0.8715493800059567,
 0.29677021537522563,0.3227435147209726,0.8466878202209829,0.8033474374590068,
 0.011976285147533128,0.9052251604373589,0.6978852068307543,
 0.9763598499891382,0.02153857743296228,0.9029245497058103,0.4567517312592563,
 0.047509394930732585,0.058090507043004114,0.13070826077217268,
 0.9602650958960408,0.3606209328064839,0.8604007550506244,0.8535725689014874,
 0.6941285473425648,0.7737768542979852,0.528863107229959,0.9157662733944859,
 0.44216628888352605,0.4286465194569413,0.1203979355460637,0.8276072391134123,
 0.059648744280009214,0.02319069061769441,0.8512277044941584,
 0.15509295050257255,0.26298047791213786,0.3141532194052976,
 0.034361882622405915,0.7849851337160887,0.7262252250717688,
 0.40487480565488077,0.6745804366391237,0.299381685355974,0.8934014801950911,
 0.805099172171445,0.6181916273393733,0.5797953714938887,0.07987223909221364,
 0.29944166737101396,0.13201116147230285,0.2980127865035206,
 0.7241938491877913,0.2931244547137347,0.8939739458781866,0.2216460572756651,
 0.5591692765585391,0.7352884640485602,0.18052849653633496,0.5488930713610226,
 0.5556809298142126,0.9108429047868761,0.8658110306836498,0.4696134675596334,
 0.4418800568642194,0.8285936965654636,0.6587658635034181,0.28502348917640274,
 0.9631902683350254,0.9159366698787487,0.9933232595635918,0.5052270250408722,
 0.5510138039139145,0.8968257943742947,0.5886752843150541,0.8991388488362465,
 0.5620918862060976,0.2477148911827438,0.9029226120492726,0.5749476964934173,
 0.057572731506531305,0.432361175770422,0.8397713133908432,0.1494475231675061,
 0.15685340976812534,0.4943837453371862,0.5910130604443231,0.6642566414948093,
 0.5004383582947228,0.7634603122056555,0.3615252598931664,0.500728114505388,
 0.1712493302318444,0.16427662393129405,0.2941755081513926,0.6330725210133449,
 0.3925547417719152,0.8903036711590383,0.06483868120128211,0.8309872932881198,
 0.6809150474427118,0.23731566904163692,0.2437276840054753,0.5451484848808529,
 0.708900820003613,0.19725309370836974,0.6320356351827866,0.44443981489706297,
 0.9264540898460134,0.6284806579888162,0.8982576664524407,0.25605572776553565,
 0.6422625825760073,0.254066333141838,0.7535342979562619,0.3385158880121475,
 0.9587840303251126,0.7076031518996566,0.246100098104031,0.8131761786982639,
 0.9576616070888362,0.0739010391153282,0.13163447620644875,0.8887352385792474,
 0.3779696566029639,0.47148191296383557,0.371535700453259,0.4670184354363357,
 0.12127292694250225,0.7337173461609372,0.056060755413699925,
 0.8451962887713769,0.2824076850074164,0.035258811327887774,
 0.9623381245113558,0.048104113409543126,0.9231961067155964,0.18409989113366,
 0.6100458465607774,0.9847811463253405,0.8253909385804501,0.39768318667753866,
 0.8086318677550918,0.27677546156465727,0.09969047466555259,
 0.5287417265675758,0.9256152947744934,0.8058515517679034,0.3881379046477905,
 0.23194507970158962,0.345303558032404,0.9855268795567675,0.19560467358621658,
 0.6326692542945623,0.8819707185486803,0.36240678675875015,0.9553931241266506,
 0.10770794933283001,0.15109904502586946,0.11177652705613861,
 0.5174686354111457,0.820362343686546,0.5520052086963699,0.2670775910998826,
 0.8112952135627434,0.7336124482000786,0.10008637804656184,0.9103148227038265,
 0.6986781983238176,0.2673851544186291,0.5810835145917173,0.8634478211828812,
 0.4206430692273896,0.08438142823599892,0.3303184675258126,0.7535542561320552,
 0.033539558314123696,0.4243752334764097,0.28391251206332746,
 0.5454094352716489,0.6174848896976457,0.5083767631936583,0.9594337539962353,
 0.6407606599139429,0.6078886585249882,0.12349477294711175,0.3743400450034544,
 0.633349264694849,0.8813516682778555,0.31769446280396774,0.08684657028216702,
 0.05128979979245629,0.35548549680540265,0.3228909049705664,
 0.16160839800155424,0.5889362736765572,0.7303929899140923,0.8066601506089762,
 0.23958486734225048,0.9331941002908983,0.16651598016573887,
 0.9965518525325963,0.9540708512234355,0.4070266386506003,0.17156031954884376,
 0.044979526765039135,0.7260987445880132,0.8610080155186187,0.618347432971911,
 0.9132527370537982,0.6263323836879517,0.19734389350356119,
 0.22751376839479986,0.3605113786620373,0.048267065366299544,
 0.9937752551254204,0.04842575217397238,0.16403384665010268,
 0.9602909324921644,0.966989885047469,0.11153169110072259,0.8306078619092169,
 0.4071055232780849,0.7265744059301515,0.9016226685263067,0.46405596823941064,
 0.5197513513324062,0.6932725184320805,0.11773134204934743,
 0.36810459818361596,0.5167503565005097,0.6303686697172511,0.6010394584508734,
 0.9366780105746109,0.04906202935690396,0.7569863188092919,
 0.36519271903112815,0.4479855802811148,0.5842934577433609,0.943450348656949,
 0.7785030359069423,0.6965323114690299,0.9255489621336815,0.9559656220725833,
 0.5556969129263907,0.4441487101896009,0.1999119549884555,0.9580946894096961,
 0.40418567736427546,0.21644539304110577,0.30095459195897534,
 0.8200750706554334,0.8358442643899568,0.05972364327210822,0.7847130411428237,
 0.7724094189335112,0.539735605739832,0.0787903415864818,0.071651733082151,
 0.6286136340103667,0.3358917854850657,0.8004660356818496,0.5332201502711238,
 0.5359680561388536,0.2873956929494641,0.4598813392291379,0.6701139012146554,
 0.5465792782175972,0.511176442215413,0.3657861974471035,0.5141550194007313,
 0.7440399678461589,0.9280178184391552,0.37548595808684826,0.8858832957981945,
 0.5667431944352893,0.17183704392427213,0.6263013363032242,
 0.047122738977792955,0.5657735604480488,0.09163190800081533,
 0.2763592553980244,0.6770297573349573,0.9363916720822958,0.31218338161525805,
 0.7353345602981327,0.835998563818977,0.6004016893562519,0.8459735126007235,
 0.9245858026559028,0.3279604078794547,0.07743499231935558,0.8585543222231581,
 0.039241472061182,0.15968795838429428,0.26170813786330005,0.092768490322642,
 0.5449632105208538,0.7297562691568159,0.5614371346684078,0.5538546120718725,
 0.7782147180240957,0.05914128387013695,0.015916212530276197,
 0.798569266850291,0.07568065811629088,0.26241309537310276,0.8744950933094261,
 0.23465271488095707,0.5624151833780211,0.28724373756948907,
 0.8427672340661176,0.8412864100771796,0.5112938414776289,0.2999087728689832,
 0.9828564564167653,0.13363570313797257,0.08146466585066581,
 0.6967622839179414,0.1288915459945641,0.07384842437429961,
 0.06689468694875544,0.9284456339635113,0.7973459441666161,0.6992287472636043,
 0.981728336490861,0.4154965131290953,0.06102475379613592,0.23257607154800142,
 0.6975135773686931,0.04770991906719324,0.07545747994234508,
 0.24297106954809877,0.6389122054373108,0.25088843554259344,
 0.2615041630225926,0.6552705437607783,0.5290287103249789,0.353448958193445,
 0.15943952328001343,0.4045898107760395,0.6374165373241869,0.9153579971555378,
 0.41264689732114046,0.5931495856544066,0.021073976490865287,
 0.8336403513715122,0.07452630678021044,0.8830291573999598,0.3932443183205643,
 0.9719761512592834,0.43781554297881264,0.5501850219603793,0.8341520788768136,
 0.2305650479238207,0.032609433058333526,0.7688294968888416,0.738798589405131,
 0.19833809609337705,0.09186837663889824,0.5065373674045212,
 0.7113419098653222,0.5383996135273477,0.16177319155756176,0.7908132398746556,
 0.3989297434636573,0.6150710894644085,0.08759511892069682,0.6501959081119202,
 0.31085395798102344,0.30179700858136294,0.0770870452219693,
 0.49616330122764596,0.344327530187428,0.29678573761163485,
 0.40267805075949015,0.8985480556603797,0.2396252167291898,0.4560026831863614,
 0.33580422309759006,0.2165022790826181,0.940277483615777,0.8188007394810918,
 0.16606532093278614,0.335510881223659,0.7814000584758954,0.24934493969893534,
 0.9297009471406873,0.8408031476216137,0.4861105323432203,0.7906738673710565,
 0.39919593228999306,0.10031096140756157,0.5099032890627682,
 0.7669089935502279,0.2749644188343192,0.7986518060502277,0.603403831246129,
 0.14990806944312163,0.36683180482236066,0.652493344289869,0.5477247845532921,
 0.6365098539541272,0.6089012957379157,0.30214277078066243,0.6337594454528902,
 0.9554666549032519,0.48992852608618764,0.6114207875422619,0.7519626596365293,
 0.6025385586713422,0.3584819275839457,0.057447195640764614,
 0.6647098718405057,0.9168605278443324,0.45324553395955447,0.7328281140008385,
 0.10314210625708742,0.014541715067512007,0.0175567467201041,
 0.07256593503707454,0.82796461614056,0.21976974207268118,0.4672499827086968,
 0.19223730653745663,0.21805684064521647,0.530815479615552,
 0.11596901259311809,0.9000817147004275,0.08470226123271074,
 0.5370770751581028,0.84979982604438,0.2797439277465852,0.08818814701119537,
 0.3628223722441012,0.7097636117940239,0.3880484010012716,0.2833411835786446,
 0.03894598944348582,0.5268035150903432,0.07725077301842853,
 0.6919326633927614,0.501934928361395,0.17093393612294483,0.6671787787438249,
 0.6785731232252714,0.7017955553256072,0.7973823563241249,0.15795217300527553,
 0.7238184906328119,0.8849816497161342,0.18525312471984823,0.9688513836827355,
 0.25289262494234477,0.5924635354782012,0.43375802612281744,
 0.5838645392158669,0.6062510253451476,0.538225625178333,0.3530341676491189,
 0.5229002989158497,0.052469884252514865,0.05693413471118858,
 0.8531724285130878,0.1244211525346085,0.5265967321478794,0.6269189864236884,
 0.305459724286002,0.7687940942801811,0.13495304869277158,0.9496628857328482,
 0.2700566021572981,0.16171427617075784,0.9208209323728904,0.8975654057924045,
 0.7446958865585862,0.8740028526445457,0.8337399730118227,
 0.016089756466904914,0.6293756319502725,0.13036024279023173,
 0.32322702915915846,0.9731314976659795,0.023346711100680984,
 0.6061020717331504,0.3185749768023577,0.7230388330380156,0.3119764406270725,
 0.39808216890690884,0.2247225743515261,0.8782375348974909,0.9277690327374666,
 0.7705213587437242,0.2185835802266599,0.8872386644055752,0.5561823751794773,
 0.9777867749372046,0.28974756756584674,0.38620852875725964,
 0.5974961573208166,0.08570579928932465,0.7087518802064077,
 0.47164118570432656,0.34245115307825247,0.3708390729953398,
 0.7144711586567556,0.6919870028376216,0.41459042564089366,
 0.03860148095376659,0.6296637913010737,0.4306794778258455,0.9774909350453046,
 0.5833693167762385,0.9992873854209816,0.07588414774446128,
 0.12078367794404321,0.734494455428378,0.8644325462100566,0.5751373189061394,
 0.6362343125604015,0.61246068326034,0.2551701234529883,0.7771510990325706,
 0.6241316514988985,0.09836570940473899,0.2277621231886071,0.5885523216962638,
 0.6549981611828051,0.9009123274698153,0.6921809525713769,
 3.887382942467932e-4,0.5404877239492305,0.8394154345508937,
 0.5423789342581895,0.7949706562011434,0.5837528158861486,0.5356592399486554,
 0.4394681032236536,0.10165281115320934,0.7085772256361836,0.4639218404304173,
 0.7449615044517205,0.3062657453218175,0.8782159056405295,0.1573310093264204,
 0.1335978056608529,0.84081953872368,0.5709320549485104,0.31831526930573617,
 0.11285252013865421,0.9602733291338078,0.09611323784848214,
 0.3768690913380981,0.4275581231344173,0.15964624203286404,0.9807818517184321,
 0.2841869170472897,0.6108844849837289,0.5766313887156477,0.9499797161381888,
 0.26119341299252674,0.29349513969521035,0.010286262795686563,
 0.5849716144170853,0.5801139851125943,0.18027805351492,0.583570767950824,
 0.4319851887484212,0.7167117213705365,0.017286147391107676,
 0.39314023900211403,0.7231465257032645,0.7929379813601536,
 0.16681857461446992,0.8480239554172707,0.518586755472032,0.8362715230190441,
 0.9779630042194298,0.8056539446742811,0.21927534109678726,
 0.16180638440098427,0.1208255970969474,0.8659326618554151,0.6942855796493009,
 0.5702900973824977,0.7167348530178621,0.8227019489644732,0.5731338627254621,
 0.8930150080031694,0.8527708323505525,0.7564011122594765,0.31055420311418513,
 0.8136355286658974,0.8039125719064923,0.32728158219706593,0.9348837830733803,
 0.2895904756608151,0.7093515127550195,0.8712226001229209,0.38585629527912146,
 0.1261371122257291,0.727496291019523,0.5097582703205101,0.052933838717116544,
 0.6788956573618629,0.16456514319943094,0.2541387637206539,0.9109600499008917,
 0.37525316110317664,0.7619328568693557,0.17497151313784443,
 0.2617549354859453,0.47593575891806783,0.746193151065824,0.723075423822537,
 0.2893448048545231,0.8061590385182353,0.8247473523531956,0.3548491736999011,
 0.27776476414538265,0.1980802632563905,0.16890113302616383,
 0.9468414124914913,0.28386755789176965,0.37893405576801376,
 0.7454804710114253,0.42247915001959,0.5437484280239587,0.45605252537897933,
 0.6139426489226836,0.6927834181444875,0.8790271757267742,0.4267854112040844,
 0.296904908526168,0.9495082215007861,0.2804173181856229,0.858558230361397,
 0.3929552635571594,0.2032160721989068,0.9623768329516337,0.8682754584279108,
 0.7458797402582263,0.9321437523505087,0.25562373266377236,0.5781659681569662,
 0.03782437976387787,0.30847120878413914,0.5446770483628842,
 0.10452307972855279,0.996869692440405,0.45151191878254804,0.2057220961959043,
 0.6359816972659281,0.7011891633067361,0.9405903030926608,0.590351510221516,
 0.7066959213555164,0.39242842747722273,0.6741555343119772,0.6842906911612289,
 0.8479379376385228,0.7110833567934802,0.32779391689229065,0.8452657597708881,
 0.6976355979779646,0.5515856871784278,0.6460344178538142,0.4779705922948456,
 0.1299537642748837,0.6581546303942034,0.6654047345493719,0.9490958592400905,
 0.029367638813315167,0.5447502395481505,0.7184516833512818,
 0.7956918755891089,0.38236804302061733,0.3486931492967136,
 0.08995998080721412,0.35028375491394714,0.27938666340019447,
 0.5786893746551695,0.7642715883092942,0.807202438920995,0.043489723103373845,
 0.3711494947059477,0.097697490256206,0.0019195853710811672,
 0.5782387023719457,0.30393807090266645,0.8982946636351347,
 0.07765004058486746,0.9896357581958282,0.5147608362163768,0.2792938524861508,
 0.44384883733134006,0.9704383147351651,0.36768124902387633,
 0.9695448290420967,0.18094610724332783,0.24396778794901763,
 0.4845228734357966,0.7699187744989155,0.5297257537782312,0.4262893227913157,
 0.9192369532104142,0.3053558203123473,0.8646105951400384,0.7897040743641345,
 0.16144087576979038,0.4551559330142956,0.06815878595245517,
 0.9629706619887086,0.5922534107375822,0.4074088381263582,0.7647384347159121,
 0.3733882859158191,0.7172263783798551,0.829686144472508,0.25808315635977896,
 0.5428095661467606,0.09223211447888424,0.05652103621523086,
 0.28494080847763814,0.06003050712557845,0.5732860759736348,0.916191857867438,
 0.7471985623536384,0.9339323361664996,0.09264354682094011,0.8344069863421916,
 0.4289769150497992,0.9567534412342468,0.5139092869217111,0.521750605185477,
 0.3150682595903942,0.6407922579364232,0.1645341473348365,0.08525908067427879,
 0.06444613804117638,0.9070992839534937,0.010689769713835395,
 0.9750064325220367,0.870961133639742,0.48305445156172533,0.13638546676450924,
 0.4797603906813518,0.98930847650404,0.7293603784344665,0.43921194086886484,
 0.3410000594473629,0.030017863581802362,0.189289823371526,0.6269671456237087,
 0.9470781690952962,0.744419942178451,0.12705220220144864,0.9595353400116384,
 0.8018162546269709,0.4668414373670965,0.4369536470207678,0.670770188187656,
 0.2820155588125901,0.658672940971972,0.1678896643874811,0.9954453940317864,
 0.44736404580736944,0.7773574581935297,0.17949895794730486,
 0.47352814400939014,0.29731155405972487,0.49390470794341623,
 0.3607414253920299,0.21351189590717135,0.4164209205431407,
 0.10645853757404966,0.010613374126081987,0.5188562532950234,
 0.2814016507627899,0.5492002185009025,0.7942768505411126,0.4029994327237405,
 0.37737493753927676,0.8303731077837386,0.47005545129200477,
 0.9746024268649268,0.47669060278198905,0.634238872884531,0.9919834587457077,
 0.9524250256405865,0.7773929146909401,0.267779396362235,0.5945096790357903,
 0.9243317366086178,0.8390447325743482,0.8650000817861154,
 0.057569194285328806,0.9871580181534735,0.467042358652201,0.686987546222807,
 0.9246232891413636,0.5309016727189988,0.22084058960688235,
 0.46784477184655904,0.7423178379609569,0.11839389976347492,
 0.03709298894271851,0.037328314673165996,0.049956263141742396,
 0.8780074665742926,0.3252480372142723,0.863439602464902,0.35755910042081507,
 0.6514767973893749,0.8543708479777644,0.45388804925473547,
 0.30651112772028355,0.6163820563447115,0.4777436350396975,0.7641794533014948,
 0.9177363145451849,0.6360300010426638,0.9786861696414099,0.39041865604807513,
 0.9415520673478978,0.8101974928742044,0.04649069571666786,
 0.41219414257406983,0.35053747410501446,0.970141917874092,0.2683187163452564,
 0.8893870675803505,0.3516552659434228,0.9183759010905941,0.49882441918060316,
 0.12105047119794232,0.24187962173244815,0.27614012400395704,
 0.45773070162441687,0.5384392239388103,0.004784640701469556,
 0.9058906438860411,0.963659176137533,0.3932051459182291,0.29122684401429255,
 0.44525131446462307,0.8761103628312011,0.7910643875845412,0.6438901371782189,
 0.9654754558852083,0.9734695327980889,0.4963097363804997,0.9082896038629539,
 0.1935516562482762,0.5986780022933503,0.009773398614476014,
 0.27643810973734484,0.9348554383022628,0.6574508014712171,0.8200317479109895,
 0.32102454617605336,0.8808007976778548,0.9008865778328027,
 0.21858735405485463,0.06069207422999068,0.484250363189474,0.985096179604138,
 0.6808331557440704,0.551905275590831,0.16194434446482409,0.2622459564992483,
 0.11623262102799414,0.31060668737888153,0.9037858634437158,
 0.9936800363567446,0.7301880956688612,0.8441783448888954,0.5384049217446535,
 0.22993507874139452,0.03739303723878773,0.17972471360103326,
 0.6499253886263588,0.8247258122714642,0.6288954676225023,0.6840911042449442,
 0.1466442509390501,0.8878613363844342,0.13855162301599822,
 0.38946831848244245,0.43886587543519795,0.08938502855555919,
 0.966916678673261,0.3908599619835774,0.967848608686118,0.5749867590106964,
 0.759198884207189,0.6451690434270495,0.7487437219112213,0.9577321890112971,
 0.7565504685360251,0.2883893117584412,0.3844215135535122,0.6084341673182119,
 0.7509832620574342,0.783456681267193,0.8534923455205998,0.1789735610051889,
 0.8155634438364306,0.19986641349413914,0.007738828906589723,
 0.8300866882092623,0.9843001619326971,0.8234656276580025,0.9543667021916882,
 0.07778264538327728,0.1546930123739403,0.7393892308469074,0.1267068618512297,
 0.5395397790485922,0.5301792688417233,0.2429545837961108,0.15475895088211578,
 0.07844421441133287,0.6644242418757267,0.5364102721882054,
 0.47101644771592943,0.48854814987327655,0.06165568453687442,
 0.6439908790449467,0.40574704982421805,0.12058670001238225,
 0.03032700295085189,0.6157623556388371,0.2731976419146802,0.2435374503451857,
 0.3208977413341876,0.42051676355111256,0.2496966888780755,0.5120665236998938,
 0.30506740539160104,0.5460578633920377,0.6440842299954341,0.7878007738793963,
 0.7913058879141976,0.8881322393535045,0.541329079461713,0.48962893375746896,
 0.22418037853926331,0.7062498675617547,0.09669354839050559,
 0.15290478650957107,0.9478760524703169,0.132531738744864,0.5809656495010878,
 0.9173421235906707,0.286533500123324,0.9325125778077727,0.6612288796825389,
 0.37688336982511483,0.06209237431489889,0.5197376240586319,
 0.7117691348452001,0.6781615854464319,0.3847569873137906,0.6236509352936762,
 0.32252660214015616,0.08527986082355599,0.5138761436928017,
 0.20877574623149664,0.38109754104896665,0.8778071163784729,
 0.4013720120633466,0.18135407984455465,0.3031954500693621,0.5519837982833881,
 0.9676079803671892,0.9616744238626515,0.6868911576019152,0.4224337855769653,
 0.4776376576945668,0.32554246518644847,0.5595958559601241,0.5009012423862815,
 0.024157805621641715,0.8153285878947901,0.08472391707128457,
 0.1055026546967498,0.06179700388517895,0.6995685311255291,
 0.10405862909857044,0.4759863122377712,0.3654461470408069,0.7366612764045519,
 0.8615653312696671,0.9078482815268889,0.6991111298222816,0.07783253624145647,
 0.13555936575345096,0.6117062457037532,0.960974909677398,0.5894091371087802,
 0.1890021179955289,0.17362689577141066,0.12429652865448215,
 0.8550344469581108,0.27918017641098436,0.35054229621430855,
 0.3059878265482846,0.025200906611230645,0.9954609820138297,
 0.3828702437137985,0.4499345140467427,0.9539870584527254,0.7502380186152663,
 0.49123968996073164,0.6694338088629217,0.9142889792826278,0.4622946191027105,
 0.7890831334736281,0.3061580645930587,0.6202760376166633,0.9897756282497306,
 0.34097491033254834,0.4940459986944683,0.39457523568437036,
 0.9125206915570651,0.9060499527320973,0.33221642600854706,
 0.025890579625119536,0.40175844987716214,0.4365514632193581,
 0.08007166664140375,0.36382025125773776,0.32943691851095513,
 0.6599785359304392,0.9510288416027842,0.8511467429220828,0.5737396592752246,
 0.24583168596277394,0.06287648441035254,0.7439127545780353,
 0.8394999226421654,0.21500377764166956,0.3334847905320272,
 0.27591912668929974,0.6705579155766126,0.08625691427381532,
 0.4324758310223856,0.1145371103653966,0.8117927538139362,0.6504895236193713,
 0.30999436096543975,0.6030891546559541,4.6330764326574236e-4,
 0.7283341240424174,0.22457747799167727,0.20678660701217455,
 0.39709135907541593,0.3263383510666581,0.9612040979490801,
 0.47181287380143244,0.2727165431608818,0.6602452173907296,
 0.33103185214458086,0.8015079785875246,0.984030002349138,0.3901305133741636,
 0.6434140346089245,0.9030256928849049,0.47847195240453866,0.6620036611648434,
 0.9517002295215277,0.5280520017459711,0.8171334638881972,0.5283893313312709,
 0.04345651472473855,0.7512260161808183,0.3051743118175253,0.7494341302406342,
 0.11371817688300301,0.5765496684690218,0.469596325569522,0.3204203275048334,
 0.48667058125742946,0.3950569338694443,0.41629358140617123,
 0.7883400886545064,0.7122134353597971,0.6421349826417964,0.8398167487704445,
 0.39504137446409926,0.49333330465151937,0.17877745639626752,
 0.9761821115961767,0.7937967090641866,0.2964631324845146,0.1628973163578824,
 0.04432290554001206,0.021543250918851786,0.49207664490657843,
 0.3787862904049468,0.8478979338268626,0.45935627477842333,0.884022671815447,
 0.07480505112148816,0.37358361207329394,0.6350010225552716,
 0.8531018628638098,0.2850508720021202,0.20680636721717272,0.9541752850260936,
 0.508549618685985,0.843999379278355,0.3781124423012423,0.9931036775648163,
 0.8083216967802865,0.237983814924433,0.7944932206201755,0.38924800579107677,
 0.7118689473106369,0.48708917658800344,0.5482426233479777,0.9131266223086446,
 0.11940599689196973,0.32189010350055525,0.6983452437241727,
 0.7278339423999614,0.10415840925958364,0.8474045677391584,0.9990818712550382,
 0.7955513629355698,0.18717284906257337,0.7773390208758157,
 0.09075350710453045,0.3811498530075076,0.6879132291132863,
 0.31757378132182834,0.4465137792043946,0.563912478072611,0.12559937586423664,
 0.9690866202652989,0.5475177325993114,0.8115729354133121,0.7483526914056674,
 0.6303296629169255,0.8030876007186125,0.27513771637677886,
 0.06588478270937326,0.4632003965230028,0.4272872437462778,0.9984588087068496,
 0.9407590473776712,0.32140218323706715,0.6172926655213125,0.6653290837518371,
 0.7458239419093553,0.6547305814741747,0.7106029478589669,0.16435335091216918,
 0.6086949051172138,0.7190624730935407,0.9564344221111596,0.9034954184628857,
 0.142873327451691,0.5353997852439156,0.523639956269451,0.226412308497682,
 0.14989518897278864,0.520997449323883,0.782122487150572,0.8758316153931233,
 0.6660664898022016,0.5106746803303117,0.0577710728435582,0.905888084799779,
 0.059386364135808,0.24684457628970036,0.6167775609250103,
 0.045665824398730015,0.8982356526959213,0.8795102632419634,
 0.5814973591315341,0.4249317984834645,0.8218128184273387,0.5699383709362545,
 0.2822740926776339,0.3908505441884521,0.021158081772074788,
 0.4499921034657065,0.8742792919460004,0.09746803603782206,0.5132703141249066,
 0.431135292360438,0.16339793985400364,0.32655288366052015,0.1048504325588836,
 0.7244040795462173,0.13872775314228025,0.39387253765495556,0.375186544038018,
 0.6281832536312906,0.6851563358515865,0.5371585775448872,0.43164959472907305,
 0.251640893951365,0.7221369145686202,0.8475209880780494,0.724789664649118,
 0.08306119844801729,0.528306300790018,0.8576340806812685,
 0.034532408635486234,0.2796446259369142,0.20235347974184315,
 0.29512820776041315,0.8653045835381445,0.9706810071964798,
 0.04888867179134604,0.16612550679950266,0.8738244806075651,
 0.5656080565521204,0.2627982021484475,0.25677883497266885,
 0.022196952651600266,0.5895208768490393,0.3811929752076766,
 0.7378442507158627,0.38604527688154666,0.4676992677638401,
 0.39885893628298386,0.27001218208782996,0.6895348042867349,
 0.016409647257354165,0.23636955654026737,0.0840301517201768,
 0.8566842348933992,0.059464923646576184,0.2105438823553465,
 0.08575586898003151,0.16664538573281962,0.4431504125134683,
 0.8337807817003856,0.5563178761890084,0.7029106412274934,0.03553990475849478,
 0.5148828718384119,0.3826139489157797,0.6515095255676683,0.22247682025543047,
 0.5567360267449407,0.33816173599713495,0.29456621896700186,
 0.045697996731355106,0.5222095193206289,0.5582753305220531,
 0.5329874940478255,0.3810355743544198,0.16069552487195604,0.4446510655594731,
 0.87530440971839,0.49588705937872524,0.2810901871856304,0.3558520397233055,
 0.4428599018174598,0.8391081002898844,0.857880331465452,0.3916945796737634,
 0.22285775261084417,0.4786766287003906,0.9001133947674876,
 0.11635815191979337,0.5880649226770936,0.7284810363854576,
 0.09597275589190213,0.26042144986254034,0.05465145459311849,
 0.10429201294361301,0.41216377490618916,0.48295656862130676,
 0.6172373926656683,0.18373220130528656,0.8255016039615378,0.8723362644219654,
 0.023515927798944625,0.47493951454075023,0.20239364407428106,
 0.5451245115228625,0.09557011719369979,0.7108212981707381,0.4173839159105349,
 0.660232878531802,0.2833327682978184,0.8321359831665716,0.2624672733740172,
 0.6674869858010583,0.5622914014934374,0.08898590626448877,
 0.14125394980476969,0.026201452362596966,0.025226584036514343,
 0.5681767466038676,0.07051367925286611,0.8625773145959563,0.8569700981609889,
 0.6971082999722293,0.7475782604424455,0.2851905039957301,0.9049505770830764,
 0.10907015997292147,0.900222923833007,0.4812055717794441,0.37147457052513877,
 0.9987751801689455,0.8435857596773584,0.9388238486062314,0.13227279861464536,
 0.04731349357819892,0.3003040074554135,0.45537838330993274,
 0.3136323802007833,0.4027930649277398,0.7551831220899324,0.5195136875507321,
 0.6410262653975606,0.5380326935700984,0.20068648980597736,
 0.023307318402471355,0.07572894542536024,0.9990191854574935,
 0.09825819774206368,0.8049882571304044,0.05909227650547433,
 0.5047881342337094,0.46396885553966505,0.04196687467018556,
 0.45409034652947144,0.16042032004452933,0.006754727876793765,
 0.8413848418753139,0.7122389166121447,0.7639610749792044,
 0.026479288811428248,0.4209744004147844,0.4247155208502984,
 0.18097314321980074,0.07348011277136368,0.8909272385881335,
 0.9569663866247697,0.41623961361759354,0.28826042515285377,
 0.017073791882657785,0.3471247771822743,0.720614050940908,0.9286239899225657,
 0.3036740082314695,0.002644145762074279,0.8316301217347165,
 0.5082297486290445,0.5927959223387648,0.04525796654575154,
 0.10713644209594397,0.9838483492383456,0.5057639831657297,0.2952036778718369,
 0.09827664042180018,0.5036660178694068,0.8667726381352547,
 0.38268342719218484,0.960986649838097,0.8073355622918594,0.816746995387911,
 0.34082586055802233,0.18981989067899785,0.9039693354510873,
 0.3378999058841031,0.06153473281790134,0.8575520353872069,0.8854310110884119,
 0.10033146706693552,0.669038597605577,0.3531168091770064,0.8706167561842263,
 0.7288009974158354,0.4490994196556586,0.38796119243065186,
 0.20928370287148046,0.23326959984625373,0.4382087084737929,
 0.4837343534207069,0.7806474849073737,0.5767107412230543,0.5026149039277866,
 0.9953519956592468,0.6301445736939529,0.17672992059367765,
 0.44848259912352284,0.6796150941273704,0.606381202797262,0.5707694626528761,
 0.392511615656862,0.5212278207016962,0.2196592674929806,0.16004352740284034,
 0.50669794300597,0.7801318900572879,0.7449728644982178,0.6409569901879442,
 0.07449739867593219,0.8112372921452693,0.10863377740486146,
 0.6010811260187326,0.7799992956536035,0.5067881712910197,0.7650837937516831,
 0.8807999751870057,0.48470900311388343,0.02682963792553572,
 0.9624811958910323,0.29522643126149184,0.17611535386369426,
 0.014038969933744427,0.6361405774715642,0.9967784915034457,
 0.37695272664577084,0.14522183331572902,0.9331451169087603,
 0.6358757097448473,0.8609561975908411,0.6517716026256815,0.27416975437806235,
 0.529870042372422,0.4383618586064283,0.13064165148724638,0.7839655850498927,
 0.4073552954615415,0.9019427926766391,0.6437138529384937,0.8067002024388156,
 0.21157697888367022,0.43246764796440007,0.36699637779248107,
 0.2068539698779278,0.33930921931724034,0.4950642985477417,
 0.34296619378948856,0.4265372437000233,0.20611861119084152,
 0.7685725764504722,0.05100497138125859,0.5474424513267034,0.3151959878660864,
 0.17021109612102214,0.4164016460035085,0.4647251656330813,
 0.19312583125589367,0.683274435785373,0.5509534057554765,0.3582070425249104,
 0.18767469411459248,0.8803598550929058,0.732233851331743,0.8228841309911252,
 0.7394293659770744,0.7876225481619195,0.27207513792719673,
 0.30429943118049974,0.029649389800114,0.9037456883704899,0.5661567055492548,
 0.15700743989553234,0.5370906379786575,0.47135457024610805,
 0.6991169210550326,0.35634377818101215,0.9582360151236468,0.7438606312703508,
 0.4367416883739046,0.32390541335032974,0.031480588820458344,
 0.6146008203451081,0.34828121088136355,0.13442850224340874,
 0.10559080262823406,0.8402179586481653,0.36149718393348595,
 0.5651788857913043,0.11733742415556214,0.047560365346459355,
 0.7729509822240589,0.1312589777802291,0.6928242650422132,0.8048156396723879,
 0.6658662615848304,0.18664920624189107,0.6163298507466151,0.8936147720187968,
 0.01652521397541684,0.2742071336943802,0.7735997868193191,0.0735126990270144,
 0.27841495520623094,0.6295520707837339,0.6368878368954198,0.9380640660431944,
 0.206760414082081,0.3244911483411923,0.5488045232309697,0.07901243838583016,
 0.37597812458747226,0.5247680075151474,0.04136628813720522,
 0.5247914247240368,0.20001291670356935,0.7206881377973023,
 0.13258136548483157,0.6027978615312399,0.7214338938291561,
 0.32895666664230994,0.710331766827437,0.6150423368476816,0.3268284055113906,
 0.9581399753170168,0.7881597347498428,0.9793553253035657,0.2254174340637758,
 0.850166060329013,0.06373652387958484,0.8701744352620855,0.3315651237485008,
 0.474916267615332,0.94873132303716,0.3046467780674018,0.5022092208360835,
 0.5159827426603977,0.6841617889120318,0.7975750630333822,0.820122422792586,
 0.385634334951269,0.050926553947455044,0.04836324436371864,
 0.11940899203954947,0.0990528997247958,0.4655980646521154,0.2708261155722135,
 0.309033644946311,0.9096120602468534,0.23448478428402553,0.16617261086056012,
 0.09555731656960875,0.4440391832363647,0.660808075497536,0.07260230965268899,
 0.5670077667151088,0.6600302852438009,0.11976955993098637,0.8031238691997884,
 0.47925862934891916,0.376266751524196,0.06959603358688027,
 0.06356388011008562,0.5772066465512277,0.48904708456376145,
 0.3808653090415427,0.21270697495717095,0.41259583964753255,0.385983525246561,
 0.7863282858320066,0.9686990992040829,0.1489263098146093,0.6956590144290506,
 0.23926411806220782,0.045211580933737006,0.35722260037341635,
 0.32352631862424075,0.2914917310789039,0.8513041439079214,
 0.22786358605560486,0.16583514206330385,0.5440862659962703,
 0.42569876070945245,0.6070371141596722,0.2125151745525381,0.9831944439641286,
 0.3724759603177562,0.9839085099123734,0.4829248994065114,0.24202064616732566,
 0.2062569726685276,0.8515621753259298,0.38233553350263705,0.7705101304001741,
 0.4415518638768955,0.8167058032000383,0.7805057381444602,0.20368707705953126,
 0.09240206770826664,0.5593302955975306,0.4362344634720692,0.9099474380726822,
 0.5224141173103523,0.33604939340246787,0.7440432116693809,
 0.32140967408431587,0.3373463991936858,0.727652303735704,0.08043766967050237,
 0.15176482909004863,0.09967026845870641,0.20368341514529797,
 0.5129442280190462,0.2510174191093635,0.7621439132009206,0.47987414097074066,
 0.40357014471717545,0.20841925225077818,0.16424552282854576,
 0.4360639432207818,0.47231596471009085,0.19370306403928472,0.329224694070797,
 0.009596972138733117,0.5482489768031095,0.06768312424207712,
 0.9630023843672911,0.05563244687139379,0.7242036529446787,0.4728831282301754,
 0.17500628386045025,0.8383429259678123,0.8124821535760786,0.8249479979472333,
 0.18283191337370241,0.7697068523535101,0.6109780958023188,0.4855776408545872,
 0.026329322710136993,0.920156633319861,0.4976719957704967,0.2154168443329807,
 0.6497276504612387,0.9487714927383477,0.06625159124183222,0.3069837909591212,
 0.31606665323576366,0.9192606683848927,0.5938461629226314,
 0.36791930808563755,0.8425112776012884,0.9290828496381516,
 0.43856477241666814,0.8633767077819545,0.45594945529533504,
 0.7368837791553371,0.9609348491608232,0.7578714651030912,0.378698535500978,
 0.6259739033702824,0.4032880331248141,0.5193908003250256,0.07912829372385044,
 0.47234624398671887,0.6796837476273002,0.3534043265841451,0.9486080696941098,
 0.8793393239710943,0.28169518329115517,0.29570111043143366,
 0.6046767728656667,0.8164794096002063,0.1418782691946161,0.6472890790602628,
 0.8756121220968217,0.7332262449444179,0.81753409941187,0.06098560497578065,
 0.7275698316421787,0.2801314667148386,0.2125145153135496,0.5541249073133214,
 0.03261690808939455,0.8314828023037324,0.710053836760429,0.4176998009925216,
 0.5472021805876295,0.20622365554317834,0.7071149837997031,0.7254850760035718,
 0.46513480424175657,0.8860851913365599,0.0691469939461159,0.839070187738068,
 0.6170782687333987,0.3546059930584782,0.7818886768471519,0.0747999647642984,
 0.3473914701335441,0.4750994272115905,0.8094373930788105,0.48313780436350384,
 0.026540162425476854,0.20992518179372877,0.32133543368597195,
 0.27600604942860474,0.6978258182989976,0.28650974369221605,
 0.2043839129081052,0.11182679330677991,0.655530257039624,0.6135400157256575,
 0.01619839864736805,0.10716061099310292,0.9855877794811846,
 0.08483738279582032,0.5626270113486479,0.3420749174811917,0.555536452534504,
 0.7410178606877502,0.3957808425584344,0.9225354307700055,0.8186669916002491,
 0.8048830859251735,0.22584826551503445,0.9328668375270079,0.5028110712060679,
 0.6953955013880827,0.20524061650533432,0.1935371238150987,0.993892920094258,
 0.23002352944917148,0.3956501144960515,0.059502320844431966,
 0.29448381937757073,0.837749307006291,0.8827945118549168,0.882861505182463,
 0.304404482855516,0.1349031482473274,0.24961439635000127,0.9677474045059463,
 0.6730370439241266,0.034810697421089776,0.1041928859252842,
 0.9697823669365765,0.1494145403928303,0.6174996391080702,0.5265136500098848,
 0.16794804530252283,0.776292548976704,0.8436721027962303,0.4935706403521669,
 0.19067900772588464,0.9245520771462395,0.9964371169522516,0.3185380453739568,
 0.9451115734060411,0.8470681894804496,0.7464178066725778,0.8886898337012957,
 0.6485219959931403,0.5105924462888882,0.7208210478738619,0.5128742211824011,
 0.9982716447265066,0.6361303195884246,0.6941792177943819,0.14602640195704586,
 0.9316831780466001,0.2815674694271192,0.11287678475104146,
 0.011489339505932561,0.3641595793446134,0.7336407204954137,
 0.5694151684568848,0.46232814734078265,0.11722747612229567,
 0.6345565262674124,0.8648996770742388,0.2868081012896537,0.7686523373960344,
 0.4104854059135882,0.06303301783187687,0.8465315816149626,
 0.49801829405135656,0.2836439699590281,0.26545822357056015,0.477319831163842,
 0.15095749263345393,0.9867277377507514,0.05616203695047883,
 0.5442339839628176,0.9671973742408487,0.4332335005837846,0.1633687292392395,
 0.723410225598299,0.6415294965127624,0.2732080897517244,0.43834750742113404,
 0.5932318186921148,0.32526775684773046,0.605225013609937,0.6540268130640184,
 0.37941461844893465,0.23561701449810424,0.47499881647071585,
 0.5293010565148251,0.09629844987611824,0.46174036337706426,
 0.038237562232998124,0.5495546850367423,0.6014073950335884,
 0.29109387152658284,0.3275286083272917,0.14475819663795364,
 0.7821832753048518,0.8883946421752714,0.4420877603686053,0.4578423469968017,
 0.020510531108345886,0.7098407989456484,0.9248798757667388,
 0.7241058584767364,0.9395502702525804,0.7641931227432428,0.19414235268746816,
 0.5218696724894416,0.7024100054732577,0.511676764506908,0.769655934290516,
 0.3287308915228615,0.663202010713645,0.4999568200844272,0.16348605381391168,
 0.7535314569104368,0.4419567553499567,0.7995971040882514,0.42514966350955674,
 0.72500903098386,0.2749329472915292,0.7638756484459905,0.10703913496103179,
 0.2606250961086469,0.522900225707549,0.5955182073750465,0.688667124562891,
 0.13446537131051506,0.370602897893318,0.16969260759787919,0.5807615269336946,
 0.10597318680033563,0.2297500863460149,0.7630564657024081,
 0.021552568850855325,0.2860045200299216,0.6409398333481184,
 0.08367438385833714,0.0494009525165513,0.3181171148890727,
 0.29937543705574976,0.280406168794037,0.8923855768959793,0.23441016873240306,
 0.7501116956535742,0.8294355048796471,0.1108009313083207,0.4915286353818532,
 0.9886979757929178,0.24814688750944935,0.3384621403531829,0.8723756999737369,
 0.8470046473967214,0.419633499642462,0.8280761595202519,0.4081014020866136,
 0.6366767420736223,0.22322591176459072,0.9897593616930618,0.912656063699072,
 0.6051304162279664,0.6209171926037561,0.7587808877364136,0.20358716816865052,
 0.9395378859372118,0.679567351523661,0.7505621276318664,0.49715947929018234,
 0.8223105029777646,0.7721222115761506,0.33258687451328095,0.5023214831233951,
 0.47992712962007855,0.5178341573129602,0.7423110045177514,0.3078498628996723,
 0.23163106973967618,0.47483856902917043,0.9867338299333492,
 0.4154305803308028,0.8047763913936427,0.9006904695967626,0.7691925280660473,
 0.3971430676705556,0.13935422577797751,0.4632485483090324,
 0.21609431765778508,0.4477848471575179,0.6928056798013232,0.18358546737309,
 0.07169292781931513,0.8978187256251655,0.5608082982509017,
 0.09400909959013093,0.3246460342426992,0.9674194513665881,0.3051494528084271,
 0.4228312202766592,0.3608727813685232,0.29135585028193256,0.7307300005507364,
 0.3531907260569517,0.023398843316017603,0.8171190757288757,
 0.5935867060006798,0.0785087070283792,0.8674467529136147,0.5224003644437719,
 0.5063347783397577,0.011031553421976148,0.1096398508923474,
 0.34836115895417874,0.1589592128012831,0.7626006715957037,0.8012020305875015,
 0.7942693567242506,0.5403956765603803,0.9606460279827748,0.4946847607125664,
 0.1325962755563055,0.034217038291540325,0.4999495145005537,
 0.8070421116873234,0.7940605248650499,0.715346652774535,0.5852974359357541,
 0.6148430939514647,0.14981179596637584,0.5185386433493829,0.6310984881679519,
 0.11281428379789993,0.16771335399278542,0.7346939403637733,
 0.4903029538051352,0.7538825100140077,0.4557668885471916,0.3895006979504909,
 0.1220162506205239,0.9445296336831241,0.9404761444561057,0.1271986898862486,
 0.33360513148332516,0.44876617091608373,0.19553841924082183,
 0.7349236158774619,0.9086004061142092,0.15074373024315035,
 0.07691681644094961,0.9935038667225764,0.3145830481076074,0.707076662580066,
 0.19945610946770698,0.2708127573790088,0.43397524066637483,
 0.3911184918961925,0.36645400479405155,0.6190352667917843,0.5091975312738306,
 0.7638063026381353,0.8889944439801792,0.7313423377476691,
 3.7481594703392407e-4,0.2731791615808553,0.5648966747854794,
 0.2861077258309336,0.33699233146202145,0.9522170035597122,
 0.33827500011393796,0.9331427117440781,0.06782845788708458,
 0.3255988308389062,0.519294289694957,0.7852398123149792,0.7173315595735574,
 0.82418590959544,0.5592240135790307,0.8087958898124207,0.8876371931597237,
 0.09854413096456072,0.18906137242743004,0.285876195561516,0.7471972849967812,
 0.3390461491636062,0.8686038483326363,0.5239206918644246,0.40487351282938544,
 0.3021440664014001,0.8634069447834581,0.09484731727626328,0.2075705970814038,
 0.19873368792302848,0.3984309710852493,0.6171920092486244,
 0.07587701478525244,0.5499736704657755,0.1321006757332145,0.7005840153261932,
 0.7924651603501456,0.7116727194272179,0.7762070528518047,0.753582226566663,
 0.2744849654937166,0.5512917819305599,0.7479040015355932,0.24307845241417592,
 0.4766453757192325,0.02076040688137859,0.9059275461396041,0.9127404865735054,
 0.8718108864041015,0.5604058360193407,0.9905447204211097,0.44022102364960514,
 0.264111527331628,0.0049724628654534975,0.9221024432263988,
 0.8818421423877437,0.1051200546134281,0.21862031456447661,
 0.28788892013741296,0.548553824633546,0.35161462461160686,
 0.22511894543893307,0.6501048928485561,0.7102557685594474,0.1850568795907367,
 0.8765007188831716,0.4668682422034509,0.24639647462993086,
 0.07811417999559356,0.7113373999436846,0.8478961244579545,0.2698516848210277,
 0.6871396433974295,0.7630523803827395,0.9984763878310187,0.3997867494081412,
 0.13066707705842007,0.2643963322951658,0.4376836336204164,
 0.24059367088814243,0.7149752972052345,0.6170571822677009,0.629311776235842,
 0.1567873744646412,0.21824712057149254,0.8943628302090483,0.3115205366681849,
 0.2848319787729877,0.8465254971321103,0.9769383299796781,0.6173133241443791,
 0.694268444916759,0.10660943387135091,0.5111604498328137,0.8390351881318997,
 0.2784621279715348,0.3406398966288263,0.4328691810009757,0.08067331008268042,
 0.46720863759256215,0.9949866402627333,0.29013417419405385,
 0.9789951820128333,0.5176013328721222,0.7529368956610185,0.36446912477421545,
 0.6722594383717408,0.07216905128166062,0.8471278451257768,0.2059856325516617,
 0.5585819169404179,0.8750974637199783,0.9899500057099319,0.7823337678748186,
 0.1712280669582935,0.2831874564419421,0.06880969247277535,
 0.28578590489401356,0.32814827296135407,0.8495865767359493,
 0.9403170835612522,0.21575701648182188,0.9932261611741438,0.4992994923635071,
 0.04621997298170211,0.11161708351500199,0.7875417570723221,
 0.5432536077113173,0.7134320667010396,0.6375950901285635,0.11682837796187595,
 0.897874264086147,0.10790002199441306,0.36642401885769527,0.9721890837520422,
 0.9342323225161278,0.7740099384907845,0.5289153648928062,0.8186928753867984,
 0.11087858613938795,0.3123869270043358,0.4990959802079844,0.5927494928008137,
 0.0778629455721197,0.8364619197699654,0.32381653458091286,0.1399711023961061,
 0.9920123549661841,0.747779085447611,0.22645874186560333,0.7755807044019095,
 0.6713493800882839,0.7849765885205168,0.9734040926047489,0.04127082111321101,
 0.9784420525227782,0.7370947996008627,0.3542911862520096,0.6468234738540471,
 0.14594198291933025,0.1650541873143987,0.11820272205305893,
 0.18033299731761043,0.8732348764111065,0.43534736909934757,
 0.2809813544432451,0.27240227002776696,0.7413480243777001,0.3435351889298165,
 0.5737977250459899,0.6605388600463912,0.3748465744977587,0.408632180814849,
 0.8893237086859058,0.8315515605632192,0.14858893628094538,0.3170796553888332,
 0.8925018691360481,0.19284712319781427,0.48450545320222216,
 0.027595065026960297,0.1665481915503001,0.5085899548418747,
 0.5005723912001919,0.05422308990218305,0.8470817268330637,
 0.13337386619426084,0.07422043264295519,0.16231718027308029,
 0.04208872515503159,0.7127295575524101,0.003987765406972543,
 0.9087602263545494,0.4947176608147881,0.5290812754745842,0.1184619681001825,
 0.9264547271930053,0.5075267795760442,0.018376451841690677,
 0.9717356334644758,0.9711362869759615,0.817927258146307,0.2451731569428749,
 0.20494243579999072,0.5524166244308633,0.5697549585352422,
 0.028264866386954912,0.3338547950261359,0.7515588688633298,
 0.6617310638677942,0.9290093704274333,0.19371325298309916,0.2129317788633427,
 0.3231111947978048,0.17193856721989176,0.46731974704593204,
 0.8335180699926971,0.90103882750863,0.8613444603652475,0.8046748272074042,
 0.6920950041383266,0.7007778562359407,0.024997116789734375,
 0.8131153749186141,0.8436640539451036,0.7595483882354519,0.6571000134375882,
 0.856523720710335,0.3767634275537697,0.1402691346933438,0.11715930690521914,
 0.2722847492711775,0.8068881986956904,0.9487091163245517,
 0.0024500959446590587,0.026439817110661012,0.005934135356007952,
 0.8860738424741298,0.08515071209659786,0.8101562503302574,0.7437892703403515,
 0.6174001473269568,0.3571200058229427,0.6460870162131179,0.5557316752844599,
 0.9574899891562795,0.33425351508688794,0.5414144994393896,
 0.03802909202056182,0.7088611182518684,0.08120066134286497,
 0.34657070722894134,0.05664997571420827,0.12773938032060173,
 0.8364226245800165,0.3533648686188705,0.22996596187980622,
 0.21897460235634303,0.5618699498966975,0.7601333702524313,
 0.19318669489049922,0.04551951174984481,0.43998040802739324,
 0.3788881764322394,0.8157282381281786,0.9732248564502858,0.6250983014908422,
 0.17511185081260872,0.8864487373369224,0.8762649480686845,0.1568797698047023,
 0.5000395367944992,0.2525479773957643,0.7491128279596557,0.6001222915040526,
 0.5958819207281773,0.3353177773755446,0.07747675486881911,0.2403149791079856,
 0.2482655475148262,0.3384543953177791,0.6245591875734059,0.21161976522006465,
 0.1466706128114902,0.8820343158555412,0.21664528649940884,0.5078396512711979,
 0.7460381106325221,0.4944761864745921,0.3441818794206585,0.5144851358400928,
 0.8307096134212711,0.4041859736597305,0.4968989268878292,0.5234583228105509,
 0.2552267658728099,0.1636039815806296,0.3441000088471279,0.25785115520624324,
 0.5743543702988266,0.7874479754863063,0.0989435433127197,0.9434835166208565,
 0.2331604546915962,0.8820971408599353,0.8120337146685533,0.2166142211828097,
 0.9737096414667427,0.06063223665083006,0.5556053429794364,
 1.517267794096666e-4,0.5599465322509509,0.1334664526374637,
 0.9189947040100362,0.9106682323704623,0.6367902255034295,0.9489323332092952,
 0.925156183164141,0.6179189456959789,0.18117105240661957,0.23060233247989448,
 0.15143080149185817,0.0611197100846097,0.5658660940196467,0.4389870355949016,
 0.8302099728502353,0.5292275569456086,0.049085125811869634,
 0.48097972744180517,0.676034660935944,0.6617660217872225,0.7018134406448215,
 0.6205883282970814,0.32949063320059213,0.31948240512155857,
 0.5290567734963699,0.2447488484730398,0.02877182134449552,0.6115226209386956,
 0.8581618320258211,0.27203954582693324,0.6513668604288252,0.3886906111798899,
 0.38828677583665205,0.42626838928843025,0.6098467534106651,
 0.6359004045638403,0.6000058206403351,0.8462966380072713,0.1537515170066166,
 0.7229683993268488,0.8241106038136015,0.42789228501865584,0.836578616728772,
 0.5776991073482596,0.8304907685936652,0.3063166167399626,0.9392928538565674,
 0.8684579945658667,0.9881461995349072,0.08656214171881194,0.5161914905643208,
 0.9663416355431019,0.21857856561100708,0.18266492723066252,
 0.06105369499887647,0.22496423922233333,0.5851916783225605,
 0.5250857404420534,0.4658339413376176,0.33311380339448116,
 0.20630464640427704,0.8327206761706897,0.27055784483795553,
 0.5164674077572815,0.17633303512286347,0.878373525406091,0.11009523746654282,
 0.7480395119014086,0.23843298026059145,0.26058974947856695,
 0.9608435181106572,0.334704787831011,0.3045433763943366,0.5814514489408065,
 0.3047830127787893,0.9999734369850053,0.23323918254846676,
 0.47423367675277517,0.6266246064277626,0.28134420447457864,
 0.6551218807659107,0.5310569921176326,0.728396501727727,0.8267628715141487,
 0.07852751060582208,0.43359997050400945,0.8456502216183122,
 0.8027901300199456,0.9918278091870172,0.055157921797631015,
 0.9477647731492291,0.3272350742481438,0.6169824093450549,0.928535900823189,
 0.8466738740326053,0.3956185546399543,0.4933763838787635,0.9207685040582545,
 0.27212892598049676,0.5077043303226515,0.9469109534935205,
 0.12049656016413945,0.015199256772276293,0.7032379848240453,
 0.4593200809032303,0.9222171477986996,0.12122526660146771,0.8669905146949869,
 0.41802612799272776,0.11304377532677201,0.9781921276847434,
 0.048258805133501226,0.44513640968053825,0.6603207297417918,
 0.10993350526371104,0.8877996708501678,0.2972506201730034,0.691888193784092,
 0.6820507273788758,0.7976193941997556,0.7587405310911715,0.8203391660571477,
 0.19641259987855098,0.5690481234771658,0.14471258781024343,
 0.09008084009959194,0.6076518966900375,0.7558536956842965,0.6891232952130764,
 0.873625462419125,0.6049216103709196,0.07839944391568676,0.935552103130039,
 0.9678674529494389,0.05266153789207495,0.4121387683729747,
 0.24059807106967512,0.4819289694758402,0.20795547392030367,
 0.5138870832055707,0.3688868336620408,0.18259009647871194,0.4266392360481386,
 0.6365391089274044,0.6482895505794062,0.8393478122337644,0.7490186903511213,
 0.8527073130870126,0.2897426061959918,0.2592357129380881,0.9473364219022689,
 0.97380826263484,0.013261922321904485,0.5548555804386313,0.5058197806143762,
 0.7627631412868591,0.001268694189233921,0.7868956358593492,
 0.6674913856630758,0.8969888642531385,0.036260610878680266,
 0.27216070066491493,0.024056908095247853,0.07367888040608772,
 0.21603581565164598,0.0410024590241741,0.1279353745521522,
 0.49607183442423375,0.7380458103518642,0.10001622619327799,
 0.18772385937878056,0.1712602551091521,0.18442672516295333,
 0.9691507936239788,0.8542642765371948,0.46952627708115424,
 0.42738908556286714,0.15226143416965232,0.7286392269208113,
 0.04100128955799964,0.0454125484118274,0.5034085111810795,0.4514046065370316,
 0.7616211578987812,0.035533014281449726,0.1734909621812616,
 0.5297240180496366,0.4775694251223839,0.8778756260146613,0.15315339748629375,
 0.433719628219587,0.8232900247118726,0.7426359155231116,0.9343996306678283,
 0.34438870187615955,0.14801210197270653,0.7280260495407612,
 0.13365174996551532,0.5931063093089206,0.48126235975186726,
 0.9085492385342396,0.8223032879879587,0.7587894734861833,0.6841497227991351,
 0.41278497137676307,0.13086099698745146,0.42555319779666634,
 0.28849668947034357,0.029062776752627584,0.6752145077798409,
 0.6810838014637745,0.740762647227102,0.15467232551915178,0.41819262864878803,
 0.540694731042306,0.12380733546742761,0.7016364248021483,0.36857784611168465,
 0.4174755336882402,0.0782435324906654,0.17374775437183132,
 0.38417971113941907,0.5854917794985265,0.8755966220330926,0.2504908194014378,
 0.5525990810057462,0.7050935483398554,0.7242537210255379,0.12957838597177607,
 0.9578490143959867,0.30745079970993194,0.4933820523637008,0.9896470657031201,
 0.33673364382049886,0.6465599179744035,0.8352669688582953,
 0.08849348503431231,0.739365106008901,0.5328505490860137,0.44255649846452083,
 0.5402228527801151,0.6300558011985902,0.7574533725576211,0.44197612854377166,
 0.07222685602782453,0.31416065397714676,0.9897609399205606,
 0.7676846664192583,0.08097729252007158,0.2115725228802524,0.8171449042514974,
 0.0878967620713571,0.27545331405503015,0.7509310706380031,0.7987930510945439,
 0.25517209127322293,0.13664167338544608,0.9913981231732174,
 0.8544668423858508,0.4933547210906921,0.9157606958574483,0.5933471470030258,
 0.015229564854874145,0.014209228477288782,0.473028513116712,
 0.1584619227632994,0.610986225731043,0.21397956688228903,0.36155975758373493,
 0.6930920846627664,0.862746328369512,0.09948825524820049,0.4046198722133647,
 0.5167460320743733,0.011614421116081353,0.5679013530246938,
 0.1364503377832642,0.138117642709093,0.10545326489647622,0.6842220038990381,
 0.8510446648485485,0.5989750031786081,0.3359699669616836,0.7066878678804315,
 0.40788812620966686,0.20685881527633365,0.6985816160990885,
 0.5601212299233024,1.9864610072195532e-4,0.7897348699734057,0.36722118429435,
 0.9134002319959599,0.7107809024224071,0.25572201967798736,0.4869965129355919,
 0.7390713316578209,0.4718248375038365,0.5416699507777122,0.3596442128512448,
 0.5543996757666975,0.48579378358048,0.630465712390589,0.9959619938097233,
 0.21753689305768253,0.7079781765376943,0.2674248821623748,0.6410851206899271,
 0.41485509708436785,0.609029793195216,0.41965900535073053,0.7450328564569478,
 0.9139027162486835,0.42519816766554086,0.7102863736062647,0.1393666294072593,
 0.2137287824985461,0.5692524620575369,0.7016782191842104,0.1804679291130794,
 0.37855086268086424,0.08511651974985279,0.7827589516531888,
 0.8004927653415076,0.5494806392548901,0.31854284632281216,0.8597058479814089,
 0.6642965166985245,0.42770924417387757,0.8075970883685542,0.7027093132082285,
 0.06237597113221982,0.17332095444629025,0.9555731615001571,0.260563045113014,
 0.36048188858235075,0.20294578269032337,0.5661774746526682,
 0.32907485972700556,0.8312034894312568,0.9609860137624524,0.6494131812344716,
 0.4811951697060203,0.9847017209671989,0.5875252615817499,0.5912625780632146,
 0.13102482345090338,0.9086992458307512,0.4105291608476114,0.6917189952524022,
 0.8273863293195189,0.7450756610053328,0.9579747066778046,0.7829540731675807,
 0.008936187497612269,0.9428910908337705,0.25638341398527054,
 0.9054339103835922,0.7875289375622923,0.9551576862487222,0.5549741015556664,
 0.5055656592166241,0.3477384440182708,0.47839619240398723,
 0.47879939183764586,0.9993927127785502,0.19687568262802102,
 0.8181921689130035,0.5450098912725666,0.9303129107053704,0.6683765575563765,
 0.2620195969568734,0.49071988157745894,0.5068842347598139,0.5432357275752576,
 0.9245056368585772,0.2183783811637624,0.7333741406104609,0.14806592493342907,
 0.6253766650318969,0.3729842980214575,0.597117103855207,0.17388388035611746,
 0.040632628597835674,0.047801763235549966,0.5171222378990306,
 0.441127329330101,0.47435447826550503,0.5414417542180245,0.3786565068309167,
 0.2972877986915976,0.07080969201911258,0.7873833513912019,
 0.49641179242274036,0.5512665704065588,0.4677628892664377,0.4468304443531983,
 0.6162399617508321,0.6799187624613655,0.24635553177170144,0.8720155257744517,
 0.8666739355152279,0.01181926768946917,0.5482454182420023,0.3449649283506153,
 0.6811585862908754,0.8544674288491978,0.9106560080997768,0.1456247177403931,
 0.5914973676153387,0.555675017534658,0.3855910888087417,0.7679194318422403,
 0.8096231861655924,0.748894133009599,0.4466406720015622,0.4115851118784728,
 0.87842935910734,0.44855377467074453,0.17664010263297547,0.0920446090588598,
 0.2559102223481924,0.1361200125101969,0.5880729415474891,0.5572157906192451,
 0.752835565820529,0.9610942102319866,0.11947247715803522,0.9990800774391003,
 0.5267630729327601,0.4364522701337078,0.7120664558546155,0.3187354261849077,
 0.6859549989355954,0.24305732926658286,0.8615465913484337,0.9947735279630008,
 0.4740339974283083,0.8608713784839602,0.3480566824228837,0.582183763240853,
 0.07064847024366716,0.6247197872568795,0.7915626743588903,0.8187004083309176,
 0.6988344629314326,0.2338947086758646,0.551383432667707,0.6684435798821674,
 0.4073050570909087,0.6286989789908674,0.5907720110253135,0.7018223480588086,
 0.27102754155846853,0.02884350300790417,0.9656477944060121,
 0.5467356434000066,0.3822058522167656,0.3727534407794566,0.9950525798606303,
 0.6091000804877573,0.9933733401390932,0.6336733780707151,0.9838997801244711,
 0.765201606118536,0.23324197807051228,0.998040330265822,0.9269973406493773,
 0.08003551229457806,0.11345299385913643,0.48679259472637737,
 0.06744957861658873,0.6196214313432791,0.7024383325753286,0.4882999764486179,
 0.654031942753095,0.904868538164932,0.8745612051104628,0.8519384367969856,
 0.6916202412775649,0.49678327088878116,0.11214267847494974,
 0.2455280848842355,0.8232567830388459,0.09174563235728617,
 0.11582488800314616,0.8548902864781323,0.512756607696801,0.21746847050936724,
 0.483559789394696,0.29106248632857845,0.7482611993713347,0.882473515361543,
 0.11446301390244129,0.29413757640857985,0.3385365821005507,
 0.04460860640955633,0.6385875804788501,0.45244883769423705,
 0.6879159692209023,0.16468974276640436,0.11878929382236436,
 0.9345124709997497,0.4411949250147601,0.030826660151240137,0.71566773647287,
 0.6676570529230027,0.15315363167447693,0.1931102757462978,0.2747338535303345,
 0.6474619654501741,0.014004203535501136,0.7570158731574081,
 0.6898841573468819,0.5630213215397694,0.4974107619947258,0.4858622309761036,
 0.104889841522808,0.1672222187058603,0.38954990785442867,0.48058482036456573,
 0.3539567921512575,0.35419228041417383,0.6064003062325773,
 0.34804609975471223,0.37013389961809917,0.4514860218081741,
 0.49092249178256653,0.2875075966851077,0.8389031042670038,
 0.32917018141173315,0.28280000016592544,0.5347285882608047,
 0.5790125285605329,0.6573298056478009,0.42172185159373254,0.5245277663146173,
 0.03440727780143893,0.6880074918755397,0.928925864113473,0.19929320271059647,
 0.585513529794618,0.4051616327802495,0.301190083506949,0.5384720664049555,
 0.7819177150806906,0.8439567319818363,0.056081994969104376,
 0.43967225415268596,0.6505886779352699,0.6860936159861941,0.558343989471174,
 0.7212642878572096,0.540549441596715,0.5294978759668644,0.1957454809918311,
 0.3012204500988498,0.5738548911420265,0.14297919923311875,0.5494196418356304,
 0.4006537660178012,0.33680738906312124,0.2574139871275031,0.3089645134186224,
 0.1577203705933603,0.611326064716685,0.25377202038945035,0.02073428914622677,
 0.049431058753532486,0.8205526067150718,0.800717940645785,
 0.04329706761788732,0.5122305175183509,0.7992890777117887,
 0.42615288166834964,0.3339786757290911,0.9426268812690688,0.2346405154665846,
 0.5691274822629994,0.7207460587365857,0.31937937898493707,0.8504576271755234,
 0.8939019388335921,0.9613121287607069,0.6925397315216363,0.06885723154055279,
 0.7279902883944237,0.8271438054704023,0.4652622531702656,0.2148603083688062,
 0.8639936265018954,0.33519904204366224,0.2911335995172085,0.9560067994252665,
 0.8103847111373483,0.7675264750532171,0.9953989825564173,0.9114577603898746,
 0.779488121981991,0.05156333540620106,0.5447367543823907,0.7850124407138397,
 0.753696111578956,0.19487053678939426,0.6271045668384178,0.8504175520004414,
 0.24395217968869742,0.015298599610866215,0.30729102267291586,
 0.05394434524342173,0.4209662457323401,0.5802707261835575,0.1694582055292988,
 0.7887858698460242,0.10451643967473001,0.6501987623723323,0.5440428112031367,
 0.43946663990248336,0.1496937142140795,0.9306078563499823,
 0.48826778982168584,0.4246531344835849,0.31892536027828344,
 0.8820955290975414,0.04870248843960889,0.7626603312115801,
 0.17869683181169294,0.26189523089964295,0.15662070535519457,
 0.06539952267497207,0.6939772926694648,0.012348857358766119,
 0.4054874695334726,0.9373367569804246,0.1982463188354402,0.8977966687221097,
 0.4524005907243448,0.9204071993068984,0.24955224883858362,0.758553148624567,
 0.637364157038691,0.363904042691765,0.8688169951428011,0.056388682872215434,
 0.3068801632060012,0.8321907536683271,0.6057841558722796,0.5219092261554448,
 0.5449530629726816,0.3453193802268094,0.8623864749443317,
 0.026463110271617962,0.08004688987821207,0.7874397794418394,
 0.43867342320450264,0.7106576564480098,0.5251776280873293,0.9049128957451171,
 0.5872844209043042,0.9550774181736834,0.48512494987402666,0.935174196734965,
 0.4559280097405779,0.8126544969452584,0.5292873423935571,
 0.005169943455675674,0.394826455307265,0.9243275892818654,
 0.44792893122464317,0.6102632633075584,0.765937367699025,0.0929810125149434,
 0.6040077262715893,0.7366332642060489,0.4542009706198863,0.8086350499205465,
 0.090604852558957,0.3668078252721486,0.0359126435428988,0.19242013689748194,
 0.16057208836811576,0.6045599215482893,0.6886209772240028,0.8899151545384497,
 0.33947156646736176,0.37344844586898873,0.12779663720175405,
 0.1476976969603958,0.269371245541685,0.21131148762340946,0.7332792798899515,
 0.4725130145744124,0.9764313712507656,0.7873725569995982,0.7409682885769766,
 0.8193057397229011,0.2337360478471454,0.28413535061285755,0.7299981914166561,
 0.009855998857968729,0.5783568485956523,0.5998731094395876,0.80208789846396,
 0.39807722721608674,0.16466191458850532,0.8395458302597371,
 0.9967873674996697,0.6544379965849488,0.2255114865633504,0.803328566482627,
 0.45812290289021607,0.9892366852816339,0.46777328845529464,
 0.32294247080651173,0.9491139501022708,0.6673708326934151,0.5765985648835225,
 0.5970245363228135,0.21939515084204364,0.7638548542002445,
 0.26551304743848736,0.483942469311062,0.7178941380507589,0.5566014209086101,
 0.9192701214820461,0.022460988508795077,0.5018893966455736,
 0.8289706602912942,0.7953605922069006,0.07368752541028223,
 0.24138898017954247,0.3374866654077846,0.52347583296947,0.1345720933709169,
 0.04471293555028555,0.03077412666514212,0.09282971303709031,
 0.015365754715127466,0.3382500956795155,0.6237439587692771,
 0.6458423336819291,0.8108580813635221,0.5117601829085584,
 0.025328589060972573,0.6248967163971734,0.6795461580732334,
 0.6699726367027498,0.9464862640099571,0.29824982379218024,0.6821765571286511,
 0.9874013522628061,0.6191657883003339,0.123675287823521,0.4856514189915577,
 0.7543054507904994,0.012598093573441882,0.8676361343764302,
 0.012168649230480999,0.7497091042150834,0.8949174625753713,
 0.06760303808650314,0.23763316357838737,0.8167119812470918,
 0.8716535562516344,0.538571251015121,0.541691486633503,0.050530266358341125,
 0.8640833274674369,0.6246506990639691,0.906096170467331,0.2916053353372534,
 0.5227272195621557,0.020510044006543215,0.34826303462187236,
 0.17691025214108347,0.9911793554960295,0.8094640481992597,0.4235758162498693,
 0.9060620078486408,0.5606032416604061,0.7088455615755644,0.31448597170051695,
 0.797049999478087,0.34229163134338225,0.7177440225204389,0.6803154825493726,
 0.152404862287276,0.2486007302410448,0.05470457163736997,0.17160685635062456,
 0.9628914794805347,0.9941103209354514,0.1332145023214606,0.5702973068421333,
 0.2348611151133826,0.655907257770637,0.7623920016461951,0.881481946471127,
 0.3211156116954743,0.9708161116314143,0.5760053553084598,0.7095904473357167,
 0.5687025773306327,0.5841759678618262,0.9359767093351142,0.24287848499819875,
 0.6464911092021465,0.3113235603613749,0.32627084136924533,
 0.36281647284850027,0.24544820833646508,0.811645526688164,
 0.047071815991956645,0.8577519641083622,0.8575878175538255,
 0.29565748462279673,0.8077445104465805,0.1214695961283987,0.7349885488565161,
 0.8143721803377397,0.05725826496411013,0.07888259262883679,
 0.6620082901534291,0.41993925500565454,0.8926387587822251,0.3767723293132663,
 0.4396668494287409,0.8457750541534313,0.5751022014249503,0.9949220518116153,
 0.5608422652763368,0.7902612169572129,0.9315019668604462,0.06957657700448305,
 0.7795136390109952,0.6917980777915668,0.5972761150653976,0.47405996249110127,
 0.8647093288583563,0.45767448009627654,0.8738683930407958,
 0.014030414802678748,0.718983201691094,0.5147424003676455,
 0.13677232337479805,0.3266171478550036,0.7360672030482248,0.6262039713376162,
 0.3936747387132912,0.5625636149818296,0.8362964909349667,0.9822405983965522,
 0.7876032474880411,0.9819804924010597,0.8332186471892883,0.06270193536053248,
 0.40274596463391554,0.652626314675097,0.9181845091084124,0.7586878743026839,
 0.6719794650553292,0.4125484772699628,0.8948922765697866,0.19232320862072028,
 0.10495053333366222,0.5917502904278767,0.048594328085200034,
 0.949304576943819,0.4773673599948661,0.3797695417838075,0.016762881064069313,
 0.8736464553723245,0.14068144816373807,0.1710682987303599,
 0.36165028754511275,0.13886649022641107,0.3554521881044126,
 0.16976824237249188,0.42237915559744066,0.7755233563984049,0.936865980427803,
 0.4297374519902085,0.3388228386739578,0.13350329507806147,0.7925148303663778,
 0.835093239943737,0.2682170298862767,0.1307259177674709,0.5670091090932533,
 0.2304993637221967,0.9859235038132332,0.19692648512683153,0.3881036457168384,
 0.6954897131199436,0.9386998397788455,0.7874933356461485,0.926615030869149,
 0.9044669372922366,0.8573633891096557,0.6371272974594042,0.06940346264833885,
 0.28179552752021597,0.6507225304004581,0.0797854703465789,
 0.18055921565971678,0.3159047959154301,0.1521217704287403,0.8164840199126224,
 0.28786560417432994,0.2237129781448023,0.21144045945934842,
 0.21775513365240218,0.12485638756570794,0.9905899897866672,
 0.4864283660566592,0.4209593928357829,0.3268804007120626,0.18159933963231145,
 0.2918169353000545,0.9182036849925322,0.40488707208814745,
 0.04103384838147417,0.29985727459430844,0.7876439606556451,
 0.9300436941700411,0.2341630020523645,0.3639910671547796,0.9219995476393893,
 0.3596238313446396,0.6071743862734887,0.7788195143509634,0.5584357465832371,
 0.3585519966734775,0.7092135740069544,0.18647197807773364,0.8732317539566357,
 0.8406036804384253,0.2620729093572103,0.25814968502965074,
 0.22859898532996614,0.13216281951383302,0.4150595953155636,
 0.054051374751785675,0.21100961148344866,0.69132765645687,
 0.023361705402797917,0.2890967152417919,0.06795865137103185,
 0.5829639548090916,0.2857142567087513,0.012283543792080631,
 0.30275809049070423,0.4265539891181539,0.15079336666080145,
 0.45531122480854624,0.6630628047320541,0.3957933854522122,0.8597249331285398,
 0.11096652840365728,0.746335345795193,0.570807012683794,0.9651033785806883,
 0.08583344715861718,0.8000944555246622,0.8661536462629591,
 0.048978349286082246,0.007279992644937283,0.8711453749565806,
 0.5231927301742274,0.9989717484327225,0.9116102569934139,0.7288361553143733,
 0.8985905275933395,0.48022992446047996,0.40760548471187974,
 0.7839282457188742,0.8459943788589306,0.1587070595684651,
 0.050578415399998744,0.4730476544265886,0.09285526222480067,
 0.7874382795588744,0.6333460185577022,0.9161658823221035,0.6347340109140102,
 0.4394220995193471,0.23441362490730133,0.8645496010277137,0.8486555317598151,
 0.7826692171561972,0.6364641953126484,0.794258916407833,0.5048752130357772,
 0.7114564770948895,0.9208300571161236,0.6210357845961725,0.4366926275438594,
 0.7758650887913712,0.8459848796930607,0.1811343871455121,0.9816289811125045,
 0.7646406184705558,0.6182152855976296,0.09927928899960559,0.780002478205477,
 0.6539277571488339,0.2703488805357398,0.537833029189621,0.7440752432655378,
 0.8436695688877106,0.012164933508516218,0.045799601600276185,
 0.0731880312745592,0.8087438848960592,0.33742005924258367,0.8171595293192656,
 0.4804261330522288,0.2862953596530802,0.4334241998055296,0.05502017448644769,
 0.3586764238556567,0.8325481287684304,0.4880912164640696,0.8989626963953852,
 0.17100504191653687,0.34635780402897354,0.4504615614746672,
 0.3605148638643947,0.39670771840956087,0.9292733938859645,0.8668738928545969,
 0.549723727783638,0.7790376576466463,0.6543186886852063,0.496670470223581,
 0.5332124972659765,0.5325362218778923,0.3183303453816426,0.34155737781183937,
 0.031143835850228507,0.7792443746869577,0.5702113443090957,
 0.40206334796957277,0.7065899595357988,0.32377075731463256,
 0.1666839788074127,0.39417780232975297,0.04774614327195492,
 0.9167674978180919,0.4028634642972785,0.20630332557850406,
 0.03824325774194948,0.14927362688588064,0.4558313729949568,
 0.30272353297195687,0.23402471486217025,0.21567068617413399,
 0.1461458877010563,0.14636753810522984,0.24769677787666233,
 0.5717988385677325,0.3816036480031112,0.31508709168215443,0.3048524817317336,
 0.1726067367757661,0.894305743523344,0.26309999401864526,0.9539984071888313,
 0.2141319545492555,0.5305742494307649,0.09765908195268569,0.8594678214218876,
 0.3024436202404561,0.22069206583102519,0.730004054027578,0.29074545157535736,
 0.8071765846823284,0.4436620547984167,0.28086849409447345,0.5349735952538749,
 0.37777687667642845,0.34504408418180543,0.8258404720324886,
 0.4823575645697521,0.32520244965842204,0.4511110994414721,0.6806090382797584,
 0.506540117246744,0.23083026065402257,0.8263496386244373,0.25521438862604706,
 0.720307632198138,0.21811623019647364,0.40038149021986047,
 0.41703055584787696,0.16680205670992299,0.9072118430034928,
 0.6545115323903731,0.8444510951009753,0.9895423061872607,0.2404948557220331,
 0.6580099288503404,0.17330467350229528,0.9444427307016752,0.7275888246873501,
 0.9443436174910672,0.40452818356128395,0.2750584412503938,0.0917497277746484,
 0.5148584854677885,0.8370812446243572,0.96322861572336,0.4507045765809361,
 0.6756996068977835,0.9539485985150596,0.02388119012084089,0.5736172241761622,
 0.2876255978559109,0.6970707974589992,0.25608571867495356,0.0183051691767685,
 0.8318767674669854,0.10208811047037158,0.6100086223847839,0.3827389567582712,
 0.7130865622174652,0.6479582543249895,0.49849708820343785,0.0890148003808019,
 0.3406706568032113,0.09715849883955818,0.6573427516770693,0.7374163698675378,
 0.8226935566385909,0.9655093081608727,0.4846028153051166,0.5579254795057453,
 0.8926785502135433,0.29219540620082907,0.4963819298327271,
 0.14476893352809406,0.10909851385392333,0.06671693733104866,
 0.9790240580057042,0.421391657556044,0.378456029046924,0.8724228671445473,
 0.6907908568474745,0.2170024748642726,0.7893337051323768,0.9810369646524865,
 0.6858574347779013,0.1298480128856777,0.8600090702110127,0.9610271184204144,
 0.23788812836811457,0.10519063095304937,0.3982763203156251,
 0.9746939116277529,0.16932333180059966,0.6903192813413154,
 0.07362338563857829,0.5034459848517142,0.3317665611049455,
 0.015794053781077455,0.02139764861834559,0.6259044605658678,
 0.1984445244980403,0.23853233602584967,0.44847332233127546,
 0.9036927140481364,0.283606202080927,0.5299138079902973,0.1987311147913967,
 0.24504537420544237,0.3688459063562316,0.8712248807308796,0.5224618398506451,
 0.5196531787763403,0.39307581377910106,0.3120940783956859,
 0.011924630097356115,0.4973401967885984,0.9641770582170157,
 0.29290007369577586,0.5094788036839455,0.5866535912095616,0.6557919193608425,
 0.7170230648302989,0.8546685875479294,0.4769440924959365,0.03856407956924346,
 0.668063493871673,0.6589647047479884,0.450831690865377,0.5403843032382427,
 0.32219679700731085,0.8758238395920032,0.45152258037425475,
 0.9531528581934597,0.8957436156988086,0.3418887223731809,0.832240743891389,
 0.6012278228002357,0.10554106404343933,0.8717061224282632,
 0.49301803396639554,0.719084149880598,0.7807172398143094,0.17383654538398963,
 0.7437712616452886,0.32285839121403803,0.31322579537147754,
 0.20165857932533005,0.0977753441960284,0.5532458487700539,
 0.09881070856493046,0.2883003150033424,0.6537509786916886,0.6667467031465254,
 0.3396761512262183,0.5295265116934763,0.46648339814569095,0.6406567332764812,
 0.8077870887523289,0.7729680838162841,0.09928544435775344,
 0.33786511844043676,0.2988511559290046,0.43415025089525994,
 0.4396534026940443,0.24853782518788858,0.9074529612808322,0.5729829771895072,
 0.1582659439387386,0.3281328284867534,0.38066806682468945,0.9237293838100712,
 0.4722922800409899,0.8578643424769433,0.26761662414328413,0.2889978270931568,
 0.34446041531199256,0.43745626303042817,0.09932776810552923,
 0.4206025027955268,0.04186006479968141,0.18040739172904108,
 0.13239279020079364,0.021469200621326512,0.9991793872819614,
 0.9994446511205726,0.9638521979854456,0.7165750901464841,0.9192063639457447,
 0.8976402569694416,0.5014014826668919,0.3527895710354585,0.2579954168904589,
 0.36154993506035804,0.38781028855154487,0.4755969841574986,
 0.41613869276840787,0.9708332935913557,0.8346849444019946,
 0.43116379258726434,0.914939239494434,0.5690829936902977,0.5959449327967505,
 0.12599458036285893,0.3283444087383378,0.8972446049582581,0.9167268162549919,
 0.5413855061186295,0.43462689567526025,0.5928331109497678,0.9520102522017051,
 0.7594322505124859,0.5077308726925068,0.876110187536115,0.7744519353244055,
 0.30925733582697146,0.5039076406075272,0.3635618788337496,0.9134163427337477,
 0.35510668302219406,0.2623588952307553,0.3604448068590993,0.4910050170101681,
 0.5211781530525291,0.41184282787872994,0.2583221565083137,0.1667727100341314,
 0.17314398623773497,0.7010004953448046,0.554623473628368,0.6983364770114582,
 0.5165001346220139,0.4373366331807602,0.27514137723414667,0.318198551589126,
 0.3218017967601987,0.2285675289874951,0.6410669448869019,0.7778246276234422,
 0.6534731256223347,0.70748589099269,0.9828788476621162,0.4917142467758201,
 0.6246524915321248,0.8616619529664027,0.4020652129773442,0.7812991787557096,
 0.2541624559893163,0.8501520186466794,0.2819995340846616,0.7696477443341547,
 0.6416606470774404,0.02829513152600238,0.3602822526255939,0.9548577794171436,
 0.07934459931216853,0.7817364541217122,0.1352516453853171,0.3003148755982511,
 0.6127829962665806,0.5027101237499766,0.8146699494443439,0.3863131883122648,
 0.9685218514844512,0.38884682732331854,0.18478741267232635,
 0.8443997765954716,0.9955002301163207,0.3643869192368072,0.503397814373723,
 0.7797104602921514,0.49286783821639546,0.1411247724102357,0.6826968966636181,
 0.7843177668701632,0.9448296637094193,0.03879783050026597,0.731889267053462,
 0.40162023602068775,0.4854362424054216,0.16279393284035903,
 0.8440715820510651,0.2169393910045868,0.48505118228636424,
 0.28954986617356415,0.901567567426435,0.8655709783892581,0.14459713772264338,
 0.5927020435244456,0.9675503040991087,0.63487383475752,0.06291948537413317,
 0.9720760889200278,0.4434027231352169,0.5576617746364954,0.578505076845031,
 0.06552752607420154,0.038055088221921496,0.8810561010882032,
 0.9680145594507913,0.27527155289350513,0.7127992095134166,0.9225492316969638,
 0.6826366778882653,0.04477166113007358,0.015456551787504602,
 0.5239856186485315,0.1180231763734293,0.0408105050995301,0.6172750619810502,
 0.2619223414807951,0.19542390321143888,0.17698655932674434,
 0.30501889911040336,0.3268676651090867,0.82527682031677,0.1840301328823104,
 0.6172513823147526,0.048673861643569016,0.06337033517356017,
 0.12351534767349548,0.663527740388293,0.2149176971647645,0.4543408975296781,
 0.8764125369466017,0.5403415078535735,0.6098480174235074,0.22751294958153268,
 0.31501331376288255,0.1245285966155758,0.22662970778980673,
 0.2606225771797921,0.6762246663182677,0.2347288725578257,0.8547521457165801,
 0.2921398966663721,0.1652765079879498,0.4426185290058201,0.3983185540657974,
 0.22954059815973382,0.5402897366897474,0.7253822543375436,0.8422433413293026,
 0.19695320487284929,0.422808157276125,0.6520873297737236,0.11422149547311955,
 0.40709565912885104,0.8029053107304034,0.30127498240740336,
 0.6101152280113148,0.5792264545937985,0.35404327445319794,0.9507271004828749,
 0.2200205171923202,0.3835106970314708,0.09065401436724796,
 0.43512677173421344,0.360584853250668,0.10427954219617097,0.7079585021553436,
 0.3218488251850631,0.8300874837864969,0.03289015008831786,
 0.06289788812828301,0.72265605918568,0.13095512729287695,0.2138600662906649,
 0.288902479258194,0.6154349774856678,0.9140356897800679,0.521230481726053,
 0.13426608228376335,0.8318057215093372,0.31493618401092516,
 0.34510192350356883,0.664147977245755,0.32746609886606115,0.8797210837507508,
 0.02431381324011428,0.48372266290940613,0.3733290233681088,
 0.7789281905425998,0.6517531299910848,0.18512439930471136,0.1826622732393297,
 0.5212171559448546,0.15248846470468902,0.5940584171101052,0.3585221988827578,
 0.48660621268898274,0.02410838588433839,0.4686252775254317,
 0.14549390050142885,0.1473736004607924,0.5597806603888198,
 0.21766172577207943,0.5071228853128135,0.4247520026634266,0.2431926034604558,
 0.8397511994194554,0.8498650914857763,0.46533809359338174,0.3294466363366493,
 0.9972652988574432,0.028320702265822506,0.35658076213999235,
 0.5760819504029714,0.896560080585746,0.16810844946451053,0.48741350427352337,
 0.3092540643436197,0.20617612343488445,0.4172415784279132,0.1739585776461905,
 0.40013026944294405,7.997312894472142e-4,0.8627427984816557,
 0.4395485938631778,0.436467740276124,0.5061947938384782,0.3870296445507708,
 0.4049064739687598,0.3790895575422677,0.5258081915542446,0.6222018777616491,
 0.6000348007346538,0.9689435931100208,0.26255762633250757,0.7298975003538595,
 0.2582574683350478,0.5856070021073078,0.8243689706971358,0.6480902789215885,
 0.6272050285909675,0.6842355327893982,0.08241961003635201,0.5437890561079941,
 0.9095984965444158,0.8596668804623346,0.9718518323315203,0.980319730656005,
 0.8705910581452474,0.8765909074080304,0.5507742351263052,0.3040950448400056,
 0.8532377478860551,0.3421009159690316,0.3632917855819595,0.2918839131997206,
 0.8825135265103916,0.3181300335008145,0.051466340522970944,
 0.41817583643141165,0.9670135250527806,0.9469215497810867,0.8723995813407567,
 0.7829502975044239,0.3975008110354644,0.5345567447688832,0.6399638529254069,
 0.8634964832036127,0.6031705846272044,0.08958396058262452,0.1703591961919788,
 0.13192497785304003,0.3531703414643421,0.021826466433427893,
 0.4783289587378018,0.29514837982938424,0.3081939813361587,0.5223684833681378,
 0.27755157947086995,0.3624001709376309,0.892310383021502,0.40722983358073506,
 0.8505893918838632,0.34966429835223833,0.013374678086851999,
 0.022526106490296405,0.2565409291481513,0.9208400194740688,
 0.5567971184656043,0.47218756911851656,0.6940103285032372,0.7233731319208652,
 0.515702033766319,0.38779950864512236,0.2686917877742271,0.6113938105193225,
 0.1634678839950109,0.46741254815026345,0.8795226481244025,
 0.10486063710330729,0.5127505414034029,0.9132381264130836,
 0.18377166992449867,0.9685203209519787,0.6531879501149013,0.7517646479711809,
 0.4066186803100681,0.6592083842590708,0.9954943487200404,0.6755052626538979,
 0.41177764116480964,0.15203513313049333,0.17386698792855337,
 0.9990391440276746,0.4120580927528673,0.6242206235851442,0.28142717406831363,
 0.5177198402436116,0.9898291044028663,0.1380598222805337,0.7301365023711945,
 0.8439289814962657,0.36393701056099415,0.36727606748364805,
 0.6006898941580644,0.26954342511969465,0.05400287341991983,
 0.4154083060288656,0.22563121880183834,0.925754907464417,0.4174895514643766,
 0.04755290390910283,0.9332628360181094,0.45738933840796503,
 0.032852081360132956,0.4716262339491226,0.4358846177076663,
 0.05277424369019734,0.6130030563419885,0.23323434275566313,
 0.5628967675798453,0.4979100110803436,0.5137977139687234,0.9416467948997992,
 0.012434895416745362,0.22399165955898892,0.976649614952334,
 0.6747726579565173,0.1091664075146952,0.2902229821840354,0.4209527592138932,
 0.5916643327714441,0.05870612165619915,0.03256843875775495,
 0.9755069591187675,0.8095254909942847,0.9537402406140774,0.28873400048389264,
 0.05304842156004397,0.3678600759580891,0.016529330521899088,
 0.09780158607476719,0.906423970572347,0.455245140707232,0.2485633774605207,
 0.14061524511563062,0.7998098281589722,0.21188605568242036,
 0.9966857004201635,0.23450152543805447,0.17554064112020662,
 0.6160287286697017,0.05325630197013109,0.37348500402946827,
 0.5752542149684713,0.546959723727383,0.9723501070379493,0.8245695497173333,
 0.782235994082583,0.35525913812897736,0.9597295784631814,0.08901076478180514,
 0.1120810836924292,0.11101683979767352,0.6798479248143967,0.9240152147129811,
 0.5181993912386864,0.7595816958438455,0.17554346311775626,0.878008251566139,
 0.01351312825779405,0.43989960683832185,0.61457299492991,0.8287278684991399,
 0.13606102963762978,0.3324514427099984,0.6860699904274726,0.8800586868330855,
 0.8462659287880197,0.7646704640793625,0.08613665361729705,0.7643085310755012,
 0.2706801037972717,0.4443428824940242,0.5383786867978162,
 0.033957529249209806,0.17947850847251345,0.1382927988440933,
 0.25327844486106754,0.13996924989039472,0.9287075069798422,
 0.3284341732556283,0.06559111077838742,0.5244676235470219,
 0.05452365700690531,0.1540248135633221,0.6792254627330994,0.295699049881492,
 0.5064879192854501,0.9872396593181609,0.09756707044789614,0.6335690937807648,
 0.7644060917282107,0.6288731415838429,0.8751059125887088,0.43306273478912893,
 0.870846410156817,0.7108432186507316,0.5301555297520546,0.517113036998544,
 0.5271232918423955,0.8563463977869591,0.5164723270313019,0.6556309170347818,
 0.5388773973240767,0.5103429801489138,0.9561118564372972,0.8078157741110689,
 0.928762682546909,0.38252895998133396,0.15958875992949384,
 0.07617750232977483,0.27721223492432734,0.6553543841918811,
 0.5288687490446736,0.11469611869373475,0.2763899617792861,0.6793233127513522,
 0.003474160846618557,0.382557087189186,0.8226472628526614,
 0.38218612053127465,0.472900049079957,0.2683985477981493,0.4143282942176477,
 0.12396643536320273,0.874431447899272,0.2584701611715422,0.04246102571674415,
 0.5646665708633748,0.9059950743144303,0.8007705587522479,0.09041209417321039,
 0.3063519056305668,0.381009964287917,0.3774563840461528,0.4931376418527843,
 0.517936552758011,0.920889408828703,0.02733145143797555,0.9294579255414258,
 0.009924988915027022,0.4931249329498798,0.6113496386356978,
 0.6273084367164266,0.8162752138613353,0.869837222261671,0.4747363305281065,
 0.15994315632372458,0.8813068794715956,0.9131059995014699,0.7253403227151245,
 0.4060981175767593,0.27952658257388663,0.47037163628713285,
 0.8377649817773847,0.9229038411695624,0.22942020937774776,
 0.08501298787028133,0.4157102383668698,0.5892000565133876,0.6247329533344232,
 0.5444912201631277,0.9971417724040789,0.146434648511236,0.19112828408154103,
 0.6541612442752702,0.9994011841140039,0.24256703753955877,0.7666577144703725,
 0.5986340884430407,0.35502468904245543,0.4852930779423421,
 0.44659485516637665,0.036687833294631544,0.3154895313406011,
 0.42041674167886556,0.4880229557616427,0.38273694207656284,
 0.9682795240202788,0.8418059655929337,0.32944870489659417,
 0.19154053658257175,0.11688813596018066,0.8002105028644496,
 0.5711994251697708,0.9105297314028782,0.8061425037226381,0.25320279046045646,
 0.6666952808818307,0.7705377995549922,0.5030437103146466,0.9641745803971383,
 0.03909988108669982,0.3973252372033298,0.5316976743452405,
 0.10894548994324516,0.6218236041456328,0.5181690782168576,0.267008993254102,
 0.9183250036049732,0.19357195483810608,0.39767724825343,0.45121313593568446,
 0.4330550965563922,0.7653567061682616,0.4713118321129648,0.18265750903259914,
 0.4860861603576092,0.7525236463670184,0.77997871338653,0.0879483257478505,
 0.8611589850019423,0.13199996037101758,0.157181387508728,0.5612473011406525,
 0.13551387290790518,0.07395126455179135,0.13556740189485716,
 0.2596230834156559,0.45114376030812253,0.5485095950592614,0.8163583901900723,
 0.7668730103825182,0.4998925789057733,0.3619556057935628,0.26772290666271603,
 0.061392972007197844,0.6198787650784111,0.11408290643842944,
 0.41435601796349764,0.10895447858337715,0.5176216727324439,
 0.7528639277991526,0.6152769314135886,0.36859957720588143,
 0.27923641291582435,0.32525001369812556,0.12344085512468161,
 0.018708901158713487,0.5101231323183909,0.8911360351768676,
 0.6301850809350064,0.5905174620241922,0.21648070957086596,
 0.06115855575265039,0.5859192978707471,0.45455128070813844,
 0.20389059528127962,0.7847498209702275,0.9469751743839938,0.7623260724836625,
 0.2560018033219451,0.5275798276185522,0.0770374725774301,0.46051357280364846,
 0.10145969888790785,0.9427295121387879,0.8731017437298394,0.8640031233010013,
 0.8497039758898094,0.32955576735979863,0.685992611734275,0.10592469372255431,
 0.8339907066586377,0.248166342043742,0.18611460404372837,0.41466494137114396,
 0.12120172571580523,0.2669153771316798,0.40813054993247166,
 0.6874077384603035,0.5304547264255182,0.29554030580142465,
 0.26933325270402897,0.4838731695420041,0.45879182645306604,
 0.9331987634223027,0.7884441989963868,0.7548091326074077,0.7241388049353825,
 0.11226342157659497,0.8212582253928524,0.6749341741354837,0.9372754471392549,
 0.6578459452120194,0.8290456508293632,0.6300211937285041,0.08986151262836684,
 0.7096977499921564,0.037856453268782064,0.10777618457471516,
 0.7375434468124795,0.047665968376876644,0.3033229977457982,
 0.28808484309723437,0.6638894637472869,0.6609492139318345,
 0.15044206510333635,0.7551992980461456,0.6773768092116286,0.9885888186107,
 0.2249970599470742,0.0770507442130618,0.4346401111194197,0.09447731379946034,
 0.39944162380780823,0.18429701907498175,0.5577463923222654,
 0.17900793163681694,0.38156757073654013,0.5020439977538994,
 0.8159176601163337,0.3602979430682144,0.9729718002948896,0.6988251699089817,
 0.9869451462575123,0.4958169043706019,0.7371636579282826,0.7541541218751687,
 0.39193622303085873,0.703809479549981,0.780441492587679,0.032874006755062024,
 0.7345704590738991,0.7013769060924687,0.7761554002061398,0.40019847921520135,
 0.03851131636007843,0.581860257476114,0.12854923950130503,0.3419371348925463,
 0.6510466039013114,0.22259282985422635,0.289477589518372,0.9059535578473772,
 0.7243339157953136,0.9193860972576222,0.8461955765072027,0.852465965670777,
 0.49801020316568945,0.0224614335444413,0.5211065823718865,0.3918323966359115,
 0.5468475983560804,0.1397213156944036,0.7158399849278755,0.45777275500244985,
 0.28660375713143205,0.18415987871145745,0.7303952492706266,
 0.2694886029457628,0.0491419156029711,0.30977436345914033,0.8838515928197479,
 0.920963010624341,0.46635650096945447,0.8690156092760835,0.5958674351144269,
 0.8739036717037308,0.0811358652804538,0.6257920361754474,0.10191957834676935,
 0.2980754387612917,0.5480065359654254,0.7650416812025345,0.09239235611345376,
 0.47560650154632045,0.6683254995194295,0.7781971431076679,
 0.019369348688002552,0.6949670250089786,0.8033246673963381,
 0.8312657479598109,0.8650547435383379,0.5608252997037246,0.9895769584226866,
 0.6119890774566411,0.1579760286991161,0.6154013087590208,0.6386966226359504,
 0.6305311182911084,0.9874397958149556,0.27894534179674413,0.5585852446598891,
 0.2651096099972746,0.3779216734311246,0.7838241972683249,0.524820246848393,
 0.345042774276332,0.7527084232927304,0.9136143695079285,0.21869872282991742,
 0.4328269016673518,0.5863416066523002,0.812849644578312,0.2631297428767757,
 0.14176161331278858,0.33021226332838194,0.8821207165354982,
 0.7243732227252422,0.5625837459798334,0.5938672668145214,0.33064241040231246,
 0.7666951586664966,0.8444743521708674,0.07444534557316684,0.6845691148736013,
 0.9744929838044457,0.0017978755331995444,0.49135475213396707,
 0.7792667197770498,0.9679649924618876,0.4099820023540698,0.548126412582688,
 0.7275170308735717,0.3021874557951635,0.8516744198222643,0.20632378115774053,
 0.7225558988310977,0.39826184797553865,0.13121869951443887,
 0.30930701222282864,0.6114387894439987,0.8456854314830184,0.9087323038986074,
 0.5901835007963877,0.35879613995412696,0.6612417202780281,0.6417370477249114,
 0.5491911775603718,0.4089245738018962,0.055127378783809466,
 0.3309657843172512,0.12115238926622418,0.8071130959892803,0.4883060287549912,
 0.563680890014976,0.44785017398327,0.4649018868758915,0.25035005228790985,
 0.8026962798148882,0.1577819301439053,0.7266181682614679,0.7182891549234045,
 0.9318247933894487,0.9216657077173687,0.632026770717989,0.5694747500023093,
 0.17954016062967937,0.5676839539741863,0.5564397303649492,
 0.16521561630426584,0.43349513119994554,0.016353273653322953,
 0.3462459859223701,0.6977188691238516,0.5336638973025247,0.22429478301181183,
 0.850229831261006,0.9809445265519682,0.49918444978927634,0.7576600449425642,
 0.17092679108911224,0.36788591509731683,0.08427555801289488,
 0.7139802886790755,0.9028078021502342,0.5652670219384955,0.6960714127954183,
 0.604748261155085,0.8042538779274401,0.30937261075737443,0.0883891787161315,
 0.9277941375613479,0.27481660064718283,0.06780807133010536,
 0.8248189190363283,0.5818687352873017,0.8708317874451543,0.24883706888442092,
 0.43651440296703514,0.03229879773409694,0.7829841998870357,
 0.7921859165516781,0.849435040643789,0.17599183654839157,0.09209398782484657,
 0.9393729334063725,0.22686367084300219,0.9149737487678097,0.6306677296250708,
 0.5323046769525329,0.6285468698399529,0.7746464565790647,0.5638477991484525,
 0.520835033055896,0.868763427466224,0.8117945214641477,0.2739621869407335,
 0.6633891664490548,0.026109841308702864,0.5905093971178746,
 0.7834000337978094,0.3774701933583604,0.5275224791873813,0.9244047893053372,
 0.34974642168068215,0.7088980937751556,0.6460216628605682,
 0.42640804096140794,0.944502196106803,0.14701478778141086,0.2000419444921132,
 0.22303033529608285,0.9486426692338789,0.11795800398997891,
 0.49452927752246234,0.5499903692992292,0.5813229508630418,0.7450504346247335,
 0.5322177617197283,0.5849799119154432,0.5169821227552273,0.5329469881649271,
 0.7145701118682204,0.574537720711179,0.728499734432417,0.03739517916072965,
 0.9778339348618706,0.03128638909883874,0.42399298920231865,
 0.6280349440067425,0.13392924065779843,0.986746072458241,0.44955189573075405,
 0.19430288016790065,0.08066056919124676,0.473461061751111,0.1594081575685018,
 0.3391851431997547,0.22147700668540138,0.6714582210605692,0.6266442765307155,
 0.8413957881048086,0.6590021569879932,0.7950443287625377,0.13583807634502398,
 0.25662277976527315,0.8437773149089791,0.9454937899813175,0.5333193014120579,
 0.7667553885831929,0.9147229393481708,0.6889230080895824,0.6133701336793901,
 0.02909339554933399,0.775607893402679,0.6753166944995388,0.6554668934101133,
 0.5542909409547218,0.9596972148453544,0.35096171410974975,0.7341707353373198,
 0.09985978734426926,0.9320765441287974,0.8287432292000949,0.2881386237393583,
 0.8434930505571252,0.004251852895803876,0.7208880538008706,
 0.010103280298156547,0.8599735835353107,0.4298166143678974,
 0.8804249292578884,0.41387077270700934,0.6095322468619179,
 0.14811160126357525,0.09132680738714727,0.6212915199812139,
 0.5049284819154772,0.32161897213788915,0.9075234637821243,0.5320852610228649,
 0.8893164076792536,0.8087988247011225,0.15336399890454178,0.66624604201487,
 0.2997407579883793,0.6083241249611004,0.8490181279889916,0.49815575628512,
 0.8549855787035842,0.931464274946597,0.8850177508605203,0.5243354688537951,
 0.04619748682375702,0.9635160764245381,0.8602161428914028,
 0.13491737755484523,0.800590479096885,0.9892945348629176,0.8325984775199997,
 0.3091822432495537,0.09150477186477857,0.8508680903700951,
 0.08635326140013855,0.04028965427058884,0.30485442841417787,
 0.1989170607330929,0.49490935590754115,0.2746433300231933,0.627264976479511,
 0.1959894654306178,0.39828289939904793,0.3936345434897972,0.3110939660112738,
 0.7780443989678325,0.5157989787691297,0.10666321906452647,0.8494179764460893,
 0.17006413638596962,0.0518283531490884,0.559037922750882,0.2544675154718158,
 0.20960272172623395,0.7615378574464928,0.5907073024643574,
 0.11718746689209258,0.29855783159675464,0.9391784969504942,
 0.08699559009159186,0.9318482039324099,0.8107698114036185,
 0.12306372565023005,0.0660055252272258,0.6531671956140829,
 0.060294832049483826,0.12859148492921169,0.6819047135303204,
 0.9199467748843905,0.13630296936856257,0.7372180655283223,0.4949382639937975,
 0.42794901346884473,0.12459658222062708,0.2967916768782052,
 0.40852924963246073,0.9733369861590835,0.8767828863613407,0.1927202971303863,
 0.602801631395511,0.18108812341704073,0.17066349067535502,0.5434020744950113,
 0.7365050429131468,0.05586641022088301,0.10160581586653017,
 0.5714442306465011,0.9498330287590264,0.6014671242911209,0.8208641750006272,
 0.1865549315038626,0.4976313177315297,0.401781549204417,0.7823041325324531,
 0.35848298180734095,0.9218588220422342,0.6149036423403499,0.4954103792980875,
 0.7365063285049205,0.25445977859865676,0.8513488166301709,0.9709867545295136,
 0.26195502694288064,0.011226340141365698,0.7030960282630411,
 0.9722723197405706,0.4431328928777599,0.019108898584154432,
 0.9626422846612498,0.98703192645324,0.30070016504011565,0.5181125919884065,
 0.2970008822009339,0.45599003641938496,0.2592587542826661,0.9416199590681468,
 0.6632270152705323,0.9275092040753029,0.14998999710019267,0.5069097252341893,
 0.6510127193591011,0.3861965484490604,0.16389696211462057,0.9074005049264985,
 0.3018425722246211,0.6171073505221913,0.15229415256584222,0.2859238356110668,
 0.15615851661343294,0.6257211210638185,0.19508060330416355,
 0.27550105917200474,0.741023960154459,0.8643751766521984,0.17319222208895857,
 0.7129800122044199,0.4558750058940453,0.6104443801972647,0.20843420050278594,
 0.16098630120739665,0.7783079695434034,0.43693591286491884,
 0.8850968855673075,0.14668652377467428,0.471036518040995,0.17395702655829826,
 0.7508896185498188,0.6581377426339576,0.8162406508570483,0.9783640587542787,
 0.2032626987956797,0.43367630539053614,0.9953028713559505,0.2639462737403896,
 0.7908898757541797,0.7118560849311171,0.6750402350149302,0.275747078710454,
 0.8956342652983099,0.7105502613947033,0.9489866822446209,0.24979901049563424,
 0.4387673059107525,0.9047654630835734,0.5627521717716069,0.03942336405718827,
 0.6396226576121267,0.47447516738822215,0.431676918263738,0.316279000710732,
 0.696124305396135,0.03136246246448959,0.5764691906819523,0.409109644345936,
 0.21183384030815322,0.5767048849731216,0.0030666197243299775,
 0.5928489758261057,0.7759936434526996,0.4032935138829251,0.26602001861005253,
 0.6198563146193878,0.6418571446670137,0.25723705305316225,0.4353696934099527,
 0.1643442959621232,0.840132359015448,0.5946554903291427,0.8862514905645817,
 0.8637957384916576,0.8003353369148138,0.9720332012869075,0.09351267248056816,
 0.7814390390987451,0.42423027414251735,0.9717066623174986,
 0.05723818006275527,0.6614455867658174,0.5622121584062291,0.4098545924683721,
 0.6969817909466994,0.6734921178025668,0.08635813113629043,0.8434711236555353,
 0.24085217627529154,0.6425681724432398,0.9532295905913694,
 0.13333502262284025,0.25940874279301906,0.22704151537920325,
 0.9143904791596744,0.6312798301974273,0.4695768776539174,0.5005355988418257,
 0.8641130875578393,0.5292761615225866,0.7412398139950583,0.6801137139384025,
 0.38143672566454156,0.8226023736001687,0.7481016874011985,0.7366853752816578,
 0.75775649718624,0.8065816133573886,0.09399882150179373,0.7948809078889856,
 0.5773506935876672,0.48049075594076285,0.7429625078018218,
 0.16871804341143104,0.19460128899262996,0.9942223026767287,
 0.4893163733613515,0.7147931790570426,0.6466660439627171,0.21675929338867594,
 0.8068975529510879,0.17252571825008045,0.06794741536205096,
 0.46125273241773246,0.27462258619391444,0.2585347957446411,
 0.43039794474999626,0.4663466638276994,0.33951654486771254,
 0.7317988055911511,0.9571833761480081,0.8596185167095329,0.02085134314550119,
 0.9624065828908127,0.34497762765746876,0.15772702242994252,
 0.11102759633665449,0.5861813468600208,0.028489186698809843,
 0.7106579089183346,0.13947389038872982,0.31830510811921353,
 0.46202733028176235,0.14879795007404895,0.08721695107110816,
 0.19880260960525298,0.016713107573373298,0.49472689897964517,
 0.6086550105711205,0.3288436393901979,0.9753799776489006,0.34856688656944446,
 0.8995062457939571,0.8726419562425355,0.7620682411894246,0.7242768920868529,
 0.4495345466748275,0.2647994894840666,0.34704128618855934,0.7734105241206921,
 0.23943198639939012,0.42732398708600705,0.7160516388556023,
 0.06373385747519844,0.294219316820183,0.05722221463035382,
 0.20929813089174143,0.5419578195863608,0.38891264571297435,
 0.4227680940712826,0.3966672380946301,0.7589574056401464,0.7643383596163205,
 0.6331523677773542,0.6367109010435413,0.5237434148459711,0.33425896226049456,
 0.9031925657851778,0.4960275438671068,0.08526310389178415,
 0.41149181508794774,0.7232123775031786,0.4675531864290018,0.7265004652303925,
 0.2217134244323138,0.938527092614487,0.7362903825888808,0.12792105400425235,
 0.14966378961136206,0.5069028082500897,0.8685968898733379,
 0.29276421553110044,0.7132610233718681,0.17532774986476318,
 0.9550587520546441,0.07447116440301527,0.4875713894411706,0.9227107010052273,
 0.8379987824903223,0.49158404631245,0.7050804342493318,0.5769646112556047,
 0.04028156058055854,0.14135757867456178,0.6310067131477505,
 0.0643573564394202,0.1313177563832041,0.15245511526837463,0.8218428086745262,
 0.07598401814388445,0.2879356490306759,0.3589667807481063,0.7960799224792662,
 0.323738816333787,0.3965889897209177,0.05615268105371174,0.23165972399653534,
 0.9887017720423023,0.4271063342487742,0.4970686814718597,0.7629280057638749,
 0.5326674584416059,0.1898213285180157,0.7727402188191176,0.32188055411433136,
 0.7712592242601602,0.1722256457054946,0.1824669224114449,0.4573442248491224,
 0.29157956205591606,0.3741046405370295,0.46332952308820663,
 0.9784975738250404,0.7032450198296886,0.9815150168081378,0.31665024520675356,
 0.7325175364442259,0.1387065821254111,0.10549171501504795,0.9755887372062869,
 0.39875794681567267,0.28374495798761123,0.6451886111282945,
 0.5809386193348226,0.6257448080119831,0.3192518069833853,0.6178011396624724,
 0.8745785352139428,0.14280255009025455,0.009091461065903084,
 0.025792092381515697,0.9768997535266524,0.9140112847857849,
 0.21180720714005985,0.09330494862596328,0.6319690412483928,0.736297255144285,
 0.49996647805080696,0.05281499673214918,0.20757707927430846,0.61792139896911,
 0.22174249087111275,0.2783064558565108,0.7040882850536277,0.3369224172461863,
 0.9059542219945691,0.12067072219755914,0.3498300135344967,0.7150022885254733,
 0.13795450180484103,0.9916385879689779,0.11501646944001276,
 0.7850380012997917,0.7706370291649568,0.6660428669516436,0.6099829867614296,
 0.7037488761741071,0.5244618012814286,0.7852427641568379,0.22675674669501344,
 0.1050258316656163,0.7704082056646314,0.27680549945170296,
 0.07254257185831947,0.35512354341256536,0.5711755401540841,
 0.2628066525400137,0.6107971763199084,0.26707507208106185,0.1563866089264504,
 0.29457623927355847,0.8808018078191404,0.42058038007116916,
 0.37118768875166563,0.6619193631838701,0.4937003137340842,0.8861980734949833,
 0.836038373193706,0.360124257983502,0.944071861022336,0.6399168504983165,
 0.2143737735261595,0.1817066608354201,0.8134939345845191,0.4558301390161813,
 0.43617618737066366,0.6985381219278269,0.3619584049485023,0.698595629641867,
 0.7895422559145512,0.7636956018152351,0.9725845593245644,0.9497325213830471,
 0.08202950846490431,0.96761001786535,0.9077422934311413,0.05255991244330416,
 0.35929735787944805,0.15661899553982883,0.017738993104603695,
 0.9211050342255054,0.24876083300478813,0.7037676063846118,0.6898631751793823,
 0.616568385678282,0.8352907211032171,0.852771186012006,0.9345407010808459,
 0.34931226368941015,0.5519626214781309,0.1936159083129695,
 0.47990033132464227,0.5453122551012775,0.9606668697925658,0.7802885425394965,
 0.1735826279921011,0.5606214331302085,0.9879770615642408,0.11311569945806033,
 0.05490053715387966,0.568660590965894,0.7321086480988337,0.3939560502847803,
 0.21535540152216204,0.37593193582454154,0.6822386604682917,0.720461260307891,
 0.7915769077077245,0.33783095968017274,0.7310190974951619,0.6018971448005082,
 0.05573937874216994,0.5062419549338358,0.7938535739669065,0.9376407150757006,
 0.31549123069951346,0.3194100229210548,0.9756415315330023,0.9153481985781491,
 0.002344280005255217,0.5762541339502856,0.34041496866771614,
 0.868530322718505,0.6597834494660807,0.7654454405123362,0.7216445226962931,
 0.8827746149285874,0.04086819285287113,0.7589878996163644,0.6321120628418513,
 0.4452184081836208,0.5196064998972731,0.15774546335721495,
 0.020866173372519836,0.8733060034600558,0.7999025539805069,0.552835477489574,
 0.009470632680069535,0.6719823327105534,0.8684709800474637,
 0.9332439517047009,0.8506084788385615,0.16830315217551362,0.0527467788774304,
 0.1376905621551141,0.4507379857153574,0.4341807619916428,0.8493398779559538,
 0.7749967401556344,0.04068339138231336,0.4752885947382498,0.6150493677492633,
 0.6445739145879937,0.07886981896627632,0.12623463949696812,
 0.27883868793026245,0.9705285747000252,0.4770743686020279,
 0.06424087482987084,0.9061952791107293,0.5071140863029201,0.9009774765542694,
 0.02910354501330681,0.8183960575462857,0.37366019903803693,
 0.8856687850514448,0.17008821178029032,0.6051499810989751,0.5036806492416465,
 0.6322579881756479,0.8087767497961282,0.779562409869621,0.08473852268509563,
 0.3707862354053235,0.40238639887044625,0.9919472613019091,0.8049113105509956,
 0.529351951146594,0.06106541438153823,0.4878239457334903,0.5203024409310684,
 0.5947826816972083,0.6424205001215013,0.36498186957848255,
 0.12289222152727197,0.010887453451064122,0.43245681869335495,
 0.18990815954635154,0.33981168476084356,0.786609728423364,0.8350647631270246,
 0.7650937858584783,0.593609055294334,0.42263436617393424,0.4147213693523507,
 0.9955367274923344,0.9786734000721518,0.8933583152546103,0.4872292584523531,
 0.5678374610952426,0.7907813377565974,0.3133837195005347,0.9001274313116749,
 0.5980703356875783,0.7649403541768142,0.9767367135726421,0.48690723773328426,
 0.7626220740221376,0.12941092975307467,0.8521914620818196,0.3555493598630408,
 0.8108610584145357,0.43839908095915014,0.25048779552307854,
 0.9309191560910732,0.9407585541227742,0.13357793826593645,
 0.36607346155367304,0.3155899084875833,0.005734103976268501,
 0.42005046919251865,0.994694594038256,0.636591706673408,0.6771405802143039,
 0.9938170304519526,0.5352619009260189,0.25654000336370486,0.701440575293019,
 0.2934597500452303,0.25771710935062353,0.37237839391101524,
 0.12370903566797575,0.9649031923690712,0.4277390948975861,0.5590078361425954,
 0.3126171856741222,0.4709431622866259,0.5453674044517834,0.6089311334846765,
 0.8209525915300108,0.3318665711005263,0.5521532247682167,0.6809587063599424,
 0.5579350020817441,0.5203603043704118,0.21780770960455298,0.9051102647020608,
 0.21970847183943087,0.48758877782826404,0.9965610061830568,
 0.11613765191450665,0.5687298817755936,0.4892488552879497,0.5673312178297919,
 0.9079455423296636,0.7753056859984033,0.09483299099356068,
 0.20291197631063684,0.842098045536865,0.5818268351628614,0.508515844178827,
 0.2698031412730527,0.782593199691531,0.25019928255930635,0.8017289277296358,
 0.9190210061747642,0.020000191455752936,0.36800881555720644,
 0.9697690594779933,0.8155003870055256,0.9186383526065908,0.29082875142079945,
 0.02461304165598044,0.23190098387509428,0.09989375305539927,
 0.8676392137995004,0.013574433716134982,0.2963213106591228,
 0.3782585794682817,0.5012868095744946,0.628073567601996,0.04775525236495648,
 0.31668643348903247,0.03272774651283894,0.5261238768057698,
 0.7686715042639183,0.05192609182843144,0.7484180851041756,0.5016619090149583,
 0.3672536489857845,0.11803948520239127,0.7434060155475882,0.2958466500873064,
 0.4595289406884824,0.8119265710713106,0.39721874072979346,0.7780724799142433,
 0.9175736250765749,0.6483805309999594,0.5611880646733501,0.5500932892165558,
 0.40508698585347513,0.09125030072170326,0.3331139683385297,
 0.7664243277916412,0.7244801455942019,0.292821643661929,0.6650886123592262,
 0.803815151720813,0.7453110486531995,0.40252994382151897,0.5960868421859405,
 0.29933226205281216,0.22774485698650504,0.4778864491482967,0.579052966519625,
 0.6063628744089887,0.9485776533949776,0.7610599184663778,0.47695085340867027,
 0.5456370437037997,0.10645510982348705,0.37981360460725055,
 0.7571807623019337,0.28178063960841615,0.04908657751473067,
 0.5444852966317373,0.8677928324513036,0.6214378557942408,0.9973089807119669,
 0.1004581038517407,0.875240010639654,0.035712660286382425,
 0.14857114782167813,0.4048029414147233,0.3289450246276324,
 0.012510721164814331,0.9434333299770477,0.5080781535021521,
 0.9585767749686376,0.4983016074679496,0.08842795924602165,0.8839981701425832,
 0.3114674243044734,0.7785558052208571,0.13023022934120554,0.8433143359647202,
 0.6945403279469672,0.42836839538577554,0.9129998582621469,
 0.29764347273496006,0.43368322069310683,0.5626869994111448,
 0.5562285487339481,0.8202337703637356,0.7035802503375203,0.41381964906455715,
 0.4027613093476925,0.2920604302743297,0.29656359616134065,
 0.49152448945627214,0.023257212394405036,0.9779492262267748,
 0.05950811850792892,0.20433310790393178,0.5533228468023785,
 0.9191833754417058,0.8920216824010089,0.9475918456146852,0.9681143989311947,
 0.2280869038247081,0.04345787287610714,0.525736391182963,0.8367783315847254,
 0.40457752914937983,0.10664102186188695,0.9256641375911119,
 0.7460229074441297,0.04956603135622317,0.7839884406430048,
 0.13737769982999237,0.9701024806207065,0.999883325877458,0.5413207496967337,
 0.6491245143081901,0.4997518294610058,0.5726770911605833,0.9034420635847019,
 0.16202151988942193,0.06583134696142456,0.7901502496131098,
 0.7921373263196781,0.922679328640724,0.22774114375572685,0.9787600020989914,
 0.8077836579067592,0.6335964327387287,0.8776220039899978,0.6973205117524537,
 0.9770225775286251,0.7833367412348612,0.7009809696970535,0.6686150917246946,
 0.9968386107904111,0.18437478100438032,0.39368830625255735,0.887738587009518,
 0.5034587258214206,0.5065568647250848,0.033542800252762794,
 0.39920365780059974,0.6333251468296088,0.05313745701687267,
 0.5181656384710942,0.6844487813019143,0.595735679146478,0.8612245393011686,
 0.9776168719697692,0.2943367412687694,0.3810900522574512,0.9131128452698762,
 0.9445300701661398,0.4179516432042585,0.46608912617513654,0.980395272370062,
 0.9700318596334103,0.252377126106761,0.21310078650566655,0.3641695441118611,
 0.9363366314403938,0.4595435846084527,0.1950094839753458,0.5642438074499179,
 0.21152129619601168,0.898915032119621,0.6459674369822576,0.42953680656120563,
 0.9413171208851467,0.7053388437158866,0.5260126347756533,0.7093399795802483,
 0.6894766758289564,0.4291797262208734,0.3300405131774773,0.3821160878629605,
 0.8245746359139087,0.6344306991490185,0.9517428444359548,
 0.014756206816441075,0.6271590414066587,0.9816830484400298,
 0.49155133122787076,0.42220914303386137,0.8533409328991882,0.280615301348514,
 0.7692882207176976,0.8150432586437134,0.8480890187829772,0.7193405106483385,
 0.2660531377269775,0.7324343967231915,0.486984681393723,0.16760845608990793,
 0.8021862454857106,0.9371275294911664,0.7424546277854026,0.44847864307706276,
 0.4423841288842737,0.8057201317402166,0.21948647276680888,0.7639129473394337,
 0.5407637680057108,0.5441918602021619,0.8358946257116986,0.38694888979907405,
 0.6578135338111876,0.30778699300386303,0.23301995943906817,
 0.6688654217523143,0.7287691679745647,0.6673486052263786,0.7419933033402282,
 0.24484942174921853,0.6646145819763758,0.013803383003220593,
 0.9258156491616875,0.666857096144355,0.2481384012326857,0.9158570562437096,
 0.8034335382932717,0.31639677037988956,0.6674885666459696,0.2045115033629068,
 0.8798016821743412,0.18690112708636875,0.8235640398560625,0.8996078517136676,
 0.7545808670818754,0.6209361906390513,0.5832483014691036,0.5546859353445838,
 0.38196104474311565,0.027389289121958815,0.07897255416622229,
 0.9926543896167677,0.5923201801120479,0.5071977719401827,0.2673829721684847,
 0.43847933659269744,0.4127394120035712,0.7002343292611477,0.8611342379823577,
 0.42577283414192424,0.5061794239721934,0.6193730210034625,0.2927767405910342,
 0.23447632939547913,0.7202102324851494,0.27130410539013083,
 0.5238648014747642,0.31975049856973503,0.14395506107451772,
 0.22332017892114864,0.9112487805543389,0.8798753807394272,
 0.13159518384344238,0.04798876208842606,0.1232396659880195,
 0.31234949857631433,0.2655749623727963,0.8789482292553361,0.5467305209516604,
 0.7845275322562911,0.04044382005965552,0.18128436451150864,
 0.23592462498374345,0.1917570855644708,0.2816520748685203,0.4168524926156716,
 0.8244594396893821,0.9306496623018675,0.35943164059244126,0.5700124852801746,
 0.8444824351362082,0.13062878258589694,0.05015042526553626,
 0.2945565652743103,0.9430673883121163,0.2079014957241544,0.8133428012483511,
 0.13941298451943474,0.9165517736790304,0.3505089782211105,0.8765131629052778,
 0.39084112463996235,0.813764625000357,0.6708842984168054,0.04252242378473803,
 0.05231720516257632,0.5735956670347666,0.525536325264386,0.9220377292466475,
 0.02855977646781227,0.6160217049780206,0.6967173528152287,
 0.17689451439923465,0.24365833447196872,0.6969917306315482,
 0.14317482290263606,0.2238533471460331,0.18521412601508191,
 0.5132837119103375,0.06079060867716457,0.7874323135725457,0.9040821898220952,
 0.5888339613570105,0.10676461697942607,0.030294797215384084,
 0.16750023284932036,0.46470037371441697,0.03640332530402912,
 0.31850926775036925,0.27309861985219386,0.412383930154936,0.0545321712453245,
 0.19356530284412365,0.601052305993184,0.2508994840307157,0.7898179699857176,
 0.3616365248795328,0.03624764236443567,0.7961467899928866,0.362073698542969,
 0.20859296972044583,0.8020856547805923,0.365485579366401,0.6730211084499926,
 0.7326367673814883,0.4446379005967762,0.02822481461217663,0.6649441342135093,
 0.09358511594794139,0.8329689480259361,0.5880954693270964,0.4377103945576084,
 0.16693321992040044,0.6821383078910754,0.7375894301715269,0.6232865880520874,
 0.1763536936256237,0.5899252921783906,0.8322439636598462,0.3351343459013352,
 0.9967761537999503,0.26630065981855355,0.6968681022268632,0.6940532190132672,
 0.8937383995274992,0.8109010343789762,0.1609037207529952,0.285057239283132,
 0.26134879928757904,0.39666302319200053,0.0048921796951652,
 0.9106548005895546,0.12494498988834501,0.9938679668987853,0.7629584878654363,
 0.18454814011385923,0.4939163374557447,0.5306096791010984,
 0.09301220384137221,0.8868583211801582,0.03466299409646467,
 0.15324276347747834,0.2936300654699382,0.853245279670397,0.2822703992026827,
 0.15100208695141926,0.3384351168949299,0.19756500819070955,
 0.8498990269135268,0.16285428056508444,0.9739449118770739,
 0.09558966435710059,0.5624472861708104,0.9478912262438078,0.2864350460397953,
 0.1848061575851312,0.598311875787364,0.9283973079213199,0.6727893865403283,
 0.31036617675355027,0.5438391961888834,0.5105741288316084,0.454939929993606,
 0.26866426187408243,0.28791966435890015,0.03787374646838704,
 0.05553823891476495,0.9648335288719172,0.5097583777692192,0.3437163908125165,
 0.6298370826867717,0.4399526134686126,0.1391573008381941,0.12300407633622079,
 0.3819031307601747,0.4301044083054265,0.08159662194726391,
 0.30415069237961156,0.454608195969432,0.05526491447359727,0.8129715184877812,
 0.4075625883324564,0.5958472400434841,0.6908082694860628,0.7613581360647159,
 0.08216183035460833,0.6714658853728904,0.8664163389157197,0.9367246249389657,
 0.4624469259762177,0.34173932457396616,0.24383247251031537,
 0.10352589538678003,0.3292365471992055,0.7070531966998195,0.7746701233044049,
 0.48531791852937034,0.18650685772543873,0.47658257698981055,
 0.44873143069607346,0.1602519479895237,0.2519578212655115,
 0.047672350923641416,0.2850553625079335,0.7113987258869607,0.919056793524083,
 0.31611077007981214,0.25154774716023465,0.08796230695008045,
 0.18274906506754074,0.350122732576043,0.5133136556286146,0.42736720317568533,
 0.9019956707345878,0.32213156682282906,0.6688961338847119,0.732336721259296,
 0.5730533636912314,0.2799007081371335,0.8344872496622084,0.6640376113313358,
 4.278966780604243e-4,0.9729625982890799,0.07413821928543607,
 0.3296327730124964,0.1067241247503854,0.005479939481587204,
 0.7123581049537908,0.4241860550874522,0.29332039002544485,
 0.10919941480979523,0.9066440591993234,0.1786214139384522,
 0.09476065292056002,0.6771973300257454,0.3923464517008415,
 0.19293865821138156,0.3934775886788763,0.23649017126390903,
 0.4436819083833218,0.21512580122062597,0.3839104859178296,
 0.38925724093975245,0.8124956008722816,0.29895531562655353,0.892822832470266,
 0.676590534037388,0.04814054396719891,0.2496858707797508,0.4791874684366719,
 0.9907677022650548,0.5113420129763606,0.05708490502678243,0.9688076601378429,
 0.9897352791071026,0.9802327693768172,0.11820614042543198,0.416617033174564,
 0.11675138882631819,0.751644905195214,0.28856149055403224,0.8117989227380699,
 0.9097476055315851,0.8349381824753064,0.7536150176777354,0.6555074120137729,
 0.032486199361531076,0.5403045270386475,0.9123660849646935,0.894068025193044,
 0.8806232899668439,0.9918483705778715,0.9794906859763384,0.08058290768795096,
 0.2908513263803736,0.932158242226028,0.6276986006259414,0.7786333239645329,
 0.01591066821966791,0.9642335424441909,0.5640903566094062,0.6783763189103411,
 0.2884377043333798,0.42685629313197593,0.891458442910072,0.21451036731106232,
 0.6646896038176385,0.04675023109396759,0.046107118009122816,
 0.42739935450104494,0.527439819311203,0.7865871257313187,0.7091064891440425,
 0.8245143937406972,0.8751984931085135,0.16580871208194203,0.2670022529692938,
 0.3799966758903085,0.5684347032476554,0.7433976755497996,0.8792549021640379,
 0.43862152256630205,0.8426385919606175,0.9035098705988731,0.648447876266618,
 0.5521790167918214,0.04694682467527844,0.6822477635509473,0.0382935109742808,
 0.6829008700230692,0.03397402148923978,0.05646427412077126,
 0.16393193873676581,0.3117267891465999,0.6320812463462874,0.6947811249638146,
 0.09775516478411705,0.49823811385229666,0.07378289050916242,
 0.6314674819968571,0.6570406862533827,0.023804166128310045,
 0.5109863198778762,0.29535896487594016,0.3146093968850605,0.4069064737437489,
 0.3622074923157733,0.9502814672877635,0.6388099322525443,0.7925187500308231,
 0.809612142634906,0.40517532946417056,0.4551154983143286,0.4208362600888136,
 0.48515387234228247,0.13575072257530874,0.15456168547427263,
 0.12541660341946914,0.4075164213849828,0.8172037375737057,0.8459012309042034,
 0.6414173834983646,0.1260705795319449,0.7505426783110057,0.33359091483733483,
 0.9367732574898489,0.41415501216858486,0.9659349955131703,0.538592086060283,
 0.050427505655401106,0.221358120553103,0.733120792315365,0.16813180027478714,
 0.7063880959909516,0.12217998910434191,0.5248933760384776,0.9408484932003722,
 0.477533720594418,0.7922074026745823,0.5374519167954256,0.24695206171173512,
 0.2780975228655491,0.9058225189665479,0.5117192247595211,0.9149203009850038,
 0.7335725290534997,0.35880324473011427,0.05106268140356829,
 0.4464753731857588,0.1605213166344377,0.6587671112939757,0.4422865441438296,
 0.8136733305375111,0.011937648868690531,0.8237609742172343,
 0.7946321377176537,0.8577198411347103,0.09831936817253217,
 0.39394142920866515,0.663697554961646,0.7659992886601668,0.2700394679967424,
 0.3823801085049108,0.45953339891898726,0.22436177315651307,
 0.8676035757820529,0.7490183032931151,0.7187245648725404,0.17158236031745733,
 0.15112328354311233,0.22566223559029008,0.44610858317465674,
 0.7750342228511756,0.9670913998665193,0.051337624821829575,
 0.47354358611507297,0.4712129061228624,0.6285572252118947,0.1879462184611459,
 0.9087696863339098,0.44474372978090415,0.8960648421940677,
 0.02321843879336316,0.8188963855783296,0.33996318316098395,
 0.5078795738919624,0.8929641199732639,0.7479455301308084,0.9137910693020523,
 0.37395198070968727,0.7080454032748797,0.17856645063179866,
 0.2049596815692889,0.3489263208795712,0.9913062919710481,0.4532368311269749,
 0.7978495707921892,0.7102485465402427,0.15706371393813434,0.9137657749857353,
 0.9139267942886533,0.45664963316790375,0.682718437668457,0.6209627352651054,
 0.9042828803709788,0.6594609026662266,0.2713971287340813,0.9147068651414616,
 0.9361119689301869,0.4592274523369144,0.18404925022246277,
 0.018414387323338555,0.08028438011639882,0.8553789147212579,
 0.5868850759788825,0.19414861985933007,0.518574479818662,0.5144612388858661,
 0.24778160667897908,0.4717226218110788,0.6443807571207862,0.5545573161485321,
 0.11057435652582415,0.7710534461029829,0.7012358185037735,0.5169241214219069,
 0.9131779531588562,0.46174132883886077,0.6529799147142014,0.6407877064866553,
 0.6318191574134004,0.22923419464847838,0.3447384777500091,0.7919565785938628,
 0.799212565082534,0.558060777205192,0.21239332677065903,0.4468568093302966,
 0.013526135716971144,0.22383536560210748,0.3054581990755617,
 0.39677870808824867,0.9754164052209594,0.7505132980991207,
 0.37172053916364833,0.8415096548560188,0.3464005058146661,0.9420828701996666,
 0.347963719439938,0.6512031562115989,0.35479054370543583,0.21845197153585505,
 0.08881204966906209,0.22897287923331433,0.6875396181614244,
 0.8978288671988336,0.3593215776788419,0.9164519984240297,0.31297888092405113,
 0.5493688135140891,0.605920696236137,0.6532783006413548,0.7482113657102932,
 0.93066419165943,0.6765950038831994,0.896168703009209,0.05825675748586845,
 0.2915039328914196,0.8973246717913432,0.5505347384174789,0.7863807510566313,
 0.7624906599692594,0.7956596936135455,0.24814257887750124,0.7623109772055997,
 0.5151918333503157,0.07996238748263063,0.5057065051620314,
 0.04677752936715429,0.2308234682168251,0.7656327856155,0.6259027620033525,
 0.3217450925973181,0.8952883250220711,0.8882122334317792,
 0.0034607872883691826,0.6086324699514861,0.9714010142257513,
 0.7737061411739975,0.21733105907907346,0.5768519710335662,
 0.21009319137477933,0.8341296246987426,0.7432284093949738,0.3283462510055749,
 0.7762926254234677,0.5585348785862646,0.08130676940505943,0.9852494395762286,
 0.9553134032255995,0.7903323946304235,0.2611499169830679,0.16275847126562892,
 0.5057397856643352,0.6974859702445493,0.6977731604809395,0.26347724268097594,
 0.036524888347482465,0.587128909473238,0.759140228200541,0.4378162472711786,
 0.9818476571453877,0.6219968850186667,0.02571861678184273,0.8484332027136406,
 0.7138562000244925,0.5797673383253461,0.029729648364511196,
 0.9450233891164068,0.08993099438962071,0.25736102708696385,
 0.4723818772196008,0.26472992359327585,0.2682660002991203,
 0.24881853630485196,0.69718215319203,0.31464805555493536,0.6698412309566617,
 0.8374218951906386,0.9449549236876194,0.7397007256961284,0.9993448204321562,
 0.7947025726008202,0.4613143713490494,0.3954260452116103,0.5211683361633626,
 0.9060974601081161,0.32256486265978324,0.26848773261951586,
 0.34347188933791206,0.8964183548631356,0.2125559935576793,
 0.28278908228825717,0.31749506885113743,0.7151341800782721,
 0.9415634390318091,0.4167795818670905,0.8930662294879244,0.652954948377721,
 0.7919585762970243,0.645769421700963,0.6726896095450279,0.29616269407953166,
 0.4438817415553964,0.6536527254444179,0.031463243723561574,
 0.06181269982005666,0.07387567929323569,0.980448393448763,0.9676160417700265,
 0.47047897357091184,0.8157060325170986,0.6595711268291554,0.9892229524073003,
 0.6371656540022373,0.4495128507451236,0.24327520347196419,
 0.38325454625516353,0.6557064814489695,0.20896262458643167,
 0.3463466634500161,0.7632174589688399,0.3189899933650555,0.25082224453040725,
 0.8841559657003007,0.024422378599814,0.40912074128907683,0.505191639867214,
 0.9042872126636803,0.12377277359535399,0.2937455246643349,0.5147955202880272,
 0.9109664917100845,0.06228614891925843,0.540509591473927,0.2052452090792103,
 0.5211523187606169,0.2694233952940672,0.8063720697568298,0.0687676200259758,
 0.3502889731629396,0.7250202161957481,0.8324705292827878,0.8457320732116529,
 0.8771566915351883,0.34469803761250595,0.1453059425750104,0.7248958463822104,
 0.8929183620356002,0.924043732535444,0.20579087673258467,0.285046056378485,
 0.6910788819461472,0.24177899065075803,0.4664254456235172,0.39916329654281,
 0.4900940632960117,0.015489106509800732,0.4663181956707845,
 0.5421741338244723,0.1830487809111392,0.10436272079827136,0.2943286001153169,
 0.5407570595793756,0.6014210569233628,0.4531150057715416,0.8873442482533276,
 0.6426091543816903,0.968899946060255,0.18168118865952676,0.10316982164669086,
 0.11107134018376863,0.2281223681641693,0.39408251934303307,
 0.26105589859123124,0.23774016126571096,0.8087737241198921,
 0.5953982746771856,0.5499958878355931,0.3182668638326822,0.5767606609518985,
 0.7275657379793561,0.009861360735917812,0.6316713588108582,
 0.05724556141656123,0.9399154334919119,0.05728570423748913,
 0.26465590704129327,0.3123411143003726,0.8766655037683501,0.2906310204030964,
 0.2520502014320579,0.907467946723849,0.4204826398353907,0.9395327896256558,
 0.5399454615936103,0.847594581968053,0.3063537595125613,0.4825302278888136,
 0.0702372823065005,0.49594370169282,0.1519790937451283,0.7890944101959606,
 0.5085524729329225,0.797511477554447,0.3960243681426703,0.005526098093300402,
 0.48483239633958186,0.613071192118098,0.2745087330671233,0.111911907820479,
 0.4425386674577837,0.9496164325013648,0.12411036575969991,
 0.46232707000242423,0.32697445334157305,0.07156377309969864,
 0.9701279960495598,0.3901245582148434,0.36143230026616846,
 0.23784055593116238,0.6804318858298032,0.30452321485790856,
 0.30965849149535296,0.2937330878607225,0.3043041205489424,
 0.052416416135726784,0.28158011149719364,0.8085679342992292,
 0.38558644932244723,0.4148745893883288,0.406169924580673,0.40900210426991,
 0.9185521976467269,0.3619651909247369,0.4985190040433468,0.5583864347373121,
 0.43748510953724395,0.8261767053414886,0.14786706175234143,
 0.7033309658915534,0.8745594602783279,0.07591883423442791,0.7479300593500886,
 0.8581527060662458,0.03563746565638959,0.2723249784860464,
 0.27558934272312585,0.32249022607437405,0.12938149978364455,
 0.5144174185232329,0.4341313474692684,0.59233335202287,0.147726264687296,
 0.826170384573673,0.3121445212844767,0.6832620763683457,0.5344115426243226,
 0.5870441915305303,0.4860620177202162,0.48297917065193396,
 0.42205241784532177,0.36847893120270603,0.9657695523901119,
 0.6205806716248838,0.8815991130396028,0.3757651815694639,0.6366187408917084,
 0.302405983481397,0.4665753527625923,0.6714347816834605,0.9333050320314696,
 0.8512678616009213,0.1418975486072862,0.906953056767222,0.6424394243191082,
 0.9783529339833196,0.6566944551235191,0.8987351742072431,0.30394287257085595,
 0.05823809528497792,0.4705418512629582,0.21976867692690705,
 0.6706311794736137,0.9532771537617223,0.9171741551050426,0.6484247135008778,
 0.002828485401344505,0.2918563000359261,0.6300596717577286,
 0.6292274720890397,0.2772232653575798,0.26459772447338625,
 0.11917570480802053,0.050835947042154994,0.6244224177251085,
 0.9140358258390268,0.8439302160060095,0.41029940195633907,
 0.025378571374442438,0.7505242359063654,0.14038480340517123,
 0.7875232403541954,0.5141028249237324,0.22826751145510127,0.5224754316718443,
 0.9408704011807432,0.1452439801240787,0.08894536617263615,
 0.05880877650575078,0.1512420629179485,0.0028289222186923535,
 0.41661533076611357,0.11392198770973538,0.6948068260189225,
 0.8083901950641039,0.9417740136270907,0.4525247500135293,0.39462310604057604,
 0.18008377242997242,0.14102126235523482,0.7178375959485902,
 0.0209259512853317,0.42121696297177924,0.6547899743127902,0.5561564372758037,
 0.8459256808300821,0.47761821294449713,0.6894961616182225,0.5272882743894207,
 0.9733132966833957,0.43162993782128356,0.6722278186052711,0.9854622192254766,
 0.5957531037397771,0.3534717917375455,0.8877591467202539,0.6711031010870614,
 0.6131750849537478,0.5331561106895367,0.03305418189580578,0.9676526174521749,
 0.1514539257376739,0.07861919741658585,0.6661850774390461,0.7042955737596213,
 0.396372137848662,0.6265287962582481,0.9939031815173487,0.505836215997426,
 0.4121099969178549,0.26910318920106513,0.0705674529410133,0.404961154775887,
 0.5898875360238404,0.26082757727465133,0.9400819827229914,0.8079453290482219,
 0.4957543668481693,0.6453976718890284,0.618255116729315,0.7254856711344385,
 0.6571449376959015,0.8391095164954924,0.9563530814689812,0.4162752694899148,
 0.27362095172694767,0.38356448421427436,0.5886462750915783,
 0.38766091515340273,0.5442495562148811,0.8068277524231712,
 0.10300269528778705,0.6888331523212182,0.6901845962016533,
 0.12754225241901196,0.4424675686082803,0.30380720819320095,
 0.44524353075706913,0.77288100683314,0.8156813913156361,0.9511629825044775,
 0.4747230877029044,0.023707740920432596,0.847351489850929,
 0.38094549852961845,0.17938402531344722,0.38620849326712714,
 0.1956148967923942,0.850785436345151,0.9542909817128649,0.8219519742176418,
 0.19391113222946132,0.9903789028322377,0.22358532188080849,
 0.5521092158343743,0.5997925623298122,0.32122334917622175,0.5116342367639031,
 0.6747874114136155,0.8716041305060084,0.4021247473773215,0.8545144617945412,
 0.6484287491112883,0.3660510269142738,0.605963900572511,0.8570993974256241,
 0.6718283523214647,0.4279503058452412,0.7782151204448889,
 0.046741478562439154,0.4771226094937753,0.7292902903070738,
 0.3825141661642919,0.23867176305665894,0.5600322563804094,
 0.20816157210527153,0.8416970948358737,0.46243018035363725,
 0.17614086305185506,0.16030182737124732,0.11916274068040678,0.92158348307909,
 0.5523344738934697,0.034057029118042315,0.1131051619724528,
 0.17742625900227404,0.34283126531494224,0.43310836547505327,
 0.1348969222301888,0.09833735735807336,0.44529930084692726,
 0.8128776195826735,0.1769659598423371,0.2906103373692762,
 0.011640634415141293,0.8030919541926695,0.2766893973424598,
 0.43013724798247643,0.2764883997883397,0.012715084952323985,
 0.6292264068084137,0.027412886865707642,0.5498205723783096,
 0.7931060102754036,0.7734109188194506,0.4415599172570458,0.31119614029868214,
 0.679436764818698,0.025934912416515132,0.41698040078370213,
 0.46059233338003036,0.8459323247252013,0.2645066488074118,0.419797199885395,
 0.05532340329619123,0.18668149352413388,0.48566902891242847,
 0.49007412875608547,0.21620790687269276,0.7477186179321594,0.758641316688791,
 0.9460526979226602,0.4236775238560674,0.7895015564635118,0.37772573881759275,
 0.4447709297689002,0.8924408461761767,0.9548861201392821,0.18954816128127938,
 0.6039602029108347,0.8603162640787312,0.34570849468034315,0.6257865235241781,
 0.3524328336585396,0.2745938247942651,0.6807360660054764,0.7044892630369912,
 0.6923821277354151,0.7209569208579623,0.5852398085195702,0.858361758288563,
 0.11742418615160288,0.46416207823379907,0.5467802850115607,
 0.7303743599993278,0.1279304514601095,0.02596621763427398,0.7065487908834786,
 0.1948189399608785,0.9334205986256837,0.23863341359169243,0.0856721848291182,
 0.2840436295839368,0.9439773849412013,0.05460735989672638,0.7828880272173147,
 0.09244720896431613,0.7336147935374875,0.8325571421194915,
 0.17683970819754524,0.7800488495039504,0.4933870108962706,0.634507779226815,
 0.9747087219297792,0.611770562901763,0.7183999324926591,0.40557943988099476,
 0.3498194341981158,0.49589536548991586,0.7494779401937992,0.6702936351785036,
 0.9361430775524788,0.823164948929778,0.6049108377575019,0.33214459835269683,
 0.4368266068258062,0.3004567394131785,0.20641602127338798,0.2351848463388022,
 0.8185668760363709,0.8036388684197291,0.41347542478428956,0.566620889900655,
 0.3238805041350089,0.46257361367390337,0.6988582862687125,
 0.43822437593261065,0.6292876665251065,0.6597481260218819,0.7144828736139808,
 0.8809417389749126,0.9275612106199267,0.7165256273396551,0.8747612734001045,
 0.7492080384470332,0.23487545631103512,0.14513632892898654,
 0.5459423892512596,0.024426631593693626,0.12751221039734595,
 0.872742714557488,0.43385893198897085,0.9580474651972689,0.7231704431477716,
 0.0878504057398517,0.46664656183267517,0.42626896256507885,
 0.7373919069571304,0.9679083498836581,0.12523144043414525,0.7767899790846201,
 0.39768515365760304,0.6012853680124814,0.2904540673419862,
 0.13055503919586486,0.25906571930637545,0.35381963048829757,
 0.5874567021892502,0.4082667233770092,0.9250777999612818,0.952155838515331,
 0.22783640702864616,0.2857572111550126,0.4519475268657226,
 0.32103341558680976,0.7863218606656156,0.5099885161503374,
 0.25887967115776933,0.8875118543009864,0.026008271409839412,
 0.38084175665278863,0.5460459383519269,0.9600193993941165,0.5202958884819749,
 0.29781498000129425,0.9434137470459965,0.325330693749589,0.06736805397231926,
 0.9737889763976594,0.3120604292038953,0.30190357232549814,0.6648379431291649,
 0.8743097426421865,0.29270454666561607,0.15179295387404856,
 0.5235930053442145,0.7687444438903793,0.9792213021176314,0.6285046730524039,
 0.8129630526099495,0.7412624048660166,0.9682924533574536,0.29883939181570096,
 0.3138889497614459,0.40189017878431177,0.6951862659978523,
 0.016157462940027845,0.46166578419364934,0.026935800022611422,
 0.4146270476507383,0.5811189573401808,0.31613645129990786,0.29624430345482,
 0.2874987614088348,8.909575930206026e-4,0.7162454116868864,
 0.49231279708609244,0.678458934982218,0.2983824810844956,0.7227056352052006,
 0.712232467556898,0.21290110183363276,0.2934936795051626,0.9779769343024975,
 0.4934595721528886,0.4739828950897702,0.03991805353727618,
 0.17287022388361306,0.8102297146458071,0.9319713826706872,0.4680746826946861,
 0.7663970057282161,0.31847857029315174,0.5737176006200929,
 0.15442342880233628,0.6439245948983175,0.5912827469144111,0.9099805939173176,
 0.44443006050727596,0.07569040248707426,0.03840287432160894,
 0.17003308057342625,0.8794729238159642,0.8172163501911064,0.8658677597032707,
 0.49827146688976776,0.9131670358625323,0.9864595948718461,0.6891267698082437,
 0.8040305806170716,0.5217162230430118,0.2089768705188726,
 0.029182685937467978,0.17613740545725132,0.009725168050576993,
 0.7352074807172867,0.5115375197616017,0.7013583358931322,0.5373156860540075,
 0.45463540068474095,0.8460812096493266,0.5522722838802664,0.5555163592529774,
 0.5471221950132843,0.8663567353101479,0.16644711498844655,0.463795736916638,
 0.9384526000900631,0.32985398019650924,0.5985757574922376,
 0.31863443521995116,0.0716184145441584,0.6761858301369021,0.6386289951779097,
 0.3771239483323058,0.16548912534848026,0.6915847904923235,
 0.26218013594939016,0.14108400566248547,0.8309987864277985,
 0.48617706969561414,0.45235427437085907,0.8599969731973703,
 0.8852244205072857,0.47220012643346787,0.3598464794143629,0.6612339503748516,
 0.9449581592710031,0.5999868605164979,0.18099535223750085,
 0.26609398494380054,0.20876933535123732,0.7279837695983856,
 0.28046231525379306,0.02935146712789838,0.6173826368564246,0.558824602652358,
 0.18613478350484425,0.3658479524645917,0.02765095514353011,
 0.8315991788198419,0.9953903929942686,0.9782450284684673,0.9569996556716852,
 0.5059919313583296,0.5512469320370411,0.3403118896395594,0.26578705933158153,
 0.659907379903208,0.3278193666095369,0.176465286895511,0.708307925953447,
 0.7170311443068486,0.6358815234052326,0.3135254022267523,
 0.018245708370834834,0.7689777524912346,0.6404836914791369,
 0.5072555910190826,0.06775523292018648,0.4998129246562726,
 0.16810289456282135,0.07552731706057325,0.6214308695813577,
 0.16461428501851638,0.01733010972786475,0.17716810429748497,
 0.19529971724466177,0.2958159381484713,0.2637012196435291,0.3470854512223064,
 0.4881033498306623,0.23880758405795766,0.1989277118501369,0.3390251234342342,
 0.025564997693729552,0.3297217235698099,0.8421227955726354,
 0.7023301694865872,0.4836429853012121,0.5098406083714372,0.7166967364445859,
 0.05926726313790098,0.665427786746365,0.1767909838458528,0.7020366409845483,
 0.5717132380293167,0.9074345018641781,0.7469629357961189,0.08813641030984021,
 0.3902981751711023,0.6393128695793637,0.502708045666722,0.910939792342448,
 0.9592586006942552,0.8003230932628909,0.8126752128469794,0.588485767659823,
 0.8097754955603553,0.3686217283283213,0.34173084930773556,
 0.43128016008375303,0.9893296761978191,0.008850772467922208,
 0.3560218667404518,0.5235498812183352,0.4921088053465603,0.6955618882760131,
 0.15483804804141732,0.3052409677968768,0.7066113823037868,0.8814633292023979,
 0.6029019675124144,0.6679556696583926,0.7397923169106988,0.6793544919586231,
 0.5346239676104472,0.6499985853117768,0.6672633713657572,0.4489272759046944,
 0.3986753730382433,0.7655178861724565,0.3127544589943603,0.43006835749855565,
 0.14555633035257687,0.8004573610099572,0.9348883720747653,
 0.13631871751332103,0.2843030149811192,0.46414470596724744,
 0.5027941839340692,0.8909954211799356,0.7594012731940679,0.27890513998459876,
 0.21492176902597127,0.006497970529870445,0.33447546807962225,
 0.950831311361402,0.33236755441275523,0.047537617114659314,
 0.8044371988990614,0.31176332410564256,0.7159291449812452,
 0.49007852734500545,0.5577837747533707,0.17599858404837188,
 0.9174378318765235,0.6130945808798742,0.9706293417048979,0.09589056147057717,
 0.15452561000256837,0.9087257227551381,0.8924549380100187,0.7983078600005565,
 0.3520725164096542,0.003774390818400919,0.0437421039980056,
 0.8917863010553083,0.9462782121562371,0.41988211059774816,
 0.012571814628147271,0.3267178535990932,0.709951371052513,0.6930978812545939,
 0.4631195760698855,0.20930007080427837,0.35692041415172626,
 0.8060785430101207,0.27636113529136574,0.5732280100291975,
 0.17281672560898942,0.18046977837282396,0.47684657540781006,
 0.4066187849102373,0.7513276151629152,0.34495159981149615,
 0.31112329251629756,0.4405933595109961,0.5173555979151033,0.9141839597947262,
 0.3501534474307578,0.7920286184008744,0.8742664427481519,0.9430433749541728,
 0.1940255977135601,0.4032271364179737,0.575986849687287,0.06691079313944426,
 0.5831981193371585,0.10933025689097142,0.13077198049911576,
 0.7154466469347681,0.8080090453898772,0.24166568069167893,
 0.30562828482605786,0.2093050880100834,0.5503187400381561,0.8102133891919897,
 0.6176742455631089,0.9108135889834551,0.2705094454041146,0.2127312417441064,
 0.008813204171541189,0.8843918677747578,0.5489351452862403,
 0.5365615437305385,0.06473589074763764,0.2303686941944606,0.9244775357590798,
 0.699618275778235,0.35935218797492874,0.9089994973645715,0.03037121176551283,
 0.8837378216669168,0.8708167489008631,0.16792593043400983,0.1006968253012338,
 0.344646057155265,0.7537685025549845,0.919610120701447,0.707014820943707,
 0.3636909072456248,0.33550842994301555,0.21940179997866094,
 0.8308875287554944,0.3164269187506753,0.042835881644396334,
 0.08609720932236697,0.6893344376862249,0.8848931548091273,
 0.42460680778921345,0.6986223004620743,0.044400606795266695,
 0.5169000154566927,0.24153162948464213,0.012010285695830403,
 0.5493628150928664,0.3252496510072276,0.143535385127475,0.3295536040715318,
 0.5952817034409942,0.19147822801283443,0.1201139619295396,
 0.14943591298370285,0.5550218618496698,0.8672615259876194,0.7534293012373608,
 0.13560237272635633,0.35260341012122565,0.9504653925633335,
 0.011521714134846373,0.10190052087326318,0.5090147883084104,
 0.3672936172051121,0.9720829377836973,0.5895829635899538,0.8302311660039937,
 0.8842514872925613,0.7707570835510716,0.19568721574780035,
 0.06781626799292706,0.7453560499165668,0.3331370581795736,0.8683020922729104,
 0.19504835261018683,7.838826510158103e-4,0.6783860314764821,
 0.8969189346101247,0.8021354569250652,0.28378846157375115,0.7654821322496456,
 0.6981911953158202,0.19648823404712557,0.6106297719419791,
 0.39157510655756833,0.7108316571545517,0.8842437807277778,
 0.20550166631068956,0.5517054661373882,0.4613251273157143,0.437599804360821,
 0.4854070545677145,0.035009576731817704,0.7720747970642875,
 0.3254013439035399,0.41885729230954205,0.37226879414991243,
 0.14248489166789624,0.4424561319650707,0.9381455251953192,0.7285623277951302,
 0.24288503009686646,0.8161813659501577,0.6121523342722405,0.552141620595316,
 0.6489032092621816,0.8047388081310436,0.8437286413977325,0.4888219417563663,
 0.9535943295905818,0.8747705210980883,0.7695743060705895,0.35797472284414966,
 0.7768929694511774,0.8667031714243031,0.34200314321718506,0.292505450140884,
 0.02707731155568094,0.5669826299963271,0.7068137050037262,
 0.44452657450933064,0.7822431623720142,0.6465216503075514,0.5160737437946148,
 0.191685310103198,0.840356053642581,0.7866730623890715,0.40724932451921825,
 0.15723887590557006,0.22997774423182327,0.6471222340734379,
 0.3835987690730329,0.5226321015704876,0.603204227943575,0.19739728309714732,
 0.6849948623246878,0.4124095768002396,0.19770388266822736,0.7817716320769351,
 0.07182693434367726,0.4714662258403004,0.689500509316625,0.9513772963662276,
 0.6467608697260766,0.021128014960343777,0.9623357282285171,
 0.012446436600610178,0.04644712707134657,0.9073383127621432,
 0.46549079906185475,0.7583150060403238,0.5672725486848467,0.6305588974794999,
 0.295290277025531,0.1930254788465331,0.8116609148187445,0.03005450293408285,
 0.6676094837643137,0.061084127836185,0.6560969574453266,0.6862754172746481,
 0.7095783667209122,0.5668614481872882,0.6427579972782868,0.1542104976943367,
 0.9734841833704045,0.14637094076538193,0.8712367789250832,0.5371004414796948,
 0.1718281354455663,0.9785099814380027,0.8644787464321217,0.6136236612894398,
 0.4345826651675334,0.3189371100960835,0.9302840838759945,0.9915665736207651,
 0.4316770816983577,0.6907047398670746,0.7002208994260654,0.8709293177995407,
 0.3613053166811492,0.6979628114390009,0.2606393993236983,0.49315569864240705,
 0.48853917852981954,0.8970328698033598,0.55931080009399,0.38293606890614396,
 0.17773941278367678,0.9418873312436863,0.8380647650029783,
 0.28607102423643005,0.14272841184478602,0.6385528888618637,
 0.4452136431397168,0.525338812426254,0.6311907560722487,0.780319369749694,
 0.5210440928484251,0.70157300327669,0.6926223849113213,0.6719488270511531,
 0.01931836782655505,0.8603112625349433,0.9418144783297256,0.5801652688249682,
 0.6447205541547504,0.7517926610084644,0.4185963010479655,0.36172630650417636,
 0.24954678929251095,0.23722321172723582,0.6013495433656272,0.902958401371599,
 0.18636090776567238,0.7768162846534551,0.8447887511975293,0.9230288361722262,
 0.580817329955378,0.31055231991772736,0.31160868631536665,0.9163105827475223,
 0.073130487788589,0.14292594686466986,0.5690437354221656,0.5019079323101643,
 0.3313807679652231,0.5759992146272739,0.4680050506021063,0.9931951690555167,
 0.12383111435071881,0.2784595986175975,0.7995338876240523,
 0.19989703829537486,0.9366873393056929,0.6269824407096833,
 0.15442013168815338,0.13634622779604388,0.4402837958301191,
 0.04489479760600401,0.10316392849921163,0.4175981642583826,
 0.17116633619644828,0.6483297385096546,0.8558299991687975,0.6148891964492033,
 0.12822299865780706,0.3462156635758966,0.2530118885974594,0.2745871875284639,
 0.6583836896911216,0.43784900421894024,0.4896111646107517,0.6607719098974814,
 0.679570458048133,0.47891886380443793,0.35703738260352647,0.6999283322985991,
 0.6669903408257098,0.19271842899535252,0.6954817401367662,
 0.17869619426381989,0.6855560434992835,0.421364130339856,0.8594212823684222,
 0.5639626791592415,0.9577229649154364,0.3325083897290225,0.7282200668282431,
 0.6929730578803229,0.07524930138473174,0.08443681046994977,
 0.7869294562715795,0.2505503663043913,0.0011388688151531268,
 0.2690424961769372,0.07174055402477875,0.20193218263853052,
 0.9934845762692693,0.943372662018164,0.28035276878640036,0.6966598696675906,
 0.19427467364778217,0.9323660963969002,0.7352459703083444,0.6358344739955158,
 0.3164500140166342,0.5304718496296843,0.5337116478617957,0.36735454917432364,
 0.9397311324963414,0.022047911471974047,0.9218755289940318,
 0.9722855987398703,0.2977701478743471,0.02776447644547908,
 0.032209953425886395,0.6027327989998086,0.6394269759090448,
 0.6720560356456162,0.7466368122696933,0.07055466617610939,0.5234664388404762,
 0.14602422853879315,0.317118887369807,0.8845596190512538,0.9371776648070222,
 0.49300941511418206,0.5121898944825232,0.21235995321811663,
 0.7027540922177095,0.9372803989571988,0.03744202551059461,0.9800045887424881,
 0.5651539532193497,0.8290798958852529,0.5269483372000433,0.23855677465370373,
 0.2818562428969501,0.6456569673600333,0.005931903396276028,
 0.5848229713841672,0.9442667731656945,0.3802543964321311,0.4443449555991845,
 0.7667431904557833,0.18756156643542787,0.0981498559604631,0.0939851252819508,
 0.13878885593362478,0.24691131065703564,0.5906235125641277,
 0.16468516559924995,0.8185643128042315,0.9376553726764785,0.5131178562193311,
 0.3625863426268049,0.13486567142455486,0.6519930191367345,0.8477089257856306,
 0.6072374170493994,0.30724774079458117,0.8414909528132944,0.169721699964827,
 0.711478162612536,0.3350808758949846,0.03119464085702539,0.783687538067455,
 0.8576098353156857,0.034026541716244973,0.552653905964763,0.7751801685923172,
 0.21910569736400975,0.3529655562594095,0.17332164358392088,
 0.02284840690112011,0.700032308209661,0.40405832030889477,0.6106978390102926,
 0.006614177403874599,0.11859591228107902,0.4944028533343132,
 0.016993567875283278,0.2984166813211251,0.3917894201929868,
 0.4037564541924753,0.6085166923950279,0.058500018520250996,
 0.10239293225180135,0.2904720704182576,0.39332123052383394,
 0.7375285571225715,0.24977564318806078,0.1696194662023235,0.3316447262919763,
 0.9701625248670502,0.4801212896443543,0.776170969513767,0.39346253378342977,
 0.5252771698043561,0.22186003365215268,0.02902996237911981,
 0.6584815348855704,0.8937711942940223,0.08098260339928565,0.2117435026655322,
 0.8795027770379326,0.4380380039446621,0.34069715798988975,
 0.06996083194286318,0.5711410042226832,0.3439731658034777,0.8027022528962802,
 0.06821515879648332,0.2888751877043938,0.415730835575422,0.718092565346507,
 0.6112532992330229,0.8072496207701473,0.141558999879962,0.20801800564353945,
 0.6707185518370153,0.6737555024192309,0.5534490007046728,0.18132282604112326,
 0.32542163274260205,0.16626357259890856,0.15071687056798366,
 0.4291797487517671,0.6460361838454481,0.045481349035160545,
 0.6030555416019845,0.44741231061867304,0.557780432424208,0.3356408117840557,
 0.8331103809002135,0.5809410416797158,0.4543013999623394,0.7398353863530543,
 0.7473401772150255,0.2910534966170152,0.8480250106860675,0.7310917234261484,
 0.4456126731736366,0.2887800778210776,0.5998224687595893,0.2171797477904358,
 0.5826373173330255,0.08124194934745732,0.27300020746292675,
 0.3529993221995227,0.4291207291208985,0.28409798121396834,0.5971541048957908,
 0.08379869502470016,0.9760868459312239,0.48259056208729467,
 0.9446243563438954,0.23834455345860062,0.6820924707300181,0.5239147506638883,
 0.013011934217010745,0.19687702724204625,0.7068050144581233,
 0.44486716613262756,0.7804317743826289,0.39130374588993644,
 0.6707534954653274,0.18203064263990043,0.5934521289575491,0.8348568924616316,
 0.5420427139290939,0.0012422377225532744,0.2389101460107781,
 0.7846041382923415,0.6564827156729853,0.8183952303871411,0.11966247767158977,
 0.9916277540976594,0.29114509341011163,0.8850913675065069,0.9353605042919474,
 0.4087220269863514,0.950659015818851,0.872364632694555,0.025277829579403055,
 0.906483907214139,0.3032088037607631,0.4601191665190777,0.052320312445825023,
 0.0437771638444443,0.28619148200100886,0.5208939156523832,0.4861525962565477,
 0.8579169779481584,0.060175932522570585,0.17345966879455033,
 0.35105011245209006,0.6971133237053775,0.10844383599952312,
 0.09549230216400506,0.34902863531555417,0.4938357037420612,
 0.38405862901453125,0.6178216266154904,0.2583610183989077,
 0.20055591174044718,0.6347910790090895,0.6568924625432137,0.8134477736390396,
 0.7509974420456598,0.23287176484211014,0.9815338284847239,0.3497656480840303,
 0.544383913357086,0.3422201899486126,0.20737520003376186,0.28785243777584324,
 0.12622705075180085,0.851686915363157,0.17098397781377972,0.7221746765201728,
 0.4315261374350642,0.7307524467780766,0.29368986211508863,
 0.43331106457815216,0.6490051298850488,0.6962460701857114,
 0.28571614377092625,0.09817291129482586,0.7595673194144399,
 0.7312719564307278,0.9170081427685364,0.2998873371245582,0.615372471428476,
 0.27442579951503143,0.8501903714245944,0.8982424529958712,
 0.27512889514954364,0.08300488479557355,0.42199787702992975,
 0.9479511125292703,0.30122024935925296,0.3927281002236295,0.9621004937458918,
 0.26892891093330673,0.8145944231494955,0.6501196886961593,0.6261046559247324,
 0.4414635223766551,0.23853679521663806,0.8865801722734769,0.6796094236409022,
 0.17241470542423187,0.19065442268014154,0.868287451296472,0.8308770879503942,
 0.20955424085346963,0.537421840843591,0.9458364370567952,0.4479994699320947,
 0.4978331311364924,0.7007396588412191,0.851816097914562,0.6265519379825208,
 0.6572591091981831,0.7933877006931114,0.5793740339352128,0.6801130182940036,
 0.5545710965685348,0.45874931744630953,0.8522581065327156,
 0.18071297220503035,0.16603220616036818,0.21463299408105263,
 0.126011558274163,0.2285390672833516,0.006243847186806595,0.5768586599208846,
 0.1879317699438856,0.22640461017878266,0.2177665409355506,0.8360120324112269,
 0.8076960407416949,0.4693217416568566,0.11014013543695289,0.9394861635768292,
 0.7014542692256733,0.8384456028132174,0.6012921465777828,0.254100485609158,
 0.9424894724537087,0.6024249921475338,0.8715658667996375,0.5182736037407463,
 0.3932848223867381,0.7956025359738145,0.6117766386587694,0.22995111032348425,
 0.46006995958597574,0.8621766397029318,0.789502798855577,0.8107140602243005,
 0.915378566530136,0.6160869498314245,0.5077928174052339,0.6176478665543742,
 0.14054792285022055,0.06928388457913448,0.20760318330097438,
 0.8016348479142876,0.34774162227451766,0.40414904581174715,
 0.2745441534688309,0.4224617747681273,0.040119256324848585,
 0.2789500987401051,0.08570768744941648,0.8103288865162395,
 0.20748934123902818,0.46821037914753927,0.2002619143146247,
 0.2894962271604772,0.7406396931829459,0.09035598867146633,
 0.19752209187663472,0.7196008399919989,0.8570648049559106,0.7034548986516478,
 0.843472473530422,0.7362650256616118,0.011747038685899103,
 0.29905604419520115,0.3711682632253618,0.7720892838317208,0.6103336501464202,
 0.03334653368598561,0.2431023821646615,0.2544061708564702,0.9618532575140487,
 0.4697425198551195,0.35495974192377266,0.6489054796404851,
 0.35150499065302254,0.39477553533267185,0.7081899591382776,
 0.9913428026876755,0.37122132607042946,0.6465360657320859,0.4475185291978382,
 0.96412323925893,0.6723868798407193,0.8606081490264421,0.7041866567693759,
 0.2729680092796696,0.3583084726560991,0.3820026116777401,0.33307297824620086,
 0.9209802048581375,0.6574458354931886,0.7419252350763045,0.6162809284356121,
 0.21795202583408257,0.05723890599837689,0.9850782036000877,
 0.5445909971879449,0.24071308252326373,0.37928744569260864,
 0.46363180284090344,0.6170497208850411,0.14433819090856304,
 0.3489518626162429,0.12683738003577283,0.3013734752849828,0.8651408879172039,
 0.3980050315201923,0.609417322014694,0.22422448406674234,0.6226845638572558,
 0.738885505972168,0.7739749833606848,0.5852723089639165,0.47515684780968126,
 0.9239789223927279,0.940026309096556,0.5717563091321496,0.2480540336898427,
 0.8546037706572718,0.7707009330714415,0.6378537470319262,0.9091640206745373,
 0.4414113324389277,0.6562675642531902,0.3063849422391116,0.5173417133779457,
 0.5354093548736699,0.9115232453196987,0.07880044405114683,0.7380481922964289,
 0.9045202768065418,0.5448595854827814,0.6959812280233103,0.43384452201695956,
 0.8556117048070077,0.4673884303838054,0.8159228370310218,
 0.013422295839265574,0.7652841754899451,0.5381389029830782,
 0.6340048840914945,0.9524598258832342,0.9612296700322304,
 0.053434363424920694,0.11556598407658603,0.7263205897774427,
 0.0831877871290051,0.33338992511072274,0.818175133464599,0.9267216679512421,
 0.6685565378800984,0.15177445894827868,0.4211843200203156,0.9337663319553844,
 0.5486945038868944,0.7714596455379307,0.4671765633880002,0.9040338988534797,
 0.3238138799089594,0.27529809968123864,0.7832030075068309,0.8436127837528417,
 0.43129541117726744,0.9788768017753314,0.6004028513519148,
 0.04828669961864518,0.7787995879052464,0.43467153339508235,
 0.7027126105720854,0.5290839589161948,0.3162241806565651,0.6315478542303608,
 0.9553777645288223,0.8737127145116004,0.05867810887163971,0.6070768104410499,
 0.26049535721612205,0.9298567060211176,0.5234729621951207,
 0.029678994595653818,0.11917433229915919,0.593796570043928,
 0.029358923435005235,0.5085543009420431,0.28309710825116585,
 0.04150905704858676,0.09407200506070457,0.961495336620104,
 0.36794390170640046,0.3622656024790153,0.07081955194244693,
 0.9288584801892944,0.0748331556838131,0.9418264337440049,0.8082442583256982,
 0.6089855613859372,0.5612549155832812,0.9519519453772535,0.6775713535434174,
 0.847175183216305,0.031153215395392286,0.8482625698734323,0.4551555030014588,
 0.5730563318099967,0.7450379442816862,0.45427425312503333,
 0.47241266988503783,0.9692963137985796,0.376510681552755,0.9904242943516948,
 0.5495998633623749,0.6866479640055165,0.06923663540351088,0.9914383981316277,
 0.41618693661023787,0.0983366709606941,0.6440608988253816,0.9483892670527936,
 0.3182502675029817,0.022281918247079813,0.5477552997666529,
 0.5216854012506323,0.9986863561189817,0.6584369801894374,0.13575646140755382,
 0.5221719611282585,0.019340783377557957,0.28535505274317263,
 0.5166678753021579,0.20444515815070985,0.46064335844619086,
 0.9772866225049024,0.6294764570829038,0.2832205413302178,0.763151018174605,
 0.9346878848024631,0.549595674608969,0.7308202715395822,0.15322813907141541,
 0.4010594874751928,0.24715485083870126,0.5496902765573924,0.8754059543249664,
 0.9018920328993293,0.5664700207391725,0.018781496694581845,
 0.4657363382577736,0.8686008724087746,0.0020770031342512985,
 0.8665979867232684,0.6630062069319858,0.22458035642580831,0.6060641284729793,
 0.885066975900425,0.4921177929385724,0.47759458575397984,0.9367566052701878,
 0.1240537725595906,0.3361566626753697,0.1403280461537284,0.18834720676844297,
 0.8337946680168178,0.877082586282488,0.6516198400023069,0.8180060964005397,
 0.7517512712919628,0.5877186011760908,0.03691969214188029,0.7904981381634718,
 0.029101903697511977,0.8334165228880381,0.4430666883983604,
 0.11069480661168196,0.2656729767695264,0.40979489190681473,
 0.37949325279524226,0.13009248221606262,0.8428318811640825,
 0.03430976634498584,0.5663470318251597,0.5278131138136605,0.6832971497521686,
 0.7826150540199163,0.1808014264983342,0.25323349867009304,0.766028908470721,
 0.5907365864982125,0.09684034299916189,0.4924776107383,0.08377621151225101,
 0.9875307078333441,0.8545209866350035,0.00421833333838173,0.3767290634399082,
 0.7291572533993455,0.19710465313241088,0.7439915583056446,0.6370602701216296,
 0.13747190763614547,0.5942993313564219,0.24279847918557618,
 0.9749198186143495,0.8972974034638193,0.15977432690144266,0.9760559843104323,
 0.5029807167413148,0.07896625434156623,0.8396896776508949,
 0.49612452753271075,0.11854912597624678,0.9560266551127585,
 0.6539211272496901,0.3597114347434536,0.8961680003944865,0.43303663070662235,
 0.9790883735584984,0.7380609501297509,0.9383042126701182,0.8443157644702072,
 0.08736757576401233,0.4407461636661695,0.19018325143618098,
 0.4959949047842055,0.6728041358786947,0.3001450360130259,0.09419738781632869,
 0.47291647826962036,0.9092536396202199,0.3076316569278835,0.7915644628971562,
 0.3070184944632641,0.9250938928081138,0.8681681312584092,0.30922431114708804,
 0.7227795387388509,0.47535412327567783,0.5867404384841461,0.7741971523027666,
 0.3148633835900526,0.9381585082718443,0.9510434953395968,0.8877465664307753,
 0.28145765061854067,0.011850779238995468,0.9796591904727876,
 0.11378426116546381,8.557388532302035e-4,0.5629662709902172,
 0.3611502856061948,0.8241436250640164,0.7432665000619473,0.3052221323286477,
 0.914622600635365,0.9101343042373153,0.7321516498941816,0.8610172851748523,
 0.7232464061941619,0.3936794405081123,0.3993543827596687,0.9180674420353183,
 0.03464107707680397,0.2959080571180158,0.5053484558789922,0.8999922271572235,
 0.33167176698596323,0.2310905146225487,0.05266845953255217,
 0.7087333370375034,0.4331045890724482,0.20910780436199272,
 0.36479210433897347,0.029571623977201256,0.687921410019966,
 0.9330137817449642,0.8644204277099113,0.9138413415143243,0.31716574660293284,
 0.2743999696867111,0.527535151504329,0.6542773572776979,0.6796712051206882,
 0.24939805056265563,0.028376035787297083,0.09529195940724311,
 0.9687699689748028,0.9188146697790756,0.14044515158252724,0.1886179264450275,
 0.4635698796494361,0.8247262566050944,0.6603544458632187,0.8212170395004782,
 0.4979701658094863,0.3404632652695325,0.9192829617008662,0.0535698272043037,
 0.9836231731439476,0.27159831313458205,0.5050527500204446,0.8105203763806659,
 0.003810185122539167,0.6718226525722519,0.5137292367158398,
 0.2929045841140957,0.8739545286025479,0.4337787165432089,0.6895071120259454,
 0.7822127728323409,0.12232543782174843,0.9000721876533258,
 0.06001843495543557,0.8557461040229147,0.5303142053087015,0.8038393313505818,
 0.4331955664642757,0.4968651338691845,0.4482875532315026,0.7134993699706121,
 0.6012655657380931,0.36504162505253557,0.47314073447476146,
 0.8393285014789087,0.6663602458207023,0.17844755827657433,
 0.18190597923188434,0.5402917424195064,0.3878618312115357,
 0.08519891673519775,0.7042392218075829,0.1285374598956468,0.3981161033942364,
 0.3991424200752616,0.3405603732358087,0.510562767179047,0.6273218082758983,
 0.10112808382034733,0.587116849952237,0.2065520506095435,0.7028167004121626,
 0.2588721873252502,0.04766561464400898,0.014575053378552805,
 0.554914888238697,0.9337079006869335,0.5768041552718441,0.8971712500972149,
 0.2781586374481745,0.9006648924641768,0.2528969655160147,0.8694229089789283,
 0.9688666631340093,0.9731111061029244,0.8783712924362739,0.48372127586699254,
 0.9037593784464448,0.8495792121697557,0.45654464222983826,0.56285213661498,
 0.4414857014243282,0.4135380538884652,0.05501193205355648,0.6577228829336138,
 0.08658323057275608,0.9694297369772366,0.02520441257271311,
 0.3690238669430931,0.6718451560413747,0.6589760616211698,0.9630307472643053,
 0.5796196719818539,0.4854479751655968,0.2921303054689599,0.1544732345795038,
 0.64275690823932,0.48276280966631613,0.1391744916729969,0.6843616328053124,
 0.44823510664833754,0.3238862937476563,0.9291703724974584,
 0.010478885346727584,0.2530265505534891,0.9558740395092804,
 0.7944589681626246,0.14070003141343257,0.30026252802407194,
 0.22595512027944686,0.03370098436710933,0.7023150227806858,
 0.5663144945574932,0.3458779089955907,0.20360008635150917,0.0773431588779061,
 0.6515669357169591,0.6363862764854824,0.012135815593932064,
 0.4622609016608583,0.28373568597113863,0.34234994391320306,
 0.6790915903024346,0.5864699359524005,0.6511205915066708,0.24011241681023965,
 0.1829553653522975,0.2389008747973691,0.2964382933946833,0.37116950348941213,
 0.04043450156850137,0.5419189839983073,0.4444756926477519,0.6030727750588035,
 0.08774917642473457,0.8676991512314322,0.29629244322737047,
 0.09839413900099792,0.8162283143679653,0.07450476074231649,0.846722554913496,
 0.7941900754010609,0.07246217806527888,0.7022971404878996,
 0.22728601841063512,0.7721122498121563,0.866758545841941,0.48476748074159104,
 0.1064769313664452,0.23540204515529906,0.3502212956937609,
 0.02936356826724662,0.91017926825856,0.517342333047243,0.4001924337476277,
 0.8192965920683299,0.12034389998507744,0.1494229195250033,
 0.40145290821182067,0.937058703630604,0.2677203036242337,0.47280743605041453,
 0.35762128012489447,0.6975737070597734,0.5291346538079778,
 0.06605496457525029,0.3437695506656666,0.9982884510898462,0.2518552001079485,
 0.8413974408884355,0.7317903898816529,0.6649034436781012,0.8463342707448982,
 0.8971178064071058,0.7708485770931899,0.23095143101069693,
 0.05809701720252802,0.5067029468657265,0.8704427688306758,0.3866344550849783,
 0.18756445678234335,0.9369313210161438,0.9491424212047983,0.6329789865767288,
 0.3806976631759216,0.700945267748974,0.6097993627733451,0.2803267189509272,
 0.4039923279416108,0.174610022359727,0.4683314836380653,0.7909131760560509,
 0.5441039358094948,0.11305079768068504,0.6542508375977677,0.1173171578669171,
 0.5798743569229308,0.5318784134070973,0.7750491593217148,0.6482316494073446,
 0.19328710122316084,0.9666129613945674,0.28136910861836495,
 0.30275401425588866,0.038632757681359875,0.0439995044543009,
 0.7545950071061975,0.10546432393803351,0.7722979731803725,
 0.27823815683598396,0.17697308558302116,0.6934098073466451,
 0.5282162705630398,0.94208741243545,0.262721907652837,0.09280388492345637,
 0.9059317077752674,0.7880823586631445,0.27571452727907186,0.8948490522767234,
 0.7319572223971018,0.6020198944998634,0.32265561414245425,0.8421145190397707,
 0.5214803917696124,0.9748908845146611,0.3229708461137941,0.241495689738897,
 0.4215250725875198,0.7640279473844696,0.9503185401096951,0.18825867663353335,
 0.06340767742245856,0.9419805115533562,0.4017937310490043,0.7484604714588835,
 0.721043753885223,0.2857485722208656,0.16262747287885426,0.6674459039920412,
 0.1255193460046432,0.30109136145333526,0.5717752333754792,0.7597385522399888,
 0.1839764710102504,0.570769972779573,0.014261566210530763,
 0.40104392747727347,0.34173632972598256,0.6067364122172029,
 0.9306112385741101,0.36933838376055184,0.8492663100031792,0.2577078305909146,
 0.15124491125342332,0.013559414921438084,0.55647257732032,
 0.07318513502233326,0.4582308107939451,0.25310251777747117,
 0.2730741952288004,0.8305314698959712,0.28736338698439956,
 0.34277676984788275,0.9922252155292315,0.3772994272771716,0.6372255536699319,
 0.8564633825365154,0.6945805844628963,0.48567564595094503,0.566568651523993,
 0.3169502023099162,0.0666721278750787,0.9571528581023603,0.7250651151157441,
 0.3609903731101983,0.33043036328717323,0.8115288422402873,0.9514693168896589,
 0.8509447134707222,0.06990767134217368,0.6323812415496123,0.2457134097474063,
 0.4005761773003861,0.8559341682138352,0.9319401170396728,0.09316932586275006,
 0.8550003354512461,0.5123125065520389,0.7295204787339516,0.15792887985028292,
 0.5599542336955735,0.8085992678594834,0.39865510557890504,
 0.22078182792906376,0.8798597798390462,0.26508630044151216,0.621136923871997,
 0.7242647179543028,0.5470899478360763,0.1911882662313622,0.7682540929068717,
 0.16770426592590604,0.7869607850619305,0.26565574516040324,
 0.3399606776265126,0.6145221973236709,0.736151000229338,0.3145088843505446,
 0.8006444300295272,0.46984353260289724,0.04056011271843918,
 0.7704113849088249,0.3669550639039867,0.7777785973062419,0.9666846765086251,
 0.5799676579052199,0.3660543421382726,0.8694706729350583,0.39419633284408473,
 0.29218811622194063,0.8365356283362676,0.46122821951124193,
 0.46647851477974966,0.25541649587353954,0.9585032137702773,
 0.7200548554792755,0.933159543900368,0.9657694024676717,0.17631223541300012,
 0.8309015641875038,0.8125621596517751,0.29293294141908244,0.7327502540671854,
 0.0849122504837666,0.09892944088774913,0.35892135568426053,
 0.16153916970185134,0.03158240904522047,0.27457118272017444,
 0.8390496725504373,0.9657663402360575,0.44292286501287437,0.5006420029852364,
 0.031075079805791272,0.20259500214151238,0.5426440598778034,
 0.43826681691313196,0.7780048536890892,0.5034869736767166,0.3405841249771445,
 0.773373249731332,0.12205116610231914,0.8480197478802898,0.4722259245939615,
 0.00213401719709172,0.2332864899795397,0.5956269421964796,0.6192590143951481,
 0.5420670319911194,0.6059434610969682,0.15754593043357967,0.3294383900478459,
 0.8772714488745699,0.6047464766490397,0.05856485713607307,0.6868842933724305,
 0.48645529983135893,0.8068098332690306,0.966044865384161,0.9300536351350186,
 0.19360814020041572,0.3575734897908769,0.13391373692545028,
 0.47607180640704294,0.4977593085062798,0.5268636270841675,0.1762168907735464,
 0.47578592425310484,0.2538186968614293,0.8228867544695188,0.6898132105756551,
 0.40477102333105797,0.9927938388110936,0.29354579459985486,
 0.9002937684858494,0.22246199128593835,0.3827469449330021,0.7796170324799054,
 0.4328397876817851,0.96714438048408,0.5313492464568529,0.6537234174835871,
 0.012473288713792274,0.9408077915896385,0.4148677457296285,
 0.8079987012016946,0.1411843905307636,0.1716669909245131,0.21436484619172325,
 0.29043009985762136,0.4082032054696726,0.35183807215415863,
 0.9582415125184544,0.6351190492540735,0.39434356588198016,
 0.06789277647059888,0.3511366219853149,0.024957304076266063,
 0.12821195291184162,0.4467171434068682,0.9210724411895633,0.8945902261055605,
 0.8018236569933435,0.248121699618718,0.908990860205486,0.08705963618206425,
 0.36104179974761197,0.7241402693628857,0.129511277366833,0.17558979402322727,
 0.3892380023223696,0.7078559857476672,0.7131171707339747,0.38251151292112195,
 0.5958435895678015,0.8652442287666787,0.7937752935417028,0.3059206916527387,
 0.3017702493045119,0.16181373604659155,0.7589068417421483,0.352641084598063,
 0.236303472198647,0.7493209963595772,0.8191247568165113,0.9629233681380953,
 0.25897745029168684,0.23250649957973868,0.0698004430895367,
 0.24884706999468276,0.3857284657171828,0.18250189022609564,
 0.7844667945170486,0.6848791586777144,0.6905215438131137,0.12187748170101431,
 0.38869086329166536,0.5298508593840361,0.03253050601743668,
 0.9452757337674907,0.7416298600883424,0.41860968106462093,
 0.29051273978106673,0.9748561773671229,0.9204950101732697,0.760022662216227,
 0.29167089742060026,0.7718856184860504,0.2921830275138898,0.7733923514757925,
 0.2725054380430356,0.5032707047044213,0.2160631496005555,0.09913942565779688,
 0.05640798718825568,0.13779845132326685,0.3719318775260517,
 0.7574949891283191,0.45031350211677357,0.9999457169633,0.1896305804227454,
 0.3040318230334126,0.31054569585092795,0.5209671450460469,0.940183024465149,
 0.45658056229856037,0.6889469286349947,0.10184116334999938,
 0.5981920387591301,0.21097274456163184,0.37361443281510776,
 0.07780534197255418,0.6953493199413304,0.8943962195067753,0.6013989260009958,
 0.6647817183091529,0.935704151398933,0.8430013066962139,
 0.0017201474156371965,0.04435783690671202,0.15915590612836283,
 0.4032685485532159,0.4562735019350861,0.039400290663204274,
 0.9074894282089385,0.7944599279900901,0.63266288495484,0.8753428247876693,
 0.25157795768553093,0.34955597778733327,0.5874871629303211,
 0.6320254506076038,0.6590341574379928,0.7651616524318708,0.8954252231438953,
 0.41703761856179566,0.6920205390705565,0.6530128336279208,0.6374919379993831,
 0.8115267427134393,0.96122675950449,0.4050195693460774,0.8425916748261123,
 0.5069813133954967,0.6532398038177131,0.7107645046709367,0.827549108658274,
 0.31842759365389894,0.28720991667318385,0.9344740360017201,
 0.09655194405053147,0.5477962831571807,0.7337313226040484,0.6882756537953005,
 0.6156782876557516,0.403875771357166,0.5825800168438368,0.6426655967410523,
 0.2334496696843984,0.5228717412166357,0.434421921351242,0.1364219689980124,
 0.9385486461377345,0.1622195159391615,0.4764874602967625,0.7278975659665193,
 0.4527638269292543,0.9284748091175827,0.5446781579605902,0.1063595720369801,
 0.7279013562532128,0.9308694703064607,0.5358435356826405,0.04442206927979542,
 0.8662404440164043,0.6908041789132162,0.22080968541016632,0.5123312442778662,
 0.7869135124583378,0.9511064875344644,0.6414749826858596,0.6863188058938336,
 0.5178118880071928,0.9201782810708953,0.316714725019719,0.2762895128047832,
 0.5118771550058075,0.6138445975474147,0.9701870670354662,0.7046622264734186,
 0.2560063363406473,0.7106744650726647,0.5753599065671775,0.7046666933551229,
 0.2412715506143297,0.3234331770921781,0.15251507479361004,0.431771006880905,
 0.006801435081812968,0.919327545023065,0.3085993172324014,0.6759670872412208,
 0.7206571789918781,0.34603843292049885,0.4662828818578363,
 0.48073521355608007,0.09233179875482089,0.24135761178113602,
 0.40513302733765244,0.9427729026499446,0.5751731773353401,0.5325055007703174,
 0.6678122867557257,0.308844726529071,0.9500672264583213,0.9849576135614366,
 0.02062957580146918,0.7306981355132042,0.3190278859942449,
 0.14666817502585494,0.896883976962417,0.16661806724969508,0.3909465156678771,
 0.5721896718862376,0.7181364337067588,0.9420798631127898,0.5125584413623724,
 0.6301633155369408,0.9371743469342122,0.47185206545612013,
 0.22597552910121477,0.4888601520623257,0.8405843264430145,
 0.16760845043798767,0.8678565455456285,0.21248606325691577,
 0.5272079649179029,0.749568363028912,0.524935729268233,0.36508371914746796,
 0.8181027096604166,0.1549184195897111,0.8122330476544352,0.900394417244222,
 0.2177352615485032,0.29272584360333387,0.7712217835390106,0.1432774467363499,
 0.6625629741854676,0.20182065943278604,0.017370742158667074,
 0.8110665182546875,0.07120408292349711,0.4664307296739054,0.8532142662696958,
 0.3810105498284536,0.6498401562316076,0.5800798349727081,0.2513590697188731,
 0.37654879693423604,0.22070526154732617,0.9272466502291155,
 0.8340834857926778,0.5946485500048317,0.7461428543028771,0.8139224082732888,
 0.21030072908194164,0.9007433307474806,0.06749376287063003,
 0.6588506002171327,0.7728135020437283,0.9308879570013012,0.9978830086107771,
 0.5913593635159262,0.4134308846851069,0.8409490941388684,0.3909040569148815,
 0.7237891160115492,0.5616942463005287,0.16844730976865363,0.0612694233686949,
 0.5721834614686437,0.42604422236971407,0.0339119286277364,0.5119767313146628,
 0.7047409349443486,0.06683122007591424,0.4733707979661874,
 0.45817921592917443,0.4252369707990443,0.6825489841959544,
 0.12044218602226764,0.18762158685558272,0.6091968984354833,
 0.22478822444920032,0.4024923462161646,0.7242237050475862,0.4054255164041066,
 0.6568617536124787,0.08207314101910435,0.9546189313421096,
 0.18598669472267382,0.8230153620537513,0.9004430916948556,0.9601258643902351,
 0.9180946490239523,0.7659496066495958,0.5637518697174019,0.32953828397331264,
 0.12642223647958817,0.2337212343837538,0.044279779596061775,
 0.32657515209961063,0.43357285815801716,0.33612621336296045,
 0.0011892334897757317,0.8589533385955298,0.223565896255713,0.683284569861696,
 0.09657958174306103,0.8818985452681964,0.006324838190781112,
 0.8750666398465021,0.3132895539363252,0.3073936021674749,0.35874430713328176,
 0.356574947872899,0.8827521948396262,0.994508675413808,0.682693298293491,
 0.11715454886861465,0.5538819184305862,0.44072056466452114,
 0.7751821230685525,0.7409152303032902,0.5296529004562022,0.4311763326856498,
 0.627745334659083,0.8477956305934771,0.3110256505135611,0.19878584124603071,
 0.19798237115560924,0.07937693921146116,0.012165539272259385,
 0.5148001240210295,0.7038303689507488,0.5910679152710464,0.2331191293555036,
 0.4724919160619777,0.28135629915435756,0.20923780296743044,
 0.21176391376400827,0.07953618709359422,0.18720519124359192,
 0.5848954497078399,0.6322921122781217,0.6378992691867085,0.6572917401930747,
 0.6499286631388618,0.12218585390404346,0.6463316950394221,0.365204955531907,
 0.1541422509576582,0.8986723684362848,0.3995047091793982,0.6472580198482115,
 0.03311008554649553,0.008311814307331855,0.8779336973507286,
 0.4922046310249366,0.45535882842445863,0.46738552895674956,
 0.6856590987829819,0.6709506326330136,0.47577519665668455,0.7284882564725804,
 0.12433023913300978,0.26033274669470075,0.18692403804185753,
 0.9015271790778446,0.7626897288885038,0.6841461870151515,0.8332052777281733,
 0.46411595336388256,0.8509763656333974,0.7783613717170044,0.3208310161982775,
 0.42107756338267077,0.5211838935746733,0.23938621160746432,
 0.3346942678073994,0.17820936737121107,0.1703118990300625,0.8027807799662484,
 0.42712359790221255,0.07843227690568455,0.7349575386128775,
 0.26888027947018434,0.8691999699224384,0.6388781603989423,0.9157620442457503,
 0.2614478673363877,0.020634067035018355,0.9747604920585026,
 0.15015337990448752,0.7266389382118186,0.3536843029293423,0.6398927341412383,
 0.1413623557048046,0.9173699948125752,0.33114358239580877,0.9266864778441666,
 0.19628681580531449,0.4470768595056849,0.9784320151265898,0.8557218058694193,
 0.11022544822828328,0.6247695582692911,0.540992061662739,0.9850924310213398,
 0.31854152325533547,0.009163489348692266,0.9185221664031133,
 0.2426941957078993,0.8524923562599145,0.7916202310110455,0.28204370728152917,
 0.5683594892990382,0.06789365509438727,0.4422184150609003,0.51181871348837,
 0.5345886138324079,0.14977932771360247,0.5420192749685669,
 0.24108309055588717,0.5702968036914099,0.31372149690035034,
 0.47174212365427903,0.32010490523738466,0.3653629471781098,
 0.7640522784585626,0.633816664002437,0.8639329189019508,0.02267166274959509,
 0.4683898077870997,0.9048888233335947,0.524420666638564,0.04811800563009838,
 0.4237528686958467,0.6453237298202698,0.6385369673727177,
 0.003108382332252857,0.47120459437757845,0.18221799836097363,
 0.6271604243939017,0.6199399468533934,0.5746783646347425,0.39793253611096935,
 0.005898812585413471,0.060965389930464964,0.7692708371697032,
 0.7420809788593958,0.9424498541617423,0.4574675611856005,0.4498297941487629,
 0.7459904362383224,0.32276100492828386,0.03146089310332656,
 0.5679121749181293,0.39254560654609805,0.21047871029993892,
 0.9555681767186522,0.9979004263999313,0.35130188476180135,0.9147445575229468,
 0.22314258300854306,0.7108007887204247,0.3136192718141444,0.6236076554473148,
 0.5270528417967562,0.46200175073132543,0.39962480812930656,
 0.6340804295429018,0.6402466526845337,0.9756570855748109,0.849230885584129,
 0.19389251223308934,0.7271576101586426,0.920632658759262,0.592686426003986,
 0.7939989858212917,0.7015770368327144,0.039456968710810614,
 0.035226541947780965,0.7402161164505779,0.0507788097754307,
 0.04191898208883982,0.14871455168968362,0.5041476740445391,
 0.34188064981578103,0.39351227958776513,0.5561374362541666,
 0.1756319854311449,0.35395483526187,0.8163441900639563,0.030622688787298125,
 0.8514386940181695,0.748217211316549,0.7326689851124755,0.06156152197556897,
 0.9044744805063974,0.7002070745403874,0.6739704873202478,0.5577183988193929,
 0.8893850866660891,0.7238957580501051,0.07935464712490459,0.3948165567090286,
 0.3042690105165198,0.651314172260812,0.27865960962038694,
 0.013767185918941882,0.3379053345669938,0.5799351723159144,
 0.7884523087406312,0.5551401991413245,0.3503835661414473,0.3056192704341537,
 0.37653637768562065,0.6277454527687808,0.2077608964157851,0.2579549220377507,
 0.06976636793848434,0.660125011816874,0.49439555279573155,0.5897824549014189,
 0.04569385850785801,0.8045663076564243,0.48738709634782607,
 0.41859192551007607,0.4098570601734616,0.6694199073176288,
 0.006066477333186215,0.41874170805028244,0.31797874197571696,
 0.11328416360904103,0.6221715416515703,0.8775317033622186,
 0.013754729154746181,0.19433057334441017,0.7703993393245561,
 0.20493982649472708,0.1150813313852177,0.9404776344178466,
 0.13335676069772218,0.44717413181734667,0.7649586704561254,
 0.04923930253727804,0.6147647564524464,0.9707087857574318,
 0.11328631318383386,0.8645229199657024,0.556070888854118,0.5051493145779132,
 0.7355741181784337,0.43300113231594084,0.5369891776132917,0.6744527729172612,
 0.07217995430714363,0.05054404408083768,0.49288877413327836,
 0.8354701556150612,0.0186286716984716,0.8888997035804996,0.2085324848793454,
 0.6455054736296254,0.008721034685615803,0.586028252695443,
 0.45957257941752905,0.19593859469168007,0.012489920588687808,
 0.8798606489785546,0.46914344308809275,0.8151210049867873,
 0.25298780809957944,0.39208853965297075,0.8950245399328408,
 0.5793011365088812,0.7135371788977527,0.759594604496531,0.44139968378823236,
 0.7902242889008778,0.7260503333083744,0.036634099562937616,
 0.6118255301493053,0.47240038171679966,0.8750678165661206,
 0.005184568878044082,0.4512436433692062,0.8277256012703287,
 0.24868313494907968,0.17194442566374923,0.907951821922368,0.5125909029498169,
 0.12802930538000257,0.5686427797539497,0.37739906749095564,
 0.02265283289287734,0.9132069296930966,0.6999241468594068,0.5886680899696121,
 0.7416190762040364,0.66761816121494,0.8812053474590676,0.6197031703847105,
 0.5082512513688234,0.3377598672740645,0.31733028475150094,
 0.053779468119472096,0.3135821261352545,0.6609032323044369,
 0.01272826408749983,0.2850417497209726,0.8919113048627925,
 0.35966285731486325,0.27369004913108497,0.9511935593359628,
 0.9956479374144321,0.2564078938608275,0.8646938536460546,0.397259152118111,
 0.25970092938935474,0.6072268067850668,0.29920785665355176,
 0.08219420679871792,0.36765849294814956,0.15794437979463205,
 0.9069280158683104,0.19996172547413504,0.02515817390159647,
 0.4977231909386872,0.8090447316143341,0.21898810432311988,0.7266045936697471,
 0.3358858956612075,0.2994999141154636,0.06723306648391647,
 0.22030169955135437,0.968698042063672,0.8361040184142877,0.316971040119279,
 0.005068456720509595,0.7594733322008846,0.13561160855818044,
 0.1706056785561485,0.9999406969611878,0.4211224674784486,0.8078622812560362,
 0.18201939339631323,0.9991500020545834,0.12440058400224285,0.544945459928895,
 0.17588854645264995,0.2001617663430778,0.6359988920156914,0.5919081386706075,
 0.9251317292447345,0.38125621750471406,0.4095451456306951,
 0.12135786607224519,0.29881395386302534,0.27460786225061096,
 0.8666286549268257,0.8253581915167587,0.17407455777841097,0.6844779392384667,
 0.6685823104224285,0.48465735812544686,0.18502106690755082,
 0.2818875568149275,0.10920037802957527,0.37173215578583163,
 0.5807610998687411,0.637985213242749,0.09638570598079355,0.19154734698717069,
 0.9752314366271064,0.09626907397797224,0.46075378876753803,
 0.25288178033189157,0.8134860883313187,0.29087150818660745,
 0.9736879392866288,0.8700336379506178,0.37265847928365936,0.6395065291409423,
 0.5318615245725069,0.341197463137896,0.28698526623076304,0.13474133831479373,
 0.5634182306196306,0.94116689874941,0.06099992248680253,0.6460617808502461,
 0.2919201392278914,0.8352616775549102,0.02333378943172615,
 0.31621265946668553,0.5572448504136543,0.6247130025638624,0.4319876664651049,
 0.19667426906020846,0.8926123239756302,0.7007947031609156,0.9004178335003375,
 0.21626114328951884,0.7805588329887664,0.7126680513318359,0.6759373562848513,
 0.6723709967464242,0.6888226922829646,0.3844392117715705,0.8598352288105618,
 0.2386185266018107,0.1592025744028699,0.5793356945760639,0.35713401746650875,
 0.42746887426889346,0.3891844910636332,0.029717217134125007,
 0.6198809822297294,0.1641447032225959,0.6458320977561064,0.33071288303174984,
 0.5828848725695508,0.28764124817301484,0.6914518891155124,
 0.22120442989467637,0.23063421229615533,0.23356720144504628,
 0.8721412869374455,0.34560433826703674,0.5174504484470386,
 0.11582486779506995,0.08248300367224792,0.7787266068589632,
 0.6415182264813847,0.23289190644677238,0.5690015870466967,
 0.10794485314338986,0.8690846457272378,0.11362663223000402,
 0.2667831148807215,0.4913275471849686,0.6425379036475494,0.23512880838964045,
 0.26886175382551836,0.6057954571339029,0.6903831890892537,0.187708550820485,
 0.15450488548777974,0.5698536278414171,0.4135858675567319,0.9178678041365849,
 0.6012872409024501,0.4309380564530182,0.6868695351377927,0.6251481987067933,
 0.17715845954061438,0.4867202202942842,0.6339374800415589,0.4735239536440352,
 0.6081415340227052,0.40320404438692337,0.23416654516550972,
 0.4845940479662507,0.6190953932383916,0.5055793041949143,0.4841650576693649,
 0.26819057780725697,0.8109091212644037,0.6333901551156181,0.6930355010045417,
 0.29582719206824104,0.5491414344758466,0.987070182856182,0.8612656369765549,
 0.9156373743521642,0.01126788163900061,0.4684344274713753,0.2900860141339626,
 0.5915291463972403,0.44144676617877243,0.21339070540202476,
 0.26393137297156166,0.4660503706571385,0.921923301481033,0.24537221848383828,
 0.8699054465804668,0.3063307157847399,0.49014710987324583,0.6590566697555093,
 0.5347549045924033,0.9483210580176089,0.059627208521822905,0.891033339327924,
 0.03303878691190354,0.3504651267557559,0.5044461467570247,0.5943353559287754,
 0.3904451818458732,0.28924459715909534,0.5391571923478302,0.810952543674442,
 0.7038143783788121,0.569537947805939,0.8099576448682628,0.024898714963756508,
 0.5745199341608643,0.0946256839653482,0.6918878302236346,0.557358201377888,
 0.018985683026486844,0.3267500929340732,0.44935245155171555,
 0.018976121047659777,0.9131321377311653,0.31152707754821984,
 0.003903915206105668,0.299722747058944,0.2633495572292658,0.2975525739931091,
 0.5347218885743867,0.8222481558476613,0.3143198023135684,0.04093193119499616,
 0.34383179096896954,0.19980951948402026,0.43794827362714817,
 0.34297768465433487,0.6692250492614397,0.6815913527791337,0.5031933336829432,
 0.4008651954589013,0.12661133738310393,0.4268770227350409,
 0.37438205598234997,0.8908118404768135,0.09794332493628344,
 0.5202303296703219,0.9958207529697175,0.11850750038179858,
 0.03160048109192126,0.8009089549944937,0.9826908559061972,0.2336911416929628,
 0.7054459223031185,0.8075606659649801,0.3553601660225636,0.6722914982559375,
 0.35339003570141325,0.6614421693907179,0.9060078325383679,0.4117464763503347,
 0.08092112610377367,0.566604423440305,0.7350704492665812,0.3366536307556005,
 0.08513493409252848,0.06730376251623282,0.3009796891506211,
 0.16071815885726326,0.23639804964016597,0.5748848160205529,
 0.3701273511030688,0.15433843743770292,0.6180082137771846,
 0.04771162483152336,0.5671187508401965,0.8121215302852962,0.8070065639333768,
 0.7867092674774417,0.08833185743840843,0.8269220008894207,
 0.06294014249613344,0.5202064318917825,0.15804771548004926,
 0.9157903693458012,0.43032888869708086,0.8320278244303099,
 0.39273944703483854,0.5375365580177475,0.27147040389567945,
 0.4735831901276679,0.04929732612822779,0.4049176327220503,0.6831023901360673,
 0.2305024253433876,0.759939612879143,0.6701553237893632,0.5389899784181377,
 0.36944452612426526,0.7911506937945121,0.7720459686246043,
 0.0029600955462333323,0.8694334684792442,0.8952521100850949,
 0.769066173599661,0.6404108647864428,0.819800232959869,0.5132873731567187,
 0.3555889035720321,0.6625534609658178,0.12203656409892638,0.5902488025481214,
 0.8578735868428989,0.07014664939951942,0.09447991956322044,
 0.9629291441901952,0.9594462687097911,0.03278331505987142,0.5042174697336321,
 0.2824077016348674,0.22058788467654344,0.6566137733793183,0.7586874214846037,
 0.978485741626301,0.07532249556691839,0.004444851375853043,
 0.21030543367627053,0.2783067778465106,0.7877775147631648,0.3244787931839992,
 0.5020483546557042,0.04006631827601681,0.47246122820793135,
 0.8350325643039683,0.4720533619468348,0.8684490432231184,0.6451881908416605,
 0.47590363004754654,0.3178482542287898,0.2691070107444361,
 0.14382885498844344,0.8438261769024312,0.4450492261412484,
 0.43173905507968335,0.1246973426012491,0.44137584771498495,
 0.9429498352746583,0.48546706297753217,0.9185522763217882,0.709159404794101,
 0.06494630204930796,0.5067006908590698,0.8847724614488448,
 0.10190253199464872,0.21877093279889026,0.5009157794218855,
 0.8902385999095174,0.9981469382826449,0.8705407319345673,0.8734990532302082,
 0.5064027308641266,0.9248698410124807,0.3549218635112018,0.3003661928131611,
 0.4146713936393428,0.42609072265800974,0.540773493349774,0.09392908603988481,
 0.8839324590253947,0.21724903065675705,0.45736739491403167,
 0.5951174921048632,0.8163194858662893,0.8529737088766892,0.6758845436846626,
 0.36534439710081545,0.5381857715555869,0.16089358368643025,
 0.14570085011794287,0.2812545894284939,0.5705179633148648,
 0.21622405773491715,0.3957140499043551,0.7774420256641926,0.6220892863063398,
 0.7463873857026866,0.38768579990614493,0.4433097278074657,0.8753497965666408,
 0.48653950241067956,0.6511832684405051,0.09050735510779484,
 0.09322440129046639,0.5046244317685737,0.6878785278115525,0.4081620297596934,
 0.2386774492602104,0.07563500367783882,0.1864952314834999,0.5841114879771103,
 0.516953282016011,0.7296218947696367,0.23231960878028146,0.4036306195550885,
 0.6476704180283291,0.03653514361000809,0.3580253004452023,0.7345549641264884,
 0.8886592905454876,0.905531707759873,0.8739422185889646,0.7342836776713068,
 0.6068876054460652,0.8482118796017974,0.7660015033188912,0.31440608833711625,
 0.32183517781628224,0.6714137189819467,0.9875233902041595,
 0.38666111383356583,0.18304491034424808,0.8228849056528227,
 0.3550321154918693,0.2250739496006644,0.25193022320793235,0.535404618318473,
 0.28760891576403114,0.5287272009949183,0.20871325974089483,
 0.6722492983867541,0.3731399845258122,0.6496406639450164,0.8920734083186334,
 0.16012867804579645,0.44314502239892484,0.673440834060449,0.2896499067690155,
 0.3392013441293322,0.10514311735893533,0.8212366948925415,
 0.06554788525188016,0.025130003373570187,0.9212710021395109,
 0.1631156364391857,0.19718110676195966,0.7813468693614399,0.5915562350209113,
 0.42394788311775367,0.9067021799571732,0.9069991164564415,
 0.07914645792865915,0.931236235292113,0.5890124101809237,0.8395716650122349,
 0.6502656856252181,0.946919528308917,0.6243987867518883,0.2781276481446021,
 0.6513927323917962,0.3025395700260882,0.5699207210400051,0.7712837622696537,
 0.6049997756293707,0.4582064659958769,0.14680642697660717,0.8724366095994988,
 0.23370949713720213,0.31552983925034883,0.25921600042111437,
 0.9728555033453878,0.6827833536274922,0.9051014801980604,0.08834634645122008,
 0.6826646108239609,0.6622002665093931,0.009195415652322736,
 0.3089123498953281,0.3254078965577285,0.7305086398811333,0.5421386693282917,
 0.8566433554092513,0.9055689188654203,0.9486559707055124,0.3545269276435854,
 0.766682968432186,0.9620392170284494,0.3619991652657706,0.49296431877403546,
 0.7621396695197497,0.9468578813389602,0.8075237197853131,0.8550954717251879,
 0.8428299052030486,0.7530730834240518,0.5505045382343451,0.0475376939961496,
 0.9872776426861594,0.6940985820680948,0.4768524987453985,0.01311787515335916,
 0.7091620265655121,0.28903536643115557,0.31289040563639936,
 0.9704087739500122,0.19301075843581295,0.9533734275997394,
 0.11802496666007933,0.9324641261098138,0.9644294080576907,0.7797925027403158,
 0.3086916939946487,0.30491718877108986,0.9540680722132361,0.6146359582479363,
 0.25874411545273257,0.7418345649352653,0.7397688913355943,0.7923993934685063,
 0.00418125384238166,0.597270079136953,0.9769074131191452,0.3626200321220563,
 0.6924600275781482,0.01942865436002894,0.8659557270696686,0.4167134191757341,
 0.4215850017021434,0.7731416358863945,0.4400789113345741,0.7246522740455995,
 0.8362674704761076,0.8375917816897169,0.9018579534626774,0.18478741157045642,
 0.6979493522055786,0.35791453678264085,0.7403648582908798,0.5759319468658016,
 0.3921482549894584,0.9917782414265928,0.31511449007878645,0.2780985964361239,
 0.002294839638390722,0.32008720344532704,0.67405596154014,0.7455361283110387,
 0.4355528174090131,0.8631876913114575,0.19446041221390709,0.4454855567654945,
 0.9121963343469344,0.27829086522759017,0.5876939851615155,0.4282972200248021,
 0.8210063790840301,0.9452060777482921,0.9112194350821451,0.7949624161505491,
 0.4189594868958061,0.849162209663821,0.9811673701762842,0.6555264700606176,
 0.21244290955763556,0.5043479190378408,0.821740669843374,0.24146048087778216,
 0.3815302340535006,0.9410488162147813,0.8565792676804552,0.17049606952710228,
 0.7062074142112653,0.4883300323937795,0.7861117947472233,0.04370010990550788,
 0.6879714932792389,0.1091316381505616,0.06983947624075137,0.6930675123345181,
 0.9120488664689679,0.41445013111931994,0.6411513933661447,0.2996600140411083,
 0.09222004752917146,0.9132075515810026,0.28993630452224584,
 0.9948574962014841,0.5426123830550408,0.33971117051702926,
 0.09667792964193733,0.09960962402579432,0.4144676862402159,
 0.1475102548174143,0.40543836152248347,0.7754293824324786,
 0.46139227653599524,0.18771156450597515,0.8270032295148162,
 0.9193394448334488,0.05853226079653362,0.6034438096243769,0.4496427911239411,
 0.5139401978221779,0.9074887604244057,0.8196093338849958,0.35163760344318173,
 0.9525527505766542,0.6613562595765676,0.49304881905405495,0.5869722852607783,
 0.1889424619208886,0.4060674551376737,0.480059958390852,0.01823537335003389,
 0.36623995513253493,0.0027941605057553165,0.15547751279631095,
 0.27058245888072996,0.41908561664559996,0.9151046931794129,
 0.05279162349591504,0.10328491852407706,0.6794159469268402,
 0.6323523136843021,0.2687319186738615,0.03314945187125662,
 0.27546602589757574,0.18717293718350858,0.7190297565329509,
 0.21393508842312903,0.07568444203271807,0.5450165783623274,
 0.8947356516367309,0.18198827695619424,0.12548504627178814,
 0.4415474821960269,0.5791700646088405,0.08631956440731892,
 0.29910032871431413,0.1783679162476428,0.8026197830300855,
 0.16850585778445937,0.4237151371583767,0.705915696527564,0.815192167342658,
 0.21222840733952741,0.0594831206065346,0.9848218296046247,0.5872656432537957,
 0.5579167454117947,0.365142440041605,0.3638605278850874,0.9996910073894586,
 0.29086882209889775,0.416701186355217,0.670162791626442,0.6796016043461484,
 0.9906874848994724,0.16533431038007018,0.11377468492765819,0.694982816636251,
 0.9801806399980605,0.3788973492426597,0.5393360189595509,0.6549774347878884,
 0.09158631107847837,0.8655671648314968,0.6651304872300615,0.592911787112635,
 0.8365775734016587,0.6018356978948679,0.15823272192761195,
 0.46010511348508765,0.3305201643440334,0.49892430198943116,
 0.7757409305530903,0.9939678155107046,0.6509493814933837,0.41493947130002107,
 0.4697747026864132,0.4967380074224361,0.5281015859401608,0.8174450753481403,
 0.02275849898258786,0.6581475600199218,0.8130511722452455,0.596718887622439,
 0.4784424076200148,0.8538303074710367,0.8730831341592462,0.19216269181884416,
 0.256099919727507,0.9240300138915508,0.8485675134964394,0.48717455715902436,
 0.3531953893289552,0.3361343294578367,0.785681946434272,0.720388249870903,
 0.07753452969727281,0.1555459328023101,0.05064649478844141,
 0.5268405290492287,0.25529924452930663,0.6365163581212812,0.2952504666688107,
 0.44482244845272456,0.07444626305518764,0.8029437262663708,
 0.8412613526780512,0.657775559346198,3.776357975247002e-4,0.2968985044338901,
 0.33554361996290105,0.9279901161736194,0.36929267361361473,
 0.43543245993919266,0.7559504326154955,0.9864029718731078,0.9611389071933191,
 0.8782473702217993,0.21339278134023554,0.5834070166731048,0.9704521979050067,
 0.8113735128788813,0.8664246816730234,0.3013574022477128,0.9137322806791388,
 0.03832448860911708,0.5632049297398843,0.6182179957973329,
 0.031107906522426565,0.5839211631545522,0.1271115781662524,
 0.17070455999675072,0.7942187248089583,0.3025909840363892,0.4508360208592965,
 0.22484629238764642,0.8606058082919343,0.6795965577452565,0.4176085813792111,
 0.3270956508507502,0.18337221040674578,0.7992067217448013,0.5982210520656338,
 0.20724363754665787,0.6801754321153902,0.4802042815731854,
 0.04138766365505331,0.9328696705836241,0.2651656375393441,0.7441577800212861,
 0.28503815696498647,0.2975637772753894,0.8624741086728089,0.2526247105860411,
 0.6607788534112351,0.2801490109492363,0.22526829894301015,
 0.20399030007288643,0.6842702542229241,0.2106572259592876,0.2804810175246557,
 0.7391739538296298,0.7981661460604825,0.48006736678657236,0.5886468165272574,
 0.43127540605198456,0.36919978339544746,0.5687848602121558,
 0.9683019915596077,0.19598728785291042,0.7900987555535152,0.884480974992495,
 0.8990519278091617,0.5337105584868671,0.5871405635956419,0.9821106994767059,
 0.9895297774932701,0.4410478482407564,0.7854828611778849,0.3106898398379728,
 0.23593447708706805,0.7376008793419934,0.8685411345182158,0.5512630728053258,
 0.975432697107733,0.682986888251109,0.811148361774239,0.4546046470891293,
 0.06895710707750802,0.5971618493575231,0.02658389748146328,
 0.0350591050956689,0.9286931822499467,0.9589391880410862,0.12841567405901777,
 0.6367816666940949,0.47906928367509793,0.29701319821032435,
 0.9566424912901608,0.6082743743061892,0.8368970953506111,0.6078637992928149,
 0.34820740948990503,0.05650535848643168,0.6552641952284395,
 0.5212927644319589,0.523416364256161,0.6635812144259737,0.30095553013322673,
 0.19065078180235717,0.2886858939760836,0.5327813015067148,0.6807545556633098,
 0.6034196737121751,0.506135194679762,0.2336912786327543,0.9138325850214257,
 0.02356490418201851,0.3431713565864183,0.6082074635604034,0.813476043336248,
 0.258358489510114,0.9099125258814691,0.20512471161595625,
 0.005986253362540328,0.34654687280218,0.29335991553795415,0.7170525656983009,
 0.6539403848259991,0.2136741672976874,0.12617139131548372,
 0.46582933955599193,0.46662519273547876,0.6753708314424706,
 0.6280650472971246,0.9225903249684397,0.0332256751360942,0.22303333772512834,
 0.6578317578966282,0.9573737037705953,0.21782067086656653,0.7849100254004029,
 0.4605125363445952,0.348337784081988,0.0560234748502445,0.15280588289353425,
 0.5798519040371065,0.8121450636767497,0.5087629420071956,0.9007609795729855,
 0.5018611498344546,0.13604511032875188,0.2975920206240773,
 0.14419052646585162,0.08753706401319494,0.33243500642318247,
 0.6729581034965731,0.958920826199086,0.8786611906557308,0.7009655193612213,
 0.6498467013688235,0.8034585478184874,0.0498988375424545,0.2943434087900415,
 0.6989437552804733,0.23866270010914103,0.021572801153238896,
 0.9611290325165056,0.07215684019192048,0.09129862128985922,0.128140337490789,
 0.35139081760954305,0.4243285547578829,0.4669609835378913,0.9892509750009796,
 0.5054822217311332,0.9996134130277657,0.8530866761206507,0.7504612777479235,
 0.9690254291236302,0.7423795443858691,0.574599000577719,0.6469144072241516,
 0.6990440132712814,0.3496205107138636,0.9210452884307865,0.08852497328265119,
 0.8697395595297912,0.6787138056649803,0.038870854498544216,
 0.11682095682302696,0.06949656974459661,0.5829573016902677,
 0.13452991860357955,0.7614017212960471,0.7778501557309964,0.5464701833504368,
 0.5746779956568342,0.24776741229666654,0.702442703065088,0.18762686197895684,
 0.22048959501118015,0.6272987278278036,0.9268684007273893,
 0.13588615110250923,0.4328362608032742,0.8337936287088858,0.2584688254920402,
 0.4340239414628071,0.3504200841235442,0.12094982862568993,
 0.05691653094830862,0.45930971047488434,0.8246414870829025,
 0.4464623926080292,0.012127827484298193,0.041141125649692856,
 0.8502852311819549,0.07354925328061779,0.1766326060647072,0.4184265076073722,
 0.4719091411037287,0.03295377617122497,0.5503798428900282,0.9642373255731658,
 0.0028525637432365025,0.8033565604157878,0.6511222965904307,
 0.6658980838188961,0.15174562046302842,0.9497750677608466,0.9324011249179854,
 0.5902587409763744,0.3771564049662094,0.4980319247604248,0.9802867583864272,
 0.46200913452470294,0.8218766699877444,0.6137183824403427,0.1414711303166707,
 0.5129339535607914,0.4934615037997854,0.6096948302416098,0.3331764102917095,
 0.7439476606218483,0.5452179707648529,0.4469828796969749,0.9233100309296406,
 0.8133795093318318,0.23409550237294263,0.5356977120197729,
 0.03755588161765877,0.9080378769714543,0.8233788511988462,0.9618147273077932,
 0.25813885659717306,0.9236097617277712,0.42917456253358144,
 0.1274291682437797,0.45516083461374257,0.48120269714674235,
 0.7477158444862969,0.6202242843542625,0.2357523457778662,0.6154845576461986,
 0.047747050150684034,0.20843507162263963,0.2076093108872784,
 0.8385622702185076,0.8483105106430437,0.6363032775270457,0.8775185429792175,
 0.09981439645653079,0.9764515634767891,0.9853309898080174,0.607338622782239,
 0.055629987120948465,0.992030994826208,0.43663079778996317,
 0.45814543172278455,0.42492757836863315,0.17092218858545882,
 0.4305817792829467,0.729144350624092,0.37405549152052864,0.9271289277726266,
 0.9417672240822282,0.15563743614085412,0.9303644470874057,0.5781988557606745,
 0.5996840640271243,0.8180843535074832,0.9070432366074181,0.9487243593516976,
 0.8613942141123639,0.6663829376469379,0.9540092345442731,0.8634322082741671,
 0.0921891244984332,0.6329270646458016,0.19972244512251192,0.4217199013563748,
 0.7934453948404313,0.04545043308314323,0.3800185877130776,0.2015181764614662,
 0.880844928391544,0.008305618398241177,0.39111800847030376,
 0.5524796772134131,0.6171601018511894,0.11063139481375339,0.7742503691589172,
 0.3067770207460224,0.010825443959748049,0.2902316511472375,
 0.8522999236580449,0.3783124379796652,0.02024422973642992,0.3182901814967697,
 0.6460667467527269,0.846667648712222,0.4438123116623891,0.4316966963956914,
 0.6660163361944165,0.572239542961353,0.7294451276882187,0.5322387968474814,
 0.451394887340837,0.4379864147166209,0.5120365051798704,0.10006358638681456,
 0.8276674278869708,0.9615974692276229,0.8032093786100818,0.08016231905631888,
 0.3585912272248921,0.044613264483582604,0.279733617285441,
 0.03921414651706823,0.5584223079214586,0.16832066513533428,
 0.2047270279325445,0.1597173252417231,0.8693080335362963,0.06722473873180135,
 0.6878908140299197,0.37101659242455376,0.9977363696745669,
 0.23852046244764014,0.4402198915961274,0.47193347104186356,
 0.4427777538832348,0.17443901739511514,0.9486015561833246,0.9887328548109215,
 0.15504334474253723,0.6000484959689822,0.841784387306194,0.4331515051434056,
 0.12093326573260754,0.6260214389330135,0.04360541865517975,
 0.8662580223027744,0.5821793981258007,0.3016241542345943,0.1714356510160432,
 0.3619986430147826,0.7577605568785483,0.05696162320707032,0.9703554212722862,
 0.731817313219509,0.9544647665097288,0.6196892999289693,0.7515588836961182,
 0.6830589710601039,0.30691005330362064,0.28239084025083094,
 0.7717186107519748,0.9788600203606561,0.45539070300460105,
 0.40538132991219467,0.6317692428470668,0.8823762528846284,0.1374378529384339,
 0.053692542760912176,0.2591385414991254,0.27709063748552853,
 0.4596623052969362,0.8886072248457819,0.06671395998572316,
 0.48060856714557654,0.7035212974397081,0.6273954617583668,0.364285366638735,
 0.6161409555966003,0.8330250542565619,0.7624646102846875,0.9347194668091079,
 0.4404034469742486,0.03032136793946283,0.05896480931921566,0.999229293679575,
 0.477051420278518,0.31167956398866015,0.6193981308068282,0.8832130504650527,
 0.37158087712155874,0.8932115404960195,0.006830744730420335,
 0.1596141185497415,0.9496836277751586,0.8082449101177696,0.7212913524194964,
 0.4825219590813906,0.21826572485200535,0.9834895473552832,0.543821000883725,
 0.5461484962074681,0.5918542485641375,0.7686912888358177,0.5360549044659365,
 0.6156941362970996,0.45594830463194647,0.5220367942465814,
 0.03096906581906944,0.858651171975491,0.9472020445078564,0.33077694120358225,
 0.3527744617063482,0.23153301733325815,0.8596821004455374,
 0.011145952882180121,0.209813838707547,0.3473850380074105,0.8080398480241373,
 0.33583646749411,0.15442232158080693,0.7578403374798923,0.5064963168714207,
 0.889026377526609,0.028921977705547386,0.11831856228499638,
 0.5231046175081838,0.3095082625460984,0.7761534652840142,0.2764325039373742,
 0.4472957215427651,0.3376791999394111,0.96505000193797,0.21339573849660476,
 0.2986053652388999,0.6504254774825434,0.13105374184142726,0.4837263691209577,
 0.4805486841105404,0.839032922490031,0.2666883076192518,0.8733981706187396,
 0.12690843051539158,0.43252818188045006,0.06335570754120778,
 0.5140053276251584,0.6606473113613386,0.8826866444946811,0.3785896020059908,
 0.0011590068895536154,0.6000509742351583,0.006068361523146937,
 0.024796581713269927,0.5244580088637232,0.43996403258115824,
 0.8434830249611917,0.22567009322951392,0.3194533915439218,
 0.09875368807675255,0.18137663528670145,0.3783002040571245,
 0.9374032622822281,0.8380661773396845,0.09470191530651972,
 0.10670133790317671,0.4116471509717411,0.49866015205710745,
 0.9544631649233718,0.7221829309872687,0.09739164966486125,0.7722113291289774,
 0.6408828044823426,0.49691759228430965,0.8550327972955379,0.818725708162184,
 0.9140985857798215,0.6585416044229102,0.8202986984634703,0.9016873967594858,
 0.07235797590243054,0.9307746112043824,0.5380467271933586,0.6456547315626371,
 0.28761762979183414,0.8332636178258771,0.5316482263249812,0.8370044750543972,
 0.7716505362593276,0.2498127509580257,0.09432699977045489,0.922942655756671,
 0.08514200506275715,0.33397138705485363,0.7550871157682029,
 0.8271106284696609,0.2087118866346308,0.5781784188243252,0.2644535347585495,
 0.5235804427158652,0.030168374075498328,0.9219706770296606,
 0.23453690751091594,0.323245889960888,0.8680072814813831,0.5913638042087591,
 0.7616862866868556,0.45535491154468744,0.9674867893259451,0.2364891231925691,
 0.8811048558586703,0.5226478544423213,0.26388863097565374,0.9011753596042597,
 0.40929267183784446,0.7155799088894806,0.12336594072284479,
 0.9313299472307408,0.5937419826471001,0.6972028511794032,0.47552477099597745,
 0.9516613521655573,0.4137899512918981,0.1255772260490171,0.5662991729659995,
 0.8335822627486875,0.7683253002962211,0.3404046973697016,0.8248000193757875,
 0.2975367263231433,0.33547232784865444,0.6392715742623409,0.6127390860869263,
 0.45228486375523724,0.4339121803124467,0.10654525041418661,
 0.02752534111236904,0.9226506175501211,0.4679687665048021,0.5599798726013545,
 0.30695717611499085,0.11554828734776978,0.12536664821921395,
 0.31142567991375936,0.5368598527300114,0.7038826312389032,0.5072791823472251,
 0.3169247649783802,0.067979552513734,0.17075252804948482,0.2022057991451518,
 0.1828445411680637,0.19778716845236322,0.051614441323380666,
 0.2665625942891091,0.7206711640978785,0.6571083479337146,0.33134790125200364,
 0.2662384550488953,0.7614804099450698,0.21187022851082238,
 0.24977349893876388,0.9092585792467268,0.01872811899514415,
 0.49467323497246996,0.25250347259495876,0.46025220117210053,
 0.8665673581459064,0.38417302467811354,0.21072482858518682,
 0.031061864687281227,0.20505555309947732,0.4651778715911543,
 0.17428228343375363,0.8210144133778337,0.1687274537503085,0.0995559013617816,
 0.13664916093886448,0.19303589119909947,0.906620308940086,0.9502057591949167,
 0.36044921492275683,0.37331912542004375,0.6349402104099715,0.578698042932076,
 0.19614014970734028,0.658645304232932,0.9809986352997108,0.8679369548080209,
 0.9835074568275513,0.08928391100665556,0.6710651341798878,0.6613373076986365,
 0.7379167654927183,0.1993849965708615,0.8391947225120431,0.6634034137432187,
 0.4630102789297855,0.6140560830951324,0.3968318472344772,0.2249265819431796,
 0.038488060633222076,0.297385675993199,0.578203203558715,0.5990171131809537,
 0.003955690309302207,0.5650509360939008,0.013245227546374028,
 0.25881403028347283,0.6106188459448657,0.2905240243261663,
 0.33201339789422135,0.38116097529411497,0.8163590638154116,
 0.3067644621427418,0.13209802600326903,0.7072801845415296,0.9591423008680862,
 0.9890688498007989,0.4716097063493113,0.05163270671316278,
 0.18705833741608224,0.48319897262043454,0.15303712628541444,
 0.4137450076145114,0.876601381668295,0.12119369849648354,0.7361501786571238,
 0.5126299737953228,0.17321353841193743,0.29134977345139956,
 0.3505927207824292,0.2020089631200478,0.7084807262187248,0.9393286024708676,
 0.45742247408039305,0.219991162175718,0.6583628878187269,0.9530802058566453,
 0.4142382173560064,0.15898299889252043,0.2649509682470187,
 0.11254262333207077,0.638575613066956,0.830526051053981,0.37673558641118987,
 0.6427003816310679,0.13914964822789733,0.3355121372785086,0.7989227281171182,
 0.8553366551987401,0.34130892043371175,0.31183883168484383,
 0.8239950605319386,0.5425926952410189,0.3764690875617883,0.6691144094436154,
 0.38433212327479194,0.23429926028643677,0.0946444769645951,
 0.8199855459739387,0.13162409033409772,0.5742787383696958,0.7946123574439737,
 0.0027863978266909895,0.5317555091088337,0.2858337947373857,
 0.9431967155517387,0.8828857247956262,0.2804291844677367,0.3414499321861918,
 0.10718030861571504,0.2787087468445214,0.08063869818875624,
 0.7191807428192809,0.1903589369561084,0.02098472011670982,
 0.33699210095286214,0.5406199623860463,0.4456917043051739,0.7852145452517251,
 0.5684495750775564,0.3755843877293803,0.10041308965801088,0.7725812236970555,
 0.472636635589204,0.5727134603198566,0.5131569557305836,0.9503124892533283,
 0.9544929392405822,0.6386578002903814,0.05221342295537856,0.9967237694843041,
 0.5316560500364089,0.9851897548252844,0.6090116942902285,0.07506634623193564,
 0.5963428791613807,0.2848699150698134,0.4296045103240047,0.15776798433922812,
 0.73175294504627,0.703871956198394,0.48712318488277107,0.12375613831687249,
 0.9326041675465955,0.7748138707055708,0.6332096093834239,0.7668654476728318,
 0.5653190274969901,0.9226639038930248,0.5583827982275515,0.7397988612855645,
 0.6798964532424416,0.07559269771649413,0.38315917492993945,
 0.9626781225502351,0.7820993250636337,0.191875444034452,0.21634339607695519,
 0.39102096942345177,0.9896948651700085,0.9218029528583059,0.5978605047028928,
 0.8479391166341422,0.3818636905067829,0.3021821640798603,0.8210035990118792,
 0.7104675090311879,0.37447272728150727,0.12096897764193204,
 0.30491787977028495,0.10637116630035304,0.8169639283425216,
 0.12051473561425141,0.6976724460686295,0.6149156885803398,0.971433220760495,
 0.7840932130448729,0.6601605329684823,0.7868778903945051,0.8252176885297052,
 0.4882999056410434,0.33569090434602256,0.4907228763867415,
 0.37715722544896324,0.9182550553083224,0.4440517357743191,0.268276060115453,
 0.7399112466844343,0.8135290378196124,0.5557195359141953,0.72223973344292,
 0.72345362271541,0.7840534638002509,0.859123605694602,0.913912499965874,
 0.4328761883932528,0.7452174311751057,0.027407624249494167,
 0.23583594736703406,0.8928715019492918,0.5067985364639555,0.626516519616882,
 0.7422073844029458,0.5392319113500292,0.3594225854195172,0.8299437522957515,
 0.38990969789047114,0.004458328821497437,0.4280254726805248,
 0.28323306298792494,0.46466033755654224,0.7811670748110413,
 0.8485997488568675,0.947639342137832,0.1213542781046607,0.2992263845455724,
 0.5457017945108281,0.7037458399452676,0.11978300190697722,0.7805871301538421,
 0.017101792392376414,0.6419352557537116,0.8157545385867664,
 0.08489880725119614,0.23969408106082413,0.6496661467048168,0.505875270843089,
 0.9837529315259697,0.9041345670238058,0.08196879765951703,0.6514872987911399,
 0.6481464394043502,0.749327391216194,0.7529274023823862,0.7307783213049196,
 0.9458952330018453,0.6525085599369387,0.4022197995113519,
 0.025139027167282668,0.20167418543895754,0.18327428854579564,
 0.2495075452051092,0.44466495384439575,0.583208564603565,0.5008970221737993,
 0.1561290382988908,0.03442444676119616,0.4900883517274208,
 0.021408539989677844,0.2549477210942058,0.3277529098549682,
 0.4194977864542686,0.1974420509484046,0.14447909599558417,0.1718926319009204,
 0.3507475300335994,0.6323483902015876,0.6215719839852402,0.9698091989191823,
 0.4402824029075607,0.32989907432803434,0.7046937478862776,0.7968106209021819,
 0.4040901478992249,0.5640604819878338,0.9483729061543809,0.5984257854342633,
 0.3144012647342366,0.2978476523603527,0.48542777863206577,0.1799248545132731,
 0.07150200247082228,0.6387098860147141,0.94292069001198,0.6841583926651257,
 0.21415310676456178,0.5183926195676838,0.38187237035888744,
 0.050535719023954706,0.2537355516487184,0.5102153735149867,
 0.24806605902893197,0.4098374418234463,0.09880357584889421,
 0.22076514698197747,0.9100287185634078,0.6503685611330369,0.5679813314081058,
 0.21393177105056205,0.26953416741881164,0.42960121946410657,
 0.7198906572315911,0.6134437218917823,0.6536723002795137,0.4414472575143724,
 0.39472602120738454,0.4565989426366446,0.8936299541599143,0.3563565999341618,
 0.40877894123495273,0.2443282330035852,0.8222005482482299,0.2173434892007231,
 0.04513252390035061,0.736008814760574,0.22198915494391402,0.5334526063005552,
 0.4000562451271483,0.9893440099021813,0.4391046576067067,0.04150554892031477,
 0.31506407878798826,0.1901004379617307,0.6325381990954763,0.8476449359412964,
 0.7110212265866824,0.6675830405866595,0.017216353078632674,
 0.08074949879350479,0.6480909547609356,0.24064374196388139,
 0.021361223312257205,0.6911489984322207,0.46946439196084844,
 0.05154129985075018,0.508782767722562,0.2668050222321572,0.3936759600071702,
 0.1588776346313071,0.11761845542312621,0.28395394334517976,
 0.6854550755102222,0.5593730679102,0.24000937106049303,0.11187553723515642,
 0.16732141761640518,0.6723467613823422,0.1248960459049715,0.5166616952982116,
 0.6784972935373574,0.020033549007734686,0.49597982558923015,
 0.8289783044863462,0.4039511394262554,0.4864090029802285,0.3261624079500691,
 0.592064648062577,0.20822429665984477,0.5080811662387041,0.47450945336668193,
 0.27656232833315464,0.2800887839753924,0.8231066337858249,
 0.17893362988974681,0.10071859153004392,0.1099685215331413,
 0.2526563064333708,0.3031005917633163,0.18320508987278705,0.8659640421525953,
 0.6019588196350071,0.2972806196495138,0.19095262626200205,0.4538931571575062,
 0.6972212432203428,0.5359695877884529,0.857266847211558,0.5426351888661116,
 0.2738006168154017,0.8521182018918697,0.16123621774461605,0.3915615114551815,
 0.5007140068093012,0.4077101843093933,0.8999905790315542,0.0905304757220522,
 0.8061590654577708,0.23402419334505276,0.6212160175805881,0.6323876387883415,
 0.3200058889685399,0.16680214170342844,0.7187385188252611,0.4728386790780099,
 0.43588600717531834,0.39976121884107185,0.07150940635167424,
 0.38479211891180054,0.9705017370253326,0.6874020812243167,0.239756286009429,
 0.9896977820123581,0.5233951448000393,0.757725806863736,0.31741045556081127,
 0.6525222853795073,0.34964349919231796,0.28624041190842875,
 0.32209468784130346,0.17496231484783697,0.06846486785758277,
 0.48310838607163387,0.48434542630348787,0.9334895340702462,
 0.8083228518491525,0.5702041504771591,0.6559296982894322,0.12823115265019547,
 0.2463494731320014,0.045082806385939866,0.7718293054510798,
 0.3475205767145275,0.14877367548692622,0.0061761097867936154,
 0.33450524400927495,0.17978239864029089,0.7984577423708321,
 0.6928892514760809,0.5254936599968416,0.3899024482384277,0.3786342867956869,
 0.31352822770457245,0.7421197560965146,0.9744052903251835,0.2557519302236738,
 0.8568968821306302,0.9709132585280738,0.9203235912507881,0.5571128327222756,
 0.9650268502218243,0.50477173545795,0.019299737857503052,0.8214864433651102,
 0.664215000679398,0.37400201156706203,0.6025900967008431,0.4544406966455653,
 0.8008660215287473,0.6825499458942682,0.03335513022052272,0.3624182359483832,
 0.6407240819647863,0.9101351772809771,0.9927738871608092,0.19468093672038567,
 0.2528790402271146,0.41730895491897824,0.41589088367515126,
 0.12176417428533326,0.7827589198096578,0.9088009464804946,0.7343855944815662,
 0.3084047605716096,0.3073241170065367,0.7791863253228206,0.725576504921371,
 0.7295198851440999,0.10235234492481637,0.7979629146568686,0.4136901175095222,
 0.2804953238978365,0.03943923766797497,0.05232823998699032,
 0.2731455686574279,0.6963303743326106,0.8398107110636507,0.1100922920435895,
 0.7352774558544547,0.47104460513462487,0.953311129241803,0.2367118315285961,
 0.46017413974270716,0.040376845710213494,0.7478299924766352,
 0.9600775073690424,0.39900777003795473,0.9129332553278584,0.8227315465701638,
 0.16317909654518103,0.0674829902882037,0.45640505279537114,
 0.5247946347814859,0.5847113656978464,0.9843411822745857,0.6825876756780836,
 0.4137104798451876,0.2687614498316381,0.6479064872572151,0.6593860241324557,
 0.5743912653879839,0.026212027760657453,0.9546040427607229,
 0.25185017136549626,0.4469530992252866,0.42471485357419514,
 0.7024044023538889,0.12269593040671678,0.5998705388307218,0.2744054798692883,
 0.1822132087088837,0.6882951079153768,0.10099373845032844,
 0.43718708807342677,0.12761322624175808,0.967701357545131,0.6204842751522611,
 0.2924835211276102,0.9028745880078972,0.47860660855471915,0.8125932979180821,
 0.9590046946609174,0.03383733182180604,0.12324333927276032,
 0.8854233449592064,0.07223534718985225,0.26277437497020606,
 0.42557945730331104,0.8098537986204536,0.4306829151010103,0.2931308997090928,
 0.3009814144982188,0.4676229627239762,0.8617710774140883,0.9007595432005413,
 0.3322975836701836,0.17261642771437746,0.1305585411105611,0.5432020120778551,
 0.5863147026459412,0.5058678972372399,0.66262061839405,0.2667113949412828,
 0.2824884004262682,0.9771826488155791,0.14156194002965627,
 0.09556724717766907,0.20579313576447855,0.7839778990280056,0.546677865949666,
 0.6277864327247169,0.7692270341930603,0.6246302858320206,0.5693721300513078,
 0.7287906044154705,0.12786879440143295,0.857393191535306,0.4299076502000465,
 0.8505726658204154,0.12602477136250156,0.4580927447408544,0.9335529035750998,
 0.4276158960463031,0.3000141118588925,0.3891858950092062,0.8659169216985163,
 0.005564477006880897,0.5298170698409198,0.29224969603332007,
 0.5708988910543836,0.24607743438370144,0.23757064252266813,
 0.9801652171235194,0.9820557247793728,0.7896831526409261,0.7275299756030479,
 0.7850748417200297,0.44545761076195145,0.8768556369796777,0.7130791403447716,
 0.27292270459583123,0.6109769034198678,0.3263414885882072,0.4456825308496083,
 0.39650078860475,0.04632988426805362,0.45019000188399827,0.08820364302227479,
 0.12078951386643222,0.7430378114513437,0.10453691033739188,
 0.9236873658311311,0.9530068733667374,0.8039806587844075,0.23498070248617609,
 0.5238256820327132,0.8938180230946431,0.2840820269064609,0.26619891639123083,
 0.8170069627834062,0.6537975363461964,0.1452683440445115,0.14516564980384195,
 0.1242363263043379,0.7588282841900765,0.8911074981302989,0.688478985747153,
 0.6877537293903988,0.7878676529413958,0.7600808510627298,0.9156298370588154,
 0.2396589285340155,0.9227306479695481,0.40584711941392115,
 0.06577578422443231,0.6071291537549866,0.6939834154744499,0.9945466043599339,
 0.5078579674741411,0.9530232364921283,0.40707681287200315,0.7328242869712255,
 0.118422940741028,0.9259494627572294,0.375823137363964,0.01325330456410101,
 0.44655033891123486,0.8087973716275971,0.19534896047797512,
 0.8436643436729768,0.5238404502564465,0.40299202447033433,0.3591105609247115,
 0.10152886371249248,0.3540953180967076,0.5974291502503011,0.7879426603320134,
 0.3548805425368122,0.3193125519753395,0.9885808946382039,0.8636418978883625,
 0.8171325971667734,0.6068999425007288,0.23006870396010448,0.9818704135822955,
 0.6807752135767092,0.05149859040055893,0.858657767141591,0.20362965436966485,
 0.5240812887741746,0.5755261513709757,0.47407254219531947,
 0.024257810358087717,0.7917278832916155,0.43931641729788407,
 0.44716977575226047,0.10150395039895943,0.9902869383915746,
 0.43496398299695493,0.3743683896183454,0.8181663271187379,0.9415699817057089,
 0.6566861906953084,0.8222635975915831,0.5631483141558847,0.464157051157436,
 0.1453907646307646,0.8371612885823965,0.6236302357613482,0.6596580734427545,
 0.535206760656862,0.4192103367367147,0.1382669645489919,0.43200946318222755,
 0.160672060727457,0.6460988923830209,0.9732439347367131,0.7950176990363025,
 0.01610759369678727,0.1787676570970147,0.6987520493460863,0.3075322499716937,
 0.5238618016111655,0.08340284979858836,0.265271662549626,0.5951671086954878,
 0.807561358754522,0.18015278253826572,0.5474178023664705,0.03646180452169534,
 0.398955200442047,0.09131198357358306,0.07770986200816354,0.3403785176364983,
 0.4053820391993874,0.76088479019749,0.40933495822964305,0.509823889134149,
 0.3233449292527919,0.07627467078275352,0.779779461482065,0.6710216128794048,
 0.5898058341183402,0.31308558488576543,0.9691252183169018,0.5825168897836973,
 0.764967376872032,0.1053992259999903,0.5810980692738981,0.11582024139651448,
 0.3135291752637812,0.7787962803802082,0.27388198200366376,
 0.18709710837299032,0.7420909462489022,0.3625051118459318,0.887346490156412,
 0.24155972955072025,0.954095087690381,0.5624445795137565,0.44491187681334954,
 0.9169341942729407,0.022935452339115203,0.27392825087760575,
 0.9123614752515078,0.4251113174027922,0.1939124343115124,0.30558303754157057,
 0.17284548391898524,0.45718199733359466,0.18128594394648068,
 0.14195823308996147,0.054210089132619266,0.6875559005661969,
 0.21648130239819963,0.9331173406763078,0.9126756385055036,0.783815226540936,
 0.8400740820796107,0.3076931319721795,0.5768518379483087,0.3003119435145152,
 0.46996130076399434,0.21482538027545006,0.13851118939926432,
 0.35532948214869975,0.6618612213177889,0.21974599011565965,
 0.7638774681321372,0.15159646187683784,0.26223916475723985,
 0.6144199256994153,0.5948908978629843,0.09387505503951787,0.5701416159908753,
 0.40661796082846313,0.6282967333746139,0.2484572019707465,
 0.16411227841147102,0.5338483071024777,0.9889235392575942,
 0.042506360155540435,0.9427025604760402,0.48473226242135503,
 0.5335797839902906,0.33198558580555193,0.1852787598244966,
 0.46955549726142887,0.23741892509203433,0.16587757814789095,
 0.9648714068589045,0.6525555995166419,0.25223965041538055,
 0.21733463477585124,0.6989765175150567,0.28750246672867374,
 0.12684473791635464,0.31902633556031124,0.8668901700613475,0.676183244340361,
 0.046288176122249336,0.27643601465255097,0.9754548260595207,
 0.6653748697832678,0.06238076839807816,0.6472594857091578,0.5108393718126769,
 0.8754126104877368,0.9192933095277291,0.993590368193928,0.8531852449419358,
 0.7747738010969185,0.18616726634626313,0.19494761715206854,
 0.3646665289594627,0.4920510252956912,0.6460559719125436,0.5841144793607016,
 0.8786797723811928,0.9020347806011951,0.833598884475961,0.6573320075162795,
 0.1717362364596453,0.33748277452656206,0.2673496154173859,0.8048760138284343,
 0.46928992290845384,0.30207728940783174,0.19595311775536084,
 0.3446506869252536,0.7157646850137341,0.07774447949628249,0.1494875729318702,
 0.2086890735967888,0.7876756956757839,0.968454710587356,0.47228670531607253,
 0.40239504774897394,0.6886329771029404,0.5033540098310069,0.4541226739669063,
 0.07740513137276206,0.11591983568925068,0.9034882450512768,
 0.9086220204754236,0.2664485139443361,0.7474476929673622,0.2213279236389909,
 0.10343867057790623,0.6915177525583914,0.961631011216322,0.8365467721458101,
 0.08898880849135105,0.19733589775729787,0.0761155580364794,
 0.5937526192585671,0.4444934548055607,0.5340327158270647,0.3945963957016887,
 0.5451065987114819,0.5532160586430321,0.204639403293143,0.8601265440107503,
 0.2506443269857994,0.513286046307514,0.23368409595653072,
 0.022835931156137912,0.20077145776619,0.5203894332831469,0.37802645291519943,
 0.8362244476572979,0.13368174286272116,0.3776634676705496,0.2060963756121168,
 0.5551850431691862,0.5529147976564639,0.018505571073401228,
 0.007892440154357083,0.12146143964827738,0.1731489186588695,
 0.519426950405242,0.24378996607584102,0.21977305872957587,0.6909737351326791,
 0.9119462424989331,0.1729058085230234,0.20452252787505287,0.4072345367446165,
 0.6235755853225624,0.028115241441008854,0.9106000715865672,
 0.9068156355908107,0.03799645751447356,0.9541395482372254,0.8215553844297381,
 0.9347956443129758,0.6065635364872973,0.2976425035505236,0.8253158664742228,
 0.31899433268844846,0.6772621329823195,0.7890974458527118,0.90702027176449,
 0.8445405475233219,0.8912399375040413,0.316334585259199,0.24241363000396166,
 0.6114476365830588,0.21800334527877163,0.20329840524732912,
 0.0994558035869143,0.6680633103150512,0.15730887186909526,
 0.33204173428061123,0.9302289734161278,0.7559079420642055,
 0.13180329271683489,0.6952583397790985,0.30502459548112093,
 0.3699159302821262,0.017966498410442622,0.04452036312213914,
 0.226672734393204,0.8025763975024636,0.13954934388281492,0.6774089432390503,
 0.3766602983789743,0.8485946583548887,0.1614404643698203,0.473591196348112,
 0.5454025128913974,0.30973980857715777,0.07788324727516582,
 0.6230095737932132,0.9600586955282513,0.49482014259265716,
 0.17074526250904876,0.18715885810755306,0.06704356841763448,
 0.5773928015970817,0.4794398049524673,0.7682064814104806,0.22529353441947586,
 0.9332763802197703,0.42632139219792053,0.06027768693643576,
 0.5894134143638748,0.5992159606768029,0.515414020369815,0.6277551485291301,
 0.19819406192994526,0.4966362366542173,0.13072822622136704,
 0.0724715704233202,0.25999283185492095,0.04024160863774562,
 0.9999718873021661,0.5933559298801686,0.1284265429534075,0.7384499547605465,
 0.9902341321961887,0.8251014900833158,0.013560268287596822,
 0.4813321471688802,0.4475477573960265,0.8778039325300236,0.4155818234557471,
 0.6121542612586156,0.048933933278058506,0.7360768687764601,
 0.3836332993635392,0.31496911507162106,0.2597296936612786,0.944568761526551,
 0.37390716837712945,0.02549646704316022,0.31802754594620686,
 0.25795652935675495,0.6946315473769915,0.4634924068815607,
 0.42187985802617367,0.31521447130738556,0.6014467730162694,
 0.03695714225573221,0.6644679994569865,0.40522651697999745,
 0.6973855745800672,0.9819118060260599,0.06553493085719309,
 0.03159821147451547,0.6616051500278243,0.8302577897000428,0.9921184186826981,
 0.30358425135153666,0.16121584094830754,0.1987757353716787,
 0.8099877857053305,0.9124996684640907,0.39976322658202257,0.662440297734205,
 0.4730129195407772,0.28519019831814785,0.08245464580825301,
 0.4296211954641739,0.9671133127289918,0.7266062172261235,0.32431809826379976,
 0.04047811127202128,0.30320352089585245,0.2996031787001402,
 0.8066854489316008,0.5437507423074992,0.23647536125504331,0.9863466104948484,
 0.14281985450521928,0.4874061486937519,0.8399526302194057,0.317841405152011,
 0.07755768646562133,0.7888318214178468,0.6209097181354224,0.970928945751489,
 0.7913218778216836,0.7420090940424713,0.17610947592522708,
 0.15267473349219507,0.6670335042383823,0.6495295247847028,0.4117299382369318,
 0.20993686630202668,0.3734028389702695,0.43818426565740376,
 0.5366730478933299,0.5320182844380065,0.82866795818839,0.8426610426502099,
 0.26224645008067415,0.5139170464036835,0.6947191779975279,0.9225041210917299,
 0.2424165590977052,0.6253572336661132,0.6477319871231564,0.9875236763240524,
 0.214750880645036,0.017678400305622954,0.7466193100131066,0.8139780854600607,
 0.9925081438010183,0.6389930779072249,0.08367893817378869,
 0.33954448696093864,0.801012361212165,0.6468037467652008,0.8386791728502234,
 0.026912065295650822,0.8176350711570471,0.060500527870101295,
 0.09637233571998094,0.3726885792062812,0.4523011752146312,0.3982654602619359,
 0.27733665956925346,0.4494717081452625,0.059202743320216245,
 0.1057757585733361,0.8271733249185109,0.4799751093796655,0.8384152094518486,
 0.6783261770804636,0.8117336880347558,0.8827347414765476,0.5181452701824213,
 0.48648834169111577,0.057333214739099025,0.6962724011119087,
 0.3288498803507798,0.7239589011789526,0.2213108058401,0.4203565095195897,
 0.6034361710114531,0.8175119854904797,0.6013934695392443,0.7434907168249295,
 0.6244002884346443,0.7773116742119658,0.8363888445842962,0.3903157585383342,
 0.45671090010592663,0.5769502203875945,0.1135574005761637,0.5833981680690191,
 0.9490325499336878,0.8379501477591337,0.5934940129034841,0.9034278022929038,
 0.1538274662049911,0.971348224756154,0.7460205242510216,0.6150191353240516,
 0.21947118827891865,0.16370482845737544,0.44253208870143146,
 0.10369280711088724,0.5541038045222529,0.09195188681569932,
 0.21451414776170386,0.1309380321534881,0.424432090021468,0.6556911911147024,
 0.41952542630828094,0.4136234130503873,0.2563447621914938,0.8042217231480765,
 0.8323474127098994,0.30806591456934684,0.21031612029961622,
 0.9134151656473823,0.15798161373629704,0.7126383324578858,
 0.07575836947743331,0.7654435298625393,0.1890343510885868,0.7637657202041132,
 0.5335384198991745,0.9020482159764457,0.3972953415445353,0.8316291556161888,
 0.61383229936862,0.37574369464914537,0.3450927491809521,0.29327941621696074,
 0.6810834989485144,0.8454357803438362,0.34717674121729714,0.5565285348403243,
 0.41265318024011366,0.6830478199111835,0.20936578187332766,
 0.8701059460960504,0.029048750937145984,0.08029212085564863,
 0.6491513973485148,0.9166362245114367,0.5840207600848982,0.4188309002315008,
 0.15465080718789614,0.9729672462656117,0.4785967043772503,0.1038948357688837,
 0.7902052865476635,0.3989646312346826,0.2500726588855393,0.38331404248847356,
 0.7311754539933271,0.934489768274779,0.5710511345249737,0.6284664115939166,
 0.11498639719996073,0.35283806690156694,0.34208314419798214,
 0.9094890893793686,0.1391209077210641,0.18633673068554224,
 0.17824650146516507,0.2619479805672489,0.16719228386531193,
 0.07270996194254076,0.10139507300070205,0.03247378524287925,
 0.7026997475819087,0.08106985060680505,0.5751181885649875,
 0.012150805377141527,0.6473321184592226,0.39345972858922684,
 0.1347735894592218,0.29181772790866334,0.18396234016931257,
 0.3500139366430802,0.2233630427624076,0.3190895892659147,
 0.015864566881219955,0.2778258264118243,0.5510652529563055,
 0.9157800732673897,0.28752637014913385,0.41730366303412236,
 0.8453792610460155,0.927391975926249,0.34861118982729233,0.3829198207043605,
 0.7462536875755214,0.11115543420277985,0.03271238888775774,
 0.5758963022201001,0.18995245125924076,0.7169489975179152,0.7623420202486912,
 0.8904051403411558,0.025151932855094872,0.4665380434778632,
 0.5350956985743965,0.6328572377202928,0.7843266234976259,0.46691076991482694,
 0.027456829143198136,0.7734098917844874,0.7866735953491831,
 0.8729706727537602,0.8400495863208856,0.9684036165882534,0.8150565484516646,
 0.3224001827627999,0.43540823649656757,0.9136814560455454,0.8525156771979567,
 0.9002012307759225,0.5678714263953708,0.5785297984978024,0.27877002263931583,
 0.06544550984708208,0.20908542394142393,0.4610624442899959,
 0.20383152561814488,0.6280356372422724,0.7453071696257291,
 0.42863444694591746,0.8428749000345801,0.7268088935707505,0.0955113754430672,
 0.40420561511194353,0.17046310009406074,0.9057552579592515,
 0.24429430598732949,0.46148300615554005,0.32195770045223504,
 0.7080222530981467,0.9131176402288848,0.6829447473883168,0.6560507633006858,
 0.43604840821625945,0.8557162228391504,0.6019934191997909,
 0.40623094601436405,0.43166103779534004,0.16627741392429163,
 0.3278483756422944,0.6257543009774045,0.5666058837129638,0.2828915085075375,
 0.17626594288939812,0.713051268238784,0.7584705647126684,0.9983913949501194,
 0.7985660440151564,0.15474257195586893,0.5458802491291268,
 0.026404905480857055,0.44903409208947687,0.07358895406637656,
 0.1289702671275581,0.7924553783225773,0.4897698015502111,0.5855720471069353,
 0.002356875520931667,0.9276054759222492,0.25327965829073074,
 0.3149736072818642,0.06314708372655331,0.6441041461272997,0.3401227595871633,
 0.3346270243031664,0.9024863711849145,0.1646956047176249,0.08627414588061721,
 0.8558085301870827,0.4989550019678163,0.015949892072413707,
 0.6990052389043943,0.015500792185709877,0.7289389261743741,
 0.13202431352619814,0.6809045099768058,0.8828003198280088,0.4163996387796651,
 0.16269482508947997,0.9676842745073673,0.16182127919500222,0.871307319809346,
 0.4876552657256098,0.08574847096462868,0.9171317454953029,0.6208300474847622,
 0.0791942211244081,0.5466886395583231,0.8845350126955748,0.2657038122670977,
 0.5533962951915784,0.3307016439356798,0.05027292585643439,
 0.33398122884186965,0.33643624809421746,0.6406485606854037,
 0.9924916014556942,0.40321188656484086,0.6054232786617925,0.5852702334779997,
 0.6745416007120152,0.8893956388740649,0.858668963019026,0.7075279848434696,
 0.4919673177187518,0.1854033718850956,0.9729157714942752,0.18909444568303035,
 0.43674016397301685,0.4359525840889833,0.37148315719765834,
 0.16289027268350031,0.6710190247858325,0.42190921360438705,0.835262162542749,
 0.6781676200232193,0.91783704993194,0.7849330915121328,0.9988763389385144,
 0.6572556503057839,0.7560419627412646,0.5813304332526175,0.04692954658482007,
 0.8160613307259263,0.02082834844593573,0.8991765472954139,0.7355413246819976,
 0.42825875887464826,0.035651375738232516,0.28266109035623843,
 0.3989601997657515,0.8158448206591736,0.969523145823239,0.6802564170919934,
 0.8389636208664677,0.7712939938984368,0.0023764290410406286,
 0.8262522693549903,0.9120398855359053,0.9379971181631531,0.8230615800194891,
 0.05316057589666001,0.8621390563881601,0.9557978858549475,0.5045602505280363,
 0.017871182946751163,0.2245867339824914,0.6637967611647985,
 0.45375807251946587,0.9319512487563589,0.47964917895879144,
 0.016781826016642243,0.4698575645290306,0.5765978389266846,
 0.11161860146831748,0.5995783241020357,0.5857676958943896,
 0.47129867412573967,0.16106445667519798,0.6365472931209599,
 0.29118418878877617,0.7843180171844641,0.38984524335862347,
 0.17716995123367552,0.7604780389070744,0.7552225878991231,
 0.31292119161143706,0.8694855965731381,0.7991467712318496,0.7126531741471123,
 0.5909174522084731,0.5339132354547949,0.45023510322435123,0.1991228488301744,
 0.8269280248446076,0.07809341177271045,0.08286509056006741,
 0.18924332317677195,0.9905807468321179,0.1158717345985375,0.9472090163364432,
 0.3801818981526266,0.9102645689137605,0.37401836839408964,
 0.14897194694541027,0.37748806050349515,0.21132351615806466,
 0.5913574741255108,0.6240964885705362,0.3061363699993829,0.44458236269714935,
 0.5526072985070829,0.9799064861467569,0.6676803431754446,0.9885448725907161,
 0.42135685639811926,0.9775206374646787,0.3873875556566583,0.7323704414936292,
 0.02594080738538762,0.9602927756077224,0.6485156046249477,0.8507332004338206,
 0.8382718120240154,0.7129540083161873,0.044835344558300516,
 0.6119142649376711,0.1596608294110362,0.024167214215428334,
 0.08214715165665842,0.8485360761665245,0.17164551104060022,
 0.44829101920055203,0.3360750264119343,0.16724868561868866,
 0.10881361092253927,0.25727870349857906,0.07217566583357593,
 0.9952255861998716,0.8004449229658154,0.19910988747697445,0.7995103155447721,
 0.07352006563834912,0.4639763827165879,0.6546992664869018,
 0.28168198619665097,0.837042815408367,0.19201466217897822,
 0.024311267486977095,0.6280001663098895,0.3391215979804003,
 0.5931113307475395,0.6887625744762624,0.4074468204978242,0.23845743679568077,
 0.4952026831314198,0.18695959928843664,0.11046852631424608,0.583555683747431,
 0.5243343234912587,0.6808600284924425,0.6540239275473375,0.7401992957436114,
 0.6544613511310026,0.9663896851240393,0.920295861463565,0.08007725820811318,
 0.9215787696817811,0.698545264249434,0.3691211861576361,0.7480575235431403,
 0.5865382864694283,0.22554821724408425,0.15615035697116897,
 0.17160390569677053,0.5773690333201793,0.44630940317007695,
 0.1834631347711928,0.64529876919544,0.4194324905597846,0.77468984543046,
 0.366037684644564,0.33688033355367697,0.8795481440229859,
 0.015649862905484313,0.003312697980028645,0.39286442970488766,
 0.590216159427085,0.2132597547217029,0.4948890630609526,0.45870974409301557,
 0.880739018040896,0.3152391215295791,0.6269913085685207,
 0.0039034363438789077,0.5307947102804069,0.6618618831336702,
 0.0761292307343362,0.4755803308748896,0.4741062228169699,0.5831569724063688,
 0.7281969883486517,0.5728495701814033,0.4791969566028944,0.3637075928695995,
 0.8473798195469383,0.7495286708136806,0.2639333733569943,0.29544773681347525,
 0.5787667016341764,0.7858799609518768,0.11638997231387482,0.716318977149684,
 0.6551274798455606,0.054787219217110605,0.06132970317254671,
 0.4315413777932805,0.18364393887086083,0.3485448772220341,0.823214238398635,
 0.2561894257145396,0.9431610932308517,0.4534104639345339,0.4946940149303216,
 0.1630526851899443,0.4500957618152577,0.510558725415416,0.8744633359038623,
 0.3747688731128678,0.8125402839589989,0.9199710316058888,0.36094374935242324,
 0.5966924260730594,0.828288944460514,0.979371187903708,0.9803737877272157,
 0.8887863211417367,0.9508368689562093,0.15452802947388733,0.9895987621132952,
 0.17158531201928895,0.8628697361596508,0.34270841604050417,
 0.6047679463740933,0.5431206128124221,0.3817680008581166,0.3982878702780307,
 0.27284483157132255,0.9346697316642589,0.6289600102281107,0.2049730739852954,
 0.6444330021061873,0.827601644400134,0.5602385353524214,0.05644077989008456,
 0.08189757735315895,0.18104464817079935,0.16081601269368395,
 0.2144811435427051,0.07866600394785728,0.17355615872101238,
 0.040491316642086606,0.9007235829235314,0.4330361611608732,
 0.2427025236378636,0.4208500360557398,0.4557666623295913,0.2587989218802397,
 0.3913933200769294,0.13097705141594096,0.2544104154462584,0.7117660875439668,
 0.5180447694168218,0.14374651060128574,0.9505820901979574,0.1662890274621729,
 0.1924840202533531,0.09889960857752456,0.12328272235434767,
 0.7827623021623251,0.868858311617405,0.6763154504057287,0.6108961957609346,
 0.1429270034272777,0.3524523932287731,0.14832547039793398,0.6608774597648213,
 0.655074024424647,0.05606591403218453,0.5126697300875576,0.23991111559230494,
 0.5406663065328501,0.0854425098104179,0.04879904304029814,0.5294476533601367,
 0.9455693478583127,0.6025166448672554,0.5391689404369429,0.9127988830218311,
 0.3084570308981982,0.29728865093289913,0.7663462992486083,0.6131159022808879,
 0.039980053860036224,0.3336023636659565,0.2882006661939154,
 0.14961822195802854,0.7208753828752179,0.3875813701808628,
 0.10706687960423422,0.7168135385834908,0.46205459527687665,
 0.7904843292738362,0.29378334996669986,0.3371589439428485,
 0.37877517640325964,0.3088100974293364,0.059213862476965695,
 0.13333800838460086,0.6383537330711478,0.06537522719933042,
 0.5897528548004332,0.8444796482421896,0.6365144179478296,0.18265291537371153,
 0.8807947621955937,0.32924501969470155,0.891502914588,0.11792571219915648,
 0.7595280197380511,0.4988484393772009,0.9397883598295858,0.10805705382049813,
 0.8485234251378261,0.08216732175072028,0.35390872113748706,
 0.43108739575311517,0.9635418217219994,0.34612722195895496,
 0.27136528945766714,0.7876112531543201,0.988334702141686,0.04112275149640243,
 0.8409713802400371,0.8548945171726824,0.11799686773874707,
 0.26582656109940694,0.46125606235016336,0.19134565087239552,
 0.8788995381885689,0.942301335216935,0.9055163611238541,0.37622305068752293,
 0.8973431437514465,0.934494153543066,0.5236799201052658,0.3938204930382626,
 0.49329639483363075,0.08046759630092937,0.7053412671885767,0.827362937130671,
 0.9532703602151348,0.26243631760394925,0.02555040074077919,
 0.2979506289234811,0.8916383449628277,0.3467059842901363,0.09216082345597731,
 0.47876273573561756,0.42096284726354616,0.6378418884240231,
 0.41702617783493334,0.16993458794684868,0.5897090026637697,
 0.3200177788521976,0.7537952894377083,0.9510198744407347,0.8882508730993095,
 0.5562122888178657,0.8946047223958009,0.04244089521223948,
 0.14754722009198984,0.4651296272471497,0.61461472674624,0.705185996985666,
 0.8063988836059323,0.8931404862723269,0.6563684384991187,0.5250091203231293,
 0.4313640820288096,0.15780040085061453,0.7471541449343374,0.7866004586121058,
 0.006518643586776562,0.69862703120092,0.3305443900328262,0.0682699044260403,
 0.07036614688822451,0.13944798529674873,0.2373065390550987,
 0.9266034496934099,0.9526976120312901,0.8761073680306796,0.48561109941339764,
 0.6700087690826825,0.1573633977744262,0.23009135569035655,0.2085037025733104,
 0.8461714281606388,0.021227487482476604,0.976876242201356,0.6606472250987375,
 0.6022089184105711,0.5289619570301337,0.8731031584094839,0.775433227312388,
 0.31098718001071224,0.3527617926433987,0.9707532638305886,0.2549072039936452,
 0.28296999366870323,0.3791448800079442,0.20291859206388585,
 0.34507432409241834,0.03984596375780114,0.1474109731030795,0.506389337980684,
 0.0034715824188183975,0.01654050147018804,0.9341145897326119,
 0.30110239444140274,0.6330919892203174,0.8923195370322062,0.4691291720078742,
 0.7805955838482148,0.974057532245676,0.40660742067553834,0.35064899924133863,
 0.40821558255584656,0.18535587748440807,0.052059979511437704,
 0.7212023175728168,0.928629780816326,0.7877638243170102,0.6641889184175228,
 0.7508789322650298,0.4941904071824753,0.9747134760029219,0.7711412997706061,
 0.5718314310339281,0.09255592811500568,0.9572092053269284,0.7253189480430796,
 0.47667586512322124,0.7273712097480978,0.8244624457361547,0.3909210988610634,
 0.1216761785811522,0.14893714401215385,0.020011908499858122,
 0.7438804041464113,0.039838174418089456,0.16858032514497623,
 0.013041025285895458,0.6457883618228978,0.4400156276626551,
 0.3263527882265056,0.01015517776139685,0.139300807165455,0.3880440660289953,
 0.005083167533734945,0.8114257409408071,0.5761730833967066,
 0.5004345555537363,0.007184896832665766,0.807947762566581,0.715719022548413,
 0.8241614019173324,0.6554143089725516,0.11652542670669441,0.8895612288866602,
 0.21002337568650453,0.8093483070281225,0.12993268276940972,
 0.9385461065318258,0.4281821439099247,0.04347730405441819,0.5510544059518696,
 0.7750908997490359,0.3005166338135622,0.45802749458829917,0.4853927608605646,
 0.08829771024747712,0.9695974520557757,0.6794823083970651,0.8013787416460227,
 0.6938534739249584,0.55063263462908,0.2809996848991553,0.20126666989519926,
 0.46602136937838845,0.4916407741368213,0.665993852777475,0.11307544750759735,
 0.47493731440129205,0.2809485618674541,0.1842748805907123,
 0.09273740024063926,0.17810393418340253,0.19760126531773037,
 0.02258546056859556,0.7276794199752528,0.4620302201234273,0.9606504630436822,
 0.6024756576290363,0.2828478381808276,0.19057450913705043,
 0.38746946701964446,0.2518509008512322,0.910247182738337,0.40154545113066786,
 0.6955027617324877,0.5321557831935142,0.16469250248875023,
 0.21932136519999434,8.215779856989291e-4,0.8509020163504953,
 0.8199791538033686,0.4016046102409838,0.14242100316412554,0.8245459931296174,
 0.06388700947151538,0.31296121127810705,0.9422913769204087,
 0.8669212625389789,0.6250043115458561,0.985952574448005,0.45405273847194816,
 0.522458762465143,0.7474068086075791,0.3551383211754404,0.9692264961420302,
 0.86638970474956,0.3570185006384258,0.7020692162058201,0.28524058836571353,
 0.18491090739198668,0.5113102378782554,0.4118411636355016,0.181643953599692,
 0.8368308902235065,0.5731584854351636,0.8146440011482108,0.8491423027258221,
 0.5697404606205223,0.5371800067335903,0.8374869166281884,0.41646594634698053,
 0.662175687359961,0.2270612268635136,0.24820055802965169,0.27208801193694465,
 0.5383559943204925,0.8033458494219827,0.6681883175237224,0.7814828224042126,
 0.2520565980784767,0.058630390488245254,0.7463460510892456,
 0.5255330173424932,0.6447252581837033,0.7165585302768963,0.5880952516165913,
 0.264466749591447,0.952235227839068,0.6010725504016654,0.03787671317623453,
 0.5544011889333563,0.028463449711558342,0.5347331328508628,
 0.1389210045962227,0.7628233284596028,0.22878633691043404,0.2710865845760726,
 0.07944462523421547,0.9940220230594184,0.19295340431091224,
 0.34486479640122936,0.8316282049725063,0.7839554819496418,0.7960637615596455,
 0.16348265292090125,0.21026826278275612,0.821513319498873,
 0.12443801688863987,0.4724900013171689,0.5631189574947641,0.791224817220397,
 0.49982444304075524,0.2026733823502881,0.5561744444141205,0.4126640774212148,
 0.9666413169863902,0.4074159097436114,0.08843587155558774,0.6086233585242625,
 0.617160686569207,0.9712621763855078,0.06961234256714677,0.4845981147560565,
 0.7797158527671353,0.5174015651225176,0.7634718997004245,0.942305949336151,
 0.6782375822312905,0.2613337098630115,0.05512971153932844,0.633397387012542,
 0.7234320919244828,0.30955466616680427,0.33174013238119593,
 0.7199616746247648,0.9284202386853602,0.14797606008771957,0.8070194148539569,
 0.4462104488276847,0.07797715490015189,0.3205131644112982,
 0.23203328648960564,0.4750758568201948,0.5164462190211904,0.5920087905608834,
 0.4559315554835951,0.6860211514852371,0.42021577945404587,0.608759898893405,
 0.713752602332305,0.6759329658937551,0.6970250059446763,0.5346141231078062,
 0.8131403477138875,0.56642380725101,0.5837049472341567,0.41211715210195565,
 0.5676518862729153,0.5662799730983764,0.0884903182025355,0.878569719581169,
 0.749546001484178,0.9502878802612745,0.4264648803092669,0.15309577705205069,
 0.13582219216307978,0.8048604295273484,0.9308849796394466,0.8194321296130948,
 0.9231977938096708,0.37220469135508727,0.012531377064661453,
 0.14043075899698299,0.36157417700245564,0.030434745791780182,
 0.3847227034698145,0.30504047934298273,0.8598127089087845,0.3602236394999936,
 0.07802849350594077,0.7438923932595224,0.4747026924333725,0.8159433999140402,
 0.996809640213016,0.5977787661427704,0.3051312361730587,0.6493139840269844,
 0.7251845659876632,0.41790336885166934,0.19840234980878046,
 0.5049447016317061,0.1193047529241521,0.37897200183548674,0.5314000157428221,
 0.7329179234567494,0.4810537196362342,0.24472915258114836,0.961202717581923,
 0.652435623613333,0.3951064238558739,0.9806684274486643,0.3682494535749975,
 0.7758446729586681,0.4146235652034498,0.5218965494333476,0.17253570805872287,
 0.8366193953672679,0.5065002578581376,0.8253546547250852,0.2111218682419913,
 0.22821505446062562,0.751445953431843,0.7637225893248025,0.6955665788549683,
 0.7126032094962418,0.06649643461838362,0.8638487114251628,0.5703988079381873,
 0.1914206489058924,0.43975012264433144,0.02761396750818279,
 0.7873447189231481,0.14611369987810063,0.9320205383895934,0.8476328197121847,
 0.28694249572062847,0.6401726383009765,0.5793372153182772,0.9513206423015247,
 0.23067415249688605,0.3399361120679697,0.3744452426513787,0.2017837423948795,
 0.41056126566108986,0.1141023346647273,0.24150478976656709,
 0.0632089153161034,0.5378326585145013,0.6905303120189563,0.29170090453472663,
 0.16129294120611204,0.7996508546715005,0.22557553762204552,0.538555181470144,
 0.9384796644097236,0.2710342550543914,0.5160582502331864,0.9684737009549252,
 0.04063421781689158,0.2942046908064987,0.5636634090603705,0.6532505924879679,
 0.5627431739573668,0.20143703824280756,0.6592691113119744,0.9389725493413295,
 0.4962427613940986,0.587410555161167,0.2747534342456385,0.1804859918472589,
 0.3913845774321928,0.26423898914273614,0.9712121250751018,0.7517763196932132,
 0.7718690049606551,0.49746122873949317,0.131547003642963,0.9810935960184265,
 0.847194478414465,0.418739430494814,0.39649624236525083,0.025081680323397526,
 0.34703503070688135,0.33589145311937985,0.9849788699894808,
 0.7789722225353306,0.5337695352547817,0.7403185957250373,0.9513358348321942,
 0.6923712602657534,0.8561894077343364,0.5131329851022809,0.22147750588652226,
 0.6756708334272478,0.37402019344817317,0.8419279525831418,0.5655453526922465,
 0.31568944989774217,0.0668199272190314,0.1277710081079212,0.7141526214111698,
 0.8091810974009577,0.08144290794223397,0.5023811893428365,0.8047989464243168,
 0.46271631642074185,0.35835678217585176,0.4913335651223204,
 0.5196813372090199,0.7894505276370897,0.17016819549596807,0.1666292881997239,
 0.2996812283156167,0.04897991774392785,0.8817463137175836,
 0.015262049039628911,0.9189327499173261,0.6579673118655938,
 0.07393046594485086,0.45512993057238693,0.16160011447721678,
 0.3474350318656062,0.532459173598095,0.8763602127851031,0.8536016974925712,
 0.7700490502830133,0.571398121458379,4.240399321320254e-4,0.5569375381904992,
 0.13427344646753325,0.5671806589873306,0.19848570468190674,
 0.7585537464439784,0.32895836629923714,0.5460696926357061,
 0.22215120072599726,0.9311902119589028,0.9348025750623189,0.2984776782715902,
 0.49308165365819867,0.20720854360212837,0.3611862242281765,
 0.5125878685759953,0.07235977186840326,0.04540594291294586,
 0.04600206094507364,0.08085645326683477,0.6317549147715223,
 0.24077597222600655,0.056717262378882305,0.15353466456239662,
 0.4753061027000477,0.46511605676787826,0.654083354278043,0.6488988945708094,
 0.833751638064498,0.19741957659623033,0.5801425088535105,0.4603012601234998,
 0.05838426862182178,0.25574984401014433,0.7906033653472332,
 0.025878487423530605,0.15103401974075603,0.10267717120273145,
 0.4388176389960039,0.9264570265595575,0.37021584353549364,0.6049312496926772,
 0.7518383511977064,0.5428277940146007,0.339301933962876,0.6555525287371401,
 0.16115411190031936,0.3993358212850395,0.3973582312755567,0.4618926730009647,
 0.4895817823869627,0.5531046579633934,0.6488038072202562,0.763332158010823,
 0.6284773983676912,0.6031135159561337,0.4693765764117177,0.4768814686485361,
 0.8870378341067551,0.913276136107186,0.6690271298035866,0.38378230513388145,
 0.336344877374232,0.3379303197184592,0.40371785968018825,0.12256594730024595,
 0.4811041370320638,0.09071618719874464,0.7944978896966853,0.64751798030188,
 0.2195176624637032,0.8549479633081294,0.6271830922078707,0.8770742722921894,
 0.7436117793959997,0.1285619557878549,0.9333174113220413,0.42179432157732566,
 0.30427924626106406,0.2620588761387994,0.40526581209614776,
 0.4508879334990701,0.5430404554303291,0.8963454307606951,0.8246064446054389,
 0.8479804251434511,0.03198043068698564,0.6538852640005031,0.2907400664201527,
 0.7993881992533156,0.19071717405607014,0.39422053213461594,
 0.12647695406592174,0.17913544097978296,0.22803107295588443,
 0.19122720854074804,0.5231131171105778,0.08240056924107686,
 0.5274068035629591,0.030030944590705544,0.6134173268175501,
 0.14435751422904586,0.9537377370588627,0.8452687818869629,0.3715454024338466,
 0.3498801592845061,0.785096707089667,0.9197105236213619,0.37234850396901154,
 0.3071116380254343,0.17744171189045554,0.3496588322466043,0.4254189982768677,
 0.48419789090760434,0.2675536706372693,0.6135442158715971,0.9501911726083593,
 0.24150893486229785,0.275571363876971,0.2008440201605196,0.533150508400642,
 0.0461312850104868,0.46791427159938215,0.4722601447273177,0.5548671559804168,
 0.6337144640109411,0.31970119989828194,0.6229351967472765,
 0.33939596066403843,0.8873122219098822,0.05919804839507292,0.803542612584982,
 0.013358473431720475,0.9754835777523422,0.38531045540031084,
 0.09635570776483227,0.7354273630146719,0.9205713813110976,0.9459756348977683,
 0.2328354586828978,0.2327396398741537,0.667973752281083,0.20621768995862122,
 0.006337901228868903,0.8622183878500811,0.9702023648650775,
 0.7954139806038665,0.49260591394686637,0.6277568462823472,0.4569786124444668,
 0.605689566812967,0.3417473464864007,0.7978286250092431,0.9707237529588314,
 0.658673240262829,0.03374540548953375,0.02054833319589322,0.6707483376037096,
 0.7768085182212642,0.6994988670956355,0.12015084916037133,0.3821855379459833,
 0.7581887989384041,0.05433104777920428,0.8894450306807974,0.7054446914980605,
 0.809698050048163,0.1631280502646173,0.5479825566256423,0.3465745024736431,
 0.667859160615156,0.4210501511638194,0.4421852153725797,0.017687548416975174,
 0.7968721387921525,0.6513990365902436,0.36284059975373273,0.9168490388324482,
 0.19510113148540664,0.8213727916847638,0.9490940398890779,
 0.24191186739041304,0.9323885964597096,0.4079907418266655,0.5069837476879772,
 0.8922861049167589,0.17908847581316378,0.10771484346542926,
 0.9630834698701233,0.33771113651336393,0.5505093893854491,
 0.08697000104615038,0.16053158680058177,0.4737766855561918,
 0.7724757667171893,0.2916591332651066,0.6202418611624201,0.03546792292499312,
 0.5317027105367755,0.2070187674336199,0.07108089660911265,0.4931137639233997,
 0.06426773535401353,0.5464165650075038,0.21068774105340626,
 0.8146673207794162,0.059577143689374834,0.9224341567531126,
 0.4353880742563919,0.6477982933561207,0.12097558552105359,0.3290855916446992,
 0.7967991602854554,0.5050099191127956,0.8571946610993383,0.1259501217381933,
 0.3126266661148982,0.6572914581352616,0.09209281188150564,0.95987512461804,
 0.9363821236754153,0.6079842916716659,0.7733533206731938,0.03725813616815454,
 0.41025466166962343,0.06932478819133303,0.9672820675130867,
 0.6928308348414264,0.5130217803641605,0.09294021529541341,0.5778273870075367,
 0.0906145255942169,0.38955215726700343,0.8849662529734715,0.5541145758044812,
 0.46227148783368854,0.533669181898783,0.01379517941414421,0.9025234378127248,
 0.14245613115677758,0.1659321044761468,0.004009596155530648,
 0.09209370487615054,0.5611978325366332,0.882191891031665,0.3935771588464618,
 0.3103307539696022,0.3985323046355804,0.5401771849831941,0.5358096651218612,
 0.8864784902365803,0.09840628176405342,0.4653582662221831,0.6570476542401044,
 0.34731797192530156,0.1348565960090441,0.8931289710646438,
 0.15301076907793376,0.5332562007459306,0.4134318250890703,0.9886320628099167,
 0.5702087500561019,0.6512961540036318,0.4346517043949919,0.9319367626344421,
 0.38673112656964503,0.5059372161404205,0.9857332558971899,0.6283344691689775,
 0.22194727668303194,0.9755608859267996,0.3056550337423909,
 0.43450020648263055,0.6496621770030332,0.4775051674633223,0.8344774439091799,
 0.41376581221288766,0.39706838028703295,0.5518127571196,0.996799636393637,
 0.3823067025647313,0.0015695603433244498,0.06489573685422667,
 0.9678971869306822,0.931592618414264,0.06937739349433258,0.2640450270366199,
 0.6710703424302724,0.4177874094472068,0.03709263654624584,0.5412605389312274,
 0.49343905865421134,0.19018847779967052,0.029020409600894692,
 0.0745638001670147,0.4255588629200687,0.9521782574297034,0.8548012318695726,
 0.5462327427761119,0.35325437326754605,0.895040753903741,0.19885796830124525,
 0.9482430135919054,0.08320691743267794,0.6924829026683129,0.9323556206441326,
 0.9650173970441036,0.19876106361553636,0.8226703665950681,0.8358431954950042,
 0.26313869652201827,0.2628412463806894,0.5624941361587693,0.8328073431765697,
 0.36324152988537006,0.4829568522147697,0.8985967391318153,0.3900084799873229,
 0.7161320677070804,0.4681038639163567,0.18354314652425752,
 0.32562143984440617,0.8080396396630781,0.16465826615598955,
 0.049767384299128414,0.9230284630844334,0.964338874413826,0.9543394108224424,
 0.8039880137769387,0.0802185238146691,0.1288411365414902,0.3580001456866535,
 0.4951237486448282,0.6699025146874504,0.4709720201927321,0.7852220602644928,
 0.2969788319685982,0.16186082429342097,0.09245168183423125,
 0.9026935868695075,0.6378888633856717,0.2564484304961637,0.06384273992822631,
 0.06490301282845468,0.64530123669207,0.3485427414440325,0.9791922713323544,
 0.32242981444824526,0.28048603146821927,0.9443422371548245,
 0.8227511156092239,0.3236707724176652,0.21092420451621918,0.527430035036131,
 0.8777107879048812,0.8059601451643592,0.4577437191335457,0.679325743481158,
 0.7294170457082387,0.6554048050181467,0.092168037320418,0.9298560408168066,
 0.3496041802757749,0.8315702144075755,0.6955107823188409,0.46549056468163974,
 0.8489129072246232,0.19095269133204928,0.2357910836549063,0.4923180160220605,
 0.20043644597473864,0.9616880210799155,0.36402448427603296,
 0.8497772875427081,0.3017159064986943,0.24888661378293475,0.1604478523671644,
 0.7961079695465433,0.35467050372472997,0.12238679924049367,
 0.17458342288825068,0.16396492583233635,0.4384128051467422,
 0.40459774342764043,0.5120586442838732,0.6931746097157413,
 0.21257980391917908,0.007979810802658749,0.5333516287095609,
 0.06345207920953755,0.5333229850436285,0.769742706547507,0.34833762578956307,
 0.9763132517525956,0.4802067590015785,0.09339689387058148,0.8795973750108582,
 0.9328869311435212,0.8875290830368346,0.7087707940872888,0.9412295722818603,
 0.6023540405206224,0.10762682779063026,0.09275402917964681,
 0.38730265876375336,0.10581350681047952,0.599804330993982,0.7543035155546512,
 0.19626369442407687,0.45728395316048964,0.7565508132378381,
 0.004855295919649061,0.15602493485502444,0.015380419558225333,
 0.2792251428207201,0.8992749893726157,0.0012253048802908229,
 0.30065108519732153,0.3599600527505352,0.7322657945678193,0.8149399505152963,
 0.6143713808425565,0.5071118804073453,0.8714879508160746,
 0.009174481496244558,0.5169181996702472,0.641433112009346,
 0.038321824964473894,0.0221102893316798,0.27941027841661603,
 0.5004440569523066,0.24210663893016815,0.09738640804131138,
 0.6761815811936638,0.5090099345143555,0.7814915000714157,0.6265242580431389,
 0.9125022265368206,0.7729825261424317,0.4695787805492071,0.9994334840087569,
 0.669473300684898,0.130155172467323,0.03799229985288666,0.5219884479800828,
 0.9004983373009783,0.07819381841075668,0.4898182002890328,
 0.07064813690033556,0.6133162845473136,0.23365922383774096,
 0.048677303292881224,0.9500858708123802,0.652720447300145,0.5790693525448414,
 0.3553317376784202,0.563875930219951,0.4898450237337728,0.17029613629782303,
 0.773994154157478,0.7609415776995997,0.48505003970904725,0.32045063620261516,
 0.6075091083753542,0.39094112288088045,0.08101417651942833,
 0.4279787859648061,0.8773686636805792,0.04011789078496908,0.5687216828060543,
 0.2068600239041929,0.8615438493095674,0.9619940427580107,0.06039149676162092,
 0.9441544303820579,0.6541459692393434,0.9947577008457386,0.8136675224677805,
 0.8928911458167013,0.8773288261845203,0.3625013266358732,0.7863314859707978,
 0.8635231380973521,0.8014129759214316,0.5383011891914127,0.5399645879053938,
 0.26826003527171327,0.8256033060962189,0.022592109908535662,
 0.10997342272956467,0.35710331115184324,0.5092079491349757,
 0.26719994539665026,0.5303932662012617,0.8000953203060082,0.4429172667347353,
 0.25974292814535205,0.9001965093257694,0.553247545763611,0.06849969624326424,
 0.3029505718583728,0.3537572483464182,0.20140071442114416,0.1710971503087998,
 0.8694170052213204,0.9853578831813486,0.6301398423929645,0.29078586324040334,
 0.6325420526716877,0.19155817312442824,0.9024130503581945,0.6266085429670519,
 0.6075397021141088,0.058048830750246805,0.28500152988323235,
 0.769223875817855,0.25474217463070936,0.21990408761658164,
 0.10407765821759662,0.79867279687312,0.5898077703600072,0.40449283680293047,
 0.10827761299354888,0.7142502584316126,0.370707023652624,0.11432575837968506,
 0.11144385821209857,0.8306915790959141,0.07015682132250123,
 0.7636392260539393,0.29231418721892444,0.5064150988029534,0.916537937370999,
 0.9820511814571438,0.4506493123950124,0.9608045101317856,0.6035203029245485,
 0.42004528241221106,0.25850475567063014,0.7216039339092633,
 0.8111448372096417,0.4755652822226756,0.5037863101134222,0.16436671979859996,
 0.7432783015376266,0.9259338211000806,0.6743495056762832,0.33028738467361185,
 0.007260002911176566,0.5015192755437754,0.10458916253121997,
 0.7367648231409436,0.4857776944895069,0.1885928621546955,0.11302428492968886,
 0.5870906590147709,0.004270167319683216,0.7804294006076211,
 0.8415852831328642,0.62837084588491,0.6778765970147174,0.24854364311984423,
 0.4696312977845759,0.8953842507859116,0.5510405821347729,0.541122849029851,
 0.9752506367401743,0.8767214746937425,0.1287358533140216,0.6711406219883088,
 0.7521780650119207,0.7409319565858831,0.5841179693124625,0.18343010001959725,
 0.486450969020897,0.866961615365971,0.5677835438044034,0.44579653566646316,
 0.6821182899287406,0.23627363819628533,0.23244146675186783,
 0.7055596588630281,0.6521995218381681,0.63181893409034,0.06790253740100616,
 0.658209871608364,0.5988574278366707,0.5583924457202584,0.4313709346204796,
 0.8692765585729625,0.5641213479612864,0.16196061698187136,0.5826961632376103,
 0.6562103214123506,0.025558255477108993,0.31873053304448584,
 0.925596971878808,0.4387603317651926,0.021597308041953367,0.8429572421827524,
 0.3842366232862777,0.27024681914457815,0.66084734510542,0.05434364297847505,
 0.126956448920432,0.8882148617275282,0.1838095252049219,0.8901606017229976,
 0.5638972105199077,0.5815007993394102,0.7458294636568649,0.4637203830196217,
 0.7388180719728136,0.09310330223530483,0.22961624321658414,
 0.9058072504214226,0.5806774952015481,0.5957830163490216,0.3417464481283081,
 0.4536928539297541,0.579306037678894,0.31289681622371446,0.44126529940608183,
 0.9245665627830669,0.8843355305574928,0.9179166787193492,
 0.056703926714771646,0.06208459383507858,0.9098650973239858,
 0.48560028578766323,0.19740070229765883,0.5338837533833705,
 0.5954620241107252,0.18871590018303064,0.7171769016199422,
 0.07874378361259349,0.9100415004903669,0.8953523947705716,0.6836055095827439,
 0.0522489989797168,0.478660508795976,0.3374968066710886,0.4880427887083406,
 0.12226100400121964,0.8790489411080508,0.8767560904645644,
 0.043298057714066696,0.38491852107376945,0.005429376909733885,
 0.8917098084375578,0.4748831208483333,0.2017482317831082,0.12308233446061978,
 0.09261905164236339,0.1383796907054471,0.5438105226383881,0.7746768993892232,
 0.21654677032302616,0.2976891874864571,0.010950007215025148,
 0.9982283387866471,0.2875742918802874,0.9686453866173098,0.3255723380196458,
 0.9583720626828399,0.8963953757614681,0.3740323570897123,0.6491848663439341,
 0.3644460210859546,0.8997378556838351,0.711029113806439,0.6661315942668945,
 0.9618115056887013,0.16037185152679023,0.5414300294769905,0.7420600374223599,
 0.43035782894871455,0.5578266279095812,0.8283169239099581,0.5849281084808693,
 0.39111399295575033,0.5205925816645456,0.25742148873712123,
 0.7100511920290075,0.804625566636053,0.742807305259877,0.15697662026469494,
 0.4977759255452734,0.22859958828040183,0.5384658026993537,0.326807193198822,
 0.8219922368126844,0.7322698413617221,0.584503539210827,0.05298728007697062,
 0.03232525183697188,0.8119767599798283,0.0316657154597888,
 0.43990417584981145,0.2059110798951167,0.15848860698098088,
 0.015386968968832182,0.27751183360086684,0.987389051758567,
 0.6909887702001639,0.6851013712251082,0.4264205409523367,0.26415051664936995,
 0.3663656633234791,0.5090349784581336,0.615615531240911,0.6438461094841862,
 0.4450019367031508,0.2889781113846245,0.37913624147731206,
 0.05379985743042315,0.2740551469110821,0.7107866879731743,
 0.36147836173475956,0.13187101901590326,0.7453972711028573,
 0.3943487436422384,0.7482504103015506,0.9156189809976195,0.7068471829636352,
 0.04375193826696422,0.2690990129255605,0.09928870226464193,
 0.3588615824176098,0.7064507522967521,0.8305264846290116,0.609269609467257,
 0.8170472753408291,0.5488762890867573,0.740667013026189,0.6995194513079296,
 0.8076228312804382,0.08906897271162484,0.328472891925372,0.6035764055778016,
 0.8639300768961233,0.1271866503771214,0.8344168424213366,0.10253712725025743,
 0.34413302818671676,0.6937853351370433,0.7448786689456892,0.2116462963095087,
 0.6070224720086095,0.5660481385534226,0.05520988374000424,0.7407382374067295,
 0.04854407260787286,0.31792040981501857,0.2710548047853868,
 0.25566522537826575,0.5886924604572933,0.9607702604282384,0.7297881980716517,
 0.6830056511416301,0.11876144774033648,0.34381052452054517,
 0.6986945492825445,0.018690224644044462,0.48191345512652417,
 0.805020148226517,0.7735737559715028,0.7150403718659992,0.5542638311213969,
 0.6822277687055,0.3957742193420649,0.5881463188598305,0.3433206041389668,
 0.8646234270005436,0.3812954420067617,0.4461782598324885,0.9859015248380564,
 0.41039669989407723,0.9286897043039037,0.8748970597816018,
 0.22588432254838675,0.6815777274363712,0.8864065252994077,0.6675702957700399,
 0.1955414006547722,0.8442051856565066,0.4288546355341092,0.9440886406112335,
 0.4778372652448981,0.6694191956279009,0.32018225179851934,0.4668096371962329,
 0.20890273384015035,0.743688368605885,0.6903813462758228,0.7441161909168851,
 0.4823695023525715,0.027086807167730154,0.5834666227558689,
 0.35195566137648504,0.7931225800900426,0.802785123816288,0.6108538093847182,
 0.5245819537777007,0.004056650418816532,0.48237427225322815,
 0.499868984871231,0.6711381460839305,0.10818701926777374,0.6803903786742729,
 0.8435865737949514,0.5444264851749693,0.5805922946368854,0.5267122567610459,
 0.5061288141580715,0.5418970386992503,0.21571285088441028,0.6000198393395156,
 0.04674351233346652,0.9137861988068885,0.48844438306302096,
 0.5571450149801183,0.5210782207123319,0.698754918061443,0.4686906366165592,
 0.7719312441119022,0.5777237892127195,0.18786315698193312,0.5149623452671389,
 0.09733214457748451,0.616873820301006,0.2730805679822117,0.2212997341391889,
 0.5973373611220294,0.7841131207543968,0.6579695726707817,0.9693465233153343,
 0.04130715074334712,0.7769991534605057,0.3856076622038246,0.744177326254764,
 0.20341219059857196,0.6667260510843759,0.3651453930067994,
 0.43604860029599535,0.6242724998980131,0.8624930100106745,
 0.09104508136883904,0.3073492593474886,0.6520780399519384,0.6522914838409757,
 0.7289172248891185,0.14818809214578743,0.43464433213643316,
 0.8712417003546984,0.3085500995755529,0.97653014699184,0.2640986909190356,
 0.6936457166631401,0.2955870597903546,0.7090568611687221,0.09190991380666158,
 0.06284689611055905,0.4743682552185572,0.435705822777649,
 0.045934284595282704,0.891218888428343,0.6946966066399077,0.5422910636655172,
 0.29148136755548704,0.6813009754175333,0.1242092930841191,
 0.05961907414912215,0.7806754476852352,0.16225125813082697,
 0.4865143124543656,0.6033109168198986,0.8232870221631204,0.13058375966121283,
 0.8494259386918106,0.3691542466347848,0.5077064225781238,
 0.005999358247588016,0.17429571793300536,0.3632112567417466,
 0.29651870873904784,0.11872826818168059,0.2109906877283001,
 0.03489110467493517,0.13126563888652665,0.7563097692972596,0.966414848305673,
 0.9994175893142097,0.2937296642549543,0.22453856805780803,
 0.06871198772953813,0.7135342091807881,0.013260021688473622,
 0.3697101184464042,0.8405405941975221,0.08965036113936842,0.3383298090035687,
 0.2742182487367897,0.5652955249607967,0.45591586708864573,0.5304828317966992,
 0.8318625229692131,0.8883207435184751,0.18785176383908897,
 0.39121919499354363,0.9901954658495462,0.681220446646672,0.5599808327564135,
 0.39716301860951886,0.48205810968459994,0.9644174233481262,
 0.6845202750886936,0.5524210691539467,0.08029775134489281,
 0.11935383477662276,0.5917312112821667,0.29919880410082633,
 0.7250953697252683,0.3001236887692478,0.0690077115523492,0.33008605504393684,
 0.2559461790705583,0.5622710404386194,0.24360412503121553,0.8971182842897363,
 0.9412778894422715,0.9864994631401146,0.055550825731987996,
 0.7652215238633707,0.15620888794914356,0.9641845604722705,0.3410866756236153,
 0.213575153478761,0.6883795239552801,0.42381678270251977,0.08153041004744221,
 0.2740024665367873,0.3515397603221545,0.8568098634791359,0.787709863144905,
 0.6774360158194426,0.6901778274669909,0.2569783385797386,0.10990196575198186,
 0.841955104749624,0.1319588513580815,0.11782129777623673,0.2603526421881929,
 0.4878818880071144,0.47776595412243716,0.9434575866970208,
 0.48787413773908384,0.3779279326201651,0.7580612531290155,0.943586162637422,
 0.14859547791993166,0.7882092273315051,0.40135830055432176,
 0.02423031321474789,0.9752929762300685,0.8316086760314868,
 0.25162847880973427,0.27836394803784037,0.2740878344525175,
 0.38636957802778416,0.051416649680078086,0.27343657103274477,
 0.7376898133804575,0.6693224566745452,0.9484538748236705,0.12273506853509386,
 0.09188779371964839,0.6128206594520363,0.1346041895119321,0.2702024938307487,
 0.5135712774377984,0.8148707806674069,0.3747779991035386,0.07600409435498612,
 0.4885260446179124,0.34178616741440393,0.32889684048684376,
 0.01778318942156043,0.936227980859899,0.4443929753136969,0.9922201611352759,
 0.9198556364788483,0.8736128187608233,0.738262499414021,0.7972334624349282,
 0.5102723626638096,0.01910832668514073,0.36519273248713013,
 0.7108145758027681,0.7149022573346847,0.3141424583626319,0.2709267219662066,
 0.1723560021598145,0.12241734439922813,0.808464156478337,0.09418296652181646,
 0.2529435070875613,0.3694444109226317,0.3041655641929767,0.5320000938122676,
 0.6463958404184922,0.4368359654834525,0.7522495412891167,0.7385324515339152,
 0.39442471094707965,0.39195231657611485,0.46593259775110385,
 0.8512913077152093,0.7234248988877289,0.8533892637004521,0.4638653703092822,
 0.8432886612611119,0.44625692215828494,0.18927004084076937,
 0.9454043642576775,0.6795857560509236,0.03392625715838493,
 0.44626401756850587,0.41777050304137464,0.15938794179887728,0.13236309008543,
 0.08365104676714841,0.5564780514922112,0.19676435292181071,
 0.2705440175392475,0.1421398720268392,0.4306776851129904,0.9643129735352254,
 0.3087951920954025,0.5847269828067907,0.7892864117388997,0.7547467038962169,
 0.5078232870251599,0.5142729457671428,0.6967242672100423,0.5045714445854705,
 0.42183860319597366,0.15320971648138082,0.3452549581836426,
 0.38623888699075204,0.17510863272511745,0.45932657712319946,
 0.14670011120105453,0.12896425489142838,0.8307160797144512,
 0.6668424261570197,0.9624375130447731,0.24849903557818076,
 0.07744992513997173,0.23713989354849008,0.9536240681903627,
 0.20258679481238717,0.9985064393231741,0.6471815983654097,0.1718053584292898,
 0.43288025816170983,0.47937547298152217,0.7719394829186939,
 0.5145060205499252,0.22245334184017906,0.6831687580825865,0.4760453756694589,
 0.27971701015611905,0.7490023671203684,0.22049870569393692,
 0.7976227032312391,0.33458721447115736,0.09751390743832211,
 0.20407667742807523,0.0753021164429406,0.8047654009359027,
 0.31446998183226693,0.9602414280778208,0.02859036141354887,
 0.5436571871525812,0.6948812257405577,0.23306939946550298,
 0.20750385906981772,0.9907074857923597,0.33676389611574864,
 0.4728054892913778,0.5080020432984441,0.5267860951355567,0.6328834407003906,
 0.6344911922775505,0.035346194656948327,0.47569306733034844,
 0.9303520250627247,0.49111449201514956,0.7954631943251987,0.5837479620380364,
 0.47147564924587515,0.2282263106345792,0.26953274992084386,
 0.8664395291256032,0.08348249599420998,0.9561050216525633,0.5478444527736218,
 0.040101372643863264,0.6312285048731067,0.725611361173568,0.6603673373144284,
 0.15841792172319402,0.11743567611124472,0.02705237987261533,
 0.9270342203663057,0.4864421539299604,0.5492442987821253,0.7402014509157981,
 0.28354480180012764,0.9136711572888965,0.48878570664888177,
 0.30243106820094035,0.06350955997259711,0.9618031936477767,
 0.01406127451852257,0.9423064616242354,0.6588286259466454,
 0.09766062320420521,0.5339701124094015,0.5496487570119158,0.7223359344222408,
 0.2794498041214061,0.9249680985017548,0.36568936992637424,0.8142545688514671,
 0.9760756702250106,0.7756770411763242,0.4004998157393026,0.38610304269908113,
 0.37621962056635927,0.40478006497677543,0.6421063629505686,
 0.8185795670511243,0.1438638298616599,0.7510441856777198,0.7010335654061883,
 0.43617549800771127,0.15918565608130653,0.1269147155543987,
 0.47424006540026764,0.6081790138986292,0.6555794517541575,
 0.26026476101973295,0.4523680634815499,0.7469948596670712,
 0.15473876435199896,0.9967514555615758,0.406556656570777,0.9768186611243765,
 0.045219750878440435,0.507632109362364,0.12315904987317228,
 0.9373343615623546,0.3800314058421048,0.4544035235683629,0.7472660729113723,
 0.5938728799501736,0.5974154428456007,0.7730479597423631,0.15299508528435923,
 0.692029982406691,0.7129662824042082,0.1760069358596037,0.11775188741170528,
 0.5407158927822034,0.07634558197802843,0.009149666891074548,
 0.801696781337309,0.469569320264246,0.5019195782604141,0.5452652755566012,
 0.08462715515487995,0.8365974702768378,0.8524335233772242,
 0.13001051337135838,0.6198072363042378,0.4988049337803826,0.3976922413705599,
 0.06396793610628859,0.7641887419289605,0.7330117252927446,0.8645672438800355,
 0.44064948707586016,0.05751283874964619,0.7213818575741199,
 0.4895713418980573,0.41886298799303023,0.08638442785967682,0.473455772558524,
 0.7348265769439554,0.5633259400305707,0.1260616925528425,0.20107050790794412,
 0.7328084765925975,0.9231632458700713,0.47109652922229983,
 0.08247234776694978,0.2536868128590133,0.5150351346673798,
 0.009764822848770338,0.5861466248603237,0.5157188544895207,0.550969691181077,
 0.7043455440714158,0.19161729781262837,0.4479780040431064,0.9045349134253526,
 0.8190669857420585,0.5082099975695539,0.10493300239749503,0.8833415183221501,
 0.5451872252368499,0.2602306793026503,0.32711448317976066,0.8254712967163772,
 0.7086501700222322,0.962976083823283,0.8475535770854922,0.98646403858556,
 0.918229572080742,0.8937714309322387,0.3178069986717684,0.8231002870677608,
 0.07142316178188723,0.7350345390667862,0.1775814985619103,0.9501482600636458,
 0.3315877316621775,0.9978739470015063,0.775481987604007,0.9069304639320231,
 0.607249958685459,0.04283885997236214,0.1541058388587324,0.4105008363097349,
 0.28524121309422235,0.8125381206167057,0.028759706644540972,
 0.1453507405274984,0.302907394616592,0.075215581255348,0.3607731027887603,
 0.36791374613626593,0.5871465461487821,0.8546315757320123,0.64591687961434,
 0.3375726425707055,0.270220791689076,0.8658932121609753,0.047141012121265646,
 0.4533632316315489,0.30454658796847645,0.23007046248094176,
 0.5962644600751318,0.04654500491556879,0.3795707568830078,0.7476414982159388,
 0.3374670058511432,0.25587543685627345,0.9509338608566775,0.5169275360628952,
 0.3359120232763253,0.1658623969396541,0.46912314826876267,0.4194151668135796,
 0.2358612547696849,0.9447920361480916,0.970307479021195,0.09833880207888068,
 0.9437025084224462,0.5786067236020895,0.3146395190930913,0.277443571546481,
 0.7904038766425098,0.11848082491591994,0.23746850922405383,
 0.2865659953294263,0.2389866320227031,0.951439950940493,0.6531096155344679,
 0.3738151051232955,0.11673225344959237,0.2215337560862709,0.5802902849635082,
 0.14343577074108715,0.12714670370379333,0.04650280282580144,
 0.06626673477628142,0.9964436753611082,0.029256927321268722,
 0.7094036625026081,0.8650668725348032,0.37547597197369686,0.6779118676043662,
 0.6098604190548208,0.6329531291504472,0.9326853156043068,0.6258042260714337,
 0.2662617765107519,0.06240727288847947,0.07718070003818411,
 0.09446073742785488,0.9811402889806188,0.9056659573935527,
 0.011645947779989618,0.6451725552321713,0.9969417106042665,
 0.3562819903548353,0.2029816173615262,0.6706316029789632,0.8153228415913366,
 0.8960355486077699,0.6880445673666418,0.7432629720711357,0.9316509892396526,
 0.40045515498095563,0.4055625346677991,0.8684019532026883,0.2535902716047067,
 0.6960493057904819,0.006082371928010422,0.08967805222535463,
 0.9538728143557689,0.2332151284729358,0.015958646237634322,
 0.43814824379930595,0.955884022918179,0.4959398604514256,0.4347528842911801,
 0.8752137809730309,0.427291922372667,0.1251291167471975,0.13257932187773913,
 0.22912514647859838,0.6883122230945372,0.865203382973745,0.06244239569091392,
 0.8747786752794383,0.23697586152038763,0.27370858037572765,
 0.9132182801140692,0.8403591437003162,0.48713438071618864,0.9952852101922997,
 0.049670900649592675,0.44758070106509684,0.132168478358896,
 0.035416058416612994,0.8209856898139611,0.4381579667835993,
 0.5642785201163666,0.7027930155672812,0.5274215219330483,0.19433356614241826,
 0.830109278679461,0.33607575660547173,0.4709215737228214,0.8358011353505564,
 0.8155132266527756,0.5933221105566978,0.9536080725406373,0.45132661165127197,
 0.20714803924397485,0.9791162820573781,0.19891820005700434,
 0.04821048700345976,0.08821220994018764,0.7508382841425932,
 0.3714652764968205,0.9641234361661256,0.7445195087877456,0.5198784925195141,
 0.07343272526636768,0.18077519512075457,0.6075295009232764,
 0.10755488415524317,0.9300988199462057,0.7508234252476957,0.9613161045760961,
 0.9194811229658333,0.10543113930549397,0.45807393597469903,
 0.044456811911592675,0.3885106221494111,0.2755389898612032,
 0.2806216011933128,0.6794192774404127,0.05334858677536769,0.5343545260869945,
 0.11461714465808082,0.09009815229936202,0.11525881747328393,
 0.19119280589042398,0.7501085044941366,0.8599283768755985,
 0.12961401151604046,0.9651166628726793,0.7267567672787532,0.460552201258028,
 0.8832485311985279,0.4890994306037628,0.861910660074603,0.8253545039996627,
 0.7249963774245843,0.6579809735530738,0.6186347621567894,0.8659336422918347,
 0.2057969990003341,0.4843663813092931,0.4271927111154121,0.57647717409235,
 0.046947336817456664,0.49674394328889604,0.8383825002228144,
 0.16543828506504743,0.5503414730465233,0.4097219085461027,0.7671147403715145,
 0.5216710922871017,0.8876693379548448,0.9555113938429465,0.8588687629211037,
 0.4746366215713623,0.7388145138131978,0.6487442743991619,0.6402127625180389,
 0.9658643252934798,0.1903630272074407,0.4038833220438386,0.7592187600374539,
 0.6445788306431979,0.5006907981893788,0.8838047868065031,0.3431131706149366,
 0.7070443162181641,0.13649230823097813,0.5541840433633611,
 0.05054001275501596,0.06700303471163715,0.9200159207765168,
 0.7603931876140884,0.24182903584039284,0.02501083335668819,
 0.9649227425757961,0.8584539630601997,0.8860467221194649,0.3919699811168781,
 0.008495434529594337,0.4695531841037589,0.9252900669441455,
 0.11926960529955144,0.6316825410194812,0.052182753444684704,
 0.8641606070090715,0.6847017729709919,0.26359466704159773,0.7951799304963529,
 0.4988567991960652,0.30845004485125893,0.6290493280558545,0.6316070182147138,
 0.5639738309692298,0.9151891338396786,0.03371929887271863,0.697766597283283,
 0.7133930104549266,0.12645183967214657,0.4225805029274897,
 0.42572108247526763,0.30122313296873837,0.252283631752661,0.615872544363663,
 0.17628924651764133,0.24171088824616727,0.5588021627217544,
 0.2872681017823553,0.359253784916003,0.9095528910268282,0.04649797073884332,
 0.4184574561159715,0.3065999574865653,0.8068048468905941,0.44546007769300555,
 0.5939898377258699,0.3410872007760112,0.25310482917676536,
 0.28358193171394697,0.16030514195703038,0.36964232151583976,
 0.26579196978793984,0.9210671411347731,0.18041625362413938,
 0.06782925080169189,0.8602578485241559,0.8495059521589212,0.5902467799195754,
 0.878307177941736,0.9002576436706644,0.801954842850997,0.0581266669371483,
 0.6046220668055243,0.8052078997734236,0.6628552914980006,0.5241591333062555,
 0.5234471770308161,0.044937571224658024,0.13095841239442807,
 0.1606450326275315,0.3007828787629635,0.32887781905196745,0.3911318919539609,
 0.9868479639450329,0.07850688508684822,0.882980725565515,0.871128277484144,
 0.9038062840360594,0.6521282400891613,0.045672451356715715,
 0.15010662906141092,0.2142347214146194,0.36718472475188446,
 0.7792994032119004,0.39991127746396227,0.7123823495142079,0.426913165270402,
 0.2698761098804704,0.08202329516163198,0.7274516108073382,0.8948003095361267,
 0.9072425844994022,0.34565103364402816,0.012209787526932026,
 0.9209771344677188,0.11353808760242601,0.2357597992806365,0.8007921135770523,
 0.9327423588268163,0.6500637624673696,0.03390480069642765,0.6541867667090212,
 0.7169803146060719,0.6576140747225078,0.3695273255116732,0.7350344294049546,
 0.12479956408207504,0.1665439160253981,0.7860645328301635,0.3533070353958917,
 0.8857719520523879,0.09603795792713066,0.7305851572878155,0.895006056943845,
 0.8354340092397461,0.11302560453524702,0.09872998827610002,
 0.7062015736325965,0.39993301330296616,0.4452063676853356,0.8727906616845497,
 0.8599450696160543,0.9789369506053347,0.7921657651388722,0.275197949388878,
 0.11693884495390838,0.8584815564506857,0.6383320123872849,
 0.24523000719105004,0.5942257875892878,0.3632945281939467,0.3863845317483101,
 0.871969027045431,0.022103391170562148,0.33457895509930946,
 0.08794083741628822,0.41258907903975106,0.9899601793622173,
 0.6798321896220286,0.8381357351970684,0.3429208947014867,
 0.032472693322901924,0.9095411029481584,0.8048802310938518,
 0.014248569241117837,0.7899342910601066,0.682812018131232,
 0.30502450548274074,0.43807998517293734,0.5675871730606122,
 0.1856772692293429,0.7391043645149258,0.36282447777818305,0.8388400239973426,
 0.880614130310921,0.8094390863315299,0.9081264722280716,0.7924842575700745,
 0.33458661847585536,0.01749422899406783,0.5537468215608006,
 0.7213191331981106,0.8083866939479036,0.7911658813480714,0.20667426217200013,
 0.4230975231251418,0.8559032323184719,0.1527773881481308,0.15332056653172865,
 0.20453895331052263,0.06918423664422924,0.4354947400028568,
 0.9466004294285352,0.14599777597742503,0.7765459355908666,
 0.058475836957327165,0.9905447756216602,0.13780083800885834,
 0.8978432733000528,0.11072132880641694,0.2063148660409978,0.3513623980810954,
 0.21467413300346638,0.2790537996146626,0.33151277161440096,
 0.5440257638231599,0.15260678042901965,0.20252843501631346,
 0.5650368746898194,0.6097756838491442,0.0035652513349332393,
 0.7575630753186636,0.04061324340229899,0.392003957830211,0.09778195330670802,
 0.055527561894559985,0.37169305519536466,0.48543138305525624,
 0.34946806940409136,0.22714885978157862,0.1431181617641184,
 0.3064162975755882,0.32644024909760583,0.5561276510416986,
 0.41356737446117564,0.05967453503653497,0.9583557677770406,
 0.6570705907999187,0.6565729172294483,0.741332119989379,0.2899871580619868,
 0.24256493591432782,0.9478104717486098,0.736213067518297,0.12777235026206935,
 0.9233593430507241,0.4536480098810659,0.534092786048653,0.6172442855622274,
 0.9326202351702688,0.36121509549287856,0.9858943008883735,
 0.32984892895131446,0.6599313193265748,0.01925253557524087,
 0.13978084674432556,0.41858284487797304,0.34631679139487137,
 0.9618976302886245,0.5544460099837512,0.320036032128417,0.2702440727120361,
 0.5029955213169034,0.8930421276518652,0.40632293147646603,
 2.802940660964204e-4,0.3028488085681703,0.5200559144408958,
 0.48542015915924674,0.3312780413907047,0.2895294228967036,0.3242521638497504,
 0.972788711095606,0.26893995456173003,0.28286813689055224,0.5778128200279744,
 0.2714466156027413,0.40550474325557073,0.661125344680287,0.9476287123818781,
 0.860999694441533,0.23141841814612507,0.34201227691118774,0.7352081444432809,
 0.5762594378334981,0.6106502258212639,0.35995370086007317,0.3927408495359985,
 0.20832849280538912,0.16885117343523415,0.18382313862326394,
 0.9944952883526976,0.8797479037190687,0.8690025910011252,0.18103775252009136,
 0.7815839466805081,0.02718418846786652,0.8489225137490358,0.6467381373515826,
 0.4469093656971309,0.73643341098946,0.486300733589483,0.22485348601851807,
 0.2073185928513327,0.4795150738597136,0.8392452568250344,0.252266669050086,
 0.9770975486422893,0.04833388589554699,0.44463382320553935,
 0.7872911813562011,0.30733620763227665,0.8684065553083232,0.7094690316031743,
 0.8320365674422171,0.5940627728343368,0.2037151756824669,0.4909334019512065,
 0.21809866564572422,0.28985927061270667,0.5794891411262487,
 0.30892066622521985,0.45767114172061507,0.9566194352579319,
 0.4644222537436563,0.8112703652833365,0.12510588976227732,0.3201122183469517,
 0.01933091063524195,0.31016039448877997,0.12136078252360538,
 0.7034451410508575,0.6156403120772835,0.3784622382656072,0.11630750608680829,
 0.25058317335387825,0.26713911582392935,0.04681826657944277,
 0.8977331384715856,0.599075292308946,0.208086431038773,0.09345589128144449,
 0.6365875315950377,0.34301870364072273,0.05541522564863233,
 0.03595319543524844,0.9889620784112333,0.5957234752931495,0.6528252302714634,
 0.8844510166223606,0.615406675855484,0.38273801206685076,0.11571656498512373,
 0.44974487690987763,0.8203512520530325,0.15116348601050322,
 0.9863160981596337,0.25913680415677753,0.1641316329846626,
 0.39573639585087805,0.7458107409031675,0.7522175295930965,
 0.41338898152186765,0.46921356625437016,0.7471982522982411,
 0.9178605698092182,0.6412442823875069,0.47826021668386964,
 0.38958227367715015,0.07248858864874907,0.04290832133849454,
 0.5540785373576017,0.4083566767301181,0.5506649896459066,0.11111095149748529,
 0.6501188469202702,0.3247664407935845,0.9267240317802196,0.9273051936318897,
 0.47582488762767783,0.4361061787892728,0.08527918386782718,
 0.43043990205713023,0.059865192366114095,0.7832320888460991,
 0.036644404532788055,0.6243007898675793,0.6954122069070581,
 0.28956514934295285,0.2654898561780916,0.7719276404402557,0.2642525675677774,
 0.902034424320633,0.9211214151717029,0.19577984226798772,0.3321221817689284,
 0.1236316706151499,0.8503589165673677,0.5558791244785948,0.9173192493973878,
 0.8478231467631824,0.5077006693873181,0.2456936229321376,0.3233226224586181,
 0.24404880780096094,0.7091731498587294,0.36249527630942724,
 0.26956736461817965,0.3046302451264571,0.3347912114097934,0.1616690353276613,
 0.8861165978109905,0.1853122296698292,0.19069659270851302,0.6274506933520073,
 0.29125458171694807,0.37808911982535454,0.8054884421667751,0.754943247446694,
 0.731753337047056,0.925410655141291,0.08429034664031498,0.05948334244399789,
 0.6559245988266924,0.5487232662070373,0.35416790880744053,0.2547067976603429,
 0.015627385647621228,0.5316937499615607,0.9766441758672354,
 0.6947868334710228,0.21613906574173203,0.6482298367528178,0.7522822524786378,
 0.1684857896797063,0.7411472320348568,0.6492952230475685,0.5331933131438178,
 0.3706952548387229,0.49673137284984237,0.20803390670259736,
 0.7573000795252038,0.6018311875491034,0.39387897663790195,0.672307116080348,
 0.8128086825141413,0.8391062385267533,0.8107691233844354,0.8275500278092837,
 0.9067485188195258,0.4332012609716108,0.04201094707367514,
 0.08684060290465967,0.8613424445301202,0.33533866013102687,
 0.30147665462201345,0.7535717062634,0.17809183867769696,0.9028016652301846,
 0.6897665664647921,0.2842660846910131,0.8508488656313207,0.39418999196394533,
 0.5738445948465208,0.6358621688766286,0.7925221836215128,0.23153176068278158,
 0.3375300153608851,0.04532204336023726,0.07884255219383185,
 0.6033569164588481,0.024195626383117674,0.4626287865919456,
 0.017619202208366103,0.40596941405733555,0.3053681627836684,
 0.18378603426980722,0.13442197242979104,0.17160115359502848,
 0.7186724738607698,0.3146964037837763,0.22378033000236486,
 0.29516303456669446,0.09896640212895225,0.6584998031545407,
 0.06450919370884656,0.7264184454631469,0.9460293344848798,0.5262292330151039,
 0.15842184082670996,0.34783886328967817,0.08354533381121854,
 0.3488135823300954,0.2056709649588433,0.07238349437348812,0.7268546043426517,
 0.26206096516034716,0.758702034447263,0.47606203814364634,
 0.12633760588905463,0.7368438533807014,0.3822960353195186,0.795497673264474,
 0.4368467724178473,0.7947116834344444,0.6205001080507924,0.9843474738810759,
 0.9454752062580288,0.6581082817498585,0.6197484075288344,0.5158347980675848,
 0.32675717993511355,0.3320408551695567,0.4883216662345261,0.1398837821547575,
 0.07208760618898713,0.5779197879632928,0.4025807145829541,0.4989478359495959,
 0.32092826945688857,0.3339906026356936,0.15604217689897204,
 0.18031957062440218,0.43983515170446086,0.6656228048516866,
 0.8203082046975517,0.8063169800427127,0.0431414702851981,0.761046907068227,
 0.6958307681675066,0.541167142710056,0.20751597473288974,0.719410298372198,
 0.749067490194349,0.4211178343760652,0.35404857684779667,0.5792063373715275,
 0.6578046706832841,0.14949259574156692,0.37069602009290525,
 0.9626255642657268,0.42443933007961343,0.5859611372493527,0.4005049827823881,
 0.5711644422202922,0.7057075681181105,0.14705681630959,0.6800583392571764,
 0.4664370379545638,0.8472202129160891,0.9722828416934274,0.8219480983431873,
 0.7005498758539963,0.662981698164767,0.7066874230150981,0.4167345299217047,
 0.7673537653488403,0.906037697156319,0.7752121784062047,0.29160004104086823,
 0.8472642284297195,0.2913418490184677,0.35607289648050033,
 0.29229959050219156,0.09916925492354434,0.06588761762759388,
 0.4202318892308934,0.5138818557391062,0.5966769884837583,0.9493136379703069,
 0.5814375343474674,0.7751994623455257,0.4998388329332373,0.10480606117126168,
 0.30749673215370044,0.625458843997241,0.8038793388137921,0.4028223580709298,
 0.25687912609270114,0.6289934845096835,0.02344340488005414,
 0.6955869347345233,0.7623934474117593,0.4630766497064861,0.4059364394144781,
 0.6634381549430831,0.8262387387613115,0.09511342799683087,0.5413013967883221,
 0.45136008507293335,0.7156567431249579,0.8041948442821432,
 0.029215804843360837,0.7780952107770119,0.34716570992246165,
 0.9813408816324984,0.7262972960581898,0.25577369807758044,0.5920198121446744,
 0.029481380534838064,0.24116223678152127,0.469800951945048,
 0.09408298812887306,0.7575783728373612,0.1258301120310823,0.5595125293816329,
 0.3558375136673232,0.42043266753970465,0.7155681501545073,
 0.19246457823841534,0.06024037495249268,0.22041078636701106,
 0.10237652190870605,0.6295030541603829,0.2568906959861811,0.2768663400688266,
 0.8950839003280259,0.026937753392332064,0.6806425436895241,
 0.41567863717413067,0.7563950166898405,0.8370608228253505,0.5919503338980703,
 0.5705800870310049,0.5144677490817899,0.8385365718003605,0.848451726693686,
 0.8032871776016909,0.3346626952930394,0.4878623798219628,
 0.044342954273894675,0.32438775822997057,0.6851125213724925,
 0.46413897365618484,0.015125806727451674,0.9514984806141262,
 0.443391528090349,0.5636553792899872,0.7182422269836821,0.935035019793336,
 0.6246858310716258,0.49304098400603524,0.4833920461982668,0.5868760085955977,
 0.161432652013346,0.4659932884292777,0.6462707729008637,0.5639934434865763,
 0.40590904938838923,0.8026926522374435,0.6867654821904059,0.6266262215088444,
 0.09743350437658183,0.07356328271676882,0.6493923677500126,
 0.5931901222800987,0.379205650376805,0.16001036978208782,0.4154205952006552,
 0.38437267283211696,0.9260948733755012,0.8539594579916683,
 0.07324649814464967,0.5690663915522404,0.45875236477549264,
 0.5710243034113553,0.40747924809270664,0.9792867054506504,0.8770330863576606,
 0.42468447626901296,0.9549875224397756,0.20742759909620612,
 0.6763724785746082,0.7132070077300612,0.6032479460357173,0.6703896390036159,
 0.9780279055125425,0.7570746016426988,0.3208474595278853,0.31953355682737494,
 0.7257474648374194,0.7143051202263199,0.31530559588685503,0.383400584394363,
 0.8397340303801453,0.39146259229854186,0.6275595981638421,0.748081786281375,
 0.6705895122511186,0.09568077641601558,0.355596910650549,0.20495499783272664,
 0.2755762698699997,0.5185094354400123,0.7800162843550904,0.4337992191298795,
 0.7200620800379445,0.41426697775509835,0.1054888457636981,0.6753909850378869,
 0.3763145099234847,0.9260542330746864,0.3069612559674183,0.37203626127733536,
 0.5350310887498702,0.9234104213771972,0.7957225610038471,0.5813768622509078,
 0.8902371943625182,0.3377554583278909,0.5156346082124912,0.5456785558252488,
 0.029119924359182092,0.3508524639368157,0.21487824393120525,
 0.9934201059828298,0.5313713051488895,0.8916673220550427,0.8305186938776922,
 0.35781816343298634,0.2360930935807828,0.02235372802155655,0.73752919350895,
 0.5867595047127319,0.2622991255067755,0.8782167368743298,0.26306861976742557,
 0.7784695618796117,0.9539014288192986,0.715139112834218,0.4128602307376298,
 0.18830991339744407,0.9595629306085252,0.251721960249893,
 0.043966918268286914,0.966065825847747,0.22534694458869875,
 0.09070755922490548,0.1829994958775386,0.1640987948883761,0.8037598301429086,
 0.3339428448939421,0.0908418892827264,0.7092367995057677,0.34387540222796886,
 0.5540116203597071,0.35856154361611214,0.7321779779644433,0.4674464643108516,
 0.8718932321366235,0.4656930560573531,0.2277428635194363,0.15200122672887462,
 0.054178459125273415,0.11898179060040548,0.7029666991725718,
 0.2341304221406202,0.6196417462825726,0.9907036424255841,0.8598079394297016,
 0.9620140117800619,0.8090730667956386,0.19162320867390037,0.2679421790721188,
 0.043335871152565764,0.40291539236012497,0.6142588638111144,
 0.9998967504141382,0.8903491054546521,0.24505631450769993,0.4144619965750217,
 0.9361307809862309,0.49439353849580103,0.039699972739730915,
 0.8249037322955598,0.518292360340366,0.847106358366985,0.23833104494575497,
 0.016493195152240014,0.5640617504273997,0.5296366067159259,
 0.48454682504403257,0.2376058462457885,0.06938441544817553,
 0.15187774284432687,0.631583391247851,0.43800214433838325,
 0.027331045544289112,0.8835869320917167,0.697574227194362,0.6793262261570978,
 0.11878256623642991,0.616536581578693,0.5322698443714065,0.2882522549780395,
 0.5793654796725257,0.09417556193786558,0.709408445322906,0.7712875522122703,
 0.06841618927287463,0.06120064611472942,0.31689619488640064,
 0.5492537686229777,0.4670447766907778,0.42655522834328263,
 0.07567924886901733,0.2990777746060964,0.891247928994673,0.2138488286562069,
 0.5547306415739798,0.09820164533731945,0.25105122368675625,
 0.2137600529658178,0.1043954904160459,0.6134529122640516,0.4807529660186195,
 0.35885447409486815,0.4115549439009206,0.6256739812147499,0.4908196154868705,
 0.7968156337403967,0.45312258915272496,0.2233664192058189,0.8144867003245267,
 0.2604978687557177,0.426780511936924,0.4597443773804315,0.38433757333595775,
 0.8825068663349822,0.4629466674457323,0.6702993702422707,0.6510675677286588,
 0.6401539327749916,0.8168188100081594,0.022979505459310268,0.968465720334476,
 0.34626536195925894,0.37510324929746097,0.5968433911939638,
 0.6681847423785073,0.023384548858213505,0.5417144666285648,
 0.8604691678432401,0.7213102551305707,0.9192606414105723,0.8756914255011543,
 0.8321626434286538,0.8074012675318719,0.19511695387812034,0.9723913706309439,
 0.6951569319904839,0.4831504091368124,0.061398932132876394,
 0.28610635830806763,0.2631707112488153,0.043067664322680166,
 0.46075584198233677,0.689247540025482,0.05361266092824335,0.7700398209554141,
 0.1898368474176496,0.8863176949729045,0.16749297323223356,0.1755770959254208,
 0.7960999434203513,0.6583908688019742,0.2437298943677071,0.5325274554844783,
 0.37656275079139423,0.972942198742801,0.13199059645107092,0.6670758835734915,
 0.5539924171006074,0.9471528179275182,0.8869617888127554,0.8827112069912705,
 0.24384576170700512,0.803444220330451,0.5595367582281515,0.10337557439008349,
 0.6397093302732106,0.5542351784671743,0.7886060973905211,0.6578072182444431,
 0.41140596734853785,0.5400049681070814,0.6300540818085472,0.2608405347056567,
 0.30507634900466085,0.4079637347769457,0.1850360247615518,0.7260658194066497,
 0.7127495220239312,0.5178369635827432,0.2352768340836222,0.7618977476641915,
 0.3781600155188666,0.5399057259195202,0.7227886863890568,0.49481240847929564,
 0.20694547981134825,0.17519018403863362,0.7728825891716996,
 0.9146360578302406,0.3621357592376233,0.4438812453849217,0.9046290129689089,
 0.9905083295901423,0.6512084528652423,0.10336139574588088,0.6667686836007741,
 0.7690547911510878,0.01788670667859127,0.9609664466347032,0.8987384971696761,
 0.4600031625439288,0.5190863483694056,0.9885783008936796,0.4410886709880445,
 0.4872483720508701,0.3572781610072655,0.29022774766671533,0.5280706553983073,
 0.05603902303836317,0.22419021397752847,0.5023147313509824,
 0.8788408189349276,0.049749618985930666,0.3856659475209132,
 0.39076853619739893,0.4302974427701608,0.6452118176423459,0.3749022623792193,
 0.006621790481219358,0.9344162812943081,0.3255651971517636,
 0.08388784942004224,0.4470989654549977,0.6964964715951321,0.6375596722225617,
 0.6449148129364403,0.3817832541472044,0.4968215534099517,0.4627924284542052,
 0.3200326299863797,0.7582670393809694,0.07643061667422524,0.4426155314821806,
 0.6466689784315391,0.09986772273451705,0.2570342220185873,0.7172762184298678,
 0.08961367844982426,0.7336988755266818,0.6991474459302616,0.1082583840348984,
 0.48992519604548,0.9542865836943897,0.4849714581555564,0.8371728222196182,
 0.359848632423754,0.024508357797213498,0.8507827971123288,
 0.27466351055590565,0.1365374286369232,0.7319293942680493,0.1506097830112848,
 0.6410872544471119,0.016589853494293094,0.9882165315205783,
 0.3602256660657496,0.8817106341944653,0.7019239602140105,0.13353785756161152,
 0.7228993508642759,0.9290000563874485,0.3074374577582444,0.08621212630383546,
 0.40130714444670146,0.9426490201259716,0.7269445706993944,0.237254591309642,
 0.48000857946698894,0.8656520786963287,0.2578456911291117,0.4041418021057708,
 0.15486200157381436,0.1599203304037522,0.05618172083415729,
 0.051655706923146694,0.49699188330903943,0.9431711747023015,
 0.1343739110936476,0.04977095032588963,0.3687452687462863,0.3705938755700723,
 0.9533086771519654,0.4127113649906081,0.3475333240113756,0.5556240330659743,
 0.2378078257515689,0.4812286272557821,0.27726798415928267,0.3205050094857792,
 0.3966111352723538,0.2653003253611367,0.281173811403045,0.08730849258718609,
 0.8878311956241207,0.3122877437298719,0.5134625702957974,0.40037703143096415,
 0.20406919584568606,0.3298366149031158,0.8908393822223902,0.571363009358004,
 0.6882013586656387,0.023231795630995866,0.7858866543689186,
 0.5394598475180883,0.8738072253900222,0.01642313970712861,0.5913532838679787,
 0.7080986619636551,0.18968515381141737,0.05152029413654913,
 0.21169092258161648,0.19401262912591344,0.8251710956696269,
 0.3138014263882918,0.28061508132465807,0.834103771854199,
 0.030317872618924713,0.04373728124645371,0.37077085548762323,
 0.01864296325877557,0.25519691813417633,0.7637743284383401,
 0.6190593098354028,0.6931630811768971,0.06919309086388326,0.8952402468269347,
 0.10784285616258127,0.29792487848711047,0.9005518557989647,
 0.4543555787338097,0.321890476741601,0.557710132572139,0.1830178882217236,
 0.8088064794159406,0.4125745288298972,0.26435553551477875,0.2384020771994554,
 0.9044940529356147,0.25246261267341363,0.8955196869451689,
 0.07496794041802479,0.8949039319926583,0.6543825097560232,
 0.28250355089250845,0.7115769034899313,0.7751978535524544,0.5560325966609123,
 0.671566256428626,0.08021090449660118,0.9800497454782783,0.7339767229382508,
 0.8997310347185355,0.6826689294135946,0.8513853522172793,0.8163516403642886,
 0.7681191958432019,0.4656268034790304,0.5237833027607527,0.04058584926088804,
 0.4129158492188383,0.5257316574046219,0.2523988948481377,0.5967662726394714,
 0.3909747621328399,0.2374579986398554,0.8774372270326347,0.872503619208394,
 0.16650549090780065,0.11395437110895956,0.18201930549199374,
 0.7548385845791414,0.8410995941349174,0.8310243925425108,0.41568610912076087,
 0.9733859726205459,0.09899858527564032,0.5302980068166053,0.7131460705237137,
 0.7482709713268789,0.3597028939936411,0.9761511730385136,0.07974056133181284,
 0.2491143166379206,0.4418954258955645,0.12279801260377732,0.8482547902230421,
 0.12772689126101433,0.49681781984702633,0.22873158002260296,
 0.22982447772093362,0.44944719667939936,0.7352495139250481,
 0.3331127778564281,0.7060829997300347,0.1651429584744385,0.28910359525828255,
 0.7163406668484573,0.9441205156101808,0.9379626884003377,0.5959251777457849,
 0.3535084131095859,0.36379519468314103,0.5390494057761978,
 0.012544437055972302,0.9460742949521037,0.2781283719392521,
 0.4868016038176748,0.9399758495619014,0.34445033219831955,0.983880803739505,
 0.27110000971427417,0.5458030365591532,0.9378360925584234,
 0.25433515869133505,0.4142279725231657,0.014792851768883453,
 0.5522747503727523,0.11969998115266933,0.7743758187244426,0.5079012315666516,
 0.36982678509869393,0.10967021674045518,0.05565780556882394,
 0.15046159353683886,0.6794235030973973,0.7090294864989073,
 0.16577460312728654,0.7203880578966761,0.10650790755717388,0.489340978021676,
 0.7745807776803078,0.3638880252799077,0.19241948735940462,
 0.08723407335709144,0.22569263635846704,0.7675861600158838,
 0.3494714014649276,0.6968499026426201,0.48816935530884864,0.8494707142926173,
 0.14882096980955806,0.5999649299528675,0.21230118604025572,
 0.26912358749194576,0.42962390890440205,0.8063938290536857,
 0.7736746054089341,0.46916478132964046,0.9762653280085201,0.3845928621061002,
 0.7811582647173133,0.758465747488639,0.4881842732635546,0.9391560885222364,
 0.9334976508787887,0.3940251035867124,0.6613539730460151,0.46495058155684066,
 0.746071536513624,0.6466220438896662,0.609959720488721,0.30365820586780135,
 0.639737160877026,0.41569816317639974,0.36900255851529395,0.8676197304366163,
 0.7702401787310797,0.12894743907262352,0.43725334632697543,
 0.7894308647622582,0.2803785441318001,0.9794401542459091,0.6635810591859381,
 0.3279487631293917,0.8704518833772295,0.2442948339783334,0.32827846454956033,
 0.783938894046164,0.9951405088253243,0.5538503467745515,0.4505589174701877,
 0.3595255374036411,0.6671887405204308,0.07665794876938126,0.3318978709997046,
 0.9684030937463499,0.2874870570703182,0.49305942769155686,0.7051344747198292,
 0.40917538267598674,0.5397910490041387,0.4965177498671747,
 0.41404558954901527,0.9701274558882385,0.9467094752196458,
 0.36238898523173346,0.32533363266452675,0.20181880564677857,
 0.35763297429431173,0.345922495351368,0.6105329432348275,0.8063891863675597,
 0.3480143333424447,0.6403656119388381,0.8684573309033667,0.23342848761098833,
 0.30860579511789077,0.3319369063984159,0.5220771948073349,0.4533397221338613,
 0.9529942690128038,0.8097200860909737,0.24334720955077827,0.7182890386359552,
 0.8388920789437011,0.9736130098370479,0.052178717832881194,
 0.9486176249020558,0.063238319539479,0.6669772352363373,0.871635174079593,
 0.7871981253642184,0.7409238605736238,0.8999988863844642,0.8108856043242669,
 0.17909389096978645,0.16635284432560038,0.9060405174553187,
 0.8855438819477773,0.8754190781911382,0.9895952544919686,0.3385644898916593,
 0.6721703903979473,0.7992495361537191,0.8811917123952491,
 0.011244559998095593,0.7017941720431089,0.21188048447674168,
 0.7310881236834698,0.2590235611570394,0.5112659768352482,0.4303903923637393,
 0.02039398966963968,0.3657597329417056,0.7178449824944116,
 0.45340918140128905,0.2903554393886304,0.38177599432672815,
 0.7899103721158188,0.5010818669217355,0.8901240419054075,0.25792452820841416,
 0.9611283787681159,0.5728243162269149,0.5099448538047208,0.2264155902230458,
 0.7325672102018004,0.2515898303479239,0.11053884276789505,0.4457956664293061,
 0.7223325741496045,0.574666747255472,0.8950976845717938,0.17565582119620127,
 0.6021271844249119,0.13320154952839058,0.4824623919684453,0.8890484472773881,
 0.3998725776268811,0.6253266996740797,0.30484996164400113,
 0.42509036600971595,0.39037993571579754,0.5157252291973644,
 0.3548103109135958,0.19092786241802195,0.8695470984079346,
 0.07449771915798709,0.6015949152094919,0.48953104356082,0.9040658153830922,
 0.5486589947865873,0.2709537099778677,0.922008547607572,0.6136249170185794,
 0.6898745091095339,0.9370427728339837,0.3826967545001395,0.9721163373842732,
 0.10414769539430502,0.6741651710411402,0.16324081368102028,
 0.5290908722894725,0.7561656591284002,0.02505386068706794,0.6840227673059116,
 0.6523466161096931,0.5959088028929475,0.5758608741485476,0.6465670232932571,
 0.7058719472205058,0.6796580985246271,0.7421096089148621,0.15232347228913157,
 0.12084886111091864,0.16695333775070587,0.14363856971494138,
 0.20334227483001066,0.8335537790322819,0.0010793749839439482,
 0.38793597038138994,0.2759586522043843,0.8601727901903314,0.6396523788867866,
 0.6064855504941231,0.08335431253478265,0.975162022888724,0.7885894273962745,
 0.03875549620438001,0.6890738558445897,0.297938104010091,0.2771430932724309,
 0.28813099125077946,0.3398115286624993,0.17242856412766439,
 0.3257746842457495,0.1579894038068247,0.4520482989186825,0.9502946606214414,
 0.8143299597246473,0.059961091874383365,0.3118046358649722,
 0.39286973266287406,0.7858306117937394,0.9580184101270925,0.4454666503625102,
 0.9259226018444785,0.5619763106097336,0.7343835417645058,0.01169599254890763,
 0.6004633213362613,0.49890031939266877,0.7128320141860504,0.6411546163833232,
 0.6323704370481463,0.8557345677409381,0.7137061702559379,0.6567653221081985,
 0.4578310931366496,0.7782915602003575,0.4799000311428715,0.6112234734819019,
 0.30722126695722296,0.6268331755963749,0.0636381672080466,
 0.44583594480617794,0.8315730088376367,0.8273814895005807,0.7527477821073111,
 0.5709352113752945,0.9308609076246513,0.6999330996043829,0.4835305755161513,
 0.06554317448302849,0.5192985665600116,0.934102549908905,0.5554732656959439,
 0.050010689926636154,0.054489459229463355,0.1530394822485105,
 0.7338025879855847,0.44578459488391986,0.12273532847288404,
 0.3649442463075706,0.6434337359290362,0.6563859020324598,
 0.019911092386201923,0.33141399522115944,0.9955898657099325,
 0.12615516569048857,0.983191033409826,0.9473948752277053,0.9959983796483127,
 0.45223048348310757,0.25460135370781145,0.5002486018934342,
 0.6413626129131654,0.5244671890728445,0.3277221546848663,0.10300271126113425,
 0.5636734225482547,0.13381453292070333,0.6172647920008932,
 0.37870497221761623,0.10278842799755461,0.20477864476745133,
 0.44292006790435656,0.34983654629061245,0.5103554843669711,
 0.9429100683899523,0.7695854324918451,0.6369297323406732,0.8205486200761326,
 0.7547694485425507,0.004363542627499717,0.49630175539317045,
 0.6868902229414525,0.08675536496893643,0.5693155096406347,
 0.29183649692460556,0.6746016585749195,0.02502967293357561,
 0.5403427354168461,0.1528155401325696,0.010230287853700615,
 0.6789517658989465,0.5206070495188764,0.7435304461267174,0.4840758257544361,
 0.38936644962494515,0.1358691542176853,0.8362048725684451,
 0.20669794895457627,0.021689768252717068,0.9408190589298393,
 0.5240775915890319,0.34251880855883066,0.3970151184621873,0.9679749995874355,
 0.04887729225730164,0.9104557426013427,0.7013701865440449,
 0.11817978621258352,0.8608694904904199,0.3162976935599514,
 0.32164948919169045,0.8904603203254535,0.4788590574306161,0.234251471840766,
 0.00534785475233901,0.8338142501768575,0.5820005077037492,0.3089444767254711,
 0.21086933623516624,0.605985894702413,0.4467784988810559,0.3364628216608635,
 0.1526306214249703,0.10877106378087076,0.3676642276301818,0.634551510014658,
 0.1536313557722856,0.8663044258532042,0.46500708817369074,
 0.06874120931613292,0.936279352273275,0.8060961188550672,0.9661047355634494,
 0.3923303035613046,0.6282977637124607,0.6335198590284619,0.880636047929647,
 0.3267301480894107,0.17173784625531152,0.30203338146780034,
 0.7593316186588395,0.6557040392162862,0.8537010396296097,0.03899611524242308,
 0.2821396945028637,0.5644586018774533,0.8849074757857647,0.3662643108572907,
 0.3688822400337314,0.25692632371450663,0.6356872719323211,0.9505085915807786,
 0.761271280231362,0.4047484633084121,0.5340317839616069,0.45843497203606165,
 0.38284770576392435,0.5017937928604941,0.8046718294756464,
 0.07055818697841798,0.2807625658866708,0.4109270283767641,0.3779123702764696,
 0.7443508589672811,0.7122919291502304,0.5168369003522897,0.7951950153004459,
 0.21176116887194474,0.5483489989658208,0.30790861454104723,0.1694341164241,
 0.12732332961792414,0.22020167326103957,0.7288967717925351,
 0.18837460901946035,0.8601992050107853,0.0973172561882415,
 0.14118167740867793,0.0255566853159378,0.34161883379976665,
 0.8723696364419249,0.7895919110351489,0.07229631432821337,0.2705705938891383,
 0.9704097333069215,0.8664654921211391,0.040916115417729215,
 0.8279543014961884,0.8693511779011557,0.8398165482401482,0.4024970400579527,
 0.9200747733361911,0.5492418664503543,0.37674486010713115,0.7519085936170391,
 0.5740332578421614,0.33558001834713,0.29390786024230553,0.5327840597139416,
 0.8099005177265971,0.6454139779982107,0.2844473102910696,0.1058397174179353,
 0.6697428426508658,0.7727084599060048,0.9740542523008603,0.8757175168525398,
 0.24007409263546908,0.16019489877043724,0.4772349062999588,
 0.07867162184528897,0.31741036964634894,0.4785523972013047,
 0.8459646783598361,0.581443100005486,0.853831944213965,0.4524000841262521,
 0.42683689398856006,0.035809983990129246,0.6407651355232525,
 0.6337975631092864,0.5622944505365879,0.9678206553360345,0.5509172742453158,
 0.49978148814710943,0.35247523335306175,0.5462837134747833,0.931516808102444,
 0.4412310950074275,0.19897875541432408,0.5176922077322682,
 0.17863998268464232,0.5939298453568771,0.19655502921606394,
 0.20480971956644978,0.11771930088881877,0.22733442191574738,
 0.6675234473348777,0.07133645534373989,0.839670534627879,0.4287002398315838,
 0.12655295790479903,0.9417008092373362,0.6398587109801414,0.8449579908051758,
 0.38244123860703405,0.9820851491215394,0.033090625099695736,
 0.47280254038904135,0.8990704750179135,0.48653497388479006,
 0.06569124579143992,0.48754436951865,0.5815711816164927,0.3747804978503766,
 0.11210550056880375,0.18348385018555302,0.19592717880662203,
 0.6428548220913095,0.4332851074246101,0.1338013422993124,0.13862305253496676,
 0.45061023601861083,0.5200599010047209,0.4187709919719482,0.6323404123025341,
 0.839080115504946,0.5513161806720166,0.9284517069211806,0.021755609227140926,
 0.22699775913958087,0.5710794863574216,0.43283168763029234,0.798988193151409,
 0.13190026649566666,0.24453877241928523,0.3802593964858244,
 0.7741799865984654,0.8591801523435019,0.16233572133971452,
 0.34767858052415446,0.4019068399091362,0.3681612666044427,0.7833714811807545,
 0.9648877704423137,0.1643324557590694,0.9689837058622932,0.8098957243972472,
 0.13546134909186303,0.6987741674770271,0.3332206017218089,0.9096045238235136,
 0.5673161340949698,0.6250427804452763,0.08499384018325684,0.9374896771567373,
 0.3069943331490015,0.5932292639932838,0.720344482713566,0.1348325243008086,
 0.2564777539798996,0.9205382567013053,0.7211394350767854,
 0.015202481322177941,0.9707927463757494,0.4727101858554297,
 0.05849786316513517,0.6970435715279227,0.9851738605159285,
 0.18958516461203945,0.6040584104103229,0.5039490953058053,0.9482516126775987,
 0.46450180228568916,0.8090211266370715,0.2171769769322638,
 0.06274177855896279,0.41055669736876566,0.9666988030067661,
 0.8868651743761226,0.8858820935075375,0.10851288605761211,0.6911538698438273,
 0.7109128156237444,0.7091682543660345,0.3080064719487565,0.48624590806973733,
 0.7021837829666387,0.5943572306551652,0.2940665354572213,0.7032955594879134,
 0.33483332215265527,0.33872826169641046,0.8940634888658638,
 0.33719504693839375,0.13955750259438893,0.9599111463470269,
 0.056628919674573064,0.714785660467261,0.32300640963708427,
 0.5185600587311938,0.17405275462926073,0.6901784862742014,
 0.25084172691925033,0.1670255704385315,0.7093788984130558,0.5838436880616139,
 0.10449004092729741,0.16404047595418103,0.9531631452348938,
 0.7251098587219346,0.3218075967146422,0.47716591375352435,0.5696099009035845,
 0.33114534494703307,0.2879283983201756,0.8953559643884756,0.4847642373554405,
 0.7607926312278216,0.8575158001649155,0.46004877467582517,0.6397381157625992,
 0.2639917873998523,0.34811919090612553,0.36343236754314234,
 0.7893410715602832,0.07009392262308567,0.5994056925205917,
 0.27502958307905023,0.44788614561359286,0.665593404455552,0.7327667695852275,
 0.31516321785080414,0.14606451274914134,0.03464456596062071,
 0.3252812954079305,0.40497908812475625,0.6742333530627197,
 0.42082082662221887,0.28428259918477883,0.44080627037240017,
 0.5673480956632194,0.9621120573298413,0.6301029986737003,0.23691596788903702,
 0.5957475769745522,0.12349634561739686,0.48328437001953617,
 0.12346350025856268,0.5350784803867975,0.7700209703013348,0.9914226206219386,
 0.9288037353105529,0.9499417374466185,0.3508327619369913,0.7629636075361413,
 0.3111155866237326,0.8040283495601337,0.7227393713995013,0.5671612248098258,
 0.6435513447422444,0.8110160585591696,0.7484712262029919,0.21505053291198228,
 0.6301686061983027,0.007238359373433978,0.4244294249938684,
 0.25319638331243455,0.4876930970817628,0.7307693909179737,0.6293320613227625,
 0.5846594523318943,0.6217996093526275,0.34450731583025884,0.8760620261330901,
 0.18360926922869725,0.9919814332576378,0.7569486806352281,0.8198034914256283,
 0.7957536420209532,0.5445134074043547,0.4906202754049779,
 0.015226015369612766,0.8016843468640585,0.37378365975017835,
 0.4408073708261828,0.25685503392567544,0.03600881257632382,
 0.8537931820615128,0.7433561086021051,0.7717279401137123,0.3120851640254494,
 0.530327999334655,0.27039393633852615,0.096011780925636,0.4951396056314936,
 0.23678301801331836,0.7796548622349944,0.6990973318938106,0.9836594560217534,
 0.9712615024920703,0.3682172859979841,0.29142821573692956,0.242938498958436,
 0.6836112404554923,0.12494145645651566,0.7038863344958375,0.5496242871798324,
 0.7353589935170595,0.3528744149075307,0.9577679496412081,0.2524087312592408,
 0.9480115457616893,0.5166980537959923,0.43209790130557924,
 0.25463860411509287,0.7501094024703667,0.7656951317454543,0.4404306189313899,
 0.9923423188688217,0.4611609283235568,0.4519716807907692,0.10380264268263573,
 0.2148104872878447,0.30582200131998427,0.518500186614771,0.3650998915271527,
 0.798684288384603,0.8783155168259524,0.697519461088119,0.9340025732878464,
 0.8069402727289254,0.6340296082635782,0.4991529811077511,0.5218090298855288,
 0.7575080302808117,0.34162665601708675,0.8768723622658924,0.9592358260874896,
 0.6072140548332545,0.23575075948419455,0.9657353558463382,
 0.24952073818114318,0.2236065202277585,0.3355985497989442,0.9123730903649853,
 0.8144349766089503,0.7543153139744956,0.7017729545231086,
 0.038431577482245904,0.8257670291868345,0.34797503609515323,
 0.9102321809496194,0.4046270790132298,0.033073949719913176,
 0.9818740349121945,0.7249219712954058,0.5452622141819203,0.2591436163403358,
 0.778939836016465,0.525250666449375,0.011600160674844928,0.7430490199838445,
 0.8015016989389459,0.020137718118241277,0.09376591801928003,
 0.730637188571951,0.8727109958947067,0.6757447935065707,0.021377982899149828,
 0.8023801490051975,0.01754689566436518,0.8836346483723355,
 0.36800438497317955,0.04160136318505625,0.017124718403475847,
 0.8474935014935892,0.16584840298857106,0.8878083745698538,
 0.13945114503740297,0.7681447668094191,0.5251371610264625,0.7708629727988747,
 0.2039902171515744,0.8100182788328508,0.7401587132361374,0.4373011866973151,
 0.6283807902144064,0.35149413364160365,0.8766231806731556,0.8926760482390761,
 0.17622910583289342,0.028938903718946474,0.6541433248137197,
 0.9841782359701308,0.12351248298229234,0.2586835147775952,
 0.09958634297340885,0.19574308610426794,0.6067095401351406,
 0.31739679391654596,0.5993509462031876,0.5723041790142204,0.5093119404104928,
 0.9477822315249678,0.21690746854563348,0.8975061614123204,0.5849834998613487,
 0.6490922998652593,0.14154318308081937,0.6078091596143393,0.1269513330105302,
 0.652859267623771,0.7051138346888636,0.5642132786407346,0.21313227910369426,
 0.2230165911344073,0.43382533768660125,0.25866488510894925,
 0.4221836877939882,0.9110664906113722,0.1036484569181848,0.07411292013224946,
 0.42137230422719607,0.06727729171185215,0.5498112567110521,
 0.03977249156154483,0.13251979347297194,0.9017015896191158,
 0.08186706307199632,0.3858719728077189,0.0196224835393064,0.7026386410472705,
 0.9059793737115459,0.8086275558306617,0.5197194745614322,0.47847455210839906,
 0.9091355041925676,0.2727931458204934,0.6433559630873713,0.5830152660785429,
 0.8617607736400232,0.4050412270549636,0.18902428745191702,0.3474989579296307,
 0.6286103473448929,0.7933161871605814,0.04669122739318032,0.1588478463467361,
 0.13277728781383025,0.7601354048266245,0.8786037862158939,0.9018926757611785,
 0.6051292891018497,0.5833783060631829,0.3465406436984616,0.10694611003004839,
 0.8114717131664351,0.9640450130890431,0.9796639971360932,0.598363656768072,
 0.9498912449252435,0.08932261116613638,0.8767889815221215,0.97485081765857,
 0.009234757723494691,0.6604137267137097,0.07430412945339904,
 0.5053149238104871,0.692167065805193,0.5817385888361418,0.5445152372913415,
 0.893336654768897,0.22650027326916755,0.04045340387660212,0.349695443004271,
 0.9371895908273401,0.24266932535508834,0.1998789730940752,0.7281300257941095,
 0.6736506545992846,0.30130579129481705,0.7295315074939347,0.5212208226470103,
 0.15908248939724656,0.6173778501642091,0.7679577893207875,0.3159811106972701,
 0.2646865954776808,0.46811084404286185,0.006743386916728267,
 0.8726511784226356,0.21884295507646023,0.7755744621633508,
 0.14968246917219052,0.04627078543908891,0.9200093995355075,
 0.018631639395511046,0.9241674173096303,0.6063213893444171,
 0.6945491446196091,0.5570810648229498,0.8530288965940704,0.2987956961824908,
 0.6040409817057595,0.960926279414627,0.26664570669752985,0.33760991805469776,
 0.8656902608169963,0.7349880248821508,0.19976274345829326,0.9679402780655312,
 0.35926468161418035,0.8550979958777559,0.03167504404964594,
 0.1224666758371169,0.7682131281283053,0.6389307587861758,0.31188185458596873,
 0.262238210467849,0.8992630916219132,0.3281639881311985,0.18749810984996151,
 0.7886052663752429,0.3423718744749825,0.6065221990112888,0.2417445711281867,
 0.3010633939736954,0.9901083015231713,0.9542004031104742,0.4712646235145892,
 0.9878271073368928,0.48082500441143583,0.199722102199833,0.7022050682260258,
 0.07770566508317922,0.04339080288322461,0.6748585561142603,
 0.6368663955401256,0.2215699100652745,0.7805316068263183,0.6102697652019721,
 0.8365823485589726,0.9974549928921161,0.3216376071993945,0.9122796322793592,
 0.8652550119506063,0.4526716209975987,0.2073143978580323,0.32327258460720676,
 0.948821206755576,0.6119813101602343,0.15251092214207718,0.638649695182458,
 0.11764388623169664,0.5464262493079567,0.6101415339904301,0.5775882783985644,
 0.12542761918901846,0.38036724089304885,0.23357278521290958,
 0.9098468845136111,0.6477054467810123,0.6907621249797059,0.986876194696619,
 0.2760731566880533,0.7081234330497308,0.8188287868783379,0.6883366629142192,
 0.39632296259623545,0.4597421441132905,0.8896978239975799,0.6930903342795482,
 0.6785466561999558,0.7724151358586381,0.17806645209245053,0.7083857869404102,
 0.5009733801294326,0.20530664650730046,0.7442633185881473,0.9947769329331668,
 0.8443418643573124,0.476821386910662,0.08862305427155504,0.6470150797930762,
 0.6909300668045272,0.7281391000517177,0.8601947201141449,0.48380421954717845,
 0.8461163707100355,0.4937613912036889,0.4462590171905768,0.43197368888648624,
 0.025291247141300266,0.5516287468803576,0.8468559400388727,
 0.32644745344826287,0.8184883049395745,0.8469194117351586,0.5520979557984463,
 0.6524442784239872,0.4262297447890435,0.6228546399961433,0.6715817449112164,
 0.12234051721260708,0.2653899189748188,0.25920337467467525,0.597556984302527,
 0.871336148191665,0.3655114030672728,0.7255481401159326,0.38146126019340254,
 0.5275482218599212,0.5012226479734926,0.33133077921731713,0.7581590904764185,
 0.8128078982929094,0.4820117416534776,0.025708605584710553,
 0.7308494936999735,0.020566095888070235,0.07394486129478217,
 0.6484235673935761,0.27152682900777925,0.7597944561694194,0.4782749114717094,
 0.8207123769509215,0.9142172978693878,0.45066283496592896,0.4986471873372661,
 0.7559957266218171,0.22169278305798035,0.7413440887270566,
 0.45132510825465555,0.8511186014263706,0.11758875819175751,
 0.09932894195512953,0.6247489328085422,0.08361716435970756,
 0.6506431905671701,0.8652632431099954,0.085220948694117,0.7244695934243086,
 0.8102639260559001,0.586649949290494,0.34005096690669046,
 0.057299698433095125,0.7251204760693502,0.1809379041035729,
 0.7289515147841619,0.6833380655875337,0.014471371647267861,
 0.09258079776228734,0.8499359606411926,0.8128990298046925,
 0.49693880708214355,0.3821555096050743,0.4994192919018152,0.7805101361915876,
 0.30585676327813405,0.10967810132166145,0.11161386420121522,
 0.4027597949617643,0.8715132439127813,0.29028394596213103,0.8827562202970229,
 0.08644085915043098,0.37842856471470476,0.00975057830450532,
 0.7188302799993838,0.367536710376391,0.7832298906340247,0.9432876649249265,
 0.2965050315160196,0.7803893725463327,0.7858887787080802,0.23636863530649188,
 0.14909957234007698,0.19170845822912197,0.6626362872600541,
 0.29946995648179864,0.9067631085812276,0.18855248846028316,
 0.012635816211589712,0.4219284530896309,0.5848931074161479,0.313831061003343,
 0.9826653299856192,0.014098848317571466,0.8345039784577516,
 0.11394234187494878,0.6576095977349101,0.7939568684321291,0.478698653039088,
 0.27212781272687414,0.5597315951084593,0.2864870330300504,
 0.24111300532157687,0.19563119592698197,0.20258060662612432,
 0.6516797152941701,0.6391356864624501,0.8684915035546305,0.513458193737004,
 0.5227601171759443,0.4534462978630238,0.6091611607902605,0.07589854015911568,
 0.6040351157308436,0.7414785025108176,0.15762195259728895,0.3892516182842056,
 0.4321310066298104,0.06686914019539558,0.10850424238507472,0.455663972235715,
 0.8010441506065864,0.48653808698464074,0.15272067353031116,
 0.5406163736424019,0.7116268083357543,0.17406715432215247,0.6201659325757629,
 0.1964798158932315,0.605390065746187,0.31236546942560317,0.2895159194563641,
 0.7645293141019562,0.012211070370214294,0.6273727795512067,
 0.8246992641757901,0.32337639730398693,0.9472034491573686,0.2239956261406273,
 0.8448021988975242,0.9660241475291456,0.8276332492758262,0.9531314065947235,
 0.6434518899461703,0.6554027115096804,0.9163955615739642,0.07011726614139657,
 0.4095242434790698,0.24219486168843807,0.6104229852639027,
 0.16087607231719947,0.006681973746663794,0.4545737554871684,
 0.9612704666831928,0.3428919586875906,0.9545220928425551,0.6476909113918674,
 0.8419640827844245,0.6003961573782007,0.8411359520455769,0.6941942612926253,
 0.7096384151317804,0.8088250722903789,0.8991943085664501,
 0.023315054602764396,0.8737875910669111,0.9312880513032007,
 0.1749535717564824,0.3413600913442607,0.09483933590478222,0.9632360518592974,
 0.345088198914945,0.22968943542606135,0.5997075722132603,0.49083380330842297,
 0.22991575281362686,0.8710192184851033,0.01676734995625251,
 0.7437893405861358,0.1510700502133776,0.04126428406960014,0.7493838223832041,
 0.486166350950495,0.17615947260913034,0.05469152843156555,0.6821842141103601,
 0.902092516251964,0.017470260707326712,0.37140902605655857,
 0.5153525213169976,0.733828428651081,0.7239970469656687,0.7339180449495379,
 0.1558451420492012,0.3936204679853228,0.34073951208309583,0.7042590130781845,
 0.6549271431999829,0.6249368009388896,0.6223723085413225,0.9781297316912185,
 0.7675380003525231,0.8017322348964978,0.7879523692255024,0.1372806534619011,
 0.4796216466477735,0.6052640473901789,0.6491122620415154,0.09458143046865042,
 0.9447093635990682,0.8185922364142894,0.8833908642910642,0.7365589383181221,
 0.44659100607761126,0.9529007497683666,0.9910021433599601,0.4876744456950397,
 0.2132338521376953,0.3143020395367776,0.5659168881862129,0.9063299069708505,
 0.5951425072470113,0.572023445041601,0.7058244348692507,0.9205557588821172,
 0.4203454337652385,0.02764569082940249,0.6422956728100581,0.7331717202541504,
 0.8933891432740181,0.6200891723142252,0.8496361912785618,0.32945762001215606,
 0.07756534855739483,0.2565236923961929,0.798453260179104,0.7777224146492447,
 0.4944675713214314,0.11157107029337898,0.308888267893742,0.8424993036574269,
 0.6218961533191443,0.03513399075568269,0.21908327266928174,
 0.7714033163830283,0.36368182429706486,0.2529052057711775,0.3965267187105673,
 0.8825970215123188,0.9828185824809968,0.3346353960459647,0.5741900100100227,
 0.04505151322715939,0.1556321449447895,0.1259171588902056,0.3413371547869166,
 0.4965894364641865,0.8816788515178502,0.4516828306589662,0.36892200931142793,
 0.14836559131739335,0.5693390430926152,0.4596150014909228,0.776803847447763,
 0.8558557778005842,0.20789612066216234,0.772325255562373,0.07480175561117897,
 0.9167212135505884,0.6030644776576336,0.4348999425386606,0.3470395010922589,
 0.8937765035573889,0.8182274918892004,0.09301026678485824,0.4055431060576209,
 0.6980303848621175,0.9924988986413734,0.703168480685256,0.23288140604299667,
 0.5501859123043198,0.7359087553187079,0.29569790553130415,
 0.06123025007180072,0.6829868075545835,0.7801180014977662,0.8753429001548588,
 0.20730226336495994,0.03394107152550807,0.24508194484390244,
 0.5838676073699809,0.7479062440009674,0.12343931443106093,0.6051433816884773,
 0.44497364943078044,0.3329987290606701,0.08565901001848442,
 0.34402587862245826,0.7007589792024557,0.534291901314898,0.3709586022885406,
 0.8358959316774073,0.16789398983900805,0.22610732755938534,
 0.8262068432948763,0.5588494505109334,0.544954033477416,0.18258986856420945,
 0.10389546357292778,0.4769650876778403,0.07538779104623217,
 0.9747586840527936,0.7213588400447988,0.7305362582483623,0.5779074419657632,
 0.7911434420546839,0.9050615804700062,0.9244151274760053,0.6273949689105616,
 0.21342169337562977,0.7992515156387017,0.0933607005022995,0.8692309664288832,
 0.17492144787498476,0.6616857508435605,0.7749565124558957,
 0.16249255766343396,0.8914109404387016,0.8973833545820202,0.2449924670658381,
 0.45188508261402194,0.7457771175577846,0.5365540455567657,0.9836237764455172,
 0.5942388570036262,0.22124684241429726,0.8200935377295993,0.3243452824989075,
 0.10563244463762078,0.002551151092683668,0.027071195328883357,
 0.029681629943839938,0.10990851644714583,0.0632739736456952,
 0.1169563681420237,0.9263809977881213,0.10431609372768913,0.9497764333199373,
 0.1675437584035938,0.40708842148312696,0.5035844728607963,0.6647838304553386,
 0.3372401883846363,0.6770189318322014,0.6827487292590627,0.25516244227099194,
 0.18301262331886237,0.7628244871123183,0.6481746396338626,0.8305302941494259,
 0.3620888201296222,0.2700806578795094,0.5071007650490755,0.5533445994007307,
 0.45815578587959493,0.22370756023498584,0.2916584691418216,
 0.8016688863535365,0.5797675492278372,0.595161605890249,0.015611745543423372,
 0.19192143761707003,0.6143855134079845,0.06440026049891945,
 0.25604000417308415,0.4438960072480018,0.129135412662577,0.4719871818761753,
 0.409402879743449,0.5972524736170034,0.42660055847734013,0.6191550033098321,
 0.2506974794052631,0.2186612362527125,0.4861195581907046,0.8080492103588324,
 0.7726252507984787,0.9636822582411346,0.7600170932860786,0.6374836010021719,
 0.4691848268804515,0.3452633398848113,0.14970197263752505,0.9758982449920386,
 0.28044649941362776,0.7626418815746403,0.9130035080145581,0.271252347453368,
 0.5662014154899162,0.9695429208914,0.06207104535807351,0.6221382200362121,
 0.5857497900074753,0.23283103398430538,0.012110382487308557,
 0.8921981268698473,0.8151423459261691,0.4656122121785069,0.5841435735838536,
 0.5700324129858864,0.16747774440981167,0.7851484313726096,
 0.23710397962785634,0.2290502535414981,0.012339087629247736,
 0.2942886526170969,0.5461453905421627,0.32430625180391737,0.5324247767315792,
 0.7660529529903924,0.05638282592564958,0.9055114223542067,0.5173171810170315,
 0.43795610861675394,0.36701030845177396,0.6480692127391564,
 0.7132093457299387,0.17503585955757683,0.9671884791365638,
 0.01786332134633062,0.24636768007179444,0.0030476792958509247,
 0.21861338683404496,0.229525032023834,0.40454409199896124,0.5113887094251153,
 0.4304790950709555,0.745485258578517,0.1966683916255525,0.11874334154272947,
 0.38829868266301193,0.6577306822157555,0.23789224180726853,
 0.3672639992829426,0.5296537504387296,0.5996179637271797,0.12249364471864477,
 0.6605138588977133,0.04317209044446002,0.18666314444449383,
 0.027462918205978903,0.384278143669576,0.11893572555533294,
 0.6839672014754865,0.06334340074766864,0.19925986265861262,
 0.6770012222060406,0.8506764324361176,0.2886255168304641,0.6972720646458344,
 0.19158267981533372,0.6618449123884229,0.6801379871837033,0.6928212837685304,
 0.7446437694448809,0.019444933768645978,0.07309632015652201,
 0.5629679592444478,0.0723820295594918,0.5706723427287269,0.3158440764730367,
 0.19545669217780393,0.13103313145283169,0.6862840870782596,
 0.28921816118506183,0.9803102437044784,0.9103798049686538,
 0.030236430992243712,0.8055868239208281,0.3661941908734826,
 0.1575577445817582,0.01532945623034676,0.5354994481276316,0.6160898880653318,
 0.11571272765631513,0.4455583518261823,0.9938057293235496,0.8214886115836307,
 0.8700252364682874,0.3573322411124541,0.7651097430145832,0.4596135957399673,
 0.1294373574624097,0.11651686663429128,0.002036912296322857,
 0.2071473722232281,0.6575601801035657,0.2762037624188779,0.49342296882331704,
 0.060361082588937465,0.03404660841247398,0.25967735237052536,
 0.0917470384524508,0.48871654901464223,0.6846069636432923,0.811776073308309,
 0.29560556899925505,0.5179887700797503,0.5435312438821117,0.6118875390016312,
 0.8989283631217074,0.1306025831048695,0.55436661404111,0.011632748399195036,
 0.9688927033604995,0.797852885008843,0.19921808592753953,0.6027368433196625,
 0.6141663762488839,0.02388868157464452,0.3254785209208426,0.3442763380447863,
 0.7292511424254218,0.8765018513383783,0.6238902541579765,0.7792892771798661,
 0.6601237388031795,0.8450666975680803,0.06197108530200934,
 0.019539337613866392,0.6727121129255235,0.42237515390648217,
 0.2434472578196717,0.2619345681035753,0.9964643714771381,0.5291821116670873,
 0.1784696881504091,0.37491063939395985,0.6643180033101996,0.5838214461615147,
 0.9684148926026047,0.48530961449716115,0.9011036024062568,0.3777888535062536,
 0.2406225001104848,0.24267550241271352,0.19530821566139545,0.608880706713473,
 0.7327005878113044,0.429643603029477,0.7805839357843002,0.21767131195752298,
 0.32836701189878226,0.7283400029234844,0.029264779648921824,
 0.2045349297187672,0.49539093154805225,0.9396971761620501,0.3058172330932645,
 0.9481666350793985,0.11438525487615336,0.32090278270498906,
 0.2726231358259871,0.37286688979899174,0.6818503752903177,0.1928370671262788,
 0.9737700640139336,0.422892269298615,0.14313980706689589,0.5142197700935528,
 0.3280452933190606,0.12816733944055825,0.11111449968701059,
 0.42720433232663424,0.4318043259533336,0.9495459815685826,0.315443609054182,
 0.3928671663957929,0.882679492751193,0.6732615246497511,0.0265922100698901,
 0.8092723092567434,0.5172812969914812,0.8191479726437918,0.37918583993969,
 0.19640164151798978,0.7