/* ------------------------------------------------------------------------- */
/*
 *  tokenmap.h
 *
 *  Copyright (c) 2004 - 2008, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Thu 09 Oct 2008 06:44:00 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_TOKENMAP_H
#define CLX_TOKENMAP_H

#include <string>
#include <map>
#include "tokenizer_func.h"
#include "lexical_cast.h"

namespace clx {
	/* --------------------------------------------------------------------- */
	//  basic_tokenmap
	/* --------------------------------------------------------------------- */
	template <
		class TokenizerFunc,
		class KeyT = char,
		class ValueT = std::basic_string<char>,
		class Container = std::map<KeyT, ValueT>
	>
	class basic_tokenmap : public Container {
	public:
		typedef TokenizerFunc functor;
		typedef KeyT key_type;
		typedef ValueT value_type;
		typedef typename functor::char_type char_type;
		typedef std::basic_string<char_type> string_type;
		
		// constructor and destructor
		explicit basic_tokenmap(functor f = functor()) :
			super(), f_(f) {}
		
		template <class InputIterator>
		explicit basic_tokenmap(InputIterator first, InputIterator last, functor f = functor()) :
			super(), f_(f) { this->assign(first, last); }
		
		explicit basic_tokenmap(const string_type& s, functor f = functor()) :
			super(), f_(f) { this->assign(s); }
		
		explicit basic_tokenmap(const char_type* s, functor f = functor()) :
			super(), f_(f) { this->assign(s); }
		
		virtual ~basic_tokenmap() {}
		
		//operator
		void reset() {
			f_.reset();
			this->clear();
		}
		
		template <class InputIterator>
		basic_tokenmap& assign(InputIterator first, InputIterator last) {
			string_type tmp;
			string_type token;
			
			this->reset();
			while (f_(first, last, tmp, token)) {
				key_type key = lexical_cast<key_type>(tmp);
				value_type val = lexical_cast<value_type>(token);
				this->insert(std::pair<key_type, value_type>(key, val));
				if (first == last) break;
				++first;
			}
			return *this;
		}
		
		basic_tokenmap& assign(const string_type& s) {
			typename string_type::const_iterator pos = s.begin();
			return this->assign(pos, s.end());
		}
		
		basic_tokenmap& assign(const char_type* s) {
			string_type tmp(s);
			return this->assign(tmp);
		}
		
		template <class InputIterator>
		basic_tokenmap& operator()(InputIterator first, InputIterator last) {
			return this->assign(first, last);
		}
		
		basic_tokenmap& operator()(const string_type& s) { return this->assign(s); }
		basic_tokenmap& operator()(const char_type* s) { return this->assign(s); }
		
	private:
		typedef Container super;
		
		functor f_;
	};
	
	typedef basic_tokenmap<format_separator<char>, char, std::basic_string<char> > tokenmap;
#ifdef CLX_USE_WCHAR
	typedef basic_tokenmap<format_separator<wchar_t>, wchar_t, std::basic_string<wchar_t> > wtokenmap;
#endif // CLX_USE_WCHAR
	
	/* --------------------------------------------------------------------- */
	//  create_tokenmap
	/* --------------------------------------------------------------------- */
	template <class KeyT, class ValueT, class TokenizerFunc>
	inline basic_tokenmap<TokenizerFunc, KeyT, ValueT> create_tokenmap(TokenizerFunc f) {
		return basic_tokenmap<TokenizerFunc, KeyT, ValueT>(f);
	}
}

#endif // CLX_TOKENMAP_H
