package jp.ac.kyoto_u.jfftw3;

import java.nio.ByteBuffer;
import java.nio.DoubleBuffer;
import java.nio.ByteOrder;
import java.util.Set;

/**
 * 複素数から複素数へのフーリエ変換を行うクラスです。
 * @author Katsutoshi Itoyama (itoyama[at]kuis.kyoto-u.ac.jp)
 * @version 2009.10.04
 */
public final class PlanC2c
  extends Plan {

  static {
    System.loadLibrary("jfftw3");
  }

  private
  PlanC2c(final int size) {
    super(size, size);
  }

  /**
   * 1次元変換のプランを作成します。
   * @param n 変換のサイズ
   * @param sign 変換の方向
   * @param flagSet プラン作成時のフラグ
   * @see Sign
   * @see Flag
   * @throws NonPositiveTransformSizeException 指定された変換のサイズが零または負の場合
   * @throws NullPointerException 指定された変換の方向もしくはプラン作成時のフラグがnullである場合
   */
  public static final
  PlanC2c newPlan(final int n,
                  final Sign sign,
                  final Set<Flag> flagSet) {
    if (n <= 0)
      throw new NonPositiveTransformSizeException(n);
    if (sign == null)
      throw new NullPointerException("sign");
    if (flagSet == null)
      throw new NullPointerException("flagSet");

    final int size = n * 2;
    final PlanC2c plan = new PlanC2c(size);
    plan.fftwPlanDft1d(n, sign, flagSet);
    return plan;
  }

  /**
   * 2次元変換のプランを作成します。
   * @param n0 変換の第1次元のサイズ
   * @param n1 変換の第2次元のサイズ
   * @param sign 変換の方向
   * @param flagSet プラン作成時のフラグ
   * @see Sign
   * @see Flag
   * @throws NonPositiveTransformSizeException
   *  指定された変換のサイズが零または負の場合
   * @throws NullPointerException
   *  指定された変換の方向もしくはプラン作成時のフラグがnullである場合
   */
  public static final
  PlanC2c newPlan(final int n0,
                  final int n1,
                  final Sign sign,
                  final Set<Flag> flagSet) {
    if (n0 <= 0)
      throw new NonPositiveTransformSizeException(n0, "n0");
    if (n1 <= 0)
      throw new NonPositiveTransformSizeException(n1, "n1");
    if (sign == null)
      throw new NullPointerException("sign");
    if (flagSet == null)
      throw new NullPointerException("flagSet");

    final int size = n0 * n1 * 2;
    PlanC2c plan = new PlanC2c(size);
    plan.fftwPlanDft2d(n0, n1, sign, flagSet);
    return plan;
  }

  /**
   * 3次元変換のプランを作成します。
   * @param n0 変換の第1次元のサイズ
   * @param n1 変換の第2次元のサイズ
   * @param n2 変換の第3次元のサイズ
   * @param sign 変換の方向
   * @param flagSet プラン作成時のフラグ
   * @see Sign
   * @see Flag
   * @throws NonPositiveTransformSizeException
   *  指定された変換のサイズが零または負の場合
   * @throws NullPointerException
   *  指定された変換の方向もしくはプラン作成時のフラグがnullである場合
   */
  public static final
  PlanC2c newPlan(final int n0,
                  final int n1,
                  final int n2,
                  final Sign sign,
                  final Set<Flag> flagSet) {
    if (n0 <= 0)
      throw new NonPositiveTransformSizeException(n0, "n0");
    if (n1 <= 0)
      throw new NonPositiveTransformSizeException(n1, "n1");
    if (n2 <= 0)
      throw new NonPositiveTransformSizeException(n2, "n2");
    if (sign == null)
      throw new NullPointerException("sign");
    if (flagSet == null)
      throw new NullPointerException("flagSet");

    final int size = n0 * n1 * n2 * 2;
    PlanC2c plan = new PlanC2c(size);
    plan.fftwPlanDft3d(n0, n1, n2, sign, flagSet);
    return plan;
  }

  /**
   * 多次元変換のプランを作成します。
   * @param n 変換のサイズ。
   *  配列の長さが階数を、各要素が各階のサイズを表す
   * @param sign 変換の方向
   * @param flagSet プラン作成時のフラグ
   * @see Sign
   * @see Flag
   * @throws NullPointerException
   *  指定された変換のサイズ、変換の方向、
   *  プラン作成時のフラグのいずれかがnullである場合
   * @throws IllegalArgumentException
   *  指定された変換の階数が零の場合
   * @throws NonPositiveTransformSizeException
   *  指定された変換のサイズのいずれかの要素が零または負の場合
   */
  public static final
  PlanC2c newPlan(final int[] n,
                  final Sign sign,
                  final Set<Flag> flagSet) {
    if (n == null)
      throw new NullPointerException("n");
    if (n.length == 0)
      throw new IllegalArgumentException("n.length must be positive");
    for (int i = 0; i < n.length; i++)
      if (n[i] <= 0)
        throw new NonPositiveTransformSizeException
          (n[i], String.format("n[%d]", i));
    if (sign == null)
      throw new NullPointerException("sign");
    if (flagSet == null)
      throw new NullPointerException("flagSet");

    final int size = Misc.prod(n) * 2;
    PlanC2c plan = new PlanC2c(size);
    plan.fftwPlanDft(n, sign, flagSet);
    return plan;
  }

  private final native
  void fftwPlanDft1d(int n,
                     Sign sign,
                     Set<Flag> flagSet);

  private final native
  void fftwPlanDft2d(int n0,
                     int n1,
                     Sign sign,
                     Set<Flag> flagSet);

  private final native
  void fftwPlanDft3d(int n0,
                     int n1,
                     int n2,
                     Sign sign,
                     Set<Flag> flagSet);

  private final native
  void fftwPlanDft(int[] n,
                   Sign sign,
                   Set<Flag> flagSet);

  // public static void main(String[] args) {
  //   int size = 4;
  //   double[] sa = new double[size * 2];
  //   double[] da = new double[size * 2];
  //   Plan plan = PlanC2c.newPlan(size, Sign.FORWARD, java.util.EnumSet.of(Flag.ESTIMATE));
  //   DoubleBuffer sb = plan.src();
  //   DoubleBuffer db = plan.dst();

  //   sa[0] = 1.0;
  //   sa[2] = 1.0;
  //   sa[4] = 1.0;
  //   sa[6] = 1.0;

  //   sb.put(sa);
  //   System.out.println(java.util.Arrays.toString(sa));
  //   System.out.println(java.util.Arrays.toString(da));
  //   db.put(0, 1.0);
  //   plan.execute();
  //   db.get(da);
  //   System.out.println(java.util.Arrays.toString(sa));
  //   System.out.println(java.util.Arrays.toString(da));
  // }

  public static final void main(String[] args) {
    final int size = 16384;
    final int iter = 10000;
    final Plan plan =
      PlanC2c.newPlan(size, Sign.FORWARD, java.util.EnumSet.of(Flag.ESTIMATE));
    final DoubleBuffer sb = plan.src();

    for (int i = 0; i < iter; i++) {
      sb.put((int)(Math.random() * size * 2), Math.random());
      plan.execute();
    }
  }
}
