package jp.ac.kyoto_u.jfftw3;

import java.nio.ByteBuffer;
import java.nio.DoubleBuffer;
import java.nio.ByteOrder;
import java.util.Set;

/**
 * 複素数から実数へのフーリエ変換を行うクラスです。
 * @author Katsutoshi Itoyama (itoyama[at]kuis.kyoto-u.ac.jp)
 * @version 2009.10.04
 */
public final class PlanC2r
  extends Plan {

  static {
    System.loadLibrary("jfftw3");
  }

  private PlanC2r(final int srcSize, final int dstSize) {
    super(srcSize, dstSize);
  }

  /**
   * 1次元変換のプランを作成します。
   * @param n 変換のサイズ
   * @param flagSet プラン作成時のフラグ
   * @see Flag
   * @throws NonPositiveTransformSizeException
   *  指定された変換のサイズが零または負の場合
   * @throws NullPointerException
   *  指定されたプラン作成時のフラグがnullである場合
   */
  public static final
  PlanC2r newPlan(final int n,
                  final Set<Flag> flagSet) {
    if (n <= 0)
      throw new NonPositiveTransformSizeException(n);
    if (flagSet == null)
      throw new NullPointerException("flagSet");

    final int srcSize = (n / 2 + 1) * 2;
    final int dstSize = n;
    final PlanC2r plan = new PlanC2r(srcSize, dstSize);
    plan.fftwPlanDftC2r1d(n, flagSet);
    return plan;
  }

  /**
   * 2次元変換のプランを作成します。
   * @param n0 変換の第1次元のサイズ
   * @param n1 変換の第2次元のサイズ
   * @param flagSet プラン作成時のフラグ
   * @see Flag
   * @throws NonPositiveTransformSizeException
   *  指定された変換のサイズが零または負の場合
   * @throws NullPointerException
   *  指定されたプラン作成時のフラグがnullである場合
   */
  public static final
  PlanC2r newPlan(final int n0,
                  final int n1,
                  Set<Flag> flagSet) {
    if (n0 <= 0)
      throw new NonPositiveTransformSizeException(n0, "n0");
    if (n1 <= 0)
      throw new NonPositiveTransformSizeException(n1, "n1");
    if (flagSet == null)
      throw new NullPointerException("flagSet");

    final int srcSize = n0 * (n1 / 2 + 1) * 2;
    final int dstSize = n0 * n1;
    final PlanC2r plan = new PlanC2r(srcSize, dstSize);
    plan.fftwPlanDftC2r2d(n0, n1, flagSet);
    return plan;
  }

  /**
   * 3次元変換のプランを作成します。
   * @param n0 変換の第1次元のサイズ
   * @param n1 変換の第2次元のサイズ
   * @param n2 変換の第3次元のサイズ
   * @param flagSet プラン作成時のフラグ
   * @see Flag
   * @throws NonPositiveTransformSizeException
   *  指定された変換のサイズが零または負の場合
   * @throws NullPointerException
   *  指定されたプラン作成時のフラグがnullである場合
   */
  public static final
  PlanC2r newPlan(final int n0,
                  final int n1,
                  final int n2,
                  Set<Flag> flagSet) {
    if (n0 <= 0)
      throw new NonPositiveTransformSizeException(n0, "n0");
    if (n1 <= 0)
      throw new NonPositiveTransformSizeException(n1, "n1");
    if (n2 <= 0)
      throw new NonPositiveTransformSizeException(n2, "n2");
    if (flagSet == null)
      throw new NullPointerException("flagSet");

    final int srcSize = n0 * n1 * (n2 / 2 + 1) * 2;
    final int dstSize = n0 * n1 * n2;
    final PlanC2r plan = new PlanC2r(srcSize, dstSize);
    plan.fftwPlanDftC2r3d(n0, n1, n2, flagSet);
    return plan;
  }

  /**
   * 多次元変換のプランを作成します。
   * @param n 変換のサイズ。
   *  配列の長さが階数を、各要素が各階のサイズを表す
   * @param flagSet プラン作成時のフラグ
   * @see Flag
   * @throws NullPointerException
   *  指定された変換のサイズもしくはプラン作成時のフラグの
   *  いずれかがnullである場合
   * @throws IllegalArgumentException
   *  指定された変換の階数が零の場合
   * @throws NonPositiveTransformSizeException
   *  指定された変換のサイズのいずれかの要素が零または負の場合
   */
  public static final
  PlanC2r newPlan(final int[] n,
                  final Set<Flag> flagSet) {
    if (n == null)
      throw new NullPointerException("n");
    if (n.length == 0)
      throw new IllegalArgumentException("n.length must be positive");
    for (int i = 0; i < n.length; i++)
      if (n[i] <= 0)
        throw new NonPositiveTransformSizeException
          (n[i], String.format("n[%d]", i));
    if (flagSet == null)
      throw new NullPointerException("flagSet");

    final int srcSize = Misc.prod2(n);
    final int dstSize = Misc.prod(n);
    final PlanC2r plan = new PlanC2r(srcSize, dstSize);
    plan.fftwPlanDftC2r(n, flagSet);
    return plan;
  }

  private final native
  void fftwPlanDftC2r1d(int n,
                        Set<Flag> flagSet);

  private final native
  void fftwPlanDftC2r2d(int n0,
                        int n1,
                        Set<Flag> flagSet);

  private final native
  void fftwPlanDftC2r3d(int n0,
                        int n1,
                        int n2,
                        Set<Flag> flagSet);

  private final native
  void fftwPlanDftC2r(int[] n,
                      Set<Flag> flagSet);
}
