package jp.ac.kyoto_u.jfftw3;

import java.nio.ByteBuffer;
import java.nio.DoubleBuffer;
import java.nio.ByteOrder;
import java.util.Set;

/**
 * 実数から実数への各種変換を行うクラスです。
 * @see R2rKind
 * @author Katsutoshi Itoyama (itoyama@kuis.kyoto-u.ac.jp)
 * @version 2009.10.04
 */
public final class PlanR2r
  extends Plan {

  static {
    System.loadLibrary("jfftw3");
  }

  private
  PlanR2r(final int size) {
    super(size, size);
  }

  /**
   * 1次元変換のプランを作成します。
   * @param n 変換のサイズ
   * @param kind 変換の種類
   * @param flagSet プラン作成時のフラグ
   * @see R2rKind
   * @see Flag
   * @throws NonPositiveTransformSizeException 指定された変換のサイズが零または負の場合
   * @throws NullPointerException 指定された変換の種類もしくはプラン作成時のフラグがnullである場合
   */
  public static final
  PlanR2r newPlan(final int n,
                  final R2rKind kind,
                  final Set<Flag> flagSet) {
    if (n <= 0)
      throw new NonPositiveTransformSizeException(n, "n");
    if (kind == null)
      throw new NullPointerException("kind");
    if (flagSet == null)
      throw new NullPointerException("flagSet");

    final int size = n;
    final PlanR2r plan = new PlanR2r(size);
    plan.fftwPlanR2r1d(n, kind, flagSet);
    return plan;
  }

  /**
   * 2次元変換のプランを作成します。
   * @param n0 変換の第1次元のサイズ
   * @param n1 変換の第2次元のサイズ
   * @param kind0 第1次元の変換の種類
   * @param kind1 第2次元の変換の種類
   * @param flagSet プラン作成時のフラグ
   * @see R2rKind
   * @see Flag
   * @throws NonPositiveTransformSizeException 指定された変換のサイズが零または負の場合
   * @throws NullPointerException 指定された変換の種類もしくはプラン作成時のフラグがnullである場合
   */
  public static final
  PlanR2r newPlan(final int n0,
                  final int n1,
                  final R2rKind kind0,
                  final R2rKind kind1,
                  Set<Flag> flagSet) {
    if (n0 <= 0)
      throw new NonPositiveTransformSizeException(n0, "n0");
    if (n1 <= 0)
      throw new NonPositiveTransformSizeException(n1, "n1");
    if (kind0 == null)
      throw new NullPointerException("kind0");
    if (kind1 == null)
      throw new NullPointerException("kind1");
    if (flagSet == null)
      throw new NullPointerException("flagSet");

    final int size = n0 * n1;
    final PlanR2r plan = new PlanR2r(size);
    plan.fftwPlanR2r2d(n0, n1, kind0, kind1, flagSet);
    return plan;
  }

  /**
   * 3次元変換のプランを作成します。
   * @param n0 変換の第1次元のサイズ
   * @param n1 変換の第2次元のサイズ
   * @param n2 変換の第3次元のサイズ
   * @param kind0 第1次元の変換の種類
   * @param kind1 第2次元の変換の種類
   * @param kind2 第3次元の変換の種類
   * @param flagSet プラン作成時のフラグ
   * @see R2rKind
   * @see Flag
   * @throws NonPositiveTransformSizeException 指定された変換のサイズが零または負の場合
   * @throws NullPointerException 指定された変換の種類もしくはプラン作成時のフラグがnullである場合
   */
  public static final
  PlanR2r newPlan(final int n0,
                  final int n1,
                  final int n2,
                  final R2rKind kind0,
                  final R2rKind kind1,
                  final R2rKind kind2,
                  final Set<Flag> flagSet) {
    if (n0 <= 0)
      throw new NonPositiveTransformSizeException(n0, "n0");
    if (n1 <= 0)
      throw new NonPositiveTransformSizeException(n1, "n1");
    if (n2 <= 0)
      throw new NonPositiveTransformSizeException(n2, "n2");
    if (kind0 == null)
      throw new NullPointerException("kind0");
    if (kind1 == null)
      throw new NullPointerException("kind1");
    if (kind2 == null)
      throw new NullPointerException("kind2");
    if (flagSet == null)
      throw new NullPointerException("flagSet");

    final int size = n0 * n1 * n2;
    final PlanR2r plan = new PlanR2r(size);
    plan.fftwPlanR2r3d(n0, n1, n2, kind0, kind1, kind2, flagSet);
    return plan;
  }

  /**
   * 多次元変換のプランを作成します。
   * @param n 変換のサイズ。
   *  配列の長さが階数を、各要素が各階のサイズを表す
   * @param kind 各階の変換の種類
   * @param flagSet プラン作成時のフラグ
   * @see R2rKind
   * @see Flag
   * @throws NullPointerException 変換のサイズ、変換の種類、プラン作成時のフラグのいずれかがnullである場合
   * @throws IllegalArgumentException 指定された変換の階数が零の場合、もしくは変換の階数と変換の種類の長さが一致しない場合
   * @throws NonPositiveTransformSizeException 指定された変換のサイズが零または負の場合
   */
  public static final
  PlanR2r newPlan(final int rank,
                  final int[] n,
                  final R2rKind[] kind,
                  final Set<Flag> flagSet) {
    if (n == null)
      throw new NullPointerException("n");
    if (n.length == 0)
      throw new IllegalArgumentException("n.length must be positive");
    for (int i = 0; i < n.length; i++)
      if (n[i] <= 0)
        throw new NonPositiveTransformSizeException
          (n[i], String.format("n[%d]", i));
    if (kind == null)
      throw new NullPointerException("kind");
    if (kind.length != n.length)
      throw new IllegalArgumentException
        (String.format("n.length and kind.length do not match: " +
                       "n.length = %d, kind.length = %d",
                       n.length, kind.length));
    if (flagSet == null)
      throw new NullPointerException("flagSet");

    final int size = Misc.prod(n);
    final PlanR2r plan = new PlanR2r(size);
    plan.fftwPlanR2r(n, kind, flagSet);
    return plan;
  }

  private final native
  void fftwPlanR2r1d(int n,
                     R2rKind kind,
                     Set<Flag> flagSet);

  private final native
  void fftwPlanR2r2d(int n0,
                     int n1,
                     R2rKind kind0,
                     R2rKind kind1,
                     Set<Flag> flagSet);

  private final native
  void fftwPlanR2r3d(int n0,
                     int n1,
                     int n2,
                     R2rKind kind0,
                     R2rKind kind1,
                     R2rKind kind2,
                     Set<Flag> flagSet);

  private final native
  void fftwPlanR2r(int[] n,
                   R2rKind[] kind,
                   Set<Flag> flagSet);
}
