/*
 * Original coded: 2003/12/08
 *
 */

/*
 * LICENSE:
 * 
 *     Copyright (C) 2006 Hidenao Abe (COIN Project)
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License along
 *    with this program; if not, write to the Free Software Foundation, Inc.,
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
package coin.mining_result_evaluator.classificationRule;

import java.util.Vector;
import java.util.Enumeration;
import java.util.Arrays;
import java.io.BufferedWriter;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.IOException;

import coin.Attribute;
import coin.Attributes;
import coin.Clause;
import coin.Evaluator;
import coin.Instance;
import coin.Instances;
import coin.Rule;
import coin.Ruleset;

/**
 * @author Hidenao Abe  (hidenao@users.sourceforge.jp)
 *
 */
public abstract class RuleEvaluator extends Evaluator{
	
	protected Vector rulesets; // set of classifier set
	protected int RuleCountNumber;
	
	public RuleEvaluator(){
		super();
		
		rulesets = new Vector();
	}
	
	public RuleEvaluator(String name){
		super(name);
		
		rulesets = new Vector();
	}
	
	public void evaluation(){
	}

	public void addRuleset(Ruleset inputRules){
		rulesets.add(inputRules);
		int i;
		for(i=0; i<inputRules.size(); i++){
			results.put(inputRules.getRule(i),new Double(0.0f));
		}
	}

	public void countAllOfRulesStats(){

		Rule rule;
		Ruleset rules;
		Instances dset;
		int i=0,j=0;
		
		for(i=0; i<rulesets.size(); i++){
			rules = (Ruleset)rulesets.elementAt(i);
			dset = (Instances)dsets.get(evalTable.get(rulesets.elementAt(i)));
			for(j=0; j<rules.size(); j++){
				rule = rules.getRule(j);
				rule.initRuleStats();
		
				int k=0;
				for(k=0; k<dset.size(); k++){
					isMatch(dset.getAttributeInfo(),dset.getInstance(k), rule);
				}
				RuleCountNumber++;
			}
		}
	}

	protected boolean isMatch(Attributes attInfo, Instance data, Rule rule){
		boolean result = false;
		
		int i=0;
		
		for(i=rule.getNumOfAntecedents()-1; i>=0; i--){
			if(! isMatch(attInfo, data, rule.getAntecedent(i))){
				result = false;
				break;
			}
			else{
				result = true;
			}
		}
		if(result){
			rule.incrementMatches();
			if(isMatch(attInfo, data, rule.getConsequent())){
				rule.incrementCorrects();
			}
		}
		
		return result;
	}
	
	protected boolean isMatch(Attributes attInfo, Instance data, Clause clause){
		
		boolean result = false;
		
		Attribute att = attInfo.getAttribute(clause.getAttName());
		if(att==null){ System.out.println(clause.toString()); }
		String value = data.getValueAt(att.getId());
		
		if(value==null){ System.out.println(clause.toString()); }
		
		if(!value.equals(Instance.UNKNOWN_VALUE)){
			switch(att.getType()){
				case Attribute.NOMINAL:
					result = matchStringAsString(value,clause);
				break;
				case Attribute.NUMERIC:
					result = matchStringAsFloat(value,clause);
				break;
				case Attribute.STRING:
					result = matchStringAsString(value,clause);
				break;
				default:
				break;
			}
		}
		
		return result;
		
	}
	
	protected boolean isMatch(Attribute att, String value, Clause clause){
		
		boolean result = false;

		if(!value.equals(Instance.UNKNOWN_VALUE)){
			switch(att.getType()){
				case Attribute.NOMINAL:
					result = matchStringAsString(value,clause);
				break;
				case Attribute.NUMERIC:
					result = matchStringAsFloat(value,clause);
				break;
				case Attribute.STRING:
					result = matchStringAsString(value,clause);
				break;
				default:
				break;
			}
		}
		
		return result;
	}
	
	protected boolean matchStringAsFloat(String strValue, Clause clause){
		
		boolean result = false;
		
		float value=0.0f, l_value=Float.MIN_VALUE, u_value=Float.MAX_VALUE;
		
		try{
			if(! strValue.equals(Instance.UNKNOWN_VALUE)){
				value = Float.parseFloat(strValue);
			}
			else{
				value = Float.NEGATIVE_INFINITY;
			}
			l_value = Float.parseFloat(clause.getLowerValue());
			if(clause.getUpperValue() != null && !clause.getUpperValue().equals(Instance.UNKNOWN_VALUE)){
				u_value = Float.parseFloat(clause.getUpperValue());
			}
			else{
				u_value = Float.MAX_VALUE;
			}

			switch(clause.getOperator()){
				case Clause.EQUAL:
					if(value == l_value){
						result = true;
					}
					else{
						result = false;
					}
				break;
				case Clause.LESS:
					if(value <= l_value){
						result = true;
					}
					else{
						result = false;
					}
				break;
				case Clause.GREATER:
					if(value > l_value){
						result = true;
					}
					else{
						result = false;
					}
				break;
				case Clause.BETWEEN:
					if(l_value < value && value <= u_value){
						result = true;
					}
					else{
						result = false;
					}
				break;
				default:
				break;
			}
		}
		catch(NumberFormatException nfe){
			nfe.printStackTrace();
		}
		
		return result;
		
	}
	
	protected boolean matchStringAsString(String value, Clause clause){
		
		boolean result = false;
		
		switch(clause.getOperator()){
			case Clause.EQUAL:
				if(value.equals(clause.getValue())){
					result = true;
				}
				else{
					result = false;
				}
			break;
			default:
			break;
		}
		
		return result;
	}
	
	public int countInstances(Instances dset, Clause c){
		
		int result=0;
		
		int i=0;
		
		for(i=0; i<dset.size(); i++){
			if(isMatch(dset.getAttributeInfo(),dset.getInstance(i),c)){
				result++;
			}
		}
		
		return result;
	
	}
	
	public int countInstances(Instances dset, String attName, String value){
		
		int result = 0;
		Attribute att = dset.getAttributeInfo().getAttribute(attName);
		Clause c = new Clause(attName, Clause.EQUAL, value);
		Instance instance;
		
		int i=0;
		for(i=0; i<dset.size(); i++){
			instance = dset.getInstance(i);
			if(matchStringAsString(instance.getValueAt(att.getId()),c)){
				result++;
			}
		}
		
		return result;
	}
	
	public Rule[] sortByResults(){ // sort key with result value in ascending order
		
		Rule[] rules = new Rule[results.size()];
		
		Enumeration e = results.keys();
		
		int i=0;
		while(e.hasMoreElements()){
			rules[i++]= (Rule)e.nextElement();
		}
		
		Arrays.sort(rules,this);
		
		return rules;
	}

	public void store(OutputStream os, boolean lower_best){
		
		Rule rules[] = sortByResults();
		BufferedWriter bw = new BufferedWriter(new OutputStreamWriter(os));
		
		rules = sortByResults();
		
		int i;
		if(lower_best){
			for(i=0; i < rules.length; i++){
				try{
					bw.write(rules[i].getRuleset().getID()+",");
					bw.write(rules[i].getID()+",");
					bw.write(results.get(rules[i]).toString()+"\n");
					bw.flush();
				}
				catch(IOException ioe){
					ioe.printStackTrace();
					break;
				}
			}
		}
		else{
			for(i=rules.length-1; i>=0; i--){
				try{
					bw.write(rules[i].getRuleset().getID()+",");
					bw.write(rules[i].getID()+",");
					bw.write(results.get(rules[i]).toString()+"\n");
					bw.flush();
				}
				catch(IOException ioe){
					ioe.printStackTrace();
					break;
				}
			}
		}
		
		try{
			bw.close();
		}
		catch(IOException ioe){
			ioe.printStackTrace();
		}
		
	}

}
